## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #	FILE: "indentation.tcl"
 #					  created: 27/7/97 {1:08:08 am}	
 #				     last update: 11/16/2001 {14:17:17 PM}	
 #	Author:	Vince Darley
 #	E-mail:	<vince@santafe.edu>
 #	  mail:	317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #	   www:	<http://www.santafe.edu/~vince/>
 #	
 # 
 #  modified   by  rev reason
 #  ---------- --- --- -----------
 #  2000-12-07 DWH 0.1 updated help text for electricReturn
 #  2001-09-26 JK      call to text::inCommentBlock replaced by call 
 #                     to text::isInDoubleComment (the former is obsolete)
 # ###################################################################
 ##

alpha::flag electricBraces 0.1 Electrics {global C C++ Java Tcl Perl} help {
    Enabling the 'Electric Braces' feature tells Alpha to treat the 
    left or right brace '{', '}' keys as special keypresses which 
    enter the '{' or '}' character, followed by a return and then 
    indent the following line correctly.  It is useful for those 
    programming modes in which '{' and '}' are used to delineate 
    blocks of code in 'for' loops or 'if-then-else' groups etc.
}

alpha::flag electricSemicolon 0.1 Electrics {global C C++ Java Perl} help {
    Enabling the 'Electric Semicolon' feature tells Alpha to treat the 
    semicolon key ';' as special keypresses which enters the ';' 
    character followed by a return and then indents the following line 
    correctly.  It is useful for some programming modes in which ';' 
    normally ends a line.
    
    The ';' key is context-dependent so you can still enter a 
    for( ; ; ) loop in C mode (for instace) without Alpha messing 
    things up.
}

alpha::declare flag electricReturn 0.1 {global} {
    if {[info tclversion] >= 8.0} {
	linkVar indentOnReturn
    }
    set indentOnReturn 0
    lappend flagPrefs(Electrics) electricReturn
} {set indentOnReturn 1} {set indentOnReturn 0} help {
    Enabling the 'Electric Return' feature tells Alpha to indent the 
    following line automatically whenever you press return.
    
    Press <control>-<return> to over-ride this behavior, to insert a
    carriage return without indenting.
}

alpha::flag electricColon 0.1 Electrics {global} help {
    Enabling the 'Electric Colon' feature tells Alpha to carry out a 
    special action when the user presses colon.
}

alpha::flag autoContinueComment 0.1 Electrics {global} help {
    Enabling the 'autoContinueComment' feature tells Alpha to check when
    the users hits return whether the current line is a comment, and if
    so, to indent and insert comment characters so that the following
    line continues the comment.
}

alpha::flag indentUsingSpacesOnly 0.1 Electrics {global TeX} help {
    If set, do not use tabs to indent, but spaces only.  This is mostly
    useful for modes in which the 'tab' character has a special meaning,
    such as python or TeX (the latter usually only for TeX as a programming
    language, not as a document preparation system).
}

alpha::flag commentsArentSpecialWhenIndenting 0.1 Electrics {global TeX} help {
    Indent lines to level of previous line if set, otherwise to level 
    of previous non-comment line (in which case Alpha will search 
    backwards for some distance).  If you're in the habit of indenting 
    your comments to the same level as your code, this setting 
    shouldn't matter (and setting it is slightly more efficient).
    
    One case in which it can be _much_ more efficient is when your 
    files contain vast comments (especially .dtx files in TeX mode, 
    for instance).  For these files, you should activate this feature.
}

namespace eval indent {}
namespace eval Bind {}
namespace eval text {}

proc IndentLine {} { bind::IndentLine }

proc typeText {t} {
    if {[isSelection]} {
	deleteSelection
    }
    insertText $t
}

proc normalLeftBrace {} {
    typeText "\{"
}
proc normalRightBrace {} {
    typeText "\}"
    blink [matchIt "\}" [pos::math [getPos] - 2]]
}
			
#  Electric indentation  #
proc bind::LeftBrace {} {
    if {[isSelection]} { deleteSelection }
    global electricBraces mode
    if {!$electricBraces} {
	insertText "\{"
	return
    }
    mode::proc electricLeft
}

proc ::electricLeft {} {
    if {![catch {search -s -l [lineStart [pos::math [lineStart [getPos]] - 1]] \
      -f 0 -r 0 "\}" [getPos]} res]} {
	set end [getPos]
	if {[pos::compare [getPos] != [maxPos]]} {
	    set end [pos::math $end + 1]
	}
	
	if {[regexp -- "\}\[ \t\r\n\]*else" [getText [lindex $res 0] $end]]} {
	    set res2 [search -s -f 0 -r 1 {else} [getPos]]
	    oneSpace
	    set text [getText [lindex $res2 0] [getPos]]
	    if {[lookAt [pos::math [getPos] - 1]] != " "} {
		append text " "
	    }
	    replaceText [pos::math [lindex $res 0] + 1] [getPos] " $text\{\r"
	    bind::IndentLine
	    return 
	}
    }
    set pos [getPos]
    set i [text::firstNonWsLinePos $pos]
    
    if {([pos::compare $i == $pos]) || ([lookAt [pos::math $pos - 1]] == " ")} {
	insertText "\{\r" [text::indentString $pos] [text::Tab]
    } else {
	insertText " \{\r" [text::indentString $pos] [text::Tab]
    }
}

proc ::electricRight {} {
    set pos [getPos]
    set start [lineStart $pos]
    
    if {[catch {matchIt "\}" [pos::math $pos - 1]} matched]} {
	beep
	message "No matching '\{'!"
	return
    }
    set text [getText [lineStart $matched] $matched]
    regexp "^\[ \t\]*" $text indentation
    if {[string trim [getText $start $pos]] != ""} {
	insertText "\r" $indentation "\}\r" $indentation
	blink $matched
	return
    }
    set text "${indentation}\}\r$indentation"
    replaceText $start $pos $text
    goto [pos::math $start + [string length $text]]
    blink [matchIt "\}" [pos::math $start - 2]]
}

proc bind::RightBrace {} {
    if {[isSelection]} { deleteSelection }
    global electricBraces mode
    if {!$electricBraces} {
	insertText "\}"
	catch {blink [matchIt "\}" [pos::math [getPos] - 2]]}
	return
    }
    mode::proc electricRight
}

proc bind::electricSemi {} {
    global electricSemicolon
    if {!$electricSemicolon} {
	typeText ";"
    } else {
	mode::proc electricSemi
    }
}

proc ::electricSemi {} {
    if {[isSelection]} { deleteSelection }
    set pos [getPos]
    set text [getText [lineStart $pos] $pos]
    
    set inFor 0
    if {[string first "for" $text] != "-1"} {
	set len [string length $text]
	for {set i 0} {$i < $len} {incr i} {
	    switch -- [string index $text $i] {
		"("	{ incr inFor }
		")"	{ incr inFor -1 }
	    }
	}
    }
    
    insertText ";"
    if {!($inFor != 0 || [text::isInComment $pos] \
      || [text::isInString $pos] || [text::isEscaped $pos])} {
	mode::proc carriageReturn
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "bind::CarriageReturn" --
 #	
 #  General purpose CR procedure.  Should be bound to 'return' for all
 #  modes really.  Calls a mode-specific procedure if required.
 #  
 # -------------------------------------------------------------------------
 ##
proc bind::CarriageReturn {} {
    if {[isSelection]} { deleteSelection }
    global autoContinueComment
    if {$autoContinueComment} {
	set p [getPos]
	if {[text::isInComment $p start]} {
	    insertReturnAndContinueComment $start
	    return
	}
    }
    mode::proc carriageReturn
}

proc bind::continueComment {} {
    if {[isSelection]} { deleteSelection }
    set p [getPos]
    # special case for beginning of line
    if {[pos::compare $p == [lineStart $p]]} {
	backwardChar
	set p [getPos]
    }
    if {![text::isInComment $p start]} {
	global prefixString
	if {[info exists prefixString]} {
	    set start $prefixString
	} else {
	    set start "/"
	}
    }
    insertReturnAndContinueComment $start
}

# We could possibly hook into this procedure in the future to allow
# different comment style handling.
proc insertReturnAndContinueComment {prefixOf} {
    # Use this sequence so the user can easily remove
    # the extra stuff inserted, with cmd-x
    insertText "\r"
    goto [getPos]
    replaceText [getPos] [getPos] $prefixOf
}

proc ::carriageReturn {} {
    insertText "\r"
    global indentOnReturn
    if {$indentOnReturn} {bind::IndentLine}
}

proc bind::IndentLine {} {
    mode::proc indentLine
}

proc insertActualTab {} { typeText "\t" }

proc text::surroundingWord {} {
    if {[isSelection]} {return [list [getPos] [selEnd]]}
    set pos [getPos]
    backwardWord ; set pos0 [getPos]
    forwardWord  ; set pos1 [getPos]
    forwardWord  ; set pos2 [getPos]
    backwardWord ; set pos3 [getPos]
    goto $pos
    if {[pos::compare $pos >= $pos0] && [pos::compare $pos <= $pos1]} {
	return [list $pos0 $pos1]
    } elseif {[pos::compare $pos >= $pos3] && [pos::compare $pos <= $pos2]} {
	return [list $pos3 $pos2]
    } else {
	return [list $pos $pos]
    }
}

#  Indentation utility routines  #

proc lineEnd {pos} { return [pos::math [nextLineStart $pos] -1] }

proc posX {pos} {return [lindex [posToRowCol $pos] 1] }
# the above version doesn't work!
if {[info tclversion] < 8.0} {
proc posX {pos} {
    return [string length [text::maxSpaceForm [getText [lineStart $pos] $pos]]]
}
}

proc beginningOfLineSmart {} {
    set start [getText [lineStart [getPos]] [getPos]]
    if {[string length $start] && ([string trim $start] == "")} {
	beginningOfLine
    } else {
	goto [text::firstNonWsLinePos [getPos]]
    }
}

proc beginningOfLineSelectSmart {} {
    set start [getText [lineStart [getPos]] [getPos]]
    if {[string length $start] && ([string trim $start] == "")} {
	beginningLineSelect
    } else {
	select [text::firstNonWsLinePos [getPos]] [selEnd]
    }
}

proc beginningOfLineSmart {} {
    set firstNonWsPos [text::firstNonWsLinePos [getPos]]
    if {[pos::compare [getPos] == $firstNonWsPos]} {
	beginningOfLine
    } else {
	goto $firstNonWsPos
    }
}

proc text::firstNonWs {pos} {
    set p [text::firstNonWsPos $pos]
    if {[pos::compare $p > [minPos]]} {
	return [lookAt $p]
    } else {
	return ""
    }
}

## 
 # -------------------------------------------------------------------------
 #   
 # "text::firstNonWsPos" --
 #  
 #  This returns the position of the first non-whitespace character from
 #  the start of pos' line.  It need not return something on the same
 #  line.
 # -------------------------------------------------------------------------
 ##
proc text::firstNonWsPos {pos} {
    if {[catch {lindex [search -s -f 1 -r 1 "\[^ \t\r\n\]" [lineStart $pos]] 0} res]} {
	return [lineStart $pos]
    } else {
	return $res
    }
}

proc text::firstNonWsLinePos {pos} {
    if {[catch {lindex [search -s -f 1 -r 1 "\[^ \t\]" [lineStart $pos]] 0} res]} {
	return [lineStart $pos]
    } else {
	return $res
    }
}

proc text::indentation {pos} {
    return [search -s -m 0 -f 1 -r 1 "^\[ \t\]*\[^ \t\]" [lineStart $pos]]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "text::minSpaceForm" --
 # 
 #  Converts to minimal form: tabs then spaces.  Uses one regsub to do
 #  the job.  Note that the regexp used relies upon the left-to-right
 #  priority of branch matching.  If the regexp library used is more
 #  sophisticated and finds maximal matches, then this is no good.
 #  In that case use:
 #		regsub -all $sp $ws "\t" ws
 #		regsub -all " +\t" $ws "\t" ws
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.1} {
    proc text::minSpaceForm {ws} {
	if {[string length [set sp [spacesEqualTab]]]} {
	    regsub -all "($sp| +\t)" $ws "\t" ws
	} else {
	    regsub -all "\t" $ws "" ws
	}
	return $ws
    }
} else {
    proc text::minSpaceForm {ws} {
	if {[string length [set sp [spacesEqualTab]]]} {
	    regsub -all $sp $ws "\t" ws
	    regsub -all " +\t" $ws "\t" ws
	} else {
	    regsub -all "\t" $ws "" ws
	}
	return $ws
    }
}


## 
 # -------------------------------------------------------------------------
 # 
 # "text::maxSpaceForm" --
 # 
 #  Converts it to maximal form - just spaces.
 #  Just uses one funky regsub to do the job!  Takes account of tab-size,
 #  spaces interspersed with tabs,...
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.1} {
    proc text::maxSpaceForm {ws} {
	if {[string length [set sp [spacesEqualTab]]]} {
	    regsub -all "(($sp)*) *\t" $ws "\\1$sp" ws
	} else {
	    regsub -all "\t" $ws "" ws
	}
	return $ws
    }
} else {
    proc text::maxSpaceForm {ws} {
	if {[string length [set sp [spacesEqualTab]]]} {
	    regsub -all $sp $ws "\t" ws
	    regsub -all " +\t" $ws "\t" ws
	    regsub -all "\t" $ws "$sp" ws
	} else {
	    regsub -all "\t" $ws "" ws
	}
	return $ws
    }
}


## 
 # -------------------------------------------------------------------------
 # 
 # "spacesEqualTab" --
 # 
 #  Return the number of spaces equivalent to a single tab. If tabs are too
 #  big, this won't work.
 # -------------------------------------------------------------------------
 ##
proc spacesEqualTab {} {
    getWinInfo a
    string range "              " 1 $a(tabsize)
}

proc doubleLookAt {pos} {return [getText $pos [pos::math $pos + 2]]}

set bind::_IndentSpaces "                                                   \
                                         "
set bind::_IndentTabs "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"

proc text::indentOf {size} {
    global bind::_IndentSpaces bind::_IndentTabs indentUsingSpacesOnly
    if {$indentUsingSpacesOnly} {
	return [string range ${bind::_IndentSpaces} 1 $size]
    } else {
	getWinInfo a
	if {$a(tabsize) == 0} {
	    return [string range ${bind::_IndentSpaces} 1 $size]
	} else {
	    set ret [string range ${bind::_IndentTabs} 1 [expr {$size / $a(tabsize)}]]
	    append ret [string range ${bind::_IndentSpaces} 1 [expr {$size % $a(tabsize)}]]
	}
    }
    return $ret
}

# returns the indent string of the line named by 'pos'
proc text::indentString {pos} {
    set beg [lineStart $pos]
    regexp "^\[ \t\]*" [getText $beg [nextLineStart $beg]] white
    return $white
}

# returns the indent string of the line up to position 'pos' 
proc text::indentTo {pos} {
    regexp "^\[ \t\]*" [getText [lineStart $pos] $pos] white
    return $white
}

## 
 # -------------------------------------------------------------------------
 # 
 # "text::indentBy" --
 # 
 #  Take the given block of text, and insert/remove spaces and tabs to
 #  indent it $by spaces to the left or right. This version should work
 #  ok for Tcl 7.5/8.0/8.1
 # -------------------------------------------------------------------------
 ##
proc text::indentBy {text by} {
    global bind::_IndentSpaces indentUsingSpacesOnly
    set sp [spacesEqualTab]
    # Convert all leading whitespace to spaces
    if {[string length $sp]} {
	while {[regsub -all "((^|\r|\n)($sp)*) *\t" $text "\\1$sp" text]} {}
    } else {
	while {[regsub -all "(^|\r|\n)( *)\t" $text "\\1" text]} {}
    }
    set sby [string range ${bind::_IndentSpaces} 1 [expr {abs($by)}]]
    if {$by < 0} {
	# need to indent less
	regsub -all "(^|\r|\n)$sby" $text "\\1" text
    } else {
	# need to indent more: add spaces to beginning of each line,
	# apart from blank lines and the final line
	regsub -all "(\[\r\n\])(\[^\r\n\])" $sby$text "\\1$sby\\2" text
    }
    # We already converted everything to spaces, so we only convert
    # to tabs if the user wants them.
    if {!$indentUsingSpacesOnly && [string length $sp]} {
	while {[regsub -all "((^|\r|\n)\t*)$sp" $text "\\1\t" text]} {}
    }
    return $text
}

proc text::halfTab {} {
    global indent_amounts
    return [string range "              " 1 $indent_amounts(1)]
}
proc text::Tab {} {
    global indentationAmount
    return [text::indentOf $indentationAmount]
}

proc text::getTabSize {} {
    getWinInfo a
    return $a(tabsize)
}

#  General purpose indentation  #

proc indentSelection {} {
    mode::proc indentRegion
}

## THE FOLLOWING PROC IS OBSOLETE --- SUPERSEEDED BY text::isInDoubleComment
 # -------------------------------------------------------------------------
 # 
 # "text::inCommentBlock" --
 # 
 #  Returns 'startpos endpos' if true, else returns an error.  Not 
 #  particularly robust, but not too bad either
 # -------------------------------------------------------------------------
 ##
proc text::inCommentBlock {pos} {
    set chars [comment::Characters Paragraph]
    set start [string trim [lindex $chars 0]]
    set end [string trim [lindex $chars 1]]
    if {$start == $end} {
	error "No"
    }
    set cS [search -s -f 0 -r 0 -l [pos::math $pos - 1000] $start $pos]
    set cE [search -s -f 1 -r 0 -l [pos::math $pos + 1000] $end [lindex $cS 1]]
    if {[pos::compare $pos >= [lindex $cE 1]]} {	
	error "No"
    } else {
	return [list [lindex $cS 0] [lindex $cE 1]]
    }
}


# Tom's new regexp which I don't use now.  Shame.
#set commentRegexp       {/\*[^*]*\*+([^/*][^*]*\*+)*/}


## 
 # -------------------------------------------------------------------------
 # 
 # "::indentLine" --
 # 
 #  This procedure can be used by any mode which defines its
 #  'correctIndentation' procedure.  It is therefore no longer
 #  necessary for a mode to define its own indentLine procedure.
 #  
 #  The advantage of this is (i) extra code-sharing, and (ii)
 #  the 'position after indentation' preference may be global.
 # -------------------------------------------------------------------------
 ##
proc ::indentLine {} {
    # get details of current line
    set beg [lineStart [set p [getPos]]]
    set text [getText $beg [nextLineStart $beg]]
    regexp "^\[ \t\]*" $text white
    set next [pos::math $beg + [string length $white]]

    set lwhite [mode::proc correctIndentation $p [getText $next [nextLineStart $next]]]
    set replacement [text::indentOf $lwhite]
    if {$white != $replacement} {
	replaceText $beg $next $replacement
    }
    global positionAfterIndentation
    if {$positionAfterIndentation && [string length [string trim $text]]} {
	# Keep relative position.
	set to [pos::math $p + [string length $replacement] - [pos::diff $beg $next]]
	if {[pos::compare $to < $beg]} {
	    goto $beg
	} else {
	    goto $to
	}
    } else {
	goto [pos::math $beg + [string length $replacement]]
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "::correctIndentation" --
 # 
 #  This is a fallback procedure for all modes which do not
 #  define their own 'correctIndentation' procedure.  Most complex
 #  modes will probably want to over-ride this.
 # -------------------------------------------------------------------------
 ##
proc ::correctIndentation {pos {next ""}} {
    set pos [lineStart $pos]
    # Find last previous non-comment non-white line and get its leading
    # whitespace:
    while 1 {
	if {[pos::compare $pos == [minPos]] \
	  || [catch {search -s -f 0 -r 1 -i 0 -m 0 \
	  "^\[ \t\]*\[^ \t\r\n\]" [pos::math $pos - 1]} lst]} {
	    # search failed at top of file
	    set line "#"
	    return 0
	}
	set lst0 [lindex $lst 0]  ;# the start of the found line
	set lst1 [lindex $lst 1]  ;# just after first non-white on the line

	global commentsArentSpecialWhenIndenting
	if {!$commentsArentSpecialWhenIndenting} {
	    if {[text::isInDoubleComment $lst1 res] \
	      || [text::isInSingleComment $lst1 res]} {
		if {[pos::compare [lindex $res 0] < $lst0]} {
		    set pos [lindex $res 0]
		} else {
		    set pos $lst0
		}
		# look further back.  ($pos is smaller than in previous
		# loop)
		continue
	    }
	}
	
	# the essence is really the following --- all the above was special-case yoga
	set line [getText $lst0 [pos::math [nextLineStart $lst0] - 1]]
	set lwhite [posX [pos::math $lst1 - 1]]
	return $lwhite
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "::correctBracesIndentation" --
 # 
 #  This is a procedure which can be used by modes which don't want
 #  to write a complex <mode>::correctIndentation procedure of their
 #  own, but require braces to be correctly indented.
 # -------------------------------------------------------------------------
 ##
proc ::correctBracesIndentation {pos {next ""}} {
    set pos [lineStart $pos]
    if {[pos::compare $pos != [minPos]]} {
	global commentsArentSpecialWhenIndenting
	# Find last previous non-comment line and get its leading whitespace
	while 1 {
	    if {[pos::compare $pos == [minPos]] \
	      || [catch {search -s -f 0 -r 1 -i 0 -m 0 \
	      "^\[ \t\]*\[^ \t\r\n\]" [pos::math $pos - 1]} lst]} {
		# search failed at top of file
		set line "#"
		set lwhite 0
		break
	    }
	    if {!$commentsArentSpecialWhenIndenting && \
	      [text::isInDoubleComment [lindex $lst 0] res]} {
		set pos [lindex $res 0]
	    } else {
		set line [getText [lindex $lst 0] \
		  [pos::math [nextLineStart [lindex $lst 0]] - 1]]
		set lwhite [posX [pos::math [lindex $lst 1] - 1]]	
		break
	    }
	}
	
	regexp "(\[^ \t\])\[ \t\]*\$" $line "" nextC
	global indentationAmount electricColon
	if {($nextC == "\{")} {
	    incr lwhite $indentationAmount
	} elseif {$nextC == ":" && $electricColon} {
	    incr lwhite [expr {$indentationAmount /2}]
	}
	set text [getText [lineStart $pos] $pos]
	append text $next
	if {[regexp ":\[ \t\r\n\]*\$" $text] && $electricColon} {
	    incr lwhite [expr {-$indentationAmount / 2}]
	}
	if {[string index $next 0] == "\}"} {
	    incr lwhite [expr {-$indentationAmount}]
	}
    } else {
	set lwhite 0
    }
    return $lwhite
}


proc ::indentRegion {} {
    set from [lindex [posToRowCol [getPos]] 0]
    set to [lindex [posToRowCol [selEnd]] 0]
    select [getPos]
    while {$from <= $to} {
	goto [rowColToPos $from 0]
	bind::IndentLine
	incr from
    }
}
