# init.tcl --
#
# Default system startup file for Tcl-based applications.  Defines
# "unknown" procedure and auto-load facilities.
#
# SCCS: @(#) init.tcl 1.86 97/08/08 10:37:39
#
# Copyright (c) 1991-1993 The Regents of the University of California.
# Copyright (c) 1994-1997 Sun Microsystems, Inc.
# Some additions copyright (c) 1997-2000 Vince Darley.

set errorCode ""
set errorInfo ""

if {[info commands tclLog] == "" \
  || [string first "puts stderr" [info body tclLog]] >= 0} {
    proc tclLog {string} {
	message [string trim $string "\r"]
    }
}
if {[info tclversion] >= 8.0} {
    namespace eval index {}
    namespace eval procs {}
    # used to force some child namespaces into existence
    ;proc namesp {var} {
	if {[catch "uplevel global $var"]} {
	    set ns ""
	    while {[regexp "^(::)?($ns\[a-zA-Z_\]+::)" $var ns]} {
		uplevel "namespace eval $ns {}"
	    }
	}
    }
} else {
    ;proc namesp {var} {}
    # Remove 'load' command.  We implement our own 'evaluate'
    # which takes an optional argument.
    rename load {}
}

# 7.1 doesn't rename unbind in the actual application
if {[info commands unBind] == ""} { 
    if {[info commands unbind] != ""} {rename unbind unBind}
}

# define compatibility procs for menu, bind, unbind
if {[info commands bind] == ""} {
    proc bind {args} { uplevel 1 Bind $args }
    proc unbind {args} { uplevel 1 unBind $args }
    proc menu {args} { 
	regsub -all "\{menu " $args "\{Menu " args
	uplevel 1 Menu $args 
    }
}
namespace eval file {}
# determine platform specific directory symbol
regexp {Z(.)Z} [file join Z Z] "" file::separator
# To get rid of the stupid {} variable created by the above line.
# We 'catch' in case a future version of Tcl fixes this silliness.
catch {unset {}}

proc catchNoClobber {script args} {
    global errorCode errorInfo
    set oldErrorCode $errorCode
    set oldErrorInfo $errorInfo
    if {[set ret [uplevel 1 [list catch $script] $args]]} {
	set errorCode $oldErrorCode
	set errorInfo $oldErrorInfo
	return $ret
    } else {
	return 0
    }
}

# Note: if this DOES exist (e.g. in Alphatk, and possibly Alpha 8),
# then procs like auto_load must already exist, and have been loaded
# in from Tcl's core library routines.
if {![info exists useStandardTclIndices]} {
## 
 # -------------------------------------------------------------------------
 # 
 # "unknown" --
 # 
 #  Almost the same as standard Tcl 8 unknown.  Since we're on a Mac,
 #  I removed the auto_execok flag, and for some reason had to change
 #  'history change $newcmd 0' to 'history change $newcmd'
 # -------------------------------------------------------------------------
 ##
# unknown --
# This procedure is called when a Tcl command is invoked that doesn't
# exist in the interpreter.  It takes the following steps to make the
# command available:
#
#	1. See if the autoload facility can locate the command in a
#	   Tcl script file.  If so, load it and execute it.
#	2. If the command was invoked interactively at top-level:
#	    (a) see if the command exists as an executable UNIX program.
#		If so, "exec" the command.
#	    (b) see if the command requests csh-like history substitution
#		in one of the common forms !!, !<number>, or ^old^new.  If
#		so, emulate csh's history substitution.
#	    (c) see if the command is a unique abbreviation for another
#		command.  If so, invoke the command.
#
# Arguments:
# args -	A list whose elements are the words of the original
#		command, including the command name.
proc unknown args {
    global auto_noload env unknown_pending tcl_interactive
    global errorCode errorInfo
    
    # Save the values of errorCode and errorInfo variables, since they
    # may get modified if caught errors occur below.  The variables will
    # be restored just before re-executing the missing command.
    
    set savedErrorCode $errorCode
    set savedErrorInfo $errorInfo
    set name [lindex $args 0]
    if {![info exists auto_noload]} {
	#
	# Make sure we're not trying to load the same proc twice.
	#
	if {[info exists unknown_pending($name)]} {
	    return -code error "self-referential recursion in \"unknown\" for command \"$name\"";
	}
	set unknown_pending($name) pending;
	set ret [catch {auto_load $name} msg]
	unset unknown_pending($name);
	if {$ret != 0} {
	    return -code $ret -errorcode $errorCode \
	      "error while autoloading \"$name\": $msg"
	}
	if {![array size unknown_pending]} {
	    unset unknown_pending
	}
	if {$msg} {
	    set errorCode $savedErrorCode
	    set errorInfo $savedErrorInfo
	    set code [catch {uplevel 1 $args} msg]
	    if {$code ==  1} {
		#
		# Strip the last five lines off the error stack (they're
		# from the "uplevel" command).
		#
		
		set new [split $errorInfo \n]
		set new [join [lrange $new 0 [expr {[llength $new] - 6}]] \n]
		return -code error -errorcode $errorCode \
		  -errorinfo $new $msg
	    } else {
		return -code $code $msg
	    }
	}
    }
    if {([info level] == 1) && ([info script] == "") \
      && [info exists tcl_interactive] && $tcl_interactive} {
	set errorCode $savedErrorCode
	set errorInfo $savedErrorInfo
	if {$name == "!!"} {
	    set newcmd [history event]
	} elseif {[regexp {^!(.+)$} $name dummy event]} {
	    set newcmd [history event $event]
	} elseif {[regexp {^\^([^^]*)\^([^^]*)\^?$} $name dummy old new]} {
	    set newcmd [history event -1]
	    catch {regsub -all -- $old $newcmd $new newcmd}
	}
	if {[info exists newcmd]} {
	    tclLog "\r$newcmd"
	    history change $newcmd
	    return [uplevel $newcmd]
	}
	
	set ret [catch {set cmds [info commands $name*]} msg]
	if {[string compare $name "::"] == 0} {
	    set name ""
	}
	if {$ret != 0} {
	    return -code $ret -errorcode $errorCode \
	      "error in unknown while checking if \"$name\" is a unique command abbreviation: $msg"
	}
	if {[llength $cmds] == 1} {
	    return [uplevel [lreplace $args 0 0 $cmds]]
	}
	if {[llength $cmds] != 0} {
	    if {$name == ""} {
		return -code error "empty command name \"\""
	    } else {
		return -code error \
		  "ambiguous command name \"$name\": [lsort $cmds]"
	    }
	}
    }
    return -code error "invalid command name \"$name\""
}

## 
 # -------------------------------------------------------------------------
 # 
 # "auto_load" --
 # 
 #  I use this separate proc to be closer to the standard Tcl 8 system
 #  of unknown-loading.
 # -------------------------------------------------------------------------
 ##
proc auto_load cmd {
    set f [procs::find $cmd]
    if {$f != ""} {
	uplevel \#0 source [list $f]
	return [expr {[llength [info commands $cmd]] != 0}]
    }
    if {[regsub {^::} $cmd "" cmd]} {
	set f [procs::find $cmd]
	if {$f != ""} {
	    uplevel \#0 source [list $f]
	    return [expr {[llength [info commands $cmd]] != 0}]
	}
    }
    # to cope with some Tcl 8 package stuff
    if {[info tclversion] < 8.0} {
	return 0
    }
    global auto_index auto_oldpath auto_path

    set namespace [uplevel {namespace current}]
    set nameList [auto_qualify $cmd $namespace]
    # workaround non canonical auto_index entries that might be around
    # from older auto_mkindex versions
    lappend nameList $cmd
    foreach name $nameList {
	if {[info exists auto_index($name)]} {
	    uplevel #0 $auto_index($name)
	    return [expr {[info commands $name] != ""}]
	}
    }
    if {![info exists auto_path]} {
	return 0
    }

    if {![auto_load_index]} {
	return 0
    }

    foreach name $nameList {
	if {[info exists auto_index($name)]} {
	    uplevel #0 $auto_index($name)
	    if {[info commands $name] != ""} {
		return 1
	    }
	}
    }
    return 0
}

# auto_load_index --
# Loads the contents of tclIndex files on the auto_path directory
# list.  This is usually invoked within auto_load to load the index
# of available commands.  Returns 1 if the index is loaded, and 0 if
# the index is already loaded and up to date.
#
# Arguments: 
# None.

proc auto_load_index {} {
    global auto_index auto_oldpath auto_path errorInfo errorCode

    if {[info exists auto_oldpath]} {
	if {$auto_oldpath == $auto_path} {
	    return 0
	}
    }
    set auto_oldpath $auto_path

    # Check if we are a safe interpreter. In that case, we support only
    # newer format tclIndex files.

    set issafe [interp issafe]
    for {set i [expr {[llength $auto_path] - 1}]} {$i >= 0} {incr i -1} {
	set dir [lindex $auto_path $i]
	set f ""
	if {$issafe} {
	    catch {source [file join $dir tclIndex]}
	} elseif {[catch {set f [open [file join $dir tclIndex]]}]} {
	    continue
	} else {
	    set error [catch {
		set id [gets $f]
		if {$id == "# Tcl autoload index file, version 2.0"} {
		    eval [read $f]
		} elseif {$id == \
		    "# Tcl autoload index file: each line identifies a Tcl"} {
		    while {[gets $f line] >= 0} {
			if {([string index $line 0] == "#")
				|| ([llength $line] != 2)} {
			    continue
			}
			set name [lindex $line 0]
			set auto_index($name) \
			    "source [file join $dir [lindex $line 1]]"
		    }
		} else {
		    error \
		      "[file join $dir tclIndex] isn't a proper Tcl index file"
		}
	    } msg]
	    if {$f != ""} {
		close $f
	    }
	    if {$error} {
		error $msg $errorInfo $errorCode
	    }
	}
    }
    return 1
}

# auto_qualify --
#
# Compute a fully qualified names list for use in the auto_index array.
# For historical reasons, commands in the global namespace do not have leading
# :: in the index key. The list has two elements when the command name is
# relative (no leading ::) and the namespace is not the global one. Otherwise
# only one name is returned (and searched in the auto_index).
#
# Arguments -
# cmd		The command name. Can be any name accepted for command
#               invocations (Like "foo::::bar").
# namespace	The namespace where the command is being used - must be
#               a canonical namespace as returned by [namespace current]
#               for instance.

proc auto_qualify {cmd namespace} {

    # count separators and clean them up
    # (making sure that foo:::::bar will be treated as foo::bar)
    set n [regsub -all {::+} $cmd :: cmd]

    # Ignore namespace if the name starts with ::
    # Handle special case of only leading ::

    # Before each return case we give an example of which category it is
    # with the following form :
    # ( inputCmd, inputNameSpace) -> output

    if {[regexp {^::(.*)$} $cmd x tail]} {
	if {$n > 1} {
	    # ( ::foo::bar , * ) -> ::foo::bar
	    return [list $cmd]
	} else {
	    # ( ::global , * ) -> global
	    return [list $tail]
	}
    }
    
    # Potentially returning 2 elements to try  :
    # (if the current namespace is not the global one)

    if {$n == 0} {
	if {[string compare $namespace ::] == 0} {
	    # ( nocolons , :: ) -> nocolons
	    return [list $cmd]
	} else {
	    # ( nocolons , ::sub ) -> ::sub::nocolons nocolons
	    return [list ${namespace}::$cmd $cmd]
	}
    } else {
	if {[string compare $namespace ::] == 0} {
	    #  ( foo::bar , :: ) -> ::foo::bar
	    return [list ::$cmd]
	} else {
	    # ( foo::bar , ::sub ) -> ::sub::foo::bar ::foo::bar
	    return [list ${namespace}::$cmd ::$cmd]
	}
    }
}

# auto_mkindex:
# Regenerate a tclIndex file from Tcl source files.  Takes two arguments:
# the name of the directory in which the tclIndex file is to be placed,
# and a glob pattern to use in that directory to locate all of the relevant
# files.
proc auto_mkindex {dir {files *.tcl}} {
    # Due to some peculiarities with current working directories
    # under some MacOS/HFS+/other conditions, we avoid using
    # 'cd' and 'pwd' explicitly if possible.
    set dir [file nativename $dir]
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    if {$dir == ":" || $dir == "."} {
		set dir [pwd]
	    }
	}
	default {
	    if {$dir == "."} {
		set dir [pwd]
	    }
	}
    }
    # So we can handle relative path names
    if {[file pathtype $dir] == "relative"} {
	set dir [file join [pwd] $dir]
    }
    if {[info tclversion] < 8.0} {
	if {![catchNoClobber {file readlink $dir} _root]} {
	    set dir $_root
	}
    } else {
	if {[file type $dir] == "link"} {
	    set dir [file readlink $dir]
	}
    }
    set dir [string trim $dir :]
    append line "# Tcl autoload index file: each line\
      identifies a file (nowrap)\n\n"
    set indexvar "[file tail [string trim $dir :]]_index"
    append line "set \"${indexvar}\" \{\n"
    
    set cid [scancontext create]
    # This pattern is used to extract procedures when the 'scanfile'
    # command is used below.  We don't do anything too dramatic if
    # the procedure name can't be extracted.  The most likely cause
    # is a garbled file.
    scanmatch $cid "^\[ \t\]*proc\[ \t\]" {
	if {[regexp "^\[ \t\]*proc\[ \t\]+((\"\[^\"\]+\")|(\{\[^\}\]+\})|(\[^ \t\]*))" \
	  $matchInfo(line) match procName]} {
	    append line "$procName "
	} else {
	    message "Couldn't extract a proc from '$matchInfo(line)'!"
	}
    }
    foreach file [glob -dir $dir -- $files] {
	watchCursor
	set f ""
	append line "\{[file tail $file]\14 "
	message [file tail $file]
	if {[catch {open $file r} fid]} {
	    lappend errors $fid
	    lappend errorFiles $file
	} else {
	    if {[catch {scanfile $cid $fid} err]} {
		lappend errors $err
		lappend errorFiles $file
	    }
	    close $fid
	}
	append line "\}\n"
    }
    
    scancontext delete $cid
    
    append line "\}\n"
    if {[info exists errors]} {
	if {[dialog::yesno -y "View the error" -n "Continue" \
	  "The following files: [join $errorFiles ,] were unable\
	  to be opened or scanned for procedures to store in Tcl index\
	  files.  This is a serious error.  Alpha will not be\
	  able to find procedures stored in those files, and will\
	  therefore fail to function correctly.  You should\
	  ascertain the cause of these\
	  problems and fix them.  Your disk may be damaged.\r\
	  To avoid some of these problems, the Tcl index file\
	  in $dir will not be replaced."]} {
	    dialog::alert [join $errors "\r"]
	}
    } else {
	if {![file writable $dir]} {
	    # This is a read-only directory, so there isn't
	    # a problem that we couldn't write to it.  Probably
	    # it's a system directory such as the base Tcl library.
	    if {[file exists [file join $dir tclIndexx]]} {
		message "'$dir' is read-only, so I'll use the existing\
		  AlphaTcl index."
	    } elseif {[file exists [file join $dir tclIndex]]} {
		message "'$dir' is read-only, so I'll use the existing\
		  Tcl index."
	    } else {
		message "'$dir' is read-only, and there is no Tcl index.\
		  This means Alpha will not be able to auto-load\
		  procedures in that directory"
	    }
	} else {
	    if {[catch {open [file join $dir tclIndexx] w} fid]} {
		if {![file::isLocal $dir]} {
		    message "The Tcl index file in $dir could not\
		      be rewritten, probably because it is on a remote\
		      volume and you don't have permission."
		} else {
		    dialog::alert "The Tcl index file in $dir could not\
		      be rewritten.  Perhaps the file is locked or read-only?\
		      The old index will be left intact, but you should fix\
		      this problem so Alpha can index new files in\
		      this directory."
		}
	    } else {
		if {[catch {puts -nonewline $fid $line} err]} {
		    if {[dialog::yesno -y "View the error" -n "Continue" \
		      "The Tcl index file in $dir was successfully opened,\
		      but Alpha encountered an error while writing to the\
		      file.  This is a very serious problem, and Alpha will\
		      probably no longer function correctly.  At the very\
		      least you will need to reinstall that directory, and\
		      perhaps all of Alpha."]} {
			dialog::alert $err
		    }
		}
		catch {close $fid}
	    }
	}
	foreach i [info vars $indexvar] {
	    global $i
	    unset $i
	}
    }
    
}

## 
 # -------------------------------------------------------------------------
 # 
 # "auto_reset" --
 # 
 #  After rebuilding indices, Tcl retains its old index information unless
 #  we tell it not to.
 # -------------------------------------------------------------------------
 ##
proc auto_reset {} {
    global auto_path
    foreach path $auto_path {
	if {![file exists $path]} continue
	set index "[file tail $path]_index"
	global $index
	catch {unset $index}
    }
}

proc procs::findIn {cmd pathlist} {
    regsub -all {[][\$?^|*+()\{\}]} $cmd {\\&} cmd
    foreach path $pathlist {
	if {![file exists $path]} continue
	if {[info tclversion] < 8.0} {
	    if {![catchNoClobber {file readlink $path} _path]} {
		set path $_path
	    }
	} else {
	    if {[file type $path] == "link"} {
		if {[catchNoClobber {set path [file readlink $path]}]} {
		    # forget about this one
		    continue
		}
	    }
	}
	set index "[file tail $path]_index"
	global $index
	if {![info exists $index]} {
	    if {![file exists [file join $path tclIndexx]]} continue
	    if {[catch [list uplevel \#0 source [list [file join $path tclIndexx]]] err]} {
		alertnote "Tcl index in $path is corrupt.  It throws an error: $err"
	    } else {
		if {![info exists $index]} {
		    alertnote "Tcl index in $path is incorrectly formed.  It\
		      should set the variable $index but doesn't.  You should\
		      fix this problem."
		}
	    }
	}
	if {[info exists $index]} {
	    if {[regexp "\n\{(\[^\14\]+)\14\[^\n\]* \[\"\{\]?(::)?${cmd}\[\"\}\]? " [set $index] dummy file]} {
		return [file join $path $file]
	    }
	}
    }
    return ""
}

proc procs::find {cmd} {
    global auto_path
    procs::findIn $cmd $auto_path
}

} else {
    # If we're using standard Tcl indices
    proc procs::findIn {cmd pathlist} {
	for {set i [expr {[llength $pathlist] - 1}]} {$i >= 0} {incr i -1} {
	    set dir [lindex $pathlist $i]
	    set f ""
	    if {[catch {set f [open [file join $dir tclIndex]]}]} {
		continue
	    } else {
		set error [catch {
		    set id [gets $f]
		    if {$id == "# Tcl autoload index file, version 2.0"} {
			eval [read $f]
		    } elseif {$id == \
			"# Tcl autoload index file: each line identifies a Tcl"} {
			while {[gets $f line] >= 0} {
			    if {([string index $line 0] == "#")
				    || ([llength $line] != 2)} {
				continue
			    }
			    set name [lindex $line 0]
			    set auto_index($name) \
				"source [file join $dir [lindex $line 1]]"
			}
		    } else {
			error \
			  "[file join $dir tclIndex] isn't a proper Tcl index file"
		    }
		}]
		if {$f != ""} {
		    close $f
		}
		if {$error} {
		    continue
		}
	    }
	}
	set namespace [uplevel {namespace current}]
	set nameList [auto_qualify $cmd $namespace]
	# workaround non canonical auto_index entries that might be around
	# from older auto_mkindex versions
	lappend nameList $cmd
	foreach name $nameList {
	    if {[info exists auto_index($name)]} {
		return [lindex $auto_index($name) 1]
	    }
	}
	return ""
    }
    
    proc procs::find {cmd} {
	set entry [uplevel 1 [list findIndexEntry $cmd]]
	if {[string length $entry]} {
	    return [lindex $entry 1]
	}
	return ""
    }

    # Basically the same as 'auto_load', but doesn't load the
    # command, instead it returns the index entry which should
    # be used.
    proc findIndexEntry {cmd {namespace ""}} {
	global auto_index auto_path

	if {[string length $namespace] == 0} {
	    set namespace [uplevel {namespace current}]
	}
	set nameList [auto_qualify $cmd $namespace]
	# workaround non canonical auto_index entries that might be around
	# from older auto_mkindex versions
	lappend nameList $cmd
	foreach name $nameList {
	    if {[info exists auto_index($name)]} {
		return $auto_index($name)
	    }
	}
	if {![info exists auto_path]} {
	    return 0
	}

	if {![auto_load_index]} {
	    return 0
	}

	foreach name $nameList {
	    if {[info exists auto_index($name)]} {
		return $auto_index($name)
	    }
	}
	return ""
    }
    
    # We do not want to have auto_reset destroy the core Alphatk procedures,
    # so we use this modified version.
    proc auto_reset {} {
	global auto_execs auto_index auto_oldpath
	catch {unset auto_execs}
	catch {unset auto_index}
	catch {unset auto_oldpath}
    }

    # auto_mkindex:
    # Regenerate a tclIndex file from Tcl source files.  Takes two arguments:
    # the name of the directory in which the tclIndex file is to be placed,
    # and a glob pattern to use in that directory to locate all of the relevant
    # files.  For Alpha's core files we cannot use the standard Tcl 8
    # 'auto_mkindex' because it sources the files in question, and many of
    # Alpha's files have nasty side-effects when sourced (e.g. AlphaBits.tcl!)
    #
    # We could look into using 'auto_mkindex_old', but this version here provides
    # much better error reporting...
    proc auto_mkindex {dir {files *.tcl}} {
	# Due to some peculiarities with current working directories
	# under some MacOS/HFS+/other conditions, we avoid using
	# 'cd' and 'pwd' explicitly if possible.
	set dir [file nativename $dir]
	global tcl_platform
	switch -- $tcl_platform(platform) {
	    "macintosh" {
		if {$dir == ":" || $dir == "."} {
		    set dir [pwd]
		}
	    }
	    default {
		if {$dir == "."} {
		    set dir [pwd]
		}
	    }
	}
	# So we can handle relative path names
	if {[file pathtype $dir] == "relative"} {
	    set dir [file join [pwd] $dir]
	}
	if {![catchNoClobber {file readlink $dir} _root]} {
	    set dir $_root
	}
	set dir [string trim $dir :]
	# This line is very important, or Tcl will reject the file...
	append index "# Tcl autoload index file, version 2.0\n"
	
	set cid [scancontext create]
	# This pattern is used to extract procedures when the 'scanfile'
	# command is used below.  We don't do anything too dramatic if
	# the procedure name can't be extracted.  The most likely cause
	# is a garbled file.
	scanmatch $cid "^\[ \t\]*proc\[ \t\]" {
	    if {[regexp "^\[ \t\]*proc\[ \t\]+((\"\[^\"\]+\")|(\{\[^\}\]+\})|(\[^ \t\]*))" \
	      $matchInfo(line) match procName]} {
		set procName [lindex [auto_qualify $procName "::"] 0]
		append index "set [list auto_index($procName)]"
		append index " \[list source \[file join \$dir [list [file tail $file]]\]\]\n"
	    } else {
		message "Couldn't extract a proc from '$matchInfo(line)'!"
	    }
	}
	foreach file [glob -dir $dir -- $files] {
	    watchCursor
	    set f ""
	    message [file tail $file]
	    if {[catch {open $file r} fid]} {
		lappend errors $fid
		lappend errorFiles $file
	    } else {
		if {[catch {scanfile $cid $fid} err]} {
		    lappend errors $err
		    lappend errorFiles $file
		}
		close $fid
	    }
	}
	
	scancontext delete $cid
	
	if {[info exists errors]} {
	    if {[dialog::yesno -y "View the error" -n "Continue" \
	      "The following files: [join $errorFiles ,] were unable\
	      to be opened or scanned for procedures to store in Tcl index\
	      files.  This is a serious error.  Alpha will not be\
	      able to find procedures stored in those files, and will\
	      therefore fail to function correctly.  You should\
	      ascertain the cause of these\
	      problems and fix them.  Your disk may be damaged.\r\
	      To avoid some of these problems, the Tcl index file\
	      in $dir will not be replaced."]} {
		dialog::alert [join $errors "\r"]
	    }
	} else {
	    if {[catch {open [file join $dir tclIndex] w} fid]} {
		if {[file exists [file join $dir tclIndex]] \
		  && ![file writable $dir]} {
		    # This is a read-only directory, so there isn't
		    # a problem that we couldn't write to it.  Probably
		    # it's a system directory such as the base Tcl library.
		    message "'$dir' is read-only, so I'll use the existing Tcl index."
		} else {
		    dialog::alert "The Tcl index file in $dir could not\
		      be rewritten.  Perhaps the file is locked or read-only?\
		      The old index will be left intact, but you should fix\
		      this problem so Alpha can index new files in\
		      this directory."
		}
	    } else {
		if {[catch {puts -nonewline $fid $index} err]} {
		    if {[dialog::yesno -y "View the error" -n "Continue" \
		      "The Tcl index file in $dir was successfully opened,\
		      but Alpha encountered an error while writing to the\
		      file.  This is a very serious problem, and Alpha will\
		      probably no longer function correctly.  At the very\
		      least you will need to reinstall that directory, and\
		      perhaps all of Alpha."]} {
			dialog::alert $err
		    }
		}
		catch {close $fid}
	    }
	}
	
    }

}



if {[info tclversion] < 8.0} {
    proc ensureNamespaceExists {cmd} {}
    proc namespace_exists {ns} {
	return [expr {[llength [info commands ${ns}::*]] > 0}]
    }
} else {
    proc ensureNamespaceExists {cmd} {
	set ns ""
	while {[regexp "^((::)?$ns\[a-zA-Z_\]+::)" $cmd ns]} {
	    namespace eval $ns {}
	}
    }
    if {[info tclversion] <= 8.3} {
	proc namespace_exists {ns} {
	    if {![catch {namespace children ::$ns}]} {
		return 1
	    } else {
		return 0
	    }
	}
    } else {
	# Vince's patch is in Tcl 8.4, so we have 'namespace exists'.
	proc namespace_exists {ns} {
	    uplevel 1 [list namespace exists $ns]
	}
    }
}

proc alpha::ensureAutoPathOk {} {
    global HOME tcl_platform
    if {[info exists tcl_platform(isWrapped)]} {
	return
    }
    if {![file exists $HOME]} {
	global alpha::platform
	alertnote "Alpha's home directory '$HOME' does not seem to exist. This\
	  must be found."
	while {1} {
	    if {[catch {get_directory -p "Where is Alpha's home directory"} new_home]} {
		return
	    }
	    if {[file exists [file join $new_home Tcl]]} {
		set HOME $new_home
		break
	    }
	    # Probably running on Alphatk
	    if {[file exists [file join $new_home Alpha Tcl]]} {
		set HOME [file join $new_home Alpha]
		break
	    }
	    if {${alpha::platform} == "alpha"} {
		alertnote "That didn't seem to be Alpha's home directory.\
		  The home directory must contain the Alpha application and\
		  the 'Tcl' subdirectory."
	    } else {
		alertnote "That didn't seem to be Alpha's home directory.\
		  The home directory must contain alphatk and \
		  the 'Alpha' subdirectory."
	    }
	}
	# Remove anything which has gone from the auto_path
	set new_auto_path {}
	foreach dir $auto_path {
	    if {[file exists $dir]} {
		lappend new_auto_path $dir
	    }
	}
	set auto_path $new_auto_path
	unset new_auto_path
    }
}

proc alpha::makeAutoPath {{check_dups 1} {skipPrefs 0}} {
    global HOME auto_path
    if {$check_dups} {
	set lcmd lunion
    } else {
	set lcmd lappend
    }
    alpha::ensureAutoPathOk
    set root [file join $HOME Tcl]
    if {![catchNoClobber {file readlink $root} _root]} {
	set root $_root
    }
    
    foreach dir {SystemCode Modes Menus Completions Packages UserModifications} {
	if {($dir == "Packages") || ($dir == "UserModifications")} {
	    if {$skipPrefs} {
		# If we're skipping preferences these two folders
		# are not added to the auto path.
		continue
	    }
	}
	$lcmd auto_path [file join $root $dir]
	foreach d [glob -types d -nocomplain -dir [file join $root $dir] *] {
	    $lcmd auto_path $d
	}
    }
    if {[file exists [file join $HOME Tclextensions]]} {
	$lcmd auto_path [file join $HOME Tclextensions]
    }
}

proc rebuildTclIndices {} {
    global auto_path HOME
    # Make sure nothing weird has happened.
    alpha::ensureAutoPathOk
    foreach dir $auto_path {
	# if directory exists
	if {[file isdirectory $dir]} {
	    # if there are any files
	    if {![catch {glob -dir $dir *.*tcl} err]} {
		if {[file writable $dir]} {
		    message "Building [file tail $dir] index"
		    # use 'catch' also in case directory is write-protected
		    if {[catch {auto_mkindex $dir *.*tcl} err]} {
			message "Problem rebuilding directory $dir : $err"
		    }
		} else {
		    message "Skipping rebuilding [file tail $dir] index\
		      because directory is not writable"
		}
	    } else {
		message "Directory '$dir' contains no Tcl files!"
	    }
	} else {
	    message "Directory '$dir' doesn't appear to exist."
	}
    }
    # make alpha forget its old information so the new stuff is loaded
    # when required.
    catch {auto_reset}
    message "Tcl indices rebuilt."
}

# 'exit' kills Alpha without allowing it to save etc.
# 'quit' handles a smooth shutdown for us
if {[info commands exit] != ""} {
    rename exit ""
    proc exit {{returnCode ""}} {quit}
}
