## -*-Tcl-*- (nowrap)
 # ==========================================================================
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "marks.tcl"
 #                                    created: 06/04/99 {09:12:06 am} 
 #                                last update: 11/14/2001 {18:16:41 PM}
 #
 #  Marks for front window.
 # ==========================================================================
 ##

#  Search Menu Procs  #

proc gotoFileMark {} {
    set text [getSelect]
    if {[string length $text] && ([string length $text] < 32)} {
	gotoMark [listpick -p "Go to Mark :" -L [list $text] [getNamedMarks -n]]
    } else {
	gotoMark [listpick -p "Go to Mark :" [getNamedMarks -n]]
    }
}

proc gotoFunc {} {
    set l [parseFuncsAlpha]
    if {[set ind [lsearch -exact $l "\(-"]] >= 0} {
	array set pos [lrange $l [expr {$ind + 2}] end]
    } else {
	array set pos $l
    }
    set res [listpick -p "Go to Function :" [lsort [array names pos]]]
    goto $pos($res)
}

proc namedMarkProc {menu item} {
    switch -- $item {
	"sortAlphabetically"    {sortMarksFile}
	"sortByPosition"        {orderMarks}
	default                 {$item}
    }
}

proc unnamedMarkproc {menu item} {
    switch -- $item {
	"setUnnamedMark"        {setMark}
	default                 {$item}
    }
}

proc findViaFileMarks {dir} {
    set pos [getPos] 
    set markAbovePos ""
    set markBelowPos ""
    
    set nm [getNamedMarks]
    foreach n $nm {
	set posOf_n [lindex $n 3]
	if {[pos::compare $posOf_n < $pos]} {
	    set markAbovePos [lindex $n 0]
	} elseif {[pos::compare $posOf_n == $pos]} {
	    continue 
	} else {
	    set markBelowPos [lindex $n 0]
	    break
	}
    }
    
    if {$dir} {
	if {$markBelowPos != ""} {
	    gotoMark $markBelowPos 
	}
    } else {
	if {$markAbovePos != ""} {
	    gotoMark $markAbovePos 
	}
    }
}

# ===========================================================================
# 
#  ----  #
# 
#  Named Mark Procs  #
#       

proc buildMarksMenu {} {
    lappend items "Mark File" "Set Mark" "Remove Mark" "Clear Marks" \
      "Float Marks" "(-)"
    eval lappend items [getNamedMarks -n]
    return [list build $items {marksMenuProc -m -c} {}]
}

proc markTrim  {markName {limit 31}} {
    if {[info tclversion] < 8.0} {
        regsub -all "\{" $markName "(" markName
        regsub -all "\}" $markName ")" markName
    }
    if {[string length $markName] > $limit} {
        set markName "[string range $markName 0 [expr {$limit - 1}]]"
    }
    return $markName
}

proc marksMenuProc {menu item} {
    # Note menus with '-c' should in the future not strip ellipses, so
    # we should remove the workarounds here as soon as Alpha 8 is fixed
    # in that regard.
    switch -- $item {
	"Mark File" {markFile}
	"Set Mark" - "Set Mark" {setNamedMark}
	"Remove Mark" - "Remove Mark" {removeNamedMark -w [win::Current]}
	"Clear Marks" {clearFileMarks}
	"Float Marks" {floatNamedMarks}
	default       {
	    if {[catch {gotoMark $item}]} {
		gotoMark "${item}"
	    }
	}
    }
}

proc markFile {} {
    
    set refloat [unfloatNamedMarks]
    if {[llength [getNamedMarks -n]]} {
	removeAllMarks "" 1
    }
    mode::proc MarkFile
    if {$refloat} {floatNamedMarks}
}

proc ::MarkFile {} {
    global mode 
    status::msg "\"$mode\" mode does not support file marking."
}

proc displayNamedMarks {} {
    
    global mode
    
    # Return all mark names in a new window.
    set    result "\rCurrent named marks for \"[win::CurrentTail]\" :\r\r"
    append result [join [getNamedMarks -n] \r]
    new -n "* [win::CurrentTail] Marks *" -m $mode -info $result
    shrinkWindow 1
}

proc floatNamedMarks {} {
    global windowMarks windowMarksMenu defTop defWidth

    set w1 [list [win::Current]]
    set w2 [list [win::CurrentTail]]

    if {![llength [getNamedMarks]]} {
	set msg "No marks found in \"[lindex $w2 0]\"."
    }
    if {[info exists msg]} {beep ; status::msg $msg ; return}

    # Create a floating menu with mark names.
    if {[win::IsFile [lindex $w1 0]]} {
	set windowMarks $w1
    } else {
	set windowMarks $w2
    }
    eval [list lappend windowMarks "Re-Mark File" "-"] [getNamedMarks -n]
    set  windowMarks [lremove -all -regexp $windowMarks {^mark[0-9]+$}]
    Menu -n "Marks" -p goToWindowMark -c -m [lrange $windowMarks 1 end]
    # Determine the proper width for the floating menu.
    set width 125
    foreach item [lrange $windowMarks 1 end] {
	set newWidth [expr [string length $item] * 7]
	if {[expr $newWidth > $width]} {set width $newWidth}
    }
    if {$width > 280} {set width 280}
    # Float the menu.
    set windowMarksMenu \
      [float -m "Marks" -w $width -t $defTop -l [expr $defWidth + 20]]
}

proc unfloatNamedMarks {{refloat 1}} {
    global windowMarks windowMarksMenu 
    
    set wm  [info exists windowMarks] 
    set wmm [info exists windowMarksMenu] 
    if {!$wm || !$wmm} {
	return 0
    } elseif {[win::CurrentTail] != [file tail [lindex $windowMarks 0]]} {
	return 0
    } else {
	return [unfloat $windowMarksMenu]
    }
}

proc clearFileMarks {} {removeAllMarks}

proc removeAllMarks {{pat ""} {quietly 0}} {
    
    if {$pat == ""} {set pat *}
    set win [win::Current]
    if {[catch {foreach mark [getNamedMarks -n] {
	if {[string match $pat $mark]} {removeNamedMark -w $win -n $mark}}}]
    } {
	# some marks contain curly braces!
	# (This will be fixed in Alpha8)
	foreach mark [quote::Regfind [getNamedMarks -n]] {
	    if {[string match $pat $mark]} {
		removeNamedMark -w $win -n $mark
	    }
	    if {[string index $mark 0] == "\{"} {
		set mk [string range $mark 1 [expr {[string length $mark] -1}]]
	    }
	    if {[string match $pat $mark]} {
		removeNamedMark -n $mark -w $win
	    }
	}
	
    }
    if {!$quietly} {
	status::msg "All marks for \"[win::CurrentTail]\" have been removed."
    } 
    return [unfloatNamedMarks]
}

proc sortMarksFile {{ignoreCase 1}} {
    set marks    [getNamedMarks]
    set question "Really sort all marks alphabetically?"
    if {![llength $marks]} {
	status::msg "No marks found in \"[win::CurrentTail]\"."
    } else {
	set refloat [unfloatNamedMarks]
	set w       [win::Current]
	set wCT     [win::CurrentTail]
	foreach mark $marks {
	    removeNamedMark -n [lindex $mark 0] -w $w
	    lappend marks2 $mark
	}
	if {$ignoreCase} {
	    set marks2 [lsort -ignore $marks2]
	} else {
	    set marks2 [lsort $marks2]
	}
	foreach mark $marks2 {
	    set name [lindex $mark 0]
	    set disp [lindex $mark 2]
	    set pos  [lindex $mark 3]
	    set end  [lindex $mark 4]
	    setNamedMark $name $disp $pos $end
	}
	if {$refloat} {floatNamedMarks}
	status::msg "All named marks for \"$wCT\" have been sorted alphabetically."
    }
}

proc orderMarks {} {
    set marks    [getNamedMarks]
    set question "Really re-order all marks by position?"
    if {![llength $marks]} {
	status::msg "No marks found in \"[win::CurrentTail]\"."
    } else {
	set refloat [unfloatNamedMarks]
	set w       [win::Current]
	set wCT     [win::CurrentTail]
	foreach mark $marks {
	    removeNamedMark -n [lindex $mark 0] -w $w
	    set name   [lindex $mark 0]
	    set disp   [lindex $mark 2]
	    set pos    [lindex $mark 3]
	    set end    [lindex $mark 4]
	    set pos    [zeroadd $pos]
	    set mark2  [list $pos $disp $name $end]
	    lappend marks2 $mark2
	}
	foreach mark [lsort -ignore $marks2] {
	    set name [lindex $mark 2]
	    set disp [lindex $mark 1]
	    set pos  [lindex $mark 0]
	    set end  [lindex $mark 3]
	    setNamedMark $name $disp $pos $end
	}
	if {$refloat} {floatNamedMarks}
	status::msg "All named marks for \"$wCT\" have been sorted by position."
    }
}

# From Mark Nagata.  Once we have Tcl 8, we can get rid of this and use
# 'lsort -index 0 -dictionary' above.

if {${alpha::platform} == "alpha"} {
    proc zeroadd {num} {return [format "%0[string length [maxPos]]d" $num]}
} else {
    proc zeroadd {num} {return $num}
}

proc goToWindowMark {menuName item} {
    
    global win::Active windowMarks
    
    if {[llength ${win::Active}] && $item != "Re-Mark File"} {
	set pB 1 ; placeBookmark
    } else {
	set pB 0
    }
    set w [lindex $windowMarks 0]
    if {[catch {win::OpenQuietly $w}]} {
	set msg "Couldn't find the window \"$w\""
	status::msg "Error: $msg" ; error $msg
    }
    if {[icon -q]} {icon -o} 
    if {$item == "Re-Mark File"} {markFile ; return}
    if {$pB} {status::msg "Press <Ctrl-.> to return to original position"}
    if {[catch {gotoMark "$item"}]} {
	if {[info tclversion] < 8.0 && [string length $item] > 24} {
	    set msg "this mark's name is longer than 24 characters."
	} else {
	    set msg "couldn't find the mark \"$item\"."
	}
	beep ; status::msg "Sorry, $msg"
    } 
}

proc editMark {fname mname args} {
    getOpts
    # Try to open the window.
    if {[catch {win::OpenQuietly $fname} window]} {
	status::errorMsg "Couldn't find the window '$fname'."
    }
    # First find out if we have any marks at all ...
    if {![llength [set mNames [getNamedMarks -n]]]} {
	catch {mode::proc MarkFile}
	set mNames [getNamedMarks -n]
    }
    # Now try to find the mark.
    set result 0
    if {[set index [lsearch $mNames "${mname}"]] >= 0} {
	# Found the exact mark name.
	gotoMark [lindex $mNames $index]
	set result 1
    } elseif {[set index [lsearch $mNames "*${mname}*"]] >= 0} {
	# Found a mark name that seems to resemble what we're looking for.
	gotoMark [lindex $mNames $index]
	set result 1
    } 
    # If we still didn't find anything ...
    if {!$result} {
	set pat1 [quote::Regfind $mname]
	set pat2 [string trimright [string trimright $pat1] ]
	if {![catch {search -s -f 1 -r 1 -i 0 $pat1 [minPos]} match]} {
	    # Found something in the file at least.
	    goto [lindex $match 0]
	    set result 1
	} elseif {![catch {search -s -f 1 -r 1 -i 0 $pat2 [minPos]} match]} {
	    # Found something in the file at least.
	    # (Mark was probably truncated using 'win::MakeTitle'.)
	    goto [lindex $match 0]
	    set result 1
	}
    } 
    # Finish up.
    if {[info exists opts(-r)]} {winReadOnly}
    return $result
}

# ===========================================================================
# 
#  ----  #
# 
#  Parse Funcs  #
# 
# Used to create a popup of all funcs in window.  Routine should return
# list containing, consecutively, proc name and start of definition.
# 

proc buildParseMenu {} {
    global currentParseItems
    
    set currentParseItems [parseFuncsAlpha]
    
    set items [list]
    set skip 0
    foreach item $currentParseItems {
	if {!$skip} {
	    lappend items $item
	} 
	set skip [expr {!$skip}]
    }
    
    return [list build $items {parseMenuProc -m -c} {}]
}

proc parseMenuProc {menu item} {
    global currentParseItems
    
    set num [lsearch -exact $currentParseItems $item]
    set pos [lindex $currentParseItems [expr {$num + 1}]]
    
    if {$pos < 0} {
        optClickTB_Pick $item
    } else {
	goto $pos
    }
}

proc parseFuncsAlpha {} {mode::proc parseFuncs}

proc ::parseFuncs {} {
    global sortFuncsMenu funcExpr parseExpr mode
    if {![info exists funcExpr] || ! [info exists parseExpr]} {
	# Give an informative error message
	set msg "\"$mode\" mode does not support function parsing."
	status::msg $msg ; error $msg
    }
    set pos [minPos]
    set m {}
    if {$sortFuncsMenu} {
	while {[set res [search -s -f 1 -r 1 -i 0 -n $funcExpr $pos]] != ""} {
	    if {[regexp -- $parseExpr [eval getText $res] dummy word]} {
		lappend m [list $word [lindex $res 0]]
	    }
	    set pos [lindex $res 1]
	}
	set m [eval concat [lsort -ignore $m]]
    } else {
	while {[set res [search -s -f 1 -r 1 -i 0 -n $funcExpr $pos]] != ""} {
	    if {[regexp -- $parseExpr [eval getText $res] dummy word]} {
		lappend m $word [lindex $res 0]
	    }
	    set pos [lindex $res 1]
	}
    }
    return $m
}

