#  AlphaTcl - core Tcl engine
# Menu creation procs
	
namespace eval menu {}
namespace eval global {}
namespace eval file {}

proc menu::buildBasic {} {
    # These are built on the fly
    Menu -n File -p menu::generalProc {}
    Menu -n Edit -p menu::generalProc {}
    Menu -n Text -p menu::generalProc {}
    Menu -n Search {}
    Menu -n Utils {}
    Menu -n Config {}
    
    insertMenu File Edit Text Search Utils Config
    
    help::buildMenu
}

proc global::listAllBindings {} {
    new -n {* All Key Bindings *} -m Tcl -info [bindingList]
}

proc global::listGlobalBindings {} {
    set text ""
    set tmp [mode::listAll]
    foreach b [split [bindingList] "\r"] {
	set lst [lindex [split $b  " "] end]
	if {[lsearch $tmp $lst] < 18} {
	    append text "$b\r"
	}
    }
    new -n {* Global Key Bindings *} -m Tcl -info $text
}

## 
 #  ------------------------------------------------------------------------
 # 
 # "global::listPackages" --
 # 
 # Creates the file "Packages", saves it in the Help folder (over-writing
 # if necessary.)  Includes hyperlinks to any available package help. 
 # ------------------------------------------------------------------------
 ##
proc global::listPackages {} {
    global HOME index::feature timeStampStyle index::maintainer \
      alpha::packageRequirementsFailed
    cache::readContents index::maintainer
    foreach i [array names index::maintainer] {
	set j [lindex [set index::maintainer($i)] 1]
	set au($i) "[lindex $j 0], [lindex $j 1]"
    }
    foreach p [lsort -ignore [array names index::feature]] {
	set v [alpha::package versions $p]
	if {[lindex $v 0] == "for"} {
	    set v "for [lindex $v 1] [lindex $v 2]"
	}
	if {[lcontains alpha::packageRequirementsFailed $p]} {
	    append tbad "\r[format {  %-30s %-10s } \
	      [concat package: $p] $v]"
	    if {[info exists au($p)]} { append tbad $au($p) }
	    set requires [lindex [alpha::package requirements $p] 1]
	    catch {uplevel \#0 $requires} res
	    append tbad "\r     $res"
	    continue
	}
	switch -- [lindex [set index::feature($p)] 2] {
	    "1" {
		if {[lindex [alpha::package versions $p] 0] != "for"} {
		    # Usual Menus (tm1)
		    append tm1 "\r[format {  %-30s %-10s } \
		      [concat package: $p] $v]"
		    if {[info exists au($p)]} { append tm1 $au($p) }
		} else {
		    # Other possible packages (tm2)
		    append tm2 "\r[format {  %-30s %-10s } \
		      [concat package: $p] $v]"
		    if {[info exists au($p)]} { append tm2 $au($p) }
		}
	    }
	    "0" {
		# Features (tp1), enabled through "Preferences -> Features"
		append tp1 "\r[format {%s %-30s %-10s } \
		  [package::active $p { { }}] [concat package: $p] $v]"
		if {[info exists au($p)]} { append tp1 $au($p) }
	    }
	    "2" {
		# Features (tp2), enabled as flag preferences
		append tp2 "\r[format {%s %-42s %-10s } \
		  [package::active $p { { }}] [concat package: $p] $v]"
		if {[info exists au($p)]} { append tp2 $au($p) }
	    }
	    "-1" {
		# Auto-loading features (ta)
		append ta "\r[format {  %-30s %-10s  } \
		  [concat package: $p] $v]"
		if {[info exists au($p)]} { append ta $au($p) }
	    }
	}
    }
    # Is the current window "Packages" ?  If so, close it, so that it can
    # be over-written.
    if {[win::CurrentTail] == "Packages"} {
	killWindow
    }
    # Create a new file, insert title information at the top.
    new -n {Packages} -m Text
    set created [mtime [now] $timeStampStyle]
    set alphaV  [alpha::package versions Alpha]
    set alphatclV  [alpha::package versions AlphaTcl]
    append t "\rCurrently installed packages, "
    global alpha::platform
    if {[set alpha::platform] == "alpha"} {
	append t "Alpha v $alphaV, AlphaTcl $alphatclV\r"
    } else {
	append t "Alphatk v $alphaV, AlphaTcl $alphatclV\r"
    }
    append t "as of $created\r\r"
    append t "\rTo update this file, use the "
    append t "\"Config -> List Packages\" menu item.\r\r"
    append t "columns are: name, version, and maintainer\r"
    insertText "$t \r" ; set t ""
    # Modes
    append t "\r\t  \tModes:\r\r"
    append t "\rSee also \"Examples Help\" for example syntax files.\r\r"
    foreach p [lsort -ignore [alpha::package names -mode]] {
	# put version numbers back
	set v [alpha::package versions $p]
	append t "\r[format {  %-16s %-8s  } [concat package: $p] $v]"
	if {[info exists au($p)]} {append t $au($p)}
    }
    insertText "$t \r" ; set t ""
    #     Mode Specific Completion Tutorials
    append t "\r\r\t  \t \tMode Specific Completion Tutorials:\r\r"
    set td [glob -dir [file join $HOME Tcl Completions] *Tutorial*]
    foreach tFile $td {
	append t "\r    \"[file tail $tFile]\""
    }
    insertText "$t \r" ; set t ""
    # Menus
    append t "\r\r\t  \tMenus:\r\r"
    #     Usual Menus
    append t "\r\"Usual menus\" are designed to be used globally.\r\r"
    if {[info exists tm1]} {append t "$tm1 \r" ; unset tm1}
    #     Other Possible Menus
    append t "\r\r\"Other possible menus\" are designed for specific modes.\r\r"
    if {[info exists tm2]} {append t  $tm2 ; unset tm2}
    insertText "$t \r" ; set t ""
    # Features
    append t "\r\r\t  \tFeatures:\r\r"
    append t "\r  '' = active as of $created\r"
    #     Enabled via "Config -> Preferences -> Features"
    append t "\r\rThese are enabled through "
    append t "\"Config -> Preferences -> Features\"\r\r"
    if {[info exists tp1]} {append t $tp1 ; unset tp1}
    insertText "$t \r" ; set t ""
    #     Enabled as flag preferences
    append t "\r\rThese are enabled through "
    append t "\"Interface\" or \"Input-Output\" Preferences\r\r"
    if {[info exists tp2]} {append t $tp2 ; unset tp2}
    insertText "$t \r"; set t ""
    # Auto-loading features
    append t "\r\r\t  \tAuto-loading features:\r\r"
    if {[info exists ta]} {append t $ta ; unset ta}
    insertText "$t \r\r" ; set t ""
    # Environment
    append t "\r\t  \tEnvironment:\r\r"
    insertText "[global::listEnvironment] \r\r" ; set t ""
    # Incompatible packages
    if {[info exists tbad]} {
	append t "\r\t  \tIncompatible packages:\r\r"
	append t $tbad ; unset tbad
	insertText "$t \r\r" ; set t ""
    }
    # Hyperize, color and mark file
    goto [minPos]
    help::hyperiseEmailAddresses
    help::hyperiseUrls
    help::colourHeadingsEtc
    # remove the strings "package: "
    global alpha::platform
    if {${alpha::platform} == "alpha"} {
	set pos [minPos]
	while {[llength [set range [search -s -n "package: " $pos]]]} {
	    set pos [lindex $range 1]
	    replaceText [lindex $range 0] $pos ""
	}
    }
    # Overwrite any existing "Packages" file in the Help directory
    setWinInfo tabsize 4
    setWinInfo state mpw
    global backup
    set oldBackup $backup
    set backup 0
    saveAs -f [file join $HOME Help Packages]
    set backup $oldBackup
    winReadOnly
    catch {unset index::maintainer}
}

proc global::listEnvironment {} {
    global alpha::defaultEncoding
    append t "\r[format {  %-30s %-10s  }\
      Tcl-version [info patchlevel]]"
    append t "\r[format {  %-30s %-10s  }\
      "system encoding" [encoding system]]"
    append t "\r[format {  %-30s %-10s  }\
      "AlphaTcl encoding" ${alpha::defaultEncoding}]"
    append t "\r"
    if {[info tclversion] >= 8.0} {
	foreach pkg [info loaded] {
	    set pkgName [lindex $pkg 1]
	    append t "\r[format {  %-30s %-10s  } $pkgName loaded]"
	}
    }
    return $t
}

proc global::listFunctions {} {
    new -n {* Functions *} -m Tcl -info \
      "===\r\tCommand-double-click on a function to see its\
      definition\r===\r\r[join [lsort -ignore [info commands]] \r]\r"
}

proc global::menus {} {global::menusAndFeatures 1}
proc global::features {} {global::menusAndFeatures 2}
proc global::menusAndFeatures {{mfb 0}} {
    global global::features
    dialog::pickMenusAndFeatures global $mfb
}

proc global::insertAllMenus {} {
    global global::features index::feature
    foreach m ${global::features} {
	if {[lindex [set index::feature($m)] 2] == 1} {
	    global $m
	    insertMenu [set $m]
	}
    }
}

proc global::rebuildPackageIndices {} {
    if {[dialog::yesno "You must quit Alpha immediately after rebuilding. \
      Proceed?"]} {
	alpha::rebuildPackageIndices
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::buildProc" --
 # 
 #  Register a procedure to be the 'build proc' for a given menu.  This
 #  procedure can do one of two things:
 #  
 #  i) build the entire menu, including evaluating the 'menu ...' command.
 #  In this case the build proc should return anything which doesn't
 #  begin 'build ...'
 #  
 #  ii) build up part of the menu, and then allow pre-registered menu
 #  insertions/replacements to take-effect.  In this case the procedure
 #  should return a list of the items (listed by index):
 #  
 #  0: "build"
 #  1: list-of-items-in-the-menu
 #  2: list of other flags.  If the list doesn't contain '-p', we use
 #  the standard menu::generalProc procedure.  If it does contain '-p'
 #  general prmenu procedure to call when an item is selected.  
 #  If nothing is given,
 #  or if '-1' is given, then we don't have a procedure.  If "" is given,
 #  we use the standard 'menu::generalProc' procedure.  Else we use the
 #  given procedure.
 #  3: list of submenus which need building.
 #  4: over-ride for the name of the menu.
 #  
 #  You must register the build-proc before attempting to build the menu.
 #  Once registered, any call of 'menu::buildSome name' will build your
 #  menu.
 # -------------------------------------------------------------------------
 ##
proc menu::buildProc {name proc {postEval ""}} {
    global menu::build_procs menu::posteval
    set menu::build_procs($name) $proc
    if {[string length $postEval]} {
	set menu::posteval($name) $postEval
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::insert" --
 # 
 #  name, type, where, then list of items.  type = 'items' 'submenu'
 #  
 #  Add given items to a given menu, provided they are not already there.
 #  Rebuild that menu if necessary.
 #  
 #  There are also procs 'menu::removeFrom' which does the opposite of
 #  this one, and 'menu::replaceWith' which replaces a given menu item
 #  with others.
 # -------------------------------------------------------------------------
 ##
proc menu::insert {name args} {
    if {[llength $args] < 3} { error "Too few args to menu::insert" }
    global menu::additions alpha::guiNotReady
    if {[info exists menu::additions($name)]} {
	set a [set menu::additions($name)]
	if {[lsearch -exact $a $args] != -1} { 
	    return 
	}
	# check if it's there but in a different place; we over-ride
	set dblchk [lreplace $args 1 1 "*"]
	if {[set i [lsearch -glob $a $dblchk]] == -1} {
	    unset i
	}
    }
    if {[info exists i]} {
	set menu::additions($name) [lreplace $a $i $i $args]
    } else {
	lappend menu::additions($name) $args
    }
    if {![info exists alpha::guiNotReady]} {
	# we were called after start-up; build the menu now
	menu::buildSome $name
    } else {
	hook::register startupHook "menu::buildSome [list $name]"
    }
}

proc menu::uninsert {name args} {
    global menu::additions alpha::guiNotReady
    set a [set menu::additions($name)]
    if {[set idx [lsearch -exact $a $args]] == -1} { 
	return 
    }
    set menu::additions($name) [lreplace $a $idx $idx]
    if {![info exists alpha::guiNotReady]} {
	# we were called after start-up; build the menu now
	menu::buildSome $name
    } else {
	hook::register startupHook "menu::buildSome [list $name]"
    }
}

proc alpha::buildMainMenus {} {
    # removed in 7.4
    #menu::buildProc internetUpdates package::makeUpdateMenu
    menu::buildProc packages menu::packagesBuild
    menu::buildProc mode menu::modeBuild
    menu::buildProc preferences menu::preferencesBuild
    uplevel #0 {
	source [file join $HOME Tcl SystemCode alphaMenus.tcl]
	menu::buildSome "File" "Edit" "Text" "Search" "Utils" "Config"
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::buildSome" --
 # 
 #  Important procedure which builds all known/registered menus from a
 #  number of pieces.  It allows the inclusion of menus pieces registered
 #  with the menu::insert procedure, which allows you easily to add items
 #  (including dynamic and hierarchial) to any of Alpha's menus.
 # 
 # Results:
 #  Various menus are (re)built
 # 
 # Side effects:
 #  Items added to those menus with 'addMenuItem' will vanish.
 # 
 # --Version--Author------------------Changes-------------------------------
 #    1.0     <vince@santafe.edu> original
 #    2.0     <vince@santafe.edu> more compact, more like tk
 # -------------------------------------------------------------------------
 ##
proc menu::buildSome {args} {
    set msubs {}
    foreach token $args {
	eval lappend msubs [menu::buildOne $token]
    }
    # build sub-menus of those built
    if {[llength $msubs]} {eval menu::buildSome $msubs}
    foreach token $args {
	hook::callAll menuBuild $token
    }
}

proc menu::buildOne {args} {
    global menu::additions menu::build_procs alpha::guiNotReady \
      menu::items
    set token [lindex $args 0] ; set args [lrange $args 1 end]
    set len [llength $args]
    if {$len > 0 || [info exists menu::build_procs($token)]} {
	if {$len > 0} {
	    set res $args
	} else {
	    if {[catch "[set menu::build_procs($token)]" res]} {
		alpha::reportError "The menu $token had a problem\
		  starting up ; $res"
		return ""
	    }
	}
	switch -- [lindex $res 0] {
	    "build" {
		set ma [lindex $res 1]
		if {[llength $res] > 2} {
		    set theotherflags [lrange [lindex $res 2] 1 end]
		    if {[lindex [lindex $res 2] 0] != -1} {
			set mproc [lindex [lindex $res 2] 0]
		    }
		    if {[lindex $res 3] != ""} {
			eval lappend msubs [lindex $res 3]
		    }
		    if {[lindex $res 4] != ""} { set name [lindex $res 4] }
		}
	    } "menu" - "Menu" {
		eval $res
		menu::postEval $token
		return ""
	    } default {
		menu::postEval $token
		return ""
	    }
	}
    } else {
	set ma ""
	if {[info exists menu::items($token)]} {
	    set ma [set menu::items($token)]
	    global menu::proc menu::which_subs menu::otherflags
	    if {[info exists menu::proc($token)]} {
		set mproc [set menu::proc($token)]
	    }
	    if {[info exists menu::which_subs($token)]} {
		eval lappend msubs [set menu::which_subs($token)]
	    }
	    if {[info exists menu::otherflags($token)]} {
		set theotherflags [set menu::otherflags($token)]
	    }
	}
    }

    if {![info exists name]} { set name $token }
    # add any registered items and make the menu contents
    if {[info exists menu::additions($token)]} {
	foreach ins [set menu::additions($token)] {
	    set where [lindex $ins 1]
	    set type [lindex $ins 0]
	    set ins [lrange $ins 2 end]
	    switch -- $type {
		"submenu" {
		    lappend msubs [lindex $ins 0]
		    # 'ins' may be just a menu name, or also contain various
		    # additional flags (-p proc etc)
		    set ins [list [concat Menu -n $ins [list {}]]]
		}
	    }
	    switch -- [lindex $where 0] {
		"replace" {
		    set old [lindex $where 1]
		    if {[set ix [eval llindex ma $old]] != -1} {
			set ma [eval [list lreplace $ma \
			  $ix [expr {$ix -1 + [llength $old]}]] $ins]
		    } else {
			alertnote "Bad menu::replacement registered '$old'"
		    }
		    
		}
		"end" {
		    eval lappend ma $ins
		}
		default {
		    if {![is::UnsignedInteger $where]} {
			if {[set pos [lsearch -exact $ma $where]] != -1} {
			    set where $pos
			} else {
			    alertnote "The string '$where' has not be found\
				       in menu '$name'. '$ins' will be put at\
				       the end of this menu"
			    set where [llength $ma]
			}
		    }
		    set ma [eval linsert [list $ma] $where $ins]
		}
	    }
	}
    }
    # These two lines removed due to some conflicts
    #    regsub -all {"?\(-"?([ \t\r\n]+"?\(-"?)+} $ma "(-" ma
    #    regsub -all {(^[ \t\r\n]*"?\(-"?|"?\(-"?[ \t\r\n]*$)} $ma "" ma

    # build the menu
    set name [list -n $name]
    if {[info exists theotherflags]} {
	set name [concat $theotherflags $name]
    }
    if {[info tclversion] >= 8.0} {
	lappend name -h [menu::helpText [lindex $name end]]
    }
    if {[info exists mproc]} {
	if {$mproc != ""} {
	    eval Menu $name -p $mproc [list $ma]
	} else {
	    eval Menu $name [list $ma]
	}
    } else {
	eval Menu $name -p menu::generalProc [list $ma]
    }
    menu::postEval $token
    if {[info exists msubs]} {
	return $msubs
    }
    return ""
}

proc menu::helpText {name} {
    switch -- $name {
	Config {
	    return [list "Config menu\r\rUse this menu to view and adjust\
	      your current preferences settings."]
	}
	Text {
	    return [list "Text menu\r\rUse this menu to manipulate lines,\
	      paragraphs or larger blocks of text.|Text menu\r\rIt is disabled\
	      because no windows are currently open."]
	}
	Utils {
	    return [list "Utils menu\r\rThis menu contains miscellaneous\
	      operations such as pairwise window comparison, spell-checking,\
	      plus access to various command-line 'Shells'."]
	}
	Search {
	    return [list "Search menu\r\rUse this menu to perform sophisticated\
	      find or replace operations on the contents of single or\
	      multiple windows or files."]
	}
	Edit {
	    return [list "Edit menu\r\rUse this menu to perform the standard\
	      cut, copy, paste operations, and to carry out other minor\
	      textual manipulations."]
	}
	File {
	    return [list "File menu\r\rUse this menu to open new windows,\
	      save or print existing windows, access recently used files,\
	      and revert windows to previously saved versions."]
	}
	default {
	    global index::help
	    if {[info exists index::help($name)]} {
		return [list [set index::help($name)]]
	    } else {
		return [list "This is the $name menu"]
	    }
	}
    }
}

proc menu::postEval {name} {
    global menu::posteval
    if {[info exists menu::posteval($name)]} {
	catch {uplevel \#0 [set menu::posteval($name)]}
    }
}

proc menu::replaceRebuild {name title} {
    global $name alpha::guiNotReady
    set bar [menu::inserted [set $name]]
    if {$bar} {removeMenu [set $name]}
    set $name $title
    if {![info exists alpha::guiNotReady]} {
	menu::buildSome $name
	if {$bar} {insertMenu [set $name]}
    }
}

# Move a toplevel menu to the end of the chain
proc menu::moveToEnd {menuName} {
    global $menuName
    if {[menu::inserted [set $menuName]]} {
	removeMenu [set $menuName]
	insertMenu [set $menuName]
    }
}

proc menu::packagesBuild {} {
    global alpha::package_menus package::prefs
    if {[info exists package::prefs]} {
	lappend ma [menu::itemWithIcon "packagePreferences" 84] 
	foreach pkg ${package::prefs} {
	    lappend ma "${pkg}Prefs"
	}
    }
    lappend ma "miscellaneousPackages" "\(-" "(-)"
    lappend ma "describeAPackage" "readHelpForAPackage" \
      "uninstallSomePackages" \
      "installAPackage" \
      "(-" "rebuildPackageIndices"

    # removed in 7.4
    # {Menu -m -n internetUpdates -p package::menuProc {}} 
    #return [list build $ma menu::packagesProc internetUpdates]
    return [list build $ma menu::packagesProc]
}

proc menu::packagesProc {menu item} {
    global package::prefs alpha::prefs
    if {[regexp "(.*)Prefs" $item d pkg]} {
	if {[lcontains package::prefs $pkg]} {
	    if {[info exists alpha::prefs($pkg)]} {
		dialog::pkg_options [set alpha::prefs($pkg)] \
		  "Preferences for the '[quote::Prettify $pkg]' package"
	    } else {
		dialog::pkg_options $pkg
	    }
	    return
	}
    }
    switch -- $item {
	"miscellaneousPackages" {
	    return [dialog::preferences $menu Packages]
	}
	"describeAPackage" -
	"Describe A Package" {
	    set pkg [dialog::optionMenu "Describe which package?" \
	      [lsort -ignore [alpha::package names]]]
	    package::describe $pkg
	}
	"readHelpForAPackage" -
	"Read Help For A Package" {
	    set pkg [dialog::optionMenu "Read help for which package?" \
	      [lsort -ignore [alpha::package names]]]
	    package::helpFile $pkg
	}
	"uninstallSomePackages" -
	"Uninstall Some Packages" {
	    package::uninstall
	}
	"installAPackage" -
	"Install A Package" {
	    install::fromRemoteUrl [dialog::getUrl]
	}
	"rebuildPackageIndex" {
	    alpha::rebuildPackageIndices
	}
	"packagePreferences" {
	    global::allPackages
	    #alertnote "Select a package from the group below in the menu to\
	    #edit its preferences."
	}
	default {
	    menu::generalProc global $item
	}
    }
}


proc menu::menuPackages {menu m} {
    if {[package::helpOrDescribe $m]} {
	return
    }
    # toggle global existence of '$m' menu
    global global::menus
    if {[set idx [lsearch  ${global::menus} $m]] == -1} {
	lappend global::menus $m
	global $m
	catch $m
	insertMenu [set	$m]
	markMenuItem packageMenus $m 1
    } else {
	set global::menus [lreplace ${global::menus} $idx $idx]
	global $m
	catch "removeMenu [set $m]"
	markMenuItem packageMenus $m 0
    }
    prefs::modified global::menus
}

if {[info tclversion] < 8.0} {
    proc menu::modeBuild {} {
	set ma [list "menus" "/p<Bfeatures" "/ppreferences" "editPrefsFile" \
	  "loadPrefsFile" "describeMode" "(-" "/m<UchangeMode"]
	return [list build $ma mode::menuProc "" "Mode Prefs"]
    }
} else {
    proc menu::modeBuild {} {
	global mode
	set ma [list "menus" "/p<Bfeatures" "/ppreferences" "editPrefsFile" \
	  "loadPrefsFile" "describeMode" "(-" "/m<UchangeMode"]
	if {$mode != ""} {
	    return [list build $ma mode::menuProc "" "${mode} Mode Prefs"]
	} else {
	    return [list build $ma mode::menuProc "" "Mode Prefs"]
	}
    }
}

proc menu::preferencesBuild {} {
    global flagPrefs
    
    set ma [list "Menus" "Arrange Menus" \
      "/p<U<BFeatures" "/p<USuffix Mappings" \
      "Save Preferences Now" "Edit Prefs File" "(-" \
      [menu::itemWithIcon "Interface Preferences" 84]]
    lappend ma Appearance Completions Electrics Help Text Tiling Window "(-" \
      [menu::itemWithIcon "Input-Output Preferences" 84]
    lappend ma Backups Files Printer Tags WWW "(-" \
      [menu::itemWithIcon "System Preferences" 84]
    eval lunion ma [lsort [lremove [array names flagPrefs] Packages]]
    return [list build $ma {dialog::preferences -m}]
}

proc menu::removeFrom {name args} {
    global menu::additions alpha::guiNotReady
    if {[info exists menu::additions($name)]} {
	if {[set i [lsearch -exact [set menu::additions($name)] $args]] != -1} {
	    set menu::additions($name) [lreplace [set menu::additions($name)] $i $i]
	    if {![info exists alpha::guiNotReady]} {
		# we were called after start-up; build the menu now
		menu::buildSome $name
	    } else {
		hook::register startupHook "menu::buildSome [list $name]"
	    }
	}
    }
}

proc menu::replaceWith {name current type args} {
    global menu::additions alpha::guiNotReady
    if {![info exists menu::additions($name)]} {
	lappend menu::additions($name) \
	  [concat [list $type [list replace $current]] $args]
    } else {
	set add 1
	set j 0
	foreach i [set menu::additions($name)] {
	    if {[lrange $i 0 1] == [list $type [list replace $current]]} {
		if {[lindex $i 1] != $args} {
		    set add 0
		    set menu::additions($name) \
		      [lreplace [set menu::additions($name)] $j $j \
		      [concat [list $type [list replace $current]] $args]]
		    break
		} else {
		    # no change
		    return
		}
	    }
	    incr j
	}
	if {$add} {
	    lappend menu::additions($name) \
	      [concat [list $type [list replace $current]] $args]
	}
    }
    if {![info exists alpha::guiNotReady]} {
	# we were called after start-up; build the menu now
	menu::buildSome $name
    } else {
	hook::register startupHook "menu::buildSome [list $name]"
    }
}

proc menu::itemWithIcon {name icon} {
    return "/\x1e${name}^[text::Ascii $icon 1]"
}

proc menu::fileProc {menu item} {
    switch -- $item {
	"open" {
	    findFile
	}
	"close" {
	    killWindow
	}
	default {
	    uplevel 1 [list menu::generalProc file $item]
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu::generalProc" --
 # 
 #  If either 'item' or 'menu::item' exists, call it.  Else try and
 #  autoload 'item', if that fails try and autoload 'menu::item'
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
    proc menu::generalProc {menu item {lower 1}} {
	if {$lower} {set menu [string tolower $menu]}
	if {[info commands ${menu}::${item}] != ""} {
	    uplevel \#0 ${menu}::$item
	} elseif {[info commands $item] != ""} {
	    uplevel \#0 $item
	} elseif {[auto_load ${menu}::$item]} {
	    uplevel \#0 ${menu}::$item
	} else {
	    uplevel \#0 $item
	}
    }
} else {
    proc menu::generalProc {menu item {lower 1}} {
	if {$lower} {set menu [string tolower $menu]}
	if {[info commands ::${menu}::${item}] != ""} {
	    uplevel \#0 ::${menu}::$item
	} elseif {[info commands $item] != ""} {
	    uplevel \#0 $item
	} elseif {[auto_load ::${menu}::$item]} {
	    uplevel \#0 ::${menu}::$item
	} else {
	    uplevel \#0 $item
	}
    }
}

proc menu::globalProc {menu item} {
    menu::generalProc global $item
}

## 
 # proc namedClipMenuProc {menu item} {
 #     switch $item {
 #         "copy"      "copyNamedClipboard"
 #         "cut"       "cutNamedClipboard"
 #         "paste"     "pasteNamedClipboard"
 #     }
 # }
 ##

proc menu::colorProc {menu item} {
    global colorInds
    
    if {[info exists colorInds($item)]} {
	set default $colorInds($item)
    } else {
	set default "65535 65535 65535"
    }
    
    set color [eval [list colorTriple "New \"$item\":"] $default]
    if {![string length $color]} { return }
    eval setRGB $item $color
    set colorInds($item) $color
    alpha::makeColourList
    prefs::modified colorInds
}

proc alpha::makeColourList {} {
    global alpha::colors colorInds alpha::basiccolors
    # Set up color indices
    foreach ind [array names colorInds] {
	eval setRGB $ind $colorInds($ind)
    }
    set alpha::basiccolors {none blue cyan green magenta red white yellow}
    set alpha::colors ${alpha::basiccolors}
    foreach c {color_9 color_10 color_11 color_12 color_13 color_14 color_15} {
	if {[info exists colorInds($c)]} {lappend alpha::colors $c}
    }
}
	
namespace eval icon {}
namespace eval file {}

proc icon::FromID {ID} {
    return "^[text::Ascii [expr {$ID - 0x1D0}] 1]"
}

proc icon::FromSig {sig} {
    global alpha::_icons
    if {[set p [lsearch -glob ${alpha::_icons} "[quote::Find ${sig}] *"]] != -1} {
	set p [lindex ${alpha::_icons} $p]
	return [lindex $p 2]
    } else {
	return ""
    }
}

proc icon::MenuFromSig {sig} {
    global alpha::_icons
    if {[set p [lsearch -glob ${alpha::_icons} "[quote::Find ${sig}] *"]] != -1} {
	set char [expr {[lindex [lindex ${alpha::_icons} $p] 2] -208}]
	if {$char < 1 || $char > 256} { return "" }
	return "^[text::Ascii $char 1]"
    } else {
	return ""
    }
}

proc menu::fileUtils {menu item} {
    if {[lsearch -exact {"insertPathName" "insertFile" "fileRemove" \
      "fileInfo" "wordCount" "textToAlpha"} $item] != -1} {return [$item]}
    switch -- $menu {
	"moreUtils" {
	    file::Utils::$item
	}
	default {
	    file::$item
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 #	"menu::buildHierarchy" --
 # 
 #  Given a list of folders, 'menu::buildHierarchy' returns a
 #  hierarchical menu based on the files and subfolders in each of
 #  these folders.  Pathnames are optionally stored in an array in the
 #  caller's context given by the argument 'filePaths'.  The path's
 #  index in this array is formed by concatenating the submenu name and
 #  the filename, allowing the pathname to be retrieved by the
 #  procedure 'proc' when the menu item is selected.
 # 
 #  The search may be restricted to files with specific extensions, or
 #  files matching a certain pattern.  A search depth may also be given,
 #  with three levels of subfolders assumed by default.
 # 
 #  See filesets code, perl or latex modes for examples.
 # 
 #  (originally written by Tom Pollard, with modifications by Vince Darley
 #  and Tom Scavo)
 # 
 # --Version--Author------------------Changes-------------------------------
 #	  1.0	  Tom Pollard			        original
 #	  2.0	  <vince@santafe.edu> multiple extensions, optional paths
 #	  2.1	  Tom Scavo	      multiple folders
 #	  2.2	  <vince@santafe.edu> pattern matching as well as exts
 #	  2.3	  <vince@santafe.edu> handles unique menu-names and does text only
 #	  2.4	  <jl@theophys.kth.se>    now also handles patterns like "*.{a,b}"
 #	  2.5	  <vince@santafe.edu> better glob, non-dup dir handling
 # -------------------------------------------------------------------------
 ##
proc menu::buildHierarchy {folders name proc {filePaths {}} {exts *} {depth 3}\
  {fset {}} {ignore ""} {top 1}} {
    global filesetmodeVars file::separator
    if { $filePaths != "" } {
	upvar 1 $filePaths fp
    }
    if {[llength $exts] > 1} {
	regsub -all {\.} $exts "" exts
	set exts "*.{[join $exts ,]}"
    } elseif {[string match ".*" $exts] && ![string match {*\**} $exts]} {
	set exts "*$exts"
    }
    incr depth -1
    set overallMenu {}
    foreach folder $folders {
	if {[file exists $folder]} {
	    if {![file isdirectory $folder]} {
		set folder [file dirname $folder]
	    }
	    if {[regexp -- "${file::separator}$" $folder]} {
		set folder [string trimright $folder ${file::separator}]
	    }
	    if {$name == 0} {
		set name [file tail $folder]
	    }
	    # if it's a fileset, we register _before_ recursing
	    if { $fset != "" } {
		set mname [filesetMenu::registerName $fset $name $proc $top]
	    } else {
		set mname $name
	    }
	    set menu {}
	    set subfolders [glob -nocomplain -types d -dir $folder *]
	    if {$filesetmodeVars(includeNonTextFiles)} {
		set filenames [glob -nocomplain -dir $folder -- $exts]
	    } else {
		set filenames [glob -types TEXT -nocomplain -dir $folder $exts]
	    }
	    
	    set orderedList [lsort -ignore [concat $subfolders $filenames]]
	    if {[llength $ignore]} {
		set ignoreList \
		  [eval [list glob -nocomplain -dir $folder --] $ignore]
		# We need '-all' because directories are probably listed twice.
		set orderedList [lremove -l -all -- $orderedList $ignoreList]
	    }
	    set last {}
	    # Note that the list of filenames may also contain some/all
	    # subfolders (if they matched the glob expression), hence
	    # we must be sure not to add them twice.
	    set count 0
	    foreach m $orderedList {
		if {[file isfile $m]} {
		    set fname [file tail $m]
		    if {([info tclversion] < 8.0) \
		      && ([string index $fname 0] == "-")} {
			set fname " $fname"
		    }
		    lappend menu "${fname}&"
		    if {$filePaths != ""} {
			set fp([file join $name $fname]) $m
		    }
		    incr count
		    if {([info tclversion] < 8.0) && ($count > 999)} {
			alertnote "There are too many items in '$name' menu.\
			  Some will be ignored."
			break
		    }
		} elseif {$depth > 0 && ($m != $last)} {
		    set last $m
		    set subM [menu::buildHierarchy \
		      [list ${m}] 0 $proc fp $exts $depth $fset $ignore 0]
		    if {[llength $subM]} { 
			lappend menu $subM 
			set first 0
		    }
		}
	    }
	    if {[llength $menu]} {
		set overallMenu [concat $overallMenu $menu]
	    }
	} else {
	    beep
	    alertnote "menu::buildHierarchy:  Folder $folder does not exist!"
	}
    }
    
    if {[llength $overallMenu]} {
	if { [string length $proc] > 1 } {
	    set pproc [list -p $proc]
	} else {
	    set pproc [list]
	}	
	if { $fset != "" } {
	    if {[string length $proc] > 1} { 
		set pproc [list -p filesetMenu::subProc] 
	    }
	}	 
	return [concat [list Menu -m -n] [list $mname] $pproc [list $overallMenu]]
    } else {
	return [list]
    }
}

# in case we've done something odd elsewhere
ensureset filesetmodeVars(includeNonTextFiles) 0

proc menu::reinterpretOldMenu {args} {
    set ma [lindex $args end]
    set args [lreplace $args end end]
    getOpts {-n -M -p}
    if {[info exists opts(-p)]} {
	lappend proc $opts(-p)
    } else {
	lappend proc "-1"
    }
    if {[info exists opts(-M)]} { lappend proc -M $opts(-m) }
    if {[info exists opts(-m)]} { lappend proc -m }
    menu::buildOne $opts(-n) build $ma $proc
}

