## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "package.tcl"
 #                                    created: 2/8/97 {6:15:10 pm} 
 #                                last update: 10/29/2001 {17:30:50 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 1997-2001  Vince Darley, all rights reserved
 # 
 #  How to ensure packages are loaded in the correct order?
 #  (some may require Vince's Additions).  Here perhaps we could
 #  just use a Tcl8-like-approach: introduce a 'package' command
 #  and have stuff like 'package Name 1.0 script-to-load'.
 #  Then a package can just do 'package require Othername' to ensure
 #  it is loaded.  I like this approach.
 #  
 #  How to initialise each package at startup?  If we use the above
 #  scheme, then the startup script is purely a sequence of
 #  'package require Name' commands.  The file 'prefs.tcl' is then
 #  purely for user-meddling.  Packages do not need to store anything
 #  there.  Sounds good to me.
 #  
 #  How to uninstall things?  One approach here is a 
 #  'package uninstall Name' command.  Nice packages would provide
 #  this.
 #  
 #  We need a default behaviour too.  Some packages require no
 #  installation at all (except placing in a directory), others 
 #  require sourcing, others need to add something to a menu.  How
 #  much of this should be automated and how much is up to the
 #  package author?
 # 
 # ----
 # 
 #  The solution below is to imitate Tcl 8.  There is a 'package'
 #  mechanism.  There exists a index::feature() array which gives for
 #  each package the means to load it --- a procedure name or a
 #  'source file' command.  The package index is compiled 
 #  automatically by recursively scanning all files in the
 #  Packages directory for 'package name version do-this'
 #  commands.
 #  
 #  There's also 'package names', 'package exists name', and an
 #  important 'package require name version' which allows one
 #  package to autoload another...
 #  
 # Pros of this approach: many packages, which would otherwise
 # require an installation procedure, now can be just dropped
 # in to the packages directory and they're installed! (After
 # rebuilding the package index).  This is because 'package'
 # can declare a snippet of code, an addition to a menu etc
 # ----
 # 
 # Thanks to Tom Fetherston for some improvements here.
 # ###################################################################
 ##

namespace eval package {}
namespace eval remote {}

set package::loaded [list]

## 
 # -------------------------------------------------------------------------
 # 
 # "alpha::findAllExtensions" --
 # 
 #  package require all extensions the user has activated
 # -------------------------------------------------------------------------
 ##
proc alpha::findAllExtensions {} {
    global global::features index::feature alpha::earlyPackages
    foreach m [array names index::feature] {
	if {[lsearch -exact [set global::features] $m] != -1} {
	    # it's on
	    if {[lsearch -exact [set alpha::earlyPackages] $m] != -1} {
		# We already did this one.
		continue
	    }
	    if {[package::do_activate $m]} {
		set global::features [lremove ${global::features} $m]
		package::throwActivationError
	    }
	} else {
	    if {[lindex [set index::feature($m)] 2] == 2} {
		package::initialise $m
	    }
	}
    }

    # remove any package which doesn't exist.
    foreach m [set global::features] {
	if {![info exists index::feature($m)]} {
	    set global::features [lremove ${global::features} $m]
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::addPrefsDialog" --
 # 
 #  Register a given package as having a standard preferences page which
 #  should be accessible from the 'Config->Packages' menu.  If the optional
 #  'mapTo' argument is given, then the actual preferences data is not
 #  stored in the array variable ${pkg}modeVars, but rather in 
 #  ${mapTo}modeVars.  This is useful if the 'pkg' name is rather long..
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
proc package::addPrefsDialog {pkg {mapTo ""}} {
    global package::prefs alpha::guiNotReady alpha::prefs
    if {[string length $mapTo]} {
	set alpha::prefs($pkg) $mapTo
    }
    lunion package::prefs $pkg
    if {![info exists alpha::guiNotReady]} {
	# we were called after start-up; build the menu now
	menu::buildSome packages
    }
}
} else {
proc package::addPrefsDialog {pkg {mapTo ""}} {
    global package::prefs alpha::guiNotReady
    if {[string length $mapTo]} {
	# I think the existence of two variables *::prefs in this proc
	# causes problems, especially with lunion's upvar call.
	set ::alpha::prefs($pkg) $mapTo
    }
    lunion package::prefs $pkg
    if {![info exists alpha::guiNotReady]} {
	# we were called after start-up; build the menu now
	menu::buildSome packages
    }
}
}

## 
 # -------------------------------------------------------------------------
 # 
 # "alpha::package" --
 # 
 #  Mimics the Tcl standard 'package' command for use with Alpha.
 #  It does however have some differences.
 #  
 #  package require ?-exact? ?-extension -mode -menu? name version
 #  package exists ?-extension -mode -menu? name version
 #  package names ?-extension -mode -menu?
 #  package uninstall name version
 #  package vcompare v1 v2
 #  package vsatisfies v1 v2
 #  package versions ?-extension -mode -menu? name
 #  package type name
 #  package info name
 #  package maintainer name version {name email web-page}
 #  package modes 
 #  
 #  Equivalent to alpha::mode alpha::menu and alpha::extension
 #  
 #  package mode ...
 #  package menu ...
 #  package extension ...
 #  
 #  For extensions only:
 #  
 #  package forget name version
 # -------------------------------------------------------------------------
 ##
proc alpha::package {cmd args} {
    global index::feature
    switch -- $cmd {
	"require" {
	    set info [package::getInfo "exact loose"]
	    global alpha::rebuilding
	    if {[llength $info]} {
		if {!${alpha::rebuilding} && [set version [lindex $args 1]] != ""} {
		    if {[info exists exact]} {
			if {[lindex $info 0] != $version} {
			    error "requested $name exact $version, had [lindex $info 0]"
			}
		    } elseif {[info exists loose]} {
			if {[alpha::package vcompare [lindex $info 0] $version] < 0} {
			    error "requested $name $version or newer, had [lindex $info 0]"
			}
		    } elseif {![alpha::package vsatisfies [lindex $info 0] $version]} {
			error "requested $name $version, had [lindex $info 0]"
		    }
		}
		if {$type == "feature"} {
		    global global::features package::loaded
		    set failed 0
		    if {[lsearch -exact [set package::loaded] $name] == -1} {
			lappend package::loaded $name
			set failed [package::do_activate $name]
		    }
		    if {!$failed} {
			if {[lsearch -exact ${global::features} $name] == -1} {
			    lappend global::features $name
			}
		    } else {
			package::throwActivationError
		    }
		}
		return [lindex $info 0]
	    }
	    if {!${alpha::rebuilding}} {
		error "can't find package $name"
	    }
	}
	"uninstall" {
	    set name [lindex $args 0]
	    if {[llength $args] > 2} {
		set version [lindex $args 1]
		global alpha::rebuilding 
		if {${alpha::rebuilding}} {
		    global rebuild_cmd_count index::uninstall pkg_file
		    switch -- [set script [lindex $args 2]] {
			"this-file" {
			    set script [alpha::actionOnFileScript "file delete" $pkg_file]
			}
			"this-directory" {
			    set script [alpha::actionOnFileScript "rm -r" [file dirname $pkg_file]]
			}
		    }
		    set index::uninstall($name) [list $version \
		      [string trim [alpha::actionOnFileScript "" $pkg_file]] $script]
		    set args [lrange $args 3 end]
		    if {[llength $args]} {
			eval alpha::package [lindex $args 0] $name $version [lrange $args 1 end]
			return
		    }
		    if {[info exists rebuild_cmd_count] && [incr rebuild_cmd_count -1] == 0} {
			return -code 11
		    }
		}
	    } else {
		global index::uninstall
		cache::readContents index::uninstall
		set ret [set index::uninstall($name)]
		catch {unset index::uninstall}
		return $ret
	    }
	}
	"forget" {
	    catch {unset index::feature($name)}
	}
	"exists" {
	    if {[package::getInfo] != ""} {return 1} else {return 0}
	}
	"type" {
	    if {[package::getInfo] != ""} {return $type} 
	    error "No such package"
	}
	"info" {
	    if {[llength [set info [package::getInfo]]]} {return [concat $type $info]} 
	    error "No such package"
	}
	"requirements" {
	    set name [lindex $args 0]
	    if {[llength $args] > 2} {
		global alpha::rebuilding 
		if {${alpha::rebuilding}} {
		    set version [lindex $args 1]
		    global rebuild_cmd_count index::$cmd
		    set data [lindex $args 2]
		    set index::${cmd}($name) [list $version $data]
		    set args [lrange $args 3 end]
		    if {[llength $args]} {
			eval alpha::package [lindex $args 0] $name $version [lrange $args 1 end]
			return
		    }
		    if {[info exists rebuild_cmd_count] && [incr rebuild_cmd_count -1] == 0} {
			return -code 11
		    }
		}
	    } else {
		global index::$cmd
		if {[info exists index::${cmd}($name)]} {
		    return [set index::${cmd}($name)]
		} else {
		    return ""
		}
	    }
	}
	"preinit" -
	"maintainer" -
	"disable" -
	"description" -
	"help" {
	    set name [lindex $args 0]
	    if {[llength $args] > 2} {
		global alpha::rebuilding 
		if {${alpha::rebuilding}} {
		    set version [lindex $args 1]
		    global rebuild_cmd_count index::$cmd
		    set data [lindex $args 2]
		    set index::${cmd}($name) [list $version $data]
		    set args [lrange $args 3 end]
		    if {[llength $args]} {
			eval alpha::package [lindex $args 0] $name $version [lrange $args 1 end]
			return
		    }
		    if {[info exists rebuild_cmd_count] && [incr rebuild_cmd_count -1] == 0} {
			return -code 11
		    }
		}
	    } else {
		cache::readContents index::$cmd
		if {[info exists index::${cmd}($name)]} {
		    return [set index::${cmd}($name)]
		} else {
		    return ""
		}
	    }
	}
	"versions" {
	    set info [package::getInfo]
	    if {[llength $info]} {
		return [lindex $info 0]
	    }
	    error "No such package"
	}
	"vcompare" {
	    set c [eval package::_versionCompare $args]
	    if {$c > 0 || $c == -3} {
		return 1
	    } elseif {$c == 0} {
		return 0
	    } else {
		return -1
	    }
	}
	"vsatisfies" {
	    if {[lindex $args 0] == "-loose"} {
		set c [eval package::_versionCompare [lrange $args 1 end]]
		return [expr {$c >= 0 || $c == -3 ? 1 : 0}]
	    } else {
		set c [eval package::_versionCompare $args]
		return [expr {$c >= 0 ? 1 : 0}]
	    }
	}
	"names" {
	    set names ""
	    package::getInfo
	    foreach type $which {
		if {[array exists index::${type}]} {
		    eval lappend names [array names index::${type}]
		}
	    }
	    return $names
	}
	"mode" -
	"menu" -
	"feature" {
	    eval alpha::$cmd $args
	}
	default {
	    error "Unknown option '$cmd' to 'package'"
	}
    }
}

proc package::getInfo {{flags ""}} {
    uplevel [list set flags $flags]
    uplevel {
	set name [lindex $args 0]
	if {[regexp -- {-([^-].*)} $name "" which]} {
	    if {[lsearch $flags $which] != -1} {
		set $which 1
		set name [lindex $args 1]			
		set args [lrange $args 1 end]			
		return [package::getInfo $flags]
	    }
	    if {[lsearch {feature mode} $which] == -1} {
		error "No such flag -$which"
	    }
	    set name [lindex $args 1]
	    set args [lrange $args 1 end]
	} else {
	    set which {feature mode}
	}
	foreach type $which {
	    if {$type != "feature"} {cache::readContents index::${type}}
	    if {[info exists index::${type}($name)]} {
		return [set index::${type}($name)]
	    }
	}
	return ""
    }	
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::_versionCompare" --
 # 
 #  This proc compares the two version numbers.  It returns:
 #  
 #  0 equal
 #  1 equal but beta/patch update
 #  2 equal but minor update
 #  -1 beta/patch version older
 #  -2 minor version older
 #  -3 major version newer
 #  -5 major version older
 #  
 #  i.e. >= 0 is basically ok, < 0 basically bad
 #  
 #  It works for beta, alpha, dev, fc and patch version numbers.
 #  Any sequence of letters starting b,a,d,f,p are assumed to
 #  represent the particular item.
 #  
 #  2.4 > 1.5 > 1.4.3 > 1.4.3b2 > 1.4.3b1 > 1.4.3a75 > 1.4p1 > 1.4
 # -------------------------------------------------------------------------
 ##
proc package::_versionCompare {v1 v2} {
    regsub -all -nocase {([a-z])[a-z]+} $v1 {\1} v1
    regsub -all -nocase {([a-z])[a-z]+} $v2 {\1} v2
    set v1 [split $v1 .p]
    set v2 [split $v2 .p]
    set i -1
    set ret 0
    set mult 2
    while 1 {
	incr i
	set sv1 [lindex $v1 0]
	set sv2 [lindex $v2 0]
	if {$sv1 == "" && $sv2 == ""} { break }
	if {$sv1 == ""} { 
	    set v1 [concat 8 0 $v1]
	    set v2 [concat 9 $v2]
	    continue
	} elseif {$sv2 == ""} { 
	    set v1 [concat 9 $v1]
	    set v2 [concat 8 0 $v2]
	    continue
	} elseif {[regexp -nocase {[a-z]} "$sv1$sv2"]} {
	    # beta versions
	    foreach v {sv1 sv2} {
		if {[regexp -nocase {[a-z]} [set $v]]} {
		    # f = 8, b = 7, a = 6, d = 5
		    regsub -nocase {([^a-z])f} [set $v] {\1 7 } $v
		    regsub -nocase {([^a-z])b} [set $v] {\1 6 } $v
		    regsub -nocase {([^a-z])a} [set $v] {\1 5 } $v
		    regsub -nocase {([^a-z])d} [set $v] {\1 4 } $v
		} else {
		    # release version = 8, so it is larger than any of the above
		    append $v " 8"
		}
	    }
	    set v1 [eval lreplace [list $v1] 0 0 $sv1]
	    set v2 [eval lreplace [list $v2] 0 0 $sv2]
	    set mult 1
	    continue
	}
	if {$sv1 < $sv2} { set ret -1 ; break }
	if {$sv1 > $sv2} { set ret 1 ; break }
	set v1 [lrange $v1 1 end]
	set v2 [lrange $v2 1 end]
    }
    if {$i == 0} {
	# major version, return 0, -3, -5
	return [expr {$ret * (-4*$ret + 1)}]
    } else {
	return [expr {$mult *$ret}]
    }
}

proc package::versionCheck {name vers} {
    set av [alpha::package versions $name]
    set c [package::_versionCompare $av $vers]
    if {$c < 0 && $c != -3} {			
	error "The installed version $av of '$name' is too old. Version $vers was requested."
    } elseif {$c == -3} {			
	error "The installed version $av of '$name' may not be backwards compatible with the requested version ($vers)."
    }			
}

proc package::reqInstalledVersion {name exact? {reqvers ""}} {
    global index::feature
    # called from installer
    set msg " I suggest you abort the installation."
    if {[info exists index::feature($name)]} {
	if {[set exact?] == ""} {return}
	set av [alpha::package versions $name]
	if {[set exact?] == "-exact"} {
	    if {[alpha::package versions $name] != $reqvers} {
		alertnote "The installed version $av of '$name' is incorrect.  Exact version $reqvers was requested.$msg"
	    }
	} else {
	    set reqvers [set exact?]
	    if {$reqvers != ""} {		
		set c [package::_versionCompare $av $reqvers]			
		if {$c < 0 && $c != -3} {			
		    alertnote "The installed version $av of '$name' is too old. Version $reqvers was requested.$msg"
		} elseif {$c == -3} {			
		    alertnote "The installed version $av of '$name' may not be backwards compatible with the requested version ($reqvers).$msg"
		} 			
	    }		
	}
    } else {
	alertnote "This package requires the prior installation of '$name'. It is not currently installed.$msg"
    }
}

proc package::checkRequire {pkg} {
    if {[catch {alpha::package require $pkg} error]} {
	global errorInfo ; echo $errorInfo
	if {[catch {alertnote "The '$pkg' package had an error starting up: $error"} ]} {
	    alertnote "The '$pkg' package had an error starting up"
	    echo $error
	}
    }	
}



proc package::queryWebForList {} {
    global defaultAlphaDownloadSite remote::site package::_pkgtemp
    set sitename [dialog::value_for_variable defaultAlphaDownloadSite\
      "Query which site?"]
    
    temp::cleanup _pkgtemp
    set package::_pkgtemp [temp::unique _pkgtemp _pkg]
    set siteurl [set remote::site($sitename)]
    
    message "Fetching remote list"
    set type [lindex [url::fetch $siteurl [set package::_pkgtemp]] 0]
    package::okGotTheList $sitename
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::okGotTheList" --
 # 
 #  Helper proc which we can also call if the listing was interrupted
 #  half-way through.
 # -------------------------------------------------------------------------
 ##
proc package::okGotTheList {{sitename ""}} {
    global defaultAlphaDownloadSite remote::site package::_pkgtemp remote::lastsite
    if {$sitename == ""} {
	if {[info exists remote::lastsite]} {
	    set sitename ${remote::lastsite}
	    unset remote::lastsite
	} else {
	    set sitename [dialog::value_for_variable defaultAlphaDownloadSite "From which site did you get the list?"]
	}
    }
    set type [lindex [url::parse [set remote::site($sitename)]] 0]
    set nm [set package::_pkgtemp]
    if {![file exists $nm] || (![file writable $nm]) || (![file size $nm])} {
	alertnote "It looks like that application returned control\
	  to me before the download was complete (otherwise there was an error)\
	  -- probably Netscape/IE.  When it's done, or if there was an error\
	  hit Ok."
    }
    if {![file exists $nm] || (![file writable $nm]) || (![file size $nm])} {
	dialog::alert "There was a problem fetching the list --- if it's still\
	  being downloaded (you hit Ok too early!), wait till it's done \
	  and then select 'Ok Got The List'\
	  from the internet updates menu."
	set remote::lastsite $sitename
	enableMenuItem -m internetUpdates "Ok, Got The List" on
	error "Error fetching list of new packages"
    } else {
	enableMenuItem -m internetUpdates "Ok, Got The List" off
    }
    set fd [alphaOpen $nm "r"]
    catch {set lines [split [read $fd] "\n\r"]}
    close $fd
    
    if {[catch [list remote::process${type}Listing $lines] listing]} {
	alertnote "Error interpreting list of new packages"
	error "Error interpreting list of new packages"
    }
    message "Processing list"
    remote::processList $sitename $listing
    message "Updated remote package information."
}

proc package::active {pkg {text ""}} {
    global global::features mode
    if {([lsearch -exact ${global::features} $pkg] != -1) \
      || ($mode != "" && ([lsearch -exact [mode::getFeatures $mode] $pkg] != -1))} {
	if {[llength $text]} { return [lindex $text 0] } else {return 1 }
    } else {
	if {[llength $text]} { return [lindex $text 1] } else {return 0 }
    }
}

proc package::_editSite {{name ""} {loc ""}} {
    if {$name == ""} {
	set title "Name of new archive site"
	set name "Ken's Alpha site"
	set loc "ftp://ftp.ken.com/pub/Alpha/"
    } else {
	set title "Archive site name"
    }
    set y 10
    set yb 105
    set res [eval dialog -w 420 -h 135 \
      [dialog::textedit $title $name 10 y 40] \
      [dialog::textedit "URL for site" $loc 10 y 40] \
      [dialog::okcancel -400 yb 0]]
    if {[lindex $res 3]} { error "Cancel" } 
    # cancel was pressed
    return [lrange $res 0 1]	
}

proc package::addIndex {args} {
    global index::feature pkg_file
    cache::readContents index::feature
    foreach f [concat $args] {
	set pkg_file $f
	message "scanning $f"
	catch {source $f}
    }
    cache::create index-extension "variable" index::feature
    unset pkg_file
}

proc package::helpFile {pkg {pointer 0}} {
    set v [alpha::package versions $pkg]
    if {[lindex $v 0] == "for"} {
	set type [lindex $v 2]
	set v [lindex $v 1]
	if {$pointer} {
	    return "The '$pkg' package is implemented by $v $type, \
	      and has no separate help."
	}
	set pkg $v
    }
    if {![catch {alpha::package help $pkg} res]} {
	set help [string trim [lindex $res 1]]
	if {[lindex [split $help " \t"] 0] == "file"} {
	    if {$pointer} {
		return "Help for this package is located in \
		  \"[lindex $help 1]\""
	    } else {
		help::openFile [lindex $help 1]
	    }
	} elseif {[string index $help 0] == "\["} {
	    if {$pointer} {
		return "You can read help for this package \
		  by holding 'shift' when\ryou select its name in the menu."
	    } else {
		uplevel \#0 [string range $help 1 \
		  [expr {[string length $help] - 2}]]
	    }
	} else {
	    if {$pointer} {
		return $help
	    } else {
		new -n "* '$pkg' Help *" -info \
		  "\rHelp for package '$pkg', version \
		  [alpha::package versions $pkg]\r\r    $help"
		message "Please wait: Colouring and marking the help file"
		help::hyperiseEmailAddresses
		help::hyperiseUrls
		help::colourHeadingsEtc
		message ""
	    }
	}
	return
    }
    if {!$pointer} {
	alertnote "Sorry, there isn't a help file for that package. \
	  You should contact the package maintainer."
    }
    return
}


## 
 # -------------------------------------------------------------------------
 # 
 # "package::helpFilePresent" --
 # 
 #  Help files must be of the same name as the package (minus 'mode' or 
 #  'menu'), but may have any combination of mode, menu, or help after
 #  that name.  Whitespace is irrelevant.
 # -------------------------------------------------------------------------
 ##
proc package::helpFilePresent {args} {
    set res ""
    cache::readContents index::help
    foreach pkg $args {
	lappend res [info exists index::help($pkg)]
    }
    return $res
}

proc package::helpOrDescribe {pkg} {
    if {[set mods [expr {[getModifiers] & 0xfe}]]} {
	if {$mods & 34} {
	    package::helpFile $pkg
	} else {
	    package::describe $pkg
	}
	return 1
    }
    return 0
}

#  Specific to 'features'  #

proc package::addRelevantMode {_feature mode} {
    global index::feature
    if {[info exists index::feature($_feature)]} {
	if {[lsearch -exact [set oldm [lindex [set index::feature($_feature)] 1]] $mode] != -1} {
	    return
	}
	lappend oldm $mode
	set index::feature($_feature) \
	  [lreplace [set index::feature($_feature)] 1 1 $oldm]
    } else {
	set index::feature($_feature) [list [list "for" $mode "mode"] $mode]
    }
}

proc package::removeRelevantMode {_feature mode} {
    global index::feature
    if {[info exists index::feature($_feature)]} {
	if {[set idx [lsearch -exact [set oldm [lindex [set index::feature($_feature)] 1]] $mode]] == -1} {
	    return
	}
	set oldm [lreplace $oldm $idx $idx ""]
	set index::feature($_feature) \
	  [lreplace [set index::feature($_feature)] 1 1 $oldm]
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::onOrOff" --
 # 
 #  Complicated procedure to accomplish a relatively simple task!
 #  
 #  Given the current mode (possibly ""), and the last mode, work out
 #  what changes have to be made to the set of on/off features to
 #  synchronise everything.
 #  
 #  This procedure is now only used by changeMode.
 #  
 #  Note that features which no longer exist are still returned by
 #  this procedure.  Hence calling procedures should possibly
 #  check whether the index::feature array entry exists.
 # -------------------------------------------------------------------------
 ##
proc package::onOrOff {curMode {lastMode ""}} {
    global global::features
    set oldfeatures ""
    set offfeatures ""
    set onfeatures ""
    set newfeatures ""
    if {[mode::exists $curMode]} {
	set pkgs [mode::getFeatures $curMode]
    } else {
	set pkgs {}
    }
    foreach m $pkgs {
	if {[string index $m 0] == "-"} {
	    set m [string range $m 1 end]
	    if {[lsearch -exact ${global::features} $m] >= 0} {
		lappend offfeatures $m
	    }
	} else {
	    if {[lsearch -exact ${global::features} $m] < 0} {
		lappend newfeatures $m
	    }
	}
    }
    if {[mode::exists $lastMode]} {
	foreach m [mode::getFeatures $lastMode] {
	    if {[string index $m 0] == "-"} {
		set m [string range $m 1 end]
		if {[lsearch -exact ${global::features} $m] >= 0} {
		    if {[set ip [lsearch -exact $offfeatures $m]] < 0} {
			lappend newfeatures $m
		    } else {
			set offfeatures [lreplace $offfeatures $ip $ip]
		    }
		}
	    } else {
		if {[lsearch -exact ${global::features} $m] < 0} {
		    lappend oldfeatures $m
		    if {[lsearch -exact $newfeatures $m] < 0} {
			lappend offfeatures $m
		    }
		}
	    }
	}
    }
    foreach m $newfeatures {
	if {[lsearch -exact $oldfeatures $m] < 0} {
	    lappend onfeatures $m
	}
    }
    return [list $offfeatures $onfeatures]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::partition" --
 # 
 #  Return either 3 lists: menus, features and then modes, if 'mode'
 #  is empty, or return 6 lists.
 #  
 #  In this second case we have 2 choices:
 #  
 #  (i) 'mode = global', we return: 
 #  usual menus, other menus, {}, usual features, other features, {}
 #  'Usual' means global, 'Other' means everything else.
 #  (ii) 'mode = some given mode', we return: 
 #  usual menus, general menus, other, usual features, general features, other
 #  'Usual' means mode-specific, 'General' means global, 'Other' means
 #  specific to other modes, or global-only
 #  
 #  Note: when we partition for a given mode, we remove all items
 #  which are currently globally on.
 # -------------------------------------------------------------------------
 ##
proc package::partition {{mode ""} {mfb 0} {ignore_flags 1}} {
    global index::feature index::flags
    set a ""
    set b ""
    set c ""
    if {$mode == ""} {
	# This is the case in which we just want everything.
	foreach n [lsort -ignore [alpha::package names]] {
	    if {$ignore_flags && ([lsearch -exact ${index::flags} $n] != -1)} {
		continue
	    }
	    if {[info exists index::feature($n)]} {
		switch -- [lindex [set index::feature($n)] 2] {
		    "1" {
			lappend a $n
		    }
		    default {
			lappend b $n
		    }
		}
	    } else {
		lappend c $n
	    }
	}
	return [list $a $b $c]
    } else {
	# Now we either want only global items, or for the given mode
	set d ""
	set e ""
	set f ""
	set partition [array names index::feature]
	if {$mode == "global"} {
	    set mode "global*"
	    set search "-glob"
	} else {
	    set search "-exact"
	    global global::features
	    set partition [lremove -l $partition ${global::features}]
	}		
	foreach n [lsort -ignore $partition] {
	    if {$ignore_flags && ([lsearch -exact ${index::flags} $n] != -1)} {
		continue
	    }
	    set ff [set index::feature($n)]
	    switch -- [lindex $ff 2] {
		"1" {
		    if {$mfb == 2} {continue}
		    if {[lsearch $search [lindex $ff 1] $mode] != -1} {
			lappend a $n
		    } elseif {[lsearch -exact [lindex $ff 1] "global"] != -1} {
			lappend b $n
		    } elseif {[lindex $ff 1] != "global-only"} {
			lappend c $n
		    }
		}
		"-1" {
		    # ignore auto-loading types
		}
		default {
		    if {$mfb == 1} {continue}
		    if {[lsearch $search [lindex $ff 1] $mode] != -1} {
			lappend d $n
		    } elseif {[lsearch -exact [lindex $ff 1] "global"] != -1} {
			lappend e $n
		    } elseif {[lindex $ff 1] != "global-only"} {
			lappend f $n
		    }
		}
	    }
	}
	return [list $a $b $c $d $e $f]
    }	
}


proc package::describe {pkg {return 0}} {
    set info [alpha::package info $pkg]
    set type [lindex $info 0]
    set v [alpha::package versions $pkg]
    if {[lindex $v 0] == "for"} {
	set v [lindex $v 1]
	set msg "Package '$pkg', designed for use by $v [lindex $v 2] is a"
    } else {
	set msg "Package '$pkg', version $v is a"
    }
    
    switch -- $type {
	"feature" {
	    switch -- [lindex $info 3] {
		"1" {
		    append msg " menu, and is "
		    global global::menus
		    if {![package::active $pkg]} {
			append msg "not "
		    }
		    append msg "in use."
		}
		"-1" {
		    append msg "n autoloading $type."
		}
		default {
		    append msg " $type, and is [package::active $pkg {active inactive}]."
		}
	    }
	}
	"mode" {
	    append msg " $type; modes are always active."
	}
    }
    global index::maintainer
    cache::readContents index::maintainer
    if {[info exists index::maintainer($pkg)]} {
	set p [lindex [set index::maintainer($pkg)] 1]
	append msg "\rMaintainer: [lindex $p 0], [lindex $p 1]\r"
	append msg [lindex $p 2]
    }
    catch {unset index::maintainer}
    if {$return} {
	return $msg
    }
    # let package tell us where its prefs are stored.
    global alpha::prefs
    if {[info exists alpha::prefs($pkg)]} {
	set pkgpref [set alpha::prefs($pkg)]
    } else {
	set pkgpref $pkg
    }
    global ${pkgpref}modeVars
    if {[array exists ${pkgpref}modeVars]} {
	append msg "\r\r" [mode::describeVars $pkg $pkgpref]
	new -n "* <$pkg> description *" -m Tcl -info $msg
    } else {
	alertnote $msg
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::deactivate" --
 # 
 #  Turns off all the packages given.  This procedure must never throw an
 #  error to its caller.
 # -------------------------------------------------------------------------
 ##
proc package::deactivate {args} {
    foreach pkg $args {
	package::tryToDeactivate $pkg log
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::activate" --
 # 
 #  Turns on all the packages given.  This procedure must never throw an
 #  error to its caller.
 # -------------------------------------------------------------------------
 ##
proc package::activate {args} {
    foreach pkg $args {
	package::tryToActivate $pkg log
    }
}

proc package::tryToDeactivate {pkg {reporting log}} {
    global index::feature alpha::guiNotReady
    if {[info exists index::feature($pkg)]} {
	set info [set index::feature($pkg)]
	if {[lindex $info 2] == 1} {
	    global $pkg
	    if {![info exists alpha::guiNotReady]} {
		return [try::level \#0 "removeMenu \$$pkg\n[lindex $info 5]\nexpr 0" \
		  -reporting $reporting -while "deactivating $pkg"]
	    }
	}
	return [try::level \#0 "[lindex $info 5]\nexpr 0" \
	  -reporting $reporting -while "deactivating $pkg"]
    } else {
	# This happens if the user completely removes a feature,
	# while some mode still wants to use it (e.g. particularly
	# with menus).
	alertnote "Something is trying to deactivate the '$pkg' feature,\
	  which no longer exists.  I will remove all references to it."
	mode::removeFeatureFromAll $pkg
	return 1
    }
}

proc package::do_activate {pkg} {
    global index::feature index::requirements alpha::guiNotReady \
      package::activationError
    if {[info exists index::requirements($pkg)]} {
	set info [lindex [set index::requirements($pkg)] 1]
	if {[string length $info]} {
	    if {[catch [list uplevel \#0 $info] package::activationError]} {
		append package::activationError " (while requiring $pkg)"
		return 1
	    }
	}
    }
    if {[info exists index::feature($pkg)]} {
	set info [set index::feature($pkg)]
	if {[set init [lindex $info 3]] != ""} {
	    message "Loading package '$pkg'"
	    if {[catch [list uplevel \#0 $init] package::activationError]} {
		append package::activationError " (while initialising $pkg)"
		return 1
	    }
	    set index::feature($pkg) [lreplace [set index::feature($pkg)] 3 3 ""]
	}
	if {[lindex $info 2] == 1} {
	    global $pkg
	    if {![info exists alpha::guiNotReady]} {
		if {[catch [list uplevel \#0 "[lindex $info 4]\ninsertMenu \$$pkg"] package::activationError]} {
		    append package::activationError " (while activating $pkg)"
		    return 1
		} else {
		    return 0
		}
	    }
	}
	if {[catch [list uplevel \#0 [lindex $info 4]] package::activationError]} {
	    append package::activationError " (while activating $pkg)"
	    return 1
	} else {
	    return 0
	}
    } else {
	# This happens if the user completely removes a feature,
	# while some mode still wants to use it (e.g. particularly
	# with menus).
	alertnote "Something is trying to activate the '$pkg' feature,\
	  which no longer exists.  I will remove all references to it."
	mode::removeFeatureFromAll $pkg
	return 1
    }
}

proc package::throwActivationError {} {
    global errorCode errorInfo package::activationError
    if {[dialog::yesno -y "View Error Trace" -n "Continue" \
      "The following problem occurred:\r [set package::activationError]\
      \rthat package will be deactivated if possible."]} {
	alpha::evaluateWhenGuiIsReady \
	  [list new -n {* Error Info *} -m Tcl -shrink -info $errorInfo]
    }
}

proc package::tryToActivate {pkg {reporting log}} {
    global index::feature index::requirements alpha::guiNotReady
    if {[info exists index::requirements($pkg)]} {
	set info [lindex [set index::requirements($pkg)] 1]
	if {[string length $info]} {
	    try::level \#0 $info -reporting $reporting -while "requiring $pkg" 
	}
    }
    if {[info exists index::feature($pkg)]} {
	set info [set index::feature($pkg)]
	if {[set init [lindex $info 3]] != ""} {
	    message "Loading package '$pkg'"
	    try::level \#0 $init -reporting $reporting -while "initialising $pkg" 
	    set index::feature($pkg) [lreplace [set index::feature($pkg)] 3 3 ""]
	}
	if {[lindex $info 2] == 1} {
	    global $pkg
	    if {![info exists alpha::guiNotReady]} {
		return [try::level \#0 "[lindex $info 4]\ninsertMenu \$$pkg\nexpr 0" \
		  -reporting $reporting -while "activating $pkg"]
	    }
	}
	return [try::level \#0 "[lindex $info 4]\nexpr 0" \
	  -reporting $reporting -while "activating $pkg"]
    } else {
	# This happens if the user completely removes a feature,
	# while some mode still wants to use it (e.g. particularly
	# with menus).
	alertnote "Something is trying to activate the '$pkg' feature,\
	  which no longer exists.  I will remove all references to it."
	mode::removeFeatureFromAll $pkg
	return 1
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "package::initialise" --
 # 
 #  Initialises all the packages given.  This procedure must never throw an
 #  error to its caller.
 # -------------------------------------------------------------------------
 ##
proc package::initialise {args} {
    global index::feature
    foreach pkg $args {
	if {[info exists index::feature($pkg)]} {
	    if {[set init [lindex [set index::feature($pkg)] 3]] != ""} {
		message "Loading package '$pkg'"
		try::level \#0 $init -reporting log -while "initialising $pkg" 
		set index::feature($pkg) [lreplace [set index::feature($pkg)] 3 3 ""]
	    }
	} else {
	    # This happens if the user completely removes a feature,
	    # while some mode still wants to use it (e.g. particularly
	    # with menus).
	    alertnote "Something is trying to activate the '$pkg' feature,\
	      which no longer exists.  I will remove all references to it."
	    mode::removeFeatureFromAll $pkg
	}
    }
}

proc package::uninstall {} {
    cache::readContents index::uninstall
    if {![llength [set pkgs [array names index::uninstall]]]} {
	alertnote "I don't know how to uninstall anything."
	return
    }
    set pkgs [listpick -p "Permanently remove which packages/modes/menus?" -l [lsort -ignore $pkgs]]
    if {![llength $pkgs]} { return }
    if {![dialog::yesno "Are you absolutely sure you want to uninstall [join $pkgs {, }]?"]} { 
	return 
    }
    global pkg_file
    foreach pkg $pkgs {
	set pkg_file [lindex [set index::uninstall($pkg)] 1]
	set pkg_file [uplevel \#0 [list subst $pkg_file]]
	set script [lindex [set index::uninstall($pkg)] 2]
	if {[regexp "rm -r\[^\r\n\]*" $script check]} {
	    if {![dialog::yesno "The uninstaller for $pkg contains a\
	      recursive removal command '$check'. Do you want to do this?"]} { 
		return 
	    }
	}
	if {[catch "uplevel \#0 [list $script]"]} {
	    alertnote "The uninstaller for $pkg had problems!"
	}
    }
    if {[dialog::yesno "It is recommended that you quit and restart\
      Alpha.  Quit now?"]} {
	quit
    }
    if {[dialog::yesno "All indices must then be rebuilt.\rShall\
      I do this for you?"]} {
	alpha::rebuildPackageIndices
	rebuildTclIndices
    } else {
	alertnote "This will probably cause problems."
    }
    if {[dialog::yesno "It is recommended that you quit and restart\
      Alpha.  Quit now?"]} {
	quit
    }
}

#  Handle remote menu  #
proc package::menuProc {menu item} {
    global remote::site
    switch -- $item {
	"Describe A Package" {
	    set pkg [dialog::optionMenu "Describe which package?" \
	      [lsort -ignore [alpha::package names]]]
	    package::describe $pkg
	}
	"Read Help For A Package" {
	    set pkg [dialog::optionMenu "Read help for which package?" \
	      [lsort -ignore [alpha::package names]]]
	    package::helpFile $pkg
	}
	"Uninstall Some Packages" {
	    package::uninstall
	}
	"Install A Package" {
	    install::fromRemoteUrl [dialog::getUrl]
	}
	"rebuildPackageIndex" {
	    alpha::rebuildPackageIndices
	}
	"listPackages" {
	    global::listPackages
	}
	"installBugFixesFrom" {
	    # this item isn't in the menu by default anymore.
	    set f [getfile "Select a bug-fix file"]
	    procs::patchOriginalsFromFile $f 1
	}
	"Update List From A Web Archive Site" {
	    package::queryWebForList
	}
	"Ok, Got The List" {
	    package::okGotTheList
	}
	"Add Web Or Ftp Archive Site" {
	    array set remote::site [package::_editSite]
	    prefs::modified remote::site
	}
	"Edit Web Or Ftp Archive Site" {
	    set sitename [dialog::optionMenu "Edit which site?" \
	      [lsort -ignore [array names remote::site]]]
	    
	    array set remote::site \
	      [package::_editSite $sitename [set remote::site($sitename)]]
	    prefs::modified remote::site
	}
	"Remove Web Or Ftp Archive Site" {
	    set sitename [dialog::optionMenu "Remove which site?" \
	      [lsort -ignore [array names remote::site]]]
	    unset remote::site($sitename)
	    prefs::modified remote::site
	}
	"Describe Item" {
	    alertnote "Select one of the packages, and I'll tell you\
	      when it was last modified, and from where it would be downloaded."
	}
	"Ignore Item" {
	    alertnote "'Ignoring' a package tells me to remove it from\
	      new and updated package lists.  It'll still be listed lower\
	      down in the menu"
	}
	"Select Item To Download" {
	    alertnote "Select one of the packages, and it will be\
	      downloaded from its site on the internet, decompressed\
	      and installed."
	}
	default {
	    remote::get $item
	}
    }
    
}


proc package::makeUpdateMenu {} {
    global remote::listing
    set l [list \
      "Update List From A Web Archive Site" \
      "(Ok, Got The List" \
      "<E<SRemove Web Or Ftp Archive Site" \
      "<S<BEdit Web Or Ftp Archive Site" \
      "<SAdd Web Or Ftp Archive Site" "(-" \
      "<S[menu::itemWithIcon {Describe Item} 81]" \
      "<S<U[menu::itemWithIcon {Ignore Item} 81]" \
      "<S[menu::itemWithIcon {Select Item To Download} 81]" ]
    foreach a ${remote::listing} {
	set type [lindex $a 1]
	regsub -all {\.(sea|tar|gz|zip|sit|bin|hqx)} [lindex $a 2] "" name
	lappend [lindex {other gone new uptodate update} [expr {$type + 2}]] $name
	if {$type == -1} {
	    lappend disable $name
	}
    }
    if {[info exists update]} {
	lappend l "(-" "/\x1e(Updated items^[text::Ascii 79 1]"
	eval lappend l [lsort -ignore $update]
    }
    if {[info exists new]} {
	lappend l "(-" "/\x1e(New items^[text::Ascii 79 1]"
	eval lappend l [lsort -ignore $new]
    }
    if {[info exists uptodate]} {
	lappend l "(-" "(Current items"
	eval lappend l [lsort -ignore $uptodate]
    }
    if {[info exists other]} {
	lappend l "(-" "(Other items"
	eval lappend l [lsort -ignore $other]
    }
    if {[info exists gone]} {
	lappend l "(-" "(Vanished items"
	eval lappend l [lsort -ignore $gone]
    }
    Menu -n "internetUpdates" -m -p package::menuProc $l
    if {[info exists disable]} {
	foreach a $disable {
	    enableMenuItem "internetUpdates" $a off
	}
    }
}

proc remote::processftpListing {lines} {
    set files {}
    foreach f [lrange [lreplace $lines end end] 1 end] {
	set nm [lindex $f end]
	if {[string length $nm]} {
	    if {[string match "d*" $f]} {
		#lappend files "$nm/"
	    } else {
		regexp {[A-Z].*$} [lreplace $f end end] time
		set date [lindex $time end]
		if {[regexp : $date] || ![regexp {^19[89][0-5]$} $date]} {
		    # reject anything pre 1996
		    lappend files [list $nm $time]
		}
	    }
	}
    }
    return $files
}

## 
 # -------------------------------------------------------------------------
 # 
 # "remote::processhttpListing" --
 # 
 #  Extract all things like  <A HREF="/~vince/pub/">Parent Directory</A>
 #  followed by a date.  Massage the date into 'Month day year'.
 #  
 #  I don't know if this will work for all http servers!  It works for
 #  mine.
 # -------------------------------------------------------------------------
 ##
proc remote::processhttpListing {lines} {
    set files {}
    foreach f $lines {
	if {[regexp "<A HREF=\"(\[^\"\]*)\">\[^<\]*</A>\[ \t\]*(\[^ \t\]+)\[ \t\]" $f "" name date]} {
	    if {![regexp {/$} $name]} {
		if {![regexp {[89][0-5]$} $date]} {
		    # reject anything pre 1996
		    set date [split $date -]
		    set md "[lindex $date 1] [lindex $date 0] "
		    append md [expr {[lindex $date 2] < 80 ? 20 : 19}]
		    append md [lindex $date 2]
		    lappend files [list $name $md]
		}
	    }
	}
    }
    return $files
}

proc remote::versionOneNewer {one two} {
    return 1
}

proc remote::processList {sitename {l ""}} {
    global remote::listing
    # removed vanished items from the menu
    regsub -all {(\.|-)[0-9]+([a-zA-Z][0-9]+)?} $l "" ll
    foreach i ${remote::listing} {
	if {[string match "*${sitename}*" $i]} {
	    regsub -all {(\.|-)([0-9]+([a-zA-Z][0-9]+)?)} \
	      [set ii [lindex $i 2]] "" ii
	    if {[lsearch -glob $ll "$ii *"] == -1} {
		# it's vanished
		lappend removed $i
		lappend _removed [lindex $i 0]
	    }
	}
    }
    if {[info exists removed]} {
	set remote::listing [lremove -l ${remote::listing} $removed]
    }
    # process new items
    foreach i $l {
	set namepart [lindex $i 0]
	set timepart [lindex $i 1]
	regsub -all {\.(sea|tar|tgz|gz|zip|sit|bin|hqx)} $namepart "" name
	regsub -all {(\.|-|_)[0-9]+([a-zA-Z][0-9]+)?} $name "" name
	if {[set idx [lsearch -glob ${remote::listing} "[quote::Find ${name}] *"]] != -1} {
	    # update old item
	    set item [lindex ${remote::listing} $idx]
	    if {[lindex $item 2] != $namepart} {
		# it's changed
		set item [lreplace $item 1 end 2 $namepart $timepart $sitename]
		set remote::listing [lreplace ${remote::listing} $idx $idx $item]
		lappend _updated $name
	    } elseif {[clock scan [lindex $item 3]] < [clock scan $timepart]} {
		# date is older
		# date has changed
		set item [lreplace $item 1 end 2 $namepart $timepart $sitename]
		set remote::listing [lreplace ${remote::listing} $idx $idx $item]
		lappend _updated $name
	    }
	} else {
	    # new package
	    lappend remote::listing [list $name 0 $namepart $timepart $sitename]
	    lappend _new $name
	}
	
    }
    prefs::modified remote::listing
    package::makeUpdateMenu
    ensureset _updated "none"
    ensureset _new "none"
    ensureset _removed "none"
    if {[catch {alertnote "Remote information, NEW: $_new, UPDATED: $_updated, REMOVED: ${_removed}."}]} {
	alertnote "Remote information, [llength $_new] new, [llength $_updated] updated and [llength $_removed] packages removed."
    }
}
proc remote::updateDatabase {idx val} {
    global remote::listing
    set item [lindex ${remote::listing} $idx]
    if {[lindex $item 1] != $val} {
	# it's changed
	set item [lreplace $item 1 1 $val]
	set remote::listing [lreplace ${remote::listing} $idx $idx $item]
    }
}

proc remote::pkgIndex {name} { 
    global remote::listing
    if {[set i [lsearch -glob ${remote::listing} "[quote::Find ${name}] *"]] == -1} {
	set i [lsearch -glob ${remote::listing} \
	  "[quote::Find [string toupper [string index ${name} 0]][string range $name 1 end]] *"]
    }
    return $i
}

proc remote::pkgDetails {name} { 
    global remote::listing
    set idx [lsearch -glob ${remote::listing} "[quote::Find ${name}] *"]
    return [lindex ${remote::listing} $idx]
}

proc remote::get {pkg} {
    global remote::listing HOME remote::site downloadFolder
    # get pkg
    if {[set idx [remote::pkgIndex $pkg]] == -1} {
	regsub -all {(\.|-|_)[0-9]+([a-zA-Z][0-9]+)?} $pkg "" pkg
	if {[set idx [remote::pkgIndex $pkg]] == -1} {
	    alertnote "Sorry, I don't know from where to download that package."
	    error ""
	}
    }
    set item [lindex ${remote::listing} $idx]
    
    if {[set mods [expr {[getModifiers] & 0xfe}]]} {
	if {$mods & 34} {
	    # just shift key demote the item in the hierarchy
	    set itm [lindex $item 1]
	    if {$itm == 0 || $itm == 2} { set itm 1 } else { set itm -2 }
	    set item [lreplace $item 1 1 $itm]
	    set remote::listing [lreplace ${remote::listing} $idx $idx $item]
	    prefs::modified remote::listing
	    package::makeUpdateMenu
	    message "Package '$pkg' demoted."
	    return
	} else {
	    # describe the item
	    alertnote "File '[lindex $item 2]', last modified [lindex $item 3], to be downloaded from [lindex $item 4], at [set remote::site([lindex $item 4])]"
	    return
	}
    }
    set file [lindex $item 2]
    set sitename [lindex $item 4]
    # get the file
    if {![file exists $downloadFolder] || ![file isdirectory $downloadFolder]} {
	alertnote "Your Download Folder does not exist.  I'll download to Alpha's home directory."
	set downloadFolder $HOME
    }
    
    # update the package database
    remote::updateDatabase $idx 1
    package::makeUpdateMenu
    
    # now download and install
    install::fromRemoteUrl "[set remote::site($sitename)]$file"
}









