## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "prefsHandling.tcl"
 #                                    created: 24/2/95 {9:52:30 pm} 
 #                                last update: 12/10/2001 {19:37:12 PM} 
 #  
 # Reorganisation carried out by Vince Darley with much help from Tom 
 # Fetherston, Johan Linde and suggestions from the alphatcl-developers mailing list.  
 # Alpha is shareware; please register with the author using the register 
 # button in the about box.
 #  
 #  Description: 
 # 
 # Procedures for dealing with the user's preferences
 # ###################################################################
 ##

# These variables should be considered private to this file.
# Use 'prefs::modified' etc.
set modifiedVars	   [list]
set modifiedArrayElements  [list]

proc viewValue {name val} {
    set header "'$name's value is:"
    set response "\r$val\r"
    if {[string length $val] > 80} {
	if {![catch {llength $val}] && (([llength $val] > 3) && ([llength $val] > 6 || [string length $val] > 160))} {
	    listpick -p "'$name's value is:" $val
	} else {
	    if {[tclLog $header$response]} {
		global tileLeft tileTop tileWidth
		new -g $tileLeft $tileTop $tileWidth 100 -n "* $name *" -m Text \
		  -info "'$name's value is:\r\r$val\r"
	    }
	}
    } else {
	global mode
	if {$mode == "Shel"} {
	    goto [maxPos]
	    tclLog $header$response
	    insertText [Alpha::Prompt]
	} else {
	    alertnote "$header\r$response"
	}
    }
}

namespace eval prefs {}

proc prefs::deleteEverything {} {
    prefs::forgetModified 0
    global PREFS
    file delete -force $PREFS
}

## 
 # -------------------------------------------------------------------------
 # 
 # "prefs::modified" --
 # 
 #  Accepts either scalar or array variables, which must have a complete
 #  namespace specification (i.e. they are not calling-namespace-relative).
 #  
 #  Adds the given variables to the list of things to save when the
 #  user 'quits' (or elects to 'save preferences now').
 # -------------------------------------------------------------------------
 ##
proc prefs::modified {args} {
    global modifiedVars modifiedArrayElements
    foreach what $args {
	# root namespace specifier is Very Bad
	# Alpha with Tcl8 crashes on startup because 
	# it automatically prepends a root namespace
	regsub {^::} $what {} what
	
	if {[regexp {([^(]+)\(([^)]+)\)} $what "" arr var]} {
	    lappend modifiedArrayElements [list $var $arr]
	} else {
	    lappend modifiedVars $what
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "prefs::removeObsolete" --
 # 
 #  Use this only for preference variables which are truly obsolete,
 #  and never referenced in code.  It 'unsets' the variables, so that
 #  accessing them again will cause errors.  To forget a users preference
 #  for something (so that it reverts to a default value), you should
 #  use 'prefs::remove' or 'prefs::removeArrayElement' (both of which
 #  can only take effect after a restart).
 # -------------------------------------------------------------------------
 ##
proc prefs::removeObsolete {args} {
    set count 0
    foreach what $args {
	if {[uplevel \#0 info exists [list $what]]} {
	    prefs::modified $what
	    uplevel \#0 unset [list $what]
	    incr count
	}
    }
    return $count
}

## 
 # -------------------------------------------------------------------------
 # 
 # "prefs::renameOld" --
 # 
 #  Useful to allow authors to rename preferences variables without
 #  inconveniencing their users.  Returns 1 if a renaming did take
 #  place (this allows the author to take an action such as telling the
 #  user).
 # -------------------------------------------------------------------------
 ##
proc prefs::renameOld {from to} {
    if {[uplevel \#0 info exists [list $from]]} {
	prefs::modified $from
	prefs::modified $to
	uplevel \#0 set [list $to [uplevel \#0 set [list $from]]]
	uplevel \#0 unset [list $from]
	return 1
    } else {
	return 0
    }
}

proc prefs::modifiedVar {args} {
    global modifiedVars
    eval lappend modifiedVars $args
}

proc prefs::modifiedModeVar {var {m ""}} {
    global modifiedArrayElements mode
    if {$m == ""} { set m $mode }
    lappend modifiedArrayElements [list $var ${m}modeVars]
}

proc prefs::modifiedArrayElement {var arr} {
    global modifiedArrayElements
    lappend modifiedArrayElements [list $var $arr]
}

proc prefs::add {def val {prefix {}}} {
    global ${prefix}prefDefs
    
    prefs::_read $prefix
    set ${prefix}prefDefs($def) $val
    prefs::_write $prefix
    catch {unset ${prefix}prefDefs}
}

proc prefs::remove {def {prefix {}}} {
    global ${prefix}prefDefs
    
    prefs::_read $prefix
    catch {unset ${prefix}prefDefs($def)}
    prefs::_write $prefix
    catch {unset ${prefix}prefDefs}
}

proc prefs::addArrayElement {arr def val} {
    prefs::add [list $arr $def] $val arr
}

proc prefs::removeArrayElement {arr def} {
    prefs::remove [list $arr $def] arr
}

proc prefs::removeArray {arr} {
    global arrprefDefs $arr
    
    prefs::_read arr
    foreach def [array names $arr] {
	catch {unset arrprefDefs([list $arr $def])}
    }
    prefs::_write arr
    catch {unset arrprefDefs}
}

proc prefs::addArray {arr} {
    global arrprefDefs $arr
    
    prefs::_read arr
    # Remove all old entries.  We have to do this because the
    # code just after will only update existing entries, so old
    # array elements which we no longer want will never disappear.
    foreach r [array names arrprefDefs] {
	if {[lindex $r 0] == $arr} {
	    unset arrprefDefs($r)
	}
    }
    foreach def [array names $arr] {
	catch {set arrprefDefs([list $arr $def]) [set ${arr}($def)]}
    }
    prefs::_write arr
    catch {unset arrprefDefs}
}

proc prefs::loadEarlyConfiguration {} {
    if {[cache::exists configuration]} {
	set filename [cache::name configuration]
	set contents [file::readAll $filename]
	prefs::_interpretPrefsContent $contents
	global alpha::home HOME alpha::homeChanged
	if {[info exists alpha::home] && (${alpha::home} != $HOME)} {
	    set alpha::homeChanged 1
	} else {
	    set alpha::homeChanged 0
	}
    }
}

if {![info exists alpha::homeChanged]} {
    set alpha::homeChanged 0
}

proc prefs::homeChanged {} {
    global alpha::homeChanged
    set alpha::homeChanged
}

proc prefs::updateHome {var {type "path"}} {
    global alpha::home HOME alpha::homeChanged
    if {![set alpha::homeChanged]} {return}
    upvar 1 $var local
    if {![info exists local]} { return }
    
    if {[array exists local]} {
	switch -- $type {
	    "name" {
		foreach item [array names local] {
		    if {[file::pathStartsWith $item [set alpha::home]]} {
			set new "${HOME}[string range $item [string length [set alpha::home]] end]"
			set local($new) $local($item)
			unset local($item)
			prefs::modified ${var}($item)
		    }
		}
	    }
	    "list" {
		foreach name [array names local] {
		    set count 0
		    foreach item $local($name) {
			if {[file::pathStartsWith $item [set alpha::home]]} {
			    set item "${HOME}[string range $item [string length [set alpha::home]] end]"
			    set local($name) [lreplace $local($name) $count $count $item]
			    prefs::modified ${var}($name)
			}
			incr count
		    }
		}
	    }
	    "path" {
		foreach name [array names local] {
		    set item $local($name)
		    if {[file::pathStartsWith $item [set alpha::home]]} {
			set local($name) "${HOME}[string range $local($name) [string length [set alpha::home]] end]"
			prefs::modified ${var}($name)
		    }
		}
	    }
	    default {
		return -code error "Bad type '$type' to prefs::updateHome"
	    }
	}
    } else {
	# Convert the value of this variable
	switch -- $type {
	    "list" {
		set count 0
		foreach item $local {
		    if {[file::pathStartsWith $item [set alpha::home]]} {
			set item "${HOME}[string range $item [string length [set alpha::home]] end]"
			set local [lreplace $local $count $count $item]
			prefs::modified $var
		    }
		    incr count
		}
	    }
	    "path" {
		if {[file::pathStartsWith $local [set alpha::home]]} {
		    set local "${HOME}[string range $local [string length [set alpha::home]] end]"
		    prefs::modified $var
		}
	    }
	    default {
		return -code error "Bad type '$type' to prefs::updateHome"
	    }
	}
    }
}

# We'll get rid of this procedure when everyone has upgraded enough.
# It is only necessary for prefs saved in some dev. releases. 
proc prefs::_interpretPrefsContent {contents} {
    global HOME tcl_platform
    regsub -all "!%HOME%!" $contents [quote::Regsub $HOME] contents
    if {$tcl_platform(platform) == "windows"} {
	# Cope with forward and backslash file separators
	regsub -all "%!HOME!%" $contents \
	  [quote::Regsub [file nativename $HOME]] contents
    }
    uplevel \#0 $contents
}

proc prefs::_read {{prefix {}}} {
    global PREFS
    set filename [file join $PREFS ${prefix}defs.tcl]
    if {![file exists $filename]} return
    
    # Read the contents, but making it HOME-independent
    if {[catch {file::readAll $filename} contents]} {
	alertnote "Your preferences file '${prefix}defs.tcl'\
	  is corrupt; I will use the backup copy instead"
	file delete $filename
	file copy [file join $PREFS backup${prefix}defs.tcl] \
	  $filename
	set contents [file::readAll $filename]
    }
    prefs::_interpretPrefsContent $contents
}

proc prefs::_write {{prefix {}}} {
    global HOME PREFS ${prefix}prefDefs tcl_platform
    
    if {![info exists ${prefix}prefDefs]} {
	catch {file delete [file join $PREFS ${prefix}defs.tcl]}
	return
    }
    
    if {![file exists "$PREFS"]} {
	file mkdir "$PREFS"
    }
    set filename [file join $PREFS ${prefix}defs.tcl]
    set fd [alphaOpen $filename "w"]
    foreach nm [array names ${prefix}prefDefs] {
	puts $fd [list set ${prefix}prefDefs($nm) [set ${prefix}prefDefs($nm)]]
    }
    close $fd
}


# So we are picked up by standard auto_mkindex
proc prefs::readAll {} {}

if {[info tclversion] < 8.0} {
    proc prefs::readAll {} {
	global prefDefs arrprefDefs
	
	catch {prefs::_read}
	foreach nm [array names prefDefs] {
	    global $nm
	    set $nm $prefDefs($nm)
	}
	catch {unset prefDefs}
	
	catch {prefs::_read arr}
	foreach nm [array names arrprefDefs] {
	    set arr [lindex $nm 0]
	    set field [lindex $nm 1]
	    set val $arrprefDefs($nm)
	    global $arr
	    set ${arr}($field) $val
	}
	catch {unset arrprefDefs}
    }
} else {
    proc prefs::readAll {} {
	global prefDefs arrprefDefs
	    
	catch {prefs::_read}
	foreach nm [array names prefDefs] {
	    ensureNamespaceExists ::$nm
	    global ::$nm
	    set ::$nm $prefDefs($nm)
	}
	catch {unset prefDefs}
	
	catch {prefs::_read arr}
	foreach nm [array names arrprefDefs] {
	    set arr [lindex $nm 0]
	    set field [lindex $nm 1]
	    set val $arrprefDefs($nm)
	    ensureNamespaceExists ::$arr
	    global ::$arr
	    set ::${arr}($field) $val
	}
	catch {unset arrprefDefs}
    }

}


proc prefs::tclRead {} {
    global PREFS
    # Use "prefs.tcl" to define or change any tcl information. 
    if {![file exists [file join $PREFS prefs.tcl]]} {
	if {![file exists "$PREFS"]} {
	    file mkdir "$PREFS"
	}
	close [open [file join $PREFS prefs.tcl] "w"]
    }
    uplevel #0 {
	if {[catch {source [file join $PREFS prefs.tcl]}]} {
	    if {[dialog::yesno "An error occurred while loading \"prefs.tcl\".  Shall I make a trace on the error?"]} {
		dumpTraces "prefs.tcl error" $errorInfo
	    }
	}
    }
}

	
proc prefs::viewSavedSetting {} {
    global prefDefs arrprefDefs

    prefs::saveModified

    set title    "The following settings have been saved:"
    set settings [listpick -p $title -l [prefs::listAllSaved]]

    foreach setting $settings {
	if {[regexp {([^(]+)\(([^)]+)\)} $setting "" arr field]} {
	    set arg [list $arr $field]
	    set val $arrprefDefs($arg)
	} else {
	    global $setting
	    set val $prefDefs($setting)
	}
	viewValue $setting $val
    }
    catch {unset prefDefs}
    catch {unset arrprefDefs}
    message ""
}

## 
 # -------------------------------------------------------------------------
 # 
 # "removeSavedSetting" --
 # 
 #  This proc shouldn't 'unset' the variables it removes, because most
 #  such variables will be in use/have default values until restart.
 # -------------------------------------------------------------------------
 ##
proc prefs::removeSavedSetting {} {
    global prefDefs arrprefDefs

    prefs::saveModified

    set title    "Remove which settings?"
    set settings [listpick -p $title -l [prefs::listAllSaved]]

    foreach setting $settings {
	if {[regexp {([^(]+)\(([^)]+)\)} $setting "" arr field]} {
	    global $arr
	    prefs::removeArrayElement $arr $field
	} else {
	    global $setting
	    prefs::remove $setting
	}
    }
    catch {unset prefDefs}
    catch {unset arrprefDefs}
    message "The saved settings will be removed when you quit."
}


proc prefs::listAllSaved {} {
    global prefDefs arrprefDefs
    
    prefs::_read
    prefs::_read arr
    
    set names [array names prefDefs]
    foreach pair [array names arrprefDefs] {
	lappend names "[lindex $pair 0]([lindex $pair 1])"
    }
    
    return [lsort $names]
}

#===============================================================================

proc prefs::findEncoding {} {
    global PREFS
    set enc [file join $PREFS encoding.txt]
    if {[file exists $enc]} {
	return [string trim [file::readAll $enc]]
    } else {
	global alpha::defaultEncoding
	return ${alpha::defaultEncoding}
    }
}

proc prefs::tclEdit {} {
    global PREFS
    if {![file exists [file join $PREFS prefs.tcl]]} {
	close [open [file join $PREFS prefs.tcl] "w"]
    }
    edit [file join $PREFS prefs.tcl]
}

# Automatically add a line to the user input file
proc prefs::tclAddLine {line} {
    global PREFS
    
    if {![file exists "$PREFS"]} {
	file mkdir "$PREFS"
    }

    set f [file join $PREFS prefs.tcl]
    set notOpen [catch [list bringToFront $f]]

    if {$notOpen} {
	set fid [alphaOpen $f "a+"]
	if {![catch {seek $fid -1 end}]} {
	    if {![is::Eol [read $fid 1]]} {
		set line "\r$line"
	    }
	}
	seek $fid 0 end
	puts $fid $line
	close $fid
    } else {
	set p [getPos]
	goto [maxPos]
	if {[lindex [posToRowCol [getPos]] 1] != 0} {
	    set line "\r$line"
	}
	insertText $line
	save
	goto $p
	message "Your prefs.tcl was saved with the necessary changes"
    }
}

# Automatically add a line to a mode's pref file -trf
proc prefs::tclAddModeLine {line} {
    global PREFS mode
    
    if {![file exists "$PREFS"]} {
	file mkdir "$PREFS"
    }
    set fid [alphaOpen [file join $PREFS ${mode}prefs.tcl] "a+"]
    if {![catch {seek $fid -1 end}]} {
	if {![is::Eol [read $fid 1]]} {
	    set line "\r$line"
	}
    }
    seek $fid 0 end
    puts $fid $line
    close $fid
}

proc prefs::saveNow {} {
    message "Saving preferences"
    global modifiedVars modifiedArrayElements global::features \
      alpha::earlyPrefs skipPrefs
    
    if {[info exists skipPrefs] && ($skipPrefs != 0)} {
	if {($skipPrefs == 2) || [askyesno "You skipped loading\
	  your saved preferences\
	  at startup.  Do you wish to save the current configuration?\
	  (it will permanently over-write the old preferences)"] != "yes"} {
	    return
	}
    }
    
    if {0 && ([info tclversion] > 8.1)} {
	# Output our current encoding.
	global PREFS
	set fout [alphaOpen [file join $PREFS encoding.txt] w]
	puts $fout [fconfigure $fout -encoding]
	close $fout
    }
    
    global HOME alpha::home
    set alpha::home $HOME
    cache::delete configuration
    cache::add configuration list global::features
    cache::add configuration variable alpha::home
    
    if {[info exists alpha::earlyPrefs]} {
	foreach f [set alpha::earlyPrefs] {
	    global $f
	    if {[info exists $f]} {
		cache::add configuration variable $f
	    }
	}
    } else {
	set alpha::earlyPrefs {}
    }

    # Copy over this old array.  It is obsolete and should not be used
    # It will be removed soon.
    global modifiedArrVars
    if {[info exists modifiedArrVars]} {
	eval lappend modifiedVars $modifiedArrVars
	unset modifiedArrVars
    }

    # Start of main preference saving
    global prefDefs arrprefDefs

    prefs::_read
    prefs::_read arr

    foreach f [lunique $modifiedVars] {
	if {[lsearch -exact [set alpha::earlyPrefs] $f] == -1} {
	    global $f
	    if {[array exists $f]} {
		# prefs::addArray $f
		foreach r [array names arrprefDefs] {
		    if {[lindex $r 0] == $f} {
			unset arrprefDefs($r)
		    }
		}
		foreach def [array names $f] {
		    catch {set arrprefDefs([list $f $def]) [set ${f}($def)]}
		}
	    } else {
		if {[info exists $f]} {
		    # prefs::add $f [set $f]
		    set prefDefs($f) [set $f]
		} else {
		    # prefs::remove $f
		    catch {unset prefDefs($f)}
		}
	    }
	}
    }
    # these two lists actually behave identically
    foreach f [lunique $modifiedArrayElements] {
	set elt [lindex $f 0]
	set arr [string trimleft [lindex $f 1] :]
	global $arr
	if {[info exists [set arr]($elt)]} {
	    # prefs::addArrayElement [set arr] $elt [set [set arr]($elt)]
	    set arrprefDefs([list $arr $elt]) [set [set arr]($elt)]
	} else {
	    # prefs::removeArrayElement [set arr] $elt
	    catch {unset arrprefDefs([list $arr $elt])}
	}
    }
    
    prefs::_write
    prefs::_write arr

    unset prefDefs arrprefDefs
    # End of main preference saving
    
    # Make backups.
    global PREFS
    # Use -force to overwrite existing file.
    file copy -force [file join $PREFS arrdefs.tcl] [file join $PREFS backuparrdefs.tcl]
    file copy -force [file join $PREFS defs.tcl] [file join $PREFS backupdefs.tcl]
    message "Preferences saved"
}

proc prefs::saveModified {} {
    prefs::forgetModified 1
}

proc prefs::forgetModified {{save 1}} {
    global modifiedVars modifiedArrayElements
    if {$save} {
	prefs::saveNow
    }
    set modifiedVars [list]
    set modifiedArrayElements [list]
}

#===============================================================================

namespace eval mode {}

## 
 # -------------------------------------------------------------------------
 # 
 # "mode::sourcePrefsFile" --
 # 
 #  Fixes 'uplevel #0' problem
 # -------------------------------------------------------------------------
 ##
proc mode::sourcePrefsFile {} { 
    global mode PREFS
    if {[file exists [file join ${PREFS} ${mode}Prefs.tcl]]} {
	uplevel #0 [list source [file join ${PREFS} ${mode}Prefs.tcl]]
    } else {
	beep; message "Sorry, no preferences file for '$mode' mode"
    }
}

proc mode::editPrefsFile {{m ""}} { 
    global PREFS mode
    if {$m == ""} { set m $mode }
    message $m
    # assume it is a mode, since we made the menu
    
    set f [file join $PREFS ${m}Prefs.tcl]
    if {[file exists $f]} {
	edit $f
    } else {
	if {[dialog::yesno "No '$m' prefs file exists, do you want to create one?"]} {
	    close [open $f "w"]
	    edit $f
	    insertText {
## 
 # This	file will be sourced automatically, immediately after 
 # the _first_ time the file which defines its mode is sourced.
 # Use this file to insert your own mode-specific preferences
 # and changes,	rather than altering the originals.
 # 
 # You can redefine menus, procedures, variables,...
 ##

	}}}
	
    hook::callAll mode::editPrefsFile
}

