## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "search.tcl"
 #                                    created: 13/6/95 {8:56:37 pm} 
 #                                last update: 11/13/2001 {14:08:45 PM} 
 #  
 # Reorganisation carried out by Vince Darley with much help from Tom 
 # Fetherston, Johan Linde and suggestions from the alphatcl-developers mailing list.  
 # Alpha is shareware; please register with the author using the register 
 # button in the about box.
 #  
 #  Description: 
 # 
 # Various procedures which deal with search/reg-search/grep type stuff
 # in Alpha.
 # ###################################################################
 ##

#================================================================================
# 'greplist' and 'grepfset' are used for batch searching from the "find" dialog.
#  Hence, you really shouldn't mess with them unless you know what you are doing.
#================================================================================
proc greplist {args} {
    global tileLeft tileTop tileWidth tileHeight errorHeight
    
    set recurse [lindex $args 0]
    set word [lindex $args 1]
    set args [lrange $args 2 end]
    
    set num [expr {[llength $args] - 2}]
    set exp [lindex $args $num]
    set arglist [lindex $args [expr {$num + 1}]]
    
    set opened 0
    set cid [scancontext create]
    
    set cmd [lrange $args 0 [expr {$num - 1}]]
    eval scanmatch $cmd {$cid $exp {
	if {!$word || [regexp -nocase -- "(^|\[^a-zA-Z0-9\])${exp}(\[^a-zA-Z0-9\]|\$)" $matchInfo(line)]} {
	    if {!$opened} {
		set opened 1
		win::SetProportions
		set w [new -n {* Batch Find *} -m Brws -g $tileLeft $tileTop $tileWidth $errorHeight -tabsize 8]
		insertText "(<cr> to go to match)\r-----\r"
	    }
	    set l [expr {20 - [string length [file tail $f]]}]
	    regsub -all "\t" $matchInfo(line) "  " text
	    insertText -w $w "\"[file tail $f]\"[format "%$l\s" ""]; Line $matchInfo(linenum): ${text}\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$f\r"}
	}
    }
    
    foreach f $arglist {
	message [file tail $f]
	if {![catch {set fid [alphaOpen $f]}]} {
	    scanfile $cid $fid
	    close $fid
	}
    }
    scancontext delete $cid
    
    if {$opened} {
	select [nextLineStart [nextLineStart [minPos]]] [nextLineStart [nextLineStart [nextLineStart [minPos]]]]
	setWinInfo dirty 0
	setWinInfo read-only 1
    }
    message ""
}


## 
 # -------------------------------------------------------------------------
 # 
 # "grepfset" --
 # 
 #  args: wordmatch ?-nocase? expression fileset
 #  Obviously we ignore wordmatch
 #  
 #  If the 'Grep' box was set, then the search item is _not_ quoted.
 #  
 #  Non grep searching problems:
 #  
 #  If it wasn't set, then some backslash quoting takes place. 
 #  (The chars: \.+*[]$^ are all quoted)
 #  Unfortunately, this latter case is done incorrectly, so most
 #  non-grep searches which contain a grep-sensitive character fail.
 #  The quoting should use the equivalent of the procedure 'quote::Regfind'
 #  but it doesn't quote () and perhaps other important characters.
 #  
 #  Even worse, if the string contained any '{' it never reaches this
 #  procedure (there must be an internal error due to bad quoting).
 # 
 # -------------------------------------------------------------------------
 ##
proc grepfset {args} {
    set num [expr {[llength $args] - 2}]
    # the 'find' expression
    set exp [lindex $args $num]
    # the fileset
    set fset [lindex $args [expr {$num + 1}]]
    eval greplist 0 [lrange $args 0 [expr {$num-1}]] {$exp [getFileSet $fset]}
}

proc grep {exp args} {
    set files {}
    foreach arg $args {
	eval [list lappend files] [glob -types TEXT -nocomplain -- $arg]
    }
    if {![llength $files]} {return "No files matched pattern"}
    set cid [scancontext create]
    scanmatch $cid $exp {
	if {!$blah} {
	    set blah 1
	    set lines "(<cr> to go to match)\n"
	}
	set l [expr {20 - [string length [file tail $f]]}]
	regsub -all "\t" $matchInfo(line) "  " text
	append lines "\"[file tail $f]\"[format "%$l\s" ""]; Line $matchInfo(linenum): ${text}\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$f\n"
    }
    
    set blah 0
    set lines ""
    
    foreach f $files {
	if {![catch {set fid [alphaOpen $f]}]} {
	    message [file tail $f]
	    scanfile $cid $fid
	    close $fid
	}
    }
    scancontext delete $cid
    return [string trimright $lines "\r"]
}

proc grepnames {exp args} {
    set files {}
    foreach arg $args {
	eval lappend files [glob -types TEXT -nocomplain -- $arg]
    }
    if {![llength $files]} {return "No files matched pattern"}
    set cid [scancontext create]
    scanmatch $cid $exp {
	lappend filenames $f
    }
    set filenames ""
    foreach f $files {
	if {![catch {set fid [alphaOpen $f]}]} {
	    message [file tail $f]
	    scanfile $cid $fid
	    close $fid
	}
    }
    scancontext delete $cid
    return $filenames
}

## 
 # -------------------------------------------------------------------------
 # 
 # "performSearch" --
 # 
 #  Call this procedure in Tcl code which wants to use the standard procs
 #  like 'replaceAll' to ensure flags like multi-file batch replace are
 #  cleared.  Otherwise replaceAll might not have the desired effect.
 #  
 #  Note that, if you are using this procedure followed by things
 #  like 'replace', you almost certainly do not want to use the '-s' flag.
 #  A rule which is probably true is that
 #  
 #  (i) search should never use '-s'
 #  (ii) performSearch should usually use '-s'
 #  
 #  This proc may be overridden by code (such as supersearch).
 # -------------------------------------------------------------------------
 ##
proc performSearch {args} {
    eval [list select] [uplevel 1 search $args]
}

proc findBatch {forward ignore regexp word pat} {
    matchingLines $pat $forward $ignore $word $regexp 
}

## 
 # -------------------------------------------------------------------------
 #	 
 #  "findPatJustBefore" --
 #	
 #  Utility proc to check whether the first occurrence of 'findpat' to
 #  the left of 'pos' is actually an occurrence of 'pat'.  It can be
 #  used to check if we're part of an '} else {' (see TclelectricLeft)
 #  or in TeX mode if we're in the argument of a '\label{' or '\ref{'
 #  (see smartScripts) for example.
 #	 
 #  A typical usage has the regexp 'pat' end in '$', so that it must
 #  match all the text up to 'pos'.  'matchw' can be used to store the
 #  first '()' pair match in the regexp.
 #	 
 #  New: maxlook restricts how far this proc will search.  The default
 #  is only 100 (not the entire file), after all this proc is supposed
 #  to look 'just before'!
 # -------------------------------------------------------------------------
 ##
proc findPatJustBefore { findpat pat {pos ""} {matchw ""} {maxlook 100} } {
    if { $pos == "" } {set pos [getPos] }
    if {[pos::compare $pos == [maxPos]]} { set pos [pos::math $pos - 1]}
    if { $matchw != "" } { upvar $matchw word }
    if {[llength [set res [search -s -n -f 0 -r 1 -l [pos::math $pos - $maxlook] -- "$findpat" $pos]]]} {
	if {[regexp -- "$pat" [getText [lindex $res 0] $pos] dum word]} {
	    return [lindex $res 0]
	}
    }
    return
}
# Look for pattern in filename after position afterPos and, if found, 
# open the file quietly and select the pattern
# author Jonathan Guyer
proc selectPatternInFile {filename pattern {afterPos ""}} {
    if {$afterPos == ""} {set afterPos [minPos]}
    set searchResult [file::searchFor $filename $pattern 1]
    if {[pos::compare [lindex $searchResult 0] >= $afterPos]} {
	placeBookmark
	file::openQuietly $filename
	eval select $searchResult
	message "press <Ctrl .> to return to original cursor position"
	return 1
    } else {
	return 0
    }
}

proc searchStart {} {
    global search_start
    select [getPos]
    setMark
    if {[catch {goto $search_start}]} {message "No previous search"}
}

set {patternLibrary(Pascal to C Comments)}  	{ {\{([^\}]*)\}}	{/* \1 */} }
set {patternLibrary(C++ to C Comments)}		{ {//(.*)}	{/* \1 */} }
set {patternLibrary(Space Runs to Tabs)}	{ { +}	{\t}}

proc getPatternLibrary {} {
    global patternLibrary
    
    foreach nm [array names patternLibrary] {
	lappend nms [concat [list $nm] $patternLibrary($nm)]
    }
    return $nms
}

# This fails if, say, search string is '\{[^}]'
# This is because the '}' ends the first argument because this
# procedure is presumably called internally with incorrect quoting.
proc rememberPatternHook {search replace} {
    global patternLibrary
    if {[catch {set name [prompt "New pattern's name?" ""]}]} {
	return ""
    }
    prefs::modified patternLibrary($name)
    set patternLibrary($name) [list $search $replace]
    return $name
}

proc deletePatternHook {} {
    global patternLibrary
    set temp [list prompt "Delete which pattern?" [lindex [array names patternLibrary] 0] "Pats:"]
    set name [eval [concat $temp [array names patternLibrary]]]
    prefs::modified patternLibrary($name)
    unset patternLibrary($name)
}

proc nextFunc {} {
    mode::proc searchFunc 1
}

proc prevFunc {} {
    mode::proc searchFunc 0
}

proc ::searchFunc {dir} {
    global funcExpr mode
    global ${mode}modeVars
    
    if {![info exists ${mode}modeVars(funcExpr)]} {
	# for modes that have no functions, just use filemarks
	findViaFileMarks $dir
	return
    }
	
    set pos [getPos]
    select $pos $pos
    
    if {$dir} {
	set pos [pos::math $pos + 1]
	set lastStop [maxPos]
    } else {
	set pos [pos::math $pos - 1]
	set lastStop [minPos]
    }
    if {![catch {search -s -f $dir -i 1 -r 1 -- $funcExpr $pos} res]} {
	eval select $res
    } else {
	goto $lastStop
	if {$dir} {
	    message "At bottom, no more functions in this direction"
	} else {
	    message "At top, no more functions in this direction"
	}
    }
}

#===========================================================================
# Juan Falgueras (7/Abril/93)
# you only need to select (or not) text and move *forward and backward*
# faster than iSearch (if you have there the |word wo|rd..).
#===========================================================================

proc quickSearch {dir} {
    if {[pos::compare [selEnd] == [getPos]]} {
	backwardChar
	hiliteWord
    }
    set myPos [expr {$dir ? [selEnd] : [pos::math [getPos] - 1]}]
    set text [getSelect]
    set searchResult [search -s -n -f $dir -m 0 -i 1 -r 0 $text $myPos]
    if {[llength $searchResult] == 0} {
	beep
	message [concat [expr {$dir ? "->" : "<-"}] '$text' " not found"]
	return 0
    } else {
	message [concat [expr {$dir ? "->" : "<-"}] '$text']
	eval select $searchResult
	return 1
    }
}

