####################################################################
#  AlphaTcl - core Tcl engine
# 
# Much by Vince Darley.
# 
#                                    created: 26/11/96 {7:08:34 pm} 
#                                last update: 02/19/2001 {16:04:15 PM}
#  Author: Vince Darley
#  E-mail: <vince@santafe.edu>
#    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
#     www: <http://www.santafe.edu/~vince/>
#  
####################################################################

## 
 #	   
 # 'rememberWhereYouAre'
 # 
 #  Given the start of a paragraph and the point to remember, this returns a
 #  record which must be passed to the following function so that it can
 #  find the spot later, even after the paragraph has had
 #  space/tabs/new-lines meddled with.  An optional last argument is a list
 #  of other characters quoted so they are regexp insensitive, which should
 #  also be ignored.  This is used so we can remember positions in text
 #  which has cosmetic characters on the left/right which are not wrapped
 #  (such as the hashes to the left here!).
 #	   
 # 'goBackToWhereYouWere'
 # 
 #  Given the beginning and end of a selection, and a previous record, where
 #  the beginning, and record correspond to a previous call of
 #  'rememberWhereYouAre', this procedure will move the insertion point to
 #  the correct place.
 #	   
 ##

proc rememberWhereYouAre {startPara pos endPara {commentReg ""}} {
    set start [pos::math $pos -20]
    if {[pos::compare $start < $startPara]} {
	set start $startPara
    }
    set __g_remember_str [getText $start $pos]
    if {[string length [string trim $__g_remember_str]] < 3} {
	# there wasn't much to remember; try the other way
	set end [pos::math $pos +20]
	if {[pos::compare $end > $endPara]} {
	    set end $endPara
	}
	set __g_remember_str [getText $pos $end]
	set __g_remember_dir 0
    } else {
	set __g_remember_dir 1
    }
    
    set __g_remember_str [quote::Regfind $__g_remember_str]
    regsub -all "\[ \t\r\n${commentReg}\]+" $__g_remember_str \
      {[ \t\r\n${commentReg}]+} __g_remember_str
    return [list $__g_remember_str $__g_remember_dir]
}

proc goBackToWhereYouWere {start end memory} {
    if {[lindex $memory 0] != "" } {
	regexp -indices ".*([lindex $memory 0]).*" [getText $start $end] \
	  "" submatch
	if {[info exists submatch]} {
	    set p [pos::math $start + [lindex $memory 1] + \
	      [lindex $submatch [lindex $memory 1]]]
	} else {
	    set p $end
	}
	goto [expr {[pos::compare $p >= $end] ? [pos::math $end - 1] : $p}]
    } else {
	goto $start
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"getLeadingIndent" --
 #	
 #  Find the indentation of the line containing 'pos', and convert it to a
 #  minimal form of tabs followed by spaces.  If 'size' is given, then the
 #  variable of that name is set to the length of the indent.  Similarly
 #  'halftab' can be set to half a tab. 
 # -------------------------------------------------------------------------
 ##
proc getLeadingIndent { pos {size ""} {halftab ""} } {
    # get the leading whitespace of the current line
    set res [search -s -n -f 1 -r 1 "^\[ \t\]*" [lineStart $pos]]
    
    # convert it to minimal form: tabs then spaces, stored in 'front'
    getWinInfo a
    set sp [string range "              " 1 $a(tabsize) ]
    regsub -all "($sp| +\t)" [eval getText $res] "\t" front
    if { $size != "" } {
	upvar $size ind
	# get the length of the indent
	regsub -all "\t" $front $sp lfront
	set ind [string length $lfront]
    }
    if { $halftab != "" } {
	upvar $halftab ht
	# get the length of half a tab
	set ht [string range "            " 1 [expr {$a(tabsize)/2}]]
    }
    
    return $front
}


proc getEndpts {} {
    if {[pos::compare [getPos] == [selEnd]]} {
	set start [getPos]
	set finish [getMark]
	if {[pos::compare $start > $finish]} {
	    set temp $start
	    set start $finish
	    set finish $temp
	}
    } else {
	set start [getPos]
	set finish [selEnd]
    }
    return [list $start $finish]
}


proc fillRegion {} {
    global leftFillColumn
    set ends [getEndpts]
    set start [lineStart [lindex $ends 0]]
    set finish [lindex $ends 1]
    goto $start
    set text [fillText $start $finish]
    replaceText $start $finish [format "%$leftFillColumn\s" ""] $text "\r"
}
	
proc wrapParagraph {} {
    set pos [getPos]
    set start [paragraph::start $pos] 
    set finish [paragraph::finish $pos]
    goto $start
    wrapText $start $finish
    goto $pos
}

proc wrapRegion {} {
    set ends [getEndpts]
    set start [lineStart [lindex $ends 0]]
    set finish [lindex $ends 1]
    if {[pos::compare $start == $finish]} {
	set finish [maxPos]
    }
    wrapText $start $finish
}
	


# Remove text from window, transform, and insert back into window.
proc fillText {from to} {
    global doubleSpaces
    set text [getText $from $to]
    regexp "^\[ \t\]*" $text front
    regsub -all "\[ \t\n\r\]+" [string trim $text] " " text
    if {$doubleSpaces} {regsub -all {(([^.][a-z]|[^a-zA-Z@]|\\@)[.?!]("|'|'')?([])])?) } $text {\1  } text}
    regsub -all " ?\[\r\n\]" [string trimright [breakIntoLines $text]] "\r${front}" text
    return $front$text
}

proc paragraphToLine {} {
    global fillColumn
    global leftFillColumn
    set fc $fillColumn
    set lc $leftFillColumn
    set fillColumn 10000
    set leftFillColumn 0
    fillRegion
    set fillColumn $fc
    set leftFillColumn $lc
}

proc lineToParagraph {} {
    global fillColumn
    global leftFillColumn
    set fc $fillColumn
    set fillColumn 75
    set lc $leftFillColumn
    set leftFillColumn 0
    fillRegion
    set fillColumn $fc
    set leftFillColumn $lc
}


#set sentEnd {[.!?](\r|\n| +)}
set sentEnd {(\r\r|\n\n|[.!?](\r|\n| +))}
set sentBeg {[\r\n ][A-Z]}

proc sentenceRegion {} {
    set ends [getEndpts]
    set start [lineStart [lindex $ends 0]]
    set finish [lindex $ends 1]
    if {[pos::compare $start == $finish]} {
	set finish [maxPos]
    }
    set t [string trim [getText $start $finish]]
    set period [regexp {\.$} $t]
    regsub -all "\[ \t\r\n\]+" $t " " text
    regsub -all {\. } $text "" text
    set result ""
    foreach line [split [string trimright $text {.}] ""] {
	if {[string length $line]} {
	    append result [breakIntoLines $line] ".\r"
	}
    }
    if {!$period && [regexp {\.\r} $result]} {
	set result [string trimright $result ".\r"]
	append result "\r"
    }
    if {$result != [getText $start $finish]} {
	replaceText $start $finish $result
    }
    goto $pos
}

proc nextSentence {} {
    global sentBeg sentEnd
    if {![catch {search -s -f 1 -r 1 $sentEnd [getPos]} mtch]} {
	if {![catch {search -s -f 1 -r 1 -i 0 $sentBeg [pos::math [lindex $mtch 1] - 1]} mtch]} {
	    goto [pos::math [lindex $mtch 0] + 1]
	}
    }
}


proc prevSentence {} {
    global sentBeg sentEnd
    if {[catch {search -s -f 0 -r 1 $sentBeg [pos::math [getPos] - 2]} mtch]} return
    if {![catch {search -s -f 0 -r 1 $sentEnd [lindex $mtch 1]} mtch]} {
	if {![catch {search -s -f 1 -r 1 -i 0 $sentBeg [pos::math [lindex $mtch 1] - 1]} mtch]} {
	    goto [pos::math [lindex $mtch 0] + 1]
	}
    }
}

#===============================================================================
# Called by Alpha to do "soft wrapping"
proc softProc {pos start next} {
    global leftFillColumn
    goto $start
    set finish [paraFinish $start]
    set text [fillText $start $finish]
    if {"${text}\r" != [getText $start $finish]} {
	replaceText $start $finish [format "%$leftFillColumn\s" ""] $text "\r"
	return 1
    } else {
	return 0
    }
}

proc dividingLine {} {
    global mode
    global ${mode}modeVars
    if {[info exists ${mode}modeVars(prefixString)]} {
	set a [string trim [set ${mode}modeVars(prefixString)]]
    } else {
	set a "#"
    }
    insertText "${a}===============================================================================\r"
}
