#  AlphaTcl - core Tcl engine
#===========================================================================
# Information about a selection or window.
#===========================================================================
proc wordCount {{text ""}} {
    if {![string length $text]} {
	if {[set chars [string length [set text [getSelect]]]]} {
	    set lines [expr {[lindex [posToRowCol [selEnd]] 0] - [lindex [posToRowCol [getPos]] 0]}]
	    set text [getSelect]
	} else {
	    set chars [pos::diff [maxPos] [minPos]]
	    set lines [lindex [posToRowCol [maxPos]] 0]
	    set text [getText [minPos] [maxPos]]
	}
    } else {
	set chars [string length $text]
        set lines [regsub -all "\r?\n" $text "&" text]
    }
    
    regsub -all {[!=;.,\(\#\=\):\{\"\}]} $text " " text
    set words [llength $text]
    alertnote [format "%d chars, %d words, %d lines" $chars $words $lines]
}

proc requireOpenWindow {{msg ""}} {
    if {[llength [winNames -f]]} {return}
    if {![string length $msg]} {
	if {[catch {info level -1} item]} {set item "this item"}
	set msg "Cancelled -- '$item' requires an open window."
    }
    status::errorMsg $msg
}

proc requireSelection {{msg ""}} {
    if {[isSelection]} {return}
    if {![string length $msg]} {
	if {[catch {info level -1} item]} {set item "this item"}
	set msg "Cancelled -- '$item' requires a selection."
    }
    status::errorMsg $msg
}

# FILE: sortLines.tcl
#
# This version of sortLines has the option of ignoring blanks/whitespace (-b)
# and case-insensitive sorting (-i), or reverse sorting, and removing duplicates
# if desired [-d]
# 	sortLines [-b] [-i] [-r] [-d]

# COPYRIGHT:
#
#	Copyright  1992,1993 by David C. Black All rights reserved.
#	Portions copyright  1990, 1991, 1992 Pete Keleher. All Rights Reserved.
#   Portions copyright (c) 1999 Vince Darley, no rights reserved.
#
#	Redistribution and use in source and binary forms are permitted
#	provided that the above copyright notice and this paragraph are
#	duplicated in all such forms and that any documentation,
#	advertising materials, and other materials related to such
#	distribution and use acknowledge that the software was developed
#	by David C. Black.
#
#	THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
#	IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
#	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#
################################################################################

# AUTHOR
#
#	David C. Black
#	GEnie:    D.C.Black
#	Internet: black@mpd.tandem.com (preferred)
#	USnail:   6217 John Chisum Lane, Austin, TX 78749
#
################################################################################

proc reverseSort {} {sortLines -r}

proc sortLines {args} {
    getOpts

    if {[info exists opts(-r)]} {
	set mode "-decreas"
    } else {
	set mode "-increas"
    }

    if {[pos::compare [getPos] == [selEnd]]} {
	set start [minPos]
	set end [maxPos]
    } else {
	# We extend last line to the end :
	if {![is::Eol [lookAt [pos::math [selEnd] - 1]]]} {
	    set end [nextLineStart [selEnd]]
	} else {
	    set end [selEnd]
	}
	# We extend first line to the start :
	set start [lineStart [getPos]]
    }
    if {![is::Eol [lookAt [pos::math $end - 1]]]} {
	set endoftext $end
    } else {
	set endoftext [pos::math $end - 1]
    }
    set text [split [getText $start $endoftext] "\n\r"]
    if {[info exists opts(-b)] || [info exists opts(-i)] \
      || [info exists opts(-d)]} {
	foreach line $text {
	    if {[info exists opts(-i)]} {
		set key [string tolower $line]
	    } else {
		set key $line
	    }
	    if {[info exists opts(-b)]} {
		regsub -all "\[ \t\]+" $key " " key
	    }
	    if {![info exists orig($key)]} {
		set orig($key) $line
		lappend list $key
	    } elseif {![info exists opts(-d)]} {
		while {[info exists dup($key)]} {
		    append key "z"
		}
		set dup($key) $line
	    }
	}
	unset text
	foreach key [lsort $mode $list] {
	    lappend text $orig($key)
	    while {[info exists dup($key)]} {
		lappend text $dup($key)
		append key "z"
	    }
	}
    } else {
	set text [lsort $mode $text]
    }
    set text [join $text "\r"]
    replaceText $start $endoftext $text
    set endoftext [pos::math $start + [string length $text] + 1]
    if {[pos::compare [maxPos] < $endoftext]} {
	set endoftext [maxPos]
    }
    select $start $endoftext
}

# Test case:
#
# a  black
# A  black dog
# a black cat
# A  Black dog
# A  black dog


## 
 # -------------------------------------------------------------------------
 # 
 # "sortParagraphs" --
 # 
 #  Sorts selected paragraphs according to their first 30 characters,
 #  it's case insensitive and removes all non alpha-numeric characters
 #  before the sort.
 # -------------------------------------------------------------------------
 ##
proc sortParagraphs {args} {
    set start [getPos]
    set end  [selEnd]
    if {[pos::compare [getPos] == [selEnd]]} {
	set start [minPos]
	set end [maxPos]
    } else {
	# We extend last line to the end :
	if {![is::Eol [lookAt [pos::math [selEnd] - 1]]]} {
	    set end [nextLineStart [selEnd]]
	} else {
	    set end [selEnd]
	}
	# We extend first line to the start :
	set start [lineStart [getPos]]
    }
    if {![is::Eol [lookAt [pos::math $end - 1]]]} {
	set endoftext $end
    } else {
	set endoftext [pos::math $end - 1]
    }
    set text [getText $start $end]

    if {[string first "" $text] != -1} {
	alertnote "Sorry, can't sort paragraphs with bullets ''."
	return
    }
    regsub -all "\[\r\n\]\[ \t\]*\[\r\n]" $text "\r" text
    set paras [split $text ""]
    unset text
    # now each paragraph ends in \r
    foreach para $paras {
	set key [string tolower [string range $para 0 30]]
	regsub -all {[^-a-z0-9]} $key "" key
	# so we don't clobber duplicates!
	if {![info exists orig($key)]} {
	    set orig($key) $para
	} else {
	    while {[info exists dup($key)]} {
		append key "z"
	    }
	    set dup($key) $para
	}
    }
    unset para
    foreach key [lsort [array names orig]] {
	lappend text $orig($key)
	while {[info exists dup($key)]} {
	    lappend text $dup($key)
	    append key "z"
	}
    }
    replaceText $start $end [join $text "\r"]
    select $start $end
}

#================================================================================
# Block shift left and right.
#================================================================================

proc shiftBy {amount} {
    set start [lineStart [getPos]]
    set end [nextLineStart [pos::math [selEnd] - 1]]
    if {[pos::compare $start >= $end]} {set end [nextLineStart $start]}
    set text [text::indentBy [getText $start $end] $amount]
    replaceText $start $end $text
    set end [pos::math $start + [string length $text]]
    if {[pos::compare [nextLineStart $start] == $end]} {
	goto [pos::math $start + [string length $text] - [string length [string trimleft $text]]]
    } else {
	select $start $end
    }
}

proc shiftRight {} {
    global indentationAmount
    shiftBy $indentationAmount
}

proc shiftLeft {} {
    global indentationAmount
    shiftBy -$indentationAmount
}

proc shiftLeftSpace {} {
    shiftBy -1
}

proc shiftRightSpace {} {
    shiftBy 1
}

proc doShiftLeft {shiftChar} {
    set start [lineStart [getPos]]
    set end [nextLineStart [pos::math [selEnd] - 1]]
    if {[pos::compare $start >= $end]} {set end [nextLineStart $start]}
    
    set text [split [getText $start [pos::math $end - 1]] "\r\n"]
    
    set textout ""
    
    foreach line $text {
	if {[regexp "($shiftChar)(.*)$" $line "" "" c]} {
	    lappend textout $c
	} else {
	    lappend textout $line
	}
    }
    
    set text [join $textout "\r"]	
    replaceText $start [pos::math $end - 1] $text
    select $start [pos::math $start + [expr {1 + [string length $text]}]]
}

proc doShiftRight {shiftChar} {
    set start [lineStart [getPos]]
    set end [nextLineStart [pos::math [selEnd] - 1]]
    if {[pos::compare $start >= $end]} {set end [nextLineStart $start]}
    
    set text [split [getText $start [pos::math $end - 1]] "\r\n"]
    
    set text "$shiftChar[join $text \r${shiftChar}]"
    replaceText $start [pos::math $end - 1] $text
    select $start [pos::math $start + [expr {1 + [string length $text]}]]
}

proc selectAll {} {
    select [minPos] [maxPos]
}

proc isSelection {} {
    return [pos::compare [getPos] != [selEnd]]
}

# Select the next or current word. If word already selected, will go to next.
proc hiliteWord {} {
    if {[pos::compare [getPos] != [selEnd]]} forwardChar
    forwardWord
    set start [getPos]
    backwardWord
    if {[pos::compare [getPos] < [lineStart $start]]} {
	goto [lineStart $start]
    }
    select $start [getPos] 
}

proc text::replace {old new {fwd 1} {pos ""}} {
    if {$pos == ""} {set pos [getPos]}
    set m [search -s -f $fwd -m 0 -r 0 -- $old $pos]
    eval replaceText $m [list $new]
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"containsSpace"	--
 #	
 #  Does the given text contain any spaces?  In general we don't
 #  complete commands which contain spaces (although perhaps future
 #  extensions should do this: e.g. cycle through 'string match',
 #  'string compare',)
 #	 
 # -------------------------------------------------------------------------
 ##
proc containsSpace { cmd } { return [string match "*\[ \t\]*" $cmd] }
proc containsReturn { cmd } { return [string match "*\[\r\n\]*" $cmd] }

## 
 # -------------------------------------------------------------------------
 # 
 # "togglePrefix" --
 # 
 #  Useful for e.g. Tcl mode to add/remove a '$', TeX to add/remove 
 #  a backslash, etc.  Only works for single character prefixes.
 # -------------------------------------------------------------------------
 ##
proc togglePrefix {pref} {
    set p [getPos]
    backwardWord
    if {[lookAt [getPos]] == $pref} {
	deleteChar
	goto [pos::math $p -1]
    } else {
	insertText $pref
	goto [pos::math $p +1]
    }
}

proc twiddle {} {
    global twiddleAdjusts
    
    set pos [getPos]
    
    # If there is a selection
    if {[string length [set text [getSelect]]]} {
	if {[string length $text] == 1} {
	    message "Select more than one character to twiddle"
	} else {
	    set sel [pos::math [selEnd] - 1]
	    set char1 [lookAt $sel]
	    set char2 [lookAt $pos]
	    replaceText $pos [pos::math $sel + 1] \
	      "$char1[getText [pos::math $pos + 1] $sel]$char2"
	    message "twiddled chars: $char1$char2"
	    select $pos [pos::math $sel + 1]
	}
    } else {
	# If there is no selection, there are three possibilities,
	# depending on the user's preference.
	if {[pos::compare $pos == [minPos]]} {return}
	
	switch -- $twiddleAdjusts {
	    0 {
		if {[pos::compare $pos == [maxPos]]} {return}
		set first $pos
		set second [pos::math $pos - 1]
	    }
	    1 {
		set first [pos::math $pos -1]
		if {[pos::compare $first == [minPos]]} {return}
		set second [pos::math $first -1]
	    }
	    2 {
		set searchResult [search -s -n -f 0 -m 0 -i 1 \
		  -r 1 {[^ \r\n\t]} [pos::math $pos - 1]]
		if {![llength $searchResult]} {return}

		set first [lindex $searchResult 0]
		if {[pos::compare $first == [minPos]]} {return}
		set second [pos::math $first -1]
	    }
	}
	
	set char1 [lookAt $first]
	set char2 [lookAt $second]
	
	replaceText $second [pos::math $first +1] "${char1}${char2}"
	message "twiddled chars: $char1$char2"
	# Emacs like movement of cursor for this preference.
	if {$twiddleAdjusts == 0} {set pos [pos::math $pos + 1]}
	goto $pos
    }
}

proc twiddleWords {} {
    global twiddleAdjusts

    set pos [getPos]
    set start1 $pos
    set end2 [selEnd]

    # If there is a selection
    if {[pos::compare $start1 != $end2]} {
	select $start1
	forwardWord; set end1 [getPos]
	goto $end2
	backwardWord; set start2 [getPos]
	# If there is no selection, there are two possibilities,
	# depending on the user's preference.
    } elseif {$twiddleAdjusts} {
	backwardWord; backwardWord; set start1 [getPos]
	forwardWord; set end1 [getPos]
	forwardWord; set end2 [getPos]
	backwardWord; set start2 [getPos]
    } else {
	backwardWord; set start1 [getPos]
	forwardWord; set end1 [getPos]
	goto $pos
	forwardWord; set end2 [getPos]
	backwardWord; set start2 [getPos]
    }

    if {[pos::compare $end1 > $start2] \
      || [pos::compare $start1 == $start2] || [pos::compare $end1 == $end2]} {
	message "twiddleWords error: two words not specified"
	if {$twiddleAdjusts} {forwardWord}
	return
    }

    set mid [getText $end1 $start2]
    set one [getText $start2 $end2]
    set two [getText $start1 $end1]
    replaceText $start1 $end2 "$one$mid$two"
    message "twiddleWords words $one with $two"
    goto $end2
}

proc insertPrefix {} {doPrefix insert}
proc removePrefix {} {doPrefix remove}
proc doPrefix {which} {
    global prefixString
    if {[pos::compare [set start [getPos]] == [set end [selEnd]]]} {
	set end [nextLineStart $start]
    }
    setMark
    set start [lineStart $start]
    set text [getText $start $end]
    replaceText $start $end [doPrefixText $which $prefixString $text]
}

proc quoteChar {} {
    message "Literal keystroke to be inserted:"
    insertText [getChar]
}

proc setPrefix {} {
    global prefixString
    if {[catch {prompt "New Prefix String:" $prefixString} res] == 1} return
    set prefixString $res
}

proc setSuffix {} {
    global suffixString
    if {[catch {prompt "New Suffix String:" $suffixString} res] == 1} return
    set suffixString $res
}

proc insertSuffix {} {doSuffix insert}
proc removeSuffix {} {doSuffix remove}
proc doSuffix {which} {
    global suffixString
    set pts [getEndpts]
    set start [lindex $pts 0]
    set end [lindex $pts 1]
    set start [lineStart $start]
    set end [nextLineStart [pos::math $end - 1]]
    set text [getText $start $end]
    set text [doSuffixText $which $suffixString $text]
    replaceText $start $end $text
    select $start [getPos]
}

proc prevLineStart { pos } {
    return [lineStart [pos::math [lineStart $pos] - 1]]
}


proc frontTabsToSpaces { start end } {
    select $start $end
    tabsToSpaces
}

proc frontSpacesToTabs { start end } {
    getWinInfo a
    set sp [string range "              " 1 $a(tabsize) ]
    set from [lindex [posToRowCol $start] 0]
    set to [lindex [posToRowCol $end] 0]
    while {$from <= $to} {
	set pos [rowColToPos $from 0]
	# get the leading whitespace of the current line
	set res [search -s -n -f 1 -r 1 "^\[ \t\]*" $pos]
	if {![llength $res]} {
	    # end of the file
	    return
	}
	regsub -all "($sp| +\t)" [eval getText $res] "\t" front
	eval replaceText $res [list $front]
	incr from
    }
}

proc beginningOfLineSmart {} {
    set firstNonWsPos [text::firstNonWsLinePos [getPos]]
    if {[pos::compare [getPos] == $firstNonWsPos]} {
	beginningOfLine
    } else {
	goto $firstNonWsPos
    }
}

proc forwardDeleteUntil {{c ""}} {
    if {$c == ""} {
	message "Forward delete up to next:"
	set c [getChar]
    }
    set p [lindex [search -s -n -f 1 -r 1 [quote::Regfind $c] [getPos]] 0]
    if {$p != ""} {
	deleteText [getPos] [pos::math $p + 1]
    }
}

proc forwardDeleteWhitespace {} {
    set p [lindex [search -s -n -f 1 -r 1 "\[^ \t\r\n\]" [getPos]] 0]
    if {$p != ""} {
	deleteText [getPos] $p
    }
}

