## -*-Tcl-*-
 # ###################################################################
 #  Part of AlphaTcl - core Tcl engine
 # 
 #  FILE: "vcsCore.tcl"
 #                                    created: 03/23/2000 {10:59:22 AM} 
 #                                last update: 12/14/2001 {12:56:12 PM} 
 #                                
 # ========================================================================
 #           Copyright (c) 1998-2001 Jon Guyer, Vince Darley
 #                         All rights reserved
 # ========================================================================
 # Permission to use, copy, modify, and distribute this software and its
 # documentation for any purpose and without fee is hereby granted,
 # provided that the above copyright notice appear in all copies and that
 # both that the copyright notice and warranty disclaimer appear in
 # supporting documentation.
 # 
 # The authors disclaim all warranties with regard to this software,
 # including all implied warranties of merchantability and fitness.  In
 # no event shall the authors be liable for any special, indirect or
 # consequential damages or any damages whatsoever resulting from loss of
 # use, data or profits, whether in an action of contract, negligence or
 # other tortuous action, arising out of or in connection with the use or
 # performance of this software.
 # ========================================================================
 #  
 #  For the moment this code is designed to work with Tcl 7.x as
 #  well as 8.x, which is why namespaces aren't handled in the most
 #  elegant way: we may not have them!
 # ###################################################################
 ##

alpha::extension vcs 0.1a12 {
    namespace eval vcs {}
    # The current version control system.
    newPref var versionControlSystem "None" vcs "" vcs::system "array"
    # This allows us to attach version control information to any fileset
    fileset::attachNewInformation * \
      {prefItemType versionControlSystem} "Version Control System" \
      None "The version control system under which these\
      files are placed" vcs::vcsSystemModified
    # Called when the user ctrl/cmd-clicks on the lock icon
    hook::register unlockHook vcs::manualUnlock *
    hook::register lockHook vcs::manualLock *
    # Add a version control prefs page, mapped to the 'vcs' storage
    package::addPrefsDialog versionControl vcs
    newPref flag addNameOfSystemToPopup 1 vcs
    
    menu::buildProc vcsMenu vcs::buildVcsMenu
} help {
    AlphaTcl's core version control functionality is provided by this
    package.
} maintainer {
    "Jon, Vince and others"
}

namespace eval vcs {}

set vcs::system(None) vcs

proc vcs::menuProc {menu item} {
    switch -- $item {
	lock {
	    # Right now this just implements non-vcs connected
	    # lock/unlock actions
	    try {setWinInfo read-only 1}
	}
	unlock {
	    # Right now this just implements non-vcs connected
	    # lock/unlock actions
	    setWinInfo read-only 0
	}
	default {
	    # add checkIn undoCheckout makeWritable checkOut
	    # refetchReadOnly fetchReadOnly
	    set name [win::Current]
	    vcs::call $item $name
	    vcs::syncLockStatus $name
	}
    }
}

proc vcs::vcsSystemModified {fset value} {
    hook::callAll vcsSystemModified $value $fset
}

proc vcs::register {type {ns ""}} {
    global vcs::system
    if {![string length $ns]} { set ns $type }
    set vcs::system($type) $ns
}

proc vcs::getNamespace {} {
    global vcs::system
    set vcs::system([vcs::getSystem])
}

proc vcs::getState {name} {
    global alpha::macos
    if {${alpha::macos}} {
	# This is meaningless if we're not on MacOS.
	if {[catch {vcs::ckid::readResource $name ckid}]} {
	    return "no-vcs"
	} elseif {$ckid(writable)} {
	    set state "checked-out"
	} else {
	    if {$ckid(mro)} {
		set state "mro"
	    } else {
		set state "read-only"
	    }
	}
    } else {
	set state "no-vcs"
    }
    
    vcs::optionalCall checkCompatibility $name
    return $state
}

# Dummy procedure
proc vcs::checkCompatibility {name} {}

proc vcs::getInfo {filename} {
    alertnote "No further information is available for '$filename'"
}

proc vcs::getFilesetInfo {infoName} {
    return [fileset::getInformation [fileset::checkCurrent] $infoName]
}

proc vcs::getSystem {} {
    global vcsmodeVars
    set fset [fileset::checkCurrent]
    if {[string length $fset] == 0} {
	if {[info exists vcsmodeVars(versionControlSystem)]} {
	    return $vcsmodeVars(versionControlSystem)
	} else {
	    return "None"
	}
    } else {
	return [fileset::getInformation $fset "Version Control System"]
    }
}

# Only call if the vcs system provides the right procedure
# This implies the vcs system must be completely loaded.
proc vcs::optionalCall {what args} {
    set proc [vcs::getNamespace]::${what}
    if {[info commands $proc] != ""} {
	if {[catch {eval $proc $args} err]} {
	    message $err
	}
	return $err
    }
    return ""
}

proc vcs::call {what args} {
    set ns [vcs::getNamespace]
    if {[catch {eval ${ns}::${what} $args} err]} {
	message $err
    }
    return $err
}

proc vcs::procOrDefault {what} {
    set ns [vcs::getNamespace]
    if {[llength [info commands ::${ns}::${what}]] > 0} {
	return ${ns}::${what}
    } else {
	return vcs::${what}
    }
}

proc vcs::syncLockStatus {name} {
    set fileReadOnly [expr {![file writable [win::StripCount $name]]}]
    getWinInfo winState
    if {$winState(read-only) != $fileReadOnly} {
	setWinInfo read-only $fileReadOnly
    }
}

proc vcs::manualUnlock {name} {
    vcs::call unlock $name
}

proc vcs::manualLock {name} {
    vcs::call lock $name
}

proc vcs::showDifferences {name} {
}

proc vcs::lock {name} {
    try {setWinInfo read-only 1}
}

proc vcs::unlock {name} {
    setWinInfo read-only 0
}

proc vcs::checkIn {name} {
}

proc vcs::checkOut {name} {
}

proc vcs::undoCheckout {name} {
}

proc vcs::refetchReadOnly {name} {
}

proc vcs::makeWritable {name} {
    global alpha::macos
    if {${alpha::macos}} {
	vcs::ckid::setMRO $name
    }
    setWinInfo -w $name read-only 0
}

proc vcs::otherCommands {state} {
    # nothing by default
}

# This is a callback routine for Alpha 8's VCS popup menu
proc ckidMenu {stateIndex locked} {
    global menu::items
    
    set state [lindex [list "no-vcs" "checked-out" "read-only" "mro"] $stateIndex]

    set state [vcs::getState [win::Current]]
    set menu::items(ckid) [vcs::menuItems $state $locked]
    menu::buildOne ckid
    return "ckid"
}

# This is used by Alphatk
proc vcs::buildVcsMenu {} {
    set state [vcs::getState [win::Current]]
    getWinInfo -w [win::Current] info
    set menuList [vcs::menuItems $state $info(read-only)]

    return [list build $menuList [vcs::procOrDefault menuProc] {}]
}

# Used in a callback from Alpha 8 via the above proc, or directly
# in Alphatk.  An empty state means AlphaTcl has no idea how to
# get any vcs information for this file (e.g. we're running
# Alphatk), a state of 'no-cvs' means this file doesn't appear
# to be under version control, but we should really double-check.
proc vcs::menuItems {state locked} {
    global vcsmodeVars
    # ckid icon suite runs from 490 to 494
    # subtract 208 (why?!?) + 256

    set vcs [vcs::getSystem]
    
    if {[string tolower $vcs] != "none"} {
	alpha::package require vc$vcs
    }
    
    if {$state == "" || $state == "no-vcs"} {
	# Unknown state
	set state [vcs::call getState [win::Current]]
    }

    if {[info exists vcsmodeVars(addNameOfSystemToPopup)]
    &&	$vcsmodeVars(addNameOfSystemToPopup)} {
	
	if {[string tolower $vcs] == "none"} {
	    set vcs "No"
	}
	lappend res "\(using${vcs}VCSTool"
	lappend res "getInfo"
    } else {
	set res [list "getInfo"]
    }
    
    # Active items should depend on whether we have a VCS system 
    # active and on the state of the file.
    # 
    # Currently 'read-only' means the file is either 'up-to-date'
    # or 'needs-patch', but we don't know which (it appears as if
    # the ckid resource doesn't give us enough information?).
    
    eval lappend res [vcs::call getMenuItems $state]
    
    if {[llength $res]} {
	lappend res "(-)"
    }
    
    if {$locked} {
	lappend res "unlock[icon::FromID 494]"
    } else {
	lappend res "lock[icon::FromID 493]"
    }
    
    # Add any other items the vcs system wants to use
    set extras [vcs::optionalCall otherCommands $state]
    if {[llength $extras]} {
	lappend res "(-)"
	eval lappend res $extras
    }
    
    set res
}

## 
 # -------------------------------------------------------------------------
 # 
 # "vcs::getMenuItems" --
 # 
 #  Called when there is no active VC System
 #  All items disabled.
 # -------------------------------------------------------------------------
 ##
proc vcs::getMenuItems {state} {
    switch -- $state {
      "no-vcs" { 
	  lappend res "\(add[icon::FromID 491]"	
      }
      "checked-out" { 
	  lappend res                                   \
	    "\(checkIn[icon::FromID 490]"              \
	    "\(undoCheckout[icon::FromID 491]"          \
	    "\(makeWritable[icon::FromID 492]"          \
	    "(-)"                                       \
	    "\(showDifferences" 
      }
      "read-only" { 
	  lappend res                                   \
	    "\(checkOut[icon::FromID 490]"             \
	    "\(refetchReadOnly[icon::FromID 491]"       \
	    "\(makeWritable[icon::FromID 492]"		\
	    "(-)"                                       \
	    "\(showDifferences"
      }
      "mro" { 
	  lappend res                                   \
	    "\(checkOut[icon::FromID 490]"		\
	    "\(fetchReadOnly[icon::FromID 491]"		\
	    "\(makeWritable[icon::FromID 492]"		\
	    "(-)"                                       \
	    "\(showDifferences"
      }
      "up-to-date" {
	  lappend res                                   \
	    "\(checkOut[icon::FromID 490]"		\
	    "\(makeWritable[icon::FromID 492]"		\
      }
      "needs-patch" { 
	  lappend res                                   \
	    "\(refetchReadOnly[icon::FromID 491]"       \
	    "(-)"                                       \
	    "\(showDifferences" 
      }
      "" {
	  # no version control registered, or not possible 
	  # to place under version control with current
	  # system
	  set res {}
      }
      default {
	  error "Bad response '$state' received from vcs system"
      }
    }
    
    return $res
}

## 
 # -------------------------------------------------------------------------
 # 
 # "vcs::generalMenuItems" --
 # 
 #  General utility function.
 #  Most VC Systems will use this to build the bulk of their items
 # -------------------------------------------------------------------------
 ##
proc vcs::generalMenuItems {state} {
    switch -- $state {
      "no-vcs" { 
	  lappend res "add[icon::FromID 491]"	
      }
      "checked-out" { 
	  lappend res                                   \
	    "checkIn[icon::FromID 490]"                \
	    "undoCheckout[icon::FromID 491]"            \
	    "\(makeWritable[icon::FromID 492]"          \
	    "(-)"                                       \
	    "showDifferences" 
      }
      "read-only" { 
	  lappend res                                   \
	    "checkOut[icon::FromID 490]"               \
	    "refetchReadOnly[icon::FromID 491]"	        \
	    "makeWritable[icon::FromID 492]"		\
	    "(-)"                                       \
	    "showDifferences"
      }
      "mro" { 
	  lappend res                                   \
	    "checkOut[icon::FromID 490]"		\
	    "fetchReadOnly[icon::FromID 491]"		\
	    "\(makeWritable[icon::FromID 492]"		\
	    "(-)"                                       \
	    "showDifferences"
      }
      "up-to-date" {
	  lappend res                                   \
	    "checkOut[icon::FromID 490]"		\
	    "makeWritable[icon::FromID 492]"		\
      }
      "needs-patch" { 
	  lappend res                                   \
	    "refetchReadOnly[icon::FromID 491]"	        \
	    "(-)"                                       \
	    "showDifferences" 
      }
      "" {
	  # no version control registered, or not possible 
	  # to place under version control with current
	  # system
	  set res {}
      }
      default {
	  error "Bad response '$state' received from vcs system"
      }
    }
    
    return $res
}

#  'ckid' resource handling  #

if {!${alpha::macos}} { return }

namespace eval vcs::ckid {}

proc vcs::ckid::readResource {name aa} {
    upvar $aa a
    
    set resid [resource open $name r]
    set a(id) [lindex [resource list ckid $resid] 0]
    set ckid [resource read ckid $a(id) $resid]
    resource close $resid
    
    if {[vcs::ckid::verifyCheckSum $ckid]} {
	binary scan $ckid IISa* a(checkSum) a(location) a(version) ckid
	binary scan $ckid Scca* a(writable) a(branch) a(mro) ckid
	binary scan $ckid SSa* a(history) a(historyLen) ckid
	binary scan $ckid IIa* a(dateTime) a(modDate) ckid
	binary scan $ckid IISSSa* a(pidA) a(pidB) a(userID) a(fileID) a(revID) ckid
	binary scan $ckid ca* count ckid
	binary scan $ckid a${count}ca* a(path) EOS ckid
	binary scan $ckid ca* count ckid
	binary scan $ckid a${count}ca* a(user) EOS ckid
	binary scan $ckid ca* count ckid
	binary scan $ckid a${count}ca* a(revision) EOS ckid
	binary scan $ckid ca* count ckid
	binary scan $ckid a${count}ca* a(filename) EOS ckid
	binary scan $ckid ca* count ckid
	binary scan $ckid a${count}ca* a(task) EOS ckid
	# sneaky bastards! comment is a wide string
	binary scan $ckid Sa* count ckid
	binary scan $ckid a${count}ca* a(comment) EOS ckid
    }
    
    return
}

proc vcs::ckid::writeResource {name aa} {
    upvar $aa a
    
    set ckid ""
    append ckid [binary format IS $a(location) $a(version)]
    append ckid [binary format Scc $a(writable) $a(branch) $a(mro)]
    append ckid [binary format SS $a(history) $a(historyLen)]
    append ckid [binary format II $a(dateTime) $a(modDate)]
    append ckid [binary format IISSS $a(pidA) $a(pidB) $a(userID) $a(fileID) $a(revID)]
    append ckid [binary format ca*x [string length $a(path)] $a(path)]
    append ckid [binary format ca*x [string length $a(user)] $a(user)]
    append ckid [binary format ca*x [string length $a(revision)] $a(revision)]
    append ckid [binary format ca*x [string length $a(filename)] $a(filename)]
    append ckid [binary format ca*x [string length $a(task)] $a(task)]
    # sneaky bastards! comment is a wide string
    append ckid [binary format Sa*x [string length $a(comment)] $a(comment)]
    
    set ckid [binary format Ia* [vcs::ckid::calculateCheckSum $ckid] $ckid]
    
    set resid [resource open $name w]
    if {[string is integer $a(id)]} {
	resource write -id $a(id) -name "Alpha" -file $resid -force ckid $ckid
    } else {
	resource write -id 128 -name $a(id) -file $resid -force ckid $ckid
    }
    resource close $resid
    
    return
}

proc vcs::ckid::verifyCheckSum {ckid} {
    binary scan $ckid Ia* checkSum remainder
    
    return [expr {[vcs::ckid::calculateCheckSum $remainder] == $checkSum}]
}

proc vcs::ckid::calculateCheckSum {remainder} {
    set sum 0
    set len [expr {[string length $remainder] / 4}]
    for {set i 0} {$i < $len} {incr i} {
        binary scan $remainder Ia* num remainder
	incr sum $num
    }
    
    return $sum
}

proc vcs::ckid::setMRO {name} {
    if {![catch {vcs::ckid::readResource $name ckid}]} {
	if {!$ckid(writable)} {
	    set ckid(mro) 1
	} 
	vcs::ckid::writeResource $name ckid
    }
}

proc vcs::ckid::getInfo {name} {
    if {[catch {vcs::ckid::readResource $name ckid} err]} {
	alert -t stop -c "" -o "" "Unable to get version control information" $err
    }
    
    if {$ckid(writable)} {
	set status "Checked out by $ckid(user) at [mtime $ckid(dateTime)]"
    } else {
	if {$ckid(mro)} {
	    set status "Modify-Read-Only by $ckid(user) at [mtime $ckid(modDate)]"
	} else {
	    set status "Checked in by $ckid(user) at [mtime $ckid(dateTime)]"
	}
    }
    
    dialog::make -title $ckid(id) \
      [list [file tail $name] \
      [list file "Local Path:" [file dirname $name]] \
      [list static "Status:" $status] \
      [list static "VCS Path:" $ckid(path)] \
      ]
}

proc vcs::ckid::checkCompatibility {name vcsNames} {
    if {![catch {vcs::ckid::readResource $name ckid} err]} {
	if {[string length $ckid(id)] > 0 
	&& [lsearch -exact $vcsNames $ckid(id)] < 0} {
	    beep
	    message "Current VCS system, [vcs::getSystem],\
	      does not match 'ckid' creator, $ckid(id)"
	} 
    }    
}
