## -*-Tcl-*- (install) (nowrap)
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 #  
 #  Chuck's Additions - an Alpha hack
 #
 #  FILE: "win.tcl"
 #  					created: 4/6/98
 #					last update: 09/06/2001 {16:03:37 PM}
 #  Author: Chuck Gregory
 #  E-mail: <cgregory@mail.arc.nasa.gov>
 #	mail: Logicon
 #		  NASA Ames Research Center, Moffett Field, CA  94035
 #
 #  Description:
 #
 #	Window handling routines. All procs are bound in AlphaBits.tcl.
 #	  Recommend the following global interface preference settings:
 #
 #					MacOS 8.0	 MacOS < 8
 #		  defLeft		 6		     0
 #		  defTop		41		    38
 #		  defWidth	       510		   510
 #		  horMargin	         6		     2
 #		  tileHeight	     [707]		   426
 #		  tileLeft	         6		     0
 #		  tileMargin		22		    20
 #		  tileTop	        41		    38
 #		  tileWidth	    [1014]		   640
 #
 #  History:
 #
 #  modified  by   rev  reason
 #  --------  ---  ---  -----------
 #  04/06/98		7.1b6 original
 #  04/08/98  czg  1.0  modified for MacOS 8
 #  07/15/98  VMD	removed lisp'ish functions
 #  07/21/98  czg  1.1  fixed margin bugs in shrinkLeft & shrinkRight;
 #			documented prefs recommendations
 # ###################################################################
 ##

proc shrinkHigh {} {
    global numWinsToTile tileTop tileHeight tileMargin
    set names [winNames -f]
    set numWins [llength $names]
    if {$numWins<2} {set numWins 2}
    if {$numWins>$numWinsToTile} {set numWins $numWinsToTile}
    set width [lindex [getGeometry] 2]
    set height [expr {($tileHeight - $tileMargin) / $numWins}]
    set text [getGeometry]
    set left [lindex $text 0]
    sizeWin $width $height
    moveWin $left $tileTop
}

proc shrinkLow {} {
    global numWinsToTile tileTop tileHeight tileMargin
    set names [winNames -f]
    set numWins [llength $names]
    if {$numWins<2} {set numWins 2}
    if {$numWins>$numWinsToTile} {set numWins $numWinsToTile}
    set width [lindex [getGeometry] 2]
    set height [expr {($tileHeight - $tileMargin) / $numWins}]
    set text [getGeometry]
    set left [lindex $text 0]
    sizeWin $width $height
    moveWin $left [expr {$tileTop + $height + $tileMargin}]
}

proc singlePage {} {shrinkFull}

proc defaultSize {} {
    global defWidth defHeight defTop defLeft
    moveWin $defLeft $defTop
    sizeWin $defWidth $defHeight
}

proc shrinkFull {} {
    global tileTop tileHeight tileLeft defWidth
    moveWin $tileLeft $tileTop
    sizeWin $defWidth $tileHeight
}

proc shrinkLeft {} {
    global horMargin tileWidth tileLeft
    set width [expr {($tileWidth-$horMargin)/2}]
    set height [lindex [getGeometry] 3]
    set text [getGeometry]
    set top [lindex $text 1]
    moveWin $tileLeft $top
    sizeWin $width $height
}

proc shrinkRight {} {
    global horMargin tileWidth tileLeft
    set width [expr {($tileWidth-$horMargin)/2}]
    set height [lindex [getGeometry] 3]
    set text [getGeometry]
    set top [lindex $text 1]
    moveWin [expr {$tileLeft + $width + $horMargin}] $top
    sizeWin $width $height
}

proc swapWithNext {} {
    set files [winNames]
    if {[llength $files] < 2} return
    bringToFront [lindex $files 1]
}
	


proc nextWindow {} {
    global win::Active 
    set files [winNames -f]
    if {[llength $files] < 2} {return}
    set f [lindex $files 0]
    if {[info tclversion] < 8.0} {
	regsub -all {[][]} $f {\\\0} f
    }
    set aind [lsearch -exact ${win::Active} $f]
    if {$aind < 0} {error "No win '$f'"}
    set rng [lrange ${win::Active} 0 [expr {$aind-1}]]
    set win::Active [concat [lrange ${win::Active} $aind end] $rng]
    set win::Active [lrange ${win::Active} 1 end]
    lappend win::Active $f
    if {[info tclversion] < 8.0} {
	regsub -all {\\([][])} [lindex ${win::Active} 0] {\1} w
    } else {
	set w [lindex ${win::Active} 0]
    }
    bringToFront $w
}


proc prevWindow {} {
    global win::Active 
    set files [winNames -f]
    if {[llength $files] < 2} {return}
    set f [lindex $files 0]
    regsub -all {[][]} $f {\\\0} f
    set aind [lsearch -exact ${win::Active} $f]
    if {$aind < 0} {error "No win '$f'"}
    set rng [lrange ${win::Active} 0 [expr {$aind-1}]]
    set win::Active [concat [lrange ${win::Active} $aind end] $rng]
    set f2 [lindex [lrange ${win::Active} end end] 0]
    set win::Active [lreplace ${win::Active} end end]
    set win::Active [linsert ${win::Active} 0 $f2]
    regsub -all {\\([][])} $f2 {\1} f2
    bringToFront $f2
}

proc bufferOtherWindow {} {
    global tileHeight tileTop tileWidth tileMargin
    global numWinsToTile
    set margin $tileMargin
    set win [win::Current]
    set numWins 2
    set hor 2
    set height [expr {($tileHeight/$numWins)-$margin}]
    set height [expr {$height + $margin / $numWins}]
    set width $tileWidth
    set ver $tileTop
    
    if {[llength [winNames]] < 2} {message "No other window!"; return}
    set res [prompt::fromChoices "Window other half" [nextWin] -command winNames]
    
    set geo [getGeometry]
    if {([lindex $geo 2] != $width) || ([lindex $geo 3] != $height) || ([lindex $geo 0] != $hor) || (([lindex $geo 1] != $ver) && ([lindex $geo 1] != [expr {$ver + $height + $margin}]))} {
	moveWin $win 1000 0
	sizeWin $win $width $height
	moveWin $win $hor $ver
	incr ver [expr {$height + $margin}]
    } else {
	if {[lindex $geo 1] == $ver} {
	    incr ver [expr {$height + $margin}]
	} 
    }
    
    set geo [getGeometry $res]
    if {([lindex $geo 0] != $hor) || ([lindex $geo 1] != $ver) || ([lindex $geo 2] != $width) || ([lindex $geo 3] != $height)} {
	moveWin $res 1000 0
	sizeWin $res $width $height
	moveWin $res $hor $ver
    }
    bringToFront $res
}

		
	
		

proc winvertically {} {
    global tileHeight tileTop tileWidth tileMargin
    global numWinsToTile defWidth tileLeft
    set margin $tileMargin
    set names [winNames -f]
    set numWins [llength $names]
    if {$numWins<=1} return
    if {$numWins>$numWinsToTile} {set numWins $numWinsToTile}
    if {$numWins == 0} {return}
    set height [expr {($tileHeight/$numWins)-$margin}]
    set height [expr {$height + $margin / $numWins}]
    set width $defWidth
    set ver $tileTop
    for {set i 0} {$i < $numWins} {incr i} {
	sizeWin [lindex $names $i] $width $height
	moveWin [lindex $names $i] $tileLeft $ver
	set ver [expr {$ver+$margin+$height}]
    }
}

proc winhorizontally {} {
    global tileHeight tileLeft tileWidth tileTop numWinsToTile horMargin
    set names [winNames -f]
    set numWins [llength $names]
    if {$numWins<=1} return
    if {$numWins>$numWinsToTile} {set numWins $numWinsToTile}
    if {$numWins == 0} {return}
    set width [expr {($tileWidth/$numWins)-$horMargin}]
    set width [expr {$width + $horMargin / $numWins}]
    set height $tileHeight
    set hor $tileLeft
    for {set i 0} {$i < $numWins} {incr i} {
	sizeWin [lindex $names $i] $width $height
	moveWin [lindex $names $i] $hor $tileTop
	set hor [expr {$hor+$width+$horMargin}]
    }
}


proc winunequalHor {} {
    global tileLeft tileHeight tileWidth tileTop numWinsToTile horMargin
    global tileProportion
    set names [winNames -f]
    sizeWin [lindex $names 0] \
      [expr {int($tileProportion*$tileWidth) - $horMargin/2}] $tileHeight
    moveWin [lindex $names 0] $tileLeft $tileTop
    sizeWin [lindex $names 1] \
      [expr {int((1-$tileProportion)*$tileWidth) - $horMargin/2}] $tileHeight
    moveWin [lindex $names 1] \
      [expr {$tileLeft + int($tileProportion*$tileWidth) + $horMargin/2}] $tileTop
}


proc winunequalVert {} {
    global tileLeft tileMargin tileHeight tileWidth tileTop numWinsToTile
    global horMargin tileProportion defWidth
    set names [winNames -f]
    set height [expr {$tileHeight + $tileMargin}]
    sizeWin [lindex $names 0] \
      $defWidth [expr {int($tileProportion*$height) - $tileMargin}]
    moveWin [lindex $names 0] $tileLeft $tileTop
    sizeWin [lindex $names 1] \
      $defWidth [expr {int((1-$tileProportion)*$height) - $tileMargin}]
    moveWin [lindex $names 1] \
      $tileLeft [expr {$tileTop + int($tileProportion*$height)}]
}


proc wintiled {} {
    global tileHeight tileWidth numWinsToTile tileTop tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    set xPan 8
	    set yPan 10
	}
	default {
	    set xPan 16
	    set yPan 16
	}
    }
    set xMarg 2
    set yMarg $tileTop
    set yMax 50
    set names [winNames -f]
    set numWins [llength $names]
    if {$numWins<1} return
    set line 0	
    set height [expr {$tileHeight-$yPan*($numWins-1)}]
    set width [expr {$tileWidth-$xPan*($numWins-1)}]
    
    for {set i 0} {$i < $numWins} {incr i} {
	set j [expr {$numWins-$i-1}]
 	moveWin [lindex $names $j] [expr {$xMarg+$i*$xPan}] [expr {$yMarg+$line}]
	set line [expr {$line+$yPan}]
	if {$line>$yMax} {set line 0}
	sizeWin [lindex $names $j] $width $height
    }
}


proc winoverlay {} {
    global defHeight defWidth numWinsToTile tileTop
    set names [winNames -f]
    set numWins [llength $names]
    if {$numWins<1} return
    for {set i 0} {$i < $numWins} {incr i} {
	moveWin [lindex $names $i] 2 $tileTop
	sizeWin [lindex $names $i] $defWidth $defHeight
    }
}

proc chooseAWindow {} {
    switch -- [llength [winNames -f]] {
	0 {
	    message "No window!"; return
	}
	1 {
	    message "No other window!"; return
	}
	default {
	    set name [prompt::fromChoices "Window" [nextWin] \
	      -command "lsort -ignore \[winNames\]"]
	    if {[string length $name]} {
		bringToFront $name
		if {[icon -q]} { icon -f $name -o }
	    }
	}
    }
}

proc closeAWindow {} {
    if {![llength [winNames]]} {message "No window!"; return}
    set name [prompt::fromChoices "Close window" [win::CurrentTail] \
      -command "lsort -ignore \[winNames\]"]
    catch {bringToFront $name; killWindow}
}

proc nextWin {} {
    global win::Active 
    set files [winNames -f]
    if {[llength $files] < 2} {return ""}
    set f [lindex $files 0]
    set aind [lsearch -exact ${win::Active} $f]
    if {$aind < 0} {error "No win '$f'"}
    if {[incr aind] < [llength ${win::Active}]} {
	return [file tail [lindex ${win::Active} $aind]]
    } else {
	return [file tail [lindex ${win::Active} 0]]
    }
}

proc iconifyAll {{iconify 1}} {
    global win::Active
    set wCT [win::CurrentTail]
    foreach w ${win::Active} {
	bringToFront $w
	if {$iconify} {
	    if {![icon -q]} {icon -t}
	} else {
	    if {[icon -q]}  {icon -o}
	}
    }
    bringToFront $wCT
}

proc iconify {} { 
    icon -t 
    if {[icon -q]} {
	nextWindow
    }
}

proc zoom {} {
    global nzmState tileHeight tileWidth zoomedGeo tileTop tileLeft
    
    set win [win::Current]
    if {[info exists nzmState($win)]} {
	if {[getGeometry] == $zoomedGeo} {
	    set state $nzmState($win)
	    moveWin [lindex $state 0] [lindex $state 1]
	    sizeWin [lindex $state 2] [lindex $state 3]
	    unset nzmState($win)
	    return
	}
    } 
    
    set nzmState($win) [getGeometry]
    moveWin $tileLeft $tileTop
    sizeWin $tileWidth $tileHeight
    
    if {![info exists zoomedGeo]} {
	set zoomedGeo [getGeometry]
    }
}

#================================================================================

proc otherThing {} {
    set win [win::Current]
    getWinInfo -w $win arr
    if {$arr(split)} {
	otherPane
    } else {
	swapWithNext
    }
}

proc winAttribute {att {win {}}} {
    if {![string length $win]} {
	set win [win::Current]
    }
    getWinInfo -w $win arr
    return $arr($att)
}

proc floatName {str} {
    if {[string match "*" $str]} {
	foreach n [info globals {*Menu}] {
	    global $n
	    if {![catch {set $n}] && ([set $n] == $str)} {
		regexp {(.*)Menu} $n dummy name
		return "[string toupper [string index $name 0]][string range $name 1 end]"
	    }
	}
    }
    return "[string toupper [string index $str 0]][string range $str 1 end]"
}
proc winDirty {} {
    getWinInfo arr
    return $arr(dirty)
}

proc winReadOnly {{win ""}} {
    if {$win == ""} {set win [win::Current]}
    if {[info tclversion] < 8.0} {
	bringToFront $win
	goto [minPos]
    } else {
	goto -w $win [minPos]
    }
    setWinInfo -w $win dirty 0
    setWinInfo -w $win read-only 1
}

proc shrinkWindow {{shrinkWidth 0}} {
    global defHeight defWidth
    # These constants work for 9-pt Monaco type
    set lineht 11
    set htoff 22
    set chwd 6
    set choff 20
    
    set wd [lindex [getGeometry] 2]
    set ht [lindex [getGeometry] 3]
    set top [lindex [getGeometry] 1]
    set left [lindex [getGeometry] 0]
    
    set mxht [expr {[lindex [getMainDevice] 3] - $top - 5 -15}]
    set mxwd [expr {[lindex [getMainDevice] 2] - $left - 5}]
    set mnht 120
    set mnwd 200
    
    set htWd [fileHtWd $shrinkWidth]
    set lines [lindex $htWd 0]
    set chars [lindex $htWd 1]
    
    if {$lines <= 1} {set lines 10}
    
    
    if {$lines > 0} {
	set ht [expr {$htoff + ( $lineht * (1 + $lines)) }]
    } elseif {$ht > $defHeight} {
	set ht $defHeight
    }
    
    if {$chars > 0} {
	set wd [expr {$choff + ( $chwd * (2 + $chars)) }]
    } elseif {$wd > $defWidth} {
	set wd $defWidth
    }
    
    if {$ht > $mxht} {set ht $mxht}
    if {$wd > $mxwd} {set wd $mxwd}
    if {$ht < $mnht} {set ht $mnht}
    if {$wd < $mnwd} {set wd $mnwd}
    sizeWin $wd $ht
}

#############################################################################
# Return the number of lines and the maximum number of characters in any 
# line of a file.  It would be nice if there was a built-in command to
# do this (i.e., compiled C code) because this is a pretty slow way to
# get the maximum line width.

proc fileHtWd {{checkWidth 0}} {
    set text [getText [minPos] [maxPos]] 
    getWinInfo arr
    set tabw [expr {$arr(tabsize) - 1}]
    
    set lines [split $text "\r\n"]
    set nlines [llength $lines]
    
    if {$checkWidth > 1} {
	set lines [eval lrange \$lines [displayedLines]]
    }
    
    set llen 0
    if {$checkWidth > 0} {
	foreach line $lines {
	    regsub {				+.*$} $line {} line
	    regsub {	} $line {    } line
	    set len [string length $line]
	    if {[set ntab [llength [split $line "\t"]]] > 1} {
		set len [expr {$len + $tabw*($ntab-1)}]
	    }
	    if { $len > $llen} {
		set llen $len
	    }
	}
    }
    #	alertnote "Text Height : $nlines ; Text Width : $llen "
    return [list $nlines $llen]
}

# Report what range of lines are displayed in any window.
# (A side effect is that the insertion point is moved to the 
# top of the window, if it was previously off-screen)
#
proc displayedLines {{window {}}} {
    if {$window == {}} { set window [win::Current] }
    
    bringToFront $window
    set oldPos [getPos]
    moveInsertionHere
    set top [getPos]
    set first [lindex [posToRowCol $top] 0]
    moveInsertionHere -last
    set bottom [getPos]
    set last [lindex [posToRowCol $bottom] 0]
    
    if {[pos::compare $oldPos < $top] || [pos::compare $oldPos > $bottom]} {
	goto $top
    } else {
	goto $oldPos
    }
    
    return [list $first $last]
}






