#!/bin/sh
#-*-tcl-*-
# the next line restarts using wish \
exec wish "$0" ${1+"$@"}

#####################################################################
#  Alphatk - the ultimate editor
# 
#  FILE: "alphatk.tcl"
#                                    created: 04/12/98 {22:45:38 PM} 
#                                last update: 12/14/2001 {14:19:15 PM}}} 
#  Author: Vince Darley
#  E-mail: vince.darley@kagi.com
#    mail: Flat 10, 98 Gloucester Terrace, London, W2 6HP
#     www: http://www.santafe.edu/~vince/Alphatk.html
#  
# Copyright (c) 1998-2001  Vince Darley
# 
# See the file "license.terms" for information on use and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# 
# In particular, while this is 'open source', it is NOT free, and
# cannot be copied in full or in part except according to the terms
# of the license agreement.
# 
#####################################################################

# Just in case we're running with tclsh.  We want Tk!
package require Tk

if {$tcl_platform(os) == "MacOS"} {
    catch {package require MacOSdefaults}
} else {
    #wm overrideredirect . 1
    proc echo {txt} {
	# Catch this in case stdout doesn't exist.
	catch {puts stdout $txt}
    }
}

set now [clock clicks]
wm withdraw .
update
# In case we started up in a strange way.
cd [file dirname [info script]]

if {[info tclversion] < 8.1} {
    error "Sorry, we require Tcl version 8.1 or newer;\
      you have [info tclversion]"
    return
}

if {[catch {package require msgcat}]} {
    echo "The msgcat package is required if you want to use\
      a localised version of Alphatk"
    namespace eval msgcat {}
    proc msgcat::mc {str} { return $str }
    proc msgcat::mclocale {args} {}
    proc msgcat::mcunknown {locale string} { return $string }
} else {
    msgcat::mcload Localisation
    # auto_load
    catch {msgcat::mcunknown blah blah}
}

if {[catch {package require Tclx}]} {
    echo "Installing Tclx speeds up some operations inside Alphatk"
}
if {[catch {package require ftp}]} {
    echo "Install the ftp package (part of 'tcllib') to let Alphatk\
      update itself more simply;"
}
if {[info tclversion] < 8.3} {
    echo "Suggestion: upgrade to Tcl 8.3 or 8.4 and Alphatk will\
      run more quickly."
}

# The AlphaTcl library uses its own implementation of unknown,
# auto_load, auto_mkindex etc.  This turns that off.  AlphaTcl still
# uses a different version of auto_mkindex/auto_reset to the core, but
# the tclIndex file formats are identical, and the core's unknown is
# therefore used.
set useStandardTclIndices 1
# For debugging, you can force this on here.
# console show

namespace eval alpha {}

if {[info commands quit] == ""} {
    rename exit quit
    proc exit {{returnCode ""}} {}

    # MacTk seems to have a bug if we exit using exit.
    # So instead we destroy the root window.  This might
    # be a better default for all platforms, to allow for
    # embedding of Alphatk, or perhaps starting up Alphatk
    # in an interpreter which forms part of a bigger application.
    proc quit {} {
	# We don't really have to do this, but it is nicer
	# of us to clean up the things we know about.
	message "Exiting"
	global alpha::unmount
	if {[info exists alpha::unmount]} {
	    vfs::zip::Unmount $alpha::unmount
	}
	destroy .
    }
}

namespace eval alpha {
    namespace eval index {}
    namespace eval cache {}
    #set haveBasicKeys 1
    proc checkFileExists {dir {isdir 0}} {
	if {![file exists $dir] || \
	  ($isdir && ![file isdirectory $dir]) \
	  || (!$isdir && ![file isfile $dir])} {
	    tk_messageBox -title "Fatal Alphatk error" -message \
	      "Alphatk cannot find its\
	      [expr {$isdir ? {directory} : {file}}]\
	      '[file join [pwd] $dir]'.\
	      You should reinstall Alphatk.  Goodbye."
	    quit
	}
    }
}

switch -- $tcl_platform(platform) {
    "unix" {
	# we have send
    }
    "windows" {
	# On windows we're supposed to use the 'dde' package instead of send.
	proc send {args} {
	    echo "No send! Ignored $args"
	}
	if {[catch {package require dde}]} {
	    echo "Problems loading dde package.  Interaction with other"
	    echo "windows applications will be much impaired."
	} else {
	    dde servername Alpha
	    update idletasks
	    # Under very strange conditions, the dde services on
	    # Windows can hang completely, and this line never returns.
	    # In this case Alphatk will not startup (it will also
	    # hang), and you should simply reboot your machine.
	    echo "Creating dde server. Current services:\
	      [dde services TclEval {}]"
	}
    }
    "macintosh" {
	if {[catch {
	    package require Comm 3
	    # let's hope this isn't used!
	    comm config -port 1197 -local 1 -listen 1
	}]} {
	    echo "Problems loading Comm package or port already in use."
	    echo "Won't be able to use remote invocation."
	}
	# On MacOS, Tk calls this procedure when it receives 
	# 'open document' events, for example if you drag
	# a file onto the Wish application.
	proc tkOpenDocument {args} {
	    foreach f $args {
		edit $f
	    }
	}
	if {[info commands ::tk::unsupported::MacWindowStyle] == ""} {
	    namespace eval ::tk::unsupported {}
	    proc ::tk::unsupported::MacWindowStyle {args} {
		eval [list unsupported1] $args
	    }
	}
    }
    default {
	error "No known platform"
    }
}

if {[file exists AlphaTcl.zip]} {
    if {![catch {
	package require vfs
	set alpha::unmount [vfs::zip::Mount AlphaTcl.zip AlphaTcl.zip]
	set HOME [file join [pwd] AlphaTcl.zip Alpha]
    }]} {
	# we mounted successfully
	catch {puts stdout "Mounted AlphaTcl vfs"}
    }
}

if {![info exists HOME]} {
    alpha::checkFileExists [file join Alpha Tcl] 1

    # Do this stuff to deal with aliases/links more easily later on.
    # We want 'HOME' to point to the parent of 'Tcl'.
    set HOME [file join [pwd] Alpha Tcl]
    if {[file type $HOME] == "link"} {
	set HOME [file readlink $HOME]
    }
    set HOME [file dirname $HOME]
    # Fix windows path
    if {$tcl_platform(platform) == "windows"} {
	catch {set HOME [file attributes $HOME -longname]}
    }

    # If we are using a packaged up AlphaTcl.zip, then mount that
    # as our home directory, if we have the vfs package available
    if {[file exists [file join $HOME AlphaTcl.zip]]} {
	if {![catch {
	    package require vfs
	    set alpha::unmount [vfs::zip::Mount \
	      [file join $HOME AlphaTcl.zip] [file join $HOME AlphaTcl.zip]]
	    set HOME [file join $HOME AlphaTcl.zip]
	}]} {
	    # we mounted successfully
	    catch {puts stdout "Mounted AlphaTcl vfs"}
	}
    }
}

alpha::checkFileExists [file join $HOME AlphaCore] 1
alpha::checkFileExists [file join $HOME AlphaCore Images] 1

foreach imagefile [glob -directory [file join $HOME AlphaCore Images] *.gif] {
    image create photo [file root [file tail $imagefile]] -file $imagefile
}
unset imagefile

switch -- $tcl_platform(platform) {
    "windows" {
	set alpha::modifier_keys [list "Alt" "alt" "Meta" "meta"]
	set alpha::command_key "Alt"
	set alpha::option_key "Meta"
	# Add these defaults in.
	event add <<Paste>> <Alt-v>
	event add <<Copy>> <Alt-c>
	event add <<Cut>> <Alt-x>
    }
    "macintosh" {
	set alpha::modifier_keys [list "Command" "cmd" "Option" "opt"]
	set alpha::command_key "Command"
	set alpha::option_key "Option"
    }
    "unix" -
    default {
	set alpha::modifier_keys [list "Alt" "alt" "Meta" "meta"]
	set alpha::command_key "Alt"
	set alpha::option_key "Meta"
    }
}

proc alpha::inAlphaHierarchy {filename} {
    global HOME PREFS tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	catch {set filename [file join [file attributes $filename -longname]]}
    } else {
	set filename [file join $filename]
    }
    if {[file pathtype $filename] == "relative"} {
	set filen [file join [pwd] $filename]
    } else {
	set filen $filename
    }
    # Is file in HOME or PREFS?
    if {([string first [file join $HOME] $filen] != 0) \
      && !([file type [file join $HOME]] == "link" \
      && [string first [file readlink [file join $HOME]] $filen] == 0)} {
	if {[info exists PREFS] && [file exists $PREFS]} {
	    if {([string first [file join $PREFS] $filen] != 0) \
	      && !([file type [file join $PREFS]] == "link" \
	      && [string first [file readlink [file join $PREFS]] $filen] == 0)} {
		return 0
	    } else {
		return 1
	    }
	} else {
	    return 0
	}
    } else {
	return 1
    }
}

set alpha::defaultEncoding macRoman

# The default encoding is ISOmacRoman if we're developing with cvs,
# but if we're just a user, we will have downloaded a macRoman encoding
# version.  However, this file itself will have been sourced in your
# system encoding!  We really need a way of distinguishing between all
# of these cases automatically.

if {0} {
scan "" %c ellipsisVal
scan "" %c bulletVal

if {($ellipsisVal == 8230) && ($bulletVal == 8226)} {
    # We were saved in a cp1252 or similar encoding, which
    # means we're probably a developer using cvs and therefore
    # the ISOmacRoman encoding
    set alpha::defaultEncoding ISOmacRoman
} elseif {($ellipsisVal == 352) && ($bulletVal == 8364)} {
    # We were saved as ISOmacRoman, but sourced into cp1252.
    set alpha::defaultEncoding ISOmacRoman
} elseif {($ellipsisVal == 201) && ($bulletVal == 165)} {
    # We were saved as macRoman, but sourced into cp1252 or other
    set alpha::defaultEncoding macRoman
}

# TURN ON FOR ENCODING DEBUGGING
if {0} {
console show
update
puts "$ellipsisVal $bulletVal"
}

unset ellipsisVal
unset bulletVal
}

# This will be over-ridden later.
proc alpha::encodingFor {filename} {
    if {[alpha::inAlphaHierarchy $filename]} {
	return $::alpha::defaultEncoding
    } else {
	return ""
    }
}

# Convert the .tcl files from their default encoding when we source
# them.  Other files (prefs etc.) will be stored in their correct
# encoding, so we only apply this to files in the Alpha:Tcl directory.
if {[lsearch -exact [encoding names] $alpha::defaultEncoding] == -1} {
    catch {puts stderr "We don't have a $alpha::defaultEncoding encoding;\
      this will cause serious problems!"}
} else {
    rename source __enc_source
    # This will be replaced later.
    proc source {args} {
	set filename [lindex $args 0]
	if {[llength $args] > 1 \
	  && [lsearch -exact {-rsrc -rsrcid} $filename] != -1} {
	    return [uplevel 1 __enc_source $args]
	}
	if {![file exists $filename] || ![file readable $filename]} {
	    return [uplevel 1 [list __enc_source $filename]]
	}
	set enc [alpha::encodingFor $filename]
	if {![string length $enc]} {
	    uplevel 1 [list __enc_source $filename]
	} else {
	    if {[catch {
		set fileid [open $filename "r"]
		fconfigure $fileid -encoding $enc 
		set contents [read $fileid]
		close $fileid
	    } err]} {
		# This is pretty desperate if we get here!
		error "Error while pre-sourcing $filename : $err"
	    }
	    if {[info tclversion] > 8.3} {
		set oldscript [info script $filename]
	    }
	    uplevel 1 $contents
	    if {[info tclversion] > 8.3} {
		info script $oldscript
	    }
	}
    }
}

# Load in either byte-compiled or source scripts.
foreach _f {balloonHelp default alpha_menus alpha_commands alpha_macros \
  alpha_dialogs alpha_gui alpha_inter_app alpha_io \
  alpha_windows alpha_vars alpha_search \
  alpha_editing alpha_colouring alpha_dnd} {
    if {[file exists [file join $HOME AlphaCore ${_f}.tbc]]} {
	source [file join $HOME AlphaCore ${_f}.tbc]
    } else {
	source [file join $HOME AlphaCore ${_f}.tcl]
    }
}
foreach _f {enhancedTextwidget syntaxColouring multiSearch} {
    if {[file exists [file join $HOME AlphaCore TextWidget ${_f}.tbc]]} {
	source [file join $HOME AlphaCore TextWidget ${_f}.tbc]
    } else {
	source [file join $HOME AlphaCore TextWidget ${_f}.tcl]
    }
}
unset _f

proc alpha::setIcon {w} {}
set alpha::useMyIcons 0
switch -- $tcl_platform(platform) {
    "windows" {
	if {[catch {wm iconbitmap . -default [file join $HOME AlphaCore Alpha.icr]}]} {
	    if {![catch {package require Winico}]} {
		source [file join $HOME AlphaCore alpha_win_ico.tcl]
		set alpha::useMyIcons 1
		set alpha::icon [alpha::loadIcon \
		  [file join $HOME AlphaCore Alpha.icr]]
		proc alpha::setIcon {w} {
		    variable icon
		    variable bigIcon
		    variable smallIcon
		    winico setwindow $w $icon big   $bigIcon
		    winico setwindow $w $icon small $smallIcon
		}
		# This doesn't work here, because Tk over-rides it later!
		#alpha::setIcon .
	    } else {
		echo "Suggestion: install the 'Winico' package or\
		  Tk 8.3.3 or newer to let Alpha use nice icons\
		  for its windows."
	    }
	}
	
	source [file join $HOME AlphaCore alpha_win_print.tcl]
    }
    "macintosh" {
    }
    "unix" -
    default {
    }
}


set alpha::platform tk

alpha::checkFileExists [file join $HOME Tcl SystemCode] 1

namespace eval pos {}
proc pos::compare {args} {
    eval text_cmd compare $args
}
proc pos::math {i args} { 
    if {[llength $args] < 2} {
	return [text_cmd index "$i [join $args ""]c"]
    } else {
	return [text_cmd index "$i +[eval expr $args]c"]
    }
}

proc pos::diff {p1 p2} {
    if {[text_cmd compare $p1 > $p2]} {
	return [string length [text_cmd get $p2 $p1]]
    } else {
	return [string length [text_cmd get $p1 $p2]]
    }
}

namespace eval global {}

alpha::checkFileExists [file join $HOME Tcl SystemCode initAlphaTcl.tcl]

if {[catch [list source [file join $HOME Tcl SystemCode initAlphaTcl.tcl]]]} {
    catch {puts stderr "Problems sourcing initAlphaTcl.tcl"}
    global errorInfo
    catch {puts stderr $errorInfo}
}

if {[info commands newPref] == ""} {
    error "No newPref command; very bad startup problem"
}

# Load in either byte-compiled or source scripts.
foreach _f {alpha_startup} {
    if {[file exists [file join $HOME AlphaCore ${_f}.tbc]]} {
	source [file join $HOME AlphaCore ${_f}.tbc]
    } else {
	source [file join $HOME AlphaCore ${_f}.tcl]
    }
}
unset _f

# Go up one level, so that default saves/opens are not inside Alphatk
cd [file dirname [pwd]]

catch {puts stdout "Startup took [expr {[clock clicks] - $now}] clicks"}
unset now

