static char rcsid[] = "$Id: code.c,v 1.3 1998/08/25 17:16:41 nickm Exp $";

/* Routines to manipulate Code sequences and the bytecode interpreter. */
/*
 *
 *  NArray - a tcl extension for manipulating multidimensional arrays
 *
 *  Author: N. C. Maliszewskyj, NIST Center for Neutron Research, August 1998
 *          P. Klosowski        NIST Center for Neutron Research
 *  Original Author:
 *          S. L. Shen          Lawrence Berkeley Laboratory,     August 1994
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************
 *
 *
 * This software is copyright (C) 1994 by the Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#ifdef NO_TCL_H
#error Cannot find tcl.h
#else
#include <tcl.h>
#endif
#include <assert.h>
#include <ctype.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "narray.h"
#include "narrayInt.h"

static CompileState state_buf;
CompileState* narray_compile_state;

#define state narray_compile_state

#define MAX_OPCODES 10

static Code code_buf[MAX_OPCODES];
static Tcl_HashTable fn_table;

/* offset into array is
 i_n + d_n ( i_(n-1) + d_(n-1) ( i_(n-2) + 
                       d_(n-2) ( ... d_2 (i_1 + d_1 i_0) ... )))
*/

/* Compute linear offset from indices passed on Narray stack
   Add dimension length to each index <= 0 so that negative indices will
   work as well.
   ******WARNING, Will Robinson******* 
   This code section may need closer scrutiny.
 */

unsigned int
linearIndexFromStack(OperandStack stack, int sp, int nDims,  NArray *array){
  int i, idx;
  int offset;

  if (nDims <= 0) return 0;

  idx = (int) stack[sp - nDims + 1].value.n;
  while(idx < 0) {
    if (nDims == 1) {
      idx += array->length;
    } else {
      idx += array->dim_length[0];
    }
  }
  offset = idx;

  for (i=1; i < nDims; i++) {
    idx = (int) stack[sp - nDims + 1 + i].value.n;
    while (idx < 0) {
      idx += array->dim_length[i];
    }
    offset = idx + array->dim_length[i] * offset;
  }
  return offset;
}

/* Compute one of the indices from linear offset 
 */
unsigned int
nDimIndexFromLinearOffset(int offset, int which, NArray *array){
  int i, result;

  for (i=array->n_dims-1; i >= which; i--) {
    result = (offset % array->dim_length[i]) ;
    offset = (offset - result) / array->dim_length[i] ;
  }
  return result;
}

/* return the length of code minus the length of its header (a OP_INFO
 * and ptr to CodeInfo*).
 */
int NArray_CodeLength(Code* code)
{
  CodeInfo* info;
  GET_CODE_INFO(info, code);
  return info->len - 2;
}

void NArray_FreeCode(Code* code)
{
  CodeInfo* info;
  GET_CODE_INFO(info, code);
  ckfree(info);
}

Code* NArray_VarMakeCode(Code first, ...)
{
  va_list args;
  int i = 0, len;
  Code* result;
  CodeInfo* info;
    
  code_buf[i] = first;
  va_start(args, first);
  while ((code_buf[i] & OPCODE_MASK) != OP_END) {
    code_buf[++i] = va_arg(args, Code);
  }
  va_end(args);
  len = i;
  result = (Code*) ckalloc(sizeof(CodeInfo) +
			   (2 + MAX_OPCODES) * sizeof(Code));
  info = (CodeInfo*) result;
  info->len = len + 2;
  info->alloced = 2 + MAX_OPCODES;
  result = (Code*) (((char *) result) + sizeof(CodeInfo));
  result[0] = OP_INFO;
  result[1] = (unsigned long) info;
  memcpy(&result[2], code_buf, len * sizeof(Code));
  return result;
}

static Code* NArrayExpandCode(Code* a)
{
  CodeInfo* a_info;
  GET_CODE_INFO(a_info, a);
  a_info->alloced *= 2;
  a_info =
    (CodeInfo*) ckrealloc(a_info,
			  (sizeof(CodeInfo)
			   + a_info->alloced * sizeof(Code)));
  a = (Code*) (a_info + 1);
  a[1] = (unsigned long) a_info;
  return a;
}

Code* NArray_AppendCodeAndFree(Code* a, Code* b)
{
  CodeInfo* a_info;
  CodeInfo* b_info;
  GET_CODE_INFO(a_info, a);
  GET_CODE_INFO(b_info, b);
  if ((b_info->len - 2) > 0) {
    while (a_info->alloced < (a_info->len + (b_info->len - 2))) {
      a = NArrayExpandCode(a);
      GET_CODE_INFO(a_info, a);
    }
    memcpy(a + a_info->len, b + 2, (b_info->len - 2) * sizeof(Code));
    a_info->len += b_info->len - 2;
  }
  NArray_FreeCode(b);
  return a;
}

Code* NArray_AppendOps(Code* a, Code first, ...)
{
  va_list args;
  CodeInfo* a_info;
  int i = 0, len;
  GET_CODE_INFO(a_info, a);
  code_buf[i] = first;
  va_start(args, first);
  while ((code_buf[i] & OPCODE_MASK) != OP_END) {
    if ((code_buf[i] & OPCODE_MASK) == OP_FN)
      code_buf[++i] = va_arg(args, Code);
    code_buf[++i] = va_arg(args, Code);
  }
  va_end(args);
  len = i;
  if (len > 0) {
    while (a_info->alloced < (a_info->len + len)) {
      a = NArrayExpandCode(a);
      GET_CODE_INFO(a_info, a);
    }
    memcpy(a + a_info->len, code_buf, len * sizeof(Code));
    a_info->len += len;
  }
  return a;
}

Code* NArray_AppendDouble(Code* code, double d)
{
  int i;
  NArray_Closure* closure = state->closure;
  for (i = 0; i < closure->n_doubles; i++) {
    if (d == closure->double_table[i])
      break;
  }
  if (i == closure->n_doubles) {
    if (closure->double_table == 0) {
      closure->alloced_doubles = 4;
      closure->double_table =
	(NArrayFloat*) ckalloc(sizeof(NArrayFloat) * closure->alloced_doubles);
    } else if (closure->n_doubles == (closure->alloced_doubles - 1)) {
      closure->alloced_doubles *= 2;
      closure->double_table =
	(NArrayFloat*) ckrealloc(closure->double_table,
				 sizeof(NArrayFloat) * closure->alloced_doubles);
    }
    closure->double_table[closure->n_doubles++] = d;
  }
  return NArray_AppendOps(code, OP_PUSHD | i, OP_END);
}

void NArray_ExpandClosureIds(NArray_Closure* closure, char* id)
{
  int old_len;
  if (closure->id_table == 0) {
    closure->alloced_ids = 8;
    closure->id_table =
      (NArrayIdSlot*) ckalloc(sizeof(NArrayIdSlot)
			      * closure->alloced_ids);
    memset(closure->id_table, 0,
	   sizeof(NArrayIdSlot) * closure->alloced_ids);
    closure->vars =
      (NArrayVar*) ckalloc(sizeof(NArrayVar) * closure->alloced_ids);
    memset(closure->vars, 0, sizeof(NArrayVar) * closure->alloced_ids);
  } else {
    old_len = closure->alloced_ids;
    closure->alloced_ids *= 2;
    closure->id_table =
      (NArrayIdSlot*) ckrealloc(closure->id_table,
				sizeof(NArrayIdSlot)
				* closure->alloced_ids);
    memset(closure->id_table + old_len, 0,
	   sizeof(NArrayIdSlot) * (closure->alloced_ids - old_len));
    closure->vars = (NArrayVar*)
      ckrealloc(closure->vars,
		sizeof(NArrayVar) * closure->alloced_ids);
    memset(closure->vars + old_len, 0,
	   sizeof(NArrayVar) * (closure->alloced_ids - old_len));
  }
}

int NArray_CreateClosureIdSlot(NArray_Closure* closure, char* id)
{
  int i, first_free;
  first_free = -1;
  for (i = 0; i < closure->alloced_ids; i++) {
    if (closure->id_table[i].flags & NARRAY_SLOT_IN_USE) {
      if (!strcmp(id, closure->id_table[i].id))
	return i;
    } else if (first_free == -1)
      first_free = i;
  }
  if (first_free == -1) {
    first_free = closure->alloced_ids;
    NArray_ExpandClosureIds(closure, id);
  }
  closure->id_table[first_free].id = (char*) ckalloc(strlen(id)+1);
  strcpy(closure->id_table[first_free].id, id);
  closure->id_table[first_free].flags = NARRAY_SLOT_IN_USE;
  return first_free;
}

Code* NArray_AppendId(Code* code, char* id)
{
  NArray_Closure* closure = state->closure;
  int i = NArray_CreateClosureIdSlot(closure, id);
  closure->id_table[i].flags |= NARRAY_SLOT_VARIABLE;
  return NArray_AppendOps(code, OP_ID | i, OP_END);
}

/*
 * Strings and Id's are stored in the same table...
 */
Code* NArray_AppendString(Code* code, char* string)
{
  NArray_Closure* closure = state->closure;
  int i = NArray_CreateClosureIdSlot(closure, string);
  closure->id_table[i].flags |= NARRAY_SLOT_STRING;
  return NArray_AppendOps(code, OP_STRING | i, OP_END);
}

void NArray_UninternId(int id)
{
  NArray_Closure* closure = state->closure;
  ckfree(closure->id_table[id].id);
  closure->id_table[id].flags = 0;
}

static Code* NArrayPrintOp(NArray* array, Code* code, Code* start)
{
  CodeInfo* info;
  NArray_Closure* closure = &array->closure;
  unsigned int id;
  int n;
    
  GET_CODE_INFO(info, start);
  DECODE_INT(code, n);
  printf("%03u %02lu %-8d ",
	 code - start,
	 (*code & OPCODE_MASK) >> OPCODE_SHIFT,
	 n);
  switch ((*code) & OPCODE_MASK) {
  case OP_PUSH:
    DECODE_INT(code++, n);
    printf("PUSH %d\n", n);
    break;
  case OP_PUSHD:
    id = *code++ & OPERAND_MASK;
    printf("PUSHD %d (%g)\n", id, closure->double_table[id]);
    break;
  case OP_ADD:
    code++; printf("ADD\n");
    break;
  case OP_SUB:
    code++; printf("SUB\n");
    break;
  case OP_MUL:
    code++; printf("MUL\n");
    break;
  case OP_DIV:
    code++; printf("DIV\n");
    break;
  case OP_FN:
    printf("FN %d @ 0x%lx\n", n, *++code);
    code++;
    break;
  case OP_REF:
    code++; printf("REF\n");
    break;
  case OP_SET:
    code++; printf("SET\n");
    break;
  case OP_NEG:
    code++; printf("NEG\n");
    break;
  case OP_NOT:
    code++; printf("NOT\n");
    break;
  case OP_EQ:
    code++; printf("EQ\n");
    break;
  case OP_LE:
    code++; printf("LE\n");
    break;
  case OP_LT:
    code++; printf("LT\n");
    break;
  case OP_GE:
    code++; printf("GE\n");
    break;
  case OP_GT:
    code++; printf("GT\n");
    break;
  case OP_OR:
    code++; printf("OR\n");
    break;
  case OP_AND:
    code++; printf("AND\n");
    break;
  case OP_JUMPT:
    DECODE_INT(code++, n);
    printf("JUMPT %d\n", n);
    break;
  case OP_JUMP:
    DECODE_INT(code++, n);
    printf("JUMP %d\n", n);
    break;
  case OP_DUPLV:
    code++; printf("DUPLV\n");
    break;
  case OP_ID:
    id = *code++ & OPERAND_MASK;
    printf("ID %d (%s)\n", id, closure->id_table[id].id);
    break;
  case OP_MARK:
    code++; printf("MARK\n");
    break;
  case OP_THIS:
    code++; printf("THIS\n");
    break;
  case OP_ARRAY:
    id = *code++ & OPERAND_MASK;
    printf("ARRAY %d (%s)\n", id, closure->array_table[id].name);
    break;
  case OP_INFO: {
    CodeInfo* info;
    GET_CODE_INFO(info, code);
    printf("INFO len=%d alloced=%d\n", info->len, info->alloced);
    code += 2;
    break;
  }
  case OP_IREF:
    printf("IREF %d\n", (int) (*code++ & OPERAND_MASK));
    break;
  case OP_STRING:
    id = *code++ & OPERAND_MASK;
    printf("STRING %d (\"%s\")\n", id, closure->id_table[id].id);
    break;
  default:
    printf("<UNKNOWN>\n");
    code++;
  }
  return code;
}

void NArray_PrintCode(NArray* array, Code* start)
{
  CodeInfo* info;
  Code* code;
  int len;
    
  GET_CODE_INFO(info, start);
  code = start;
  len = info->len;
  while (code < (start + len)) {
    code = NArrayPrintOp(array, code, start);
  }
}

int NArray_ApplyCode(NArray* array, Code* start)
{
  OperandStack stack;
  LocationStack location_stack;
  MarkStack mark_stack;
  NArray* other_array;
  CodeInfo* info;
  Code* code;
  int len, n, sp, lsp, msp, tmp_sp, i, j, k, array_length, offset;
  char* errmsg;
  NArrayFloat* aref;
  NArray_Closure* closure = &array->closure;

  GET_CODE_INFO(info, start);
  if (array == 0)
    array_length = 1;
  else
    array_length = array->length;
  for (i = 0; i < array_length; i++) {
    len = info->len;
    sp = -1;
    lsp = -1;
    msp = -1;
    code = start;
    if (array->debug & DEBUG_TRACE)
      printf("Starting code for element %d:\n", i);
    while (code < (start + len)) {
      if (array->debug & DEBUG_TRACE)
	NArrayPrintOp(array, code, start);
	    
      /* dispatch the next opcode... we convert the OP_* to
       * a small int ... this seems to allow gcc to generate
       * faster code
       */
#define UNSHIFTED_OPCODE(x) (((unsigned long) x) >> OPCODE_SHIFT)

      switch (*code >> OPCODE_SHIFT) {
      case UNSHIFTED_OPCODE(OP_PUSH):
	DECODE_INT(code, n);
	code++;
	stack[++sp].value.n = (NArrayFloat) n;
	stack[sp].type = NARRAY_TYPE_FLOAT;
	break;
      case UNSHIFTED_OPCODE(OP_PUSHD):
	stack[++sp].value.n =
	  closure->double_table[*code++ & OPERAND_MASK];
	stack[sp].type = NARRAY_TYPE_FLOAT;
	break;
      case UNSHIFTED_OPCODE(OP_ADD):
	--sp; code++;
	stack[sp].value.n = stack[sp].value.n + stack[sp + 1].value.n;
	break;
      case UNSHIFTED_OPCODE(OP_SUB):
	--sp; code++;
	stack[sp].value.n = stack[sp].value.n - stack[sp + 1].value.n;
	break;
      case UNSHIFTED_OPCODE(OP_MUL):
	--sp; code++;
	stack[sp].value.n = stack[sp].value.n * stack[sp + 1].value.n;
	break;
      case UNSHIFTED_OPCODE(OP_DIV):
	--sp; code++;
	stack[sp].value.n = stack[sp].value.n / stack[sp + 1].value.n;
	break;
      case UNSHIFTED_OPCODE(OP_FN):
	/* we copy sp to tmp_sp and back out so that
	 * it remains eligible for register allocation
	 */
	tmp_sp = sp;
	DECODE_INT(code, n);
	errmsg = ((NArray_Fn) (*++code))(n, &tmp_sp, stack,
					 closure->interp);
	sp = tmp_sp;
	code++;
	if (errmsg != 0) {
	  array->errmsg = errmsg;
	  return 0;
	}
	break;
      case UNSHIFTED_OPCODE(OP_REF):
	code++;
	stack[++sp].value.n = *location_stack[lsp--].ptr;
	stack[sp].type = NARRAY_TYPE_FLOAT;
	break;
      case UNSHIFTED_OPCODE(OP_SET):
	code++;	--sp;
	*location_stack[lsp--].ptr = stack[sp + 1].value.n;
	break;
      case UNSHIFTED_OPCODE(OP_NEG):
	code++;
	stack[sp].value.n = -stack[sp].value.n;
	break;
      case UNSHIFTED_OPCODE(OP_NOT):
	code++;
	if (stack[sp].value.n == 0) stack[sp].value.n = 1;
	else stack[sp].value.n = 0;
	break;

#define IMPLEMENT_LOG_OP(op) code++; --sp; if (stack[sp].value.n op stack[sp+1].value.n) stack[sp].value.n = 1; else stack[sp].value.n = 0

      case UNSHIFTED_OPCODE(OP_EQ):
	IMPLEMENT_LOG_OP(==);
	break;
      case UNSHIFTED_OPCODE(OP_LE):
	IMPLEMENT_LOG_OP(<=);
	break;
      case UNSHIFTED_OPCODE(OP_LT):
	IMPLEMENT_LOG_OP(<);
	break;
      case UNSHIFTED_OPCODE(OP_GE):
	IMPLEMENT_LOG_OP(>=);
	break;
      case UNSHIFTED_OPCODE(OP_GT):
	IMPLEMENT_LOG_OP(>);
	break;
      case UNSHIFTED_OPCODE(OP_OR):
	IMPLEMENT_LOG_OP(||);
	break;
      case UNSHIFTED_OPCODE(OP_AND):
	IMPLEMENT_LOG_OP(&&);
	break;
      case UNSHIFTED_OPCODE(OP_JUMPT):
	if (stack[sp--].value.n != 0.0) {
	  DECODE_INT(code, n);
	  code += n;
	} else
	  code++;
	break;
      case UNSHIFTED_OPCODE(OP_JUMP):
	DECODE_INT(code, n);
	code += n;
	break;
      case UNSHIFTED_OPCODE(OP_INFO):
	code += 2;
	break;
      case UNSHIFTED_OPCODE(OP_ID):
	n = *code++ & OPERAND_MASK;
	location_stack[++lsp].ptr = &closure->vars[n];
	break;
      case UNSHIFTED_OPCODE(OP_MARK):
	mark_stack[++msp] = sp;
	code++;
	break;
      case UNSHIFTED_OPCODE(OP_THIS):
	offset = i + linearIndexFromStack(stack,	
					  sp, 
					  sp-mark_stack[msp], 
					  array);
	aref = &(array->storage[(offset % array->length)]);
	location_stack[++lsp].ptr = aref;
	sp = mark_stack[msp];
	--msp;
	code++;
	break;
      case UNSHIFTED_OPCODE(OP_ARRAY):
	k = *code++ & OPERAND_MASK;
	other_array = closure->array_table[k].array;
	/* the code here was using other_array->storage 
	   but array->dim which is OK if arrays are the same size...
	   we'll do it right */
	offset = i + linearIndexFromStack(stack,
					  sp, 
					  sp-mark_stack[msp], 
					  other_array);
	aref = &(other_array->storage[(offset % other_array->length)]);
	location_stack[++lsp].ptr = aref;
	sp = mark_stack[msp];
	--msp;
	break;
      case UNSHIFTED_OPCODE(OP_DUPLV):
	location_stack[lsp + 1].ptr = location_stack[lsp].ptr;
	lsp++;
	code++;
	break;
      case UNSHIFTED_OPCODE(OP_IREF):
	j = *code++ & OPERAND_MASK;
	if (j >= array->n_dims) {
	  array->errmsg = "reference index out of bounds";
	  return 0;
	}
	stack[++sp].value.n = 
	  (NArrayFloat) nDimIndexFromLinearOffset(i,j,array);
	stack[sp].type = NARRAY_TYPE_FLOAT;
	break;
      case UNSHIFTED_OPCODE(OP_STRING):
	stack[++sp].value.s =
	  closure->id_table[*code & OPERAND_MASK].id;
	stack[sp].type = NARRAY_TYPE_STRING;
	code++;
	break;
      default:
	assert(("unknown", 0));
      }
      if (sp >= MAX_OPERAND_STACK) {
	array->errmsg = "operand stack overflow";
	return 0;
      }
      if (lsp >= MAX_LOCATION_STACK) {
	array->errmsg = "location stack overflow";
	return 0;
      }
      if (msp >= MAX_MARK_STACK) {
	array->errmsg = "mark stack overflow";
	return 0;
      }
    }
  }
  return 1;
}

int NArray_yyinput(char* buf, int max_size)
{
  if (state->expr_ndx >= state->expr_length)
    return 0;
  strncpy(buf,
	  state->expression + state->expr_ndx,
	  max_size);
  state->expr_ndx += max_size;
  return max_size;
}

void NArray_yyerror(char* s)
{
  if (!state->use_error_msg)
    strncpy(state->error_msg, s, sizeof(state->error_msg));
  state->ok = 0;
}

Code* NArray_Compile(NArray* array, char* expression)
{
  extern int yydebug;
  int old_debug;
  Code* result;

  old_debug = yydebug;
  if (array->debug & DEBUG_PARSE)
    yydebug = 1;
  state = &state_buf;
  state->expression = expression;
  state->expr_ndx = 0;
  state->expr_length = strlen(expression);
  state->code = 0;
  state->use_error_msg = 0;
  state->ok = 1;
  state->arity_sp = -1;
  state->closure = &array->closure;
  yyparse();
  if (state->ok)
    result = state->code;
  else
    result = 0;
  yydebug = old_debug;
  return result;
}

Code NArray_LookupFn(int fn)
{
  Tcl_HashEntry* entry;
  entry = Tcl_FindHashEntry(&fn_table, state->closure->id_table[fn].id);
  if (entry == 0) return 0;
  return (Code) Tcl_GetHashValue(entry);
}

void NArray_CreateFn(char* name, NArray_Fn fn)
{
  Tcl_HashEntry* entry;
  int new_flag;
  entry = Tcl_CreateHashEntry(&fn_table, name, &new_flag);
  Tcl_SetHashValue(entry, (ClientData) fn);
}

int NArray_CodeInit(Tcl_Interp* interp)
{
  Tcl_InitHashTable(&fn_table, TCL_STRING_KEYS);
  return TCL_OK;
}
