
package provide WinCGIPkg 2.0

# author: Evan Rempel - erempel@UVic.CA  Feedback welcome.

namespace eval WinCGISpace {
array set [namespace current]::CGISYSTEM {}
array set [namespace current]::CGIACCEPT {}
array set [namespace current]::CGIVALUE {}
variable OutFile 0
variable OutFileName ""
variable InFileName ""

# ---------------------------------------------------------------------------
# Return the url decoded string.
#
proc URLDecode { Value } {

  set NewValue ""
  regsub -all {\+} $Value { } NewValue1
  regsub -all {\\} $NewValue1 {\\\\} NewValue2
  regsub -all {"} $NewValue2 {\\"} NewValue3
  regsub -all {%([0-9][0-9])} $NewValue3 {\\\1} NewValue4
  eval set FinalValue \"$NewValue4\"
  return $FinalValue
}

# ---------------------------------------------------------------------------
# Build a name=value context array "Values" from the "String"
# element. Also return the url decoded query string.
# 
proc QueryParse {CallString CallValues} {
# create variable paramaters for last two args, calling
# them values
  upvar $CallValues Values
  upvar $CallString String

  if {[info exist Values]} then {
    unset Values
  }
  variable MyVar ""
  variable MyValue ""

  set QList [split $String {&}]
  foreach v "$QList" {
    regexp {^([^=]+)=(.*)} "$v" Filler MyVar MyValue
    if {"$MyVar" != ""} then {
      set Values([URLDecode "$MyVar"]) [URLDecode "$MyValue"]
    }
  }
  set String [URLDecode $String]
}

# ---------------------------------------------------------------------------
# Find the next [] section in the ini file, returning the "Line"
# identifying the section name.
# 
proc FindSection {INIFile Line} {
  upvar $Line CurrentLine

  if {![info exist CurrentLine]} then {
    set CurrentLine ""
  }
  while { (![eof $INIFile]) && ([string range $CurrentLine 0 0] != {[})} {
    gets $INIFile CurrentLine
  }
  return $CurrentLine
}

# ---------------------------------------------------------------------------
# Process the [CGI] section of the ini file
# place all of its pairs into the SYSTEM component, parse the query string
# and place its components into the VALUE component.
# 
proc ProcessCGI {INIFile Line} {
  upvar $Line CurrentLine
  variable CGISYSTEM
  variable CGIVALUE

  set MyVar ""
  set MyValue ""
  gets $INIFile CurrentLine
  while {(![eof $INIFile]) && ([string range $CurrentLine 0 0] != {[}) } {
    if {"$CurrentLine" != ""} then {
      regexp {^([^=]+)=(.*)} $CurrentLine Filler MyVar MyValue
      set CGISYSTEM($MyVar) $MyValue
    }
    gets $INIFile CurrentLine
  }

# process the query string
  if {[info exist "CGISYSTEM(Query String)"]} then {
    set QString "$CGISYSTEM(Query String)"
    QueryParse QString QueryValues
    set "CGISYSTEM(Query String)" $QString

# make sure tha multiple valued variables are lists
    foreach v [lsort [array names QueryValues]] {
      set ind 0
      regexp {([^_]+)_([0-9]+)} $v dummy var ind
      if {$ind == 0} then {
        set CGIVALUE($v) $QueryValues($v)
      } else {
        if {$ind == 1} then {
          set CGIVALUE($var) [list "$CGIVALUE($var)" "$QueryValues($v)"]
        } else {
          lappend CGIVALUE($var) $QueryValues($v)
        }
      }
    }
  }
}

# ---------------------------------------------------------------------------
# Process the [Accept] section of the ini file
# place all of its pairs into the ACCEPT component.
# 
proc ProcessACCEPT {INIFile Line} {
  upvar $Line CurrentLine
  variable CGIACCEPT

  set MyVar ""
  set MyValue ""
  gets $INIFile CurrentLine
  while {(![eof $INIFile]) && ([string range $CurrentLine 0 0] != {[}) } {
    if {"$CurrentLine" != ""} then {
      regexp {^([^=]+)=(.*)} $CurrentLine Filler MyVar MyValue
      set CGIACCEPT($MyVar) $MyValue
    }
    gets $INIFile CurrentLine
  }
}

# ---------------------------------------------------------------------------
# Process the [SYSTEM] section of the ini file
# place the "GMT Offset" value int othe SYSTEM component.
# 
proc ProcessSYSTEM {INIFile Line} {
  upvar $Line CurrentLine
  variable CGISYSTEM
  variable OutFileName

  set MyVar ""
  set MyValue ""
  gets $INIFile CurrentLine
  while {(![eof $INIFile]) && ([string range $CurrentLine 0 0] != {[}) } {
    if {"$CurrentLine" != ""} then {
      regexp {^([^=]+)=(.*)} $CurrentLine Filler MyVar MyValue
      if {"$MyVar" == "GMT Offset"} then {
        set CGISYSTEM($MyVar) $MyValue
      }
      if {"$MyVar" == "Output File"} then {
        set CGISYSTEM(Unique) [file tail [file rootname $MyValue]]
        set OutFileName $MyValue
      }
    }
    gets $INIFile CurrentLine
  }
}

# ---------------------------------------------------------------------------
# Process the [FORM LITERAL] section of the ini file
# placing the values into the VALUE component.
# 
proc ProcessFORMLITERAL {INIFile Line} {
  upvar $Line CurrentLine
  variable CGIVALUE

  set MyVar ""
  set MyValue ""
  gets $INIFile CurrentLine
  while {(![eof $INIFile]) && ([string range $CurrentLine 0 0] != {[}) } {
    if {"$CurrentLine" != ""} then {
      regexp {^([^=]+)=(.*)} $CurrentLine Filler MyVar MyValue
# make sure tha multiple valued variables are lists
      set ind 0
      regexp {([^_]+)_([0-9]+)} $MyVar dummy var ind
      if {$ind == 0} then {
        set CGIVALUE($MyVar) $MyValue
      } else {
        if {$ind == 1} then {
          set CGIVALUE($var) [list "$CGIVALUE($var)" "$MyValue"]
        } else {
          lappend CGIVALUE($var) $MyValue
        }
      }
    }
    gets $INIFile CurrentLine
  }
}

# ---------------------------------------------------------------------------
# Process the [FORM EXTERNAL] section of the ini file
# placing the values into the VALUE component.
# 
proc ProcessFORMEXTERNAL {INIFile Line} {
  upvar $Line CurrentLine
  variable CGIVALUE

  set MyVar ""
  set MyValue ""
  gets $INIFile CurrentLine
  while {(![eof $INIFile]) && ([string range $CurrentLine 0 0] != {[}) } {
    if {"$CurrentLine" != ""} then {
      regexp {^([^=]+)=(.*)} $CurrentLine Filler MyVar MyValue
      set Path ""
      set Size 0
# get the data from the external file
      regexp {^(.*) ([0-9]+)$} $MyValue Filler Path Size
      set tempfile [open $Path r]
      set TheValue [read $tempfile $Size]
      close $tempfile
      
# make sure tha multiple valued variables are lists
      set ind 0
      regexp {([^_]+)_([0-9]+)} $MyVar dummy var ind
      if {$ind == 0} then {
        set CGIVALUE($MyVar) $TheValue
      } else {
        if {$ind == 1} then {
          set CGIVALUE($var) [list "$CGIVALUE($var)" "$TheValue"]
        } else {
          lappend CGIVALUE($var) $TheValue
        }
      }
    }
    gets $INIFile CurrentLine
  }
}


}
# -*-*-* End of Namespace WinCGISpace

# ---------------------------------------------------------------------------
# Parse the ini file into the three name=value pair context arrays
# SYSTEM, ACCEPT and VALUE
#
proc CGIRead {inifile} {
  variable WinCGISpace::InFileName

  set WinCGISpace::InFileName "$inifile"
  if {[catch {set infile [open $inifile]}] == 0} then {
    set CGIError 0
    while {[WinCGISpace::FindSection $infile Line] != ""} {
      switch [string toupper $Line] {
        {[CGI]} {
          WinCGISpace::ProcessCGI $infile Line
        }
        {[ACCEPT]} {
          WinCGISpace::ProcessACCEPT $infile Line
        }
        {[SYSTEM]} {
          WinCGISpace::ProcessSYSTEM $infile Line
        }
        {[EXTRA HEADERS]} {
          set Line ""
        }
        {[FORM LITERAL]} {
          WinCGISpace::ProcessFORMLITERAL $infile Line
        }
        {[FORM EXTERNAL]} {
          WinCGISpace::ProcessFORMEXTERNAL $infile Line
        }
        {[FORM FILE]} {
          gets $infile Line
        }
        {[FORM HUGE]} {
          gets $infile Line
        }
        default {
          gets $infile Line
        }
      }
    }
    close $infile

# prepare the output file
    set WinCGISpace::OutFile [open $WinCGISpace::OutFileName "w"]
    set "WinCGISpace::CGISYSTEM(Output File)" $WinCGISpace::OutFile
  } else {
    set CGIError 1
  }
  return $CGIError
}

# ---------------------------------------------------------------------------
# Place a string into the return data stream
# Just like puts

proc CGIWrite { args } {
  variable WinCGISpace::OutFile

  set NumberArgs [llength $args]
  if {$NumberArgs == 1} then {
    puts $WinCGISpace::OutFile [lindex $args 0]
  } elseif {$NumberArgs == 2} then {
    if {[lindex $args 1] == "-nonewline"} then {
      puts -nonewline $WinCGISpace::OutFile [lindex $args 1]
    } else {
      error "CGIWrite: unknown argument [lindex $args 0]" 
    }
  }
}

# ---------------------------------------------------------------------------
# Return the value of a SYSTEM variable
#
proc CGISystem { Key } {
  variable WinCGISpace::CGISYSTEM

  if {[info exist WinCGISpace::CGISYSTEM($Key)]} then {
    return $WinCGISpace::CGISYSTEM($Key)
  } else {
    return ""
  }
}

# ---------------------------------------------------------------------------
# Return the value of a VALUE variable
#
proc CGIValue { Key } {
  variable WinCGISpace::CGIVALUE

  if {[info exist WinCGISpace::CGIVALUE($Key)]} then {
    return $WinCGISpace::CGIVALUE($Key)
  } else {
    return ""
  }
}

# ---------------------------------------------------------------------------
# Return the value of a ACCEPT variable
#
proc CGIAccept { Key } {
  variable WinCGISpace::CGIACCEPT

  if {[info exist WinCGISpace::CGIACCEPT($Key)]} then {
    return $WinCGISpace::CGIACCEPT($Key)
  } else {
    return ""
  }
}

# ---------------------------------------------------------------------------
# Produce a fixed pitch display of all of the information parsed.
# CGIDebug local/remote
# if local, display to standard out,
#    remote, display to CGI output channel
proc CGIDebug { flag } {
  variable WinCGISpace::CGISYSTEM
  variable WinCGISpace::CGIVALUE
  variable WinCGISpace::CGIACCEPT
  variable WinCGISpace::OutFile
  variable WinCGISpace::InFileName

  if {"$flag" == "local"} then {
    set Output stdout
  } else {
    set Output $WinCGISpace::OutFile
  }

  puts $Output "SYSTEM"
  foreach var [lsort [array names WinCGISpace::CGISYSTEM]] {
    set Value $WinCGISpace::CGISYSTEM($var)
    if {[string range $Value 0 0] == "\{"} then {
      puts $Output "    $var (Multiple value):"
      foreach v $Value {
        puts $Output "        $v"
      }
    } else {
      puts $Output "    $var: $Value"
    }
  }

  puts $Output "ACCEPT"
  foreach var [lsort [array names WinCGISpace::CGIACCEPT]] {
    set Value $WinCGISpace::CGIACCEPT($var)
    if {[string range $Value 0 0] == "\{"} then {
      puts $Output "    $var (Multiple value):"
      foreach v $Value {
        puts $Output "        $v"
      }
    } else {
      puts $Output "    $var: $Value"
    }
  }

  puts $Output "VALUE"
  foreach var [lsort [array names WinCGISpace::CGIVALUE]] {
    set Value $WinCGISpace::CGIVALUE($var)
    if {[string range $Value 0 0] == "\{"} then {
      puts $Output "    $var (Multiple value):"
      foreach v $Value {
        puts $Output "        $v"
      }
    } else {
      puts $Output "    $var: $Value"
    }
  }

  puts $Output ""
  puts $Output "WinCGI INI File"
  set infile [open $WinCGISpace::InFileName "r"]
  while {[gets $infile Line] != -1} {
    puts $Output "    $Line"
  }
  close $infile
}
