### ===========================================================================
### -package  : SimpleError
# -version    : 0.2
# -purpose    : Error handling.
# -overview   : 
#     This package centralizes the handling of errors.  A declared error is
#  made of five elements:  the error code, the error messsage, the error
#  explanation, the corrective action and a translation string.  Procedures
#  are provided to declare, delete, throw and explain errors, get information
#  about an error, obtain the list of errors and catch errors explaining them.
#
# -usage      :
#     Errors are declared via K<::Simple::Error::declare>.  An error code and
#  error message are mandatory while the error explanation, corrective action,
#  translation string and explained error message, which are used by
#  K<::Simple::error::explain> to explain an error, are optional.  In fact
#  these optional error attributes are ignored (that is, not stored) unless
#  the store details package option is true (this option is false by default).
#
#     Declared errors can be thrown via K<::Simple::Error::throw> and
#  explained via K<::Simple::Error::explain>.  The
#  K<::Simple::Package::information> procedure provides several subcommands to
#  query information about declared errors:  K<exists>, K<message>,
#  K<explanation> and K<corrective>.  Refer to the description of each
#  procedure below for further information.
#
#     The K<::Simple::Error::catch-explain> is similar to the K<catch>
#  command but explains any declared error the evaluated script may throw.
#
#  Use K<::Simple::Error::delete> to delete a declared error.
#
# -keywords   : error catch explain throw
# -commands   :
#
#  * K<::Simple::Error::declare> errorCode errorMessage errorExplanation
#    ?correctiveAction? ?translation? ?explainedErrorMessage?
#    Declares an error.
#
#  * K<::Simple::Error::delete> errorCode
#    Deletes an error.
#
#  * K<::Simple::Error::information declared> ?pattern?
#    Returns the list of declared errors.
#
#  * K<::Simple::Error::information exists> errorCode
#    Returns whether an error has been declared.
#
#  * K<::Simple::Error::information message> errorCode
#    Returns an error message format string.
#
#  * K<::Simple::Error::information explanation> errorCode
#    Returns an error explanation text.
#
#  * K<::Simple::Error::information corrective> errorCode
#    Returns an error corrective action text.
#
#  * K<::Simple::Error::information explained> errorCode
#    Returns an error explained error message.
#
#  * K<::Simple::Error::explain> errorCode
#    Explains an error.
#
#  * K<::Simple::Error::throw> errorCode ?args?
#    Throws an error.
#
#  * K<::Simple::Error::catch-explain> script ?variableName?
#    Catches an error and explains it.
#
#  * K<::Simple::Error::configure> ?-storedetails extbool?
#    Configures the package options.
#
#  * K<::Simple::Error::cget> ?-storedetails?
#    Gets the package options.
#
# -variables  :
#
#  { StoreDetails     -boolean 0     {Stores the value of the I<-storedetails>
#                                     package option}}
#
# -options    :
#
#  * I<-storedetails>:  whether the error details (error explanation,
#    corrective action and explained error message) of errors declared via
#    K<::Simple::Error::declare> are stored or not.  While not storing the
#    error details saves some memory, the K<::Simple::Error::explain>
#    procedure is really only useful for errors with stored details.  Notice
#    that only those errors declared B<after> modyfing this option are
#    affected by the change.
#
# -examples   :
#
#  # Install the package
#  package require SimpleError
#
#  # Set the package option to store the error details
#  ::Simple::Error::configure -storedetails true
#
#  # Declare an error
#  ::Simple::Error::declare FILE-NOT-FOUND {
#     no file named %s in directory %s
#  } {
#     There is no file named <file> in directory <dir>
#  } {
#     Make sure both the file and directory are correct
#  }
#
#  # Get the error correction action
#  # This displays the following:
#  #    Make sure both the file and directory are correct
#  puts [::Simple::Error::information corrective FILE-NOT-FOUND]
#
#  # Throw the error, then catch and explain it
#  # This displays the following:
#  #    ===================== Error explanation ======================
#  #    Error code   : FILE-NOT-FOUND
#  #    Error message: no file named <file> in directory <dir>
#  #    Explanation  : There is no file named <file> in directory <dir>.
#  #    Correction   : Make sure both the file and directory are correct.
#  #    ================== End of error explanation ==================
#  ::Simple::Error::catch-explain {
#     ::Simple::Error::throw FILE-NOT-FOUND foo.tcl /usr/home/
#     } result
#  puts $::errorInfo
#
#  # Delete the error
#  ::Simple::Error::delete FILE-NOT-FOUND
#
#  # The error is not decalred anymore
#  # This displays the following:
#  #    0
#  puts [::Simple::Error::information exists FILE-NOT-FOUND]
#
# -details    : 
#  * Each declared error internal represetntation is stored in one element of
#    an array named V<%Errors%> in the namespace pointed to by the error code
#    namespace qualifiers.  The name of the array element is the error code
#    tail.  The array element is a list with the following elements (only the
#    first is customary):
#     0: error message format string
#     1: error explanation text
#     2: error corrective action text
#     3: error translation string
#
# -todo       :
#  * Provide a K<::Simple::Error::modify> procedure.
#
# -history    :
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleError) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require Tcl 8.0
namespace eval ::Simple::Error::Priv {}
package provide SimpleError 0.2

# Utility procedure to install the package
proc ::Simple::Error::Priv::install {} {

   ### Create the package options
   set ::Simple::Error::StoreDetails 0

   ### Declare the package errors
   ::Simple::Error::declare ::Simple::Error::FORMAT-ERROR {
      error formatting error "%s" (error message is {%s}, arguments are {%s})
   } {
      An error ocurred when formatting error <error code>, whose error message
      is <error message>, with run-time arguments <arguments>
   } {
      Check the error message format and the number of run-time arguments
   }

   ::Simple::Error::declare ::Simple::Error::UNDECLARED-ERROR {
      undeclared error "%s"
   } {
      Error <error code> has not been declared
   } {
      Use K<::Simple::Error::declare> to declare the error
   }

   ::Simple::Error::declare ::Simple::NON-EXISTING-NAMESPACE {
      non-existing namespace "%s"
   } {
      Namespace <namespace> does not exist
   } {
      Use K<namespace eval> to create the namespace
   }

   ::Simple::Error::declare ::Simple::NO-SUBCOMMAND {
      wrong # args: should be "%s option ?arg ...?"
   } {
      The base command <command> has been invoked with no subcommand
   } {
      Invoke the base command giving a subcommand
   }

   ::Simple::Error::declare ::Simple::BAD-SUBCOMMAND {
      bad option "%s": must be %s
   } {
      Subcommand <subcommand> is not in the <list of allowed subcommands> for
      the base command
   } {
      Invoke the base command giving a subcommand among those in the list
   }

   ::Simple::Error::declare ::Simple::BAD-SUBCOMMAND-ARGS {
      wrong # args: should be "%s%s"
   } {
      Subcommand procedure <subcommand procedure> has been invoked with an
      incorrect <argument list>
   } {
      Invoke the subcommand procedure matching the argument list
   }

   ::Simple::Error::declare ::Simple::AT-LEAST-ONE-OPTION {
      at least one option required, options are %s
   } {
      No option was supplied
   } {
      Supply an option from the <list of options>
   } {<list of options>}

   ::Simple::Error::declare ::Simple::BAD-OPTION {
      bad option "%s": must be %s
   } {
      Option <option> is not in the <list of options>
   } {
      Supply an option among those in the list
   }

   ::Simple::Error::declare ::Simple::BAD-OPTION-ARG-TYPE {
      invalid value "%s" for option "%s" of type "%s"
   } {
      Option <option> has been supplied with a <bad argument>
      which does not conform to the option <type>
   } {
      Supply a correct argument for the option type
   } {<bad argument> <option> <type>}

   ::Simple::Error::declare ::Simple::BAD-OPTION-ARG-VALUE {
      invalid value "%s" for option "%s": must be %s
   } {
      Option <option> has been supplied with a <bad argument>
      which does not fulfill the <condition>
   } {
      Supply an argument for the option fulfilling the condition
   } {<bad argument> <option> <condition>}

   # Delete this procedure
   rename ::Simple::Error::Priv::install {}
}

### ===========================================================================
### -command  : ::Simple::Error::Priv::split-name
# -purpose    : Splits a name.
# -overview   : 
#     This procedure splits a name into its namespace qualifiers and tail
#  components and returns the pair.  If the name is unqualified, the global
#  namespace is assumed.
#
# -arguments  :
#  { name             -name          {Name}}
#
# -returns    : The name namespace and tail.
#
proc ::Simple::Error::Priv::split-name {
   name
} {
   ### Unqualified name
   if {![regexp -- ^(.*::)(.*)$ $name tmp namespace unqualifiedName]} {

      ### Assume global namespace
      set namespace ::
      set unqualifiedName $name
   }

   ### Return the name namespace and tail
   list $namespace $unqualifiedName
}

### ===========================================================================
### -command  : ::Simple::Error::declare
# -purpose    : Declares an error.
# -overview   :
#     This procedure is used to declare a new error.
#
#     An error is made of six elements:  the error code, the error messsage,
#  the error explanation, the corrective action, the translation string and
#  the explained error message.  The first two are mandatory while the last
#  four are optional.
#
#     The error code must be a Tcl qualified name.  The qualifiers point to
#  the namespace in which the error information will be stored; the namespace
#  must exist.  The tail is the actual code of the error to be thrown via
#  K<::Simple::Error::throw>.  The error message is a format string as
#  accepted by the K<format> command.  This format string is used by
#  K<::Simple::Error::throw> to format the error message.
#
#     The error explanation, corrective action, translation string and
#  explained error message are used by K<::Simple::Error::explain>; see that
#  procedure for the details.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#  { errorMessage     -string        {Error message format string}}
#  { errorExplanation 
#                     -string        {Error explanation text}}
#  {?correctiveAction?
#                     -string {}     {Corrective action text}}
#  {?translation?     -string {}     {Translation string}}
#  {?explainedErrorMessage?
#                     -string {}     {Formatted error message}}
#
# -examples   : See the examples section of the P<::Simple::Error::explain>
#     procedure.
#
proc ::Simple::Error::declare {
   errorCode
   errorMessage
   errorExplanation
   {correctiveAction {}}
   {translation {}}
   {explainedErrorMessage {}}
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### The namespace does not exist
   if {[catch {namespace children $namespace}]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::NON-EXISTING-NAMESPACE\
         [namespace qualifiers $namespace]
   }

   ### Add the namespace to the list of namespaces with declared errors
   set ::Simple::Error::Namespaces($namespace) [list]

   ### Clean the error message
   #   Trim and remove carriage returns
   regsub -all "\n *" [string trim $errorMessage] { } errorMessage

   ### Error details must be stored
   if {$::Simple::Error::StoreDetails} {

      ### Clean the error details
      #   Trim and remove carriage returns
      foreach item\
         {errorExplanation correctiveAction translation explainedErrorMessage} {

         regsub -all "\n *" [string trim [set $item]] { } $item
      }

      ### Store the error internal representation, including its details
      set ${namespace}%Errors%($errorCodeTail) [list $errorMessage\
         $errorExplanation $correctiveAction $translation\
         $explainedErrorMessage]

   ### Error details must not be stored
   } else  {

      ### Store the error internal representation
      set ${namespace}%Errors%($errorCodeTail) [list $errorMessage]
   }
}

### ===========================================================================
### -command  : ::Simple::Error::delete
# -purpose    : Deletes an error.
# -overview   :
#     This procedure deletes a declared error.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#
proc ::Simple::Error::delete {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Undeclared error
   if {![info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode

   ### Declared error
   } else {

      ### Delete the error
      unset ${namespace}%Errors%($errorCodeTail)
   }
}

### ===========================================================================
### -command  : ::Simple::Error::Priv::format-error-message
# -purpose    : Formats an error message.
# -overview   :
#     This procedure formats a declared error message.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#  { errorMessage     -string        {Error message format string}}
#  { arguments        -list          {Error run-time arguments}}
#
# -returns    : The formatted error message.
#
proc ::Simple::Error::Priv::format-error-message {
   errorCode
   errorMessage
   arguments
} {
   ### Error formatting the error message
   if {[catch {set answer [eval format [list $errorMessage] $arguments]}]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::FORMAT-ERROR\
         $errorCode $errorMessage $arguments

   ### No error formatting the error message
   } else {

      ### Return the formatted error message
      set answer
   }
}

### ===========================================================================
### -command  : ::Simple::Error::information
# -purpose    : Base command for package K<information> subcommands.
# -access     : Private
# -overview   :
#     This is the base command for the K<::Simple::Error::information>
#  subcommand procedures.
#
# -arguments  :
#  { args             -list          {Subcommand and subcommand arguments}}
#
proc ::Simple::Error::information {
   args
} {
   #========= %BASE-COMMAND%

   ### Get the subcommand and full command names
   set subcommand [lindex $args 0]
   set fullCommand ::Simple::Error::information\ $subcommand

   ### Evaluate the subcommand procedure
   ### Error from subcommand procedure
   #   The uplevel here ensures that the call to the base command
   #   is hidden to the command procedure body
   if {[catch {uplevel [list $fullCommand] [lrange $args 1 end]} result]} {

      ### No subcommand given
      if {[llength $args] == 0} {

         ### Throw error
         ::Simple::Error::throw ::Simple::NO-SUBCOMMAND\
            ::Simple::Error::information

      ### Non-existing subcommand
      } elseif {![string compare "invalid command name \"$fullCommand\""\
         $result]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND $subcommand\
            {corrective, declared, exists, explanation or message}

      ### Invalid arguments
      } elseif {![string compare "called \"$fullCommand\" with too many\
         arguments" $result] || [string match {no value given for\
         parameter*} $result]} {

         ### Compose the correct arguments
         switch -exact -- $subcommand {
            declared {
               set arguments { ?pattern?}
            }
            exists -
            message -
            explanation -
            corrective {
               set arguments { errorCode}
            }
         }

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND-ARGS\
            $fullCommand $arguments

      ### Other error
      } else {

         ### Rethrow the subcommand procedure error
         return -code error -errorcode $::errorCode $result
      }

   ### No error from subcommand procedure
   } else {

      ### Return the subcommand procedure return value
      set result
   }
}

### ===========================================================================
### -command  : ::Simple::Error::information declared
# -purpose    : Returns the list of declared errors.
# -overview   :
#     This procedure returns the list of declared errors matching the given
#  pattern.
#
# -arguments  :
#  {?pattern?         -pattern {}    {Pattern}}
#
# -returns    : The list of declared errors matching the given pattern.
# -remarks    :
#  * Use no pattern to obtain the complete list of declared errors.
#
proc {::Simple::Error::information declared} {
   {pattern {}}
} {
   ### Loop over namespaces with declared errors
   foreach namespace [array names ::Simple::Error::Namespaces] {

      ### Assume there are no declared errors in the namespace
      set noErrors 1

      ### Loop over declared errors
      foreach errorCode [array names ${namespace}%Errors%] {

         ### Add the error to the list of declared errors
         set tmp(${namespace}$errorCode) {}

         ### There is at least one declared error in the namespace
         set noErrors 0
      }

      ### Deleted namespace or no declared errors in it
      if {$noErrors} {

         ### Delete the namespace from the list of
         ###    namespaces with declared errors
         unset ::Simple::Error::Namespaces($namespace)
      }
   }

   ### Return the list of declared errors
   if {[string compare $pattern {}]} {
      array names tmp $pattern
   } else {
      array names tmp
   }
}

### ===========================================================================
### -command  : ::Simple::Error::information exists
# -purpose    : Returns whether an error has been declared.
# -arguments  :
#  { errorCode        -name          {Error code}}
#
# -returns    : Whether the namespace exists.
#
proc {::Simple::Error::information exists} {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Return whether the error has been declared
   info exists ${namespace}%Errors%($errorCodeTail)
}

### ===========================================================================
### -command  : ::Simple::Error::information message
# -purpose    : Returns an error message format string.
# -overview   :
#     This procedure returns a declared error message format string.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#
# -returns    : The error message format string.
#
proc {::Simple::Error::information message} {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Undeclared error
   if {![info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode
   }

   ### Return the error message format string
   lindex [set ${namespace}%Errors%($errorCodeTail)] 0
}

### ===========================================================================
### -command  : ::Simple::Error::information explanation
# -purpose    : Returns an error explanation text.
# -overview   :
#     This procedure returns a declared error explanation text.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#
# -returns    : The error explanation text.
#
proc {::Simple::Error::information explanation} {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Undeclared error
   if {![info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode
   }

   ### Return the error explanation text
   lindex [set ${namespace}%Errors%($errorCodeTail)] 1
}

### ===========================================================================
### -command  : ::Simple::Error::information corrective
# -purpose    : Returns an error corrective action text.
# -overview   :
#     This procedure returns a declared error corrective action text.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#
# -returns    : The error corrective action text.
#
proc {::Simple::Error::information corrective} {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Undeclared error
   if {![info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode
   }

   ### Return the error corrective action text
   lindex [set ${namespace}%Errors%($errorCodeTail)] 2
}

### ===========================================================================
### -command  : ::Simple::Error::information explained
# -purpose    : Returns an error explained error message.
# -overview   :
#     This procedure returns a declared error explained error message.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#
# -returns    : The error explained error message.
#
proc {::Simple::Error::information explained} {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Undeclared error
   if {![info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode
   }

   ### Return the error explained error message
   lindex [set ${namespace}%Errors%($errorCodeTail)] 3
}

### ===========================================================================
### -command  : ::Simple::Error::Priv::transform-format-string
# -purpose    : Transforms a format string specifiers to the string specifier.
# -overview   :
#     This procedure transforms all format specifiers within a format string
#  to be string specifiers, that is, "%s".
#
# -arguments  :
#  { formatString     -string        {Format string}}
#
# -returns    : The transformed format string
#
proc ::Simple::Error::Priv::transform-format-string {
   formatString
} {
   ### Protect the "%%" format specifiers
   regsub -all -- %% $formatString @%%@ formatString

   ### Transform all specifiers to the string specifier
   regsub -all -- {%([0-9]*\$)?[-+ 0#]*[0-9]*(\.[0-9]+)?[hl]?[duioxXcsfeEgG]}\
      $formatString %s formatString

   ### Revert all "%%" format specifiers to its original form
   regsub -all -- @%%@ $formatString %% answer

   ### Return the transformed format string
   set answer
}

### ===========================================================================
### -command  : ::Simple::Error::explain
# -purpose    : Explains an error.
# -overview   :
#     This procedure explains an error previously declared via
#  K<::Simple::Error::declare>.  
#
#     If the I<-storedetails> package option was not set or no error
#  explanation was provided when the error was declared, the empty string is
#  returned; otherwise the full error explanation is obtained, made of four
#  items:  the error code, the explained error message, the error explanation
#  and the corrective action.
#
#     If the explained error message was provided when the error was declared,
#  it is used.  Otherwise it is composed from the unexplained error message
#  with the format specifiers substituted by the name of the error run-time
#  arguments they represent.  The list of the run-time arguments are extracted
#  from the error explanation, in which they appear as strings between angle
#  brackets.  For example, if the error message is
#
#     "no file named %s in directory %s"
#
#  an appropriate error explanation would be 
#
#     "the file named <file> in directory <directory> could not be found",
#
#  so that the error message included in this procedure return value would be
#
#     "no file named <file> in directory <directory>". 
#
#     The translation string argument of K<::Simple::Error::declare> is used
#  here when the order in which the run-time arguments appear in the error
#  explanation is different from their order in the error message.  In this
#  case, the translation string is a sequence of items between angle
#  brackets which correspond to the names of the error message run-time
#  arguments.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#
# -returns    : The error explanation.
# -examples   :
#
#  # Set the package option to store the error details
#  ::Simple::Error::configure -storedetails true
#
#  # Declare an error
#  ::Simple::Error::declare FILE-NOT-FOUND-1 {
#     no file named %s in directory %s
#  } {
#     There is no file named <file> in directory <dir>
#  } {
#     Make sure both the file and directory are correct
#  }
#
#  # Explain the error
#  # This displays the following:
#  #    Error code   : FILE-NOT-FOUND-1
#  #    Error message: no file named <file> in directory <dir>
#  #    Explanation  : There is no file named <file> in directory <dir>.
#  #    Correction   : Make sure both the file and directory are correct.
#  puts [::Simple::Error::explain FILE-NOT-FOUND-1]
#
#  # Declare another error
#  ::Simple::Error::declare FILE-NOT-FOUND-2 {
#     directory %s has no file named %s
#  } {
#     There is no file named <file> in directory <dir>
#  } {
#     Make sure both the file and directory are correct
#  } {<dir> <file>}
#
#  # Explain the error
#  # This displays the following:
#  #    Error code   : FILE-NOT-FOUND-2
#  #    Error message: directory <dir> has no file named <file>
#  #    Explanation  : There is no file named <file> in directory <dir>.
#  #    Correction   : Make sure both the file and directory are correct.
#  puts [::Simple::Error::explain FILE-NOT-FOUND-2]
#
#  # Declare another error
#  ::Simple::Error::declare FILE-NOT-FOUND-3 {
#     directory %s has no file named %s
#  } {
#     There is no file named <file> nor <file>.BACKUP in directory <dir>
#  } {
#     Make sure both the file and directory are correct
#  } {
#  } {
#     directory <dir> has no file named <file> or <file>.BACKUP
#  }
#  #
#  # Explain the error
#  # This displays the following:
#  #    Error code   : FILE-NOT-FOUND-3
#  #    Error message: directory <dir> has no file named <file>
#  #    Explanation  : There is no file named <file> in directory <dir>.
#  #    Correction   : Make sure both the file and directory are correct.
#  puts [::Simple::Error::explain FILE-NOT-FOUND-3]
#
proc ::Simple::Error::explain {
   errorCode
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Undeclared error
   if {![info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode

   ### Declared error
   } else {

      ### Get the error details
      foreach {errorMessage errorExplanation correctiveAction translation\
         explainedErrorMessage} [set ${namespace}%Errors%($errorCodeTail)] break

      ### Error explanation available
      if {[string compare $errorExplanation {}]} {

         ### There is no explained error message
         if {![string compare $explainedErrorMessage {}]} {

            ### There is translation string
            if {[string compare $translation {}]} {

               ### Use it
               set tmp1 $translation

            ### There is no translation string
            } else {

               ### Use the error explanation
               set tmp1 $errorExplanation
            }

            ### Compose the translation list from the
            ###    angle-bracketed items in the translation string
            set translationList [list]
            while {[regexp -- {^[^<]*(<[^>]*>)(.*)$} $tmp1 tmp2 item tmp1]} {
               lappend translationList $item
            }

            ### Format the error message
            set errorMessage [::Simple::Error::Priv::transform-format-string\
               [lindex [set ${namespace}%Errors%($errorCodeTail)] 0]]
            set explainedErrorMessage\
               [::Simple::Error::Priv::format-error-message\
               $errorCode $errorMessage $translationList]
         }

         ### Format the error explanation
         # fixme: this should be formatted by the user, for example
         #        allowing him/her to provide a formatting procedure
         set answer "Error code   : $errorCode\nError message:\
            $explainedErrorMessage"
         if {[string compare $errorExplanation {}]} {
            append answer "\nExplanation  : $errorExplanation."
         }
         if {[string compare $correctiveAction {}]} {
            append answer "\nCorrection   : $correctiveAction."
         }

      ### Error explanation not available
      } else {

         ### Empty error explanation
         set answer {}
      }
   }

   ### Return the error explanation
   set answer
}

### ===========================================================================
### -command  : ::Simple::Error::throw
# -purpose    : Throws an error.
# -overview   :
#     This procedure throws a declared error.
#
# -arguments  :
#  { errorCode        -name          {Error code}}
#  { args             -list          {Error run-time arguments}}
#
# -returns    : The error is thrown.
#
proc ::Simple::Error::throw {
   errorCode
   args
} {
   ### Split the error code
   foreach {namespace errorCodeTail}\
      [::Simple::Error::Priv::split-name $errorCode] break

   ### Declared error
   if {[info exists ${namespace}%Errors%($errorCodeTail)]} {

      ### Format the error message
      set errorMessage [lindex [set ${namespace}%Errors%($errorCodeTail)] 0]
      set formattedErrorMessage [::Simple::Error::Priv::format-error-message\
         $errorCode $errorMessage $args]

      ### Throw the error
      return -code error -errorcode $errorCode $formattedErrorMessage

   ### Undeclared error
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::Error::UNDECLARED-ERROR $errorCode
   }
}

### ===========================================================================
### -command  : ::Simple::Error::catch-explain
# -purpose    : Catches an error and explains it.
# -overview   :
#     This procedure is similar to the K<catch> command.  It evaluates the
#  given script catching any error it may throw.  If the script throws a
#  declared error, the V<::errorInfo> variable is modified to include an
#  explanation of the error.  Otherwise, the procedure acts exactly as
#  K<catch> except that the call to this procedure is deleted from
#  V<::errorInfo>.
#
# -arguments  :
#  { script           -script        {Script to evaluate}}
#  {?variableName?    -name {}       {Name of the variable in the calling scope
#                                     in which the script evaluation result is
#                                     to be placed}}
#
# -returns    : The error is thrown.
# -effects    :
#  * Those of the evaluated script.
#
proc ::Simple::Error::catch-explain {
   script
   {variableName {}}
} {
   ### Evaluate the script
   set return [catch {uplevel $script} result]

   ### Error evaluating the script
   if {$return == 1} {

      ### Declared error
      set errorInfoLines [split $::errorInfo \n]
      if {[::Simple::Error::information exists $::errorCode]} {

         ### Initialise V<::errorInfo> with the error message
         set ::errorInfo [lindex $errorInfoLines 0]\n

         ### Insert the error explanation into V<::errorInfo>
         #   The error explanation is indented with four spaces
         # fixme: this should be formatted by the user, for example
         #        allowing him/her to provide a formatting procedure
         set errorExplanation [::Simple::Error::explain $::errorCode]
         if {[string compare $errorExplanation {}]} {
            set errorExplanation "===================== Error explanation\
               ======================\n$errorExplanation\n==================\
               End of error explanation =================="
            regsub -all -- \n $errorExplanation "\n    " errorExplanation
         } else {
            set errorExplanation\
               "=============== No error explanation available ==============="
         }
         append ::errorInfo "    $errorExplanation"

         ### Skip three lines from the beginning of the original V<::errorInfo>
         #   This corresponds to the call to K<::Simple::Error::throw>
         set skipFromBeginning 3

      ### Undeclared error
      } else {

         ### Initialise V<::errorInfo>
         set ::errorInfo {}

         ### Skip no lines from the beginning of the original V<::errorInfo>
         set skipFromBeginning 0
      }

      ### Skip three lines from the end of the original V<::errorInfo>
      #   This corresponds to the K<uplevel> in the evaluation of the
      #   script above
      set skipFromEnd 4

      ### Finish the composition of V<::errorInfo>
      set errorInfoTail [join [lrange $errorInfoLines $skipFromBeginning\
         [expr {[llength $errorInfoLines] - $skipFromEnd}]] \n]
      if {[string compare $errorInfoTail {}]} {
         if {[string compare $::errorInfo {}]} {
            append ::errorInfo \n $errorInfoTail
         } else {
            set ::errorInfo $errorInfoTail
         }
      }
   }

   ### Variable name given
   if {[string compare $variableName {}]} {

      ### Set the evaluation script result in the callers scope variable
      uplevel [list set $variableName $result]
   }

   ### Return the evaluation script result
   set return
}

### ===========================================================================
### -command  : ::Simple::Error::configure
# -purpose    : Configures the package options.
#
# -arguments  :
#  {-storedetails     -extbool false {Whether the error details (error
#                                     explanation and corrective action) are
#                                     stored}}
#
proc ::Simple::Error::configure {
   {arg1 {}}
   {arg2 {}}
} {
   ### No arguments
   if {![string compare $arg1 {}]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::AT-LEAST-ONE-OPTION\
         [::Simple::Error::cget]

   } else {

      ### Switch on flag
      switch -exact -- $arg1 {

         -storedetails {
         ### Handle -storedetails

            ### Invalid argument
            if {[catch {if {$arg2} {}}]} {

               ### Throw error
               ::Simple::Error::throw ::Simple::BAD-OPTION-ARG-TYPE\
                  $arg2 -storedetails -extbool
            }

            ### Set the store details package option
            if {$arg2} {
               set ::Simple::Error::StoreDetails 1
            } else {
               set ::Simple::Error::StoreDetails 0
            }
         }

         default {
         ### default, throw error
            ::Simple::Error::throw ::Simple::BAD-OPTION\
               $arg1 [::Simple::Error::cget]
         }
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Error::cget
# -purpose    : Gets the package options.
# -arguments  :
#  {-storedetails     -boolflag      {Gets whether the error details (error
#                                     explanation and corrective action) are
#                                     stored}}
#
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Error::cget {
   {arg1 {}}
} {
   ### No arguments
   if {![string compare $arg1 {}]} {

      ### Return the list of options
      return -storedetails

   } else {

      ### Switch on flag
      switch -exact -- $arg1 {

         -storedetails {
         ### Handle -storedetails
            set ::Simple::Error::StoreDetails
         }

         default {
         ### default, throw error
            ::Simple::Error::throw ::Simple::BAD-OPTION\
               $arg1 [::Simple::Error::cget]
         }
      }
   }
}

# Install the package
::Simple::Error::Priv::install

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleError-test 0.2

### ===========================================================================
### -test     : ::Simple::Error::declare
test-case declare-1 {
   ::Simple::Error::declare
} -setup {

   # Create a namespace
   namespace eval foo {}

} -script {

   # The error does not exist
   puts -nonewline [info exists ::foo::%Errors%(ERROR)]

   # Declare an error
   ::Simple::Error::declare ::foo::ERROR {} {}

   # The error exists
   puts -nonewline [info exists ::foo::%Errors%(ERROR)]

} -cleanup {

   # Delete the namespace
   namespace delete foo

} -output 01

test-case declare-2 {
   ::Simple::Error::declare
   unqualified error code
} -script {

   # The error does not exist
   puts -nonewline [info exists ::%Errors%(ERROR)]

   # Declare an error
   ::Simple::Error::declare ERROR {} {}

   # The error exists
   puts -nonewline [info exists ::%Errors%(ERROR)]

} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output 01

test-case declare-3 {
   ::Simple::Error::declare
   error details not stored
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to false
   set ::Simple::Error::StoreDetails 0

} -script {

   # Declare an error
   ::Simple::Error::declare ERROR {
      This is the error message
   } {
      This is the error explanation
   } {
      This is the corrective action
   } {
      This is the translation string
   }

   # Get the error explanation
   ::Simple::Error::information explanation ERROR

} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -return {}

test-case declare-4 {
   ::Simple::Error::declare
   stored error details
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

} -script {

   # Declare an error
   ::Simple::Error::declare ERROR {
      This is the error message
   } {
      This is the error explanation
   } {
      This is the corrective action
   } {
      This is the translation string
   }

   # Get the error details
   ::Simple::Error::information explanation ERROR

} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -return {This is the error explanation}

test-case declare-5 {
   ::Simple::Error::declare
   deleting a namespace deletes the errors therein
} -setup {

   # Declare an error in a namespace
   namespace eval ::foo {
      ::Simple::Error::declare ::foo::ERROR {} {}
   }

} -script {

   # The errors exists
   puts -nonewline [::Simple::Error::information exists ::foo::ERROR]

   # Delete the namespace
   namespace delete ::foo

   # The errors no longer exists
   puts -nonewline [::Simple::Error::information exists ::foo::ERROR]

} -output 10

test-case declare-6 {
   ::Simple::Error::declare
   error, non-existing namespace
} -script {

   # Failed declaration
   ::Simple::Error::declare ::foo::ERROR {} {}

} -error {non-existing namespace "::foo"}

### ===========================================================================
### -test     : ::Simple::Error::delete
test-case delete-1 {
   ::Simple::Error::delete
} -setup {

   # Create a namespace
   namespace eval foo {}

   # Declare an error
   ::Simple::Error::declare ::foo::ERROR {} {}

} -script {

   # The error exists
   puts -nonewline [info exists ::foo::%Errors%(ERROR)]

   # Delete the error
   ::Simple::Error::delete ::foo::ERROR

   # The error does not exist
   puts -nonewline [info exists ::foo::%Errors%(ERROR)]

} -cleanup {

   # Delete the namespace
   namespace delete foo

} -output 10

test-case delete-2 {
   ::Simple::Error::delete
   unqualified error code
} -setup {

   # Declare an error
   ::Simple::Error::declare ::ERROR {} {}

} -script {

   # The error exists
   puts -nonewline [info exists ::%Errors%(ERROR)]

   # Delete the error
   ::Simple::Error::delete ::ERROR

   # The error does not exist
   puts -nonewline [info exists ::%Errors%(ERROR)]

} -output 10

test-case delete-3 {
   ::Simple::Error::delete
   error, undeclared error
} -script {
   ::Simple::Error::delete ::ERROR
} -error {undeclared error "::ERROR"}


### ===========================================================================
### -test     : ::Simple::Error::information
test-case info-1 {
   ::Simple::Error::information
} -script {
   ::Simple::Error::information exists ::Simple::Error::UNDECLARED-ERROR
} -return 1

test-case info-2 {
   ::Simple::Error::information
   subcommand throws an error
} -setup {

   # Create a subcommand procedure
   proc {::Simple::Error::information foo} {} {
      return -code error -errorcode errorCode errorMessage
   }

} -script {
   set return [catch {::Simple::Error::information foo} result]
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
} -cleanup {

   # Delete the subcommand procedure
   rename {::Simple::Error::information foo} {}

} -output {return: {1}
result: {errorMessage}
errorCode: {errorCode}
}

test-case info-3 {
   ::Simple::Error::information
   error, no subcommand given
} -script {
   ::Simple::Error::information
} -error {wrong # args: should be "::Simple::Error::information option ?arg\
...?"}

test-case info-4 {
   ::Simple::Error::information
   error, non-existing subcommand
} -regexp -script {
   ::Simple::Error::information foo
} -error {bad option "foo": *}

test-case info-5 {
   ::Simple::Error::information
   error, invalid subcommand arguments
} -script {
   ::Simple::Error::information exists foo bar
} -error {wrong # args: should be "::Simple::Error::information exists\
errorCode"}

### ===========================================================================
### -test     : ::Simple::Error::information message
test-case message-1 {
   ::Simple::Error::information message
} -setup {

   # Declare an error
   ::Simple::Error::declare ::ERROR message {}

} -script {
   ::Simple::Error::information message ::ERROR
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -return message

test-case message-2 {
   ::Simple::Error::information message
   error, undeclared error
} -script {
   ::Simple::Error::information message ::ERROR
} -error {undeclared error "::ERROR"}

### ===========================================================================
### -test     : ::Simple::Error::information explanation
test-case explanation-1 {
   ::Simple::Error::information explanation
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare two errors
   ::Simple::Error::declare ::ERROR1 message explanation
   ::Simple::Error::declare ::ERROR2 message {}

} -script {
   puts [::Simple::Error::information explanation ::ERROR1]
   puts [::Simple::Error::information explanation ::ERROR2]
} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

   # Delete the errors
   ::Simple::Error::delete ::ERROR1
   ::Simple::Error::delete ::ERROR2

} -output {explanation

}

test-case explanation-2 {
   ::Simple::Error::information explanation
   error, undeclared error
} -script {
   ::Simple::Error::information explanation ::ERROR
} -error {undeclared error "::ERROR"}

### ===========================================================================
### -test     : ::Simple::Error::information corrective
test-case corrective-1 {
   ::Simple::Error::information corrective
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare two errors
   ::Simple::Error::declare ::ERROR1 message explanation corrective
   ::Simple::Error::declare ::ERROR2 message {}

} -script {
   puts [::Simple::Error::information corrective ::ERROR1]
   puts [::Simple::Error::information corrective ::ERROR2]
} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

   # Delete the errors
   ::Simple::Error::delete ::ERROR1
   ::Simple::Error::delete ::ERROR2

} -output {corrective

}

test-case corrective-2 {
   ::Simple::Error::information corrective
   error, undeclared error
} -script {
   ::Simple::Error::information corrective ::ERROR
} -error {undeclared error "::ERROR"}

### ===========================================================================
### -test     : ::Simple::Error::information explained
test-case explained-1 {
   ::Simple::Error::information explained
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare two errors
   ::Simple::Error::declare ::ERROR1 message explanation corrective explained
   ::Simple::Error::declare ::ERROR2 message {}

} -script {
   puts [::Simple::Error::information explained ::ERROR1]
   puts [::Simple::Error::information explained ::ERROR2]
} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

   # Delete the errors
   ::Simple::Error::delete ::ERROR1
   ::Simple::Error::delete ::ERROR2

} -output {explained

}

test-case explained-2 {
   ::Simple::Error::information explained
   error, undeclared error
} -script {
   ::Simple::Error::information explained ::ERROR
} -error {undeclared error "::ERROR"}

### ===========================================================================
### -test     : ::Simple::Error::information exists
test-case declared-1 {
   ::Simple::Error::information declared
} -script {

   # Initial number of declared errors
   set nErrors [llength [::Simple::Error::information declared]]

   # Declare an error
   ::Simple::Error::declare ::FOO-BAR {} {}
   puts [expr {[llength [::Simple::Error::information declared]] - $nErrors}]

   # Initial number of declared errors
   set nErrors [llength [::Simple::Error::information declared]]

   # Declare another error
   ::Simple::Error::declare ::FOO-GEE {} {}

   # Declared errors matching ::FOO*
   puts [lsort [::Simple::Error::information declared ::FOO*]]

} -cleanup {

   # Delete the errors
   ::Simple::Error::delete ::FOO-BAR
   ::Simple::Error::delete ::FOO-GEE

} -output {1
::FOO-BAR ::FOO-GEE
}

test-case declared-2 {
   ::Simple::Error::information declared
   namespace deletion handling
} -setup {

   # Create a namespace
   namespace eval foo {}

} -script {

   # Declared errors matching ::foo::*
   puts [::Simple::Error::information declared ::foo::*]

   # Initial number of namespaces with declared errors
   set nNamespaces [llength [array names ::Simple::Error::Namespaces]]

   # Declare an error
   ::Simple::Error::declare ::foo::ERROR {} {}

   # Declared errors matching ::foo::*
   puts [::Simple::Error::information declared ::foo::*]
   puts [expr {[llength [array names ::Simple::Error::Namespaces]] -\
      $nNamespaces}]

   # Delete the namespace
   namespace delete foo

   # Declared errors matching ::foo::*
   puts [::Simple::Error::information declared ::foo::*]
   puts [expr {[llength [array names ::Simple::Error::Namespaces]] -\
      $nNamespaces}]

} -output {
::foo::ERROR
1

0
}

### ===========================================================================
### -test     : ::Simple::Error::information exists
test-case exists-1 {
   ::Simple::Error::information exists
} -setup {

   # Declare an error
   ::Simple::Error::declare ::ERROR {} {}

} -script {

   # Existing errors
   puts -nonewline\
      [::Simple::Error::information exists ::Simple::Error::UNDECLARED-ERROR]
   puts -nonewline [::Simple::Error::information exists ::ERROR]
   puts -nonewline [::Simple::Error::information exists ERROR]

   # Non-existing errors
   puts -nonewline\
      [::Simple::Error::information exists ::foo::Non-existing-error]
   puts -nonewline [::Simple::Error::information exists Non-existing-error]

} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output 11100

### ===========================================================================
### -test     : ::Simple::Error::explain
test-case explain-1 {
   ::Simple::Error::explain
   error details not stored
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to false
   set ::Simple::Error::StoreDetails 0

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   }

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts [::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {
}

test-case explain-2 {
   ::Simple::Error::explain
   stored error details
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   }

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts \n[::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {
Error code   : ::ERROR
Error message: first "<first>" second "<second>" last "<the last>"
Explanation  : FIRST <first> SECOND <second> LAST <the last>.
Correction   : This is the corrective action.
}

test-case explain-3 {
   ::Simple::Error::explain
   stored error details
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      string "%s", percent "%%", number "%d"
   } {
      STRING <string> NUMBER <number>
   } {
      This is the corrective action
   }

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts \n[::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {
Error code   : ::ERROR
Error message: string "<string>", percent "%", number "<number>"
Explanation  : STRING <string> NUMBER <number>.
Correction   : This is the corrective action.
}

test-case explain-4 {
   ::Simple::Error::explain
   stored error details, translation string
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   } {<one> <two> <the three>}

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts \n[::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {
Error code   : ::ERROR
Error message: first "<one>" second "<two>" last "<the three>"
Explanation  : FIRST <first> SECOND <second> LAST <the last>.
Correction   : This is the corrective action.
}

test-case explain-5 {
   ::Simple::Error::explain
   stored error details, explained error message
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   } {
   } {
      This is the explained error message
   }

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts \n[::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {
Error code   : ::ERROR
Error message: This is the explained error message
Explanation  : FIRST <first> SECOND <second> LAST <the last>.
Correction   : This is the corrective action.
}

test-case explain-6 {
   ::Simple::Error::explain
   error, undeclared error
} -script {
   ::Simple::Error::explain ::ERROR
} -error {undeclared error "::ERROR"}

test-case explain-7 {
   ::Simple::Error::explain
   error, format error, bad format
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   }

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   ::Simple::Error::explain ::ERROR
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -error {error formatting error "::ERROR" (error message is {first "%s"\
second "%" last "%s"}, arguments are {<first> <second> {<the last>}})}

test-case explain-8 {
   ::Simple::Error::explain
   error, format error, too few arguments
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST
   }

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts \n[::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -error {error formatting error "::ERROR" (error message is {first "%s"\
second "%s" last "%s"}, arguments are {<first> <second>})}

test-case explain-9 {
   ::Simple::Error::explain
   error, format error, too few arguments in translation string
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   } {<one> <two>}

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -script {
   puts \n[::Simple::Error::explain ::ERROR]
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -error {error formatting error "::ERROR" (error message is {first "%s"\
second "%s" last "%s"}, arguments are {<one> <two>})}

### ===========================================================================
### -test     : ::Simple::Error::throw
test-case throw-1 {
   ::Simple::Error::throw
} -setup {

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {}

} -script {
   catch {::Simple::Error::throw ::ERROR foo bar gee} result
   puts $::errorCode
   puts $result
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {::ERROR
first "foo" second "bar" last "gee"
}

test-case throw-2 {
   ::Simple::Error::throw
   several format specifiers
} -setup {

   # Declare an error
   ::Simple::Error::declare ERROR {
      string "%s", percent "%%", number "%d"
   } {}

} -script {
   catch {::Simple::Error::throw ::ERROR foo 77} result
   puts $::errorCode
   puts $result
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {::ERROR
string "foo", percent "%", number "77"
}

test-case throw-3 {
   ::Simple::Error::throw
   error, undeclared error
} -script {
   ::Simple::Error::throw ::ERROR foo bar gee
} -error {undeclared error "::ERROR"}

test-case throw-4 {
   ::Simple::Error::throw
   error, format error
} -setup {

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%" last "%s"
   } {}

} -script {
   catch {::Simple::Error::throw ::ERROR foo bar gee} result
   puts $::errorCode
   puts $result
} -cleanup {

   # Delete the error
   ::Simple::Error::delete ::ERROR

} -output {::Simple::Error::FORMAT-ERROR
error formatting error "::ERROR" (error message is {first "%s" second "%" last\
"%s"}, arguments are {foo bar gee})
}

### ===========================================================================
### -test     : ::Simple::Error::catch-explain
test-case catch-explain-1 {
   ::Simple::Error::catch-explain
} -setup {

   # Initialise a variable
   set tmp 0

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
   }

   # With no result variable
   set return [::Simple::Error::catch-explain $script]
   puts $tmp
   puts "return: \{$return\}"

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"

} -cleanup {

   # Delete the variable and the script
   unset tmp script

} -output {foo
5
return: {0}
foo
5
return: {0}
result: {5}
}

test-case catch-explain-2 {
   ::Simple::Error::catch-explain
   return evaluating the script
} -setup {

   # Initialise a variable
   set tmp 0

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
      return 99
   }

   # With no result variable
   set return [::Simple::Error::catch-explain $script]
   puts $tmp
   puts "return: \{$return\}"

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"

} -cleanup {

   # Delete the variable and the script
   unset tmp script

} -output {foo
5
return: {2}
foo
5
return: {2}
result: {99}
}

test-case catch-explain-3 {
   ::Simple::Error::catch-explain
   break evaluating the script
} -setup {

   # Initialise a variable
   set tmp 0

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
      break
   }

   # With no result variable
   set return [::Simple::Error::catch-explain $script]
   puts $tmp
   puts "return: \{$return\}"

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"

} -cleanup {

   # Delete the variable and the script
   unset tmp script

} -output {foo
5
return: {3}
foo
5
return: {3}
result: {}
}

test-case catch-explain-4 {
   ::Simple::Error::catch-explain
   continue evaluating the script
} -setup {

   # Initialise a variable
   set tmp 0

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
      continue
   }

   # With no result variable
   set return [::Simple::Error::catch-explain $script]
   puts $tmp
   puts "return: \{$return\}"

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"

} -cleanup {

   # Delete the variable and the script
   unset tmp script

} -output {foo
5
return: {4}
foo
5
return: {4}
result: {}
}

test-case catch-explain-5 {
   ::Simple::Error::catch-explain
   error evaluating the script, undeclared
} -setup {

   # Initialise a variable
   set tmp 0

   # Create some procedures
   proc foo {arg1 arg2} {error errorMsg errorInfo errorCode}
   proc bar {arg1} {foo gee $arg1}

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
      bar 99
   }

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
   puts "errorInfo: \{$::errorInfo\}"

} -cleanup {

   # Delete the variable and the script
   unset tmp script

   # Delete the procedures
   rename foo {}
   rename bar {}

} -output {foo
5
return: {1}
result: {errorMsg}
errorCode: {errorCode}
errorInfo: {errorInfo
    (procedure "foo" line 1)
    invoked from within
"foo gee $arg1"
    (procedure "bar" line 1)
    invoked from within
"bar 99"}
}

test-case catch-explain-6 {
   ::Simple::Error::catch-explain
   error evaluating the script, declared, explanation available
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   }

   # Initialise a variable
   set tmp 0

   # Create some procedures
   proc foo {arg1 arg2} {::Simple::Error::throw ::ERROR $arg1 $arg2 LAST}
   proc bar {arg1} {foo gee $arg1}

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
      bar 99
   }

   # With no result variable
   set return [::Simple::Error::catch-explain $script]
   puts $tmp
   puts "return: \{$return\}"
   puts "errorCode: \{$::errorCode\}"
   puts "errorInfo: \{$::errorInfo\}"

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
   puts "errorInfo: \{$::errorInfo\}"

} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

   # Delete the error
   ::Simple::Error::delete ::ERROR

   # Delete the variable and the script
   unset tmp script

   # Delete the procedures
   rename foo {}
   rename bar {}

} -output {foo
5
return: {1}
errorCode: {::ERROR}
errorInfo: {first "gee" second "99" last "LAST"
    ===================== Error explanation ======================
    Error code   : ::ERROR
    Error message: first "<first>" second "<second>" last "<the last>"
    Explanation  : FIRST <first> SECOND <second> LAST <the last>.
    Correction   : This is the corrective action.
    ================== End of error explanation ==================
    (procedure "foo" line 1)
    invoked from within
"foo gee $arg1"
    (procedure "bar" line 1)
    invoked from within
"bar 99"}
foo
5
return: {1}
result: {first "gee" second "99" last "LAST"}
errorCode: {::ERROR}
errorInfo: {first "gee" second "99" last "LAST"
    ===================== Error explanation ======================
    Error code   : ::ERROR
    Error message: first "<first>" second "<second>" last "<the last>"
    Explanation  : FIRST <first> SECOND <second> LAST <the last>.
    Correction   : This is the corrective action.
    ================== End of error explanation ==================
    (procedure "foo" line 1)
    invoked from within
"foo gee $arg1"
    (procedure "bar" line 1)
    invoked from within
"bar 99"}
}

test-case catch-explain-7 {
   ::Simple::Error::catch-explain
   error evaluating the script, declared, explanation not available
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to false
   set ::Simple::Error::StoreDetails 0

   # Declare an error
   ::Simple::Error::declare ERROR {
      first "%s" second "%s" last "%s"
   } {
      FIRST <first> SECOND <second> LAST <the last>
   } {
      This is the corrective action
   }

   # Initialise a variable
   set tmp 0

   # Create some procedures
   proc foo {arg1 arg2} {::Simple::Error::throw ::ERROR $arg1 $arg2 LAST}
   proc bar {arg1} {foo gee $arg1}

} -script {

   # Script to catch
   set script {
      puts foo
      set tmp 5
      bar 99
   }

   # With no result variable
   set return [::Simple::Error::catch-explain $script]
   puts $tmp
   puts "return: \{$return\}"
   puts "errorCode: \{$::errorCode\}"
   puts "errorInfo: \{$::errorInfo\}"

   # With result variable
   set return [::Simple::Error::catch-explain $script result]
   puts $tmp
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
   puts "errorInfo: \{$::errorInfo\}"

} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

   # Delete the error
   ::Simple::Error::delete ::ERROR

   # Delete the variable and the script
   unset tmp script

   # Delete the procedures
   rename foo {}
   rename bar {}

} -output {foo
5
return: {1}
errorCode: {::ERROR}
errorInfo: {first "gee" second "99" last "LAST"
    =============== No error explanation available ===============
    (procedure "foo" line 1)
    invoked from within
"foo gee $arg1"
    (procedure "bar" line 1)
    invoked from within
"bar 99"}
foo
5
return: {1}
result: {first "gee" second "99" last "LAST"}
errorCode: {::ERROR}
errorInfo: {first "gee" second "99" last "LAST"
    =============== No error explanation available ===============
    (procedure "foo" line 1)
    invoked from within
"foo gee $arg1"
    (procedure "bar" line 1)
    invoked from within
"bar 99"}
}

### ===========================================================================
### -test     : ::Simple::Error::cget
test-case cget-1 {
   ::Simple::Error::cget
   -storedetails
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to true
   set ::Simple::Error::StoreDetails 1

} -script {
   ::Simple::Error::cget -storedetails
} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -return 1

test-case cget-2 {
   ::Simple::Error::cget
   no arguments
} -script {
   ::Simple::Error::cget
} -return -storedetails

test-case cget-3 {
   ::Simple::Error::cget
   error, incorrect option
} -script {
   ::Simple::Error::cget -foo
} -error {bad option "-foo": must be -storedetails}

### ===========================================================================
### -test     : ::Simple::Error::configure
test-case configure-1 {
   ::Simple::Error::configure
   -storedetails
} -setup {

   # Save the store error details package option
   set ::Simple::Error::%TMP% $::Simple::Error::StoreDetails

   # Set the store error details package option to false
   set ::Simple::Error::StoreDetails 0

} -script {

   # Set the store error details package option to true
   ::Simple::Error::configure -storedetails true

   # Get the store error details package option
   ::Simple::Error::cget -storedetails

} -cleanup {

   # Restore the store error details package option
   set ::Simple::Error::StoreDetails ${::Simple::Error::%TMP%}
   unset ::Simple::Error::%TMP%

} -return 1

test-case configure-2 {
   ::Simple::Error::cget
   error, no options
} -script {
   ::Simple::Error::configure
} -error {at least one option required, options are -storedetails}

test-case configure-3 {
   ::Simple::Error::cget
   error, incorrect option
} -script {
   ::Simple::Error::configure -foo
} -error {bad option "-foo": must be -storedetails}

test-case configure-4 {
   ::Simple::Error::cget
   error, incorrect option argument
} -script {
   ::Simple::Error::configure -storedetails foo
} -error {invalid value "foo" for option "-storedetails" of type "-extbool"}

} ;# End of regression testing section
