### ===========================================================================
### -package  : SimpleTest
# -version    : 0.2
# -purpose    : Regression testing framework.
# -overview   :
#     This package provides a complete framework for automated regression
#  testing.
#
#     Procedures are provided to initialise, reset and shutdown the test
#  environment, set criteria for test cases to be tested, actually perform the
#  test cases as well as others to obtain the test results and statistics.
#
# -usage      :
#     The K<::Simple::Test::initialise> procedure establishes a test
#  environment in the form of a slave interpreter in which the K<puts> command
#  has been modified so that the test cases output are collected.  Test cases
#  can be run in the test environment via the P<test-case> procedure.  By
#  comparing the output, errors and return values of the test cases with the
#  corresponding expected values each test is flagged as passed or failed. 
#
#     By using K<::Simple::Test::set-test-criteria>, one can set a criteria
#  for test cases to be tested or fully ignored.  The detailed test results
#  and the overall statistics are availabe via the P<::Simple::Test::results>
#  and P<::Simple::Test::statistics> procedures, respectively.
#
#     Use K<::Simple::Test::reset> to reset the test environment and
#  K<::Simple::Test::shutdown> to clean up after testing.
#
# -keywords   : regression test package procedure
# -variables  :
#
#  { CaseOutput       -string        {Output of each individual test case}}
#  { Results          -list          {List of tested tests cases tested each
#                                     one containing the following elements:
#                                        0: number
#                                        1: tag
#                                        2: title
#                                        3: whether it was skipped
#                                        4: whether it passed or failed
#                                        5: in case it failed, the diagnostics}}
#  { NTests           -int           {Number of tests cases tested}}
#  { NTestsPassed     -int           {Number of tests cases passed}}
#  { NTestsFailed     -int           {Number of tests cases failed}}
#  { NTestsSkipped    -int           {Number of tests cases skipped}}
#  { NTestsIgnored    -int           {Number of tests cases ignored}}
#  { TestExpr         -script        {Test cases to be tested expression}}
#
# -commands   :
#
#  * K<::Simple::Test::initialise>
#    Initialises the test environment.
#
#  * K<test-case> tag title ?-regexp? ?-prerequisites list? ?-setup script?
#    ?-script script? ?-cleanup script? ?-output any? ?-error any? ?-return any?
#    Performs a test case.
#
#  * K<::Simple::Test::set-test-criteria> testCriteria
#    Sets the criteria for test cases to be tested.
#
#  * K<::Simple::Test::criteria-to-expr> number criteria
#    Transforms a criteria into an expression.
#
#  * K<::Simple::Test::statistics>
#    Returns test statistics.
#
#  * K<::Simple::Test::results>
#    Returns the test results.
#
#  * K<::Simple::Test::reset>
#    Resets the test environment.
#
#  * K<::Simple::Test::shutdown>
#    Shutdowns the test environment.
#
#  * K<::Simple::Test::configure>
#    Configures the package options.
#
#  * K<::Simple::Test::cget>
#    Gets the package options.
#
# -examples   :
#
#  # Install the package
#  package require SimplePackage
#  ::Simple::Package::require-and-install SimpleTest
#  
#  # Initialise the test environment
#  ::Simple::Test::initialise
#  
#  # Set the test criteria: test cases number 1, 3 and following
#  ::Simple::Test::set-test-criteria 1,3-
#  
#  # Execute several test cases
#  
#  # This one passes
#  test-case test-case-1 {
#     First test case
#  } -setup {
#  
#     # Create a variable in the test environment
#     set foo 1
#  
#  } -script {
#        puts -nonewline $foo
#  } -output 1
#  
#  # This one is ignored
#  test-case test-case-2 {
#     Second test case
#  } -script {
#  } -output {}
#  
#  # This one fails
#  test-case test-case-3 {
#     Third test case
#  } -script {
#        return $foo
#  } -return 2
#  
#  # This one is skipped
#  test-case test-case-4 {
#     Fourth test case
#  } -prerequisites {
#     {$foo == 2}
#  } -script {
#  } -output {}
#  
#  # This one passes also
#  test-case test-case-5 {
#     Fifth test case
#  } -script {
#        error {ERROR MESSAGE}
#  } -cleanup {
#  
#     # Delete the variable in the test environment
#     unset foo
#  
#  } -error {ERROR MESSAGE}
#  
#  # This displays the following:
#  #    1 test-case-1     First test case: passed
#  #    2 test-case-2    Second test case: ignored
#  #    3 test-case-3     Third test case: failed
#  #    4 test-case-4    Fourth test case: skipped
#  #    5 test-case-5     Fifth test case: passed
#  foreach test [::Simple::Test::results] {
#  
#     # Get the test case information
#     foreach {number tag title ignored skipped passed diagnostics}\
#        $test break
#     if {$ignored} {
#        set status ignored
#     } elseif {$skipped} {
#        set status skipped
#     } elseif {$passed} {
#        set status passed
#     } else {
#        set status failed
#     }
#  
#     # Display a short report for each test case
#     puts [format {%1d %10s %20s %s} $number $tag $title: $status]
#  }
#  
#  # Final tests statistics
#  foreach {nTests passed failed skipped ignored}\
#     [::Simple::Test::statistics] break
#  
#  # Display some test cases statistics
#  # This displays the following:
#  #    Total test   cases 5
#  #    Test cases  passed 2
#  #    Test cases  failed 1
#  #    Test cases skipped 1
#  #    Test cases ignored 1
#  puts [format {Total test   cases %d} $nTests]
#  puts [format {Test cases  passed %d} $passed]
#  puts [format {Test cases  failed %d} $failed]
#  puts [format {Test cases skipped %d} $skipped]
#  puts [format {Test cases ignored %d} $ignored]
#  
#  # Shutdown the test environment
#  ::Simple::Test::shutdown
#
# -todo       :
#  * Consider whether it is worth it migrating the test infrastructure to use
#    the facilities of the Tcl test harness package.
#
#  * Add support for criteria for test cases to be tested based on the
#    test case tag.
#
# -history    :
#  19-feb-1999   Unreleased first version 0.1
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleTest) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleTest 0.2 -required {
   SimpleError SimplePackage
} -namespaces {
   ::Simple::Test
   ::Simple::Test::Priv
} -export {
   ::Simple::Test::test-case
} -firsttimeinstall {

### Create the package variables
set ::Simple::Test::TestingSessionTitle {}
set ::Simple::Test::NTests 0
set ::Simple::Test::NTestsPassed 0
set ::Simple::Test::NTestsFailed 0
set ::Simple::Test::NTestsSkipped 0
set ::Simple::Test::NTestsIgnored 0
set ::Simple::Test::TestExpr {}

### Declare the package errors
::Simple::Error::declare ::Simple::Test::ENVIRONMENT-NOT-SET {
   test environment not set
} {
   The attempted operation requires an initialised test environment
} {
   Use the P<::Simple::Test::initialise> procedure to initialise the test
   environment
}

::Simple::Error::declare ::Simple::Test::NO-SCRIPT-SECTION {
   no -script section for test "%s"
} {
   Test case <test case tag> lacks the customary -script section
}

::Simple::Error::declare ::Simple::Test::RETURN-AND-ERROR-SECTIONS {
   both -return and -error given for test "%s"
} {
   Test case <test case tag> has both -return and -error sections but
   these are mutually exclusive
}

::Simple::Error::declare ::Simple::Test::NO-RESULT-SECTION {
   no -output, -return nor -error given for test "%s"
} {
   Test case <test case tag> has no -output, -return nor -error section, but
   one of these is required as the test case result
}

::Simple::Error::declare ::Simple::Test::BAD-CRITERIA {
   bad criteria "%s", atom "%s" is incorrect
} {
   The criteria <criteria> has bad format; <atom> is the incorrect atom
} {
      A criteria is one or more atoms separated by commas. An atom is an
   integer or a range. A range are two integers separated by a minus sign; the
   first or second integer may not be present (but not both) meaning that the
   range is open by the left or by the right, respectively.
}

### ===========================================================================
### -command  : ::Simple::Test::Priv::test-puts
# -purpose    : Replaces the K<puts> command in the test environment.
# -overview   :
#     Works as K<puts> except if the channel to write to is stdout, stderr or
#  not given.  In that case, the output is sent to the I<CaseOutput> namespace
#  variable, unless the argument A<ignoreOutput> is set, case in which the
#  output is ignored all together.
#
# -arguments  :
#  { ignoreOutput     -boolean 1     {Whether to ignore the output to stdout
#                                     and stderr}}
#  {-nonewline        -boolflag      {Do not append a newline}}
#  {?channel?         -channel stdout
#                                    {Channel to write to}}
#  { string           -string        {String to write}}
#
# -limitations:
#  * To identify whether some arguments are given or not their default value
#    is set to "%DEFAULT%".
#
proc ::Simple::Test::Priv::test-puts {
   arg1
   arg2
   {arg3 %DEFAULT%}
   {arg4 %DEFAULT%}
} {
   ### Parse arguments
   set ignoreOutput $arg1
   set newline [string compare $arg2 -nonewline]
   if {!$newline} {
      set channel $arg3
      set string $arg4
   } else {
      set channel $arg2
      set string $arg3
   }
   set channelGiven [string compare $string %DEFAULT%]
   if {!$channelGiven} {
      set string $channel
   }

   ### Channel given but not stdout nor stderr
   if {$channelGiven &&\
      [string compare $channel stdout] && [string compare $channel stderr]} {

      ### Transfer the channel to the top interpreter
      interp transfer test-interp $channel {}

      ### Write to the given channel
      if {$newline} {
         puts $channel $string
      } else {
         puts -nonewline $channel $string
      }

      ### Transfer the channel back to the test interpreter
      interp transfer {} $channel test-interp

   ### stdout or stderr channel or no channel given at all
   } else {

      ### Do not ignore output
      if {!$ignoreOutput} {

         ### Send output to the ::Simple::Test::CaseOutput variable
         append ::Simple::Test::CaseOutput $string
         if {$newline} {
            append ::Simple::Test::CaseOutput \n
         }
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Test::Priv::assert-test-environment
# -purpose    : Raises an error if the test environment has not been set.
#
proc ::Simple::Test::Priv::assert-test-environment {
} {
   if {![interp exist test-interp]} {
      ::Simple::Error::throw ::Simple::Test::ENVIRONMENT-NOT-SET
   }
}

### ===========================================================================
### -command  : ::Simple::Test::initialise
# -purpose    : Initialises the test environment.
# -remarks    :
#  * It is customary to have called this procedure before testing.
#
# -effects    :
#  * Creates the test interpreter.
#
proc ::Simple::Test::initialise {
} {
   ### Reset the test case number and test output
   ::Simple::Test::reset

   ### Create the test interpreter
   interp create test-interp
}

### ===========================================================================
### -command  : ::Simple::Test::Priv::test-eval
# -purpose    : Evaluates a script in the test environment.
# -overview   :
#     The script is evaluated in the test environment.  Its output to stdout
#  and stderr is collected into the I<CaseOutput> namespace variable, unless
#  the argument A<ignoreOutput> is set, case in which the output is ignored
#  all together.
#
# -arguments  :
#  { script           -script        {Script to evaluate}}
#  {?ignoreOutput?    -boolean 1     {Whether to ignore the script output to
#                                     stdout and stderr}}
#
# -remarks    :
#  * It is customary to have called P<::Simple::Test::initialise> before
#  calling this procedure.
#
# -effects    :
#  * Creates an alias for K<puts> in the test interpreter.
#
#  * Those of the evaluated script in the test environment.
#
proc ::Simple::Test::Priv::test-eval {
   script
   {ignoreOutput 1}
} {
   ### Create an alias for K<puts> in the test interpreter
   interp alias test-interp puts {}\
      ::Simple::Test::Priv::test-puts $ignoreOutput

   ### Evaluate the script in the test environment
   interp eval test-interp $script
}

### ===========================================================================
### -command  : ::Simple::Test::test-case
# -purpose    : Performs a test case.
# -access     : Export
# -keywords   : test
# -overview   :
#     This is the actual test case handler.  Works by executing the test case
#  in the test environment where the output of K<puts> is available at the
#  end.  If that output matches the expected output and any error message and
#  return values match the expected ones, the test case is considered passed,
#  otherwise the test fails.  All the output of this procedure is stored and
#  can be obtained via the P<::Simple::Test::results> procedure.
#
#     If the expected output, return value or error are empty the test passes
#  only if it does not produce any output, return value or error,
#  respectively.  If the expected return value or error is not given then its
#  actual value is not checked.
#
#     By specifying the "-regexp" flag, the expected output and return value
#  or error are implemented through regular expressions so that it is possible
#  to validate non-constant values.
#
#     Before executing the test case, each prerequisite in the list of
#  prerequisites is executed.  If any of them fails, the test is skipped. 
#  Otherwise, the setup script is executed afterwards.  If there is an error,
#  the test is failed.  Otherwise, the actual test is executed followed by the
#  cleanup script.  An error in the cleanup script also results in the test to
#  be regarded as failed.
#
# -arguments  :
#  { tag              -string        {Test case tag}}
#  { title            -string        {Test case title}}
#  {-regexp           -boolflag      {Whether the expected output, return
#                                     and error shall be handled as regular
#                                     expressions}}
#  {-prerequisites    -list     {}   {List of prerequisites scripts}}
#  {-setup            -script   {}   {Setup script}}
#  {-script           -script   {}   {Test case script}}
#  {-cleanup          -script   {}   {Cleanup script}}
#  {-output           -any      {}   {Test case expected output}}
#  {-error            -any      {}   {Test case expected error}}
#  {-return           -any      {}   {Test case expected return}}
#
# -returns    : Whether the test passed or not.
# -examples   :
#
#     # A test which checks some output
#     test-case T1 {Test 1} -script {puts -nonewline "foo"} -output foo
#
#     # A test which checks an error message
#     test-case T2 {Test 2} -script {return -code error bar} -error bar
#
#     # A test which checks output using a regular expression
#     test-case T3 {Test 3} -regexp -script {puts -nonewline foo} -output {ee$}
#
# -remarks    :
#  * The A<-script> argument is customary.
#
#  * At least one of A<-output>, A<-error> or A<-return> arguments must be
#    provided.
#  * The A<-error> and A<-return> arguments are mutually exclusive.
#
#  * It is customary to have called P<::Simple::Test::initialise> before
#    calling this procedure.
#
# -effects    :
#  * Those of the evaluated scripts in the test environment.
#
proc ::Simple::Test::test-case {
   tag
   title
   args
} {
   ### Parse arguments
   set title [string trim $title]
   set regexp 0
   set expectedType {  }
   set prerequisites {}
   set setup {}
   set cleanup {}
   set expectedOutput {}
   set outputGiven 0
   set expectedError %NONE%
   set errorGiven 0
   set expectedReturn %NONE%
   set returnGiven 0
   set nArgument 0
   set nArguments [llength $args]
   while {$nArgument < $nArguments} {
      set argument [lindex $args $nArgument]
      switch -glob -- $argument {
         -regexp {
            set regexp 1
            set expectedType RE
         }
         -prerequisites {
            set prerequisites [lindex $args [incr nArgument]]
         }
         -setup {
            set setup [lindex $args [incr nArgument]]
         }
         -script {
            set script [lindex $args [incr nArgument]]
         }
         -cleanup {
            set cleanup [lindex $args [incr nArgument]]
         }
         -output {
            set outputGiven 1
            set expectedOutput [lindex $args [incr nArgument]]
         }
         -return {
            set returnGiven 1
            set expectedReturn [lindex $args [incr nArgument]]
         }
         -error {
            set errorGiven 1
            set expectedError [lindex $args [incr nArgument]]
         }
         -* {
            ::Simple::Error::throw ::Simple::BAD-OPTION $argument\
               {-cleanup, -error, -output, -prerequisites, -regexp, -return,\
               -script or -setup}
         }
         default {
            ::Simple::Error::throw ::Simple::TOO-MANY-ARGS\
               [lindex [info level 0] 0]
         }
      }
      incr nArgument
   }
   if {![info exists script]} {
      ::Simple::Error::throw ::Simple::Test::NO-SCRIPT-SECTION $tag
   }
   if {$returnGiven && $errorGiven} {
      ::Simple::Error::throw ::Simple::Test::RETURN-AND-ERROR-SECTIONS $tag
   }
   if {!$outputGiven && !$returnGiven && !$errorGiven} {
      ::Simple::Error::throw ::Simple::Test::NO-RESULT-SECTION $tag
   }

   ### Assert the test environment has been set
   ::Simple::Test::Priv::assert-test-environment

   ### Initialisation
   set passed 1
   set skipped 0
   set ignored 0
   set diagnostics {}
   set cleanupFailed 0
   set diagnosticsPrefix {>>>     }
   incr ::Simple::Test::NTests

   ### This test must be ignored
   if {![::Simple::Test::Priv::must-be-tested $::Simple::Test::NTests]} {

      ### Test ignored
      set ignored 1

      ### Test failed
      set passed 0
   }

   ### Test not failed yet
   if {$passed} {

      ### Prerequisites present
      if {[llength $prerequisites] != 0} {

         ### Loop over prerequisites
         foreach prerequisite $prerequisites {

            ### Evaluate the prerequisite in the test environment
            if {[catch {set prerequisiteResult [::Simple::Test::Priv::test-eval\
               [format {expr {%s}} $prerequisite]]} prerequisiteMessage]} {

               ### Test skipped
               set skipped 1

               # Fill in diagnostics with failure reason
               # fixme: the returned diagnostics should be a list of items
               #        which the user could format as required instead of
               #        being formatted here
               append diagnostics "${diagnosticsPrefix}Prerequisite failed:\
                  $prerequisiteMessage\n"

               ### Test failed
               set passed 0
               break

            } elseif {!$prerequisiteResult} {

               ### Test skipped
               set skipped 1

               # Fill in diagnostics with unfulfilled prerequisite
               append diagnostics "${diagnosticsPrefix}Unfulfilled prerequisite\
                  = {$prerequisite}\n"

               ### Test failed
               set passed 0
               break
            }
         }
      }
   }

   ### Test not failed yet
   if {$passed} {

      ### Setup present
      if {[llength $setup] != 0} {

         ### Evaluate the setup in the test environment
         if {[catch {::Simple::Test::Priv::test-eval $setup} setupMessage]} {

            # Fill in diagnostics with failure reason
            append diagnostics\
               "${diagnosticsPrefix}Setup failed: $setupMessage\n"

            ### Test failed
            set passed 0
         }
      }
   }

   ### Test not failed yet
   if {$passed} {

      ### Evaluate the test case in the test environment
      #   collecting the output into the CaseOutput variable
      set ::Simple::Test::CaseOutput {}
      set testcaseResult [catch {set testcaseReturn\
         [::Simple::Test::Priv::test-eval $script 0]} testcaseMessage]

      ### Cleanup present
      if {[llength $cleanup] != 0} {

         ### Evaluate the cleanup in the test environment
         if {[catch {::Simple::Test::Priv::test-eval $cleanup}\
            cleanupMessage]} {

            ### Cleanup failed
            set cleanupFailed 1

            ### Test failed
            set passed 0
         }
      }

      ### Output does not match expectations
      #   If the expected output is empty ensure the test case output
      #   is also empty, otherwise try to match the expected output either
      #   exactly or as a regular expression as required
      if {(![string compare $expectedOutput {}] && [string compare\
         $::Simple::Test::CaseOutput {}]) || (!$regexp && [string compare\
         $expectedOutput $::Simple::Test::CaseOutput]) || ($regexp &&\
         ![regexp -- $expectedOutput $::Simple::Test::CaseOutput])} {

         # Fill in diagnostics with expected and actual output
         append diagnostics "${diagnosticsPrefix}Expected output\
            $expectedType  = {$expectedOutput}\n"
         append diagnostics "${diagnosticsPrefix}Actual   output     =\
            {$::Simple::Test::CaseOutput}\n"

         ### Test failed
         set passed 0
      }

      ### Test case expected an error
      if {[string compare $expectedError %NONE%]} {

         ### Error during the test case
         if {$testcaseResult} {

            ### Error message does not match expectations
            #   If the expected error is empty ensure the test case
            #   error is also empty, otherwise try to match the expected
            #   error either exactly or as a regular expression as required
            #   but only if the expected error was given
            if {(![string compare $expectedError {}] &&\
               [string compare $testcaseMessage {}]) ||\
               (([string compare $expectedError %NONE%] && (!$regexp &&\
               [string compare $expectedError $testcaseMessage]) ||\
               ($regexp && ![regexp -- $expectedError $testcaseMessage])))} {

               # Fill in diagnostics with expected and actual error message
               append diagnostics "${diagnosticsPrefix}Expected error\
                  $expectedType   = {$expectedError}\n"
               append diagnostics "${diagnosticsPrefix}Actual   error      =\
                  {$testcaseMessage}\n"

               ### Test failed
               set passed 0
            }

         ### No error during the test case
         } else {

            # Fill in diagnostics with expected and actual error message
            append diagnostics "${diagnosticsPrefix}Expected error\
               $expectedType   = {$expectedError}\n"
            append diagnostics "${diagnosticsPrefix}Actual   error      =\
               NONE\n"

            ### Test failed
            set passed 0
         }

      ### Test case did not expect an error
      } else {

         ### Error during the test case
         if {$testcaseResult} {

            # Fill in diagnostics with expected and actual error message
            append diagnostics "${diagnosticsPrefix}Expected error\
               $expectedType   = NONE\n"
            append diagnostics "${diagnosticsPrefix}Actual   error      =\
               {$testcaseMessage}\n"

            ### Test failed
            set passed 0

         ### No error during the test case
         } else {

            ### Return value does not match expectations
            #   If the expected return value is empty ensure the test case
            #   return value is also empty, otherwise try to match the
            #   expected return value either exactly or as a regular
            #   expression as required but only if the expected return value
            #   was given
            if {(![string compare $expectedReturn {}] &&\
               [string compare $testcaseReturn {}]) ||\
               ([string compare $expectedReturn %NONE%] && ((!$regexp &&\
               [string compare $expectedReturn $testcaseReturn]) ||\
               ($regexp && ![regexp -- $expectedReturn $testcaseReturn])))} {

               # Fill in diagnostics with expected and actual return value
               append diagnostics "${diagnosticsPrefix}Expected return\
                  $expectedType  = {$expectedReturn}\n"
               append diagnostics "${diagnosticsPrefix}Actual   return     =\
                  {$testcaseReturn}\n"

               ### Test failed
               set passed 0
            }
         }
      }
   }

   ### Update the tests statistics
   if {$ignored} {
      incr ::Simple::Test::NTestsIgnored
   } elseif {$skipped} {
      incr ::Simple::Test::NTestsSkipped
   } else {
      if {$passed} {
         incr ::Simple::Test::NTestsPassed
      } else {
         incr ::Simple::Test::NTestsFailed
      }
   }

   # Update diagnostics if cleanup failed
   if {$cleanupFailed} {
      append diagnostics\
         "${diagnosticsPrefix}Cleanup failed: $cleanupMessage\n"
   }

   ### Set the test information
   lappend ::Simple::Test::Results [list $::Simple::Test::NTests\
      $tag $title $ignored $skipped $passed $diagnostics]

   ### Return whether the test passed or not
   set passed
}

### ===========================================================================
### -command  : ::Simple::Test::set-test-criteria
# -purpose    : Sets the criteria for test cases to be tested.
# -overview   :
#     This procedure sets the expression for test cases to be tested from the
#  given criteria.  Test cases matching the criteria are tested while all
#  others are fully ignored.
#
# -arguments  :
#  { testCriteria     -string        {Criteria for test cases to be tested}}
#
proc ::Simple::Test::set-test-criteria {
   testCriteria
} {
   if {[llength $testCriteria] != 0} {
      set ::Simple::Test::TestExpr\
         [::Simple::Test::criteria-to-expr ::Simple::Test::NTests $testCriteria]
   } else {
      set ::Simple::Test::TestExpr {}
   }
}

### ===========================================================================
### -command  : ::Simple::Test::criteria-to-expr
# -purpose    : Transforms a criteria into an expression.
# -overview   :
#     This procedure takes a criteria string of the form "1,2,5-10,20-"
#  and returns the equivalent expression using A<number> as the variable
#  holding the integer.
#     A criteria is one or more atoms separated by commas. An atom is an
#  integer or a range. A range are two integers separated by a minus sign; the
#  first or second integer may not be present (but not both) meaning that the
#  range is open by the left or by the right, respectively.
#     An error is thrown if the criteria has bad format.
#
# -arguments  :
#  { number           -name          {Variable name}}
#  { criteria         -string        {Criteria}}
#
# -examples   :
#
#     Using A<number> = "n".
#
#     criteria                expression
#     ----------------------- ------------------------------------------------
#     {}                      {}
#     3                       $n == 3
#     3,7                     $n == 3 || $n == 7
#     3-                      $n >= 3
#     -7                      $n <= 7
#     3-7                     $n >= 3 && $n <= 7
#     -7,10-15,20-            $n <= 7 || ($n >= 10 && $n <= 15) || $n >= 20
#
# -returns    : The expression equivalent to the criteria.
#
proc ::Simple::Test::criteria-to-expr {
   number
   criteria
} {
   ### Loop over criteria atoms
   set atoms [split $criteria ,]
   set nAtoms [llength $atoms]
   set answer {}
   foreach atom $atoms {

      ### Atom is a integer
      if {[string first - $atom] == -1} {

         ### Assert the integer has proper format
         if {![regexp {^(0|[1-9][0-9]*)$} $atom]} {
            ::Simple::Error::throw ::Simple::Test::BAD-CRITERIA\
               $criteria $atom
         }

         ### Set the subexpression
         set subexpression [format {$%s == %s} $number $atom]

      ### Atom is a range
      } else {
         foreach {limitLeft limitRight tmp} [split $atom -] break

         ### Assert the range has proper format
         if {[string compare $tmp {}] || ([string compare $limitLeft {}] &&\
            ![regexp {^(0|[1-9][0-9]*)$} $limitLeft]) || ([string compare\
            $limitRight {}] && ![regexp {^(0|[1-9][0-9]*)$} $limitRight]) ||\
            (![string compare $limitLeft {}] && ![string compare $limitRight\
            {}])} {

            ::Simple::Error::throw ::Simple::Test::BAD-CRITERIA\
               $criteria $atom
         }

         ### Range closed by the left
         if {[string compare $limitLeft {}]} {

            ### Set the subexpression
            set subexpression [format {$%s >= %s} $number $limitLeft]

            ### Range closed by the right also
            if {[string compare $limitRight {}]} {

               ### Append to the subexpression
               append subexpression [format { && $%s <= %s} $number $limitRight]

               ### There are multiple atoms
               if {$nAtoms > 1} {

                  ### Enclose the compound subexpression in parenthesis
                  set subexpression ($subexpression)
               }
            }

         ### Range open by the left
         } else {

            ### Range closed by the right only
            if {[string compare $limitRight {}]} {

               ### Set the subexpression
               set subexpression [format {$%s <= %s} $number $limitRight]
            }
         }
      }

      ### Add the subexpression
      if {[string compare $answer {}]} {
         append answer [format { || %s} $subexpression]
      } else {
         set answer $subexpression
      }
   }

   ### Return the expression equivalent to the criteria
   set answer
}

### ===========================================================================
### -command  : ::Simple::Test::Priv::must-be-tested
# -purpose    : Returns whether a test must be tested.
# -overview   :
#     This procedure returns whether the given test must be tested or not
#  by evaluating the test cases to be tested expression.
#
# -arguments  :
#  { testNumber       -int           {Test number}}
#
# -returns    : Whether the test must be tested or not.
#
proc ::Simple::Test::Priv::must-be-tested {
   testNumber
} {
   ### Test cases to be tested expression available
   if {[llength $::Simple::Test::TestExpr] != 0} {

      ### Evaluate the test expression
      set answer [expr $::Simple::Test::TestExpr]

   ### Test cases to be tested expression not available
   } else {

      ### Test all tests by default
      set answer 1
   }

   ### Return whether the test must be tested or not
   set answer
}

### ===========================================================================
### -command  : ::Simple::Test::statistics
# -purpose    : Returns test statistics.
# -returns    : A list with the following elements:
#     0: number of tests executed
#     1: number of tests passed
#     2: number of tests failed
#     3: number of tests skipped
#     4: number of tests ignored
#
proc ::Simple::Test::statistics {
} {
   list $::Simple::Test::NTests $::Simple::Test::NTestsPassed\
      $::Simple::Test::NTestsFailed $::Simple::Test::NTestsSkipped\
      $::Simple::Test::NTestsIgnored
}

### ===========================================================================
### -command  : ::Simple::Test::results
# -purpose    : Returns the test results.
# -returns    :  A list of tests cases tested containing for each case the
#  following elements:
#     0: number
#     1: title
#     2: whether it was skipped
#     2: whether it was passed or failed
#     4: in case it failed, the diagnostics
#
proc ::Simple::Test::results {
} {
   set ::Simple::Test::Results
}

### ===========================================================================
### -command  : ::Simple::Test::reset
# -purpose    : Resets the test environment.
#
proc ::Simple::Test::reset {
} {
   ### Reset the test case statistics and test results
   set ::Simple::Test::Results {}
   set ::Simple::Test::NTests 0
   set ::Simple::Test::NTestsPassed 0
   set ::Simple::Test::NTestsFailed 0
   set ::Simple::Test::NTestsSkipped 0
   set ::Simple::Test::NTestsIgnored 0
   set ::Simple::Test::TestExpr {}
}

### ===========================================================================
### -command  : ::Simple::Test::shutdown
# -purpose    : Shutdowns the test environment.
# -effects    :
#  * Deletes the test interpreter.
#
proc ::Simple::Test::shutdown {
} {
   ### Delete the test interpreter
   catch {interp delete test-interp}
}

### ===========================================================================
### -command  : ::Simple::Test::configure
# -purpose    : Configures the package options.
#
proc ::Simple::Test::configure {
} {
   ::Simple::Error::throw ::Simple::NO-OPTIONS
}

### ===========================================================================
### -command  : ::Simple::Test::cget
# -purpose    : Gets the package options.
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Test::cget {
} {
   list
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleTest-test 0.2

### Set some constants in the current and test environments
set constantsScript {
   set output         {OUTPUT VALUE}
   set outputRE       OU.*UE
   set returnValue    {RETURN VALUE}
   set returnValueRE  RE.*UE
   set errorMessage   {ERROR MESSAGE}
   set errorMessageRE ER.*GE
}
eval $constantsScript
::Simple::Test::Priv::test-eval [list set constantsScript $constantsScript]
::Simple::Test::Priv::test-eval $constantsScript

### ===========================================================================
### -test     : ::Simple::Test::Priv::test-puts
test-case test-puts-1 {
   ::Simple::Test::Priv::test-puts
   to stdout or stderr
} -script {
   puts $output
   puts stdout $output
   puts stderr $output
   puts -nonewline $output
   puts -nonewline stdout $output
   puts -nonewline stderr $output
} -output $output\n$output\n$output\n$output$output$output

test-case test-puts-2 {
   ::Simple::Test::Priv::test-puts
   to another channel
} -regexp -setup {

   # Open a pipe with no associated command for reading only
   set channel [open | r]

} -script {

   # Try to write to the pipe
   puts $channel $output

} -cleanup {

   # Close the pipe
   catch {close $channel}

} -error {channel "file.*" wasn't opened for writing}

### ===========================================================================
### -test     : ::Simple::Test::initialise
test-case initialise-1 {
   ::Simple::Test::initialise
} -script {

   # Initialise the test environment
   ::Simple::Test::initialise

   # Assert the test environment has been set
   ::Simple::Test::Priv::assert-test-environment

} -output {}

### ===========================================================================
### -test     : ::Simple::Test::Priv::test-eval
test-case test-eval-1 {
   ::Simple::Test::Priv::test-eval
} -setup {

   # Set the constants in the nested test environment
   ::Simple::Test::Priv::test-eval $constantsScript

} -script {

   # Assert that the output variable has been set
   ::Simple::Test::Priv::test-eval {set output}

} -output {}

### ===========================================================================
### -test     : ::Simple::Test::test-case
test-case test-case-1 {
   ::Simple::Test::test-case
   error, no -output, -return nor -error
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -script {
   test-case test-case-tag {Test case title} -script {}
} -error {no -output, -return nor -error given for test "test-case-tag"}

### Test: test case passed
test-case test-case-2 {
   ::Simple::Test::test-case
   no regexp, output, no return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      puts -nonewline $output
      set returnValue
   } -output $output]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-3 {
   ::Simple::Test::test-case
   regexp, output, no return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      puts -nonewline $output
      set returnValue
   } -output $outputRE]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-4 {
   ::Simple::Test::test-case
   no regexp, output, return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      puts -nonewline $output
      set returnValue
   } -output $output\
     -return $returnValue]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-5 {
   ::Simple::Test::test-case
   regexp, output, return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      puts -nonewline $output
      set returnValue
   } -output $outputRE\
     -return $returnValueRE]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-6 {
   ::Simple::Test::test-case
   no regexp, output, no return, error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      puts -nonewline $output
      error $errorMessage
   } -output $output\
     -error $errorMessage]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-7 {
   ::Simple::Test::test-case
   regexp, output, no return, error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      puts -nonewline $output
      error $errorMessage
   } -output $outputRE\
     -error $errorMessageRE]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-8 {
   ::Simple::Test::test-case
   no regexp, empty output, no return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      set returnValue
   } -output {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-9 {
   ::Simple::Test::test-case
   regexp, empty output, no return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      set returnValue
   } -output {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-10 {
   ::Simple::Test::test-case
   no regexp, no output, return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      set returnValue
   } -return $returnValue]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-11 {
   ::Simple::Test::test-case
   regexp, no output, return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      set returnValue
   } -return $returnValueRE]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-12 {
   ::Simple::Test::test-case
   no regexp, no output, no return, error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      error $errorMessage
   } -error $errorMessage]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-13 {
   ::Simple::Test::test-case
   regexp, no output, no return, error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      error $errorMessage
   } -error $errorMessageRE]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-14 {
   ::Simple::Test::test-case
   no regexp, no output, empty return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      return {}
   } -return {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-15 {
   ::Simple::Test::test-case
   regexp, no output, empty return, no error, passed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      return {}
   } -return {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 1 {}}
} -return 1

test-case test-case-16 {
   ::Simple::Test::test-case
   no regexp, unexpected output, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      puts $output
   } -output {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected output     = {}
>>>     Actual   output     = {OUTPUT VALUE
}
}}
} -return 0

test-case test-case-17 {
   ::Simple::Test::test-case
   regexp, unexpected output, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      puts $output
   } -output {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected output RE  = {}
>>>     Actual   output     = {OUTPUT VALUE
}
}}
} -return 0

test-case test-case-18 {
   ::Simple::Test::test-case
   no regexp, unexpected return, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      return $returnValue
   } -return {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected return     = {}
>>>     Actual   return     = {RETURN VALUE}
}}
} -return 0

test-case test-case-19 {
   ::Simple::Test::test-case
   regexp, unexpected return, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      return $returnValue
   } -return {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected return RE  = {}
>>>     Actual   return     = {RETURN VALUE}
}}
} -return 0

test-case test-case-20 {
   ::Simple::Test::test-case
   no regexp, unexpected  error, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
      error $errorMessage
   } -output {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected error      = NONE
>>>     Actual   error      = {ERROR MESSAGE}
}}
} -return 0

test-case test-case-21 {
   ::Simple::Test::test-case
   regexp, unexpected  error, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -regexp -script {
      error $errorMessage
   } -output {}]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected error RE   = NONE
>>>     Actual   error      = {ERROR MESSAGE}
}}
} -return 0

test-case test-case-22 {
   ::Simple::Test::test-case
   output, no regexp, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
   } -output $output]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected output     = {OUTPUT VALUE}
>>>     Actual   output     = {}
}}
} -return 0

test-case test-case-23 {
   ::Simple::Test::test-case
   return not matched, no regexp, failed
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   set testReturn [test-case test-case-tag {Test case title} -script {
   } -return $returnValue]

   # Output the test results
   puts [::Simple::Test::results]

   # Return the return value from the test procedure
   set testReturn

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected return     = {RETURN VALUE}
>>>     Actual   return     = {}
}}
} -return 0

test-case test-case-24 {
   ::Simple::Test::test-case
   the output from -prerequisites, -setup and -cleanup is ignored
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Actual test
   test-case test-case-tag {Test case title} -prerequisites {
         {![string compare [puts $output] {}]}
      } -setup {
         puts $output
      } -script {
         puts $output
      } -cleanup {
         puts $output
      } -output $output

   # Output the test results
   puts [::Simple::Test::results]

} -output {{1 test-case-tag {Test case title} 0 0 0 {>>>    \
Expected output     = {OUTPUT VALUE}
>>>     Actual   output     = {OUTPUT VALUE
}
}}
}

test-case test-case-25 {
   ::Simple::Test::test-case
   error, no script section
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -script {
   test-case test-case-tag {Test case title}
} -error {no -script section for test "test-case-tag"}

test-case test-case-26 {
   ::Simple::Test::test-case
   error, both -return and -error
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -script {
   test-case test-case-tag {Test case title} -script {} -return {} -error {}
} -error {both -return and -error given for test "test-case-tag"}

test-case test-case-19 {
   ::Simple::Test::test-case
   error, bad option
} -script {
   test-case test-case-tag {Test case title} -foo
} -error {bad option "-foo": must be -cleanup, -error, -output, -prerequisites,\
-regexp, -return, -script or -setup}

test-case test-case-20 {
   ::Simple::Test::test-case
   error, too many arguments
} -script {
   test-case test-case-tag {Test case title} foo
} -error {called "test-case" with too many arguments}

### ===========================================================================
### -test     : ::Simple::Test::set-test-criteria
test-case set-test-criteria-1 {
   ::Simple::Test::set-test-criteria
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Set the test criteria: test cases number 1, 3 and following
   ::Simple::Test::set-test-criteria 1,3-

   # Actual tests
   test-case test-case-tag-1 {Test case 1} -script {} -output {}
   test-case test-case-tag-2 {Test case 2} -script {} -output {}
   test-case test-case-tag-3 {Test case 3} -script {} -output {}
   test-case test-case-tag-4 {Test case 4} -script {} -output {}
   test-case test-case-tag-5 {Test case 5} -script {} -output {}

   # Loop over tests
   foreach test [::Simple::Test::results] {

      # Test not ignored
      if {![lindex $test 3]} {

         # Display its number
         puts -nonewline [lindex $test 0]
      }
   }

} -output 1345

### ===========================================================================
### -test     : ::Simple::Test::criteria-to-expr
test-case criteria-to-expr-1 {
   ::Simple::Test::criteria-to-expr
} -script {
   puts [::Simple::Test::criteria-to-expr n {}]
   puts [::Simple::Test::criteria-to-expr n 3]
   puts [::Simple::Test::criteria-to-expr n 3,7]
   puts [::Simple::Test::criteria-to-expr n 3-]
   puts [::Simple::Test::criteria-to-expr n -7]
   puts [::Simple::Test::criteria-to-expr n 3-7]
   puts [::Simple::Test::criteria-to-expr n -7,10-15,20-]
} -output {
$n == 3
$n == 3 || $n == 7
$n >= 3
$n <= 7
$n >= 3 && $n <= 7
$n <= 7 || ($n >= 10 && $n <= 15) || $n >= 20
}

test-case criteria-to-expr-2 {
   ::Simple::Test::criteria-to-expr
   error, incorrect creiteria
} -script {
   catch {::Simple::Test::criteria-to-expr n 1,foo,2} result
   puts $result
   catch {::Simple::Test::criteria-to-expr n 1,2-foo,2} result
   puts $result
   ::Simple::Test::criteria-to-expr n 1,-,2
} -output {bad criteria "1,foo,2", atom "foo" is incorrect
bad criteria "1,2-foo,2", atom "2-foo" is incorrect
} -error {bad criteria "1,-,2", atom "-" is incorrect}

### ===========================================================================
### -test     : ::Simple::Test::statistics
test-case statistics-1 {
   ::Simple::Test::statistics
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -setup {

   # Reset the test environment
   ::Simple::Test::reset 

} -script {

   # Set the test criteria: test cases number 1, 3 and following
   ::Simple::Test::set-test-criteria 1,3-

   # Initial tests statistics: all zeros
   puts [::Simple::Test::statistics]

   test-case test-case-tag-1 {Test case 1} -script {} -output {}
   test-case test-case-tag-2 {Test case 2} -script {} -output {}
   test-case test-case-tag-3 {Test case 3} -prerequisites {1 == 2}\
      -script {} -output {}
   test-case test-case-tag-4 {Test case 4} -script {} -output foo
   test-case test-case-tag-5 {Test case 5} -script {} -output {}

   # Final tests statistics
   puts [::Simple::Test::statistics]

} -output {0 0 0 0 0
5 2 1 1 1
}

### ===========================================================================
### -test     : ::Simple::Test::results
test-case results-1 {
   ::Simple::Test::results
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
   {[lindex [::Simple::Test::statistics] 0] == 5}
} -script {

   # Loop over tests
   foreach test [::Simple::Test::results] {
      foreach {number tag title ignored skipped passed diagnostics}\
         $test break

      puts "$number $tag $ignored $skipped $passed"
   }

} -output {1 test-case-tag-1 0 0 1
2 test-case-tag-2 1 0 0
3 test-case-tag-3 0 1 0
4 test-case-tag-4 0 0 0
5 test-case-tag-5 0 0 1
}

### ===========================================================================
### -test     : ::Simple::Test::reset
test-case reset-1 {
   ::Simple::Test::reset
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
   {[lindex [::Simple::Test::statistics] 0] == 5}
} -script {

   # Final tests statistics
   puts [::Simple::Test::statistics]

   # Reset the test environment
   ::Simple::Test::reset 

   # Tests statistics reset: all zeros
   puts [::Simple::Test::statistics]

} -output {5 2 1 1 1
0 0 0 0 0
}

### ===========================================================================
### -test     : ::Simple::Test::shutdown
test-case shutdown-1 {
   ::Simple::Test::shutdown
} -prerequisites {
   {[::Simple::Test::Priv::assert-test-environment] != 0}
} -script {

   # Shutdown the test environment
   ::Simple::Test::shutdown

   # Assert the test environment has been set, thus getting an error
   ::Simple::Test::Priv::assert-test-environment

} -error {test environment not set}

### ===========================================================================
### -test     : ::Simple::Test::configure
test-case configure-1 {
   ::Simple::Test::configure
} -script {
   ::Simple::Test::configure
} -error {this package has no options}

### ===========================================================================
### -test     : ::Simple::Test::cget
test-case cget-1 {
   ::Simple::Test::cget
} -script {
   ::Simple::Test::cget
} -return {}

} ;# End of regression testing section
