### ===========================================================================
### -package  : SimpleExtProc
# -version    : 0.2
# -purpose    : Procedures with extended argument syntax.
# -overview   :
#     This package provides a way to create procedures with extended argument
#  syntax.  This syntax includes argument types (with optional run-time type
#  checking) and argument flags with or without parameters which may be
#  interpreted or left to the user.
#
#     Procedures are provided to create, delete, rename, copy and get
#  information about procedures with extended argument syntax, as well as
#  others to handle arguments in the body of those procedures.
#
#     This package also allows to handle headers for code scripts.  Each
#  header is identified by a tag and, within a script, headers are ordered by
#  priority.  Procedures are provided to add, get and delete headers.
#
# -usage      :
#     Extended syntax procedures are created via the
#  K<::Simple::ExtProc::create> procedure.  The declaration specifies whether
#  to perform run-time checking of the arguments type as well as wether to
#  delegate the flag interpretation to the user.
#
#     Extended argument syntax procedures can be deleted via
#  K<::Simple::ExtProc::delete> or renamed via K<::Simple::ExtProc::rename>. 
#  The K<::Simple::ExtProc::information> procedure provides several
#  subcommands to query information about extended argument syntax procedures: 
#  K<is-subcommand>, K<subcommands>, K<runtimeparsing>, K<checktype>,
#  K<interpflags>, K<exists>, K<body>, K<args>, K<argsusage>, K<argslist>,
#  K<type>, K<default>, K<choices> and K<description>.  Refer to the
#  description of each procedure below for further information.
#
#     The syntax of the argument list, implemented by the P<SimpleExtVar>
#  package, is very similar to that of the P<opt> package distributed with Tcl
#  8.0, so that the upgrade should be easy; in fact this package was
#  implemented as an evolution of the P<opt> package for two reasons:  1)
#  P<opt> is B<slow> and 2) it was deprecated from Tcl 8.1.
#
#     This package encompasses the funcionality provided by the
#  P<SimpleSubcommand> package so it is recommended not to use the later if
#  using the former.  In fact, all procedures provided by this package work
#  for regular procedures also, so this is the only package needed to handle
#  regular or extended argument syntax procedures as well as regular or
#  subcommand procedures.
#
# -keywords   : procedure extended argument type flag
# -variables  :
#  { CheckType        -boolean 0     {Stores the value of the I<-checktype>
#                                     package option}}
#  { AfterFlags       -int 2         {Stores the value of the I<-afterflags>
#                                     package option}}
#
# -options    :
 
#  * I<-checktype>:  this option controls whether argument run-time type
#    checking is performed for procedures created via the
#    P<::Simple::ExtProc::create> procedure in order to check whether the
#    actual arguments match their type.  This has a performance penalty which
#    should be considered.  Notice that only those procedures created via the
#    P<::Simple::ExtProc::create> procedure B<after> modifying this option are
#    affected by the change.
#
#  * I<-afterflags>:  if the number of arguments after the last flag in
#    procedures created via the P<::Simple::ExtProc::create> procedure
#    exceedes the value of this package option, those arguments are handled
#    via an auxiliary "after flags" procedure.  Notice that only those
#    procedures created via the P<::Simple::ExtProc::create> procedure
#    B<after> modifying this option are affected by the change.
#
# -commands   :
#
#  * K<next-arg>
#    Returns the next argument in the actual argument list.
#
#  * K<previous-arg>
#    Go back one argument in the actual argument list.
#
#  * K<::Simple::ExtProc::create> ?-interpflags extbool? ?-checktype extbool?
#    procedure arguments body
#    Creates a procedure with extended argument syntax.
#
#  * K<::Simple::ExtProc::delete> procedure
#    Deletes a procedure.
#
#  * K<::Simple::ExtProc::move> sourceProcedure targetProcedure
#    Renames a procedure.
#
#  * K<::Simple::ExtProc::copy> sourceProcedure targetProcedure
#    Copies a procedure.
#
#  * K<::Simple::ExtProc::information is-subcommand> procedure
#    Returns whether a procedure is a base subcommand procedure.
#
#  * K<::Simple::ExtProc::information subcommands> command
#    Returns a base subcommand procedure subcommand list.
#
#  * K<::Simple::ExtProc::information runtimeparsing> procedure
#    Returns whether a procedure run-time parses its arguments.
#
#  * K<::Simple::ExtProc::information checktype> procedure
#    Returns whether a procedure performs argument type checking.
#
#  * K<::Simple::ExtProc::information interpflags> procedure
#    Returns whether a procedure has flag interpretation enabled.
#
#  * K<::Simple::ExtProc::information exists> procedure
#    Returns whether an extended procedure exists.
#
#  * K<::Simple::ExtProc::information body> procedure
#    Returns a procedure body.
#
#  * K<::Simple::ExtProc::information args> procedure
#    Returns an extended procedure argument names list.
#
#  * K<::Simple::ExtProc::information argsusage> procedure
#    Returns a procedure argument usage string.
#
#  * K<::Simple::ExtProc::information argslist> procedure
#    Returns a procedure argument list.
#
#  * K<::Simple::ExtProc::information type> procedure argument
#    Returns an extended procedure argument type.
#
#  * K<::Simple::ExtProc::information default> procedure argument variableName
#    Returns whether a procedure argument has a default value.
#
#  * K<::Simple::ExtProc::information choices> procedure argument
#    Returns an extended procedure argument choices list.
#
#  * K<::Simple::ExtProc::information description> procedure argument
#    Returns an extended procedure argument desription.
#
#  * K<number-flags>
#    Returns the number of flags in an extended procedure call.
#
#  * K<flag-given> flag
#    Returns whether a flag was given in an extended procedure call.
#
#  * K<arg-is-default> argument
#    Returns whether the value of an argument is the default one.
#
#  * K<::Simple::ScriptHead::add> scriptName tag priority header
#    Prepends a header to a script.
#
#  * K<::Simple::ScriptHead::get> script ?pattern?
#    Gets headers from a script.
#
#  * K<::Simple::ScriptHead::delete> scriptName ?pattern?
#    Deletes headers from a script.
#
#  * K<::Simple::ExtProc::configure> ?-checktype extbool? ?-afterflags int?
#    Configures the package options.
#
#  * K<::Simple::ExtProc::cget> ?-checktype? ?-afterflags?
#    Gets the package options.
#
# -examples   :
#
#  # Install the package
#  package require SimplePackage
#  ::Simple::Package::require-and-install SimpleExtProc
#  
#  # Enable arguments run-time type checking
#  ::Simple::ExtProc::configure -checktype true
#  
#  # Create a procedure which takes an
#  # optional flag and a required integer
#  proc-ext {
#     zero
#  } {
#     {-zero             -boolflag      "Zero flag"}
#     { integer          -int           "Integer"}
#  } {
#     if {$zero} {
#        puts 0
#     } else {
#        puts $integer
#     }
#  }
#  
#  # Get some info about the procedure
#  # This displays the following:
#  #    0
#  #     ?-zero? integer
#  #    Zero flag
#  puts [::Simple::ExtProc::information is-subcommand zero]
#  puts [::Simple::ExtProc::information argsusage zero]
#  puts [::Simple::ExtProc::information description zero -zero]
#  
#  # Valid calls
#  # This displays the following:
#  #    23
#  #    0
#  #    0
#  zero 23      
#  zero 0       
#  zero -zero 23
#  
#  # Invalid call
#  # This displays the following:
#  #    invalid value "foo" for "integer" of type "-int"
#  catch {zero foo} result
#  puts $result
#  
#  # Create a couple of subcommand procedures
#  proc-ext {
#     command foo
#  } {
#     { integer          -int           "Integer"}
#     { choice           -choice {1 2}  "Choices"}
#  } {
#     if {$integer > $choice} {
#        puts {The integer is greater than the choice}
#     } else {
#        puts {The integer is lower or equal than the choice}
#     }
#  }
#  
#  proc-ext {
#     command bar
#  } {
#     {?integer?         -int 1         "Integer"}
#  } {
#     if {[arg-is-default integer]} {
#        puts {Default value}
#     } else {
#        puts {Non-default value}
#     }
#  }
#  
#  # This displays the following:
#  #    The integer is lower or equal than the choice
#  #    The integer is greater than the choice
#  #    Default value
#  #    Non-default value
#  command foo 1 2
#  command foo 8 1
#  command bar
#  command bar 8
#  
#  # Get some info about the subcommand
#  # This displays the following:
#  #    1
#  #    foo bar
#  puts [::Simple::ExtProc::information is-subcommand command]
#  puts [::Simple::ExtProc::information subcommands command]
#  
#  # Delete one of the subcommands
#  ::Simple::ExtProc::delete {command bar}
#  
#  # Get some info about the subcommand
#  # This displays the following:
#  #    foo
#  puts [::Simple::ExtProc::information subcommands command]
#
# -details    : 
#  * For the sake of performance, this package does not implement some of the
#    more esoteric features of the P<opt> package such as argument names
#    abbreviation, type guessing, completion of incomplete arguments or out of
#    order flags (this last feature might be added in a future revision of the
#    package). 
#
#     For procedures with flags and a small number of arguments and/or with
#    run-time type checking enabled, P<SimpleExtProc> is about ten times
#    faster than the P<opt> package.  Also, several tricks are applied to
#    improve performance (see the details section of the
#    P<::Simple::ExtProc::create> procedure) which result in typical
#    procedures (those which do not require argument run-time type checking
#    and have no flags in the argument list) B<behaving identically> to
#    regular Tcl procedures (no run-time parsing is required). 
#
#     For a ten non-flag arguments procedure, the P<opt> package requires
#    25000 microseconds per call (tcl 8.0, 233 MHz AMD K6, Linux Red Hat 5.1)
#    while P<SimpleExtProc> uses a mere 15 or 2500 microseconds when run-time
#    type checking is disabled or enabled, respectively; that means
#    performance improvement factors of B<1600> or B<10>, respectively.  For a
#    ten flag arguments procedure the times are 22900 and 1965 microseconds, a
#    factor of B<11.2>.  Even for procedures with flags, their interpretation
#    can be delegated to the user so, again, no overhead is introduced if
#    desired. 
#
#  * Each extended procedure argument state is stored in several element of an
#    array named V<%ExtProcs%> in the procedure namespace.  The internal
#    representation is stored in an array element whose index is the
#    unqualified procedure name.  This array element is a list with the
#    following elements:
#     0: whether the argument is a flag
#     1: whether the argument has a default value
#     2: argument name
#     3: argument default value
#     4: argument type
#     5: argument choices (empty if type is not "-choice")
#     6: argument description
#  * The argument names list of each extended procedure argument list is
#    stored in a second element of that same array whose index is
#    "%Names%,<unqualified procedure name>".
#  * The position of the first and last flags in the argument list of each
#    extended procedure argument is stored in a third element of that same
#    array whose index is "%FlagsPositions%,<unqualified procedure name>".
#  * Whether the flag interpretation is enabled or has been delegated to the
#    user for each extended procedure is stored in a fourth element of that
#    same array whose index is "%InterpFlags%,<unqualified procedure name>".
#
# -remarks    :
#  * This package is optimized for speed and not for maintenability.  In
#    particular the run-time parsing of the variable list is done by one of
#    two different private procedures
#    (P<::Simple::ExtProc::Priv::parse-arguments> and
#    P<::Simple::ExtProc::Priv::parse-arguments-checktype>) depending on
#    whether run-time type checking is enabled or not even though they share
#    most of the code.
#
# -todo       :
#  * Provide out of order flags.
#  * It might be useful to have a way to enable and disable I<-checktype> for
#    existing procedures.
#  * Provide a package option I<-storedetails> to choose whether to store or
#    not the extended arguments descriptions.
#
# -history    :
#  19-feb-1999   Unreleased first version 0.1
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleExtProc) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleExtProc 0.2 -required {
   SimpleError SimplePackage SimpleSubcommand SimpleType SimpleExtVar
} -namespaces {
   ::Simple::ExtProc
   ::Simple::ExtProc::Priv
   ::Simple::ScriptHead
} -export {
   ::Simple::ExtProc::next-arg
   ::Simple::ExtProc::previous-arg
   ::Simple::ExtProc::number-flags
   ::Simple::ExtProc::flag-given
   ::Simple::ExtProc::arg-is-default
} -aliases {
   {proc-ext ::Simple::ExtProc::create}
} -firsttimeinstall {

### Create the package options
::Simple::ExtVar::create\
   { ::Simple::ExtProc::CheckType 
                      -boolean 0     {Whether to perform run-time type checking
                                      of the procedures arguments}}

::Simple::ExtVar::create\
   { ::Simple::ExtProc::AfterFlags 
                      -int 2         {Minimum number of arguments after the 
                                      last flag for a procedure in order for
                                      those are handled with an auxiliary 
                                      procedure}}

### Create the package variables
set ::Simple::ScriptHead::MarkerStart #=========
set ::Simple::ScriptHead::MarkerFormat\
   "$::Simple::ScriptHead::MarkerStart %-14s %4d %s"

### Declare the package errors
::Simple::Error::declare ::Simple::ExtProc::NON-EXISTING-NAMESPACE {
   can't create procedure "%s": unknown namespace
} {
   The extended procedure <procedure> could not be created because the
   namespace does not exist
} {
   Use K<namespace eval> to create the namespace
}

::Simple::Error::declare ::Simple::ExtProc::TYPE-CHECKING-AND-NO-INTERP {
   can not have "-interpflags false" and argument run-time type checking
} {
   A procedure declaration failed because argument run-time type checking was
   enabled and flag interpretation disabled
} {
   Either disable argument run-time type checking or enable flag interpretation
}

::Simple::Error::declare ::Simple::ExtProc::ARGS-MUST-BE-LAST {
   "args" must be the last argument
} {
   A procedure declaration failed because an argument
   named "args" was supplied not the last in the argument list
}

::Simple::Error::declare ::Simple::ExtProc::NON-OPT-AFTER-OPT {
   non-optional argument "%s" after at least one optional argument
} {
   A procedure declaration failed because the non-optional argument <argument>
   was supplied after an optional argument
} {
   All arguments following an optional one must be either explicitly optional
   or the special argument "args" which is implictly optional
}

::Simple::Error::declare ::Simple::ExtProc::DEFAULT-NOT-ALLOWED {
   can not have a default value for argument "%s"
} {
   A procedure declaration failed because a default value was supplied for an
   <argument> which can not have it
} {
   Default values must be supplied for optional arguments and for flags whose
   type is not "-boolflag", and only for those
}

::Simple::Error::declare ::Simple::ExtProc::ARGS-NOT-ALLOWED {
   "args" is not allowed if flag interpretation is disabled
} {
   A procedure declaration failed because an argument named "args" was
   supplied but flag interpretation is disabled
} {
   Either remove the "args" argument or enable flags interpretation
}

::Simple::Error::declare ::Simple::ExtProc::ARGS-LIST-OR-ANY {
   "args" must be of type "-list" or "-any"
} {
   A procedure declaration failed because an argument named "args" was
   supplied with a type other than "-list" or "-any"
}

::Simple::Error::declare ::Simple::ExtProc::NO-FLAGS-INTERP-AND-NO-FLAGS {
   flag interpretation disabled but no flags present
} {
   A procedure declaration failed because a flag interpretation was diabled
   but no flags were present in the argument list
} {
   Enable flag interpretation
}

::Simple::Error::declare ::Simple::ExtProc::NO-PARAM-FOR-FLAG {
   no parameter given for flag "%s" to "%s"
} {
   A call to the procedure <procedure> failed because no parameter was
   supplied for the flag <flag>
} {
   Redo the call supplying a parameter for the flag
} {
   <flag> <procedure>
}

::Simple::Error::declare ::Simple::ExtProc::NO-VALUE-FOR-PARAM {
   no value given for parameter "%s" to "%s"
} {
   A call to the procedure <procedure> failed because no value was supplied
   for argument <argument>
} {
   Redo the call using the correct list of arguments
} {
   <argument> <procedure>
}

::Simple::Error::declare ::Simple::ExtProc::CANT-BETWEEN-SUBCO-AND-NO-SUBCO {
   can't %s %ssubcommand "%s" to %ssubcommand "%s"
} {
   It is not allowed to <copy or rename> a non-subcommand procedure to\
   subcommand or viceversa
} {
} {
} {
   can't <copy or rename> [non-]subcommand "<name 1>" to
   [non-]subcommand "<name 2>"
}

::Simple::Error::declare ::Simple::ExtProc::NON-EXISTING-PROCEDURE {
   "%s" isn't a procedure
} {
   Procedure <procedure> does not exist.
} {
   Provide a valid procedure name
}

::Simple::Error::declare ::Simple::ExtProc::NON-EXISTING-EXTENDED-PROCEDURE {
   "%s" isn't an extended procedure
} {
   Procedure <procedure> does not exist or is not an extended procedure.
} {
   Provide a valid extended procedure name
}

::Simple::Error::declare ::Simple::ExtProc::NON-EXISTING-ARGUMENT {
   procedure "%s" doesn't have an argument "%s"
} {
   Procedure <procedure> does not have an argument named <incorrect argument>
} {
   Provide an argument from the procedure argument list
}

::Simple::Error::declare ::Simple::ExtProc::NO-TYPE-CHOICE {
   argument "%s" from procedure "%s" is not of type "-choice"
} {
   The extended procedure <procedure> argument <argument> is not of type
   "-choice"
}

::Simple::Error::declare ::Simple::ExtProc::CANT-BE-CALLED-FROM-TOPLEVEL {
   "%s" can't be called from the top level
} {
   It makes no sense to call procedure <procedure> from the top level
}

::Simple::Error::declare ::Simple::ScriptHead::NON-EXISTING-SCRIPT {
   non-existing script "%s"
} {
   Script <script> does not exist
} {
   Use "set scriptName {}" to create an empty script or use another script
}

::Simple::Error::declare ::Simple::ScriptHead::NON-EXISTING-HEADER {
   non-existing tag "%s" in script "%s"
} {
   Script <script> does not contain a header with tag <tag>
} {
   Use K<::Simple::ScriptHead::add> to add the header to the script
} {<tag> <script>}

::Simple::Error::declare ::Simple::ScriptHead::DUPLICATED-TAG {
   duplicated tag "%s" in script "%s"
} {
   Script <script> already contains a header with tag <tag>
} {
   Use K<::Simple::ScriptHead::delete> to delete the header with that tag
   or use another tag
} {<tag> <script>}

### ===========================================================================
### -command  : ::Simple::ExtProc::next-arg
# -purpose    : Returns the next argument in the actual argument list.
# -access     : Export
# -overview   :
#     This procedure is intended to be called in procedures with extended
#  argument syntax created via the P<::Simple::ExtProc::create> procedure with
#  the I<-interpflags false> qualifier.  It provides the next argument in the
#  actual argument list.
#
#     It can also be used in regular Tcl procedures in which the arguments
#  are called I<arg1>, I<arg2> and so on, as in extended procedures with no
#  flag interpretation.
#
# -keywords   : extended argument flag
# -returns    : The next argument in the actual argument list.
# -limitations:
#  * The pointer to the next argument is stored in a variable named
#    A<%ARG-NUMBER%> in the scope of the calling procedure.
#
# -effects    :
#  * Modifies the variable named A<%ARG-NUMBER%> in the calling scope.
#
# -examples   :
#
#  # A procedure with flags interpretation delegated to the user which
#  # uses P<next-arg> and P<previous-arg> to interpret the flags
#  proc-ext -interpflags false {
#     foo
#  } {
#     { int1       -int           {}}
#     {-boolflag   -boolflag      {}}
#     {-flagstring -string "bar"  {}}
#     { int2       -int           {}}
#  } {
#     # This is the code to interpret the arguments
#
#     # Get first argument
#     set int1 [next-arg]
#
#     # Get next argument
#     set tmp [next-arg]
#
#     # Is it -boolflag ?
#     if {[string compare $tmp -boolflag]} {
#
#        # No 
#        set boolflag 0
#
#     } else {
#
#        # Yes
#        set boolflag 1
#
#        # Eat this argument, get next one
#        set tmp [next-arg]
#     }
#
#     # Is it -flagstring ?
#     if {[string compare $tmp -flagstring]} {
#
#        # No, set the default value
#        set flagstring "bar"
#
#        # Go back one argument
#        previous-arg
#
#     } else {
#
#        # Yes, get parameter from next argument
#        set flagstring [next-arg]
#     }
#
#     # Get last argument
#     set int2 [next-arg]
#     
#     # Display the arguments
#     puts "int1        = $int1"
#     puts "-boolflag   = $boolflag"
#     puts "-flagstring = $flagstring"
#     puts "int2        = $int2"
#  }
#
# -assumes    :
#  * That the next argument exists.
#
# -seealso    :
#  * The P<previous-arg> and P<proc-ext> procedures.
#
proc ::Simple::ExtProc::next-arg {
} {
   ### Up one level
   uplevel {

      ### Increment argument pointer
      if {[info exists %ARG-NUMBER%]} {
         incr %ARG-NUMBER%
      } else {
         set %ARG-NUMBER% 1
      }

      ### Return next argument
      set arg${%ARG-NUMBER%}
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::previous-arg
# -purpose    : Go back one argument in the actual argument list.
# -access     : Export
# -overview   :
#     This procedure is intended to be called in procedures with extended
#  argument syntax created via the P<::Simple::ExtProc::create> procedure with
#  the I<-interpflags false> qualifier.  It goes go back one argument in the
#  actual argument list in order to point back to the argument returned by the
#  previous call to P<next-arg>.
#
#     It can also be used in regular Tcl procedures in which the arguments
#  are called I<arg1>, I<arg2> and so on, as in extended procedures with no
#  flag interpretation.
#
# -keywords   : extended argument flag
# -limitations:
#  * The pointer to the next argument is stored in a variable named
#    A<%ARG-NUMBER%> in the scope of the calling procedure.
#
# -effects    :
#  * Modifies the variable named A<%ARG-NUMBER%> in the calling scope.
#
# -examples   : See the examples section of the P<next-arg> procedure
# -assumes    :
#  * That P<next-arg> has been called before.
#
# -seealso    :
#  * The P<next-arg> and P<proc-ext> procedures.
#
proc ::Simple::ExtProc::previous-arg {
} {
   ### Up one level
   uplevel {

      ### Decrement argument pointer
      incr %ARG-NUMBER% -1
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::Priv::reset
# -purpose    : Deletes an extended procedure auxiliary data.
# -overview   :
#     This procedure deletes all auxiliary data (variables and procedures)
#  corresponding to a procedure with extended argument syntax. Notice that
#  the actual procedure is not deleted.
#
# -arguments  :
#  { procedure        -qualifiedname {Fully-qualified procedure name}}
#
# -remarks    :
#     All commands in this procedure are protecetd against failure via the
#  K<catch> command; this means that no error is issued even if there is no
#  auxiliary data to delete.
#
proc ::Simple::ExtProc::Priv::reset {
   procedure
} {
   ### Delete the "after flags" auxiliary procedure, if any
   catch {rename $procedure-%AFTER-FLAGS% {}}

   ### Split the procedure name
   regexp -- ^(.*::)(.*)$ $procedure tmp namespace unqualifiedProcedure

   ### Delete all extended procedure auxiliary information
   catch {unset ${namespace}%ExtProcs%($unqualifiedProcedure)}
}

### ===========================================================================
### -command  : ::Simple::ExtProc::Priv::parse-qualify-split-name
# -purpose    : Parses, fully-qualifies and splits a name.
# -overview   :
#     This procedure parses a name which may be one or two words.
#
#     In case the name contains two words, it is assumed to be a base command
#  and subcommand pair, which are validated.  In case the base command name is
#  unqualified, it is fully-qualified in the scope two levels up the stack. 
#
#     In case the name contains a single word, the name is validated.  In case
#  it is unqualified, it is fully-qualified in the scope two levels up the
#  stack.
#
# -arguments  :
#  { name             -name          {Procedure name}} 
#  {?kind?            -string procedure
#                                    {Kind of name (procedure, argument
#                                     list...)}}
#
# -returns    : The name namespace and tail.
#
proc ::Simple::ExtProc::Priv::parse-qualify-split-name {
   name
   {kind procedure}
} {
   ### Invalid number of words in name or invalid name
   set nWords [llength $name]
   if {$nWords < 1 || $nWords > 2 ||\
      ($nWords == 1 && ![::Simple::Type::is -name $name])} {

      ### Throw error
      ::Simple::Error::throw ::Simple::BAD-ITEM "$kind name" $name
   }

   ### Subcommand procedure
   if {$nWords == 2} {

      ### Get the base command and subcommand names
      foreach {command subcommand} $name break

      ### Assert the base command and subcommand names are valid
      if {![::Simple::Type::is -name $command] ||\
         ![string compare $command {}]} {

         ::Simple::Error::throw ::Simple::BAD-ITEM\
            {subcommand base command name} $command
      }
      if {![::Simple::Type::is -unqualifiedname $subcommand] ||\
         ![string compare $subcommand {}]} {

         ::Simple::Error::throw ::Simple::BAD-ITEM {subcommand name}\
            $subcommand
      }
   }

   ### Relative name
   if {![string match ::* $name]} {

      ### Qualify the name in the scope two levels up the stack
      set name [::Simple::Namespace::Priv::qualify $name 2]
   }

   ### Return the name namespace and tail
   regexp -- ^(.*::)(.*)$ $name tmp namespace unqualifiedName
   list $namespace $unqualifiedName
}

### ===========================================================================
### -command  : ::Simple::ExtProc::create
# -purpose    : Creates a procedure with extended argument syntax.
# -alias      : proc-ext
# -overview   :
#     This procedure acts as an interface to the K<proc> Tcl command to create
#  other procedures allowing a richer syntax for their arguments.  Upon
#  creation of a new procedure, its extended format argument list is parsed
#  and converted into a canonical argument list (in which each argument has
#  the format {name ?defaultValue?}) which is used to actually create the
#  procedure via the Tcl K<proc> command.  Depending on the argument list and
#  other options, the procedure body may require instrumentation to add
#  run-time handling of the argument list.
#
#     If the procedure name is two words, a subcommand procedure similar to
#  that created via the K<proc-sub> procedure of the P<SimpleSubcommand>
#  package is created, but with extended argument syntax. It is thus not
#  necessary to use K<proc-sub> when using this package.
#
#  H1<Extended format arguments>
#
#     The extended format arguments are extended format variables as described
#  in the SimpleExtVar package.  Here we describe the specifities applicable
#  to the extended argument syntax of this procedure.
#
#  H2<Argument name>
#
#     Variables whose name starts by a hyphen are flags.  Flags whose type is
#  I<-boolflag> are boolean flags and do not require parameter.  For each
#  boolean flag in the argument list, a variable is available in the procedure
#  body with the same name as the flag excluding the hyphen and whose value is
#  true or false depending on whether the flag is present or not in the actual
#  argument list.
#
#     Flags whose type is not I<-boolflag> are regular flags and require a
#  parameter; the flag type is the expected parameter type.  For each regular
#  flag in the argument list, a variable is available in the procedure body
#  with the same name as the flag excluding the hyphen and whose value is the
#  parameter supplied after the flag or its default value if the flag was not
#  given in the actual argument list.  The presence of a regular flag in the
#  actual argument list can be obtained via the P<flag-given> procedure.
#
#     Arguments whose name is enclosed between question marks are optional.
#  They differ from the flags in that they behave as optional arguments to the
#  K<proc> command, that is, they must appear at the end of the argument list.
#
#     As with the K<proc> command, the I<args> special argument collects all
#  remaining arguments in the actual argument list and combines them into a
#  list which is available in the procedure body in the variable name I<args>. 
#  If present, the I<args> special argument must be the last one.
#
#  H2<Argument type>
#
#     An actual argument whose type is I<-choice> can only hold one of the
#  values specified in the argument declaration default value field. See
#  the I<-boolflag> type in the argument name section above.
#
#  H2<Argument default value or choice list>
#
#     An argument declaration can specify this field if and only if its type
#  is I<-choice> or is an optional argument whose type is not I<-boolflag>. 
#  For arguments of type I<-choice>, this field specifies the list of allowed
#  choices otherwise the default value for optional argument.  For optional
#  arguments of type I<-choice>, the default value is the first element in the
#  choice list.
#
#  H2<Argument description>
#
#     This is simply a description of the argument.
#
#  H1<Run-time handling>
#
#     The I<-checktype> flag allows to override the package option of the same
#  name for the procedure being created.  This option controls whether a
#  run-time type check is performed to check whether the arguments contents
#  match their type.  This has a performance penalty which should be
#  considered.  The overhead comes from two fronts.  First, the contents of
#  each argument must be checked.  Second, it is not possible to apply the
#  improving performance tricks described in the details section below.
#
#     The I<-interpflags> flag allows to disable the flag interpretation which
#  is delegated to the user.  This results in better performance (the
#  procedure call is equivalent to a regular Tcl K<proc> procedure) at the
#  cost of some hassle and restrictions:
#
#  * The argument list is replaced by arguments I<arg1>, I<arg2> and so on. 
#    The number of such arguments is the maximum number of arguments the
#    procedure can take, that is, non-flags nor optional arguments plus flags,
#    their parameters, and all optional arguments. 
#
#  * All these arguments are optional, with a default value equal to
#    I<%DEFAULT%> so that it is easy to identify whether the a particular
#    argument has been supplied in the actual argument list via the
#    P<arg-is-default> procedure.  As a result, arguments default values are
#    not honored.
#
#  * The I<args> special argument is not allowed.
#
#  * It is the user responsibility to handle those arguments.  See the K<puts>
#    clone in the examples section below.
#
#  * No run-time type checking of the procedure arguments is performed.  As a
#    result, it is not allowed to supply both the I<-interpflags> and
#    I<-checktype> flags.  If the I<-checktype> package option is set to true,
#    it is not honored.
#
# -keywords   : procedure argument type flag
# -arguments  :
#  {-interpflags      -extbool true  {Whether to enable flag interpretation or
#                                     delegate it to the user}}
#  {-checktype        -extbool false {Whether to enable argument run-time
#                                     type checking for this procedure}}
#  { procedure        -name          {Procedure name}}
#  { arguments        -list          {Argument list}}
#  { body             -script        {Procedure body}}
#
# -details    :
#     If the I<-interpflags false> modifier is present, the actual procedure
#  which is created has as many arguments as described in the run-time
#  handling sectionabove.  The procedure body is not instrumented.
#
#     If run-time type checking is enabled, the actual procedure which is
#  created has no arguments.  A call to the
#  P<::Simple::ExtProc::Priv::parse-arguments-checktype> procedure is added at
#  the very beginning of the procedure body to perform the necessary run-time
#  parsing and checking.
#
#     If run-time type checking is disabled, the arguments to the procedure
#  are divided into three sets:  arguments before the first flag, arguments
#  from the first to the last flag and arguments after the last flag.  The
#  actual procedure which is created contains the first set of arguments; the
#  second and third sets are handled by a call to the
#  P<::Simple::ExtProc::Priv::parse-arguments> procedure which is added at the
#  very beginning of the procedure body.  If the number of arguments in the
#  third set exceeds the I<-afterflags> package option value, those are
#  handled via an auxiliary "after flags" procedure.
#
# -remarks    :
#  * If the procedure namespace does not exists it is not created. This mimics
#    the behaviour of the K<proc> command.
#
#  * The "after flags" procedure is named "<proc-name>-%AFTER-FLAGS%"
#    where "<proc-name>" is the fully-qualified name of the procedure being
#    created.
#
#  * The list of flags present in the actual argument list is stored in a
#    variable named "%FLAGS%" in the scope of the procedure being created.
#
#  * The header added to the script body when argument run-time type checking
#    is enabled is tagged as "%PRIV%-parse-arguments" or
#    "%PRIV%-parse-arguments-checktype" with priority 0.
#
# -examples   :
#
#  # A regular procedure
#  proc-ext name {
#     {-boolflag    -boolflag             "A boolean flag"}
#     { arg1        -string               "First argument"}
#     { arg2        -int                  "Second argument"}
#     { arg3        -choice   {foo bar}   "Choice between foo and bar"}
#     {-flag        -string   theflag     "A flag (requires parameter)"}
#     {?arg4?       -string   default     "Optional argument"}
#     { args        -list                 "Remaining args"}
#  } {
#     puts "-boolflag = <$boolflag>"
#     puts "arg1      = <$arg1>"
#     puts "arg2      = <$arg2>"
#     puts "arg3      = <$arg3>"
#     puts -nonewline "flag      = <$flag> "
#     if {[flag-given -flag]} {
#        puts "(flag given)"
#     } else {
#        puts "(default value)"
#     }
#     puts "arg4      = <$arg4>"
#     puts "args      = <$args>"
#  }
#
#  # A subcommand procedure for which the contents of
#  # their arguments is run-time type checked
#  proc-ext -checktype true {command subcommand} {
#     {-boolflag    -boolflag             "A boolean flag"}
#     { arg1        -string               "First argument"}
#     { arg2        -int                  "Second argument"}
#  } {
#     puts "-boolflag = <$boolflag>"
#     puts "arg1      = <$arg1>"
#     puts "arg2      = <$arg2>"
#  }
#
#  # A clone of K<puts> with flag interpretation
#  # delegated to the user
#  proc-ext -interpflags false name {
#     {-nonewline   -boolflag             "Do not append a newline"}
#     { channelId  -channel               "Output channel"}
#     { string      -string               "String to write"}
#  } {
#     if {![string compare $arg1 -nonewline]} {
#        puts -nonewline  $arg2 $arg3
#     } else {
#        puts             $arg1 $arg2
#     }
#  }
#
# -effects    :
#  * Creates a command named K<procedure>.
#
#  * May create an "after flags" command named K<procedure>-%AFTER-FLAGS%>.
#
proc ::Simple::ExtProc::create {
   args
} {
   ### Parse arguments
   set interpFlags 1
   set implicitCheckType 1
   set checkType $::Simple::ExtProc::CheckType
   set nArgument 0
   set nArguments [llength $args]
   while {$nArgument < $nArguments} {
      set argument [lindex $args $nArgument]
      switch -glob -- $argument {
         -interpflags {
            set interpFlags [{::Simple::Type::convert boolean}\
               [lindex $args [incr nArgument]]]
         }
         -checktype {
            set implicitCheckType 0
            set checkType [{::Simple::Type::convert boolean}\
               [lindex $args [incr nArgument]]]
         }
         -* {
            ::Simple::Error::throw ::Simple::BAD-OPTION $argument\
               [::Simple::Proc::Priv::enumerate {-checktype -interpflags}]
         }
         default {
            set remainingArgs [lrange $args $nArgument end]
            set nRemainingArgs [llength $remainingArgs]
            if {$nRemainingArgs > 3} {
               ::Simple::Error::throw ::Simple::TOO-MANY-ARGS\
                  [lindex [info level 0] 0]
            } elseif {$nRemainingArgs < 3} {
               ::Simple::Error::throw ::Simple::ExtProc::NO-VALUE-FOR-PARAM\
                  [lindex {procedure arguments body} $nRemainingArgs]\
                  [lindex [info level 0] 0]
            }
            foreach {procedure arguments body} $remainingArgs break
            set procedure [string trim $procedure]
            break
         }
      }
      incr nArgument
   }

   ### "-interpflags false" and "-checktype true" is not allowed
   if {$checkType && !$interpFlags} {

      ### Implicit check type
      if {$implicitCheckType} {

         ### Disable run-time type checking
         set checkType 0

      ### Explicit check type
      } else {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::TYPE-CHECKING-AND-NO-INTERP
      }
   }

   ### Parse and qualify the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break
   set qualifiedProcedure $namespace$unqualifiedProcedure

   ### Assert the namespace exists
   if {[catch {namespace children $namespace}]} {

      ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-NAMESPACE\
         $procedure
   }

   ### Subcommand procedure
   if {[llength $procedure] == 2} {

      ### Save the command
      ::Simple::Proc::Priv::save-name [lindex $procedure 0]

      ### Create the base command
      ::Simple::Subcommand::Priv::create-base-command\
         $namespace[lindex $unqualifiedProcedure 0] [lindex $procedure 1]
   }

   ### Delete the procedure and auxiliary data, in case it is being recreated
   catch [list rename $qualifiedProcedure {}]
   ::Simple::ExtProc::Priv::reset $qualifiedProcedure

   ### Loop over arguments
   set argsGiven 0
   set nArgs 0
   set firstFlag -1
   set lastFlag -1
   set optionalGiven 0
   set internal [list]
   set argNames [list]
   foreach argument $arguments {

      ### Assert there are no arguments after "args"
      if {$argsGiven} {
         ::Simple::Error::throw ::Simple::ExtProc::ARGS-MUST-BE-LAST
      }

      ### Parse the argument
      foreach {isOptional isFlag defaultValueGiven argName defaultValue\
         type choices description originalArgName}\
         [::Simple::ExtVar::Priv::parse argument $argument] break

      ### Handle optional arguments
      if {$isOptional} {
         set optionalGiven 1
      } elseif {$optionalGiven && [string compare $argName args]} {
         ::Simple::Error::throw ::Simple::ExtProc::NON-OPT-AFTER-OPT\
            $originalArgName
      }

      ### Handle flag arguments
      if {$isFlag} {
         if {$firstFlag == -1} {
            set firstFlag $nArgs
         }
         set lastFlag $nArgs
      }

      ### Assert the default value is required, if given
      if {$defaultValueGiven && !$isOptional &&\
         !($isFlag && [string compare $type -boolflag])} {

         ::Simple::Error::throw ::Simple::ExtProc::DEFAULT-NOT-ALLOWED\
            $originalArgName
      }

      ### Handle special argument "args"
      if {![string compare $argName args]} {
         if {!$interpFlags} {
            ::Simple::Error::throw ::Simple::ExtProc::ARGS-NOT-ALLOWED
         }
         if {[string compare $type -list] && [string compare $type -any]} {
            ::Simple::Error::throw ::Simple::ExtProc::ARGS-LIST-OR-ANY
         }
         set argsGiven 1
      }

      ### Argument is valid
      lappend internal [list $isFlag $defaultValueGiven $argName $defaultValue\
         $type $choices $description]
      if {$isFlag} {
         lappend argNames -$argName
      } else {
         lappend argNames $argName
      }

      ### Next argument
      incr nArgs
   }

   ### Assert there is at least one flag if flags interpretation is disabled
   if {!$interpFlags && $firstFlag == -1} {
      ::Simple::Error::throw ::Simple::ExtProc::NO-FLAGS-INTERP-AND-NO-FLAGS
   }

   ### Store the procedure argument list internal representation
   set ${namespace}%ExtProcs%($unqualifiedProcedure) $internal

   ### Store the procedure argument names list
   set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure) $argNames

   ### Assess whether the argument list needs run-time parsing
   #   That is, whether the argument list contains any flag
   #   unless flag interpretation is disabled
   set runTimeParsing [expr {$firstFlag != -1 && $interpFlags}]

   ### Run-time argument type checking or flag interpretation disabled
   if {$checkType || !$interpFlags} {

      # Set the position of the last flag in the
      # argument list to -1 so that it is ignored
      set lastFlag -1

   ### No run-time argument type checking and flag interpretation disabled
   } elseif {$lastFlag != -1} {

      ### "after flags" auxiliary procedure required
      #   This happens if the number of arguments after the last flag exceeds
      #   the after flags threshold, unless run-time type checking is enabled
      #   because then all arguments are parsed at run time
      #      Notice that the "after flags" procedure does not need to be
      #   exported from its namespace, even if its originator procedure is
      #   exported, as imported commands simply point to the exported one in
      #   its original namespace.
      if {$nArgs - $lastFlag > $::Simple::ExtProc::AfterFlags} {

         # Loop over arguments
         set afterFlagsArgs [list]
         set argList1 [list]
         set argList2 [list]
         foreach argument [lrange $arguments [expr {$lastFlag + 1}] end] {

            # Convert the extended argument to canonical format
            set canonicalArgument\
               [::Simple::ExtVar::convert canonical $argument]

            # Append to argument list
            set argName [lindex $canonicalArgument 0]
            lappend argList1 $argName
            lappend argList2 $$argName

            # Append to "after flags" auxiliary procedure argument list
            lappend afterFlagsArgs $canonicalArgument
         }

         # Create the "after flags" auxiliary procedure with
         # a name of the form <procedure>-%AFTER-FLAGS%
         proc $qualifiedProcedure-%AFTER-FLAGS% $afterFlagsArgs\
            [format {uplevel 2 "foreach {%s} {%s} {}"} $argList1 $argList2]

      ### "after flags" auxiliary procedure not required
      } else {

         # Set the position of the last flag in the
         # argument list to -1 so that it is ignored
         set lastFlag -1
      }
   }

   ### Store the flags positions in the argument list
   set ${namespace}%ExtProcs%(%FlagsPositions%,$unqualifiedProcedure)\
      [list $firstFlag $lastFlag]

   ### Flags interpretation requested
   if {$interpFlags} {

      # The procedure has flag interpretation enabled
      set ${namespace}%ExtProcs%(%InterpFlags%,$unqualifiedProcedure) 1

      ### Run-time type checking requested and necessary
      #   Run-time type checking is not necessary if the procedure has no
      #   arguments
      if {$checkType && $nArgs > 0} {

         ### Create the actual procedure
         if {!$runTimeParsing} {
            ::Simple::ScriptHead::add body %PRIV%-parse-arguments-checktype 0\
               [format {::Simple::ExtProc::Priv::parse-arguments-checktype %s\
               $args} [list $qualifiedProcedure]]
         } else {
            ::Simple::ScriptHead::add body %PRIV%-parse-arguments-checktype 0\
               [format {set %%FLAGS%%\
               [::Simple::ExtProc::Priv::parse-arguments-checktype\
               %s $args]} [list $qualifiedProcedure]]
         }
         proc $qualifiedProcedure args $body

      ### Run-time type checking not requested or not necessary
      } else {

         ### Get the argument list up to the first flag
         if {$firstFlag == -1} {
            set beforeFlagsExtendedArgs $arguments
         } else {
            set beforeFlagsExtendedArgs\
               [lrange $arguments 0 [expr {$firstFlag - 1}]]
         }
         set beforeFlagsArgs [list]
         foreach argument $beforeFlagsExtendedArgs {
 
            # Convert the extended argument to canonical format
            set canonicalArgument\
               [::Simple::ExtVar::convert canonical $argument]
 
            # Append to argument list
            lappend beforeFlagsArgs $canonicalArgument
         }

         ### Create the actual procedure
         if {!$runTimeParsing} {

            # Run-time parsing of the actual argument list not required
            proc $qualifiedProcedure $beforeFlagsArgs $body

         } else {

            # Run-time parsing of the actual argument list required
            ::Simple::ScriptHead::add body %PRIV%-parse-arguments 0\
               [format {set %%FLAGS%%\
               [::Simple::ExtProc::Priv::parse-arguments\
               %s $args]} [list $qualifiedProcedure]]
            proc $qualifiedProcedure [concat $beforeFlagsArgs args] $body
         }
      }

   ### Flags interpretation not requested
   } else {

      # The procedure has flag interpretation delegated to the user
      set ${namespace}%ExtProcs%(%InterpFlags%,$unqualifiedProcedure) 0

      ### Build the argument list
      #   Compute the minimum (no flags nor optional arguments given) and
      #   maximum (all flags and optional arguments given) number of arguments 
      #   a call to this procedure may have
      set nArgs 0
      set args [list]
      foreach argument $internal {

         # One argument more
         incr nArgs
         lappend args [list arg$nArgs %DEFAULT%]

         # Non-boolean flag
         if {[lindex $argument 0] &&\
            [string compare [lindex $argument 4] -boolflag]} {

            # One argument more
            incr nArgs
            lappend args [list arg$nArgs %DEFAULT%]
         }
      }

      ### Create the actual procedure
      proc $qualifiedProcedure $args $body
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::delete
# -purpose    : Deletes a procedure.
# -overview   :
#     This command is similar to the K<rename> command when its second
#  argument is the empty string but works for extended procedures as well as
#  for regular procedures.
#
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -effects    :
#  * Deletes the procedure named "A<procedure>".
#
proc ::Simple::ExtProc::delete {
   procedure
} {
   ### Parse and qualify the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break
   set qualifiedProcedure $namespace$unqualifiedProcedure
   
   ### Subcommand procedure
   if {[llength $procedure] == 2} {

      ### Save the base command name
      ::Simple::Proc::Priv::save-name [lindex $procedure 0]

      ### Delete the subcommand procedure
      ::Simple::Subcommand::delete $qualifiedProcedure

      ### Delete the base command auxiliary data
      ::Simple::ExtProc::Priv::reset $namespace[lindex $unqualifiedProcedure 0]

   ### Non-subcommand procedure
   } else {

      ### Delete the regular procedure
      uplevel [list rename $procedure {}]
   }

   ### Delete the procedure auxiliary data
   ::Simple::ExtProc::Priv::reset $qualifiedProcedure
}

### ===========================================================================
### -command  : ::Simple::ExtProc::move
# -purpose    : Renames a procedure.
# -overview   :
#     This command is similar to the K<rename> command when its second
#  argument is non-empty but works for extended procedures as well as for
#  regular procedures.
#
# -arguments  :
#  { sourceProcedure  -name          {Source procedure name}}
#  { targetProcedure  -name          {Target procedure name}}
#
# -remarks    :
#  * If the target procedure namespace does not exists, it is created. 
#    This mimics the behaviour of the K<rename> command.
#
# -todo       :
#  * Allow to rename from subcommand to regular procedure and viceversa.
#
# -effects    :
#  * Renames the procedure named "A<sourceProcedure>".
#
proc ::Simple::ExtProc::move {
   sourceProcedure
   targetProcedure
} {
   ### Parse and qualify the procedure names
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $sourceProcedure] break
   set qualifiedSourceProcedure $namespace$unqualifiedProcedure
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $targetProcedure] break
   set qualifiedTargetProcedure $namespace$unqualifiedProcedure

   ### Assert either both or none of the procedures are subcommands
   set sourceIsSubcommand [expr {[llength $sourceProcedure] == 2}]
   set targetIsSubcommand [expr {[llength $targetProcedure] == 2}]
   if {$sourceIsSubcommand && !$targetIsSubcommand} {
      ::Simple::Error::throw\
         ::Simple::ExtProc::CANT-BETWEEN-SUBCO-AND-NO-SUBCO\
         rename {} $sourceProcedure non- $targetProcedure
   } elseif {!$sourceIsSubcommand && $targetIsSubcommand} {
      ::Simple::Error::throw\
         ::Simple::ExtProc::CANT-BETWEEN-SUBCO-AND-NO-SUBCO\
         rename non- $sourceProcedure {} $targetProcedure
   }

   ### Extended procedure
   if {[{::Simple::ExtProc::information exists} $qualifiedSourceProcedure]} {

      ### Get the procedure necessary information
      set checkType\
         [{::Simple::ExtProc::information checktype} $qualifiedSourceProcedure]
      set interpFlags [{::Simple::ExtProc::information interpflags}\
         $qualifiedSourceProcedure]
      set arguments\
         [{::Simple::ExtProc::information argslist} $qualifiedSourceProcedure]
      set body [info body $qualifiedSourceProcedure]

      ### Save the procedure name
      ::Simple::Proc::Priv::save-name $sourceProcedure

      ### Delete the procedure
      ::Simple::ExtProc::delete $qualifiedSourceProcedure

      ### Delete the run-time argument parsing header, if any
      ::Simple::ScriptHead::delete body %PRIV%-parse-arguments*

      ### Create the target namespace
      namespace eval [namespace qualifiers $qualifiedTargetProcedure] {}

      ### Create the target procedure
      ::Simple::ExtProc::create -checktype $checkType\
         -interpflags $interpFlags $qualifiedTargetProcedure $arguments $body

   ### Non existing or non-extended procedure
   } else {

      ### Subcommand procedure
      if {$sourceIsSubcommand} {

         ### Save the base command name
         ::Simple::Proc::Priv::save-name [lindex $sourceProcedure 0]

         ### Rename the subcommand procedure
         ::Simple::Subcommand::move\
            $qualifiedSourceProcedure $qualifiedTargetProcedure

      ### Non-subcommand procedure
      } else {

         ### Rename the regular procedure
         uplevel [list rename $sourceProcedure $qualifiedTargetProcedure]
      }
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::copy
# -purpose    : Copies a procedure.
# -overview   :
#     This command copies a procedure, either extended or regular
#
# -arguments  :
#  { sourceProcedure  -name          {Source procedure name}}
#  { targetProcedure  -name          {Target procedure name}}
#
# -remarks    :
#  * If the target procedure namespace does not exists, it is created. 
#    This mimics the behaviour of the K<rename> command.
#
# -todo       :
#  * Allow to copy from subcommand to regular procedure and viceversa.
#
# -effects    :
#  * Creates a command named K<targetProcedure>.
#
proc ::Simple::ExtProc::copy {
   sourceProcedure
   targetProcedure
} {
   ### Parse and qualify the procedure names
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $sourceProcedure] break
   set qualifiedSourceProcedure $namespace$unqualifiedProcedure
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $targetProcedure] break
   set qualifiedTargetProcedure $namespace$unqualifiedProcedure

   ### Assert either both or none of the procedures are subcommands
   set sourceIsSubcommand [expr {[llength $sourceProcedure] == 2}]
   set targetIsSubcommand [expr {[llength $targetProcedure] == 2}]
   if {$sourceIsSubcommand && !$targetIsSubcommand} {
      ::Simple::Error::throw\
         ::Simple::ExtProc::CANT-BETWEEN-SUBCO-AND-NO-SUBCO\
         copy {} $sourceProcedure non- $targetProcedure
   } elseif {!$sourceIsSubcommand && $targetIsSubcommand} {
      ::Simple::Error::throw\
         ::Simple::ExtProc::CANT-BETWEEN-SUBCO-AND-NO-SUBCO\
         copy non- $sourceProcedure {} $targetProcedure
   }

   ### Extended procedure
   if {[{::Simple::ExtProc::information exists} $qualifiedSourceProcedure]} {

      ### Get the procedure necessary information
      set checkType\
         [{::Simple::ExtProc::information checktype} $qualifiedSourceProcedure]
      set interpFlags [{::Simple::ExtProc::information interpflags}\
         $qualifiedSourceProcedure]
      set arguments\
         [{::Simple::ExtProc::information argslist} $qualifiedSourceProcedure]
      set body [info body $qualifiedSourceProcedure]

      ### Save the procedure name
      ::Simple::Proc::Priv::save-name $sourceProcedure

      ### Delete the run-time argument parsing header, if any
      ::Simple::ScriptHead::delete body %PRIV%-parse-arguments*

      ### Create the target namespace
      namespace eval [namespace qualifiers $qualifiedTargetProcedure] {}

      ### Create the target procedure
      ::Simple::ExtProc::create -checktype $checkType\
         -interpflags $interpFlags $qualifiedTargetProcedure $arguments $body

   ### Non existing or non-extended procedure
   } else {

      ### Subcommand procedure
      if {$sourceIsSubcommand} {

         ### Save the base command name
         ::Simple::Proc::Priv::save-name [lindex $sourceProcedure 0]

         ### Copy the subcommand procedure
         ::Simple::Subcommand::copy\
            $qualifiedSourceProcedure $qualifiedTargetProcedure

      ### Non-subcommand procedure
      } else {

         ### Save the procedure name
         ::Simple::Proc::Priv::save-name $sourceProcedure

         ### Copy the regular procedure
         ::Simple::Proc::copy\
            $qualifiedSourceProcedure $qualifiedTargetProcedure
      }
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::Priv::parse-arguments
# -purpose    : Parses an argument list in run-time.
# -arguments  :
#  { procedure        -qualifiedname {Fully-qualified procedure name}}
#  { actualArgs       -list          {Actual argument list}}
#
# -returns    : The list of flags present in the actual argument list.
# -effects    :
#  * Creates the variables named after the command named K<procedure>
#    arguments in the calling scope.
#
proc ::Simple::ExtProc::Priv::parse-arguments {
   procedure
   actualArgs
} {
   # Split the procedure name
   regexp -- ^(.*::)(.*)$ $procedure tmp namespace unqualifiedProcedure

   # Flags positions flag in the argument list
   foreach {firstFlag lastFlag}\
      [set ${namespace}%ExtProcs%(%FlagsPositions%,$unqualifiedProcedure)] break

   # Argument list
   set arguments [set ${namespace}%ExtProcs%($unqualifiedProcedure)]

   # First argument
   set argument [lindex $arguments [set argumentIndex $firstFlag]]

   # First actual argument
   set actualArg [lindex $actualArgs [set actualArgIndex 0]]

   ### Loop over actual arguments
   set %FLAGS% [list]
   set actualArgsLlength [llength $actualArgs]
   while {$actualArgIndex < $actualArgsLlength} {

      ### Loop over arguments
      while 1 {

         set name [lindex $argument 2]
         set isFlag [lindex $argument 0]

         # If the argument name is empty we are beyond the last one
         if {![string compare $name {}]} {
            ::Simple::Error::throw ::Simple::TOO-MANY-ARGS\
               [::Simple::Proc::caller]
         }

         ### Handle flags
         if {$isFlag} {

            set requiresParam [lindex $argument 1]

            ### Handle match
            if {![string compare $actualArg -$name]} {

               # Flag present
               lappend %FLAGS% -$name

               ### Parameter required
               if {$requiresParam} {

                  # Take parameter from next actual argument
                  incr actualArgIndex
                  if {$actualArgIndex >= $actualArgsLlength} {
                     ::Simple::Error::throw\
                        ::Simple::ExtProc::NO-PARAM-FOR-FLAG -$name\
                        [::Simple::Proc::caller]
                  }
                  set parameter [lindex $actualArgs $actualArgIndex]

                  # Set the flag in the calling scope
                  uplevel [list set $name $parameter]

               ### Parameter not required
               } else {

                  # Set the boolflag in the calling scope: given
                  uplevel [list set $name 1]
               }

               # If the last flag has already being processed and
               # there is an "after flags" auxiliary procedure
               if {$argumentIndex == $lastFlag} {

                  # Unset the "args" special argument
                  # in the calling scope
                  uplevel [list unset args]

                  # Process the remaining arguments via the
                  # "after flags" auxiliary procedure

                  # Error evaluating the "after flags" auxiliary procedure
                  if {[catch {eval $procedure-%AFTER-FLAGS%\
                     [lrange $actualArgs [incr actualArgIndex] end]} result]} {

                     # Replace the "after flags" axuliary procedure name
                     # by the name of the caller for proper error reporting
                     regsub -all -- {"::[^"]*-%AFTER-FLAGS%} $result\
                        \"[::Simple::Proc::caller] result

                     # Rethrow the error
                     return -code error -errorcode $::errorCode $result
                  }

                  # Done
                  return ${%FLAGS%}
               }

               ### Eat argument
               set argument [lindex $arguments [incr argumentIndex]]

               ### Next actual argument
               break

            ### Handle non-match
            } else {

               ### Parameter required
               if {$requiresParam} {

                  # Set the argument in the calling scope,
                  # taking the parameter from the default value
                  set defaultValue [lindex $argument 3]
                  uplevel [list set $name $defaultValue]

               ### Parameter not required
               } else {

                  # Set the boolflag in the calling scope: not given
                  uplevel [list set $name 0]
               }

               # If the last flag has already being processed and
               # there is an "after flags" auxiliary procedure
               if {$argumentIndex == $lastFlag} {

                  # Unset the "args" special argument
                  # in the calling scope
                  uplevel [list unset args]

                  # Process the remaining arguments via the
                  # "after flags" auxiliary procedure

                  # Error evaluating the "after flags" auxiliary procedure
                  if {[catch {eval $procedure-%AFTER-FLAGS%\
                     [lrange $actualArgs $actualArgIndex end]} result]} {

                     # Replace the "after flags" axuliary procedure name
                     # by the name of the caller for proper error reporting
                     regsub -all -- {"::[^"]*-%AFTER-FLAGS%} $result\
                        \"[::Simple::Proc::caller] result

                     # Rethrow the error
                     return -code error -errorcode $::errorCode $result
                  }

                  # Done
                  return ${%FLAGS%}
               }

               ### Eat argument
               set argument [lindex $arguments [incr argumentIndex]]

               ### Next argument
               continue
            }

         ### Handle non-flags
         } else {

            if {![string compare $name args]} {

               # Get all remaining actual arguments
               uplevel [list set args [lrange $actualArgs $actualArgIndex end]]

               # Done
               return ${%FLAGS%}
            }

            # Set the argument in the calling scope
            uplevel [list set $name $actualArg]

            ### Eat argument
            set argument [lindex $arguments [incr argumentIndex]]

            ### Next actual argument
            break
         }
      }

      ### Eat actual argument
      set actualArg [lindex $actualArgs [incr actualArgIndex]]
   }

   ### Loop over not used arguments
   set argumentsLlength [llength $arguments]
   while {$argumentIndex < $argumentsLlength} {

      set name [lindex $argument  2]
      if {![string compare $name args]} {

         # No remaining actual arguments
         uplevel [list set args [list]]

         # Done
         return ${%FLAGS%}
      }

      set defaultGiven [lindex $argument 1]
      if {$defaultGiven} {

         # Set the argument in the calling scope,
         # taking the parameter from the default value
         set defaultValue [lindex $argument 3]
         uplevel [list set $name $defaultValue]

      } else {

         # If is flag
         if {[lindex $argument 0]} {
            if {!$defaultGiven} {

               # Set the boolflag in the calling scope: not given
               uplevel [list set $name 0]
            }
         } else {
            ::Simple::Error::throw ::Simple::ExtProc::NO-VALUE-FOR-PARAM $name\
               [::Simple::Proc::caller]
         }
      }

      ### Eat non-used argument
      set argument [lindex $arguments [incr argumentIndex]]
   }

   # If this point is reached, this procedure does not use the
   # special argument "args", so unset it in the calling scope
   uplevel [list unset args]

   # Done
   set %FLAGS%
}

### ===========================================================================
### -command  : ::Simple::ExtProc::Priv::parse-arguments-checktype
# -purpose    : Parses an argument list and checks their types in run-time.
# -arguments  :
#  { procedure        -qualifiedname {Fully-qualified procedure name}}
#  { actualArgs       -list          {Actual argument list}}
#
# -returns    : The list of flags present in the actual argument list.
# -effects    :
#  * Creates the variables named after the command named K<procedure>
#    arguments in the calling scope.
#
proc ::Simple::ExtProc::Priv::parse-arguments-checktype {
   procedure
   actualArgs
} {
   # Split the procedure name
   regexp -- ^(.*::)(.*)$ $procedure tmp namespace unqualifiedProcedure

   # Argument list
   set arguments [set ${namespace}%ExtProcs%($unqualifiedProcedure)]

   # First argument
   set argument [lindex $arguments [set argumentIndex 0]]

   # First actual argument
   set actualArg [lindex $actualArgs [set actualArgIndex 0]]

   ### Loop over actual arguments
   set %FLAGS% [list]
   set actualArgsLlength [llength $actualArgs]
   while {$actualArgIndex < $actualArgsLlength} {

      ### Loop over arguments
      while 1 {

         set name [lindex $argument 2]
         set isFlag [lindex $argument 0]

         # If the argument name is empty we are beyond the last one
         if {![string compare $name {}]} {
            ::Simple::Error::throw ::Simple::TOO-MANY-ARGS\
               [::Simple::Proc::caller]
         }

         ### Handle flags
         if {$isFlag} {

            set requiresParam [lindex $argument 1]

            ### Handle match
            if {![string compare $actualArg -$name]} {

               # Flag present
               lappend %FLAGS% -$name

               ### Parameter required
               if {$requiresParam} {

                  # Take parameter from next actual argument
                  incr actualArgIndex
                  if {$actualArgIndex >= $actualArgsLlength} {
                     ::Simple::Error::throw\
                        ::Simple::ExtProc::NO-PARAM-FOR-FLAG -$name\
                        [::Simple::Proc::caller]
                  }
                  set parameter [lindex $actualArgs $actualArgIndex]

                  ### Check the parameter type
                  ::Simple::ExtVar::Priv::assert-type-conformation argument\
                     $name [lindex $argument 4] $parameter [lindex $argument 5]

                  # Set the flag in the calling scope
                  uplevel [list set $name $parameter]

               ### Parameter not required
               } else {

                  # Set the boolflag in the calling scope: given
                  uplevel [list set $name 1]
               }

               ### Eat argument
               set argument [lindex $arguments [incr argumentIndex]]

               ### Next actual argument
               break

            ### Handle non-match
            } else {

               ### Parameter required
               if {$requiresParam} {

                  # Set the argument in the calling scope,
                  # taking the parameter from the default value
                  set defaultValue [lindex $argument 3]
                  uplevel [list set $name $defaultValue]

               ### Parameter not required
               } else {

                  # Set the boolflag in the calling scope: not given
                  uplevel [list set $name 0]
               }

               ### Eat argument
               set argument [lindex $arguments [incr argumentIndex]]

               ### Next argument
               continue
            }

         ### Handle non-flags
         } else {

            ### Check the argument type
            ::Simple::ExtVar::Priv::assert-type-conformation argument $name\
               [lindex $argument 4] $actualArg [lindex $argument 5]

            if {![string compare $name args]} {

               # Get all remaining actual arguments
               uplevel [list set args [lrange $actualArgs $actualArgIndex end]]

               # Done
               return ${%FLAGS%}
            }

            # Set the argument in the calling scope
            uplevel [list set $name $actualArg]

            ### Eat argument
            set argument [lindex $arguments [incr argumentIndex]]

            ### Next actual argument
            break
         }
      }

      ### Eat actual argument
      set actualArg [lindex $actualArgs [incr actualArgIndex]]
   }

   ### Loop over not used arguments
   set argumentsLlength [llength $arguments]
   while {$argumentIndex < $argumentsLlength} {

      set name [lindex $argument  2]
      if {![string compare $name args]} {

         # No remaining actual arguments
         uplevel [list set args [list]]

         # Done
         return ${%FLAGS%}
      }

      set defaultGiven [lindex $argument 1]
      if {$defaultGiven} {

         # Set the argument in the calling scope,
         # taking the parameter from the default value
         set defaultValue [lindex $argument 3]
         uplevel [list set $name $defaultValue]

      } else {

         # If is flag
         if {[lindex $argument 0]} {
            if {!$defaultGiven} {

               # Set the boolflag in the calling scope: not given
               uplevel [list set $name 0]
            }
         } else {
            ::Simple::Error::throw ::Simple::ExtProc::NO-VALUE-FOR-PARAM $name\
               [::Simple::Proc::caller]
         }
      }

      ### Eat non-used argument
      set argument [lindex $arguments [incr argumentIndex]]
   }

   # If this point is reached, this procedure does not use the
   # special argument "args", so unset it in the calling scope
   uplevel [list unset args]

   # Done
   set %FLAGS%
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information is-subcommand
# -purpose    : Returns whether a procedure is a base subcommand procedure.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : Whether the procedure is a base subcommand procedure.
# -seealso    :
#  * The P<::Simple::Subcommand::information exists> procedure.
#
proc-sub {::Simple::ExtProc::information is-subcommand} {
   procedure
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $procedure

   ### Subcommand procedure
   if {[llength $procedure] > 1} {

      ### Throw error
      ::Simple::Error::throw ::Simple::BAD-ITEM\
         {non-subcommand procedure name} [::Simple::Proc::Priv::restore-name]
   }

   ### Qualify the procedure name
   if {![string match ::* $procedure]} {
      set procedure [::Simple::Namespace::Priv::qualify $procedure]
   }

   ### Return whether the procedure is a base subcommand procedure
   {::Simple::Subcommand::information exists} $procedure
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information subcommands
# -purpose    : Returns a base subcommand procedure subcommand list.
# -arguments  :
#  { command          -name          {Base command name}}
#
# -returns    : The base subcommand procedure subcommand list.
# -seealso    :
#  * The P<::Simple::Subcommand::information subcommands> procedure.
#
# -remarks    :
#  * Procedures containing a percent sign in its name are assumed to be
#    internal to the Simple Library and are excluded from the output.
#
proc-sub {::Simple::ExtProc::information subcommands} {
   command
} {
   ### Save the base command name
   ::Simple::Proc::Priv::save-name $command

   ### Subcommand procedure
   if {[llength $command] > 1} {

      ### Throw error
      ::Simple::Error::throw ::Simple::BAD-ITEM\
         {non-subcommand procedure name} [::Simple::Proc::Priv::restore-name]
   }

   ### Qualify the base command name
   if {![string match ::* $command]} {
      set command [::Simple::Namespace::Priv::qualify $command]
   }

   ### Returns the base subcommand procedure subcommand list
   ::Simple::Subcommand::information subcommands $command
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information runtimeparsing
# -purpose    : Returns whether a procedure run-time parses its arguments.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : Whether the procedure run-time parses its arguments.
# -remarks    :
#  * This procedure returns false for non-existing or non-extended procedures.
#
proc-sub {::Simple::ExtProc::information runtimeparsing} {
   procedure
} {
   ### Parse and qualify the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Get whether the procedure run-time parses its arguments
      set answer [expr {[llength [::Simple::ScriptHead::get [info body\
         $namespace$unqualifiedProcedure] %PRIV%-parse-arguments*]] > 0}]

   ### Non existing or non-extended procedure
   } else {

      ### Return false
      set answer 0
   }

   ### Return whether the procedure run-time parses its arguments
   set answer
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information checktype
# -purpose    : Returns whether a procedure performs argument type checking.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : Whether the procedure performs argument type checking.
# -remarks    :
#  * This procedure returns false for non-existing or non-extended procedures.
#
proc-sub {::Simple::ExtProc::information checktype} {
   procedure
} {
   ### Parse and qualify the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Return whether the procedure performs argument run-time type checking
      set answer [expr {[llength [::Simple::ScriptHead::get [info body\
         $namespace$unqualifiedProcedure]\
         %PRIV%-parse-arguments-checktype]] > 0}]

   ### Non existing or non-extended procedure
   } else {

      ### Return false
      set answer 0
   }

   ### Return whether the procedure performs argument run-time type checking
   set answer
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information interpflags
# -purpose    : Returns whether a procedure has flag interpretation enabled.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : Whether the procedure has flag interpretation enabled.
# -remarks    :
#  * This procedure returns false for non-existing or non-extended procedures.
#
proc-sub {::Simple::ExtProc::information interpflags} {
   procedure
} {
   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Get whether the procedure has flag interpretation enabled
      set answer\
         [set ${namespace}%ExtProcs%(%InterpFlags%,$unqualifiedProcedure)]

   ### Non existing or non-extended procedure
   } else {

      ### Return false
      set answer 0
   }

   ### Return whether the procedure has flag interpretation enabled
   set answer
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information exists
# -purpose    : Returns whether an extended procedure exists.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : Whether there is an extended procedure with that name.
#
proc-sub {::Simple::ExtProc::information exists} {
   procedure
} {
   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Return whether there is an extended procedure with that name
   info exists ${namespace}%ExtProcs%($unqualifiedProcedure)
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information body
# -purpose    : Returns a procedure body.
# -overview   :
#     This command is similar to the K<info body> but works extended
#  procedures as well as for regular procedures.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : The procedure body.
#
proc-sub {::Simple::ExtProc::information body} {
   procedure
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $procedure

   ### Parse and qualify the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Procedure does not exist
   if {[catch {set body [info body $namespace$unqualifiedProcedure]}]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-PROCEDURE\
         [::Simple::Proc::Priv::restore-name]

   ### Procedure exists
   } else {

      ### Delete the run-time argument parsing header, if any
      if {[llength\
         [::Simple::ScriptHead::get $body %PRIV%-parse-arguments*]] != 0} {

         ::Simple::ScriptHead::delete body %PRIV%-parse-arguments*
      }

      ### Return the procedure body
      set body
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information args
# -purpose    : Returns an extended procedure argument names list.
# -overview   :
#     This command is similar to the K<info args> command but works for
#  extended procedures as well as for regular procedures.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : The procedure argument names list.
#
proc-sub {::Simple::ExtProc::information args} {
   procedure
} {
   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Return the extended procedure argument names list
      set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)

   ### Non existing or non-extended procedure
   } else {

      ### Try with a regular procedure
      uplevel [list info args $procedure]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information argsusage
# -purpose    : Returns a procedure argument usage string.
# -returns    : The procedure argument usage string.
# -overview   :
#     This command works for extended procedures as well as for regular
#  procedures.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -remarks    :
#  * The returned string is empty if the procedure has no arguments but its
#    first character is an space otherwise.
#
# -seealso    :
#  * The P<::Simple::Proc::information argsusage> procedure.
#
proc-sub {::Simple::ExtProc::information argsusage} {
   procedure
} {
   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Compose the extended procedure usage string
      set answer {}
      foreach argument [set ${namespace}%ExtProcs%($unqualifiedProcedure)]\
         name [set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)] {

         # Flag
         if {[lindex $argument 0]} {
            set type [lindex $argument 4]
            if {[string compare $type -boolflag]} {
               append answer " ?$name [{::Simple::Type::convert type} $type]?"
            } else {
               append answer " ?$name?"
            }

         # Optional
         } elseif {![string compare [lindex $argument 2] args] ||\
            [lindex $argument 1]} {

            append answer " ?$name?"

         # Non-flag nor optional
         } else {
            append answer " $name"
         }
      }

      ### Return the extended procedure usage string
      set answer

   ### Non existing or non-extended procedure
   } else {

      ### Try with a regular procedure
      uplevel [list {::Simple::Proc::information argsusage} $procedure]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information argslist
# -purpose    : Returns a procedure argument list.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : The procedure argument list.
# -seealso    :
#  * The P<::Simple::Proc::information argslist> procedure.
#
proc-sub {::Simple::ExtProc::information argslist} {
   procedure
} {
   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Compose the extended procedure argument list
      set answer [list]
      foreach argument [set ${namespace}%ExtProcs%($unqualifiedProcedure)]\
         name [set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)] {

         # Flag or optional
         if {![lindex $argument 0] && [lindex $argument 1]} {
            set name ?$name?
         }

         # -choice
         set type [lindex $argument 4]
         if {![string compare $type -choice]} {
            lappend answer [list $name $type [lindex $argument 5]\
               [lindex $argument 6]]

         # Optional
         } elseif {[lindex $argument 1]} {
            lappend answer [list $name $type [lindex $argument 3]\
               [lindex $argument 6]]

         # Otherwise
         } else {
            lappend answer [list $name $type [lindex $argument 6]]
         }
      }

      ### Return the extended procedure argument list
      set answer

   ### Non existing or non-extended procedure
   } else {

      ### Try with a regular procedure
      uplevel [list {::Simple::Proc::information argslist} $procedure]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information type
# -purpose    : Returns an extended procedure argument type.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#  { argument         -name          {Argument name}}
#
# -returns    : The extended procedure argument type.
#
proc-sub {::Simple::ExtProc::information type} {
   procedure
   argument
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $procedure

   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Locate the argument
      set position [lsearch -exact\
         [set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)] $argument]

      ### Non-existing argument
      if {$position == -1} {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-ARGUMENT\
            [::Simple::Proc::Priv::restore-name] $argument
      }

      ### Return the argument type
      lindex [lindex [set ${namespace}%ExtProcs%($unqualifiedProcedure)]\
         $position] 4

   ### Non-extended procedure
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-EXTENDED-PROCEDURE\
         [::Simple::Proc::Priv::restore-name]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information default
# -purpose    : Returns whether a procedure argument has a default value.
# -overview   :
#     This command is similar to the K<info default> command but works for
#  extended procedures as well as for regular procedures.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#  { argument         -name          {Argument name}}
#  { variableName     -name          {Name of the variable in the calling scope
#                                     in which the argument default value is
#                                     to be placed}}
#
#
# -returns    : Whether the procedure argument has a default value.
# -effects    :
#  * If the argument has a default value, the variable named A<variableName>
#    in the calling scope is set to such value.
#
proc-sub {::Simple::ExtProc::information default} {
   procedure
   argument
   variableName
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $procedure

   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Locate the argument
      set position [lsearch -exact\
         [set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)] $argument]

      ### Non-existing argument
      if {$position == -1} {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-ARGUMENT\
            [::Simple::Proc::Priv::restore-name] $argument
      }

      ### The argument has a default value
      if {[lindex [lindex [set ${namespace}%ExtProcs%($unqualifiedProcedure)]\
         $position] 1]} {

         ### Set the variable in the calling scope
         ###    to the argument default value
         uplevel [list set $variableName [lindex [lindex\
            [set ${namespace}%ExtProcs%($unqualifiedProcedure)] $position] 3]]

         ### Return true
         set answer 1

      ### The argument does not have a default value
      } else {

         ### Return false
         set answer 0
      }

      ### Return whether the procedure argument has a default value.
      set answer

   ### Non existing or non-extended procedure
   } else {

      ### Try with a regular procedure
      uplevel [list info default $procedure $argument $variableName]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information choices
# -purpose    : Returns an extended procedure argument choices list.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#  { argument         -name          {Argument name}}
#
# -returns    : The extended procedure argument choices list.
#
proc-sub {::Simple::ExtProc::information choices} {
   procedure
   argument
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $procedure

   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Locate the argument
      set position [lsearch -exact\
         [set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)] $argument]

      ### Non-existing argument
      if {$position == -1} {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-ARGUMENT\
            [::Simple::Proc::Priv::restore-name] $argument
      }

      ### Get the argument type
      set type [lindex [lindex\
         [set ${namespace}%ExtProcs%($unqualifiedProcedure)] $position] 4]

      ### The argument type is not "-choice"
      if {[string compare $type -choice]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::NO-TYPE-CHOICE\
            $argument [::Simple::Proc::Priv::restore-name]

      ### The argument type is "-choice"
      } else {

         ### Return the extended procedure argument choices
         lindex [lindex\
            [set ${namespace}%ExtProcs%($unqualifiedProcedure)] $position] 5
      }

   ### Non-extended procedure
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-EXTENDED-PROCEDURE\
         [::Simple::Proc::Priv::restore-name]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::information description
# -purpose    : Returns an extended procedure argument desription.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#  { argument         -name          {Argument name}}
#
# -returns    : The extended procedure argument desription.
#
proc-sub {::Simple::ExtProc::information description} {
   procedure
   argument
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $procedure

   ### Parse the procedure name
   foreach {namespace unqualifiedProcedure}\
      [::Simple::ExtProc::Priv::parse-qualify-split-name $procedure] break

   ### Extended procedure
   if {[info exists ${namespace}%ExtProcs%($unqualifiedProcedure)]} {

      ### Locate the argument
      set position [lsearch -exact\
         [set ${namespace}%ExtProcs%(%Names%,$unqualifiedProcedure)] $argument]

      ### Non-existing argument
      if {$position == -1} {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-ARGUMENT\
            [::Simple::Proc::Priv::restore-name] $argument
      }

      ### Return the extended procedure argument description
      lindex [lindex\
         [set ${namespace}%ExtProcs%($unqualifiedProcedure)] $position] 6

   ### Non-extended procedure
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-EXTENDED-PROCEDURE\
         [::Simple::Proc::Priv::restore-name]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::number-flags
# -purpose    : Returns the number of flags in an extended procedure call.
# -access     : Export
# -keywords   : procedure argument flag
# -returns    : Number of flags given in the actual argument list.
# -examples   :
#
#  # A procedure with two flags which only
#  # accepts one or the other but not both
#  proc-ext name {
#     {-flag1       -boolflag             "A boolean flag"}
#     {-flag2       -boolflag             "Another boolean flag"}
#  } {
#     if {[number-flags] > 1} {
#        return -code error "Only one flag allowed"
#     }
#  }
#
proc ::Simple::ExtProc::number-flags {
} {
   ### Up one level
   uplevel {

      ### Return the number of flags given in the actual argument list
      llength ${%FLAGS%}
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::flag-given
# -purpose    : Returns whether a flag was given in an extended procedure call.
# -access     : Export
# -arguments  : 
#  { flag             -flag          {Flag}}
#
# -keywords   : procedure argument flag
# -returns    : Whether the flag was given in the actual argument list.
# -examples   :
#
#  # A procedure with a single flag argument which echoes its value
#  # informing on whether that value is the default or was given in the
#  # argument list
#  proc-ext name {
#     {-flag        -string   theflag     "A flag (requires parameter)"}
#  } {
#     puts -nonewline "flag      = <$flag> "
#     if {[flag-given -flag]} {
#        puts "(flag given)"
#     } else {
#        puts "(default value)"
#     }
#  }
#
# -effects    :
#  * Modifies the variable named A<%FLAGS%> in the calling scope.
#
proc ::Simple::ExtProc::flag-given {
   flag
} {
   ### Return whether the flag was given in the actual argument list.
   upvar %FLAGS% flags
   expr {[lsearch -exact $flags $flag] != -1}
}

### ===========================================================================
### -command  : ::Simple::ExtProc::arg-is-default
# -purpose    : Returns whether the value of an argument is the default one.
# -overview   :
#     For flags or optional arguments, this procedure returns whether its
#  actual value matches its corresponding default value. For other arguments
#  the procedure returns always false.
#
# -access     : Export
# -arguments  : 
#  { argument         -name          {Argument}}
#
# -keywords   : procedure argument {default value}
# -returns    : Whether the argument has a default value and it matches its
#  actual value.
#
# -examples   :
#
#  # A procedure with an optional argument whose default value
#  # is rather estrange, value which is used to check whether
#  # the argument was given in the actual argument list
#  proc-ext name {
#     {?argument?   -string %DEFAULT%     "Optional argument"}
#  } {
#     if {[arg-is-default argument]} {
#        puts {argument not given}
#     } else {
#        puts "argument given = $argument"
#     }
#  }
#
# -limitations:
#  * This procedure must be called from another procedure, otherwise there
#    is no argument whose default value to check.
#
proc ::Simple::ExtProc::arg-is-default {
   argument
} {
   ### Get the caller name
   set caller [::Simple::Proc::caller]

   ### Save the caller name
   ::Simple::Proc::Priv::save-name $caller

   ### Assert this procedure has been called from within another
   if {![string compare $caller {}]} {
      ::Simple::Error::throw ::Simple::ExtProc::CANT-BE-CALLED-FROM-TOPLEVEL\
         arg-is-default
   }

   ### Qualify the name of the caller
   if {![string match ::* $caller]} {
      set caller [::Simple::Namespace::Priv::qualify $caller]
   }

   ### The caller is an extended procedure
   if {[{::Simple::ExtProc::information exists} $caller]} {

      ### The argument has a default value
      if {[{::Simple::ExtProc::information default}\
         $caller $argument default]} {

         ### It is a flag
         if {[::Simple::Type::is -flag $argument]} {

            ### Convert the flag to a name
            set argument [{::Simple::Type::convert flag} $argument]
         }

         ### Get whether the value of the argument
         ###    in the calling scope is the default one
         set answer [expr {![string compare $default [uplevel set $argument]]}]

      ### The argument does not have a default value
      } else {

         ### Return false
         set answer 0
      }

   ### The caller is not an extended procedure
   } else {

      ### Error getting the argument default value
      if {[catch {info default $caller $argument default} result]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtProc::NON-EXISTING-ARGUMENT\
            [::Simple::Proc::Priv::restore-name] $argument

      ### Argument default value got
      } else {

         ### The argument has a default value
         if {$result} {

            ### Get whether the value of the argument
            ###    in the calling scope is the default one
            set answer\
               [expr {![string compare $default [uplevel set $argument]]}]

         ### The argument does not have a default value
         } else {

            ### Return false
            set answer 0
         }
      }
   }

   ### Return whether the value of the argument
   ###    in the calling scope is the default one
   set answer
}

### ===========================================================================
### -command  : ::Simple::ScriptHead::add
# -purpose    : Prepends a header to a script.
# -overview   :
#     The header is enclosed between a begin and an end of header markers
#  formatted through the V<MarketFormat> package variable.  Both markers
#  contain the header tag and priority.  Successive headers within a script
#  are ordered by priority which goes from 0 to 9999, the lower the number the
#  higher the priority.  If a header with the same priority as the one being
#  prepended already exist, the old has precedence, that is, the later gets
#  added after the former.
#
# -arguments  :
#  { scriptName       -name          {Script name}}
#  { tag              -string        {Header tag}}
#  { priority         -int           {Header priority}}
#  { header           -script        {Header}}
#
# -returns    : The resulting script.
# -effects    :
#  * Prepends the header to the script called I<scriptName> in the callers
#    scope.
#
proc ::Simple::ScriptHead::add {
   scriptName
   tag
   priority
   header
} {
   ### The script does not exist
   upvar $scriptName script
   if {![info exists script]} {
      
      ### Throw error
      ::Simple::Error::throw ::Simple::ScriptHead::NON-EXISTING-SCRIPT\
         $scriptName
   }

   # Header to prepend
   set header "[format $::Simple::ScriptHead::MarkerFormat %HEADER-BEGIN%\
      $priority $tag]\n$header\n[format $::Simple::ScriptHead::MarkerFormat\
      %HEADER-END% $priority $tag]"

   ### Previous headers present
   if {[regexp -- "^($::Simple::ScriptHead::MarkerStart\
      %HEADER-BEGIN%.*\n$::Simple::ScriptHead::MarkerStart\
      %HEADER-END%\[^\n\]*)(.*)$" $script tmp previousHeaders script]} {
      # $previousHeaders contains all previous headers
      # $script contains the script after the headers

      # Initialise the remaining headers to all previous headers
      set remainingHeaders $previousHeaders

      ### Loop over remaining headers
      while {[regexp -- "$::Simple::ScriptHead::MarkerStart %HEADER-BEGIN%\
         +\[0-9\]+ (\[^\n\]*)(.*)$" $remainingHeaders tmp thisTag\
         remainingHeaders]} {
         # $thisTag contains the tag of this header
         # $remainingHeaders contains this header (but the %HEADER-BEGIN% line)
         #    plus allremaining headers

         ### Check whether the tag is already used
         if {![string compare $thisTag $tag]} {

            # Rebuild the script
            set script $previousHeaders$script

            ### Throw error
            ::Simple::Error::throw ::Simple::ScriptHead::DUPLICATED-TAG\
               $tag $scriptName
         }
      }

      # Initialise the remaining headers to all previous headers
      set remainingHeaders $previousHeaders

      ### Loop over remaining headers
      set lowerPriorityHeaders {}
      while {[regexp -- "($::Simple::ScriptHead::MarkerStart %HEADER-BEGIN%\
         +(\[0-9\]+) (\[^\n\]*))(.*)$" $remainingHeaders\
         thisPlusRemainingHeaders thisHeaderBegin thisPriority\
         thisTag remainingHeaders]} {
         # $thisPlusRemainingHeaders contains this header plus all remaining
         #    headers
         # $thisHeaderBegin contains this header %HEADER-BEGIN% line
         # $thisPriority contains the priority of this header
         # $thisTag contains the tag of this header
         # $remainingHeaders contains this header (but the %HEADER-BEGIN% line)
         #    plus all remaining headers

         ### Header with higher priority found
         if {$thisPriority > $priority} {

            ### Exit the loop
            set remainingHeaders $thisPlusRemainingHeaders
            # $remainingHeaders contains this header plus all remaining headers
            break

         ### Header with lower or equal priority found
         } else {

            ### Accumulate lower priority headers
            regexp -- "^(.*[format $::Simple::ScriptHead::MarkerFormat\
               %HEADER-END% $thisPriority $thisTag])(.*)$"\
               $remainingHeaders tmp thisHeaderEnd remainingHeaders
            # $thisHeaderEnd contains this header (but the %HEADER-BEGIN% line)
            # $remainingHeaders contains the remaining headers
            append lowerPriorityHeaders $thisHeaderBegin$thisHeaderEnd\n
         }
      }

      ### Insert the header in its place
      if {[string compare $remainingHeaders {}]} {
         set script $lowerPriorityHeaders$header\n$remainingHeaders$script
      } else {
         set script $lowerPriorityHeaders$header$script
      }

   ### No previous headers present
   } else {

      ### Prepend the header at the beginning of the script
      set script $header\n$script
   }
}

### ===========================================================================
### -command  : ::Simple::ScriptHead::get
# -purpose    : Gets headers from a script.
# -overview   :
#     This procedure returns the list of headers from a script matching the
#  given pattern.
#
# -arguments  :
#  { script           -script        {Script}}
#  {?pattern?         -pattern {}    {Pattern}}
#
# -returns    : The list of script headers matching the given pattern in the
#  format {tag priority}.
#
# -remarks    :
#  * Use no pattern to get all headers.
#
proc ::Simple::ScriptHead::get {
   script
   {pattern {}}
} {
   ### Previous headers present
   set answer [list]
   if {[regexp -- "^($::Simple::ScriptHead::MarkerStart\
      %HEADER-BEGIN%.*\n$::Simple::ScriptHead::MarkerStart\
      %HEADER-END%\[^\n\]*)" $script tmp remainingHeaders]} {
      # $remainingHeaders contains all previous headers

      ### Loop over remaining headers
      while {[regexp -- "$::Simple::ScriptHead::MarkerStart %HEADER-BEGIN%\
         +(\[0-9\]+) (\[^\n\]*)(.*)$"\
         $remainingHeaders tmp priority tag remainingHeaders]} {
         # $priority contains the priority of this header
         # $tag contains the tag of this header
         # $remainingHeaders contains this header (but the %HEADER-BEGIN% line)
         #    plus all remaining headers

         ### Tag matches pattern
         if {![string compare $pattern {}] || [string match $pattern $tag]} {

            ### Append tag/priority pair
            lappend answer [list $tag $priority]
         }
      }
   }

   ### Return the list of matching headers
   set answer
}

### ===========================================================================
### -command  : ::Simple::ScriptHead::delete
# -purpose    : Deletes headers from a script.
# -overview   :
#     This procedure deletes script headers matching the given pattern.
#
# -arguments  :
#  { scriptName       -name          {Script name}}
#  {?pattern?         -pattern {}    {Pattern}}
#
# -effects    :
#  * Deletes the headers from the script called I<scriptName> in the callers
#    scope.
#
# -returns    : The number of deleted headers.
# -remarks    :
#  * Use no pattern to delete all headers.
#
#  * If no header is deleted, an error is thrown.
#
proc ::Simple::ScriptHead::delete {
   scriptName
   {pattern {}}
} {
   ### The script does not exist
   upvar $scriptName script
   if {![info exists script]} {
      
      ### Throw error
      ::Simple::Error::throw ::Simple::ScriptHead::NON-EXISTING-SCRIPT\
         $scriptName
   }

   ### Previous headers present
   set answer 0
   if {[regexp -- "^($::Simple::ScriptHead::MarkerStart\
      %HEADER-BEGIN%.*\n$::Simple::ScriptHead::MarkerStart\
      %HEADER-END%\[^\n\]*)\n(.*)$" $script tmp remainingHeaders script]} {
      # $remainingHeaders contains all remainingHeaders headers
      # $script contains the script after the headers

      ### Tag pattern given
      set headers {}
      if {[string compare $pattern {}]} {

         ### Loop over remaining headers
         while {[regexp -- "($::Simple::ScriptHead::MarkerStart %HEADER-BEGIN%\
            +(\[0-9\]+) (\[^\n\]*))(.*)$" $remainingHeaders tmp\
            thisHeaderBegin thisPriority thisTag remainingHeaders]} {
            # $thisHeaderBegin contains this header %HEADER-BEGIN% line
            # $thisPriority contains the priority of this header
            # $thisTag contains the tag of this header
            # $remainingHeaders contains this header (but the %HEADER-BEGIN%
            #    line) plus all remaining headers

            ### Extract the header
            regexp -- "^(.*[format $::Simple::ScriptHead::MarkerFormat\
               %HEADER-END% $thisPriority $thisTag])(.*)$"\
               $remainingHeaders tmp thisHeaderEnd remainingHeaders
            # $thisHeaderEnd contains this header (but the %HEADER-BEGIN% line)
            # $remainingHeaders contains the remaining headers

            ### Tag does not match pattern
            if {![string match $pattern $thisTag]} {

               ### Do not delete header
               append headers $thisHeaderBegin$thisHeaderEnd\n

            ### Tag matches pattern
            } else {

               ### Delete header
               incr answer
            }
         }
      }

      ### Build the script
      set script $headers$script
   }

   ### No header deleted
   if {$answer == 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::ScriptHead::NON-EXISTING-HEADER\
         $pattern $scriptName
   }

   ### Return the number of deleted headers
   set answer
}

### ===========================================================================
### -command  : ::Simple::ExtProc::configure
# -purpose    : Configures the package options.
#
# -arguments  :
#  {-checktype        -extbool false {Whether to perform run-time type checking
#                                     of the procedures arguments}}
#  {-afterflags       -int 2         {Minimum number of arguments after the 
#                                     last flag for a procedure in order for
#                                     those are handled with an auxiliary 
#                                     procedure}}
#
proc ::Simple::ExtProc::configure {
   {arg1 {}}
   {arg2 {}}
} {
   ### No arguments
   if {![string compare $arg1 {}]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::AT-LEAST-ONE-OPTION\
         [::Simple::Proc::Priv::enumerate [::Simple::ExtProc::cget]]

   } else {

      ### Switch on flag
      switch -exact -- $arg1 {

         -checktype {
         ### Handle -checktype

            ### Invalid argument
            if {![::Simple::Type::is -extbool $arg2]} {

               ### Throw error
               ::Simple::Error::throw ::Simple::BAD-OPTION-ARG-TYPE\
                  $arg2 -checktype -extbool
            }

            ### Set the check type option
            set ::Simple::ExtProc::CheckType\
               [{::Simple::Type::convert boolean} $arg2]
         }

         -afterflags {
         ### Handle -afterflags

            ### Invalid argument
            if {![::Simple::Type::is -int $arg2]} {

               ### Throw error
               ::Simple::Error::throw ::Simple::BAD-OPTION-ARG-TYPE\
                  $arg2 -afterflags -int
            }
            if {$arg2 <= 1} {
               ::Simple::Error::throw ::Simple::BAD-OPTION-ARG-VALUE\
                  $arg2 -afterflags {greater than one}
            }

            ### Set the after flags option
            set ::Simple::ExtProc::AfterFlags $arg2
         }

         default {
         ### default, throw error
            ::Simple::Error::throw ::Simple::BAD-OPTION\
               $arg1 [::Simple::Proc::Priv::enumerate [::Simple::ExtProc::cget]]
         }
      }
   }
}

### ===========================================================================
### -command  : ::Simple::ExtProc::cget
# -purpose    : Gets the package options.
# -arguments  :
#  {-checktype        -boolflag      {Gets whether to perform run-time type
#                                     checking of the procedures arguments}}
#  {-afterflags       -boolflag      {Gets the minimum number of arguments
#                                     after the last flag for a procedure in
#                                     order for those are handled with an
#                                     auxiliary procedure}}
#
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::ExtProc::cget {
   {arg1 {}}
} {
   ### No arguments
   if {![string compare $arg1 {}]} {

      ### Return the list of options
      return {-checktype -afterflags}

   } else {

      ### Switch on flag
      switch -exact -- $arg1 {

         -afterflags {
         ### Handle -afterflags
            set ::Simple::ExtProc::AfterFlags
         }

         -checktype {
         ### Handle -checktype
            set ::Simple::ExtProc::CheckType
         }

         default {
         ### default, throw error
            ::Simple::Error::throw ::Simple::BAD-OPTION\
               $arg1 [::Simple::Proc::Priv::enumerate [::Simple::ExtProc::cget]]
         }
      }
   }
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleExtProc-test 0.2

### ===========================================================================
### -test     : ::Simple::ExtProc::next-arg
test-case next-arg-1 {
   ::Simple::ExtProc::next-arg
} -setup {

   # Create two variables
   set arg1 arg1
   set arg2 arg2

} -script {
   puts -nonewline [next-arg][next-arg]
} -cleanup {

   # Delete the variables
   unset arg1 arg2

   # Delete the argument counter variable
   unset %ARG-NUMBER%

} -output arg1arg2

### ===========================================================================
### -test     : ::Simple::ExtProc::previous-arg
test-case previous-arg-1 {
   ::Simple::ExtProc::previous-arg
} -setup {

   # Create a variable
   set arg1 arg1

} -script {
   puts -nonewline [next-arg]
   previous-arg
   puts -nonewline [next-arg]
} -cleanup {

   # Delete the variable
   unset arg1

   # Delete the argument counter variable
   unset %ARG-NUMBER%

} -output arg1arg1

### ===========================================================================
### -test     : ::Simple::ExtProc::create
test-case create-1 {
   ::Simple::ExtProc::create
} -setup {

   # Create two namespaces
   namespace eval ::foo {}
   namespace eval ::bar {}

} -script {

   # Create a hierarchy of procedures
   proc-ext ::foo::gee {} {
      puts [namespace current]
      puts [uplevel namespace current]
      puts [info level]
      puts [uplevel info level]
      puts [info level 0]
      puts [uplevel info level 0]
   }
   proc ::bar::zoo {} {
      ::foo::gee
   }

   # Call the hierarchy of procedures
   ::bar::zoo

} -cleanup {

   # Delete the procedures
   rename ::foo::gee {}
   rename ::bar::zoo {}

   # Delete one namespace
   namespace delete ::bar

} -output {::foo
::bar
2
1
::foo::gee
::bar::zoo
}

test-case create-2 {
   ::Simple::ExtProc::create
   flag interpretation disabled
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -setup {

   # Create a procedure which displays its arguments 
   proc display-args args {
      puts ---------------------
      foreach arg $args {
         set command "set $arg"
         puts "$arg = \{[uplevel $command]\}"
      }
   }

} -script {

   # Create a procedure
   proc-ext -interpflags false ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {display-args arg1 arg2 arg3 arg4 arg5 arg6}

   # Valid calls
   ::foo::bar
   ::foo::bar -boolflag -flag 2.0 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
arg1 = {%DEFAULT%}
arg2 = {%DEFAULT%}
arg3 = {%DEFAULT%}
arg4 = {%DEFAULT%}
arg5 = {%DEFAULT%}
arg6 = {%DEFAULT%}
---------------------
arg1 = {-boolflag}
arg2 = {-flag}
arg3 = {2.0}
arg4 = {1}
arg5 = {2}
arg6 = {3}
}

test-case create-3 {
   ::Simple::ExtProc::create
   no flags
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   namespace eval ::foo {
      proc-ext bar {
         { int1             -int           "First argument"}
         { int2             -int           "Second argument"}
         { int3             -int           "Third argument"}
      } {display-args int1 int2 int3}
   }

   # Valid calls
   ::foo::bar 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case create-4 {
   ::Simple::ExtProc::create
   no flags, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   namespace eval ::foo {

      # Create a procedure
      proc-ext -checktype true bar {
         { int1             -int           "First argument"}
         { int2             -int           "Second argument"}
         { int3             -int           "Third argument"}
      } {display-args int1 int2 int3}

      # Valid calls
      bar 1 2 3

      # Invalid calls
      bar 1 2 gee
  }

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid value "gee" for argument "int3" of type "-int"}

test-case create-5 {
   ::Simple::ExtProc::create
   no flags, optional
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
   } {display-args int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 2

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
int2 = {2}
int3 = {999}
}

test-case create-6 {
   ::Simple::ExtProc::create
   no flags, optional, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
   } {display-args int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 2

   # Invalid calls
   catch {::foo::bar 1 bar} result
   puts $result
   ::foo::bar 1 2 bar

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
int2 = {2}
int3 = {999}
invalid value "bar" for argument "int2" of type "-int"
} -error {invalid value "bar" for argument "int3" of type "-int"}

test-case create-7 {
   ::Simple::ExtProc::create
   no flags, args
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 int2 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
args = {}
---------------------
int1 = {1}
int2 = {2}
args = {3}
---------------------
int1 = {1}
int2 = {2}
args = {3 4 5}
}

test-case create-8 {
   ::Simple::ExtProc::create
   no flags, args, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 int2 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5

   # Invalid calls
   ::foo::bar 1 gee 3 4 5

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
args = {}
---------------------
int1 = {1}
int2 = {2}
args = {3}
---------------------
int1 = {1}
int2 = {2}
args = {3 4 5}
} -error {invalid value "gee" for argument "int2" of type "-int"}

test-case create-9 {
   ::Simple::ExtProc::create
   no flags, optional, args
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 int2 int3 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5 6

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
}

test-case create-10 {
   ::Simple::ExtProc::create
   no flags, optional, args, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 int2 int3 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5 6

   # Invalid calls
   catch {::foo::bar 1 bar 3 4 5 6} result
   puts $result
   ::foo::bar 1 2 bar 4 5 6

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
invalid value "bar" for argument "int2" of type "-int"
} -error {invalid value "bar" for argument "int3" of type "-int"}

test-case create-11 {
   ::Simple::ExtProc::create
   two flags, beginning, -boolflag first
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {display-args boolflag flag int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -boolflag -flag 2.0 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case create-12 {
   ::Simple::ExtProc::create
   two flags, beginning, -boolflag second
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      {-flag             -float 1.0     "A flag"}
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {display-args flag boolflag int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -flag 2.0 -boolflag 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
flag = {1.0}
boolflag = {0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
flag = {2.0}
boolflag = {0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
flag = {1.0}
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
flag = {2.0}
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case create-13 {
   ::Simple::ExtProc::create
   two flags, beginning, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {display-args boolflag flag int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -boolflag -flag 2.0 1 2 3

   # Invalid calls
   catch {::foo::bar -boolflag -flag bar 1 2 3} result
   puts $result
   ::foo::bar -boolflag -flag 2.0 1 bar 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
invalid value "bar" for argument "flag" of type "-float"
} -error {invalid value "bar" for argument "int2" of type "-int"}

test-case create-14 {
   ::Simple::ExtProc::create
   two flags, beginning, optional
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
   } {display-args boolflag flag int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar -boolflag 1 2
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -flag 2.0 1 2
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -boolflag -flag 2.0 1 2
   ::foo::bar -boolflag -flag 2.0 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case create-15 {
   ::Simple::ExtProc::create
   two flags, beginning, optional, checktype
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
   } {display-args boolflag flag int1 int2 int3}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar -boolflag 1 2
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -flag 2.0 1 2
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -boolflag -flag 2.0 1 2
   ::foo::bar -boolflag -flag 2.0 1 2 3

   # Invalid calls
   catch {::foo::bar -boolflag -flag bar 1 2 3} result
   puts $result
   catch {::foo::bar -boolflag -flag 2.0 1 bar 3} result
   puts $result
   ::foo::bar -boolflag -flag 2.0 1 2 bar

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
invalid value "bar" for argument "flag" of type "-float"
invalid value "bar" for argument "int2" of type "-int"
} -error {invalid value "bar" for argument "int3" of type "-int"}

test-case create-16 {
   ::Simple::ExtProc::create
   two flags, beginning, args
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { args             -any           "Remaining arguments"}
   } {display-args boolflag flag int1 int2 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3 4 5
   ::foo::bar -boolflag 1 2
   ::foo::bar -boolflag 1 2 3 4 5
   ::foo::bar -flag 2.0 1 2
   ::foo::bar -flag 2.0 1 2 3 4 5
   ::foo::bar -boolflag -flag 2.0 1 2
   ::foo::bar -boolflag -flag 2.0 1 2 3 4 5

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
}

test-case create-17 {
   ::Simple::ExtProc::create
   two flags, beginning, args, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { args             -any           "Remaining arguments"}
   } {display-args boolflag flag int1 int2 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3 4 5
   ::foo::bar -boolflag 1 2
   ::foo::bar -boolflag 1 2 3 4 5
   ::foo::bar -flag 2.0 1 2
   ::foo::bar -flag 2.0 1 2 3 4 5
   ::foo::bar -boolflag -flag 2.0 1 2
   ::foo::bar -boolflag -flag 2.0 1 2 3 4 5

   # Invalid calls
   catch {::foo::bar -boolflag -flag bar 1 2 3 4 5} result
   puts $result
   ::foo::bar -boolflag -flag 2.0 1 bar 3 4 5

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
args = {3 4 5}
invalid value "bar" for argument "flag" of type "-float"
} -error {invalid value "bar" for argument "int2" of type "-int"}

test-case create-18 {
   ::Simple::ExtProc::create
   two flags, beginning, optional, args
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args boolflag flag int1 int2 int3 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5 6
   ::foo::bar -boolflag 1 2
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -boolflag 1 2 3 4 5 6
   ::foo::bar -flag 2.0 1 2
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -flag 2.0 1 2 3 4 5 6
   ::foo::bar -boolflag -flag 2.0 1 2
   ::foo::bar -boolflag -flag 2.0 1 2 3
   ::foo::bar -boolflag -flag 2.0 1 2 3 4 5 6

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
}

test-case create-19 {
   ::Simple::ExtProc::create
   two flags, beginning, optional, args, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args boolflag flag int1 int2 int3 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5 6
   ::foo::bar -boolflag 1 2
   ::foo::bar -boolflag 1 2 3
   ::foo::bar -boolflag 1 2 3 4 5 6
   ::foo::bar -flag 2.0 1 2
   ::foo::bar -flag 2.0 1 2 3
   ::foo::bar -flag 2.0 1 2 3 4 5 6
   ::foo::bar -boolflag -flag 2.0 1 2
   ::foo::bar -boolflag -flag 2.0 1 2 3
   ::foo::bar -boolflag -flag 2.0 1 2 3 4 5 6

   # Invalid calls
   catch {::foo::bar -boolflag -flag bar 1 2 3 4 5 6} result
   puts $result
   catch {::foo::bar -boolflag -flag 2.0 1 bar 3 4 5 6} result
   puts $result
   ::foo::bar -boolflag -flag 2.0 1 2 bar 4 5 6

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {0}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {1}
flag = {1.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {0}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {999}
args = {}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {}
---------------------
boolflag = {1}
flag = {2.0}
int1 = {1}
int2 = {2}
int3 = {3}
args = {4 5 6}
invalid value "bar" for argument "flag" of type "-float"
invalid value "bar" for argument "int2" of type "-int"
} -error {invalid value "bar" for argument "int3" of type "-int"}

test-case create-20 {
   ::Simple::ExtProc::create
   two flags, middle, -boolflag first
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {display-args int1 boolflag flag int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
}

test-case create-21 {
   ::Simple::ExtProc::create
   two flags, middle, -boolflag second
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {-flag             -float 1.0     "A flag"}
      {-boolflag         -boolflag      "A boolflag"}
      { int3             -int           "Third argument"}
   } {display-args int1 int2 flag boolflag int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 2 -flag 2.0 3
   ::foo::bar 1 2 -boolflag 3
   ::foo::bar 1 2 -flag 2.0 -boolflag 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
flag = {1.0}
boolflag = {0}
int3 = {3}
---------------------
int1 = {1}
int2 = {2}
flag = {2.0}
boolflag = {0}
int3 = {3}
---------------------
int1 = {1}
int2 = {2}
flag = {1.0}
boolflag = {1}
int3 = {3}
---------------------
int1 = {1}
int2 = {2}
flag = {2.0}
boolflag = {1}
int3 = {3}
}

test-case create-22 {
   ::Simple::ExtProc::create
   two flags, middle, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {display-args int1 boolflag flag int2 int3}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2 3

   # Invalid calls
   catch {::foo::bar 1 -boolflag -flag bar 2 3} result
   puts $result
   ::foo::bar 1 -boolflag -flag 2.0 bar 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
invalid value "bar" for argument "flag" of type "-float"
} -error {invalid value "bar" for argument "int2" of type "-int"}

test-case create-23 {
   ::Simple::ExtProc::create
   two flags, middle, optional
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
   } {display-args int1 boolflag flag int2 int3}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 -boolflag 2
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -flag 2.0 2
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2
   ::foo::bar 1 -boolflag -flag 2.0 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
}

test-case create-24 {
   ::Simple::ExtProc::create
   two flags, middle, optional, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
   } {display-args int1 boolflag flag int2 int3}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 -boolflag 2
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -flag 2.0 2
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2
   ::foo::bar 1 -boolflag -flag 2.0 2 3

   # Invalid calls
   catch {::foo::bar 1 -boolflag -flag bar 2 3} result
   puts $result
   catch {::foo::bar 1 -boolflag -flag 2.0 bar 3} result
   puts $result
   ::foo::bar 1 -boolflag -flag 2.0 2 bar

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {999}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
invalid value "bar" for argument "flag" of type "-float"
invalid value "bar" for argument "int2" of type "-int"
} -error {invalid value "bar" for argument "int3" of type "-int"}

test-case create-25 {
   ::Simple::ExtProc::create
   two flags, middle, args
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 boolflag flag int2 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3 4 5
   ::foo::bar 1 -boolflag 2
   ::foo::bar 1 -boolflag 2 3 4 5
   ::foo::bar 1 -flag 2.0 2
   ::foo::bar 1 -flag 2.0 2 3 4 5
   ::foo::bar 1 -boolflag -flag 2.0 2
   ::foo::bar 1 -boolflag -flag 2.0 2 3 4 5

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
args = {3 4 5}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
args = {3 4 5}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
args = {3 4 5}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
args = {3 4 5}
}

test-case create-26 {
   ::Simple::ExtProc::create
   two flags, middle, args, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 boolflag flag int2 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3 4 5
   ::foo::bar 1 -boolflag 2
   ::foo::bar 1 -boolflag 2 3 4 5
   ::foo::bar 1 -flag 2.0 2
   ::foo::bar 1 -flag 2.0 2 3 4 5
   ::foo::bar 1 -boolflag -flag 2.0 2
   ::foo::bar 1 -boolflag -flag 2.0 2 3 4 5

   # Invalid calls
   catch {::foo::bar 1 -boolflag -flag bar 2 3 4 5} result
   puts $result
   ::foo::bar 1 -boolflag -flag 2.0 bar 3 4 5

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
args = {3 4 5}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
args = {3 4 5}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
args = {3 4 5}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
args = {3 4 5}
invalid value "bar" for argument "flag" of type "-float"
} -error {invalid value "bar" for argument "int2" of type "-int"}

test-case create-27 {
   ::Simple::ExtProc::create
   two flags, middle, optional, args
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 boolflag flag int2 int3 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5 6
   ::foo::bar 1 -boolflag 2
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -boolflag 2 3 4 5 6
   ::foo::bar 1 -flag 2.0 2
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -flag 2.0 2 3 4 5 6
   ::foo::bar 1 -boolflag -flag 2.0 2
   ::foo::bar 1 -boolflag -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2 3 4 5 6

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
}

test-case create-28 {
   ::Simple::ExtProc::create
   two flags, middle, optional, args, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 boolflag flag int2 int3 args}

   # Valid calls
   ::foo::bar 1 2
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 4 5 6
   ::foo::bar 1 -boolflag 2
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -boolflag 2 3 4 5 6
   ::foo::bar 1 -flag 2.0 2
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -flag 2.0 2 3 4 5 6
   ::foo::bar 1 -boolflag -flag 2.0 2
   ::foo::bar 1 -boolflag -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2 3 4 5 6

   # Invalid calls
   catch {::foo::bar 1 -boolflag -flag bar 2 3 4 5 6} result
   puts $result
   catch {::foo::bar 1 -boolflag -flag 2.0 bar 3 4 5 6} result
   puts $result
   ::foo::bar 1 -boolflag -flag 2.0 2 bar 4 5 6

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {1.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {0}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {999}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {}
---------------------
int1 = {1}
boolflag = {1}
flag = {2.0}
int2 = {2}
int3 = {3}
args = {4 5 6}
invalid value "bar" for argument "flag" of type "-float"
invalid value "bar" for argument "int2" of type "-int"
} -error {invalid value "bar" for argument "int3" of type "-int"}

test-case create-29 {
   ::Simple::ExtProc::create
   two flags, end
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
   } {display-args int1 int2 int3 boolflag flag}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 -boolflag
   ::foo::bar 1 2 3 -flag 2.0
   ::foo::bar 1 2 3 -boolflag -flag 2.0

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {0}
flag = {1.0}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {1}
flag = {1.0}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {0}
flag = {2.0}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {1}
flag = {2.0}
}

test-case create-30 {
   ::Simple::ExtProc::create
   two flags, end, checktype
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
   } {display-args int1 int2 int3 boolflag flag}

   # Valid calls
   ::foo::bar 1 2 3
   ::foo::bar 1 2 3 -boolflag
   ::foo::bar 1 2 3 -flag 2.0
   ::foo::bar 1 2 3 -boolflag -flag 2.0

   # Invalid calls
   catch {::foo::bar 1 2 3 -boolflag -flag bar} result
   puts $result
   ::foo::bar 1 bar 3 -boolflag -flag 2.0

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {0}
flag = {1.0}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {1}
flag = {1.0}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {0}
flag = {2.0}
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
boolflag = {1}
flag = {2.0}
invalid value "bar" for argument "flag" of type "-float"
} -error {invalid value "bar" for argument "int2" of type "-int"}

test-case create-31 {
   ::Simple::ExtProc::create
   subcommand, no arguments
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Create a procedure
   proc-ext {::foo::bar subcommand-1} {} {}

   # Subcommands lists
   puts [::Simple::Subcommand::information subcommands ::foo::bar]

   # Invalid calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   ::foo::bar subcommand-1
   catch {::foo::bar subcommand-1 arg1} result
   puts $result

} -output {subcommand-1
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1
wrong # args: should be "::foo::bar subcommand-1"
}

test-case create-32 {
   ::Simple::ExtProc::create
   subcommand, arguments
} -prerequisites {
   {![string compare [info args display-args] args]}
   {[lsearch -exact [namespace children] ::foo] != -1}
   {[{::Simple::Subcommand::information exists} ::foo::bar] == 1}
} -script {

   # Create a procedure
   proc-ext {::foo::bar subcommand-2} {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      {?int3?            -int 999       "Third argument"}
      { args             -any           "Remaining arguments"}
   } {display-args int1 boolflag flag int2 int3 args}

   # Subcommands lists
   puts [::Simple::Subcommand::information subcommands ::foo::bar]

   # Invalid calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   catch {::foo::bar subcommand-2} result
   puts $result

} -cleanup {

   # Delete the subcommands
   ::Simple::ExtProc::delete {::foo::bar subcommand-1}
   ::Simple::ExtProc::delete {::foo::bar subcommand-2}

   # Delete the namespace
   namespace delete ::foo

} -output {subcommand-1 subcommand-2
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1 or subcommand-2
wrong # args: should be "::foo::bar subcommand-2 int1 ?-boolflag? ?-flag float?\
int2 ?int3? ?args?"
}

test-case create-33 {
   ::Simple::ExtProc::create
   deleting a namespace deletes the extended procedures therein
} -setup {

   # Create a procedure in a namespace
   namespace eval ::foo {
      proc-ext bar {} {}
   }

} -script {

   # The extended procedure exists
   puts -nonewline [::Simple::ExtProc::information exists ::foo::bar]

   # Delete the namespace
   namespace delete ::foo

   # The extended procedure no longer exists
   puts -nonewline [::Simple::ExtProc::information exists ::foo::bar]

} -output 10

test-case create-34 {
   ::Simple::ExtProc::create
   error, -interpflags false and -checktype true
} -script {
   proc-ext -interpflags false -checktype true foo {} {}
} -error {can not have "-interpflags false" and argument run-time type checking}

test-case create-35 {
   ::Simple::ExtProc::create
   error, bad option
} -script {
   proc-ext -foo true foo {} {}
} -error {bad option "-foo": must be -checktype or -interpflags}

test-case create-36 {
   ::Simple::ExtProc::create
   error, bad procedure name
} -script {
   catch {proc-ext @foo {} {}} result
   puts $result
   proc-ext {} {} {}
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case create-37 {
   ::Simple::ExtProc::create
   error, invalid argument name
} -script {
   proc-ext foo {
      { ::foo::int1      -int           "First argument"}
   } {}
} -error {expected argument name but got "::foo::int1"}

test-case create-38 {
   ::Simple::ExtProc::create
   error, args must be the last argument
} -script {
   proc-ext foo {
      { args             -any           "Remaining arguments"}
      { int1             -int           "First argument"}
   } {}
} -error {"args" must be the last argument}

test-case create-39 {
   ::Simple::ExtProc::create
   error, non-optional argument after optional
} -script {
   proc-ext foo {
      {?int1?            -int 999       "First argument"}
      { int2             -int           "Second argument"}
   } {}
} -error {non-optional argument "int2" after at least one optional argument}

test-case create-40 {
   ::Simple::ExtProc::create
   error, can not have default value
} -script {
   proc-ext foo {
      { int1             -int 999       "First argument"}
   } {}
} -error {can not have a default value for argument "int1"}

test-case create-41 {
   ::Simple::ExtProc::create
   error, args not allowed if -interpflags false
} -script {
   proc-ext -interpflags false foo {
      { args             -any           "Remaining arguments"}
   } {}
} -error {"args" is not allowed if flag interpretation is disabled} 

test-case create-42 {
   ::Simple::ExtProc::create
   error, args must be of type -list or -any
} -script {
   proc-ext foo {
      { args             -string        "Remaining arguments"}
   } {}
} -error {"args" must be of type "-list" or "-any"}

test-case create-43 {
   ::Simple::ExtProc::create
   error, -interpflags false with no flags
} -script {
   proc-ext -interpflags false foo {} {}
} -error {flag interpretation disabled but no flags present}

test-case create-44 {
   ::Simple::ExtProc::create
   error, unknown namespace
} -script {
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {puts -nonewline [number-flags]}
} -error {can't create procedure "::foo::bar": unknown namespace}

test-case create-45 {
   ::Simple::ExtProc::create
   error, subcommand, bad command name
} -script {
   catch {proc-ext {@foo {}} {} {}} result
   puts $result
   proc-ext {{} {}} {} {}
} -output {expected subcommand base command name but got "@foo"
} -error {expected subcommand base command name but got ""}

test-case create-46 {
   ::Simple::ExtProc::create
   error, subcommand, bad subcommand name
} -script {
   catch {proc-ext {foo @foo} {} {}} result
   puts $result
   proc-ext {foo {}} {} {}
} -output {expected subcommand name but got "@foo"
} -error {expected subcommand name but got ""}

test-case create-47 {
   ::Simple::ExtProc::create
   error, subcommand, existing base command
} -script {
   proc foo {} {}
   proc-ext {foo bar} {} {}
} -error {can't create subcommand "foo bar": procedure "foo" already exists}

test-case create-48 {
   ::Simple::ExtProc::create
   error, procedure call, too many arguments
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc-ext -checktype false ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
   } {}
   proc-ext -checktype false ::foo::gee {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}

} -script {
   catch {::foo::bar 1 2} result
   puts $result
   namespace eval ::foo {
      catch {bar 1 2} result
   }
   puts $result
   catch {::foo::gee 1 2 3 4} result
   puts $result
   namespace eval ::foo {
      gee 1 2 3 4
   }
} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::foo::bar
   ::Simple::ExtProc::delete ::foo::gee

   # Delete the namespace
   namespace delete ::foo

} -output {called "::foo::bar" with too many arguments
called "bar" with too many arguments
called "::foo::gee" with too many arguments
} -error {called "gee" with too many arguments}

test-case create-49 {
   ::Simple::ExtProc::create
   error, procedure call, no value given for parameter
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc-ext -checktype false ::foo::bar {
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
   } {}
   proc-ext -checktype true ::foo::gee {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}

} -script {
   catch {::foo::bar} result
   puts $result
   namespace eval ::foo {
      catch {bar} result
      puts $result
   }
   catch {::foo::gee} result
   puts $result
   namespace eval ::foo {
      gee
   }
} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::foo::bar
   ::Simple::ExtProc::delete ::foo::gee

   # Delete the namespace
   namespace delete ::foo

} -output {no value given for parameter "int1" to "::foo::bar"
no value given for parameter "int1" to "bar"
no value given for parameter "int1" to "::foo::gee"
} -error {no value given for parameter "int1" to "gee"}

test-case create-50 {
   ::Simple::ExtProc::create
   error, procedure call, no parameter given for flag
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure
   proc-ext -checktype false ::foo::bar {
      {-flag             -float 1.0     "A flag"}
   } {}

} -script {
   catch {::foo::bar -flag} result
   puts $result
   namespace eval ::foo {
      bar -flag
   }
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

   # Delete the namespace
   namespace delete ::foo

} -output {no parameter given for flag "-flag" to "::foo::bar"
} -error {no parameter given for flag "-flag" to "bar"}

test-case create-51 {
   ::Simple::ExtProc::create
   error, procedure call, too many arguments, checktype
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc-ext -checktype true ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
   } {}
   proc-ext -checktype true ::foo::gee {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}
} -script {
   catch {::foo::bar} result
   puts $result
   namespace eval ::foo {
      catch {bar} result
      puts $result
   }
   catch {::foo::gee} result
   puts $result
   namespace eval ::foo {
      gee
   }
} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::foo::bar
   ::Simple::ExtProc::delete ::foo::gee

   # Delete the namespace
   namespace delete ::foo

} -output {no value given for parameter "int1" to "::foo::bar"
no value given for parameter "int1" to "bar"
no value given for parameter "int1" to "::foo::gee"
} -error {no value given for parameter "int1" to "gee"}

test-case create-52 {
   ::Simple::ExtProc::create
   error, procedure call, no value given for parameter, checktype
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
   } {}

} -script {
   catch {::foo::bar} result
   puts $result
   namespace eval ::foo {
      bar
   }
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

   # Delete the namespace
   namespace delete ::foo

} -output {no value given for parameter "int1" to "::foo::bar"
} -error {no value given for parameter "int1" to "bar"}

test-case create-53 {
   ::Simple::ExtProc::create
   error, procedure call, no parameter given for flag, checktype
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure
   proc-ext -checktype true ::foo::bar {
      {-flag             -float 1.0     "A flag"}
   } {}

} -script {
   catch {::foo::bar -flag} result
   puts $result
   namespace eval ::foo {
      bar -flag
   }
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

   # Delete the namespace
   namespace delete ::foo

} -output {no parameter given for flag "-flag" to "::foo::bar"
} -error {no parameter given for flag "-flag" to "bar"}

test-case create-54 {
   ::Simple::ExtProc::create
   error, bad option
} -script {
   proc-ext -foo
} -error {bad option "-foo": must be -checktype or -interpflags}

test-case create-55 {
   ::Simple::ExtProc::create
   error, too many arguments
} -regexp -script {
   proc-ext procedure arguments body foo
} -error {called "(proc-ext|::Simple::ExtProc::create)" with too many arguments}

test-case create-56 {
   ::Simple::ExtProc::create
   error, too few arguments
} -regexp -script {
   proc-ext procedure body
} -error {no value given for parameter "body" to\
"(proc-ext|::Simple::ExtProc::create)"}

### ===========================================================================
### -test     : ::Simple::ExtProc::delete
test-case delete-1 {
   ::Simple::ExtProc::delete
   regular procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a regular procedure
   proc ::foo::bar {
      int1
      int2
      int3
   } {display-args int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar 1 2 3

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

   # Invalid call
   ::foo::bar 1 2 3

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid command name "::foo::bar"}

test-case delete-2 {
   ::Simple::ExtProc::delete
   extended procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create an extended procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {display-args boolflag int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar 1 2 3

   # Delete the procedure
   namespace eval ::foo {
      ::Simple::ExtProc::delete bar
   }

   # Invalid call
   ::foo::bar 1 2 3

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {---------------------
boolflag = {0}
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid command name "::foo::bar"}

test-case delete-3 {
   ::Simple::ExtProc::delete
   subcommand procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a subcommand procedure
   proc-ext {::foo::bar gee} {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {display-args boolflag int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar gee 1 2 3

   # Delete the procedure
   namespace eval ::foo {
      ::Simple::ExtProc::delete {bar gee}
   }

   # Invalid call
   ::foo::bar gee 1 2 3

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {---------------------
boolflag = {0}
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid command name "::foo::bar"}

test-case delete-4 {
   ::Simple::ExtProc::delete
   error, non-existing procedure
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] == -1}
} -script {

   # Failed procedure deletions
   catch {::Simple::ExtProc::delete bar} result
   puts $result
   catch {::Simple::ExtProc::delete ::foo::bar} result
   puts $result
   catch {::Simple::ExtProc::delete {::foo::bar gee}} result
   puts $result

   # Ensure the namespace ::foo has not been created
   puts [lsearch -exact [namespace children] ::foo]

   # Failed procedure deletions
   namespace eval ::foo {
      catch {::Simple::ExtProc::delete bar} result
      puts $result
      ::Simple::ExtProc::delete {bar gee}
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {can't delete "bar": command doesn't exist
can't delete "::foo::bar": command doesn't exist
can't delete "::foo::bar gee": command doesn't exist
-1
can't delete "bar": command doesn't exist
} -error {can't delete "bar gee": command doesn't exist}

### ===========================================================================
### -test     : ::Simple::ExtProc::move
test-case move-1 {
   ::Simple::ExtProc::move
   regular procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a regular procedure
   proc ::foo::bar {
      int1
      int2
      int3
   } {display-args int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar 1 2 3

   # Invalid call
   catch {::bar::foo 1 2 3} result
   puts $result

   # Rename the procedure
   namespace eval ::foo {
      ::Simple::ExtProc::move bar ::bar::foo
   }

   # Valid call
   ::bar::foo 1 2 3

   # Invalid call
   ::foo::bar 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::bar::foo

   # Delete the namespaces
   namespace delete ::foo ::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
invalid command name "::bar::foo"
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid command name "::foo::bar"}

test-case move-2 {
   ::Simple::ExtProc::move
   extended procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create an extended procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {display-args boolflag int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar -boolflag 1 2 3

   # Invalid call
   catch {::bar::foo -boolflag 1 2 3} result
   puts $result

   # Rename the procedure
   namespace eval ::foo {
      ::Simple::ExtProc::move bar ::bar::foo
   }

   # Valid call
   ::bar::foo -boolflag 1 2 3

   # Invalid call
   ::foo::bar -boolflag 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::bar::foo

   # Delete the namespaces
   namespace delete ::foo ::bar

} -output {---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
invalid command name "::bar::foo"
---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid command name "::foo::bar"}

test-case move-3 {
   ::Simple::ExtProc::delete
   subcommand procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a subcommand procedure
   proc-ext {::foo::bar gee} {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {display-args boolflag int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar gee -boolflag 1 2 3

   # Invalid call
   catch {::bar::foo zii -boolglaf 1 2 3} result
   puts $result

   # Rename the procedure
   ::Simple::ExtProc::move {::foo::bar gee} {::bar::foo zii}

   # Valid call
   ::bar::foo zii -boolflag 1 2 3

   # Invalid call
   ::foo::bar gee -boolflag 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::bar::foo

   # Delete the namespaces
   namespace delete ::foo ::bar

} -output {---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
invalid command name "::bar::foo"
---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
} -error {invalid command name "::foo::bar"}

test-case move-4 {
   ::Simple::ExtProc::move
   error, non-existing procedure
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] == -1}
} -script {

   # Failed procedure renamings
   catch {::Simple::ExtProc::move bar foo} result
   puts $result
   catch {::Simple::ExtProc::move ::foo::bar ::bar::foo} result
   puts $result
   catch {::Simple::ExtProc::move {::foo::bar gee} {::bar::foo zii}} result
   puts $result

   # Ensure the namespace ::foo has not been created
   puts [lsearch -exact [namespace children] ::foo]

   # Failed procedure renamings
   namespace eval ::foo {
      catch {::Simple::ExtProc::move bar foo} result
      puts $result
      ::Simple::ExtProc::move {bar gee} {foo zii}
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {can't rename "bar": command doesn't exist
can't rename "::foo::bar": command doesn't exist
can't rename "::foo::bar gee": command doesn't exist
-1
can't rename "bar": command doesn't exist
} -error {can't rename "bar gee": command doesn't exist}

test-case move-5 {
   ::Simple::ExtProc::move
   error, can't rename non-subcommand to subcommand
} -setup {

   # Create a regular procedure
   proc foo {
      int1
      int2
      int3
   } {}

} -script {
   ::Simple::ExtProc::move foo {bar gee}
} -cleanup {

   # Delete the procedure
   rename foo {}

} -error {can't rename non-subcommand "foo" to subcommand "bar gee"}

test-case move-6 {
   ::Simple::ExtProc::move
   error, can't rename subcommand to non-subcommand
} -setup {

   # Create a subcommand procedure
   proc-ext {foo bar} {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {}

} -script {
   ::Simple::ExtProc::move {foo bar} gee
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete {foo bar}

} -error {can't rename subcommand "foo bar" to non-subcommand "gee"}

### ===========================================================================
### -test     : ::Simple::ExtProc::copy
test-case copy-1 {
   ::Simple::ExtProc::copy
   regular procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a regular procedure
   proc ::foo::bar {
      int1
      int2
      int3
   } {display-args int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar 1 2 3

   # Invalid call
   catch {::bar::foo 1 2 3} result
   puts $result

   # Copy the procedure
   namespace eval ::foo {
      ::Simple::ExtProc::copy bar ::bar::foo
   }

   # Valid call
   ::bar::foo 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::bar::foo

   # Delete the namespaces
   namespace delete ::foo ::bar

} -output {---------------------
int1 = {1}
int2 = {2}
int3 = {3}
invalid command name "::bar::foo"
---------------------
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case copy-2 {
   ::Simple::ExtProc::copy
   extended procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create an extended procedure
   proc-ext ::foo::bar {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {display-args boolflag int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar -boolflag 1 2 3

   # Invalid call
   catch {::bar::foo -boolflag 1 2 3} result
   puts $result

   # Copy the procedure
   namespace eval ::foo {
      ::Simple::ExtProc::copy bar ::bar::foo
   }

   # Valid call
   ::bar::foo -boolflag 1 2 3

} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::bar::foo

   # Delete the namespaces
   namespace delete ::foo ::bar

} -output {---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
invalid command name "::bar::foo"
---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case copy-3 {
   ::Simple::ExtProc::delete
   subcommand procedure
} -prerequisites {
   {![string compare [info args display-args] args]}
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a subcommand procedure
   proc-ext {::foo::bar gee} {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {display-args boolflag int1 int2 int3}

} -script {

   # Valid call
   ::foo::bar gee -boolflag 1 2 3

   # Invalid call
   catch {::bar::foo zii -boolflag 1 2 3} result
   puts $result

   # Copy the procedure
   ::Simple::ExtProc::copy {::foo::bar gee} {::bar::foo zii}

   # Valid call
   ::bar::foo zii -boolflag 1 2 3

} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::bar::foo
   ::Simple::ExtProc::delete {::foo::bar gee}

   # Delete the namespaces
   namespace delete ::foo ::bar

} -output {---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
invalid command name "::bar::foo"
---------------------
boolflag = {1}
int1 = {1}
int2 = {2}
int3 = {3}
}

test-case copy-4 {
   ::Simple::ExtProc::copy
   error, non-existing procedure
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] == -1}
} -script {

   # Failed procedure copies
   catch {::Simple::ExtProc::copy bar foo} result
   puts $result
   catch {::Simple::ExtProc::copy ::foo::bar ::bar::foo} result
   puts $result
   catch {::Simple::ExtProc::copy {::foo::bar gee} {::bar::foo zii}} result
   puts $result

   # Ensure the namespace ::foo has not been created
   puts [lsearch -exact [namespace children] ::foo]

   # Failed procedure copies
   namespace eval ::foo {
      catch {::Simple::ExtProc::copy bar foo} result
      puts $result
      ::Simple::ExtProc::copy {bar gee} {foo zii}
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {can't copy "bar": command doesn't exist
can't copy "::foo::bar": command doesn't exist
can't copy "::foo::bar gee": command doesn't exist
-1
can't copy "bar": command doesn't exist
} -error {can't copy "bar gee": command doesn't exist}

test-case copy-5 {
   ::Simple::ExtProc::copy
   error, can't copy non-subcommand to subcommand
} -setup {

   # Create a regular procedure
   proc foo {
      int1
      int2
      int3
   } {}

} -script {
   ::Simple::ExtProc::copy foo {bar gee}
} -cleanup {

   # Delete the procedure
   rename foo {}

} -error {can't copy non-subcommand "foo" to subcommand "bar gee"}

test-case copy-6 {
   ::Simple::ExtProc::copy
   error, can't rename subcommand to non-subcommand
} -setup {

   # Create a subcommand procedure
   proc-ext {foo bar} {
      {-boolflag         -boolflag      "A boolflag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      {?int3?            -int 99        "Third argument"}
   } {}

} -script {
   ::Simple::ExtProc::copy {foo bar} gee
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete {foo bar}

} -error {can't copy subcommand "foo bar" to non-subcommand "gee"}

### ===========================================================================
### -test     : ::Simple::ExtProc::information is-subcommand
test-case is-subcommand-1 {
   ::Simple::ExtProc::information is-subcommand
} -setup {

   # Create some procedures
   proc bar {} {}
   proc-sub {zii wuu} {} {}
   namespace eval ::foo {
      proc-ext {bar gee} {} {}
   }

} -script {

   # Non-base command procedures
   puts -nonewline [::Simple::ExtProc::information is-subcommand\
      ::Simple::ExtProc::create]
   puts -nonewline [::Simple::ExtProc::information is-subcommand ::foo]
   puts -nonewline [::Simple::ExtProc::information is-subcommand ::bar]
   puts -nonewline [::Simple::ExtProc::information is-subcommand ::bar::gee]
   puts -nonewline [::Simple::ExtProc::information is-subcommand ::foo::gee]

   # Base subcommand procedures
   puts -nonewline [::Simple::ExtProc::information is-subcommand ::foo::bar]
   puts -nonewline [::Simple::ExtProc::information is-subcommand zii]
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtProc::information is-subcommand bar]
   }

} -cleanup {

   # Delete the procedures
   rename bar {}
   ::Simple::ExtProc::delete {zii wuu}

   # Delete the namespace
   namespace delete ::foo

} -output 00000111

test-case is-subcommand-2 {
   ::Simple::ExtProc::information is-subcommand
   error, subcommand can't be base subcommand procedure
} -script {
   ::Simple::ExtProc::information is-subcommand {foo bar}
} -error {expected non-subcommand procedure name but got "foo bar"}

### ===========================================================================
### -test     : ::Simple::ExtProc::information subcommands
test-case subcommands-1 {
   ::Simple::ExtProc::information subcommands
} -setup {

   # Create some procedures
   proc gee {} {}
   namespace eval ::foo {
      proc gee {} {}
      proc-sub {bar subcommand-1} {} {}
      proc-sub {bar subcommand-2} {} {}
      proc-sub {bar subcommand-3} {} {}
   }

} -script {

   # Base subcommand procedures
   puts [lsort [::Simple::ExtProc::information subcommands ::foo::bar]]
   namespace eval ::foo {
      puts [lsort [::Simple::ExtProc::information subcommands bar]]
   }

   # Non-base command procedures
   catch {::Simple::ExtProc::information subcommands gee} result
   puts $result
   catch {::Simple::ExtProc::information subcommands ::foo::gee} result
   puts $result
   namespace eval ::foo {
      catch {::Simple::ExtProc::information subcommands gee} result
      puts $result
   }

} -cleanup {

   # Delete the procedures
   rename gee {}

   # Delete the namespace
   namespace delete ::foo

} -output {subcommand-1 subcommand-2 subcommand-3
subcommand-1 subcommand-2 subcommand-3
"gee" isn't a subcommand base command procedure
"::foo::gee" isn't a subcommand base command procedure
"gee" isn't a subcommand base command procedure
}

test-case subcommands-2 {
   ::Simple::ExtProc::information subcommands
   error, subcommand can't be base subcommand procedure
} -script {
   ::Simple::ExtProc::information subcommands {foo bar}
} -error {expected non-subcommand procedure name but got "foo bar"}

### ===========================================================================
### -test     : ::Simple::ExtProc::information runtimeparsing
test-case runtimeparsing-1 {
   ::Simple::ExtProc::information runtimeparsing
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc ::foo::luu {} {}
   proc-ext -checktype false ::foo::bar {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}
   proc-ext -checktype true {::foo::gee bar} {
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}
   proc-ext -checktype false ::foo::zii {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}
   proc-ext -interpflags false {::foo::gee zii} {
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int1             -int           "First argument"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {}

} -script {

   # Run-time parsing procedures
   puts -nonewline\
      [::Simple::ExtProc::information runtimeparsing {::foo::gee bar}]
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtProc::information runtimeparsing zii]
   }

   # No run-time parsing procedures
   puts -nonewline [::Simple::ExtProc::information runtimeparsing ::foo::ryy]
   puts -nonewline [::Simple::ExtProc::information runtimeparsing ::foo::luu]
   puts -nonewline\
      [::Simple::ExtProc::information runtimeparsing ::foo::bar]
   puts -nonewline\
      [::Simple::ExtProc::information runtimeparsing {::foo::gee zii}]

} -output 110000

### ===========================================================================
### -test     : ::Simple::ExtProc::information checktype
test-case checktype-1 {
   ::Simple::ExtProc::information checktype
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Type checking procedures
   puts -nonewline\
      [::Simple::ExtProc::information checktype {::foo::gee bar}]

   # No type checking procedures
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtProc::information checktype zii]
   }
   puts -nonewline [::Simple::ExtProc::information checktype ::foo::ryy]
   puts -nonewline [::Simple::ExtProc::information checktype ::foo::luu]
   puts -nonewline\
      [::Simple::ExtProc::information checktype ::foo::bar]
   puts -nonewline\
      [::Simple::ExtProc::information checktype {::foo::gee zii}]

} -output 100000

### ===========================================================================
### -test     : ::Simple::ExtProc::information interpflags
test-case interpflags-1 {
   ::Simple::ExtProc::information interpflags
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Flag interpretation procedures
   puts -nonewline\
      [::Simple::ExtProc::information interpflags {::foo::gee bar}]
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtProc::information interpflags zii]
   }
   puts -nonewline\
      [::Simple::ExtProc::information interpflags ::foo::bar]

   # No flag interpretation procedures
   puts -nonewline [::Simple::ExtProc::information interpflags ::foo::ryy]
   puts -nonewline [::Simple::ExtProc::information interpflags ::foo::luu]
   puts -nonewline\
      [::Simple::ExtProc::information interpflags {::foo::gee zii}]

} -output 111000

### ===========================================================================
### -test     : ::Simple::ExtProc::information exists
test-case exists-1 {
   ::Simple::ExtProc::information exists
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {

   # Flag interpretation procedures
   puts -nonewline\
      [::Simple::ExtProc::information exists {::foo::gee bar}]
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtProc::information exists zii]
   }
   puts -nonewline\
      [::Simple::ExtProc::information exists ::foo::bar]
   puts -nonewline\
      [::Simple::ExtProc::information exists {::foo::gee zii}]

   # No flag interpretation procedures
   puts -nonewline [::Simple::ExtProc::information exists ::foo::ryy]
   puts -nonewline [::Simple::ExtProc::information exists ::foo::luu]

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output 111100

### ===========================================================================
### -test     : ::Simple::ExtProc::information body
test-case body-1 {
   ::Simple::ExtProc::information body
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc ::foo::lyy {
      arg1
      arg2
      {arg3 default1}
      {arg4 default2}
      args
   } {::foo::lyy body}
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      { choice1          -choice {a b}  "A choice"}
      {-choice2          -choice {1 2}  "Another choice"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      {?int3?            -int 2         "Third argument"}
      {?choice3?         -choice {Q W}  "Yet another choice"}
      { args             -any           "More arguments"}
   } {::foo::bar body}
   proc-ext {::foo::gee zii} {
      { intA             -int           "First argument"}
      { choiceA          -choice {x y}  "A choice"}
      {-choiceB          -choice {3 4}  "Another choice"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 2.0     "A flag"}
      {?intC?            -int 4         "Third argument"}
      {?choiceC?         -choice {A B}  "Yet another choice"}
   } {::foo::gee zii body}

} -script {
   puts [::Simple::ExtProc::information body ::foo::lyy]
   puts [::Simple::ExtProc::information body ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtProc::information body {gee zii}]
   }
} -output {::foo::lyy body
::foo::bar body
::foo::gee zii body
}

test-case body-2 {
   ::Simple::ExtProc::information body
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information body @foo} result
   puts $result
   ::Simple::ExtProc::information body {}
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case body-3 {
   ::Simple::ExtProc::information body
   error, non-existing procedure
} -script {
   catch {::Simple::ExtProc::information body ::bar::foo} result
   puts $result
   ::Simple::ExtProc::information body {::foo::gee luu}
} -output {"::bar::foo" isn't a procedure
} -error {"::foo::gee luu" isn't a procedure}

### ===========================================================================
### -test     : ::Simple::ExtProc::information args
test-case args-1 {
   ::Simple::ExtProc::information args
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   puts [::Simple::ExtProc::information args ::foo::lyy]
   puts [::Simple::ExtProc::information args ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtProc::information args {gee zii}]
   }
} -output {arg1 arg2 arg3 arg4 args
int1 choice1 -choice2 -boolflag -flag int3 choice3 args
intA choiceA -choiceB -boolflag -flag intC choiceC
}

test-case args-2 {
   ::Simple::ExtProc::information args
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information args @foo} result
   puts $result
   ::Simple::ExtProc::information args {}
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case args-3 {
   ::Simple::ExtProc::information args
   error, non-existing procedure
} -script {
   catch {::Simple::ExtProc::information args ::bar::foo} result
   puts $result
   ::Simple::ExtProc::information args {::foo::gee luu}
} -output {"::bar::foo" isn't a procedure
} -error {"::foo::gee luu" isn't a procedure}

### ===========================================================================
### -test     : ::Simple::ExtProc::information argsusage
test-case argsusage-1 {
   ::Simple::ExtProc::information argsusage
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   puts [::Simple::ExtProc::information argsusage ::foo::lyy]
   puts [::Simple::ExtProc::information argsusage ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtProc::information argsusage {gee zii}]
   }
} -output { arg1 arg2 ?arg3? ?arg4? ?args?
 int1 choice1 ?-choice2 choice? ?-boolflag? ?-flag float? ?int3? ?choice3?\
?args?
 intA choiceA ?-choiceB choice? ?-boolflag? ?-flag float? ?intC? ?choiceC?
}

test-case argsusage-2 {
   ::Simple::ExtProc::information argsusage
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information argsusage @foo} result
   puts $result
   ::Simple::ExtProc::information argsusage {}
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case argsusage-3 {
   ::Simple::ExtProc::information argsusage
   error, non-existing procedure
} -script {
   catch {::Simple::ExtProc::information argsusage ::bar::foo} result
   puts $result
   ::Simple::ExtProc::information argsusage {::foo::gee luu}
} -output {"::bar::foo" isn't a procedure
} -error {"::foo::gee luu" isn't a procedure}

### ===========================================================================
### -test     : ::Simple::ExtProc::information argslist
test-case argslist-1 {
   ::Simple::ExtProc::information argslist
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   puts [::Simple::ExtProc::information argslist ::foo::lyy]
   puts [::Simple::ExtProc::information argslist ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtProc::information argslist {gee zii}]
   }
} -output {arg1 arg2 {arg3 default1} {arg4 default2} args
{int1 -int {First argument}} {choice1 -choice {a b} {A choice}} {-choice2\
-choice {1 2} {Another choice}} {-boolflag -boolflag {A boolflag}} {-flag\
-float 1.0 {A flag}} {?int3? -int 2 {Third argument}} {?choice3? -choice {Q W}\
{Yet another choice}} {args -any {More arguments}}
{intA -int {First argument}} {choiceA -choice {x y} {A choice}} {-choiceB\
-choice {3 4} {Another choice}} {-boolflag -boolflag {A boolflag}} {-flag\
-float 2.0 {A flag}} {?intC? -int 4 {Third argument}} {?choiceC? -choice {A B}\
{Yet another choice}}
}

test-case argslist-2 {
   ::Simple::ExtProc::information argslist
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information argslist @foo} result
   puts $result
   ::Simple::ExtProc::information argslist {}
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case argslist-3 {
   ::Simple::ExtProc::information argslist
   error, non-existing procedure
} -script {
   catch {::Simple::ExtProc::information argslist ::bar::foo} result
   puts $result
   ::Simple::ExtProc::information argslist {::foo::gee luu}
} -output {"::bar::foo" isn't a procedure
} -error {"::foo::gee luu" isn't a procedure}

### ===========================================================================
### -test     : ::Simple::ExtProc::information type
test-case type-1 {
   ::Simple::ExtProc::information type
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   foreach arg [::Simple::ExtProc::information args ::foo::bar] {
      puts "$arg -> [::Simple::ExtProc::information type ::foo::bar $arg]"
   }
   puts ---------------------
   namespace eval ::foo {
      foreach arg [::Simple::ExtProc::information args {gee zii}] {
         puts "$arg -> [::Simple::ExtProc::information type {gee zii} $arg]"
      }
   }
} -output {int1 -> -int
choice1 -> -choice
-choice2 -> -choice
-boolflag -> -boolflag
-flag -> -float
int3 -> -int
choice3 -> -choice
args -> -any
---------------------
intA -> -int
choiceA -> -choice
-choiceB -> -choice
-boolflag -> -boolflag
-flag -> -float
intC -> -int
choiceC -> -choice
}

test-case type-2 {
   ::Simple::ExtProc::information type
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information type @foo arg} result
   puts $result
   ::Simple::ExtProc::information type {} arg
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case type-3 {
   ::Simple::ExtProc::information type
   error, non-existing extended procedure
} -script {
   catch {::Simple::ExtProc::information type ::bar::foo arg} result
   puts $result
   catch {::Simple::ExtProc::information type {::foo::gee luu} arg} result
   puts $result
   ::Simple::ExtProc::information type ::foo::lyy arg
} -output {"::bar::foo" isn't an extended procedure
"::foo::gee luu" isn't an extended procedure
} -error {"::foo::lyy" isn't an extended procedure}

test-case type-4 {
   ::Simple::ExtProc::information type
   error, non-existing argument
} -script {
   catch {::Simple::ExtProc::information type ::foo::bar woo} result
   puts $result
   namespace eval ::foo {
      ::Simple::ExtProc::information type {gee zii} luu
   }
} -output {procedure "::foo::bar" doesn't have an argument "woo"
} -error {procedure "gee zii" doesn't have an argument "luu"}

### ===========================================================================
### -test     : ::Simple::ExtProc::information default
test-case default-1 {
   ::Simple::ExtProc::information default
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   foreach arg [info args ::foo::lyy] {
      puts -nonewline "$arg -> "
      if {[::Simple::ExtProc::information default ::foo::lyy $arg var]} {
         puts $var
      } else {
         puts {no default value}
      }
   }
   puts ---------------------
   foreach arg [::Simple::ExtProc::information args ::foo::bar] {
      puts -nonewline "$arg -> "
      if {[::Simple::ExtProc::information default ::foo::bar $arg var]} {
         puts $var
      } else {
         puts {no default value}
      }
   }
   puts ---------------------
   namespace eval ::foo {
      foreach arg [::Simple::ExtProc::information args {gee zii}] {
         puts -nonewline "$arg -> "
         if {[::Simple::ExtProc::information default {gee zii} $arg var]} {
            puts $var
         } else {
            puts {no default value}
         }
      }
   }
} -output {arg1 -> no default value
arg2 -> no default value
arg3 -> default1
arg4 -> default2
args -> no default value
---------------------
int1 -> no default value
choice1 -> no default value
-choice2 -> 1
-boolflag -> no default value
-flag -> 1.0
int3 -> 2
choice3 -> Q
args -> no default value
---------------------
intA -> no default value
choiceA -> no default value
-choiceB -> 3
-boolflag -> no default value
-flag -> 2.0
intC -> 4
choiceC -> A
}

test-case default-2 {
   ::Simple::ExtProc::information default
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information default @foo arg var} result
   puts $result
   ::Simple::ExtProc::information default {} arg var
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case default-3 {
   ::Simple::ExtProc::information default
   error, non-existing procedure
} -script {
   catch {::Simple::ExtProc::information default ::bar::foo arg var} result
   puts $result
   ::Simple::ExtProc::information default {::foo::gee luu} arg var
} -output {"::bar::foo" isn't a procedure
} -error {"::foo::gee luu" isn't a procedure}

test-case default-4 {
   ::Simple::ExtProc::information default
   error, non-existing argument
} -script {
   catch {::Simple::ExtProc::information default ::foo::bar woo var} result
   puts $result
   namespace eval ::foo {
      ::Simple::ExtProc::information default {gee zii} luu var
   }
} -output {procedure "::foo::bar" doesn't have an argument "woo"
} -error {procedure "gee zii" doesn't have an argument "luu"}

### ===========================================================================
### -test     : ::Simple::ExtProc::information choices
test-case choices-1 {
   ::Simple::ExtProc::information choices
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   foreach arg [::Simple::ExtProc::information args ::foo::bar] {
      if {![string compare [::Simple::ExtProc::information type ::foo::bar\
         $arg] -choice]} {

         puts "$arg ->\
            [::Simple::ExtProc::information choices ::foo::bar $arg]"
      }
   }
   puts ---------------------
   namespace eval ::foo {
      foreach arg [::Simple::ExtProc::information args {gee zii}] {
         if {![string compare [::Simple::ExtProc::information type {gee zii}\
            $arg] -choice]} {

            puts "$arg ->\
               [::Simple::ExtProc::information choices {gee zii} $arg]"
         }
      }
   }
} -output {choice1 -> a b
-choice2 -> 1 2
choice3 -> Q W
---------------------
choiceA -> x y
-choiceB -> 3 4
choiceC -> A B
}

test-case choices-2 {
   ::Simple::ExtProc::information choices
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information choices @foo arg} result
   puts $result
   ::Simple::ExtProc::information choices {} arg
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case choices-3 {
   ::Simple::ExtProc::information choices
   error, non-existing extended procedure
} -script {
   catch {::Simple::ExtProc::information choices ::bar::foo arg} result
   puts $result
   catch {::Simple::ExtProc::information choices {::foo::gee luu} arg} result
   puts $result
   ::Simple::ExtProc::information choices ::foo::lyy arg
} -output {"::bar::foo" isn't an extended procedure
"::foo::gee luu" isn't an extended procedure
} -error {"::foo::lyy" isn't an extended procedure}

test-case choices-4 {
   ::Simple::ExtProc::information choices
   error, non-existing argument
} -script {
   catch {::Simple::ExtProc::information choices ::foo::bar woo} result
   puts $result
   namespace eval ::foo {
      ::Simple::ExtProc::information choices {gee zii} luu
   }
} -output {procedure "::foo::bar" doesn't have an argument "woo"
} -error {procedure "gee zii" doesn't have an argument "luu"}

test-case choices-5 {
   ::Simple::ExtProc::information choices
   error, not of type -choice
} -script {
   catch {::Simple::ExtProc::information choices ::foo::bar woo} result
   puts $result
   namespace eval ::foo {
      ::Simple::ExtProc::information choices {gee zii} luu
   }
} -output {procedure "::foo::bar" doesn't have an argument "woo"
} -error {procedure "gee zii" doesn't have an argument "luu"}

### ===========================================================================
### -test     : ::Simple::ExtProc::information description
test-case description-1 {
   ::Simple::ExtProc::information description
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] != -1}
} -script {
   foreach arg [::Simple::ExtProc::information args ::foo::bar] {
      puts\
         "$arg -> [::Simple::ExtProc::information description ::foo::bar $arg]"
   }
   puts ---------------------
   namespace eval ::foo {
      foreach arg [::Simple::ExtProc::information args {gee zii}] {
         puts "$arg ->\
            [::Simple::ExtProc::information description {gee zii} $arg]"
      }
   }
} -output {int1 -> First argument
choice1 -> A choice
-choice2 -> Another choice
-boolflag -> A boolflag
-flag -> A flag
int3 -> Third argument
choice3 -> Yet another choice
args -> More arguments
---------------------
intA -> First argument
choiceA -> A choice
-choiceB -> Another choice
-boolflag -> A boolflag
-flag -> A flag
intC -> Third argument
choiceC -> Yet another choice
}

test-case description-2 {
   ::Simple::ExtProc::information description
   error, bad procedure name
} -script {
   catch {::Simple::ExtProc::information description @foo arg} result
   puts $result
   ::Simple::ExtProc::information description {} arg
} -output {expected procedure name but got "@foo"
} -error {expected procedure name but got ""}

test-case description-3 {
   ::Simple::ExtProc::information description
   error, non-existing extended procedure
} -script {
   catch {::Simple::ExtProc::information description ::bar::foo arg} result
   puts $result
   catch\
      {::Simple::ExtProc::information description {::foo::gee luu} arg} result
   puts $result
   ::Simple::ExtProc::information description ::foo::lyy arg
} -output {"::bar::foo" isn't an extended procedure
"::foo::gee luu" isn't an extended procedure
} -error {"::foo::lyy" isn't an extended procedure}

test-case description-4 {
   ::Simple::ExtProc::information description
   error, non-existing argument
} -script {
   catch {::Simple::ExtProc::information description ::foo::bar woo} result
   puts $result
   namespace eval ::foo {
      ::Simple::ExtProc::information description {gee zii} luu
   }
} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::foo::lyy
   ::Simple::ExtProc::delete ::foo::bar
   ::Simple::ExtProc::delete {::foo::gee zii}

   # Delete the namespace
   namespace delete ::foo

} -output {procedure "::foo::bar" doesn't have an argument "woo"
} -error {procedure "gee zii" doesn't have an argument "luu"}

### ===========================================================================
### -test     : ::Simple::ExtProc::number-flags
test-case number-flags-1 {
   ::Simple::ExtProc::number-flags
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {puts -nonewline [number-flags]}

} -script {
   ::foo::bar 1 2 3
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2 3
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

   # Delete the namespace
   namespace delete ::foo

} -output 0112

### ===========================================================================
### -test     : ::Simple::ExtProc::flag-given
test-case flag-given-1 {
   ::Simple::ExtProc::flag-given
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure
   proc-ext ::foo::bar {
      { int1             -int           "First argument"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      { int2             -int           "Second argument"}
      { int3             -int           "Third argument"}
   } {puts "[flag-given -boolflag] [flag-given -flag]"}

} -script {
   ::foo::bar 1 2 3
   ::foo::bar 1 -boolflag 2 3
   ::foo::bar 1 -flag 2.0 2 3
   ::foo::bar 1 -boolflag -flag 2.0 2 3
} -cleanup {

   # Delete the procedure
   ::Simple::ExtProc::delete ::foo::bar

   # Delete the namespace
   namespace delete ::foo

} -output {0 0
1 0
0 1
1 1
}

### ===========================================================================
### -test     : ::Simple::ExtProc::arg-is-default
test-case arg-is-default-1 {
   ::Simple::ExtProc::arg-is-default
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc ::foo::bar {
      arg1
      {arg2 a}
   } {
      puts [arg-is-default arg2]
   }
   proc-sub {::foo::gee zii} {
      arg1
      {arg2 a}
   } {
      puts [arg-is-default arg2]
   }
   proc-ext ::foo::lyy {
      { int1             -int           "First argument"}
      { choice1          -choice {a b}  "A choice"}
      {-choice2          -choice {1 2}  "Another choice"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      {?int3?            -int 2         "Third argument"}
      {?choice3?         -choice {Q W}  "Yet another choice"}
   } {
      puts -nonewline [arg-is-default -choice2]
      puts -nonewline [arg-is-default -flag]
      puts -nonewline [arg-is-default int3]
      puts [arg-is-default choice3]
   }
   proc-ext {::foo::woo pyy} {
      { int1             -int           "First argument"}
      { choice1          -choice {a b}  "A choice"}
      {-choice2          -choice {1 2}  "Another choice"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      {?int3?            -int 2         "Third argument"}
      {?choice3?         -choice {Q W}  "Yet another choice"}
   } {
      puts -nonewline [arg-is-default -choice2]
      puts -nonewline [arg-is-default -flag]
      puts -nonewline [arg-is-default int3]
      puts [arg-is-default choice3]
   }
} -script {
   ::foo::bar 1
   ::foo::bar 1 a
   ::foo::bar 1 b
   puts ---------------------
   ::foo::gee zii 1
   ::foo::gee zii 1 a
   ::foo::gee zii 1 b
   puts ---------------------
   ::foo::lyy 1 a
   ::foo::lyy 1 a -flag 2.0 2 W
   ::foo::lyy 1 a -flag 1.0 3 Q
   puts ---------------------
   ::foo::woo pyy 1 a
   ::foo::woo pyy 1 a -flag 2.0 2 W
   ::foo::woo pyy 1 a -flag 1.0 3 Q
} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::foo::bar
   ::Simple::ExtProc::delete {::foo::gee zii}
   ::Simple::ExtProc::delete ::foo::lyy
   ::Simple::ExtProc::delete {::foo::woo pyy}

   # Delete the namespace
   namespace delete ::foo

} -output {1
1
0
---------------------
1
1
0
---------------------
1111
1010
1101
---------------------
1111
1010
1101
}

test-case arg-is-default-2 {
   ::Simple::ExtProc::arg-is-default
   error, non-existing argument
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create some procedures
   proc ::foo::bar {
      arg1
      {arg2 a}
   } {
      puts [arg-is-default gee]
   }
   proc-ext {::foo::woo pyy} {
      { int1             -int           "First argument"}
      { choice1          -choice {a b}  "A choice"}
      {-choice2          -choice {1 2}  "Another choice"}
      {-boolflag         -boolflag      "A boolflag"}
      {-flag             -float 1.0     "A flag"}
      {?int3?            -int 2         "Third argument"}
      {?choice3?         -choice {Q W}  "Yet another choice"}
   } {
      puts [arg-is-default gee]
   }

} -script {
   catch {::foo::bar 0} result
   puts $result
   namespace eval ::foo {
     woo pyy 1 a
   }
} -cleanup {

   # Delete the procedures
   ::Simple::ExtProc::delete ::foo::bar
   ::Simple::ExtProc::delete {::foo::woo pyy}

   # Delete the namespace
   namespace delete ::foo

} -output {procedure "::foo::bar" doesn't have an argument "gee"
} -error {procedure "woo pyy" doesn't have an argument "gee"}

test-case arg-is-default-3 {
   ::Simple::ExtProc::arg-is-default
   error, can't be called from the top level
} -script {
   arg-is-default ::foo::bar
} -error {"arg-is-default" can't be called from the top level}

### ===========================================================================
### -test     : ::Simple::ScriptHead::add
test-case headadd-1 {
   ::Simple::ScriptHead::add
} -setup {

   # Initialise the base script
   set baseScript "This is the body first line\nThis is the body second line"

   # Copy the base script to the current script
   set script $baseScript

} -script {

   # Add a header to the script
   ::Simple::ScriptHead::add script tag1 100 {This is the first header}

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
This is the body first line
This is the body second line}

test-case headadd-2 {
   ::Simple::ScriptHead::add
   second header, same priority
} -prerequisites {
   {[llength [split $script \n]] == 5}
} -setup {

   # Copy the current script to the base script
   set baseScript $script

} -script {

   # Add a second header to the script
   ::Simple::ScriptHead::add script tag2 100 {This is the second header}

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  100 tag2
This is the second header
#========= %HEADER-END%    100 tag2
This is the body first line
This is the body second line}

test-case headadd-3 {
   ::Simple::ScriptHead::add
   second header, higher priority
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -setup {

   # Copy the base script to another script
   set script $baseScript

} -script {

   # Add a second header to the script
   ::Simple::ScriptHead::add script tag2 50 {This is the second header}

} -return {#========= %HEADER-BEGIN%   50 tag2
This is the second header
#========= %HEADER-END%     50 tag2
#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
This is the body first line
This is the body second line}

test-case headadd-4 {
   ::Simple::ScriptHead::add
   second header, lower priority
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -setup {

   # Copy the base script to another script
   set script $baseScript

} -script {

   # Add a second header to the script
   ::Simple::ScriptHead::add script tag2 200 {This is the second header}

} -cleanup {

   # Copy the script to the base script
   set baseScript $script

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
This is the body first line
This is the body second line}

test-case headadd-5 {
   ::Simple::ScriptHead::add
   third header, top priority
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -script {

   # Add a third header to the script
   ::Simple::ScriptHead::add script tag3 0 {This is the third header}

} -cleanup {

   # Copy the base script to another script
   set script $baseScript

} -return {#========= %HEADER-BEGIN%    0 tag3
This is the third header
#========= %HEADER-END%      0 tag3
#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
This is the body first line
This is the body second line}

test-case headadd-6 {
   ::Simple::ScriptHead::add
   third header, first header priority
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -script {

   # Add a third header to the script
   ::Simple::ScriptHead::add script tag3 100 {This is the third header}

} -cleanup {

   # Copy the base script to another script
   set script $baseScript

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  100 tag3
This is the third header
#========= %HEADER-END%    100 tag3
#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
This is the body first line
This is the body second line}

test-case headadd-7 {
   ::Simple::ScriptHead::add
   third header, between headers
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -script {

   # Add a third header to the script
   ::Simple::ScriptHead::add script tag3 150 {This is the third header}

} -cleanup {

   # Copy the base script to another script
   set script $baseScript

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  150 tag3
This is the third header
#========= %HEADER-END%    150 tag3
#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
This is the body first line
This is the body second line}

test-case headadd-8 {
   ::Simple::ScriptHead::add
   third header, second header priority
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -script {

   # Add a third header to the script
   ::Simple::ScriptHead::add script tag3 200 {This is the third header}

} -cleanup {

   # Copy the base script to another script
   set script $baseScript

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
#========= %HEADER-BEGIN%  200 tag3
This is the third header
#========= %HEADER-END%    200 tag3
This is the body first line
This is the body second line}

test-case headadd-9 {
   ::Simple::ScriptHead::add
   third header, bottom priority
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -script {

   # Add a third header to the script
   ::Simple::ScriptHead::add script tag3 300 {This is the third header}

} -cleanup {

   # Copy the base script to another script
   set script $baseScript

} -return {#========= %HEADER-BEGIN%  100 tag1
This is the first header
#========= %HEADER-END%    100 tag1
#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
#========= %HEADER-BEGIN%  300 tag3
This is the third header
#========= %HEADER-END%    300 tag3
This is the body first line
This is the body second line}

test-case headadd-10 {
   ::Simple::ScriptHead::add
   error, duplicated tag
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -script {

   # Failed addition
   ::Simple::ScriptHead::add script tag2 100 {This is the second header}

} -error {duplicated tag "tag2" in script "script"}

test-case headadd-11 {
   ::Simple::ScriptHead::add
   error, non-existing script
} -script {
   ::Simple::ScriptHead::add foo tag2 100 {This is the second header}
} -error {non-existing script "foo"}

### ===========================================================================
### -test     : ::Simple::ScriptHead::get
test-case headget-1 {
   ::Simple::ScriptHead::get
} -prerequisites {
   {[llength [split $script \n]] == 8}
} -setup {

   # Add some more headers to the script
   ::Simple::ScriptHead::add script tag1-A   0 {Header tag1-A}
   ::Simple::ScriptHead::add script tag1-B  50 {Header tag1-B}
   ::Simple::ScriptHead::add script tag1-C 150 {Header tag1-C}
   ::Simple::ScriptHead::add script tag1-D 250 {Header tag1-D}

} -script {

   # Get all headers
   puts [::Simple::ScriptHead::get $script]

   # Get header matching a regular expression
   ::Simple::ScriptHead::get $script tag1*

} -output {{tag1-A 0} {tag1-B 50} {tag1 100} {tag1-C 150} {tag2 200}\
{tag1-D 250}
} -return {{tag1-A 0} {tag1-B 50} {tag1 100} {tag1-C 150} {tag1-D 250}}

### ===========================================================================
### -test     : ::Simple::ScriptHead::delete
test-case headdelete-1 {
   ::Simple::ScriptHead::delete
} -prerequisites {
   {[llength [split $script \n]] == 20}
} -script {

   # Delete the headers matching a regular expression
   puts [::Simple::ScriptHead::delete script tag1*]

   # Get the script
   set script

} -output {5
} -return {#========= %HEADER-BEGIN%  200 tag2
This is the second header
#========= %HEADER-END%    200 tag2
This is the body first line
This is the body second line}

test-case headdelete-2 {
   ::Simple::ScriptHead::delete
   error, non-existing header
} -prerequisites {
   {[llength [split $script \n]] == 5}
} -script {

   # Try to delete a non-existing header
   ::Simple::ScriptHead::delete script tag3

} -error {non-existing tag "tag3" in script "script"}

test-case headdelete-3 {
   ::Simple::ScriptHead::delete
   error, non-existing script
} -script {
   ::Simple::ScriptHead::delete foo tag2
} -error {non-existing script "foo"}

### ===========================================================================
### -test     : ::Simple::ExtProc::cget
test-case cget-1 {
   ::Simple::ExtProc::cget
   -checktype
} -setup {

   # Save the check type package option
   set ::Simple::ExtProc::%TMP% $::Simple::ExtProc::CheckType

   # Set the check type package option to true
   set ::Simple::ExtProc::CheckType 1

} -script {
   ::Simple::ExtProc::cget -checktype
} -cleanup {

   # Restore the check type package option
   set ::Simple::ExtProc::CheckType ${::Simple::ExtProc::%TMP%}
   unset ::Simple::ExtProc::%TMP%

} -return 1

test-case cget-2 {
   ::Simple::ExtProc::cget
   -afterflags
} -setup {

   # Save the after flags package option
   set ::Simple::ExtProc::%TMP% $::Simple::ExtProc::AfterFlags

   # Set the after flags package option to 5
   set ::Simple::ExtProc::AfterFlags 5

} -script {
   ::Simple::ExtProc::cget -afterflags
} -cleanup {

   # Restore the after flags package option
   set ::Simple::ExtProc::AfterFlags ${::Simple::ExtProc::%TMP%}
   unset ::Simple::ExtProc::%TMP%

} -return 5

test-case cget-3 {
   ::Simple::ExtProc::cget
   no arguments
} -script {
   ::Simple::ExtProc::cget
} -return {-checktype -afterflags}

test-case cget-4 {
   ::Simple::ExtProc::cget
   error, incorrect option
} -script {
   ::Simple::ExtProc::cget -foo
} -error {bad option "-foo": must be -checktype or -afterflags}

### ===========================================================================
### -test     : ::Simple::ExtProc::configure
test-case configure-1 {
   ::Simple::ExtProc::configure
   -checktype
} -setup {

   # Save the check type package option
   set ::Simple::ExtProc::%TMP% $::Simple::ExtProc::CheckType

   # Set the check type package option to true
   set ::Simple::ExtProc::CheckType 1

} -script {

   # Set the check type package option to true
   ::Simple::ExtProc::configure -checktype true

   # Get the check type package option
   ::Simple::ExtProc::cget -checktype

} -cleanup {

   # Restore the check type package option
   set ::Simple::ExtProc::CheckType ${::Simple::ExtProc::%TMP%}
   unset ::Simple::ExtProc::%TMP%

} -return 1

test-case configure-2 {
   ::Simple::ExtProc::configure
   -afterflags
} -setup {

   # Save the after flags package option
   set ::Simple::ExtProc::%TMP% $::Simple::ExtProc::CheckType

   # Set the after flags package option to true
   set ::Simple::ExtProc::CheckType 1

} -script {

   # Set the after flags package option to 5
   ::Simple::ExtProc::configure -afterflags 5

   # Get the after flags package option
   ::Simple::ExtProc::cget -afterflags

} -cleanup {

   # Restore the after flags package option
   set ::Simple::ExtProc::CheckType ${::Simple::ExtProc::%TMP%}
   unset ::Simple::ExtProc::%TMP%

} -return 5

test-case configure-3 {
   ::Simple::ExtProc::cget
   error, no options
} -script {
   ::Simple::ExtProc::configure
} -error {at least one option required, options are -checktype or -afterflags}

test-case configure-4 {
   ::Simple::ExtProc::cget
   error, incorrect option
} -script {
   ::Simple::ExtProc::configure -foo
} -error {bad option "-foo": must be -checktype or -afterflags}

test-case configure-5 {
   ::Simple::ExtProc::cget
   error, incorrect argument for -checktype
} -script {
   ::Simple::ExtProc::configure -checktype foo
} -error {invalid value "foo" for option "-checktype" of type "-extbool"}

test-case configure-6 {
   ::Simple::ExtProc::cget
   error, incorrect argument for -afterflags
} -script {
   ::Simple::ExtProc::configure -afterflags foo
} -error {invalid value "foo" for option "-afterflags" of type "-int"}

test-case configure-7 {
   ::Simple::ExtProc::cget
   error, incorrect argument for -afterflags, must be greater than one
} -script {
   ::Simple::ExtProc::configure -afterflags 1
} -error {invalid value "1" for option "-afterflags":\
   must be greater than one}

} ;# End of regression testing section
