### ===========================================================================
### -package  : SimpleExtVar
# -version    : 0.2
# -purpose    : Extended format variables handling.
# -overview   :
#     This package allows to handle extended format variables.  Extended
#  variables have not only a name and a value, but also a type and a
#  description, and can be used as non-local Tcl variables or as extended
#  procedure arguments as provided by the SimpleExtProc package.
#
#     Procedures are provided to create, delete and get information about an
#  extended variable, as well as others to convert between extended and Tcl
#  canonical procedure argument formats.
#
# -usage      :
#     Extended format variables are created via P<::Simple::ExtVar::create>. 
#  The declaration specifies whether each variable shall be monitored to
#  ensure its value conforms to its type.  Extended variables can be deleted
#  via K<::Simple::ExtVar::delete>.  The K<::Simple::ExtVar::information>
#  procedure provides several subcommands to query information about extended
#  variables:  K<exists>, K<type>, K<choices>, K<description> and
#  K<monitortype>.  Refer to the description of each procedure below for
#  further information.
#
#     The K<::Simple::ExtVar::convert> pair of procedures convert between
#  extended and Tcl canonical procedure argument formats.
#
#     The extended variables format is very similar to that of the arguments
#  used by the P<opt> package distributed with Tcl 8.0:  {name type ?value? 
#  description}.  There, "name" is the variable name, "type" its type, "value"
#  its value (actual or default, depending on whether we are handling a
#  variable or a procedure argument) and "description" a string describing the
#  variable.
#
#  H2<Variable name>
#
#     This is the variable name.  It must conform to the I<-name> Simple
#  Library type (except for flags and optional arguments, see below).
#
#     Variables whose name starts by a hyphen are flags.  They are allowed
#  when the variable is intended as a procedure argument and indicates that
#  their presence in the procedure call actual argument list is optional. 
#  Variables whose name is enclosed between question marks are optional and
#  are also allowed in argument lists only.  Both the flags and the optional
#  arguments are further described in the P<proc-ext> procedure of the
#  SimpleExtProc package.
#
#  H2<Variable type>
#
#     The variable type must be a type known by the Simple library, that is,
#  any Simple library type or added via the P<::Simple::Type::add> procedure. 
#  Some types are only allowed when the variable is intended as a procedure
#  argument and is thus further described in the P<proc-ext> procedure of the
#  SimpleExtProc package.
#
#  H2<Variable value>
#
#     A variable declaration via the P<::Simple::ExtVar::create> procedure
#  must specify a value.  If the variable type is I<-choice> the value is the
#  first element of the choices list.  When the variable is intended as a
#  procedure argument, this field is regarded as the default value for the
#  variable, which must be optional.  Again, see the P<proc-ext> procedure of
#  the SimpleExtProc package for details.
#
#  H2<Variable description>
#
#     This is simply a description of the variable.
#
# -keywords   : extended variable type
# -variables  :
#
#  { MonitorType      -int 1         {Stores the value of the I<-monitortype>
#                                     package option}}
#
# -commands   :
#
#  * K<::Simple::ExtVar::create> ?-monitortype extbool? extended
#    Creates a non-local extended format variable.
#
#  * K<::Simple::ExtVar::delete> variable
#    Deletes a non-local extended format variable.
#
#  * K<::Simple::ExtVar::convert canonical> extended
#    Converts a variable from extended to canonical format.
#
#  * K<::Simple::ExtVar::convert extended> canonical
#    Converts a variable from canonical to extended format.
#
#  * K<::Simple::ExtVar::information exists> variable
#    Returns whether an extended format variable exists.
#
#  * K<::Simple::ExtVar::information type> variable
#    Returns an extended format variable type.
#
#  * K<::Simple::ExtVar::information choices> variable
#    Returns an extended format variable choices.
#
#  * K<::Simple::ExtVar::information description> variable
#    Returns an extended format variable description.
#
#  * K<::Simple::ExtVar::information monitortype> variable
#    Returns whether an extended format variable is being monitored.
#
#  * K<::Simple::ExtVar::configure> ?-monitortype extbool?
#    Configures the package options.
#
#  * K<::Simple::ExtVar::cget> ?-monitortype?
#    Gets the package options.
#
# -options    :
#
#  * I<-monitortype>:  whether to monitor the extended variables to ensure
#    their values conform to their types.  Notice that only those variables
#    created via the P<::Simple::ExtVar::create> procedure B<after> modifying
#    this option are affected by the change.
#
# -examples   :
#
#  # Install the package
#  package require SimplePackage
#  ::Simple::Package::require-and-install SimpleExtVar
#  
#  # Create an extended variable and monitor its value
#  ::Simple::ExtVar::create -monitortype true\
#     {foobar -choice {foo bar} {may contain foo or bar only}}
#  
#  # Get some information about the just created variable
#  # This displays the following:
#  #    Variable foobar is equal to: foo
#  #    Type: "-choice"
#  #    Description: "may contain foo or bar only"
#  puts "Variable foobar is equal to: $foobar"
#  puts "Type: \"[::Simple::ExtVar::information type foobar]\""
#  puts "Description: \"[::Simple::ExtVar::information description foobar]\""
#  
#  # Set the variable to a valid value
#  # This displays the following:
#  #    bar
#  set foobar bar
#  puts $foobar
#  
#  # Set the variable to an invalid value
#  # This throws an error:
#  #    can't set "foobar": invalid value "gee" for
#  #    type "-choice": must be foo or bar
#  set foobar gee
#
# -details    : 
#  * Each extended format variable internal representation is stored in one
#    element of an array named V<%ExtVars%> in the variable namespace.  The
#    name of the array element is the unqualified variable name.  The array
#    element is a list with the following elements:
#     0: variable type
#     1: variable choices (empty if type is not "-choice")
#     2: variable description
#
# -history    :
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleExtVar) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleExtVar 0.2 -required {
   SimpleError SimplePackage SimpleSubcommand SimpleType
} -namespaces {
   ::Simple::ExtVar
   ::Simple::ExtVar::Priv
} -firsttimeinstall {

### Create the package options
set ::Simple::ExtVar::MonitorType 0

### Declare the package errors
::Simple::Error::declare ::Simple::ExtVar::VALUE-REQUIRED {
   a value is required for %s "%s"
} {
   <item> <name> requires a value but no value was supplied
} {
   Supply a value for the item
}

::Simple::Error::declare ::Simple::ExtVar::ALREADY-EXISTS {
   variable "%s" already exists
} {
   Variable <name> could not be created because it already exists
} {
   Use P<::Simple::ExtVar::delete> to delete the variable
}

::Simple::Error::declare ::Simple::ExtVar::CANT-DELETE {
   can't delete "%s": no such variable
} {
   Variable <name> could not be deleted because is does not exist
}

::Simple::Error::declare ::Simple::ExtVar::OPTIONAL-FLAG {
   cannot have an explicit optional flag "%s"
} {
   Flag <name> was explicitely stated as optional while all flags are
   implicitely optional
} {
   Remove the question marks around the flag name
}

::Simple::Error::declare ::Simple::ExtVar::BAD-TYPE-BOOLFLAG {
   invalid type "-boolflag" for non-flag "%s"
} {
   Non-flag <name> type is "-boolflag" but this type is only valied for flags
} {
   Either convert the variable or argument to a flag or modify its type
}

::Simple::Error::declare ::Simple::ExtVar::AT-LEAST-TWO-CHOICES {
   %s "%s" requires at least two choices
} {
   <item> <name> is of type "-choice" but less than two choices were
   supplied
} {
   Supply at least two choices
}

::Simple::Error::declare ::Simple::ExtVar::CANT-CONVERT-FLAGS {
   cannot convert flags to canonical format
} {
   A flag was tried to be converted to Tcl canonical format {name ?value?}
   flags are not handled by this format
}

::Simple::Error::declare ::Simple::ExtVar::NO-SUCH-EXTENDED-VAR {
   no such extended format variable "%s"
} {
   The extended format variable <name> does not exist
}

::Simple::Error::declare ::Simple::ExtVar::NO-TYPE-CHOICE {
   variable "%s" is not of type "-choice"
} {
   The extended format variable <name> type is not "-choice"
}

::Simple::Error::declare ::Simple::ExtVar::MONITORING-NON-EXTENDED-VAR {
   type monitoring can not be used for non-extended variables
} {
   The procedure which monitors the extended format variables to ensure
   their values conform to their type was somehow called for a non-extended
   variable
} {
   Remove the trace on the non-extended variable
}

::Simple::Error::declare ::Simple::ExtVar::MONITORING-UNSUPPORTED {
   variable being monitored for unsupported operation "%s"
} {
   The procedure which monitors the extended format variables to ensure
   their values conform to their type was somehow called for an <operation>
   other than write
} {
   Remove the trace on the non-extended variable for that operation
}

::Simple::Error::declare ::Simple::ExtVar::BAD-VALUE {
   invalid value "%s" for%s type "%s"%s
} {
   A <variable or argument> value did not conform to its type
} {
   Either modify the <variable or argument> value or type
} {
} {
   invalid value <value> for [<variable or argument> <variable or argument
   name> of] type <type>[:  must be <choices>]
}

::Simple::Error::declare ::Simple::ExtVar::MONITORING-ALIAS {
   type monitoring for aliased variable
} {
   The procedure which monitors the extended format
   variables to ensure their values conform to their type was called for
   a posibly correct extended variable but aliased through the use of
   the K<upvar> or K<global> commands.  Due to a Tcl limitation it is
   then impossible to access the name of the original extended variable,
   and less its type, so the monitorng procedure can not work
} {
   Deactivate the variable type monitoring
}

### ===========================================================================
### -command  : ::Simple::ExtVar::create
# -purpose    : Creates a non-local extended format variable.
# -overview   :
#     Use this procedure to create a non-local (that is, namespace or global)
#  extended format variable. 
#
#     The I<-monitortype> flag allows to override the package option of the
#  same name for the varible being created.  This option controls whether the
#  variable will be monitored to ensure its values conform to its type.  An
#  error is raised if a variable being monitored is being set to a value which
#  does not conform to its type, but the variable still contains the
#  non-conforming value.
#
# -arguments  :
#  {-monitortype      -extbool false {Whether to monitor the variable being
#                                     created to ensure its values conform to
#                                     its type}}
#  { extended         -list          {Extended format variable}}
#
# -details    : 
#     The type monitoring is achieved by setting a write trace on the varible
#  via the K<trace> procedure with the P<::Simple::ExtVar::Priv::monitor> as
#  handler procedure.  This write trace is only set for monitored variables
#  but an unset trace is set for all variables in order to delete their
#  internal representation upon being unset.
#
proc ::Simple::ExtVar::create {
   arg1
   {arg2 {}}
   {arg3 {}}
} {
   ### Parse arguments
   if {![string compare $arg1 -monitortype]} {
      set monitorType [{::Simple::Type::convert boolean} $arg2]
      set extended $arg3
   } else {
      set monitorType $::Simple::ExtVar::MonitorType
      set extended $arg1
   }

   ### Check the variable format
   foreach {isOptional isFlag valueGiven variable value\
      type choices description originalName}\
      [::Simple::ExtVar::Priv::parse variable $extended 1] break

   ### Assert the variable is not optional
   if {$isOptional} {
      ::Simple::Error::throw ::Simple::BAD-ITEM {non-optional variable}\
         ?$variable?
   }

   ### Assert the variable is not a flag
   if {$isFlag} {
      ::Simple::Error::throw ::Simple::BAD-ITEM non-flag $variable
   }

   ### Assert a value has been given
   if {!$valueGiven && [string compare $type -choice]} {
      ::Simple::Error::throw ::Simple::ExtVar::VALUE-REQUIRED variable\
         $variable
   }

   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Assert the variable does not exist
   if {[info exists $variable]} {
      ::Simple::Error::throw ::Simple::ExtVar::ALREADY-EXISTS $variable
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### Store the extended format variable internal representation
   set ${namespace}%ExtVars%($unqualifiedName)\
      [list $type $choices $description]

   ### Set the variable value
   set $variable $value

   ### Monitor type
   if {$monitorType} {

      ### Set a write trace on the variable to monitor 
      ###    its values to ensure they conform to its type
      trace variable $variable w ::Simple::ExtVar::Priv::monitor
   }
}

### ===========================================================================
### -command  : ::Simple::ExtVar::delete
# -purpose    : Deletes a non-local extended format variable.
# -overview   :
#     Use this procedure to delete a non-local extended format variable
#  created via the P<::Simple::ExtVar::create> procedure.
#
# -arguments  :
#  { variable         -name          {Variable name}}
#
proc ::Simple::ExtVar::delete {
   variable
} {
   ### Save the variable name
   ::Simple::Proc::Priv::save-name $variable

   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### The variable exists
   if {[info exists ${namespace}%ExtVars%($unqualifiedName)]}  {

      ### Delete the variable
      unset $variable

      ### Delete the variable internal representation
      unset ${namespace}%ExtVars%($unqualifiedName)

   ### The variable does not exists
   } else  {
      
      ### Throw error
      ::Simple::Error::throw ::Simple::ExtVar::CANT-DELETE\
         [::Simple::Proc::Priv::restore-name]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtVar::Priv::parse
# -purpose    : Parses an extended format variable.
# -overview   :
#     This procedure parses an extended format variable ensuring its format is
#  correct and returns all useful variable details.
#
# -arguments  :
#  { kind             -string        {Kind of variable (variable, argument...)}}
#  { extended         -list          {Extended format variable}}
#  {?qualified?       -boolean 0     {Whether qualified variable names are
#                                     accepted}}
# -returns    : 
#     A list made of the following elements corresponding to the parsed
#  variable:
#     0: Whether the variable is optional.
#     1: Whether the variable is a flag.
#     2: Whether a value for the variable has been given.
#     3: Variable name.
#     4: Variable value.
#     5: Variable type.
#     6: Variable choices (empty if type is not "-choice").
#     7: Variable description.
#     8: Variable original name, including the minus for flags and question
#        marks for optional variables.
#
proc ::Simple::ExtVar::Priv::parse {
   kind
   extended
   {qualified 0}
} {
   ### Check the variable format
   set nTokens [llength $extended]
   if {$nTokens < 3 || $nTokens > 4} {
      ::Simple::Error::throw ::Simple::BAD-ITEM "extended $kind" $extended
   }
   set valueGiven [expr {$nTokens == 4}]

   ### Get tokens
   if {$valueGiven} {
      foreach {name type value description} $extended {}
   } else {
      foreach {name type description} $extended {}
      set value {}
   }

   ### Handle optional variables
   set isOptional [::Simple::Type::is -optional $name]
   if {$isOptional} {
      set name [{::Simple::Type::convert optional} $name]
      if {!$valueGiven} {
         ::Simple::Error::throw ::Simple::ExtVar::VALUE-REQUIRED\
            "optional $kind" $name
      }
   }

   ### Handle flags
   set originalName $name
   set isFlag [::Simple::Type::is -flag $name]
   if {$isFlag} {
      if {!$valueGiven && [string compare $type -boolflag]} {
         ::Simple::Error::throw ::Simple::ExtVar::VALUE-REQUIRED\
            "flag $kind" $originalName
      }
      if {$isOptional} {
         ::Simple::Error::throw ::Simple::ExtVar::OPTIONAL-FLAG ?$name?
      }
      set name [{::Simple::Type::convert flag} $name]
   } else {
      if {![string compare $type -boolflag]} {
         ::Simple::Error::throw ::Simple::ExtVar::BAD-TYPE-BOOLFLAG\
            $originalName
      }
   }

   ### Validate the variable name
   if {($qualified && ![::Simple::Type::is -name $name]) ||
      (!$qualified && ![::Simple::Type::is -unqualifiedname $name])} {

      ::Simple::Error::throw ::Simple::BAD-ITEM "$kind name" $originalName
   }

   ### Validate the type
   ::Simple::Type::Priv::assert-valid-type $type

   ### Assert the type is known
   if {![::Simple::Type::information exists $type]} {
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type
   }

   ### Handle choices
   if {![string compare $type -choice]} {
      if {!$valueGiven || [llength $value] < 2} {
         ::Simple::Error::throw ::Simple::ExtVar::AT-LEAST-TWO-CHOICES\
            $kind $originalName
      }
      set choices $value

      # Default value is first choice
      set value [lindex $choices 0]

      # Not optional nor flag
      if {!$isOptional && !$isFlag} {

         # As if no value given
         set valueGiven 0
      }

   } else {
      set choices {}
   }

   ### Validate the value
   if {$valueGiven && [string compare $type -choice]} {
      ::Simple::ExtVar::Priv::assert-type-conformation\
         $kind $originalName $type $value
   }

   ### Variable is valid
   list $isOptional $isFlag $valueGiven\
      $name $value $type $choices $description $originalName
}

### ===========================================================================
### -command  : ::Simple::ExtVar::convert canonical
# -purpose    : Converts a variable from extended to canonical format.
# -overview   :
#     Use this procedure to convert extended variables to canonical format
#  (the format used by the K<proc> command arguments, {name ?default?}).
#
# -arguments  :
#  { extended         -list          {Extended format variable}}
#
# -returns    : The variable in canonical format.
#
proc-sub {::Simple::ExtVar::convert canonical} {
   extended
} {
   ### Parse the variable
   foreach {isOptional isFlag valueGiven name value\
      type choices description originalName}\
      [::Simple::ExtVar::Priv::parse variable $extended 1] break

   ### Assert it is not a flag
   if {$isFlag} {
      ::Simple::Error::throw ::Simple::ExtVar::CANT-CONVERT-FLAGS
   }

   ### Variable name and value
   if {$valueGiven} {

      # {name value}
      set answer [list $name $value]

   ### Variable name only
   } else {

      # name
      set answer $name
   }

   ### Return the variable in canonical format
   set answer
}

### ===========================================================================
### -command  : ::Simple::ExtVar::convert extended
# -purpose    : Converts a variable from canonical to extended format.
# -overview   :
#     Use this procedure to convert canonical variables (the format used by
#  the K<proc> command arguments, {name ?default?}) to extended format.
#
# -arguments  :
#  { canonical        -list          {Canonical variable}}
#
# -returns    : The variable in extended format.
#
proc-sub {::Simple::ExtVar::convert extended} {
   canonical
} {
   set length [llength $canonical]
   if {$length < 1 || $length > 2} {
      ::Simple::Error::throw ::Simple::BAD-ITEM\
         {canonical variable} $canonical
   }
   
   ### Validate the variable name
   set name [lindex $canonical 0]
   if {![::Simple::Type::is -name $name]} {
      ::Simple::Error::throw ::Simple::BAD-ITEM {variable name} $name
   }

   ### Variable name and value
   if {$length == 2} {

      # {?name? -any value ""}
      set answer [list ?$name? -any [lindex $canonical 1] {}]

   ### Variable name only
   } else {

      # {name -any ""}
      set answer [list $name -any {}]
   }

   ### Return the variable in extended format
   set answer
}

### ===========================================================================
### -command  : ::Simple::ExtVar::Priv::assert-type-conformation
# -purpose    : Throws an error if a variable value does not conform to a type.
# -arguments  :
#  { kind             -string        {Kind of variable (variable, argument...)}}
#  { variable         -name          {Variable name}}
#  { type             -type          {Type}}
#  { value            -any           {Value}}
#  {?choices?         -list {}       {Choices}}
#
# -returns    : Error if the variable value does not conform to the type.
#
proc ::Simple::ExtVar::Priv::assert-type-conformation {
   kind
   variable
   type
   value
   {choices {}}
} {
   ### The value does not match the type
   if {![::Simple::Type::is $type $value $choices]} {

      ### Called from P<::Simple::ExtVar::Priv::monitor>
      if {[string compare [::Simple::Proc::caller]\
         ::Simple::ExtVar::Priv::monitor]} {

         set name " $kind \"$variable\" of"
      } else {
         set name {}
      }

      ### Choices given
      if {[llength $choices] != 0} {
         set choices ": must be [::Simple::Proc::Priv::enumerate $choices]"
      } else {
         set choices {}
      }

      ### Throw error
      ::Simple::Error::throw ::Simple::ExtVar::BAD-VALUE\
         $value $name $type $choices
   }
}

### ===========================================================================
### -command  : ::Simple::ExtVar::information exists
# -purpose    : Returns whether an extended format variable exists.
# -arguments  :
#  { variable         -name          {Variable name}}
#
# -returns    : Whether there is an extended format variable with that name.
#
proc-sub {::Simple::ExtVar::information exists} {
   variable
} {
   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### Return whether there is an extended format variable with that name
   info exists ${namespace}%ExtVars%($unqualifiedName)
}

### ===========================================================================
### -command  : ::Simple::ExtVar::Priv::assert-existance
# -purpose    : Throws an error if an extended format variable does not exist.
# -arguments  :
#  { namespace        -name          {Variable namespace}}
#  { unqualifiedName  -name          {Variable unqualified name}}
#
# -returns    : Error if the extended format variable does not exist.
#
proc ::Simple::ExtVar::Priv::assert-existance {
   namespace
   unqualifiedName
} {
   ### Assert the variable exists
   if {![info exists ${namespace}%ExtVars%($unqualifiedName)]} {
      ::Simple::Error::throw ::Simple::ExtVar::NO-SUCH-EXTENDED-VAR\
         [::Simple::Proc::Priv::restore-name]
   }
}

### ===========================================================================
### -command  : ::Simple::ExtVar::information type
# -purpose    : Returns an extended format variable type.
# -arguments  :
#  { variable         -name          {Variable name}}
#
# -returns    : The extended format variable type.
#
proc-sub {::Simple::ExtVar::information type} {
   variable
} {
   ### Save the variable name
   ::Simple::Proc::Priv::save-name $variable

   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### Assert the variable exists
   ::Simple::ExtVar::Priv::assert-existance $namespace $unqualifiedName

   ### Return the variable type
   lindex [set ${namespace}%ExtVars%($unqualifiedName)] 0
}

### ===========================================================================
### -command  : ::Simple::ExtVar::information choices
# -purpose    : Returns an extended format variable choices.
# -arguments  :
#  { variable         -name          {Variable name}}
#
# -returns    : The extended format variable choices.
#
proc-sub {::Simple::ExtVar::information choices} {
   variable
} {
   ### Save the variable name
   ::Simple::Proc::Priv::save-name $variable

   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### Assert the variable exists
   ::Simple::ExtVar::Priv::assert-existance $namespace $unqualifiedName

   ### Assert the variable type is "-choice"
   if {[string compare [lindex [set\
      ${namespace}%ExtVars%($unqualifiedName)] 0] -choice]} {

      ::Simple::Error::throw ::Simple::ExtVar::NO-TYPE-CHOICE\
         [::Simple::Proc::Priv::restore-name]
   }

   ### Return the list of variable choices
   lindex [set ${namespace}%ExtVars%($unqualifiedName)] 1
}

### ===========================================================================
### -command  : ::Simple::ExtVar::information description
# -purpose    : Returns an extended format variable description.
# -arguments  :
#  { variable         -name          {Variable name}}
#
# -returns    : The extended format variable description.
#
proc-sub {::Simple::ExtVar::information description} {
   variable
} {
   ### Save the variable name
   ::Simple::Proc::Priv::save-name $variable

   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### Assert the variable exists
   ::Simple::ExtVar::Priv::assert-existance $namespace $unqualifiedName

   ### Return the variable description
   lindex [set ${namespace}%ExtVars%($unqualifiedName)] 2
}

### ===========================================================================
### -command  : ::Simple::ExtVar::information monitortype
# -purpose    : Returns whether an extended format variable is being monitored.
# -overview   : 
#     This procedure returns whether an extended format variable variable is
#  being monitored to ensure its values conform to its type, that is, was
#  created with the I<-monitortype> qualifier of the
#  P<::Simple::ExtVar::create> procedure.
#
# -arguments  :
#  { variable         -name          {Variable name}}
#
# -returns    : Whether the extended format variable is being monitored.
#
proc-sub {::Simple::ExtVar::information monitortype} {
   variable
} {
   ### Save the variable name
   ::Simple::Proc::Priv::save-name $variable

   ### Qualify the variable name in the calling scope
   if {![string match ::* $variable]} {
      set variable [::Simple::Namespace::Priv::qualify $variable]
   }

   ### Split the variable name
   regexp -- ^(.*::)(.*)$ $variable tmp namespace unqualifiedName

   ### Assert the variable exists
   ::Simple::ExtVar::Priv::assert-existance $namespace $unqualifiedName

   ### Return whether the variable is being monitored to
   ###    ensure its values conform to its type
   #   that is, whether there is a write trace on the variable 
   #   directed to the monitoring procedure
   expr {[lsearch -exact [trace vinfo $variable]\
      {w ::Simple::ExtVar::Priv::monitor}] != -1}
}

### ===========================================================================
### -command  : ::Simple::ExtVar::Priv::monitor
# -purpose    : Monitors extended format variables.
# -overview   :
#     This procedure is called whenever an extended format variable created
#  through the P<::Simple::ExtVar::create> and being monitored is set.  The
#  new value is checked to ensure it conforms to the variable type, an error
#  being raised otherwise.
#
# -arguments  :
#  { var1Name         -name          {Variable name}}
#  { var2Name         -name          {Second variable name (not used)}}
#  { operation        -string        {Operation performed on the variable}}
#
# -details    : 
#     This procedure gets called through the K<trace> command set by the
#  P<::Simple::ExtVar::create> procedure.
#
proc ::Simple::ExtVar::Priv::monitor {
   var1Name
   var2Name
   operation
} {
   ### Qualify the variable in the calling scope
   set name $var1Name
   if {![string match ::* $name]} {
      set name [::Simple::Namespace::Priv::qualify $name]
   }

   ### The variable exists
   if {[info exists $name]} {

      ### Split the variable name
      foreach {namespace unqualifiedName}\
         [::Simple::Error::Priv::split-name $var1Name] break

      ### Assert the variable has extended format 
      if {![info exists ${namespace}%ExtVars%($unqualifiedName)]} {
         ::Simple::Error::throw ::Simple::ExtVar::MONITORING-NON-EXTENDED-VAR
      }

      ### Get the variable internal representation
      set variable [set ${namespace}%ExtVars%($unqualifiedName)]

      ### Variable is being written
      if {![string compare $operation w]} {

         ### Get the variable type
         set type [lindex $variable 0]

         ### Get the variable value
         upvar $var1Name value

         ### Type is not "-choice"
         if {[string compare $type -choice]} {

            ### Assert the value conforms to the type
            ::Simple::ExtVar::Priv::assert-type-conformation\
               variable $name $type $value

         ### Type is "-choice"
         } else {

            ### Get the variable choices
            set choices [lindex $variable 1]

            ### Assert the value conforms to the type
            ::Simple::ExtVar::Priv::assert-type-conformation\
               variable $name $type $value $choices
         }

      ### Variable is not being written
      } else {

         ### Throw error
         ::Simple::Error::throw ::Simple::ExtVar::MONITORING-UNSUPPORTED\
            $operation
      }

   ### The variable does not exist
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::ExtVar::MONITORING-ALIAS
   }
}

### ===========================================================================
### -command  : ::Simple::ExtVar::configure
# -purpose    : Configures the package options.
#
# -arguments  :
#  {-monitortype      -extbool false {Whether to monitor the extended variables
#                                     to ensure their values conform to their
#                                     types}}
#
proc ::Simple::ExtVar::configure {
   {arg1 {}}
   {arg2 {}}
} {
   ### No arguments
   if {![string compare $arg1 {}]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::AT-LEAST-ONE-OPTION\
         [::Simple::ExtVar::cget]

   } else {

      ### Switch on flag
      switch -exact -- $arg1 {

         -monitortype {
         ### Handle -monitortype

            ### Invalid argument
            if {![::Simple::Type::is -extbool $arg2]} {

               ### Throw error
               ::Simple::Error::throw ::Simple::BAD-OPTION-ARG-TYPE\
                  $arg2 -monitortype -extbool
            }

            ### Set the monitor type option
            set ::Simple::ExtVar::MonitorType\
               [{::Simple::Type::convert boolean} $arg2]
         }

         default {
         ### default, throw error
            ::Simple::Error::throw ::Simple::BAD-OPTION\
               $arg1 [::Simple::ExtVar::cget]
         }
      }
   }
}

### ===========================================================================
### -command  : ::Simple::ExtVar::cget
# -purpose    : Gets the package options.
# -arguments  :
#  {-monitortype      -boolflag      {Gets whether to monitor the extended
#                                     variables to ensure their values conform
#                                     to their types}}
#
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::ExtVar::cget {
   {arg1 {}}
} {
   ### No arguments
   if {![string compare $arg1 {}]} {

      ### Return the list of options
      return -monitortype

   } else {

      ### Switch on flag
      switch -exact -- $arg1 {

         -monitortype {
         ### Handle -monitortype
            set ::Simple::ExtVar::MonitorType
         }

         default {
         ### default, throw error
            ::Simple::Error::throw ::Simple::BAD-OPTION\
               $arg1 [::Simple::ExtVar::cget]
         }
      }
   }
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleExtVar-test 0.2

### ===========================================================================
### -test     : ::Simple::ExtVar::create
test-case create-1 {
   ::Simple::ExtVar::create
} -setup {

   # Create a namespace
   namespace eval ::foo {}

} -script {

   # Create an unqualified variable
   ::Simple::ExtVar::create {foo -int 99 {A foo}}

   # Access the variable
   puts $foo

   # Create a qualified variable
   ::Simple::ExtVar::create {::foo::bar -int 77 {A bar}}

   # Access the variable
   puts $foo::bar

   # Create an unqualified variable within a namespace
   namespace eval ::foo {
      ::Simple::ExtVar::create {gee -int 55 {A gee}}
   }

   # Access the variable
   puts $foo::gee

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

   # Delete the variable
   unset foo

} -output {99
77
55
}

test-case create-2 {
   ::Simple::ExtVar::create
   type choice
} -script {

   # Create a variable
   ::Simple::ExtVar::create {foo -choice {foo bar} {A foobar}}

   # Access the variable
   set foo

} -cleanup {

   # Delete the variable
   unset foo

} -return foo

test-case create-3 {
   ::Simple::ExtVar::create
   with and without type monitoring
} -script {

   # Create a variable with type monitoring set to false
   ::Simple::ExtVar::create -monitortype false {foo -int 77 {A foo}}

   # Set the variable to a value which does not conform to its type
   set foo gee

   # Access the variable
   puts $foo

   # Create a variable with type monitoring set to true
   ::Simple::ExtVar::create -monitortype true {bar -int 77 {A bar}}

   # Set the variable to a value which does not conform to its type
   set bar gee

} -cleanup {

   # Delete the variables
   ::Simple::ExtVar::delete foo
   ::Simple::ExtVar::delete bar

} -output {gee
} -error {can't set "bar": invalid value "gee" for type "-int"}

test-case create-4 {
   ::Simple::ExtVar::create
   deleting a namespace deletes the extended format variables therein
} -setup {

   # Create a variable in a namespace
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -int 77 {A bar}}
   }

} -script {

   # The extended variable exists
   puts -nonewline [::Simple::ExtVar::information exists ::foo::bar]

   # Delete the namespace
   namespace delete ::foo

   # The extended variable no longer exists
   puts -nonewline [::Simple::ExtVar::information exists ::foo::bar]

} -output 10

test-case create-5 {
   ::Simple::ExtVar::create
   error, bad variable
} -script {
   ::Simple::ExtVar::create {}
} -error {expected extended variable but got ""}

test-case create-6 {
   ::Simple::ExtVar::create
   error, required value
} -script {
   ::Simple::ExtVar::create {foo -int {A foo}}
} -error {a value is required for variable "foo"}

test-case create-7 {
   ::Simple::ExtVar::create
   error, type -boolflag
} -script {
   ::Simple::ExtVar::create {foo -boolflag true {A foo}}
} -error {invalid type "-boolflag" for non-flag "foo"}

test-case create-8 {
   ::Simple::ExtVar::create
   error, optional variable not allowed
} -script {
   ::Simple::ExtVar::create {?foo? -int 77 {A foo}}
} -error {expected non-optional variable but got "?foo?"}

test-case create-9 {
   ::Simple::ExtVar::create
   error, optional variable value required
} -script {
   ::Simple::ExtVar::create {?foo? -int {A foo}}
} -error {a value is required for optional variable "foo"}

test-case create-10 {
   ::Simple::ExtVar::create
   error, flag not allowed
} -script {
   ::Simple::ExtVar::create {-foo -int 77 {A foo}}
} -error {expected non-flag but got "foo"}

test-case create-11 {
   ::Simple::ExtVar::create
   error, flag value required
} -script {
   ::Simple::ExtVar::create {-foo -int {A foo}}
} -error {a value is required for flag variable "-foo"}

test-case create-12 {
   ::Simple::ExtVar::create
   error, explicit optional flag not allowed
} -script {
   ::Simple::ExtVar::create {?-foo? -int 77 {A foo}}
} -error {cannot have an explicit optional flag "?-foo?"}

test-case create-13 {
   ::Simple::ExtVar::create
   error, bad variable name
} -script {
   ::Simple::ExtVar::create {@foo -int 77 {A foo}}
} -error {expected variable name but got "@foo"}

test-case create-14 {
   ::Simple::ExtVar::create
   error, unknown type
} -script {
   ::Simple::ExtVar::create {foo -foo 77 {A foo}}
} -error {non-existing type "-foo"}

test-case create-15 {
   ::Simple::ExtVar::create
   error, bad type
} -script {
   ::Simple::ExtVar::create {foo @int 77 {A foo}}
} -error {expected type name but got "@int"}

test-case create-16 {
   ::Simple::ExtVar::create
   error, at least two choices
} -script {
   ::Simple::ExtVar::create {foo -choice foo {A foo}}
} -error {variable "foo" requires at least two choices}

test-case create-17 {
   ::Simple::ExtVar::create
   error, invalid value
} -script {
   ::Simple::ExtVar::create {foo -int bar {A foo}}
} -error {invalid value "bar" for variable "foo" of type "-int"}

test-case create-18 {
   ::Simple::ExtVar::create
   error, existing variable
} -setup {

   # Create a namespace with a variable
   namespace eval ::foo {
      variable bar 0
   }

} -script {
   ::Simple::ExtVar::create {::foo::bar -int 99 {A foo}}
} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -error {variable "::foo::bar" already exists}

test-case create-19 {
   ::Simple::ExtVar::create
   error, aliased variable
} -setup {

   # Create a variable
   ::Simple::ExtVar::create -monitortype true {foo -int 99 {A foo}}

   # Create a procedure which creates an alias to the variable
   proc foo {} {
      upvar foo bar
      set bar foo
   }

} -script {
   foo
} -cleanup {

   # Delete the variable
   ::Simple::ExtVar::delete foo

   # Delete the procedure
   rename foo {}

} -error {can't set "bar": type monitoring for aliased variable}

### ===========================================================================
### -test     : ::Simple::ExtVar::delete
test-case delete-1 {
   ::Simple::ExtVar::delete
} -setup {

   # Create two variables
   ::Simple::ExtVar::create {foo -int 99 {A foo}}
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -int 77 {A bar}}
   }

} -script {

   # Access the first variable
   puts $foo

   # Delete the first variable
   ::Simple::ExtVar::delete foo

   # Access the first variable
   catch {set foo} result
   puts $result

   # Access the second variable
   puts $::foo::bar

   # Delete the second variable
   ::Simple::ExtVar::delete ::foo::bar

   # Access the second variable
   catch {set bar} result
   puts $result

} -output {99
can't read "foo": no such variable
77
can't read "bar": no such variable
}

test-case delete-2 {
   ::Simple::ExtVar::delete
   error, non-existing variable
} -script {
   ::Simple::ExtVar::delete foo
} -error {can't delete "foo": no such variable}

### ===========================================================================
### -test     : ::Simple::ExtVar::convert canonical
test-case canonical-1 {
   ::Simple::ExtVar::convert canonical
} -script {
   puts [::Simple::ExtVar::convert canonical {foo -int {A foo}}]
   puts [::Simple::ExtVar::convert canonical {::foo::bar -int {A bar}}]
   puts [::Simple::ExtVar::convert canonical {?foo? -int 2 {A foo}}]
} -output {foo
::foo::bar
foo 2
}

test-case canonical-2 {
   ::Simple::ExtVar::canonical
   error, bad variable
} -script {
   ::Simple::ExtVar::convert canonical {}
} -error {expected extended variable but got ""}

test-case canonical-3 {
   ::Simple::ExtVar::canonical
   error, flag not allowed
} -script {
   ::Simple::ExtVar::convert canonical {-foo -int 2 "A foo"}
} -error {cannot convert flags to canonical format}

### ===========================================================================
### -test     : ::Simple::ExtVar::convert extended
test-case extended-1 {
   ::Simple::ExtVar::convert extended
} -script {
   puts [::Simple::ExtVar::convert extended foo]
   puts [::Simple::ExtVar::convert extended ::foo::bar]
   puts [::Simple::ExtVar::convert extended {foo 2}]
   puts [::Simple::ExtVar::convert extended {::foo::bar 2}]
} -output {foo -any {}
::foo::bar -any {}
?foo? -any 2 {}
?::foo::bar? -any 2 {}
}

test-case extended-2 {
   ::Simple::ExtVar::extended
   error, bad variable
} -script {
   ::Simple::ExtVar::convert extended {}
} -error {expected canonical variable but got ""}

### ===========================================================================
### -test     : ::Simple::ExtVar::information exists
test-case exists-1 {
   ::Simple::ExtVar::information exists
} -setup {

   # Create two variables
   ::Simple::ExtVar::create {foo -int 99 {A foo}}
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -int 77 {A bar}}
   }

} -script {

   # Existing extended variables
   puts -nonewline [::Simple::ExtVar::information exists foo]
   puts -nonewline [::Simple::ExtVar::information exists ::foo::bar]
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtVar::information exists bar]
   }

   # Delete the variables
   ::Simple::ExtVar::delete foo
   ::Simple::ExtVar::delete ::foo::bar

   # Non-existing extended variables
   set gee 1
   puts -nonewline [::Simple::ExtVar::information exists gee]
   puts -nonewline [::Simple::ExtVar::information exists foo]
   puts -nonewline [::Simple::ExtVar::information exists ::foo::bar]
   namespace eval ::foo {
      puts -nonewline [::Simple::ExtVar::information exists bar]
   }

} -output 1110000

### ===========================================================================
### -test     : ::Simple::ExtVar::information type
test-case type-1 {
   ::Simple::ExtVar::information type
} -setup {

   # Create two variables
   ::Simple::ExtVar::create {foo -int 99 {A foo}}
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -string BAR {A bar}}
   }

} -script {

   # Get the variables types
   puts [::Simple::ExtVar::information type foo]
   puts [::Simple::ExtVar::information type ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtVar::information type bar]
   }

} -cleanup {

   # Delete the variables
   ::Simple::ExtVar::delete foo
   ::Simple::ExtVar::delete ::foo::bar

} -output {-int
-string
-string
}

test-case type-2 {
   ::Simple::ExtVar::information type
   error, non-existing variable
} -script {
   ::Simple::ExtVar::information type foo
} -error {no such extended format variable "foo"}

### ===========================================================================
### -test     : ::Simple::ExtVar::information choices
test-case choices-1 {
   ::Simple::ExtVar::information choices
} -setup {

   # Create a variable
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -choice {foo bar} {A foobar}}
   }

} -script {

   # Get the variables choices
   puts [::Simple::ExtVar::information choices ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtVar::information choices bar]
   }

} -cleanup {

   # Delete the variable
   ::Simple::ExtVar::delete ::foo::bar

} -output {foo bar
foo bar
}

test-case choices-2 {
   ::Simple::ExtVar::information choices
   error, non-existing variable
} -script {
   ::Simple::ExtVar::information choices foo
} -error {no such extended format variable "foo"}

test-case choices-3 {
   ::Simple::ExtVar::information choices
   error, not of type -choice
} -setup {

   # Create a variable
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -string bar {A bar}}
   }

} -script {

   # Get the variable choices
   puts [::Simple::ExtVar::information choices ::foo::bar]

} -cleanup {

   # Delete the variable
   ::Simple::ExtVar::delete ::foo::bar

} -error {variable "::foo::bar" is not of type "-choice"}

### ===========================================================================
### -test     : ::Simple::ExtVar::information description
test-case description-1 {
   ::Simple::ExtVar::information description
} -setup {

   # Create two variables
   ::Simple::ExtVar::create {foo -int 99 {A foo}}
   namespace eval ::foo {
      ::Simple::ExtVar::create {bar -string BAR {A bar}}
   }

} -script {

   # Get the variables descriptions
   puts [::Simple::ExtVar::information description foo]
   puts [::Simple::ExtVar::information description ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtVar::information description bar]
   }

} -cleanup {

   # Delete the variables
   ::Simple::ExtVar::delete foo
   ::Simple::ExtVar::delete ::foo::bar

} -output {A foo
A bar
A bar
}

test-case description-2 {
   ::Simple::ExtVar::information description
   error, non-existing variable
} -script {
   ::Simple::ExtVar::information description foo
} -error {no such extended format variable "foo"}

### ===========================================================================
### -test     : ::Simple::ExtVar::information monitortype
test-case monitortype-1 {
   ::Simple::ExtVar::information monitortype
} -setup {

   # Create a non-monitored variable
   ::Simple::ExtVar::create -monitortype false {foo -int 99 {A foo}}

   # Create a monitored variable
   namespace eval ::foo {
      ::Simple::ExtVar::create -monitortype true {bar -string BAR {A bar}}
   }

} -script {

   # Get whether the variables are being monitored
   puts [::Simple::ExtVar::information monitortype foo]
   puts [::Simple::ExtVar::information monitortype ::foo::bar]
   namespace eval ::foo {
      puts [::Simple::ExtVar::information monitortype bar]
   }

} -cleanup {

   # Delete the variables
   ::Simple::ExtVar::delete foo
   ::Simple::ExtVar::delete ::foo::bar

} -output {0
1
1
}

### ===========================================================================
### -test     : ::Simple::ExtVar::cget
test-case cget-1 {
   ::Simple::ExtVar::cget
   -monitortype
} -setup {

   # Save the monitor type package option
   set ::Simple::ExtVar::%TMP% $::Simple::ExtVar::MonitorType

   # Set the monitor type package option to true
   set ::Simple::ExtVar::MonitorType 1

} -script {
   ::Simple::ExtVar::cget -monitortype
} -cleanup {

   # Restore the monitor type package option
   set ::Simple::ExtVar::MonitorType ${::Simple::ExtVar::%TMP%}
   unset ::Simple::ExtVar::%TMP%

} -return 1

test-case cget-2 {
   ::Simple::ExtVar::cget
   no arguments
} -script {
   ::Simple::ExtVar::cget
} -return -monitortype

test-case cget-3 {
   ::Simple::ExtVar::cget
   error, incorrect option
} -script {
   ::Simple::ExtVar::cget -foo
} -error {bad option "-foo": must be -monitortype}

### ===========================================================================
### -test     : ::Simple::ExtVar::configure
test-case configure-1 {
   ::Simple::ExtVar::configure
   -monitortype
} -setup {

   # Save the monitor type package option
   set ::Simple::ExtVar::%TMP% $::Simple::ExtVar::MonitorType

   # Set the monitor type package option to true
   set ::Simple::ExtVar::MonitorType 1

} -script {

   # Set the monitor type package option to true
   ::Simple::ExtVar::configure -monitortype true

   # Get the monitor type package option
   ::Simple::ExtVar::cget -monitortype

} -cleanup {

   # Restore the monitor type package option
   set ::Simple::ExtVar::MonitorType ${::Simple::ExtVar::%TMP%}
   unset ::Simple::ExtVar::%TMP%

} -return 1

test-case configure-2 {
   ::Simple::ExtVar::cget
   error, no options
} -script {
   ::Simple::ExtVar::configure
} -error {at least one option required, options are -monitortype}

test-case configure-3 {
   ::Simple::ExtVar::cget
   error, incorrect option
} -script {
   ::Simple::ExtVar::configure -foo
} -error {bad option "-foo": must be -monitortype}

test-case configure-4 {
   ::Simple::ExtVar::cget
   error, incorrect option argument
} -script {
   ::Simple::ExtVar::configure -monitortype foo
} -error {invalid value "foo" for option "-monitortype" of type "-extbool"}

} ;# End of regression testing section
