### ===========================================================================
### -package  : SimpleNamespace
# -version    : 0.2
# -purpose    : Namespaces and qualified names related utilities.
# -overview   :
#     This package provides utilities for namespaces and qualified names
#  handling.  Procedures are provided for qualifying names, obtaining
#  qualified names qualifiers and assessing whether a namespace exists.
#
# -keywords   : namespace qualified qualifiers
# -commands   :
#
#  * K<::Simple::Namespace::qualify> name ?level?
#    Fully-qualifies a name up the calling stack frame.
#
#  * K<::Simple::Namespace::qualifiers> name
#    Returns leading namespace qualifiers.
#
#  * K<::Simple::Namespace::information exists> namespace
#    Returns whether a namespace exists.
#
#  * K<::Simple::Namespace::configure>
#    Configures the package options.
#
#  * K<::Simple::Namespace::cget>
#    Gets the package options.
#
# -seealso    :
#  * The P<::Simple::Error::Priv::split-name> procedure.
#
# -history    :
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleNamespace) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleNamespace 0.2 -required {
   SimpleError SimplePackage
} -namespaces {
   ::Simple::Namespace
   ::Simple::Namespace::Priv
} -firsttimeinstall {

### ===========================================================================
### -command  : ::Simple::Namespace::qualify
# -purpose    : Fully-qualifies a name up the calling stack frame.
# -overview   :
#     This procedure fully-qualifies a variable or procedure name.  It is
#  somewhat similar to the P<auto_qualify> procedure but the namespace is
#  taken from a scope up the calling stack.  By default, the scope of the
#  procedure which called the one which calls this one is used, that is, two
#  levels up the stack frame.
#
# -keywords   : namespace qualify qualifier
# -arguments  :
#  { name             -name          {Variable or procedure name}} 
#  {?level?           -int 1         {Levels up the stack in which to operate}}
#
# -returns    : The fully-qualified name.
#
# -limitations:
#  * This procedure must be called from another procedure, otherwise there is
#    no scope two levels up the stack.  If the A<level> optional argument is
#    specified, the corresponding number of levels up the stack are necessary.
#
# -seealso    :
#  * The P<auto_qualify> procedure.
#
#  * The P<::Simple::Namespace::Priv::qualify> procedure.
#
proc ::Simple::Namespace::qualify {
   name
   {level 1}
} {
   ### Relative name
   if {![string match ::* $name]} {

      ### Qualify the name a scope further up the calling stack
      ::Simple::Namespace::Priv::qualify $name [incr level]

   ### Absolute namespace
   } else {

      ### Return name
      set name
   }
}

### ===========================================================================
### -command  : ::Simple::Namespace::Priv::qualify
# -purpose    : Fully-qualifies a name up the calling stack frame.
# -overview   :
#     This procedure fully-qualifies a variable or procedure name.  It is
#  somewhat similar to the P<auto_qualify> procedure but the namespace is
#  taken from a scope up the calling stack.  By default, the scope of the
#  procedure which called the one which calls this one is used, that is, two
#  levels up the stack frame.
#
# -arguments  :
#  { name             -unqualifiedname
#                                    {Unqualified variable or procedure name}}
#  {?level?           -int 1         {Levels up the stack in which to operate}}
#
# -returns    : The fully-qualified name.
#
# -limitations:
#  * This procedure must be called from another procedure, otherwise there is
#    no scope two levels up the stack.  If the A<level> optional argument is
#    specified, the corresponding number of levels up the stack are necessary.
#
# -remarks    :
#  * Notice that the given name is assumed to be unqualified.
#
# -seealso    :
#  * The P<auto_qualify> procedure.
#
#  * The P<::Simple::Namespace::qualify> procedure.
#
proc ::Simple::Namespace::Priv::qualify {
   name
   {level 1}
} {
   ### Take the namespace from a level up the stack frame
   #   This throws a "bad level <level>" error is this procedure is not
   #   called from within another one.
   set namespace [uplevel [incr level] namespace current]

   ### Namespace is not "::"
   if {[string compare $namespace ::]} {

      ### namespace::name
      set answer ${namespace}::$name

   ### Namespace is "::"
   } else {

      ### ::name
      set answer ::$name
   }

   ### Return the fully-qualified name
   set answer
}

### ===========================================================================
### -command  : ::Simple::Namespace::qualifiers
# -purpose    : Returns leading namespace qualifiers.
# -overview   :
#     This procedure returns any leading namespace qualifers for a variable or
#  procedure name.  It is similar to the K<namespace qualifiers> command but
#  works in such a way that for any name the concatenation of the value
#  returned by this procedure and the result of I<[namespace tail $name]>
#  equals the original name.  See the examples section below.
#
# -keywords   : namespace qualify qualifier
# -arguments  :
#  { name             -name          {Variable or procedure name}}
#
# -returns    : The leading namespace qualifers for the given name.
# -examples   :
#
#  name          namespace qualifiers  Namespace::qualifers  namespace tail
#  ------------  --------------------  --------------------  --------------
#  {}                 {}                    {}                    {}
#  foo                {}                    {}                    foo
#  foo::bar           foo                   foo::                 bar
#  foo::::bar         foo                   foo::::               bar
#  ::                 {}                    ::                    {}
#  ::foo              {}                    ::                    foo
#  ::foo::            ::foo                 ::foo::               {}
#  ::foo::bar         ::foo                 ::foo::               bar
#  ::foo::bar::       ::foo::bar            ::foo::bar::          {}
#
proc ::Simple::Namespace::qualifiers {
   name
} {
   if {[regexp -- {^(.*::).*$} $name tmp namespace]} {
      set namespace
   }
}

### ===========================================================================
### -command  : ::Simple::Namespace::information
# -purpose    : Base command for package K<information> subcommands.
# -access     : Private
# -overview   :
#     This is the base command for the K<::Simple::Namespace::information>
#  subcommand procedures.
#
# -arguments  :
#  { args             -list          {Subcommand and subcommand arguments}}
#
proc ::Simple::Namespace::information {
   args
} {
   #========= %BASE-COMMAND%

   ### Get the subcommand and full command names
   set subcommand [lindex $args 0]
   set fullCommand ::Simple::Namespace::information\ $subcommand

   ### Evaluate the subcommand procedure
   ### Error from subcommand procedure
   #   The uplevel here ensures that the call to the base command
   #   is hidden to the command procedure body
   if {[catch {uplevel [list $fullCommand] [lrange $args 1 end]} result]} {

      ### No subcommand given
      if {[llength $args] == 0} {

         ### Throw error
         ::Simple::Error::throw ::Simple::NO-SUBCOMMAND\
            ::Simple::Namespace::information

      ### Non-existing subcommand
      } elseif {![string compare "invalid command name \"$fullCommand\""\
         $result]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND $subcommand exists

      ### Invalid arguments
      } elseif {![string compare "called \"$fullCommand\" with too many\
         arguments" $result] || [string match {no value given for\
         parameter*} $result]} {

         ### Compose the correct arguments
         switch -exact -- $subcommand {
            exists {
               set arguments { namespace}
            }
         }

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND-ARGS\
            $fullCommand $arguments

      ### Other error
      } else {

         ### Rethrow the subcommand procedure error
         return -code error -errorcode $::errorCode $result
      }

   ### No error from subcommand procedure
   } else {

      ### Return the subcommand procedure return value
      set result
   }
}

### ===========================================================================
### -command  : ::Simple::Namespace::information exists
# -purpose    : Returns whether a namespace exists.
# -arguments  :
#  { namespace        -namespacename {Namespace}}
#
# -returns    : Whether the namespace exists.
#
proc {::Simple::Namespace::information exists} {
   namespace
} {
   if {[string compare $namespace {}]} {
      expr {![catch {namespace parent $namespace}]}
   } else {
      set answer 1
   }
}

### ===========================================================================
### -command  : ::Simple::Namespace::configure
# -purpose    : Configures the package options.
#
proc ::Simple::Namespace::configure {
} {
   ::Simple::Error::throw ::Simple::NO-OPTIONS
}

### ===========================================================================
### -command  : ::Simple::Namespace::cget
# -purpose    : Gets the package options.
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Namespace::cget {
} {
   list
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleNamespace-test 0.2

### ===========================================================================
### -test     : ::Simple::Namespace::qualify
test-case qualify-1 {
   ::Simple::Namespace::qualify
} -setup {

   # Create a hierarchy of procedures
   namespace eval foo {}
   namespace eval bar {}
   namespace eval gee {}
   proc ::foo::proc1 {} {
      puts [::Simple::Namespace::qualify foo]
      puts [::Simple::Namespace::qualify ::foo]
      puts [::Simple::Namespace::qualify foo 2]
   }
   proc ::bar::proc2 {} {
      ::foo::proc1
      puts [::Simple::Namespace::qualify foo]
      puts [::Simple::Namespace::qualify foo::::bar]
      puts [::Simple::Namespace::qualify ::foo]
      puts [::Simple::Namespace::qualify ::foo::bar]
   }
   proc ::gee::proc3 {} {
      namespace eval ::bar {
         proc2
      }
      ::bar::proc2
   }

} -script {

   # Call the hierarchy of procedures
   ::gee::proc3

} -cleanup {

   # Delete the procedures
   namespace delete foo
   namespace delete bar
   namespace delete gee

} -output {::bar::foo
::foo
::bar::foo
::bar::foo
::bar::foo::::bar
::foo
::foo::bar
::bar::foo
::foo
::gee::foo
::gee::foo
::gee::foo::::bar
::foo
::foo::bar
}

### ===========================================================================
### -test     : ::Simple::Namespace::qualifiers
test-case qualifiers-1 {
   ::Simple::Namespace::qualifiers
} -script {
   puts [::Simple::Namespace::qualifiers {}]
   puts [::Simple::Namespace::qualifiers foo]
   puts [::Simple::Namespace::qualifiers foo::::bar]
   puts [::Simple::Namespace::qualifiers ::]
   puts [::Simple::Namespace::qualifiers ::foo]
   puts [::Simple::Namespace::qualifiers ::::foo::]
   puts [::Simple::Namespace::qualifiers ::foo::bar]
   puts [::Simple::Namespace::qualifiers ::foo::bar::]
} -output {

foo::::
::
::
::::foo::
::foo::
::foo::bar::
}

### ===========================================================================
### -test     : ::Simple::Namespace::information
test-case info-1 {
   ::Simple::Namespace::information
} -script {
   ::Simple::Namespace::information exists ::
} -return 1

test-case info-2 {
   ::Simple::Namespace::information
   subcommand throws an error
} -setup {

   # Create a subcommand procedure
   proc {::Simple::Namespace::information foo} {} {
      return -code error -errorcode errorCode errorMessage
   }

} -script {
   set return [catch {::Simple::Namespace::information foo} result]
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
} -cleanup {

   # Delete the subcommand procedure
   rename {::Simple::Namespace::information foo} {}

} -output {return: {1}
result: {errorMessage}
errorCode: {errorCode}
}

test-case info-3 {
   ::Simple::Namespace::information
   error, no subcommand given
} -script {
   ::Simple::Namespace::information
} -error {wrong # args: should be "::Simple::Namespace::information option ?arg\
...?"}

test-case info-4 {
   ::Simple::Namespace::information
   error, non-existing subcommand
} -regexp -script {
   ::Simple::Namespace::information foo
} -error {bad option "foo": *}

test-case info-5 {
   ::Simple::Namespace::information
   error, invalid subcommand arguments
} -script {
   ::Simple::Namespace::information exists foo bar
} -error {wrong # args: should be "::Simple::Namespace::information exists\
namespace"}

### ===========================================================================
### -test     : ::Simple::Namespace::information exists
test-case exists-1 {
   ::Simple::Namespace::information exists
} -script {
   puts -nonewline [::Simple::Namespace::information exists {}]
   puts -nonewline [::Simple::Namespace::information exists ::]
   puts -nonewline [::Simple::Namespace::information exists ::Simple]
   puts -nonewline [::Simple::Namespace::information exists ::Simple::]
   puts -nonewline [::Simple::Namespace::information exists ::::Simple::::]
   puts -nonewline [::Simple::Namespace::information exists ::foo]
} -output 111110

### ===========================================================================
### -test     : ::Simple::Namespace::configure
test-case configure-1 {
   ::Simple::Namespace::configure
} -script {
   ::Simple::Namespace::configure
} -error {this package has no options}

### ===========================================================================
### -test     : ::Simple::Namespace::cget
test-case cget-1 {
   ::Simple::Namespace::cget
} -script {
   ::Simple::Namespace::cget
} -return {}

} ;# End of regression testing section
