### ===========================================================================
### -package  : SimplePackage
# -version    : 0.2
# -purpose    : Package handling.
# -overview   :
#     This package improves the Tcl facilities for package loading and version
#  control.  It works by each package declaring itself.  This declaration
#  includes such information as the namespaces used by the package, its
#  exported commands and aliases or the packages it requires, so that this
#  information can not only be queried later, but also used for other
#  purposes.  As an example, it is possible to obtain the list of exported
#  commands from a package plus those exported from all packages it may
#  require, recursively, before actually modifying other namespaces not owned
#  by it.
#
# -usage      :
#     Packages may be either known or unknown to Tcl, depending on whether or
#  not either K<package ifneeded> or K<package provide> has been invoked for
#  the package.  In turn, known packages may be either available or provided,
#  depending on whether or not K<package provide> has been invoked for them. 
#  In turn, provided packages may be either declared or undeclared, depending
#  on whether K<::Simple::Package::declare> has been invoked for them.  All
#  undeclared packages are considered to be installed (that is, ready to be
#  used) but declared packages may be either installed or uninstalled,
#  depending on whether or not K<::Simple::Package::install> has been invoked
#  for them.  The following drawing sketches the situation:
#
#
#             .===========.     .-----------.
#             |  unknown  |     |   known   | *
#             `==========='     `-----------'
#                                 /       \
#                      .===========.     .-----------.
#                      | available |     | provided  |
#                      `==========='     `-----------'
#                                          /       \
#                               .===========.     .-----------.
#                               | declared  | *   |undeclared |
#                               `==========='     `-----------'
#                                 /       \         /
#                      .-----------.     .===========.
#                      |uninstalled| *   | installed | *
#                      `-----------'     `==========='
#
#
#     It is possible to get the list of packages in those states which have an
#  asterisk by its box in the drawing above (known, declared, uninstalled and
#  installed) via the K<::Simple::Package::information <state>> commands
#  where "<state>" is the corresponding state.  The format of each returned
#  package is {name version}.
#
#     Each package can be considered to be in one of the four main states
#  depicted with double-line boxes in the drawing above: I<unknown>,
#  I<available>, I<declared> or I<installed>.  The
#  K<::Simple::Package::information state> command returns the state of a
#  package among these main four.
#
#     At the beginning, all packages are I<unknown>.  Where Tcl evaluates all
#  of the F<pkgIndex.tcl> files in the V<auto_path> directories, K<package
#  ifneeded> is typically executed for some packages which become
#  I<available>.  These packages can then be required with
#  K<::Simple::Package::require> or K<package require>, which should invoke
#  K<package provide> so that the packages become provided. 
#
#     During this process, K<::Simple::Package::declare> can be either invoked
#  or not.  In the later case, the package is considered I<installed>,
#  although a full install might require manually importing the package
#  namespaces via K<namespace import>.  In the former case, the package
#  becomes I<declared>, state in which many package details can be queried
#  (see below) before actually installing the package via
#  K<::Simple::Package::install>.  With this, the package finally reaches the
#  I<installed> state.
#
#     Another possibility is to use K<::Simple::Package::require-and-install>
#  which requires the package and, if declared, automatically installs it.
#
#     It is also possible to go backwards in the above described route by
#  using K<::Simple::Package::uninstall>, K<::Simple::Package::undeclare> and
#  K<package forget> (or substituting the first two calls by
#  K<::Simple::Package::uninstall-and-undeclare).  The following drawings
#  sketches these transitions.  In the drawings the prefix
#  "::Simple::Package::" is abbreviated to "::S::P::".  First, for declared
#  packages:
#
#
#                                 .===========.
#                               ,>|  unknown  |
#                              (  `==========='\
#                package forget \               \  package ifneeded
#                                \               )
#                                 .===========.<'
#     ,----------->-----------> ,>| available | -->--------->-------------.
#     |                        (  `==========='\   (::S::P::require OR    |
#     |       ::S::P::undeclare \               \  package require) AND   |
#     |                          \               ) ::S::P::declare        |
#  ::S::P::                       .===========.<'                         |
# uninstall-and-undeclare       ,>| declared  |     ::S::P::require-and-install
#     |                        (  `==========='\                          |
#     |       ::S::P::uninstall \               \  ::S::P::install        |
#     |                          \               )                        |
#     |                           .===========.<'                         |
#      `----------<-----------<-- | installed | <-----------<-------------'
#                                 `==========='
#
#     For undeclared packages:
#
#                                 .===========.
#                                >|  unknown  |
#                              (' `==========='\
#                package forget \               \  package ifneeded
#                                \               )
#                                 .===========.<'
#                                >| available |
#                              (' `==========='\   ::S::P::require OR
#                   :S::P::     \               \  package require OR
#        uninstall-and-undeclare \               ) ::S::P::install-and-require
#                                 .===========.<'
#                                 | installed |
#                                 `==========='
#
#
#     Notice that in the case of undeclared packages, a full installation
#  might require manually importing the package namespaces via K<namespace
#  import> and, conversely, a full uninstallation might require manually
#  forgetting the package namespaces via K<namespace forget>.  This is not
#  necessary for declared packages as the installation and uninstallation
#  procedures take care of that.
#
#     The purpose of having I<declared> and I<installed> states is twofold. 
#  First, it is possible to require a package and get information about it
#  without any effect in other namespaces until it is installed.  Second, it
#  is possible to "reset" the status of a package to its default configuration
#  by uninstalling and reinstalling it.
#
#     The K<::Simple::Package::information> command provides many subcommands
#  to query information about declared packages:  K<aliases>, K<aliasesall>,
#  K<collisions>, K<commands>, K<declared>, K<export>, K<exportedcommands>,
#  K<exportedcommandsall>, K<importedinto>, K<install>, K<installed>,
#  K<known>, K<namespaces>, K<packagealias>, K<packagecommand>,
#  K<packagenamespace>, K<privatecommands>, K<publiccommands>, K<required>,
#  K<requiredall>, K<requiredby>, K<state>, K<uninstall> and K<uninstalled>.
#  Refer to the description of each command below for further information.
# 
#     Notice that it is very easy to convert undeclared packages into declared
#  ones, thus benefit from all the extra functionality available, by simply
#  adding a proper call to K<::Simple::Package::declare>, for example in the
#  corresponding entry of the F<pkgIndex.tcl> file.
#
#     Upon requiring a package which gets declared in the process, all its
#  required packages are also required, recursively.  Something similar
#  happens when installing a declared package, which results in all its
#  required packages being installed as well, recursively.  Notice, though,
#  that the contrary is not true; that is, uninstalling a package does not
#  uninstall any other package.  This feature might be provided in a future
#  revision of the package.
#
#     Finally, after a package has been installed, it is possible to import
#  all its exported commands into another namespace via
#  K<::Simple::Package::import> and unimport them via
#  K<::Simple::Package::unimport>.
#  
# -keywords   : package alias export import install load require
# -commands   :
#
#  * K<::Simple::Package::information known> ?packagePattern?
#    Returns the list of known packages.
#
#  * K<::Simple::Package::information declared> ?packagePattern?
#    Returns the list of declared packages.
#
#  * K<::Simple::Package::information uninstalled> ?packagePattern?
#    Returns the list of uninstalled packages.
#
#  * K<::Simple::Package::information installed> ?packagePattern?
#    Returns the list of installed packages.
#
#  * K<::Simple::Package::information state> package ?version?
#    Returns the state of a package.
#
#  * K<::Simple::Package::require> package ?version?
#    Requires a package.
#
#  * K<::Simple::Package::declare> package version ?-required list?
#    ?-namespaces list? ?-export list? ?-aliases list? ?-install script?
#    ?-firsttimeintsall script? ?-uninstall script?
#    Declares a package.
#
#  * K<::Simple::Package::information namespaces> package ?version?
#    Returns a package namespaces.
#
#  * K<::Simple::Package::information packagenamespace> namespace
#    Returns the declared package a namespace belongs to.
#
#  * K<::Simple::Package::information packagealias> alias
#    Returns the declared package an alias belongs to.
#
#  * K<::Simple::Package::information packagecommand> command
#    Returns the declared package a command belongs to.
#
#  * K<::Simple::Package::information required> package ?version?
#    Returns a package list of required packages.
#
#  * K<::Simple::Package::information requiredall> package ?version?
#    Returns a package full list of required packages.
#
#  * K<::Simple::Package::information requiredby> package ?version?
#    Returns the list of declared packages which require a package.
#
#  * K<::Simple::Package::information install> package ?version?
#    Returns a package install script.
#
#  * K<::Simple::Package::information uninstall> package ?version?
#    Returns a package uninstall script.
#
#  * K<::Simple::Package::information aliases> package ?version?
#    Returns a package list of aliases.
#
#  * K<::Simple::Package::information export> package ?version?
#    Returns a package list of exported commands.
#
#  * K<::Simple::Package::information exportedcommands> package ?version?
#    Returns a package current list of exported commands.
#
#  * K<::Simple::Package::information privatecommands> package ?version?
#    Returns a package list of private commands.
#
#  * K<::Simple::Package::information publiccommands> package ?version?
#    Returns a package list of public commands.
#
#  * K<::Simple::Package::information commands> package ?version?
#    Returns a package list of public and exported commands.
#
#  * K<::Simple::Package::information exportedcommandsall> package ?version?
#    Returns a package full list of exported commands.
#
#  * K<::Simple::Package::information aliasesall> package ?version?
#    Returns a package full list of aliases.
#
#  * K<::Simple::Package::information collisions> package ?namespace?
#    Returns the list of collisions upon importing a package.
#
#  * K<::Simple::Package::information importedinto> package ?version?
#    Returns the namespaces into which a package has been imported.
#
#  * K<::Simple::Package::install> package
#    Installs a package.
#
#  * K<::Simple::Package::require-and-install> package ?version?
#    Requires and installs a package.
#
#  * K<::Simple::Package::import> package namespace
#    Imports a package exported commands into a namespace.
#
#  * K<::Simple::Package::unimport> package namespace
#    Unimports a package exported commands from a namespace.
#
#  * K<::Simple::Package::uninstall> package
#    Uninstalls a package.
#
#  * K<::Simple::Package::undeclare> package
#    Undeclares a package.
#
#  * K<::Simple::Package::uninstall-and-undeclare> package
#    Uninstalls and undeclares a package.
#
#  * K<::Simple::Package::configure>
#    Configures the package options.
#
#  * K<::Simple::Package::cget>
#    Gets the package options.
#
# -variables  :
#
#  { ::Simple::Package::PackageRequiredPackages
#                     -array         "Array containing for each package
#                                     its required packages in the format
#                                     {name ?version?}"}
#  { ::Simple::Package::PackageExportedCommands
#                     -array         "Array containing for each package
#                                     its exported commands"}
#  { ::Simple::Package::PackageAliases
#                     -array         "Array containing for each package
#                                     its aliases"}
#  { ::Simple::Package::PackageInstallScript
#                     -array         "Array containing for each package
#                                     its install script"}
#  { ::Simple::Package::PackageFirstInstallScript
#                     -array         "Array containing for each package
#                                     its first time install script"}
#  { ::Simple::Package::PackageUninstallScript
#                     -array         "Array containing for each package
#                                     its uninstall script"}
#  { ::Simple::Package::PackageNamespaces
#                     -array         "Array containing for each package
#                                     its namespaces"}
#  { ::Simple::Package::InstalledPackages
#                     -array         "Array whose elements are the installed
#                                     declared packages"}
#  { ::Simple::Package::DeclaredPackages -array         
#                                    "Array whose elements are the declared
#                                     packages"}
#  { ::Simple::Package::PackageImportedInto
#                     -array         "Array containing for each package
#                                     the list of packages it has been imported
#                                     into"}
#
# -examples   : 
#
#  # Install the package
#  package require SimplePackage
#  
#  # Create an ifneeded script for a package which does not declare it
#  package ifneeded UndeclaredPackage 1.0 {
#     package provide UndeclaredPackage 1.0
#  }
#  
#  # Create an ifneeded script for a package declaring it
#  package ifneeded DeclaredPackage 1.0 {
#     ::Simple::Package::declare DeclaredPackage 1.0 -required {
#        {UndeclaredPackage 1.0}
#     } -namespaces {
#        ::Declared
#     } -export {
#        ::Declared::export
#     } -aliases {
#        {alias ::Declared::public}
#     } -install {
#        puts {Installing DeclaredPackage 1.0}
#        set ::Declared::PackageVariable 1
#     } -firsttimeinstall {
#        puts {   for the first time!}
#        proc ::Declared::export {} {puts export}
#        proc ::Declared::public {} {puts public}
#     } -uninstall {
#        puts {Uninstalling DeclaredPackage 1.0}
#     }
#  }
#  
#  # No packages initially installed
#  # This displays the following:
#  #    installed:
#  puts "installed: [::Simple::Package::information installed *eclared*]"
#  
#  # Now require the declared package
#  ::Simple::Package::require DeclaredPackage
#  
#  # The undeclared package has been installed
#  # This displays the following:
#  #    installed: {UndeclaredPackage 1.0}
#  puts "installed: [::Simple::Package::information installed *eclared*]"
#  
#  # Create a couple of commands which would collide with the package exported
#  # commands and aliases upon installing it
#  proc export {} {}
#  proc alias {} {}
#  
#  # Which commands are exported by the declared package ?
#  # This displays the following:
#  #    export: export
#  puts "export: [::Simple::Package::information\
#     exportedcommands DeclaredPackage]"
#  
#  # Which commands are exported by the declared package ?
#  # This displays the following:
#  #    aliases: {alias ::Declared::public}
#  puts "aliases: [::Simple::Package::information aliases DeclaredPackage]"
#  
#  # Would those command or aliases collide with other commands or aliases
#  # in the global namespace ?
#  # This displays the following:
#  #    collisions: ::Declared::export alias
#  puts "collisions:\
#     [::Simple::Package::information collisions DeclaredPackage]"
#  
#  # Let's remove the colliding commands and assess again whether the
#  # collisions remain
#  rename export {}
#  rename alias {}
#  # This displays the following:
#  #    collisions:
#  puts "collisions:\
#     [::Simple::Package::information collisions DeclaredPackage]"
#  
#  # No collisions, we may safely install the declared package
#  # This displays the following:
#  #    Installing DeclaredPackage 1.0
#  #       for the first time!
#  ::Simple::Package::install DeclaredPackage
#  
#  # Increase and display the package variable
#  # This displays the following:
#  #    2
#  puts [incr ::Declared::PackageVariable]
#  
#  # Into which namespaces has the declared package been imported ?
#  # This displays the following:
#  #    importedinto: ::
#  puts "importedinto:\
#     [::Simple::Package::information importedinto DeclaredPackage]"
#  
#  # Which commands are provided by the declared package ?
#  # This displays the following:
#  #    commands: ::Declared::export ::Declared::public
#  puts "commands:\
#     [::Simple::Package::information commands DeclaredPackage]"
#  
#  # Ok, use them
#  # This displays the following:
#  #    public
#  ::Declared::public
#  # This displays the following:
#  #    export
#  export
#  
#  # We are done, let's get rid of the declared package
#  # This displays the following:
#  #    Uninstalling DeclaredPackage 1.0
#  ::Simple::Package::uninstall DeclaredPackage
#  
#  # The package commands are no longer available
#  # This displays the following:
#  #    invalid command name "export"
#  catch {export} result
#  puts $result
#  
#  # The undeclared package remains installed
#  # This displays the following:
#  #    installed: {UndeclaredPackage 1.0}
#  puts "installed: [::Simple::Package::information installed *eclared*]"
#  
#  # We need the declared package again!
#  # This displays the following:
#  #    Installing DeclaredPackage 1.0
#  ::Simple::Package::require-and-install DeclaredPackage
#  
#  # Verify that the package variable has its default value
#  # This displays the following:
#  #    1
#  puts $::Declared::PackageVariable
#  package require SimplePackage
#
# -todo       :
#  * Allow a I<-exact> flag in K<::Simple::Package::declare>,
#    K<::Simple::Package::require-and-install> and
#    ::Simple::Package::Priv::require-package as present in K<package
#    require>.
#
#  * Currently all packages required by a given package are installed when
#    that package is installed.  It would be nice to have a mechanism so that
#    packages not explicitely installed and no longer required by any
#    other could be automatically uninstalled.
#
#  * Provide a K<::Simple::Package::reset> command to reset the state of an
#    installed package by uninstalling and reinstalling it. If effect, this
#    may reduce to evaluating the uninstall and install scripts.
#
#  * Provide something similar to K<::Simple::Package::information collisions>
#    but informing about the collisions when importing a namespace (instead of
#    a package) into another namespace.
#
#  * Allow a {package version} pair where a package name is expected in the
#    user procedures, so that it is possible to call the procedure either with
#    a single argument {package version} or package or with two arguments
#    {package} {version}.
#
#  * Provide a K<::Simple::Package::modify> command.
#
#  * The install, first time install and uninstall package scripts should be
#    stored as procedures instead as of variables in order to benefit from 
#    its byte compilation.
#
# -history    :
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimplePackage) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
#   Utility procedure to declare the package
namespace eval ::Simple::Package::Priv {}
proc ::Simple::Package::Priv::declare {}  {

   ::Simple::Package::declare SimplePackage 0.2 -required {
      SimpleError
   } -namespaces {
      ::Simple::Package
      ::Simple::Package::Priv
   } -firsttimeinstall {

      ### Declare the package errors
      ::Simple::Error::declare ::Simple::BAD-ITEM {
         expected %s but got "%s"
      } {
         <incorrect item> was found when an <item> was expected
      } {
         Provide a valid <item>
      } {<item> <incorrect item>}

      ::Simple::Error::declare ::Simple::Package::NAMESPACE-OWNED {
         namespace "%s" owned by package "%s"
      } {
         Namespace <namespace> is alredy owned by package <package> and
         a namespace can not be shared by two packages
      }

      ::Simple::Error::declare ::Simple::Package::NAMESPACE-NOT-OWNED {
         namespace "%s" not owned by package "%s"
      } {
         Package <package> is using the namespace <namespace> as if it were its
         owner, but that is not the case
      }

      ::Simple::Error::declare ::Simple::Package::UNDECLARED {
         undeclared package "%s"
      } {
         Package <package> is undeclared but a declared package is required
      } {
         Use K<::Simple::Package::declare> to declare the package
      }

      ::Simple::Error::declare ::Simple::Package::UNINSTALLED {
         uninstalled package "%s"
      } {
         Package <package> is uninstalled but an installed package is required
      } {
         Use K<::Simple::Package::install> to install the package
      }

      ::Simple::Error::declare ::Simple::Package::INSTALLED {
         installed package "%s"
      } {
         Package <package> is installed but an uninstalled package is required
      } {
         Use K<::Simple::Package::uninstall> to uninstall the package
      }

      ::Simple::Error::declare ::Simple::Package::ERROR-EVAL-SCRIPT {
         error evaluating package "%s" %s script: %s
      } {
         Error <error> was found when evaluating package <package> <install,
         first time install or uninstall> script
      } {
         Correct the corresponding script
      } {<package> <install, first time install or uninstall> <error>}

      ::Simple::Error::declare ::Simple::Package::ALREADY-DECLARED {
         package already declared "%s %s"
      } {
         Version <version> of package <package> has been already declared via
         K<::Simple::Package::declare
      } {
         Use K<::Simple::Package::undeclare> to undeclare the package
      }

      ::Simple::Error::declare ::Simple::Package::CANT-FIND {
         can't find package %s
      } {
         Package <package> has been required but is unknown to Tcl
      } {
         Invoke K<package ifneeded> or K<package provide> for the package
      }

      ::Simple::Error::declare ::Simple::Package::COLLISION {
         couldn't %s package "%s": command (or alias) "%s" in %s collides with
         %s "%s" (package "%s")
      } {
         It was impossible to <import or install> package <package> because
         the existing command or alias <command or alias> in namespace
         <namespace> collides (that is, their names are identical) with
         <command or alias> <exported command or alias> from package <colliding
         package>.  This last package either is the one being imported or
         installed or is required by it
      } {
         Either delete or rename command or alias <command or alias>.  Use
         K<::Simple::Package::information collisions> to ensure there are
         no collisions when importing or installing a package
      }

      ::Simple::Error::declare ::Simple::Package::COLLISION-BETWEEN-PACKAGES {
         couldn't %s package "%s": command (or alias) "%s" is exported (or
         created) by both package "%s"%s and "%s"%s
      } {
         It was impossible to <import or install> package <package> because it
         exports (or creates) command (or alias) <command or alias> in two
         packages <package 1> and <package 2>.  This means that
         package <package> is badly formed
      } {
         Correct package <package>
      } {} {
         couldn't <import or install> package "<package>": command or alias
         "<command or alias>" is exported or created by both package "<package
         1>" [(namespace "<namespace 1>")] and "<package 2>" [(namespace
         "<namespace 2>")]
      }

      ::Simple::Error::declare ::Simple::Package::ALREADY-IMPORTED {
         package "%s" already imported into namespace "%s"
      } {
         It was impossible to import package <package> into namespace
         <namespace> because it was already imported into it
      } {
         Use K<::Simple::Package::unimport> to unimport the package
      }

      ::Simple::Error::declare ::Simple::Package::NOT-IMPORTED {
         package "%s" not imported into namespace "%s"
      } {
         It was impossible to unimport package <package> from namespace
         <namespace> because it was not imported into it
      } {
         Use K<::Simple::Package::import> to import the package
      }

      ::Simple::Error::declare ::Simple::NO-OPTIONS {
         this package has no options
      } {
         This package K<configure> command is not functional because the package
         has no options
      }

      ::Simple::Error::declare ::Simple::TOO-MANY-ARGS {
         called "%s" with too many arguments
      } {
         A call to the command <command> failed because
         too many arguments were supplied
      } {
         Redo the call using the correct list of arguments
      }
   }

   ### Require and declare the SimpleError package
   package require SimpleError 0.2
   ::Simple::Package::declare SimpleError 0.2 -namespaces {
      ::Simple::Error
      ::Simple::Error::Priv
   }
}

### ===========================================================================
### -command  : ::Simple::Package::information
# -purpose    : Base command for package K<information> subcommands.
# -access     : Private
# -overview   :
#     This is the base command for the K<::Simple::Package::information>
#  subcommand procedures.
#
# -arguments  :
#  { args             -list          {Subcommand and subcommand arguments}}
#
proc ::Simple::Package::information {
   args
} {
   #========= %BASE-COMMAND%

   ### Get the subcommand and full command names
   set subcommand [lindex $args 0]
   set fullCommand ::Simple::Package::information\ $subcommand

   ### Evaluate the subcommand procedure
   ### Error from subcommand procedure
   #   The uplevel here ensures that the call to the base command
   #   is hidden to the command procedure body
   if {[catch {uplevel [list $fullCommand] [lrange $args 1 end]} result]} {

      ### No subcommand given
      if {[llength $args] == 0} {

         ### Throw error
         ::Simple::Error::throw ::Simple::NO-SUBCOMMAND\
            ::Simple::Package::information

      ### Non-existing subcommand
      } elseif {![string compare "invalid command name \"$fullCommand\""\
         $result]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND $subcommand {aliases,\
            aliasesall, collisions, commands, declared, export,\
            exportedcommands, exportedcommandsall, importedinto, install,\
            installed, known, namespaces, packagealias, packagecommand,\
            packagenamespace, privatecommands, publiccommands, required,\
            requiredall, requiredby, state, uninstall or uninstalled}

      ### Invalid arguments
      } elseif {![string compare "called \"$fullCommand\" with too many\
         arguments" $result] || [string match {no value given for\
         parameter*} $result]} {

         ### Compose the correct arguments
         switch -exact -- $subcommand {
            known -
            declared -
            installed -
            uninstalled {
               set arguments { ?packagePattern?}
            }
            state -
            namespaces -
            required -
            requiredall -
            requiredby -
            install -
            uninstall -
            aliases -
            aliasesall -
            export -
            exportedcommands -
            exportedcommandsall -
            privatecommands -
            publiccommands -
            commands -
            importedinto {
               set arguments { package ?version?}
            }
            collisions {
               set arguments { package ?namespace?}
            }
            packagenamespace {
               set arguments { namespace}
            }
            packagealias {
               set arguments { alias}
            }
            packagecommand {
               set arguments { command}
            }
         }

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND-ARGS\
            $fullCommand $arguments

      ### Other error
      } else {

         ### Rethrow the subcommand procedure error
         return -code error -errorcode $::errorCode $result
      }

   ### No error from subcommand procedure
   } else {

      ### Return the subcommand procedure return value
      set result
   }
}

### ===========================================================================
### -command  : ::Simple::Package::information known
# -purpose    : Returns the list of known packages.
# -overview   :
#     This procedure returns the list of known packages (that is, those for
#  which K<package ifneeded> and/or K<package provide> has been invoked) and
#  whose name matches the given pattern.
#
#     The returned packages can be in any state but I<unknown>, that is,
#  I<available>, I<declared> or I<installed>.
#
# -arguments  :
#  {?packagePattern?  -pattern {}    {Pattern}}
#
# -returns :  The list of known packages whose name matches the given
#  pattern in the format {name version}.
#
# -remarks    :
#  * Use no pattern to get all packages.
#
proc {::Simple::Package::information known} {
   {packagePattern {}}
} {
   ### Package pattern given
   if {[string compare $packagePattern {}]} {

      ### Get packages matching the given pattern
      set packages [list]
      foreach package [package names] {
         if {[string match $packagePattern $package]} {
            lappend packages $package
         }
      }

   ### Package pattern not given
   } else {

      ### Get all packages
      set packages [package names]
   }

   ### Loop over packages
   set answer [list]
   foreach package $packages {

      ### Get the package known versions
      set providedVersion [package provide $package]
      set ifneededVersions [package versions $package]
      if {[string compare $providedVersion {}]} {

         # Provided version has precedence over equivalent ifneeded version
         set knownVersions $providedVersion
         foreach ifneededVersion $ifneededVersions {
            if {[package vcompare $ifneededVersion $providedVersion]} {
               lappend knownVersions $ifneededVersion
            }
         }
      } else {
         set knownVersions $ifneededVersions
      }

      ### Build the package list
      foreach knownVersion $knownVersions {
         lappend answer [list $package $knownVersion]
      }
   }

   ### Return the list of known packages
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information declared
# -purpose    : Returns the list of declared packages.
# -overview   :
#     This procedure returns the list of declared packages (that is, those for
#  which K<::Simple::Package::declare> has been invoked) and whose name
#  matches the given pattern.
#
#     The returned packages can be in the I<declared> or I<installed> states.
#
# -arguments  :
#  {?packagePattern?  -pattern {}    {Pattern}}
#
# -returns :  The list of declared packages whose name matches the given
#  pattern in the format {name version}.
#
# -remarks    :
#  * Use no pattern to get all packages.
#
proc {::Simple::Package::information declared} {
   {packagePattern {}}
} {
   ### Package pattern given
   if {[string compare $packagePattern {}]} {

      ### Get declared packages matching the given pattern
      set packages [list]
      foreach packagePair [array names ::Simple::Package::DeclaredPackages] {
         foreach {package version} $packagePair break
         if {[string match $packagePattern $package]} {
            lappend packages $packagePair
         }
      }
      set packages

   ### Package pattern not given
   } else  {

      ### Get all declared packages
      array names ::Simple::Package::DeclaredPackages
   }
}

### ===========================================================================
### -command  : ::Simple::Package::information uninstalled
# -purpose    : Returns the list of uninstalled packages.
# -overview   :
#     This procedure returns the list of uninstalled packages (that is, those
#  for which K<::Simple::Package::declare> has been invoked but which have not
#  been installed yet via K<::Simple::Package::install>) and whose name
#  matches the given pattern.
#
#     All returned packages are in the I<declared> state.
#
# -arguments  :
#  {?packagePattern?  -pattern {}    {Pattern}}
#
# -returns :  The list of uninstalled packages whose name matches the given
#  pattern in the format {name version}.
#
# -remarks    :
#  * Use no pattern to get all packages.
#
proc {::Simple::Package::information uninstalled} {
   {packagePattern {}}
} {
   ### Get all declared packages matching the given pattern
   set packages [{::Simple::Package::information declared} $packagePattern]

   ### Loop over packages
   set answer [list]
   foreach packagePair $packages {

      ### Uninstalled package
      if {![info exists ::Simple::Package::InstalledPackages($packagePair)]} {

         ### Add package to the list of uninstalled packages
         lappend answer $packagePair
      }
   }

   ### Return the list of uninstalled packages
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information installed
# -purpose    : Returns the list of installed packages.
# -overview   :
#     This procedure returns the list of installed packages (that is, those
#  whose state is I<installed>, either by having being required and not
#  declared or required, declared and also installed) and whose name matches
#  the given pattern.
#
#     All returned packages are in the I<installed> state.
#
# -arguments  :
#  {?packagePattern?  -pattern {}    {Pattern}}
#
# -returns :  The list of installed packages whose name matches the given
#  pattern in the format {name version}.
#
# -remarks    :
#  * Use no pattern to get all packages.
#
proc {::Simple::Package::information installed} {
   {packagePattern {}}
} {
   ### Package pattern given
   if {[string compare $packagePattern {}]} {

      ### Get packages matching the given pattern
      set packages [list]
      foreach package [package names] {
         if {[string match $packagePattern $package]} {
            lappend packages $package
         }
      }

   ### Package pattern not given
   } else {

      ### Get all packages
      set packages [package names]
   }

   ### Loop over packages
   set answer [list]
   foreach package $packages {

      ### Provided package
      set version [package provide $package]
      if {[string compare $version {}]} {

         ### Undeclared or declared and installed package
         set packagePair [list $package $version]
         if {![info exists ::Simple::Package::DeclaredPackages($packagePair)]\
            || [info exists\
            ::Simple::Package::InstalledPackages($packagePair)]} {

            ### Add package to the list of installed packages
            lappend answer $packagePair
         }
      }
   }

   ### Return the list of installed packages
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information state
# -purpose    : Returns the state of a package.
# -overview   :
#     This procedure returns I<unknown>, I<available>, I<declared> or
#  I<installed> depending on the state of the given package.  If no version is
#  given, the returned state is that of the known package version whose state
#  is higher in the rank.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -returns    :  The state of the given package, one of I<unknown>,
#  I<available>, I<declared> or I<installed>.
#
# -limitations:
#  * To identify whether a version is given or not its default value
#    is set to "0.0.0".
#
proc {::Simple::Package::information state} {
   package
   {version 0.0.0}
} {
   ### Version given
   set versionGiven [string compare $version 0.0.0]
   if {$versionGiven} {

      ### Get the version equivalent to that given
      set version\
         [::Simple::Package::Priv::equivalent-version $package $version]
   }

   ### Provided package
   set providedVersion [package provide $package]
   if {[string compare $providedVersion {}] && (!$versionGiven\
      || ![package vcompare $providedVersion $version])} {

      ### Build the package pair
      set packagePair [list $package $providedVersion]

      ### Declared but not installed package
      if {[info exists ::Simple::Package::DeclaredPackages($packagePair)] &&\
         ![info exists ::Simple::Package::InstalledPackages($packagePair)]} {

         ### Declared package
         set answer declared

      ### Undeclared ot declared and installed package
      } else {

         ### Installed package
         set answer installed
      }

   ### Non-provided package
   } else {

      ### Version given
      if {$versionGiven} {

         ### Check whether there is a package ifneeded script for that version
         set available [llength [package ifneeded $package $version]]

      ### Version not given
      } else {

         ### Check whether there is any version available
         set available [llength [package versions $package]]
      }

      ### Available package
      if {$available} {

         ### Available package
         set answer available

      ### Not available package
      } else  {

         ### Unknown package
         set answer unknown
      }
   }

   ### Return the state of the package
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::require
# -purpose    : Requires a package.
# -overview   :
#     This procedure requires a package, that is, changes its state from
#  I<available> to I<required> or I<installed> depending on whether the
#  package is declared via the K<::Simple::Package::declare> command upon
#  requiring it.
#
#     No error is thrown if the package is already provided.
#
#     An error is thrown if the package is not available.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -returns    : The required package version.
# -limitations:
#  * To identify whether a version is given or not its default value
#    is set to "0.0.0".
#
proc ::Simple::Package::require {
   package
   {version 0.0.0}
} {
   ### Require the package
   set packagePair [::Simple::Package::Priv::find-package\
      [list $package $version]]
   ::Simple::Package::Priv::require-package $packagePair
}

### ===========================================================================
### -command  : ::Simple::Package::declare
# -purpose    : Declares a package.
# -overview   :
#     This procedure declares a package including its required packages (which
#  are required), namespaces (which are created), exported commands (which are
#  exported), aliases as well as the install, first time install and uninstall
#  scripts.  
#
#     The aliases are created when the package is installed via the
#  K<::Simple::Package::install> command.  The install, first time install and
#  uninstall scripts are evaluated in the global namespace when the package is
#  installed via the K<::Simple::Package::install> command, installed for the
#  first time via the K<::Simple::Package::install> command, or uninstalled
#  via the K<::Simple::Package::uninstall> command, respectively. 
#
#     An error is thrown if the package is already declared; use
#  K<::Simple::Package::undeclare> to undeclare a package.
#
#     The most prominent difference between an alias and an exported command
#  is that the alias effect is interpreter-wide while that of the exported
#  command is namespace-wide.  As all package exported command are imported
#  into the global namespace upon installing a package, place in which are
#  accessible even from other namespaces, that difference is in practice
#  almost irrelevant.  Aliases are thus provided by this package so that a
#  command can be accessed through two names, as the
#  K<::Simple::Subcommand::create> command which can be accesses as
#  K<proc-ext>.  Notice that the alias is valid to invoke the command only,
#  not being a full-fledged substitute at all times when a command name is
#  required.  Notice also that a command call through an alias is slightly
#  slower than calling the command directly.
#
# -arguments  :
#  { package          -name          {Package name}}
#  { version          -version       {Package version}}
#  {-required         -list     {}   {Required packages list in the format
#                                     {name ?version?}}}
#  {-namespaces       -list     {}   {Package namespaces}}
#  {-export           -list     {}   {Package fully-qualified exported
#                                     commands}}
#  {-aliases          -list     {}   {Package aliases list in the format
#                                     {aliasName aliasedCommand}}}
#  {-install          -script   {}   {Package install script}}
#  {-firsttimeintsall -script   {}   {Package first time install script}}
#  {-uninstall        -script   {}   {Package uninstall script}}
#
# -effects    :
#  * Creates the package namespaces.
#
#  * Exports commands from their namespaces.
#
proc ::Simple::Package::declare {
   package
   version
   args
} {
   ### Parse arguments
   set requiredPackages [list]
   set namespaces [list]
   set exportedCommands [list]
   set aliases [list]
   set installScript {}
   set firstInstallScript {}
   set uninstallScript {}
   set nArgument 0
   set nArguments [llength $args]
   while {$nArgument < $nArguments} {
      set argument [lindex $args $nArgument]
      switch -glob -- $argument {
         -required {

            # Build the required package list
            set tmp [list]
            foreach requiredPackage [lindex $args [incr nArgument]] {
               lappend tmp\
                  [::Simple::Package::Priv::find-package $requiredPackage]
            }
            set requiredPackages $tmp
         }
         -namespaces {

            # Build the namespaces list
            set tmp [list]
            foreach namespace [lindex $args [incr nArgument]] {
               lappend tmp [string trim $namespace]
            }
            set namespaces $tmp
         }
         -export {

            # Build the exported commands list
            set tmp [list]
            foreach exportedCommand [lindex $args [incr nArgument]] {
               lappend tmp [string trim $exportedCommand]
            }
            set exportedCommands $tmp
         }
         -aliases {

            # Build the aliases list
            set tmp [list]
            foreach alias [lindex $args [incr nArgument]] {
               lappend tmp [string trim $alias]
            }
            set aliases $tmp
         }
         -install {
            set installScript [lindex $args [incr nArgument]]
         }
         -firsttimeinstall {
            set firstInstallScript [lindex $args [incr nArgument]]
         }
         -uninstall {
            set uninstallScript [lindex $args [incr nArgument]]
         }
         -* {
            ::Simple::Error::throw ::Simple::BAD-OPTION $argument\
               {-aliases, -export, -firsttimeinstall, -install, -namespaces,\
               -required or -uninstall}
         }
         default {
            ::Simple::Error::throw ::Simple::TOO-MANY-ARGS\
               [lindex [info level 0] 0]
         }
      }
      incr nArgument
   }

   ### Build the package pair
   #   K<::Simple::Package::Priv::package-pair> is not
   #   used because the equivalent version is needed below
   set oldVersion $version
   set version [::Simple::Package::Priv::equivalent-version $package $version]
   set packagePair [list $package $version]

   ### The package has not been declared
   if {![info exists ::Simple::Package::DeclaredPackages($packagePair)]} {

      ### Require the packages
      foreach requiredPackage $requiredPackages {
         ::Simple::Package::Priv::require-package $requiredPackage
      }
       
      ### Store the package namespaces information
      ::Simple::Package::Priv::add-namespaces-info $packagePair $namespaces

      ### Store the package required packages
      set ::Simple::Package::PackageRequiredPackages($packagePair)\
         $requiredPackages

      ### Create the namespaces
      ::Simple::Package::Priv::create-namespaces $namespaces

      ### Store the package exported commands
      set ::Simple::Package::PackageExportedCommands($packagePair)\
         $exportedCommands

      ### Export the exported commands
      ::Simple::Package::Priv::export $packagePair $exportedCommands

      ### Loop over package aliases
      foreach aliasPair $aliases {
         foreach {alias aliased} $aliasPair break

         ### Assert the alias is unqualified
         if {[string match ::* $alias]} {
            ::Simple::Error::throw ::Simple::BAD-ITEM\
               {unqualifed alias name} $alias
         }

         ### Assert the aliased command is fully-qualified
         if {![string match ::* $aliased]} {
            ::Simple::Error::throw ::Simple::BAD-ITEM\
               {fully-qualified aliased command name} $aliased
         }

         ### Assert the aliased command namespace is owned by the package
         ::Simple::Package::Priv::assert-package-owns-namespace\
            $packagePair $aliased
      }

      ### Store the package aliases
      set ::Simple::Package::PackageAliases($packagePair) $aliases

      ### Store the package install script
      set ::Simple::Package::PackageInstallScript($packagePair)\
         $installScript

      ### Store the package first time install script
      set ::Simple::Package::PackageFirstInstallScript($packagePair)\
         $firstInstallScript

      ### Store the package uninstall script
      set ::Simple::Package::PackageUninstallScript($packagePair)\
         $uninstallScript

      ### Add this package to the list of declared packages
      ::Simple::Package::Priv::add-declared $packagePair

      ### Provide the package
      package provide $package $version

   ### The package has been declared
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::Package::ALREADY-DECLARED\
         $package $oldVersion
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::add-declared
# -purpose    : Adds a package to the list of declared packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::add-declared {
   packagePair
} {
   set ::Simple::Package::DeclaredPackages($packagePair) {}
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::delete-declared
# -purpose    : Deletes a package from the list of declared packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::delete-declared {
   packagePair
} {
   unset ::Simple::Package::DeclaredPackages($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::add-installed
# -purpose    : Adds a package to the list of installed packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::add-installed {
   packagePair
} {
   set ::Simple::Package::InstalledPackages($packagePair) {}
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::delete-installed
# -purpose    : Deletes a package from the list of installed packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::delete-installed {
   packagePair
} {
   unset ::Simple::Package::InstalledPackages($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::require-package
# -purpose    : Requires a package.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::require-package {
   packagePair
} {
   foreach {package version} $packagePair break
   package require $package $version
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::add-namespaces-info
# -purpose    : Stores a package namespaces information.
# -overview   :
#     This procedure throws an error if a namespace belongs to a declared
#  package and stores the package namespaces otherwise.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { namespaces       -list          {Package namespaces}}
#
proc ::Simple::Package::Priv::add-namespaces-info {
   packagePair
   namespaces
} {
   ### Assert no namespace belongs to other package
   foreach namespace $namespaces {
      foreach declaredPackage\
         [array names ::Simple::Package::DeclaredPackages] {

         if {[lsearch -exact\
            $::Simple::Package::PackageNamespaces($declaredPackage)\
            $namespace] != -1} {

            ::Simple::Error::throw ::Simple::Package::NAMESPACE-OWNED\
               $namespace $declaredPackage
         }
      }
   }

   ### Store the package namespaces
   set ::Simple::Package::PackageNamespaces($packagePair) $namespaces
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::delete-namespaces-info
# -purpose    : Deletes a package namespaces information.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::delete-namespaces-info {
   packagePair
} {
   unset ::Simple::Package::PackageNamespaces($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::create-namespaces
# -purpose    : Creates namespaces from a list.
# -arguments  :
#  { namespaces       -list          {Namespaces list}}
#
# -remarks    :
#  * The namespaces may already exists and no error is thrown.
#
# -effects    :
#  * Creates the package namespaces.
#
proc ::Simple::Package::Priv::create-namespaces {
   namespaces
} {
   foreach namespace $namespaces {
      namespace eval $namespace {}
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::export
# -purpose    : Exports a list of commands.
# -overview   :
#     This procedure exports a list of commands.  The corresponding namespace,
#  which must belong to the given package, is extracted from each
#  fully-qualified command name.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { exportList       -list          {Package fully-qualified exported
#                                     commands}}
# -effects    :
#  * Exports commands from their namespaces.
#
proc ::Simple::Package::Priv::export {
   packagePair
   exportList
} {
   ### Loop over exported commands
   foreach toExport $exportList {

      ### Assert the exported command is fully-qualified
      if {![string match ::* $toExport]} {
         ::Simple::Error::throw ::Simple::BAD-ITEM\
            {fully-qualifed exported command name} $toExport
      }

      ### Assert the exported command namespace is owned by the package
      set namespace [::Simple::Package::Priv::assert-package-owns-namespace\
         $packagePair $toExport]

      ### Export the command
      namespace eval $namespace\
         [format {namespace export %s} [list [namespace tail $toExport]]]
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::assert-package-owns-namespace
# -purpose    : Throws an error if a namespace is not owned by a package.
# -overview   :
#     This procedure throws an error if the given command namespace
#  is not owned by the given package.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { command          -qualifiedname {Command name}}
#
# -returns    : The command namespace.
#
proc ::Simple::Package::Priv::assert-package-owns-namespace {
   packagePair
   command
} {
   ### Get the command namespace
   set namespace [namespace qualifiers $command]
   if {![string compare $namespace {}]} {
      set namespace ::
   }

   ### Assert the namespace belongs to the package
   if {[lsearch -exact\
      $::Simple::Package::PackageNamespaces($packagePair)\
      $namespace] == -1} {

      ::Simple::Error::throw ::Simple::Package::NAMESPACE-NOT-OWNED\
         $namespace $packagePair
   }

   ### Return the command namespace
   set namespace
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::delete-namespaces
# -purpose    : Deletes a package namespaces.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -remarks    :
#  * The namespaces may have been already deleted and no error is thrown.
#
# -effects    :
#  * Deletes the package namespaces.
#
proc ::Simple::Package::Priv::delete-namespaces {
   packagePair
} {
   foreach namespace $::Simple::Package::PackageNamespaces($packagePair) {
      catch {namespace delete $namespace}
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::find-package
# -purpose    : Finds a package version and returns the name and version pair.
# -overview   :
#     This procedure finds the most relevant version of a package.  If a
#  version is given, that is the version used, possibly modified to
#  exactly match an already known equivalent version via
#  K<::Simple::Package::Priv::equivalent-version>.  If no version is given,
#  the package provided version is used.  If there is no
#  provided version, the greatest package known version is used.  If there
#  are no known versions, an error is thrown.
#
#     If an usable version is found for the package, the package name and
#  version pair is returned.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name ?version?}}}
#
# -returns    : The package name and version pair.
# -limitations:
#  * The version is assumed to have not been given if its value is equal to
#    "0.0.0" or the empty string.
#
proc ::Simple::Package::Priv::find-package {
   packagePair
} {
   ### No version given
   foreach {package version} $packagePair break
   if {![string compare $version {}] || ![string compare $version 0.0.0]} {

      ### Guess the version

      ### Get the package known versions
      set providedVersion [package provide $package]
      set ifneededVersions [package versions $package]

      ### Provided version
      if {[string compare $providedVersion {}]} {

         ### Return the provided version
         set version $providedVersion

      ### Known versions
      } elseif {[llength $ifneededVersions] > 0} {

         ### Return the greatest knwon version
         set version [lindex\
            [lsort -decreasing -command {package vcompare} $ifneededVersions] 0]

      ### Unknown package
      } else {

         ### Throw error
         ::Simple::Error::throw ::Simple::Package::CANT-FIND $package
      }

   ### Version given
   } else {

      ### Get the version equivalent to that given
      set version\
         [::Simple::Package::Priv::equivalent-version $package $version]
   }

   ### Return the package name and version pair
   list $package $version
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::equivalent-version
# -purpose    : Returns a package equivalent version.
# -overview   :
#     This procedure returns the given package version possibly modified to
#  exactly match an already known equivalent version.  For example, if
#  K<package ifneeded> has been evaluated for "foo 1.0" and this procedure is
#  called for "foo 1", the returned version is 1.0. 
#
# -arguments  :
#  { package          -name          {Package name}}
#  { version          -version       {Package version}}
#
# -returns    : The package equivalent version.
#
proc ::Simple::Package::Priv::equivalent-version {
   package
   version
} {
   ### Validate the given version
   package vcompare 0 $version

   ### Get the package known versions
   set providedVersion [package provide $package]
   set ifneededVersions [package versions $package]

   ### Return the given version by default
   set answer $version

   ### Loop over known versions
   #   The provided version has precedence over equivalent ifneeded version
   foreach knownVersion [concat $providedVersion $ifneededVersions] {

      ### Given version equivalent to known version
      if {![package vcompare $version $knownVersion]} {

         ### Return the known version
         set answer $knownVersion
         break
      }
   }

   ### Return the package equivalent version
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::package-pair
# -purpose    : Validates a package name and version and returns the pair.
# -overview   :
#     This procedure validates a package version, if given, or gets that of
#  any declared package with the given name otherwise; in that case, if there
#  is no declared package with that name, an error is thrown.  The package
#  name and version pair is returned.
#
# -arguments  :
#  { package          -name          {Package name}}
#  { version          -version       {Package version}}
#
# -returns    : The validated package name and version pair.
# -limitations:
#  * The version is assumed to have not been given if its value is equal to
#    "0.0.0".
#
proc ::Simple::Package::Priv::package-pair {
   package
   version
} {
   ### Version given
   if {[string compare $version 0.0.0]} {

      ### Get the version equivalent to that given
      set version\
         [::Simple::Package::Priv::equivalent-version $package $version]

   ### No version given
   } else {

      ### There is a declared package with that name
      set packages\
         [array names ::Simple::Package::DeclaredPackages "$package *"]
      if {[llength $packages] > 0} {

         ### Get its version
         set version [lindex [lindex $packages 0] 1]

      ### There is no declared package with that name
      } else {

         ### Throw error
         ::Simple::Error::throw ::Simple::Package::UNDECLARED $package
      }
   }

   ### Return the package name and version pair
   list $package $version
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::assert-namespace-exists
# -purpose    : Throws an error if a namespace does not exist.
# -arguments  :
#  { namespace        -namespace     {Namespace}}
#
proc ::Simple::Package::Priv::assert-namespace-exists {
   namespace
} {
   if {[catch {namespace children $namespace}]} {
      ::Simple::Error::throw ::Simple::NON-EXISTING-NAMESPACE\
         $namespace
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::assert-package-declared
# -purpose    : Throws an error if a package has not been declared.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::assert-package-declared {
   packagePair
} {
   if {![info exists ::Simple::Package::DeclaredPackages($packagePair)]} {
      ::Simple::Error::throw ::Simple::Package::UNDECLARED $packagePair
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::assert-package-installed
# -purpose    : Throws an error if a package is not installed.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::assert-package-installed {
   packagePair
} {
   if {![info exists ::Simple::Package::InstalledPackages($packagePair)]} {
      ::Simple::Error::throw ::Simple::Package::UNINSTALLED $packagePair
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::assert-package-not-installed
# -purpose    : Throws an error if a package is installed.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::assert-package-not-installed {
   packagePair
} {
   if {[info exists ::Simple::Package::InstalledPackages($packagePair)]} {
      ::Simple::Error::throw ::Simple::Package::INSTALLED $packagePair
   }
}

### ===========================================================================
### -command  : ::Simple::Package::information namespaces
# -purpose    : Returns a package namespaces.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -returns    : The list of the package namespaces.
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
proc {::Simple::Package::information namespaces} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package namespaces
   set ::Simple::Package::PackageNamespaces($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::information packagenamespace
# -purpose    : Returns the declared package a namespace belongs to.
# -overview   :
#     This procedure returns the declared package the given namespace belongs
#  to, or the empty string if no declared package owns the given namespace.
#
# -arguments  :
#  { namespace        -namespace     {Namespace}}
#
# -returns    : The declared package a namespace belongs to.
#
proc {::Simple::Package::information packagenamespace} {
   namespace
} {
   ### Assert the namespace exists
   ::Simple::Package::Priv::assert-namespace-exists $namespace

   ### Loop over declared packages
   set answer {}
   foreach declaredPackage [array names ::Simple::Package::DeclaredPackages] {

      ### This package owns the namespace
      if {[lsearch -exact\
         $::Simple::Package::PackageNamespaces($declaredPackage)\
         $namespace] != -1} {

         set answer $declaredPackage
         break
      }
   }

   ### Return the declared package the namespace belongs to
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information packagealias
# -purpose    : Returns the declared package an alias belongs to.
# -overview   :
#     This procedure returns the declared package the given alias belongs
#  to, or the empty string if either no declared package owns the given alias
#  or the alias does not exist.
#
# -returns    : The declared package an alias belongs to.
# -arguments  :
#  { alias            -unqualifiedname
#                                    "Alias name"}
#
proc {::Simple::Package::information packagealias} {
   alias
} {
   ### Loop over declared packages
   foreach declaredPackage [array names ::Simple::Package::DeclaredPackages] {

      ### Loop over package alias
      foreach aliasPair $::Simple::Package::PackageAliases($declaredPackage) {

         ### Alias found
         if {![string compare $alias [lindex $aliasPair 0]]} {

            ### Get the aliased command namespace
            set namespace [namespace qualifiers [lindex $aliasPair 1]]

            ### Return the declared package the alias belongs to
            return\
               [{::Simple::Package::information packagenamespace} $namespace]
         }
      }
   }

   ### No declared package owns the alias
   set answer {}
}

### ===========================================================================
### -command  : ::Simple::Package::information packagecommand
# -purpose    : Returns the declared package a command belongs to.
# -overview   :
#     This procedure returns the declared package the given command belongs
#  to, or the empty string if no declared package owns the given command. 
#  It works for either package commands used in their own namespace or for
#  exported commands from other packages.
#
# -returns    : The declared package a command belongs to.
# -arguments  :
#  { command          -unqualifiedname
#                                    "Command name"}
#
proc {::Simple::Package::information packagecommand} {
   command
} {
   ### Get the original command namespace
   set namespace\
      [namespace qualifiers [uplevel namespace origin [list $command]]]
   
   ### Return the declared package the command belongs to
   {::Simple::Package::information packagenamespace} $namespace
}

### ===========================================================================
### -command  : ::Simple::Package::information required
# -purpose    : Returns a package list of required packages.
# -returns    : The package list of required packages.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
proc {::Simple::Package::information required} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package required packages
   set ::Simple::Package::PackageRequiredPackages($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::information requiredall
# -purpose    : Returns a package full list of required packages.
# -overview   :
#     This procedure returns the given package list of required packages plus
#  all packages it may require.  It only takes into account the required
#  packages of packages declared via K<::Simple::Package::declare>.
#
# -returns    : The package full list of required packages.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -seealso    :
#  * The P<::Simple::Package::Priv::requiredall> command.
#
proc {::Simple::Package::information requiredall} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package full list of required packages
   ::Simple::Package::Priv::requiredall $packagePair
}

### ===========================================================================
### -command  : ::Simple::Package::information requiredby
# -purpose    : Returns the list of declared packages which require a package.
# -returns    : The list of declared packages which require the package.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -remarks    :
#  * The given package does not need to be declared.
#
proc {::Simple::Package::information requiredby} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Loop over declared packages
   set answer [list]
   foreach declaredPackage [array names ::Simple::Package::DeclaredPackages] {

      ### This package required the given package
      if {[lsearch -exact\
         $::Simple::Package::PackageRequiredPackages($declaredPackage)\
         $packagePair] != -1} {

         ### Add this package to the list of packages
         lappend answer $declaredPackage
      }
   }

   ### Return the list of packages which require the package
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::requiredall-recursive
# -purpose    : Returns a package full list of required packages.
# -overview   :
#     This procedure returns the given package list of required packages plus
#  all packages it may require.  It only takes into account the required
#  packages of packages declared via K<::Simple::Package::declare>.
#
# -returns    : The package full list of required packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -seealso    :
#  * The P<::Simple::Package::information requiredall> command.
#
#  * The P<::Simple::Package::Priv::requiredall> command.
#
# -details    :
#     This command is recursive.
#
# -remarks    :
#  * No provision is made for removing duplicated packages, use
#    K<::Simple::Package::Priv::requiredall> instead.
#
proc ::Simple::Package::Priv::requiredall-recursive {
   packagePair
} {
   ### Loop over required packages
   set answer $::Simple::Package::PackageRequiredPackages($packagePair)
   foreach requiredPackagePair $answer {

      ### The package has been declared
      if {[info exists\
         ::Simple::Package::DeclaredPackages($requiredPackagePair)]} {

         ### Add this required package required packages
         set answer [concat $answer\
            [::Simple::Package::Priv::requiredall-recursive\
            $requiredPackagePair]]
      }
   }

   ### Return the package full list of required packages
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::requiredall
# -purpose    : Returns a package full list of required packages.
# -overview   :
#     This procedure returns the given package list of required packages plus
#  all packages they may require.  It only takes into account the required
#  packages of packages declared via K<::Simple::Package::declare>.
#  In effect, it removes the duplicates from the list returned by
#  K<::Simple::Package::Priv::requiredall-recursive>.
#
# -returns    : The package full list of required packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -seealso    :
#  * The P<::Simple::Package::information requiredall> command.
#
#  * The P<::Simple::Package::Priv::requiredall-recursive> command.
#
proc ::Simple::Package::Priv::requiredall {
   packagePair
} {
   foreach requiredPackagePair\
      [::Simple::Package::Priv::requiredall-recursive $packagePair] {

      set tmp($requiredPackagePair) {}
   }
   array names tmp
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::required-not-installed-recursive
# -purpose    : Returns a package full list of uninstalled required packages.
# -overview   :
#     This procedure returns the given package list of required packages plus
#  all not installed packages it may require.  It only takes into account the
#  required packages of packages declared via K<::Simple::Package::declare>.
#
# -returns    : The package full list of uninstalled required packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -seealso    :
#  * The P<::Simple::Package::information requiredall> command.
#
#  * The P<::Simple::Package::Priv::required-not-instaled> command.
#
# -details    :
#     This command is recursive.
#
# -remarks    :
#  * No provision is made for removing duplicated packages, use
#    K<::Simple::Package::Priv::required-not-installed> instead.
#
proc ::Simple::Package::Priv::required-not-installed-recursive {
   packagePair
} {
   ### Loop over required packages
   set answer [list]
   foreach requiredPackagePair\
      $::Simple::Package::PackageRequiredPackages($packagePair) {

      ### The package has been declared
      if {[info exists\
         ::Simple::Package::DeclaredPackages($requiredPackagePair)]} {

         ### The package has not been installed
         if {![info exists\
            ::Simple::Package::InstalledPackages($requiredPackagePair)]} {

            ### Add this package
            lappend answer $requiredPackagePair

            ### Add this package uninstalled required packages
            set answer [concat $answer\
               [::Simple::Package::Priv::required-not-installed-recursive\
               $requiredPackagePair]]
         }
      }
   }

   ### Return the package full list of required packages
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::required-not-installed
# -purpose    : Returns a package full list of uninstalled required packages.
# -overview   :
#     This procedure returns the given package list of required packages plus
#  all not installed packages it may require.  It only takes into account the
#  required packages of packages declared via K<::Simple::Package::declare>. 
#  In effect, it removes the duplicates from the list returned by
#  K<::Simple::Package::Priv::required-not-installed-recursive>.
#
# -returns    : The package full list of uninstalled required packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -seealso    :
#  * The P<::Simple::Package::information requiredall> command.
#
#  * The P<::Simple::Package::Priv::required-not-instaled> command.
#
proc ::Simple::Package::Priv::required-not-installed {
   packagePair
} {
   foreach requiredPackagePair\
      [::Simple::Package::Priv::required-not-installed-recursive $packagePair] {

      set tmp($requiredPackagePair) {}
   }
   array names tmp
}

### ===========================================================================
### -command  : ::Simple::Package::information install
# -purpose    : Returns a package install script.
# -returns    : The package install script.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
proc {::Simple::Package::information install} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package install script
   set ::Simple::Package::PackageInstallScript($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::information uninstall
# -purpose    : Returns a package uninstall script.
# -returns    : The package uninstall script.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
proc {::Simple::Package::information uninstall} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package uninstall script
   set ::Simple::Package::PackageUninstallScript($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::information aliases
# -purpose    : Returns a package list of aliases.
# -overview   :
#     This procedure returns a package list of aliases as given when
#  declaring the package via K<::Simple::Package::declare>.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -returns    : The package list of aliases.
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -seealso    :
#  * The P<::Simple::Package::information aliasesall> command.
#
proc {::Simple::Package::information aliases} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package aliases
   set ::Simple::Package::PackageAliases($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::information export
# -purpose    : Returns a package list of exported commands.
# -overview   :
#     This procedure returns a package list of exported commands as given
#  when the package was declared via K<::Simple::Package::declare>.
#
# -returns    : The package list of exported commands.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -seealso    :
#  * The P<::Simple::Package:information exportedcommands> command.
#
proc {::Simple::Package::information export} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Return the package exported commands
   set ::Simple::Package::PackageExportedCommands($packagePair)
}

### ===========================================================================
### -command  : ::Simple::Package::information exportedcommands
# -purpose    : Returns a package current list of exported commands.
# -overview   :
#     This procedure returns a package current list of exported commands as
#  given by the K<namespace export> command in each of its namespaces, that
#  is, a snapshot of the currently exported commands is taken.  Thus, the
#  returned list of exported commands might be different from the "exported
#  commands list" argument to K<::Simple::Package::declare>.
#
# -returns    : The package list of unqualified exported commands.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -seealso    :
#  * The P<::Simple::Package:information export> command.
#
#  * The P<::Simple::Package::information exportedcommandsll> command.
#
# -remarks    :
#  * All exported commands are returned, including those containing a
#    percent sign in its name which are assumed to be internal to the Simple
#    Library.  Notice that this behaviour is different from that of the
#    K<::Simple::Package::information publiccommands>,
#    K<::Simple::Package::information privatecommands> and
#    K<::Simple::Package::information commands> commands.
#
proc {::Simple::Package::information exportedcommands} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Loop over package namespaces
   set answer [list]
   foreach namespace $::Simple::Package::PackageNamespaces($packagePair) {

      ### Get the exported commands list from this namespace
      namespace eval $namespace {
         set ::Simple::Package::%TMP% [namespace export]
      }

      ### Add the exported commands to the package list of exported commands
      set answer [concat $answer ${::Simple::Package::%TMP%}]
   }

   ### Return the package list of unqualified exported commands
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::exported-commands
# -purpose    : Returns a package current list of exported commands.
# -overview   :
#     This procedure returns a package current list of exported commands as
#  given by the K<namespace export> command in each of its namespaces, that
#  is, a snapshot of the exported commands is taken.  Thus, the returned
#  list of exported commands might be different from the "exported
#  commands list" argument to K<::Simple::Package::declare>.
#
# -returns    : The package list of fully-qualified exported commands.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -remarks    :
#  * All exported commands are returned, including those containing a
#    percent sign in its name which are assumed to be internal to the Simple
#    Library.  Notice that this behaviour is different from that of the
#    K<::Simple::Package::information publiccommands>,
#    K<::Simple::Package::information privatecommands> and
#    K<::Simple::Package::information commands> commands.
#
proc ::Simple::Package::Priv::exported-commands {
   packagePair
} {
   ### Loop over package namespaces
   set answer [list]
   foreach namespace $::Simple::Package::PackageNamespaces($packagePair) {

      ### Get the exported commands list from this namespace
      namespace eval $namespace {
         set ::Simple::Package::%TMP% [namespace export]
      }

      ### Loop over exported commands
      foreach command ${::Simple::Package::%TMP%} {

         ### Add the command to the package list of exported commands
         lappend answer ${namespace}::$command
      }
   }

   ### Return the package list of fully-qualified exported commands
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::aliases
# -purpose    : Returns a package list of unqualified aliases.
# -returns    : The package list of unqualified aliases.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::aliases {
   packagePair
} {
   ### Loop over package aliases
   set answer [list]
   foreach aliasPair $::Simple::Package::PackageAliases($packagePair) {

      ### Add the alias to the package list of alias
      lappend answer [lindex $aliasPair 0]
   }

   ### Return the package list of unqualified aliases
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information privatecommands
# -purpose    : Returns a package list of private commands.
# -returns    : The package qualified list of private commands.
# -overview   :
#     This procedure returns a package list of private commands.  A private
#  command is that residing in a namespace matching the pattern *::Priv*.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -remarks    :
#  * Commands containing a percent sign in its name are assumed to be
#    internal to the Simple Library and are excluded from the output.  Notice
#    that this behaviour is different from that of the
#    K<::Simple::Package::information exportedcommands> command.
#
proc {::Simple::Package::information privatecommands} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Loop over package namespaces
   set answer [list]
   foreach namespace $::Simple::Package::PackageNamespaces($packagePair) {

      ### Private namespace
      if {[string match *::Priv* $namespace]} {

         ### Get the namespace commands
         # fixme: Tcl 8.3b2 allows [info procs ${namespace}::*]
         namespace eval $namespace {
            set ::Simple::Package::%TMP% [info procs]
         }

         ### Loop over namespace commands
         foreach command ${::Simple::Package::%TMP%} {

            ### The command name does not contain a percent sign
            #   This occurs, for example, with "after flags" auxiliary
            #   commands from the P<SimpleExtProc> package
            if {[string first % $command] == -1} {

               ### Add the command to the package list of private commands
               lappend answer ${namespace}::$command
            }
         }
      }
   }

   ### Return the package list of fully-qualified private commands
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information publiccommands
# -purpose    : Returns a package list of public commands.
# -returns    : The package qualified list of public commands.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -remarks    :
#  * Commands containing a percent sign in its name are assumed to be
#    internal to the Simple Library and are excluded from the output.  Notice
#    that this behaviour is different from that of the
#    K<::Simple::Package::information exportedcommands> command.
#
proc {::Simple::Package::information publiccommands} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Loop over package namespaces
   set answer [list]
   foreach namespace $::Simple::Package::PackageNamespaces($packagePair) {

      ### Non-private namespace
      if {![string match *::Priv* $namespace]} {

         ### Get the namespace commands
         # fixme: Tcl 8.3b2 allows [info procs ${namespace}::*]
         set ::Simple::Package::%TMP1% [list]
         namespace eval $namespace {
            set ::Simple::Package::%TMP1% [info procs]
            set ::Simple::Package::%TMP2% [namespace export]
         }

         ### Loop over namespace commands
         foreach command ${::Simple::Package::%TMP1%} {

            ### The command has not been imported from other namespace
            #   This is because Tcl 8.2.1 and beyond includes imported
            #   commands in the result of [info procs]
            if {![string compare $namespace [namespace qualifiers\
               [namespace origin ${namespace}::$command]]]} {

               ### The command name does not contain a percent sign
               #   This occurs, for example, with "after flags" auxiliary
               #   commands from the P<SimpleExtProc> package
               ### and it is not an exported command
               if {[string first % $command] == -1 && [lsearch -exact\
                  ${::Simple::Package::%TMP2%} $command] == -1} {

                  ### Add the command to the package list of public commands
                  lappend answer ${namespace}::$command
               }
            }
         }
      }
   }

   ### Return the package list of fully-qualified public commands
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information commands
# -purpose    : Returns a package list of public and exported commands.
# -returns    : The package qualified list of public and exported commands.
# -overview   :
#     This procedure returns the given package list of commands available to
#  the user.  This list includes both the exported and public commands. 
#  Notice that the package aliases are not included, use
#  K<::Simple::Package::information alises> for that.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -remarks    :
#  * Commands containing a percent sign in its name are assumed to be
#    internal to the Simple Library and are excluded from the output.  Notice
#    that this behaviour is different from that of the
#    K<::Simple::Package::information exportedcommands> command.
#
proc {::Simple::Package::information commands} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Loop over package namespaces
   set answer [list]
   foreach namespace $::Simple::Package::PackageNamespaces($packagePair) {

      ### Non-private namespace
      if {![string match *::Priv* $namespace]} {

         ### Get the namespace commands
         # fixme: Tcl 8.3b2 allows [info procs ${namespace}::*]
         set ::Simple::Package::%TMP% [list]
         namespace eval $namespace {
            set ::Simple::Package::%TMP% [info procs]
         }

         ### Loop over namespace commands
         foreach command ${::Simple::Package::%TMP%} {

            ### The command has not been imported from other namespace
            #   This is because Tcl 8.2.1 and beyond includes imported
            #   commands in the result of [info procs]
            if {![string compare $namespace [namespace qualifiers\
               [namespace origin ${namespace}::$command]]]} {

               ### The command name does not contain a percent sign
               #   This occurs, for example, with "after flags" auxiliary
               #   commands from the P<SimpleExtProc> package
               if {[string first % $command] == -1} {

                  ### Add the command to the package list
                  ### of public and exported commands
                  lappend answer ${namespace}::$command
               }
            }
         }
      }
   }

   ### Return the package list of fully-qualified public and exported commands
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information exportedcommandsall
# -purpose    : Returns a package full list of exported commands.
# -overview   :
#     This procedure returns the given package list of exported commands plus
#  those exported from all packages it may require.  It only takes into
#  account the required packages of packages declared via
#  K<::Simple::Package::declare>.
#
#     For each package, the list of exported commands is as given by
#  the K<namespace export> command in each of its namespaces, that is, a
#  snapshot of the exported commands is taken.  Thus, the returned list of
#  exported commands might be different from the "exported commands list"
#  argument to K<::Simple::Package::declare>.
#
# -returns    : The package full list of unqualified exported commands.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -seealso    :
#  * The P<::Simple::Package::information exportedcommands> command.
#
proc {::Simple::Package::information exportedcommandsall} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Get all required packages
   set requiredPackagePairs\
      [::Simple::Package::Priv::requiredall $packagePair]
   lappend requiredPackagePairs $packagePair

   ### Loop over required packages
   set answer [list]
   foreach requiredPackagePair $requiredPackagePairs {

      ### The package has been declared
      if {[info exists\
         ::Simple::Package::DeclaredPackages($requiredPackagePair)]} {

         ### Loop over fully-qualified package exported commands
         foreach command\
            [::Simple::Package::Priv::exported-commands $requiredPackagePair] {

            ### Add the command to the package list of exported commands
            lappend answer [namespace tail $command]
         }
      }
   }

   ### Return the package full list of unqualified exported commands
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information aliasesall
# -purpose    : Returns a package full list of aliases.
# -overview   :
#     This procedure returns the given package list of aliases plus those from
#  all packages it may require.  It only takes into account the required
#  packages of packages declared via K<::Simple::Package::declare>.
#
#     For each package, the list of aliases is as given when the package was
#  declared via K<::Simple::Package::declare>.
#
# -returns    : The package full list of aliases.
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
# -seealso    :
#  * The P<::Simple::Package::information aliases> command.
#
proc {::Simple::Package::information aliasesall} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Get all required packages
   set requiredPackagePairs\
      [::Simple::Package::Priv::requiredall $packagePair]
   lappend requiredPackagePairs $packagePair

   ### Loop over required packages
   set answer [list]
   foreach requiredPackagePair $requiredPackagePairs {

      ### The package has been declared
      if {[info exists\
         ::Simple::Package::DeclaredPackages($requiredPackagePair)]} {

         ### Add this package aliases to the full list of aliases
         set answer [concat $answer\
            $::Simple::Package::PackageAliases($requiredPackagePair)]
      }
   }

   ### Return the package full list aliases
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information collisions
# -purpose    : Returns the list of collisions upon importing a package.
# -overview   :
#     This procedure returns the list of fully-qualified exported commands and
#  unqualified aliases which would collide if the given package were imported
#  into the given namespace.  Collisions occur when one of the package
#  exported commands is named as an already existing command in the target
#  namespace or as a global alias, or when one of the package aliases is named
#  as an already existing global command or alias.
#
#     If no namespace is given, it defaults to the global one.  If the target
#  namespace is the global one, the list of collisions takes into account not
#  only the exported command and aliases from the given package but also those
#  from any not previously installed package it may require.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?namespace?       -namespace ::  {Namespace}}
#
# -returns    : The list of colliding fully-qualified exported commands and
#  unqualified aliases upon importing a package.
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
proc {::Simple::Package::information collisions} {
   package
   {namespace ::}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package 0.0.0]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Assert the namespace exists
   ::Simple::Package::Priv::assert-namespace-exists $namespace

   ### Get all packages to import
   set packagesToImport\
      [::Simple::Package::Priv::packages-to-import $packagePair $namespace]

   ### Get all packages to import exported commands and aliases
   #   Loop over packages to import
   set collisionCandidates [list]
   foreach packageToImport $packagesToImport {

      # Add this package exported commands and aliases
      set collisionCandidates [concat $collisionCandidates\
         [::Simple::Package::Priv::exported-commands $packageToImport]\
         [::Simple::Package::Priv::aliases $packageToImport]]
   }

   ### Get the namespace commands
   set namespaceCommandNames [list]
   foreach namespaceCommand [info commands ${namespace}::*] {
      lappend namespaceCommandNames [namespace tail $namespaceCommand]
   }

   ### Loop over collision candidates
   set answer [list]
   foreach collisionCandidate $collisionCandidates {

      ### There is a namespace command named as the collision cndidate
      if {[lsearch -exact $namespaceCommandNames\
         [namespace tail $collisionCandidate]] != -1} {

         ### Add this command to the list of collisions
         lappend answer $collisionCandidate
      }
   }

   ### Return the list of collisions
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::information importedinto
# -purpose    : Returns the namespaces into which a package has been imported.
# -overview   :
#     This procedure returns the list of namespaces into which the given
#  package has been imported or the empty list for uninstalled packages.
#
# -returns    : The list of namespaces into which the package has been imported
#     or the empty list for uninstalled packages.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -limitations:
#  * The version default value is set to "0.0.0" for
#    K<::Simple::Package::Priv::package-pair> to work.
#
proc {::Simple::Package::information importedinto} {
   package
   {version 0.0.0}
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package $version]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### The package is installed
   set answer [list]
   if {[info exists ::Simple::Package::InstalledPackages($packagePair)]} {

      ### Add the namespaces into which the package has been imported
      foreach element [array names ::Simple::Package::PackageImportedInto\
         $packagePair%*] {

         lappend answer $::Simple::Package::PackageImportedInto($element)
      }
   }

   ### Return the list of namespaces into which the package has been imported
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::packages-to-import
# -purpose    : Returns the full list of packages to import.
# -overview   :
#     This procedure returns the full list of packages to import into a given
#  namespace upon importing the given package.  If the target namespace is not
#  the global one, that list is made of the given package only; this
#  corresponds to the use by K<::Simple::Package::import>.  If the target
#  namespace is the global one, the list includes not only the given package
#  but also any not previously installed package it may require.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { namespace        -namespace     {Namespace}}
#
proc ::Simple::Package::Priv::packages-to-import {
   packagePair
   namespace
} {
   ### Non-global namespace
   if {[string compare $namespace ::]} {

      ### The package is the only one to import
      set answer [list $packagePair]

   ### Global namespace
   } else {

      ### Add the package full list of uninstalled required
      ### packages to the list of packages to import
      set answer\
         [::Simple::Package::Priv::required-not-installed $packagePair]

      ### Add the package to the list of packages to import
      lappend answer $packagePair
   }

   ### Return the full list of packages to import
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::install
# -purpose    : Installs a package.
# -overview   :
#     This procedure installs a package previously declared via the
#  K<::Simple::Package::declare> command.  The following tasks are performed
#  in this same order:
#  * All package required packages not previously installed are installed. 
#
#  * All package aliases are created.
#
#  * The package install script is evaluated in the global namespace. 
#
#  * If this is the first time the package is installed after its declaration,
#    the package first time install script is evaluated in the global
#    namespace. 
#
#  * All package exported commands (either from the package itself or from any
#    not previously installed package it may require) are imported into the
#    global namespace.
#
#     The install script is evaluated every time the package is installed
#  while the first time install script is only evaluated the first time the
#  package is installed after its declaration; upon evaluation, the first time
#  install script is deleted.  The first time install script is intended for
#  the initialisation of the package static items, notably its commands.  On
#  the other hand, the install script is intended for the initialisation of
#  the package non-static items such as its variables.  Notice that all the
#  packages required by the package being installed are guaranteed to be
#  installed when evaluating the install and first time install scripts.
#
#     An error is thrown if the package is already installed; use
#  K<::Simple::Package::uninstall> to uninstall a package.
#
#     An error is thrown if importing the package into the global namespace
#  would result in a collision; use K<::Simple::Package::information
#  collisions> to ensure there are no collisions.
#
# -arguments  :
#  { package          -name          {Package name}}
#
# -effects    :
#  * Creates the package aliases.
#
#  * Those of the evaluated scripts.
#
#  * Imports a package exported commands into the global namespace.
#
proc ::Simple::Package::install {
   package
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package 0.0.0]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Continue with the second part of the installation
   ::Simple::Package::Priv::install-2 $packagePair
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::install-2
# -purpose    : Installs a package (second part).
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::install-2 {
   packagePair
} {
   ### Assert the package is not installed
   ::Simple::Package::Priv::assert-package-not-installed $packagePair

   ### Assert no collision will occur when importing
   ###    the package into the global namespace
   ::Simple::Package::Priv::assert-no-collisions $packagePair ::

   ### Install the package
   ::Simple::Package::Priv::do-install $packagePair
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::do-install
# -purpose    : Installs a package.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
# -effects    :
#  * Creates the package aliases.
#
#  * Those of the evaluated scripts.
#
proc ::Simple::Package::Priv::do-install {
   packagePair
} {
   ### Install the package required packages
   ::Simple::Package::Priv::install-required $packagePair

   ### Create the package aliases
   foreach aliasPair $::Simple::Package::PackageAliases($packagePair) {
      foreach {alias aliased} $aliasPair break
      interp alias {} $alias {} $aliased
   }

   ### Error evaluating the package install script
   if {[catch {namespace eval ::\
      $::Simple::Package::PackageInstallScript($packagePair)} result]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Package::ERROR-EVAL-SCRIPT\
         $packagePair install $result

   ### No error evaluating the package install script
   } else {

      ### Error evaluating the package first time install script
      if {[catch {namespace eval ::\
         $::Simple::Package::PackageFirstInstallScript($packagePair)} result]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::Package::ERROR-EVAL-SCRIPT\
            $packagePair {first time install} $result

      ### No error evaluating the package first time install script
      } else {

         ### Delete the package first time install script
         set ::Simple::Package::PackageFirstInstallScript($packagePair) {}

         ### Import the package into the global namespace
         ::Simple::Package::Priv::do-import $packagePair ::

         ### Add this package to the list of installed packages
         ::Simple::Package::Priv::add-installed $packagePair
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::assert-no-collisions
# -purpose    : Throws an error if importing a package would give a collision.
# -overview   :
#     This procedure throws an error if importing the given package into the
#  given namespace would result in a collision.  Collisions occur when one of
#  the package exported commands is named as an already existing command in
#  the target namespace or as a global alias, or when one of the package
#  aliases is named as an already existing global command or alias.
#
#     If the target namespace is the global one, the list of collisions takes
#  into account not only the exported command and aliases from the given
#  package but also those from any not previously installed package it may
#  require.
#
#  There are two possibilities:
#  * A namespace command or global alias collides with an exported command
#    or package alias.
#
#  * An exported command or package alias from a namespace collides with
#    another from a second namespace.
#
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { namespace        -namespace     {Namespace}}
#
proc ::Simple::Package::Priv::assert-no-collisions {
   packagePair
   namespace
} {
   ### Get the list of collisions
   foreach {package version} $packagePair break
   set collisions\
      [{::Simple::Package::information collisions} $package $namespace]

   ### Non-global namespace
   if {[string compare $namespace ::]} {

      ### Importing to a non-global namespace
      set actionMessage import
      set namespaceMessage "namespace \"$namespace\""

   ### Global namespace
   } else  {

      ### Installing to the global namespace
      set actionMessage install
      set namespaceMessage {the global namespace}
   }

   ### A namespace command collides with an exported command or alias
   if {[llength $collisions] != 0} {

      ### The collision is an exported command
      set collision [lindex $collisions 0]
      if {[string match ::* $collision]} {

         ### Throw error
         set collidingPackage [{::Simple::Package::information\
            packagenamespace} [namespace qualifiers $collision]]
         ::Simple::Error::throw ::Simple::Package::COLLISION\
            $actionMessage $packagePair [namespace tail $collision]\
            $namespaceMessage {exported command} $collision $collidingPackage

      ### The collision is an alias
      } else {

         ### Throw error
         set collidingPackage\
            [{::Simple::Package::information packagealias} $collision]
         ::Simple::Error::throw ::Simple::Package::COLLISION\
            $actionMessage $packagePair [namespace tail $collision]\
            $namespaceMessage alias $collision $collidingPackage
      }
   }

   ### Loop over packages to import
   foreach packageToImport\
      [::Simple::Package::Priv::packages-to-import $packagePair $namespace] {

      ### Loop over exported commands and aliases
      foreach commandOrAlias [concat\
         [::Simple::Package::Priv::exported-commands $packageToImport]\
         [::Simple::Package::Priv::aliases $packageToImport]] {

         ### The command or alias would have been previously imported or created
         set commandOrAliasName [namespace tail $commandOrAlias]
         if {[info exists\
            importedCommandsAndCreatedAlias($commandOrAliasName)]} {

            ### Throw error
            foreach {package namespace}\
               $importedCommandsAndCreatedAlias($commandOrAliasName) break
            if {[string compare $namespace {}]} {
               set namespaceMessage1 " (namespace \"$namespace\")"
            } else {
               set namespaceMessage1 {}
            }
            set namespace [namespace qualifiers $commandOrAlias]
            if {[string compare $namespace {}]} {
               set namespaceMessage2 " (namespace \"$namespace\")"
            } else {
               set namespaceMessage2 {}
            }
            ::Simple::Error::throw\
               ::Simple::Package::COLLISION-BETWEEN-PACKAGES $actionMessage\
               $packagePair $commandOrAliasName $package\
               $namespaceMessage1 $packageToImport $namespaceMessage2

         ### The command or alias would not have
         ###    been previously imported or created
         } else {

            ### Register the command or alias as previously imported or created
            set importedCommandsAndCreatedAlias($commandOrAliasName)\
               [list $packageToImport [namespace qualifiers $commandOrAlias]]
         }
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::install-required
# -purpose    : Installs a package required packages.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#
proc ::Simple::Package::Priv::install-required {
   packagePair
} {
   ### Loop over required packages
   foreach requiredPackagePair\
      $::Simple::Package::PackageRequiredPackages($packagePair) {

      ### Declared but not installed package
      if {[info exists\
         ::Simple::Package::DeclaredPackages($requiredPackagePair)] && ![info\
         exists ::Simple::Package::InstalledPackages($requiredPackagePair)]} {

         ### Install the package
         ::Simple::Package::Priv::do-install $requiredPackagePair
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::do-import
# -purpose    : Imports a package exported commands into a namespace.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { namespace        -namespace     {Namespace}}
#
# -effects    :
#  * Imports a package exported commands into the namespace.
#
proc ::Simple::Package::Priv::do-import {
   packagePair
   namespace
} {
   ### Import the package exported commands into the namespace
   set ::Simple::Package::%TMP%\
      $::Simple::Package::PackageNamespaces($packagePair)
   namespace eval $namespace {
      foreach packageNamespace ${::Simple::Package::%TMP%} {

         namespace import ${packageNamespace}::*
      }
   }

   ### Add the namespace to the package list of namespaces imported into
   ::Simple::Package::Priv::add-importedinto $packagePair $namespace
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::add-importedinto
# -purpose    : Adds a namespace to a package list of namespaces imported into.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { namespace        -namespace     {Namespace}}
#
proc ::Simple::Package::Priv::add-importedinto {
   packagePair
   namespace
} {
   set ::Simple::Package::PackageImportedInto($packagePair%$namespace)\
      $namespace
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::delete-importedinto
# -purpose    : Deletes a package list of namespaces imported into.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  {?namespace?       -namespace {}  {Namespace}}
#
proc ::Simple::Package::Priv::delete-importedinto {
   packagePair
   {namespace {}}
} {
   ### No namespace given
   if {![string compare $namespace {}]} {

      ### Delete the package list of namespaces imported into
      foreach element [array names ::Simple::Package::PackageImportedInto\
         $packagePair%*] {

         unset ::Simple::Package::PackageImportedInto($element)
      }
   } else {

      ### Delete the namespace from the list of namespaces imported into
      unset ::Simple::Package::PackageImportedInto($packagePair%$namespace)
   }
}

### ===========================================================================
### -command  : ::Simple::Package::require-and-install
# -purpose    : Requires and installs a package.
# -overview   :
#     This procedure changes a package state from I<available> to
#  I<installed>.  If the package is declared via the
#  K<::Simple::Package::declare> command upon requiring it, the package is
#  then installed.  Notice that otherwise a full installation might require
#  manually importing the package namespaces via K<namespace import>.
#
#     No error is thrown if the package is already provided or installed.
#
#     An error is thrown if the package is not available.
#
# -arguments  :
#  { package          -name          {Package name}}
#  {?version?         -version 0.0.0 {Package version}}
#
# -returns    : The required package version.
# -limitations:
#  * To identify whether a version is given or not its default value
#    is set to "0.0.0".
#
proc ::Simple::Package::require-and-install {
   package
   {version 0.0.0}
} {
   ### Require the package
   set packagePair\
      [::Simple::Package::Priv::find-package [list $package $version]]
   set answer [::Simple::Package::Priv::require-package $packagePair]

   ### Declared but not installed package
   if {[info exists ::Simple::Package::DeclaredPackages($packagePair)] &&\
      ![info exists ::Simple::Package::InstalledPackages($packagePair)]} {

      ### Continue with the second part of the installation
      ::Simple::Package::Priv::install-2 $packagePair
   }

   ### Return the required package version
   set answer
}

### ===========================================================================
### -command  : ::Simple::Package::import
# -purpose    : Imports a package exported commands into a namespace.
# -overview   :
#     This procedure imports the given installed package exported commands
#  into the given namespace.
#
#     An error is thrown if the package is not installed.
#
#     An error is thrown if the package has already been imported into the
#  given namespace.  This implies that the namespace can not be the global
#  one, as any packages is imported into the global namespace upon
#  installation.
#
#     An error is thrown if importing the package exported commands and
#  aliases into the given namespace would result in a collision; use
#  K<::Simple::Package::information collisions> to ensure there are no
#  collisions.
#
# -arguments  :
#  { package          -name          {Package name}}
#  { namespace        -namespace     {Namespace}}
#
# -effects    :
#  * Imports a package exported commands into the namespace.
#
proc ::Simple::Package::import {
   package
   namespace
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package 0.0.0]

   ### Assert the package is installed
   ::Simple::Package::Priv::assert-package-installed $packagePair

   ### Assert the namespace exists
   ::Simple::Package::Priv::assert-namespace-exists $namespace

   ### The package has not been imported into the namespace
   if {![info exists\
      ::Simple::Package::PackageImportedInto($packagePair%$namespace)]} {

      ### Assert no collision will occur when importing the package
      ::Simple::Package::Priv::assert-no-collisions $packagePair\
         $namespace

      ### Import the package into the namespace
      ::Simple::Package::Priv::do-import $packagePair $namespace

   ### The package has already been imported into the namespace
   } else {
      ::Simple::Error::throw ::Simple::Package::ALREADY-IMPORTED\
         $packagePair $namespace
   }
}

### ===========================================================================
### -command  : ::Simple::Package::unimport
# -purpose    : Unimports a package exported commands from a namespace.
# -overview   :
#     This procedure unimports the given package exported commands from the
#  given namespace.
#
#     An error is thrown if the package is not installed.
#
#     An error is thrown if the package has not been imported into the
#  given namespace.
#
# -arguments  :
#  { package          -name          {Package name}}
#  { namespace        -namespace     {Namespace}}
#
# -effects    :
#  * Forgets a package exported commands from the namespace.
#
proc ::Simple::Package::unimport {
   package
   namespace
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package 0.0.0]

   ### Assert the package is installed
   ::Simple::Package::Priv::assert-package-installed $packagePair

   ### Assert the namespace exists
   ::Simple::Package::Priv::assert-namespace-exists $namespace

   ### The package has been imported into the namespace
   if {[info exists\
      ::Simple::Package::PackageImportedInto($packagePair%$namespace)]} {

      ### Unimport the package from the namespace
      ::Simple::Package::Priv::do-unimport $packagePair $namespace

   ### The package has not been imported into the namespace
   } else {

      ### Throw error
      ::Simple::Error::throw ::Simple::Package::NOT-IMPORTED\
         $packagePair $namespace
   }
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::do-unimport
# -purpose    : Unimports a package exported commands from a namespace.
# -arguments  :
#  { packagePair      -list          {Package in the format {name version}}}
#  { namespace        -namespace     {Namespace}}
#
# -effects    :
#  * Forgets a package exported commands from the namespace.
#
proc ::Simple::Package::Priv::do-unimport {
   packagePair
   namespace
} {
   ### Unimport the package exported commands into the namespace
   set ::Simple::Package::%TMP%\
         $::Simple::Package::PackageNamespaces($packagePair)
   namespace eval $namespace {
      foreach packageNamespace ${::Simple::Package::%TMP%} {

         namespace forget ${packageNamespace}::*
      }
   }

   ### Remove the namespace from the package list of namespaces imported into
   ::Simple::Package::Priv::delete-importedinto $packagePair $namespace
}

### ===========================================================================
### -command  : ::Simple::Package::uninstall
# -purpose    : Uninstalls a package.
# -overview   :
#     This procedure uninstalls a package previously installed via the
#  K<::Simple::Package::install> command.  The following tasks are performed
#  in this same order:
#  * All package exported commands are unimported from all namespace they were
#    imported into.
#
#  * The package uninstall script is executed in the global namespace.
#
#  * All package aliases are deleted.
#
#     An error is thrown if the package is not installed.
#
# -arguments  :
#  { package          -name          {Package name}}
#
# -remarks    :
#  * Notice that unlike K<::Simple::Package::install> which installs all
#    packages required by the package being installed,
#    K<::Simple::Package::uninstall> uninstalls the given package only.
#
# -effects    :
#  * Those of the evaluated script.
#
#  * Deletes the package aliases.
#
#  * Forgets a package exported commands from the namespace.
#
proc ::Simple::Package::uninstall {
   package
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package 0.0.0]

   ### Assert the package is installed
   ::Simple::Package::Priv::assert-package-installed $packagePair

   ### Unimport the package from the namespaces it has been imported into
   foreach element [array names ::Simple::Package::PackageImportedInto\
      $packagePair%*] {

      ::Simple::Package::Priv::do-unimport $packagePair\
         $::Simple::Package::PackageImportedInto($element)
   }

   ### Error evaluating the package uninstall script
   if {[catch {namespace eval ::\
      $::Simple::Package::PackageUninstallScript($packagePair)} result]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Package::ERROR-EVAL-SCRIPT\
         $packagePair uninstall $result

   ### No error evaluating the package uninstall script
   } else {

      ### Delete this package from the list of installed packages
      ::Simple::Package::Priv::delete-installed $packagePair
   }

   ### Delete the package aliases
   foreach aliasPair $::Simple::Package::PackageAliases($packagePair) {
      interp alias {} [lindex $aliasPair 0] {}
   }
}

### ===========================================================================
### -command  : ::Simple::Package::undeclare
# -purpose    : Undeclares a package.
# -overview   :
#     This procedure undeclares a package previously declared via the
#  K<::Simple::Package::install> command.
#
#     An error is thrown if the package is installed; use
#  K<::Simple::Package::uninstall> to uninstall a package.
#
# -arguments  :
#  { package          -name          {Package name}}
#
# -remarks    :
#  * Notice that unlike K<::Simple::Package::declare> which requires all
#    packages required by the package being declared,
#    K<::Simple::Package::undeclare> undeclares the given package only.
#
# -effects    :
#  * Deletes the package namespaces.
#
#  * Forgets a package, but keeps the ifneeded script, if any.
#
proc ::Simple::Package::undeclare {
   package
} {
   ### Build the package pair
   set packagePair [::Simple::Package::Priv::package-pair $package 0.0.0]

   ### Assert the package has been declared
   ::Simple::Package::Priv::assert-package-declared $packagePair

   ### Assert the package is not installed
   ::Simple::Package::Priv::assert-package-not-installed $packagePair

   ### Delete the package namespaces
   ::Simple::Package::Priv::delete-namespaces $packagePair

   ### Delete the package namespaces information
   ::Simple::Package::Priv::delete-namespaces-info $packagePair

   ### Delete the package required packages information
   unset ::Simple::Package::PackageRequiredPackages($packagePair)

   ### Delete the package list of namespaces imported into
   ::Simple::Package::Priv::delete-importedinto $packagePair

   ### Delete the package exported commands
   unset ::Simple::Package::PackageExportedCommands($packagePair)

   ### Delete the package aliases
   unset ::Simple::Package::PackageAliases($packagePair)

   ### Delete the package install script
   unset ::Simple::Package::PackageInstallScript($packagePair)

   ### Delete the package first time install script
   unset ::Simple::Package::PackageFirstInstallScript($packagePair)

   ### Delete the package uninstall script
   unset ::Simple::Package::PackageUninstallScript($packagePair)

   ### Delete this package from the list of declared packages
   ::Simple::Package::Priv::delete-declared $packagePair

   ### Set the state of the package to available or unknown
   foreach {package version} $packagePair break
   ::Simple::Package::Priv::available-or-unknown $package $version
}

### ===========================================================================
### -command  : ::Simple::Package::Priv::available-or-unknown
# -purpose    : Sets the state of a package to available or unknown.
# -overview   :
#     This procedure sets the state of a package to available or unknown
#  depending on whether or not there is an ifneeded script for the package.
#
# -arguments  :
#  { package          -name          {Package name}}
#  { version          -version       {Package version}}
#
# -effects    :
#  * Forgets a package, but keeps the ifneeded script, if any.
#
proc ::Simple::Package::Priv::available-or-unknown {
   package
   version
} {
   ### Save the package ifneeded script
   set ifneededScript [package ifneeded $package $version]

   ### Forget the package
   #   K<package forget> removes the package ifneeded script
   package forget $package $version

   ### Restore the package ifneeded script
   if {[llength $ifneededScript] > 0} {
      package ifneeded $package $version $ifneededScript
   }
}

### ===========================================================================
### -command  : ::Simple::Package::uninstall-and-undeclare
# -purpose    : Uninstalls and undeclares a package.
# -overview   :
#     This procedure changes a package state from I<installed> to
#  I<available>.  If the package was previously declared via the
#  K<::Simple::Package::install> command, the package is first uninstalled and
#  then undeclared; otherwise the package is simply "unprovided".  Notice that
#  in the later case a full uninstallation might require manually forgeting
#  the package namespaces via K<namespace forget>. 
#
#     An error is thrown if the package is not installed.
#
# -arguments  :
#  { package          -name          {Package name}}
#
# -remarks    :
#  * Notice that unlike K<::Simple::Package::install> which installs all
#    packages required by the package being installed,
#    K<::Simple::Package::uninstall-and-undeclare> uninstalls the given
#    package only.
#
# -effects    :
#  * Forgets a package, but keeps the ifneeded script, if any.
#
proc ::Simple::Package::uninstall-and-undeclare {
   package
} {
   ### Undeclared package
   if {[llength [array names\
      ::Simple::Package::DeclaredPackages "$package *"]] == 0} {

      ### Assert the package is installed
      set version [package provide $package]
      if {![string compare $version {}]} {
         ::Simple::Error::throw ::Simple::Package::UNINSTALLED $package
      }

      ### Set the state of the package to available or unknown
      ::Simple::Package::Priv::available-or-unknown $package $version

   ### Declared package
   } else {

      ### Uninstall the package
      ::Simple::Package::uninstall $package

      ### Undeclare the package
      ::Simple::Package::undeclare $package
   }
}

### ===========================================================================
### -command  : ::Simple::Package::configure
# -purpose    : Configures the package options.
#
proc ::Simple::Package::configure {
} {
   ::Simple::Error::throw ::Simple::NO-OPTIONS
}

### ===========================================================================
### -command  : ::Simple::Package::cget
# -purpose    : Gets the package options.
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Package::cget {
} {
   list
}

# Declare the package
::Simple::Package::Priv::declare

# Install the package
::Simple::Package::require-and-install SimplePackage

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimplePackage-test 0.2

### ===========================================================================
### -test     : ::Simple::Package::information
test-case info-1 {
   ::Simple::Package::information
} -script {
   expr {[::Simple::Package::information known] > 0}
} -return 1

test-case info-2 {
   ::Simple::Package::information
   subcommand throws an error
} -setup {

   # Create a subcommand procedure
   proc {::Simple::Package::information foo} {} {
      return -code error -errorcode errorCode errorMessage
   }

} -script {
   set return [catch {::Simple::Package::information foo} result]
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
} -cleanup {

   # Delete the subcommand procedure
   rename {::Simple::Package::information foo} {}

} -output {return: {1}
result: {errorMessage}
errorCode: {errorCode}
}

test-case info-3 {
   ::Simple::Package::information
   error, no subcommand given
} -script {
   ::Simple::Package::information
} -error {wrong # args: should be "::Simple::Package::information option ?arg\
...?"}

test-case info-4 {
   ::Simple::Package::information
   error, non-existing subcommand
} -regexp -script {
   ::Simple::Package::information foo
} -error {bad option "foo": *}

test-case info-5 {
   ::Simple::Package::information
   error, invalid subcommand arguments
} -script {
   ::Simple::Package::information known foo bar
} -error {wrong # args: should be "::Simple::Package::information known\
?packagePattern?"}

### ===========================================================================
### -test     : ::Simple::Package::information known
test-case known-1 {
   ::Simple::Package::information known
} -script {

   # Initial number of available packages
   set nPackages [llength [::Simple::Package::information known]]

   # package ifneeded
   package ifneeded Undeclared1 1.0 {
      package provide Undeclared1 1.0
   }
   puts -nonewline [expr\
      {[llength [::Simple::Package::information known]] - $nPackages}]
   puts ": <<[lsort [::Simple::Package::information known Undeclared*]]>>"

   # package require
   package require Undeclared1 1.0
   puts -nonewline [expr\
      {[llength [::Simple::Package::information known]] - $nPackages}]
   puts ": <<[lsort [::Simple::Package::information known Undeclared*]]>>"

   # A second package ifneeded
   package ifneeded Undeclared2 1.0 {}
   puts -nonewline [expr\
      {[llength [::Simple::Package::information known]] - $nPackages}]
   puts ": <<[lsort [::Simple::Package::information known Undeclared*]]>>"

   # package forget
   package forget Undeclared1
   puts -nonewline [expr\
      {[llength [::Simple::Package::information known]] - $nPackages}]
   puts ": <<[lsort [::Simple::Package::information known Undeclared*]]>>"

   # package provide
   package provide Undeclared3 1.0
   puts -nonewline [expr\
      {[llength [::Simple::Package::information known]] - $nPackages}]
   puts ": <<[lsort [::Simple::Package::information known Undeclared*]]>>"

   # package forget
   package forget Undeclared2
   package forget Undeclared3
   puts -nonewline [expr\
      {[llength [::Simple::Package::information known]] - $nPackages}]
   puts ": <<[lsort [::Simple::Package::information known Undeclared*]]>>"

} -output {1: <<{Undeclared1 1.0}>>
1: <<{Undeclared1 1.0}>>
2: <<{Undeclared1 1.0} {Undeclared2 1.0}>>
1: <<{Undeclared2 1.0}>>
2: <<{Undeclared2 1.0} {Undeclared3 1.0}>>
0: <<>>
}

test-case known-2 {
   ::Simple::Package::information known
   provided version has precedence over equivalent version
} -setup {

   # Two available versions
   package ifneeded Undeclared 1.0 {}
   package ifneeded Undeclared 2.0 {}

   # Provided version equivalent to one available version
   package provide Undeclared 1

} -script {
   puts [::Simple::Package::information known Undeclared*]
} -cleanup {

   # Clean the packages
   package forget Undeclared 1.0
   package forget Undeclared 2.0

} -output {{Undeclared 1} {Undeclared 2.0}
}

### ===========================================================================
### -test     : ::Simple::Package::information declared
test-case declared-1 {
   ::Simple::Package::information declared
} -script {

   # Two packages initially declared (SimplePackage and SimpleError)
   puts "declared:[llength [::Simple::Package::information declared]]"

   # Declared
   ::Simple::Package::Priv::add-declared {Declared1 1.0}

   # Declared+installed
   ::Simple::Package::Priv::add-declared {Declared2 1.0}
   ::Simple::Package::Priv::add-installed {Declared2 1.0}

   # Four packages declared
   puts "declared:[llength [::Simple::Package::information declared]]"

   # One package declared
   puts "declared:[::Simple::Package::information declared *1]"

} -output {declared:2
declared:4
declared:{Declared1 1.0}
}

### ===========================================================================
### -test     : ::Simple::Package::information uninstalled
test-case uninstalled-1 {
   ::Simple::Package::information uninstalled
} -prerequisites {
   {[llength [::Simple::Package::information declared]] == 4}
} -script {

   # One package initially uninstalled
   puts "uninstalled:[::Simple::Package::information uninstalled]"

   # Declare another package
   ::Simple::Package::Priv::add-declared {Declared3 1.0}

   # Two packages uninstalled
   puts "uninstalled:[lsort\
      [::Simple::Package::information uninstalled Declared*]]"

} -cleanup {

   # Clean the packages
   ::Simple::Package::Priv::delete-declared {Declared1 1.0}
   ::Simple::Package::Priv::delete-declared {Declared2 1.0}
   ::Simple::Package::Priv::delete-installed {Declared2 1.0}
   ::Simple::Package::Priv::delete-declared {Declared3 1.0}

} -output {uninstalled:{Declared1 1.0}
uninstalled:{Declared1 1.0} {Declared3 1.0}
}

### ===========================================================================
### -test     : ::Simple::Package::information installed
test-case installed-1 {
   ::Simple::Package::information installed
} -script {

   # No packages initially installed
   puts "installed:[::Simple::Package::information installed *eclared*]"

   # Installed via package provide
   package provide Undeclared 1.0

   # Installed via declare+install
   package provide Declared 1.0
   ::Simple::Package::Priv::add-declared {Declared 1.0}
   ::Simple::Package::Priv::add-installed {Declared 1.0}

   # Two packages installed
   puts "installed:[lsort\
      [::Simple::Package::information installed *eclared*]]"

} -cleanup {

   # Clean the packages
   package forget Undeclared
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   ::Simple::Package::Priv::delete-installed {Declared 1.0}

} -output {installed:
installed:{Declared 1.0} {Undeclared 1.0}
}

### ===========================================================================
### -test     : ::Simple::Package::information state
test-case state-1 {
   ::Simple::Package::information state
} -setup {

   # Available
   package ifneeded SimpleAvailable 1.0 script
   package ifneeded SimpleVarious 1.0 script

   # Declared
   package provide Declared 1.0
   ::Simple::Package::Priv::add-declared {Declared 1.0}
   package provide SimpleVarious 2.0
   ::Simple::Package::Priv::add-declared {SimpleVarious 2.0}

   # Installed
   package provide SimpleInstalled1 1.0
   package provide SimpleInstalled2 1.0
   ::Simple::Package::Priv::add-declared {SimpleInstalled2 1.0}
   ::Simple::Package::Priv::add-installed {SimpleInstalled2 1.0}

} -script {

   # Unknown
   puts [::Simple::Package::information state Unknown 1]

   # Available
   puts [::Simple::Package::information state SimpleAvailable 1.0]
   puts [::Simple::Package::information state SimpleVarious 1]

   # Declared
   puts [::Simple::Package::information state Declared 1]
   puts [::Simple::Package::information state SimpleVarious 2.0]

   # Installed
   puts [::Simple::Package::information state SimpleInstalled1 1.0]
   puts [::Simple::Package::information state SimpleInstalled2 1]

} -cleanup {

   # Clean the packages
   package forget SimpleAvailable
   package forget SimpleVarious 1.0
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   package forget SimpleVarious 2.0
   ::Simple::Package::Priv::delete-declared {SimpleVarious 2.0}
   package forget SimpleInstalled1
   package forget SimpleInstalled2
   ::Simple::Package::Priv::delete-declared {SimpleInstalled2 1.0}
   ::Simple::Package::Priv::delete-installed {SimpleInstalled2 1.0}

} -output {unknown
available
available
declared
declared
installed
installed
}

test-case state-2 {
   ::Simple::Package::information state
   package version not given
} -script {

   # Unknown
   puts [::Simple::Package::information state Unknown]

   # Available
   package ifneeded Undeclared 1.0 {
      package provide Undeclared 1.0
   }
   package ifneeded Undeclared 2.0 {
      package provide Undeclared 2.0
   }
   puts [::Simple::Package::information state Undeclared]
   puts [::Simple::Package::information state Undeclared 1]
   puts [::Simple::Package::information state Undeclared 2.0]
   package ifneeded Declared 1.0 {
      package provide Declared 1.0
      ::Simple::Package::Priv::add-declared {Declared 1.0}
   }
   package ifneeded Declared 2.0 {
      package provide Declared 2.0
      ::Simple::Package::Priv::add-declared {Declared 1.0}
   }
   puts [::Simple::Package::information state Declared]
   puts [::Simple::Package::information state Declared 1.0]
   puts [::Simple::Package::information state Declared 2]

   # Undeclared, installed
   package require Undeclared 1.0
   puts [::Simple::Package::information state Undeclared]

   # Declared, declared
   package require Declared 1.0
   puts [::Simple::Package::information state Declared]
   puts [::Simple::Package::information state Declared 1]
   puts [::Simple::Package::information state Declared 2.0]

   # Declared, installed
   ::Simple::Package::Priv::add-installed {Declared 1.0}
   puts [::Simple::Package::information state Declared]
   puts [::Simple::Package::information state Declared 1.0]
   puts [::Simple::Package::information state Declared 2]

} -cleanup {

   # Clean the packages
   package forget Undeclared
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   ::Simple::Package::Priv::delete-installed {Declared 1.0}

} -output {unknown
available
available
available
available
available
available
installed
declared
declared
available
installed
installed
available
}
 
test-case state-3 {
   ::Simple::Package::information state
   error, incorrect package version number format
} -setup {

   # Provide one package
   package provide Undeclared 1.0

} -script {
   ::Simple::Package::information state Undeclared incorrectVersion
} -cleanup {

   # Clean the package
   package forget Undeclared

} -error {expected version number but got "incorrectVersion"}

### ===========================================================================
### -test     : ::Simple::Package::require
test-case require-1 {
   ::Simple::Package::require
   undeclared package
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Undeclared package
   package ifneeded Undeclared 1 {
      package provide Undeclared 1
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require the package
   puts [::Simple::Package::require Undeclared 1.0]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -cleanup {

   # Clean the package
   package forget Undeclared

} -output {installed:
1
installed:{Undeclared 1}
}

test-case require-2 {
   ::Simple::Package::require
   declared package
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 0}
} -setup {

   # Declared package
   package ifneeded Declared 1.0 {
      ::Simple::Package::declare Declared 1.0 -install {
         puts {Installing Declared 1.0}
      }
   }

} -script {

   # No packages declared
   puts "declared:[::Simple::Package::information declared *eclared]"

   # Require the package
   puts [::Simple::Package::require Declared 1]

   # One package declared
   puts "declared:[::Simple::Package::information declared *eclared]"

} -cleanup {

   # Clean the package
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}

} -output {declared:
1.0
declared:{Declared 1.0}
}

test-case require-3 {
   ::Simple::Package::require
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Undeclared package
   package ifneeded Undeclared 1 {
      package provide Undeclared 1
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require the package
   puts [::Simple::Package::require Undeclared]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -cleanup {

   # Clean the package
   package forget Undeclared

} -output {installed:
1
installed:{Undeclared 1}
}

test-case require-4 {
   ::Simple::Package::require
   error, incorrect package version number format
} -script {
   ::Simple::Package::require Undeclared incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case require-5 {
   ::Simple::Package::require
   error, required package not found
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Failed require
   catch {
      ::Simple::Package::require SimpleNonExisting
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -output {installed:
can't find package SimpleNonExisting
installed:
}

### ===========================================================================
### -test     : ::Simple::Package::declare
test-case declare-1 {
   ::Simple::Package::declare
} -script {

   # No packages initially declared nor installed
   puts "declared:[::Simple::Package::information declared *eclared*]"
   puts "installed:[::Simple::Package::information installed *eclared*]"

   # Non-existing namespace
   puts [namespace children ::Simple Declared]

   # Undeclared package
   package ifneeded Undeclared 1.0 {
      puts {Requiring Undeclared 1.0}
      package provide Undeclared 1.0
   }

   # Declared package
   ::Simple::Package::declare Declared 1.0 -required {
      Undeclared
   } -namespaces {
      ::Simple::Declared
   } -install {puts {Installing Declared 1.0}}\
   -uninstall {puts {Uninstalling Declared 1.0}}

   # Declared package information
   puts [::Simple::Package::information namespaces Declared 1.0]
   puts [::Simple::Package::information required Declared 1]
   puts [::Simple::Package::information install Declared 1.0]
   puts [::Simple::Package::information uninstall Declared 1]

   # Package information
   puts "declared:[::Simple::Package::information declared *eclared*]"
   puts "installed:[::Simple::Package::information installed *eclared*]"

   # Ensure the namespace has been created
   puts [namespace children ::Simple Declared]

} -cleanup {

   # Clean the packages
   package forget Undeclared
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   namespace delete ::Simple::Declared
   ::Simple::Package::Priv::delete-namespaces-info {Declared 1.0}

} -output {declared:
installed:

Requiring Undeclared 1.0
::Simple::Declared
{Undeclared 1.0}
puts {Installing Declared 1.0}
puts {Uninstalling Declared 1.0}
declared:{Declared 1.0}
installed:{Undeclared 1.0}
::Simple::Declared
}

test-case declare-2 {
   ::Simple::Package::declare
   the greatest known version is used
} -setup {

   # Two available versions
   package ifneeded Undeclared 0.9 {
      puts {Requiring Undeclared 0.9}
      package provide Undeclared 0.9
   }
   package ifneeded Undeclared 0.10 {
      puts {Requiring Undeclared 0.10}
      package provide Undeclared 0.10
   }

} -script {

   # No packages initially declared nor installed
   puts "declared:[::Simple::Package::information declared *eclared*]"
   puts "installed:[::Simple::Package::information installed *eclared*]"

   # Declared package
   ::Simple::Package::declare Declared 1.0 -required {
      Undeclared
   } -namespaces {
      ::Simple::Declared
   } -install {puts {Installing Declared 1.0}}\
   -uninstall {puts {Uninstalling Declared 1.0}}

   # One package declared, another installed
   puts "declared:[::Simple::Package::information declared *eclared*]"
   puts "installed:[::Simple::Package::information installed *eclared*]"

} -cleanup {

   # Clean the packages
   package forget Undeclared 0.9
   package forget Undeclared 0.10
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   namespace delete ::Simple::Declared
   ::Simple::Package::Priv::delete-namespaces-info {Declared 1.0}

} -output {declared:
installed:
Requiring Undeclared 0.10
declared:{Declared 1.0}
installed:{Undeclared 0.10}
}

test-case declare-3 {
   ::Simple::Package::declare
   error, incorrect package version number format
} -script {

   # Failed declarations
   catch {
      ::Simple::Package::declare Foo incorrectVersion
   } result
   puts $result

   catch {
      ::Simple::Package::declare Foo {}
   } result
   puts $result

   # Ensure there is no namespaces information for the package
   puts [catch {::Simple::Package::Priv::delete-namespaces-info\
      {Foo incorrectVersion}}]

   # Ensure the namespace has not been created
   puts [namespace children ::Simple Declared]

   # Failed declarations
   ::Simple::Package::declare Foo 1.0 -required {{Bar incorrectVersion}}

} -output {expected version number but got "incorrectVersion"
expected version number but got ""
1

} -error {expected version number but got "incorrectVersion"}
 
test-case declare-4 {
   ::Simple::Package::declare
   error, package required package not found
} -script {

   # No packages initially declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared 1.0 -required SimpleNonExisting
   } result
   puts $result

   # No packages declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -output {declared:
can't find package SimpleNonExisting
declared:
}
 
test-case declare-5 {
   ::Simple::Package::declare
   error, namespace owned by another package
} -script {

   # Successful declaration
   ::Simple::Package::declare Declared1 1.0 -namespaces ::Simple::Declared

   # One package declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared2 1.0 -namespaces ::Simple::Declared
   } result
   puts $result

   # One package declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -output {declared:{Declared1 1.0}
namespace "::Simple::Declared" owned by package "Declared1 1.0"
declared:{Declared1 1.0}
}

test-case declare-6 {
   ::Simple::Package::declare
   error, package declared
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 1}
} -script {

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1
   } result
   puts $result

   # One package declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -cleanup {

   # Clean the package
   package forget Declared1
   ::Simple::Package::Priv::delete-declared {Declared1 1.0}

} -output {package already declared "Declared1 1"
declared:{Declared1 1.0}
}

test-case declare-7 {
   ::Simple::Package::declare
   error, unqualified exported procedure
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 0}
} -script {

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -export unqualified
   } result
   puts $result

   # No packages declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -output {expected fully-qualifed exported command name but got "unqualified"
declared:
}

test-case declare-8 {
   ::Simple::Package::declare
   error, exported procedure namespace not owned by the package
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 0}
} -script {

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -export ::foo::exported
   } result
   puts $result

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -export ::exported
   } result
   puts $result

   # No packages declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -output {namespace "::foo" not owned by package "Declared1 1"
namespace "::" not owned by package "Declared1 1"
declared:
}

test-case declare-9 {
   ::Simple::Package::declare
   error, incorrect alias format
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 0}
} -script {

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -aliases foo
   } result
   puts $result

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -aliases {foo bar}
   } result
   puts $result

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -aliases {::foo ::bar}
   } result
   puts $result

   # No packages declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -output {expected fully-qualified aliased command name but got ""
expected fully-qualified aliased command name but got ""
expected unqualifed alias name but got "::foo"
declared:
}

test-case declare-10 {
   ::Simple::Package::declare
   error, aliased procedure namespace not owned by the package
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 0}
} -script {

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -aliases {{foo ::foo::exported}}
   } result
   puts $result

   # Failed declaration
   catch {
      ::Simple::Package::declare Declared1 1 -aliases {{foo ::exported}}
   } result
   puts $result

   # No packages declared
   puts "declared:[::Simple::Package::information declared *eclared*]"

} -output {namespace "::foo" not owned by package "Declared1 1"
namespace "::" not owned by package "Declared1 1"
declared:
}

test-case declare-11 {
   ::Simple::Package::declare
   error, bad option
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 0}
} -script {
   ::Simple::Package::declare Declared1 1 -foo
} -error {bad option "-foo": must be -aliases, -export, -firsttimeinstall,\
-install, -namespaces, -required or -uninstall}

test-case declare-12 {
   ::Simple::Package::declare
   error, too many arguments
} -prerequisites {
   {[llength [::Simple::Package::information declared Declared1]] == 0}
} -script {
   ::Simple::Package::declare Declared1 1 foo
} -error {called "::Simple::Package::declare" with too many arguments}

### ===========================================================================
### -test     : ::Simple::Package::information namespaces
test-case namespaces-1 {
   ::Simple::Package::information namespaces
} -setup {

   # Declare some packages
   package ifneeded Foo0 1.0 {
      package provide Foo0 1.0
   }

   ::Simple::Package::declare Foo1 1.0

   ::Simple::Package::declare Foo2 1.0 -required {
      {Foo1 1.0} {Foo0 1.0}
   } -namespaces {
      ::Simple::Foo2
      ::Simple::Foo2::Priv
   } -aliases {
      {Foo2-Alias ::Simple::Foo2::Foo2-Public-1}
   } -export {
      {::Simple::Foo2::Foo2 Exported-1}
      ::Simple::Foo2::Foo2-Exported-2
   } -install Foo2InstallScript -uninstall Foo2UninstallScript

   ::Simple::Package::declare Foo3 1.0 -required {
      {Foo1 1.0} {Foo2 1.0}
   } -namespaces {
      ::Simple::Foo3-1
      ::Simple::Foo3-1::Priv
      ::Simple::Foo3-2
   } -aliases {
      {Foo3-1-Alias ::Simple::Foo3-1::Foo3-1-Public}
   } -export {
      ::Simple::Foo3-1::Foo3-1-Exported
      ::Simple::Foo3-2::Foo3-2-Exported
   } -install Foo3InstallScript -uninstall Foo3UninstallScript

   # Add some procedures to the packages
   namespace eval ::Simple::Foo2 {
      proc {Foo2 Exported-1} {} {}
      proc Foo2-Exported-2 {} {}
      proc Foo2-Public-1 {} {}
      proc Foo2-Public-2 {} {}
      proc Foo2%SimpleLibraryInternal {} {}
   }
   namespace eval ::Simple::Foo2::Priv {
      proc Foo2-Private {} {}
      proc Foo2%SimpleLibraryInternal {} {}
   }
   namespace eval ::Simple::Foo3-1 {
      proc Foo3-1-Exported {} {}
      proc Foo3-1-Public {} {}
   }
   namespace eval ::Simple::Foo3-1::Priv:: {
      proc Foo3-1-Private {} {}
   }
   namespace eval ::Simple::Foo3-2 {
      proc Foo3-2-Exported {} {}
      proc Foo3-2-Public {} {}
   }

} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information namespaces Foo1 1]]
   puts [lsort [::Simple::Package::information namespaces Foo2 1]]
   puts [lsort [::Simple::Package::information namespaces Foo3 1.0]]

} -output {
::Simple::Foo2 ::Simple::Foo2::Priv
::Simple::Foo3-1 ::Simple::Foo3-1::Priv ::Simple::Foo3-2
}

test-case namespaces-2 {
   ::Simple::Package::information namespaces
   package version not given
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information namespaces Foo1 1]]
   puts [lsort [::Simple::Package::information namespaces Foo2 1]]
   puts [lsort [::Simple::Package::information namespaces Foo3 1.0]]

} -output {
::Simple::Foo2 ::Simple::Foo2::Priv
::Simple::Foo3-1 ::Simple::Foo3-1::Priv ::Simple::Foo3-2
}

test-case namespaces-3 {
   ::Simple::Package::information namespaces
   error, incorrect package version number format
} -script {
   ::Simple::Package::information namespaces Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case namespaces-4 {
   ::Simple::Package::information namespaces
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information namespaces Undeclared
   } result
   puts $result
   ::Simple::Package::information namespaces Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information packagenamespace
test-case packagenamespace-1 {
   ::Simple::Package::information packagenamespace
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information packagenamespace ::Simple::Foo2]
   puts [::Simple::Package::information packagenamespace ::Simple::Foo3-1]

} -output {Foo2 1.0
Foo3 1.0
}

test-case packagenamespace-2 {
   ::Simple::Package::information packagenamespace
   error, non-existing namespace
} -script {
   ::Simple::Package::information packagenamespace ::Non-existing
} -error {non-existing namespace "::Non-existing"}

### ===========================================================================
### -test     : ::Simple::Package::information packagealias
test-case packagealias-1 {
   ::Simple::Package::information packagealias
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {
   puts [::Simple::Package::information packagealias Foo2-Alias]
   puts [::Simple::Package::information packagealias Foo3-1-Alias]
} -output {Foo2 1.0
Foo3 1.0
}

test-case packagealias-2 {
   ::Simple::Package::information packagealias
   error, non-existing alias
} -script {
   ::Simple::Package::information packagealias Non-existing
} -return {}

### ===========================================================================
### -test     : ::Simple::Package::information packagecommand
test-case packagecommand-1 {
   ::Simple::Package::information packagecommand
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {
   namespace eval ::Simple::Foo2 {

      # Foo2 1.0 procedures
      puts [::Simple::Package::information packagecommand {Foo2 Exported-1}]
      puts [::Simple::Package::information packagecommand Foo2-Public-1]
   }
   namespace eval ::Simple::Foo3-2 {

      # Foo3 1.0 procedures
      puts [::Simple::Package::information packagecommand Foo3-2-Exported]
      puts [::Simple::Package::information packagecommand Foo3-2-Public]

      # Import ::Simple::Foo2 from Foo2 1.0
      namespace import ::Simple::Foo2::*

      # Foo2 1.0 procedures
      puts [::Simple::Package::information packagecommand {Foo2 Exported-1}]
   }
} -output {Foo2 1.0
Foo2 1.0
Foo3 1.0
Foo3 1.0
Foo2 1.0
}

test-case packagecommand-2 {
   ::Simple::Package::information packagecommand
   error, non-existing procedure
} -script {
   puts [::Simple::Package::information packagecommand Non-existing]
} -error {invalid command name "Non-existing"}

### ===========================================================================
### -test     : ::Simple::Package::information required
test-case required-1 {
   ::Simple::Package::information required
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information required Foo1 1]]
   puts [lsort [::Simple::Package::information required Foo2 1]]
   puts [lsort [::Simple::Package::information required Foo3 1.0]]

} -output {
{Foo0 1.0} {Foo1 1.0}
{Foo1 1.0} {Foo2 1.0}
}

test-case required-2 {
   ::Simple::Package::information required
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information required Foo1 1]]
   puts [lsort [::Simple::Package::information required Foo2 1]]
   puts [lsort [::Simple::Package::information required Foo3 1.0]]

} -output {
{Foo0 1.0} {Foo1 1.0}
{Foo1 1.0} {Foo2 1.0}
}

test-case required-3 {
   ::Simple::Package::information required
   error, incorrect package version number format
} -script {
   ::Simple::Package::information required Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case required-4 {
   ::Simple::Package::information required
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information required Undeclared
   } result
   puts $result
   ::Simple::Package::information required Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information requiredall
test-case requiredall-1 {
   ::Simple::Package::information requiredall
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information requiredall Foo2 1]]
   puts [lsort [::Simple::Package::information requiredall Foo3 1.0]]

} -output {{Foo0 1.0} {Foo1 1.0}
{Foo0 1.0} {Foo1 1.0} {Foo2 1.0}
}

test-case requiredall-2 {
   ::Simple::Package::information requiredall
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information requiredall Foo2 1]]
   puts [lsort [::Simple::Package::information requiredall Foo3 1.0]]

} -output {{Foo0 1.0} {Foo1 1.0}
{Foo0 1.0} {Foo1 1.0} {Foo2 1.0}
}

test-case requiredall-3 {
   ::Simple::Package::information requiredall
   error, incorrect package version number format
} -script {
   ::Simple::Package::information requiredall Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case requiredall-4 {
   ::Simple::Package::information requiredall
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information requiredall Undeclared
   } result
   puts $result
   ::Simple::Package::information requiredall Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information requiredby
test-case requiredby-1 {
   ::Simple::Package::information requiredby
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information requiredby Foo0 1]]
   puts [lsort [::Simple::Package::information requiredby Foo1 1]]
   puts [lsort [::Simple::Package::information requiredby Foo2 1.0]]

} -output {{Foo2 1.0}
{Foo2 1.0} {Foo3 1.0}
{Foo3 1.0}
}

test-case requiredby-2 {
   ::Simple::Package::information requiredby
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information requiredby Foo0 1]]
   puts [lsort [::Simple::Package::information requiredby Foo1 1.0]]
   puts [lsort [::Simple::Package::information requiredby Foo2 1.0]]

} -output {{Foo2 1.0}
{Foo2 1.0} {Foo3 1.0}
{Foo3 1.0}
}

test-case requiredby-3 {
   ::Simple::Package::information requiredby
   error, incorrect package version number format
} -script {
   ::Simple::Package::information requiredby Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

### ===========================================================================
### -test     : ::Simple::Package::information install
test-case installscript-1 {
   ::Simple::Package::information install
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information install Foo1 1.0]
   puts [::Simple::Package::information install Foo2 1]
   puts [::Simple::Package::information install Foo3 1.0]

} -output {
Foo2InstallScript
Foo3InstallScript
}

test-case installscript-2 {
   ::Simple::Package::information install
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information install Foo1]
   puts [::Simple::Package::information install Foo2]
   puts [::Simple::Package::information install Foo3]

} -output {
Foo2InstallScript
Foo3InstallScript
}

test-case installscript-3 {
   ::Simple::Package::information install
   error, incorrect package version number format
} -script {
   ::Simple::Package::information install Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case installscript-4 {
   ::Simple::Package::information install
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information install Undeclared
   } result
   puts $result
   ::Simple::Package::information install Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information uninstall
test-case uninstallscript-1 {
   ::Simple::Package::information uninstall
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information uninstall Foo1 1]
   puts [::Simple::Package::information uninstall Foo2 1.0]
   puts [::Simple::Package::information uninstall Foo3 1]

} -output {
Foo2UninstallScript
Foo3UninstallScript
}

test-case uninstallscript-2 {
   ::Simple::Package::information uninstall
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information uninstall Foo1]
   puts [::Simple::Package::information uninstall Foo2]
   puts [::Simple::Package::information uninstall Foo3]

} -output {
Foo2UninstallScript
Foo3UninstallScript
}

test-case uninstallscript-3 {
   ::Simple::Package::information uninstall
   error, incorrect package version number format
} -script {
   ::Simple::Package::information uninstall Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case uninstallscript-4 {
   ::Simple::Package::information uninstall
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information uninstall Undeclared
   } result
   puts $result
   ::Simple::Package::information uninstall Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information aliases
test-case aliases-1 {
   ::Simple::Package::information aliases
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information aliases Foo1 1]
   puts [::Simple::Package::information aliases Foo2 1.0]
   puts [::Simple::Package::information aliases Foo3 1]

} -output {
{Foo2-Alias ::Simple::Foo2::Foo2-Public-1}
{Foo3-1-Alias ::Simple::Foo3-1::Foo3-1-Public}
}

test-case aliases-2 {
   ::Simple::Package::information aliases
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information aliases Foo1]
   puts [::Simple::Package::information aliases Foo2]
   puts [::Simple::Package::information aliases Foo3]

} -output {
{Foo2-Alias ::Simple::Foo2::Foo2-Public-1}
{Foo3-1-Alias ::Simple::Foo3-1::Foo3-1-Public}
}

test-case aliases-3 {
   ::Simple::Package::information aliases
   error, incorrect package version number format
} -script {
   ::Simple::Package::information aliases Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case aliases-4 {
   ::Simple::Package::information aliases
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information aliases Undeclared
   } result
   puts $result
   ::Simple::Package::information aliases Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information export
test-case export-1 {
   ::Simple::Package::information export
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information export Foo1 1]
   puts [::Simple::Package::information export Foo2 1.0]
   puts [::Simple::Package::information export Foo3 1]

} -output {
{::Simple::Foo2::Foo2 Exported-1} ::Simple::Foo2::Foo2-Exported-2
::Simple::Foo3-1::Foo3-1-Exported ::Simple::Foo3-2::Foo3-2-Exported
}

test-case export-2 {
   ::Simple::Package::information export
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information export Foo1]
   puts [::Simple::Package::information export Foo2]
   puts [::Simple::Package::information export Foo3]

} -output {
{::Simple::Foo2::Foo2 Exported-1} ::Simple::Foo2::Foo2-Exported-2
::Simple::Foo3-1::Foo3-1-Exported ::Simple::Foo3-2::Foo3-2-Exported
}

test-case export-3 {
   ::Simple::Package::information export
   error, incorrect package version number format
} -script {
   ::Simple::Package::information export Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case export-4 {
   ::Simple::Package::information export
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information export Undeclared
   } result
   puts $result
   ::Simple::Package::information export Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information exportedcommands
test-case exportedcommands-1 {
   ::Simple::Package::information exportedcommands
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -setup {

   # Export some other commands apart from
   # those specified in the package declarations
   namespace eval ::Simple::Foo2 {
      namespace export Foo2-LateExported
   }
   namespace eval ::Simple::Foo3-1 {
      namespace export Foo3-1-LateExported
   }

} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information exportedcommands Foo1 1]]
   puts [lsort [::Simple::Package::information exportedcommands Foo2 1]]
   puts [lsort [::Simple::Package::information exportedcommands Foo3 1.0]]

} -output {
{Foo2 Exported-1} Foo2-Exported-2 Foo2-LateExported
Foo3-1-Exported Foo3-1-LateExported Foo3-2-Exported
}

test-case exportedcommands-2 {
   ::Simple::Package::information exportedcommands
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information exportedcommands Foo1 1.0]]
   puts [lsort [::Simple::Package::information exportedcommands Foo2 1.0]]
   puts [lsort [::Simple::Package::information exportedcommands Foo3 1]]

} -output {
{Foo2 Exported-1} Foo2-Exported-2 Foo2-LateExported
Foo3-1-Exported Foo3-1-LateExported Foo3-2-Exported
}

test-case exportedcommands-3 {
   ::Simple::Package::information exportedcommands
   error, incorrect package version number format
} -script {
   ::Simple::Package::information exportedcommands Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case exportedcommands-4 {
   ::Simple::Package::information exportedcommands
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information exportedcommands Undeclared
   } result
   puts $result
   ::Simple::Package::information exportedcommands Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information privatecommands
test-case privatecommands-1 {
   ::Simple::Package::information privatecommands
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information privatecommands Foo1 1.0]
   puts [::Simple::Package::information privatecommands Foo2 1]
   puts [::Simple::Package::information privatecommands Foo3 1.0]

} -output {
::Simple::Foo2::Priv::Foo2-Private
::Simple::Foo3-1::Priv::Foo3-1-Private
}

test-case privatecommands-2 {
   ::Simple::Package::information privatecommands
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [::Simple::Package::information privatecommands Foo1 1]
   puts [::Simple::Package::information privatecommands Foo2 1.0]
   puts [::Simple::Package::information privatecommands Foo3 1]

} -output {
::Simple::Foo2::Priv::Foo2-Private
::Simple::Foo3-1::Priv::Foo3-1-Private
}

test-case privatecommands-3 {
   ::Simple::Package::information privatecommands
   error, incorrect package version number format
} -script {
   ::Simple::Package::information privatecommands Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case privatecommands-4 {
   ::Simple::Package::information privatecommands
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information privatecommands Undeclared
   } result
   puts $result
   ::Simple::Package::information privatecommands Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information publiccommands
test-case publiccommands-1 {
   ::Simple::Package::information publiccommands
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information publiccommands Foo1 1.0]]
   puts [lsort [::Simple::Package::information publiccommands Foo2 1]]
   puts [lsort [::Simple::Package::information publiccommands Foo3 1]]

} -output {
::Simple::Foo2::Foo2-Public-1 ::Simple::Foo2::Foo2-Public-2
::Simple::Foo3-1::Foo3-1-Public ::Simple::Foo3-2::Foo3-2-Public
}

test-case publiccommands-2 {
   ::Simple::Package::information publiccommands
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information publiccommands Foo1 1.0]]
   puts [lsort [::Simple::Package::information publiccommands Foo2 1]]
   puts [lsort [::Simple::Package::information publiccommands Foo3 1.0]]

} -output {
::Simple::Foo2::Foo2-Public-1 ::Simple::Foo2::Foo2-Public-2
::Simple::Foo3-1::Foo3-1-Public ::Simple::Foo3-2::Foo3-2-Public
}

test-case publiccommands-3 {
   ::Simple::Package::information publiccommands
   error, incorrect package version number format
} -script {
   ::Simple::Package::information publiccommands Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case publiccommands-4 {
   ::Simple::Package::information publiccommands
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information publiccommands Undeclared
   } result
   puts $result
   ::Simple::Package::information publiccommands Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information commands
test-case commands-1 {
   ::Simple::Package::information commands
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts\
      [lsort [::Simple::Package::information commands Foo1 1.0]]
   puts\
      [lsort [::Simple::Package::information commands Foo2 1]]
   puts\
      [lsort [::Simple::Package::information commands Foo3 1]]

} -output {
{::Simple::Foo2::Foo2 Exported-1} ::Simple::Foo2::Foo2-Exported-2\
::Simple::Foo2::Foo2-Public-1 ::Simple::Foo2::Foo2-Public-2
::Simple::Foo3-1::Foo3-1-Exported ::Simple::Foo3-1::Foo3-1-Public\
::Simple::Foo3-2::Foo3-2-Exported ::Simple::Foo3-2::Foo3-2-Public
}

test-case commands-2 {
   ::Simple::Package::information commands
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts\
      [lsort [::Simple::Package::information commands Foo1 1.0]]
   puts\
      [lsort [::Simple::Package::information commands Foo2 1]]
   puts\
      [lsort [::Simple::Package::information commands Foo3 1.0]]

} -output {
{::Simple::Foo2::Foo2 Exported-1} ::Simple::Foo2::Foo2-Exported-2\
::Simple::Foo2::Foo2-Public-1 ::Simple::Foo2::Foo2-Public-2
::Simple::Foo3-1::Foo3-1-Exported ::Simple::Foo3-1::Foo3-1-Public\
::Simple::Foo3-2::Foo3-2-Exported ::Simple::Foo3-2::Foo3-2-Public
}

test-case commands-3 {
   ::Simple::Package::information commands
   error, incorrect package version number format
} -script {
   ::Simple::Package::information commands Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case commands-4 {
   ::Simple::Package::information commands
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information commands Undeclared
   } result
   puts $result
   ::Simple::Package::information commands Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information exportedcommandsall
test-case exportedall-1 {
   ::Simple::Package::information exportedcommandsall
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information exportedcommandsall Foo1 1]]
   puts [lsort [::Simple::Package::information exportedcommandsall Foo2 1]]
   puts [lsort [::Simple::Package::information exportedcommandsall Foo3 1.0]]

} -output {
{Foo2 Exported-1} Foo2-Exported-2 Foo2-LateExported
{Foo2 Exported-1} Foo2-Exported-2 Foo2-LateExported Foo3-1-Exported\
Foo3-1-LateExported Foo3-2-Exported
}

test-case exportedall-2 {
   ::Simple::Package::information exportedcommandsall
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information exportedcommandsall Foo1 1]]
   puts [lsort [::Simple::Package::information exportedcommandsall Foo2 1.0]]
   puts [lsort [::Simple::Package::information exportedcommandsall Foo3 1]]

} -output {
{Foo2 Exported-1} Foo2-Exported-2 Foo2-LateExported
{Foo2 Exported-1} Foo2-Exported-2 Foo2-LateExported Foo3-1-Exported\
Foo3-1-LateExported Foo3-2-Exported
}

test-case exportedall-3 {
   ::Simple::Package::information exportedcommandsall
   error, incorrect package version number format
} -script {
   ::Simple::Package::information exportedcommandsall Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case exportedall-4 {
   ::Simple::Package::information exportedcommandsall
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information exportedcommandsall Undeclared
   } result
   puts $result
   ::Simple::Package::information exportedcommandsall Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information aliasesall
test-case aliasesall-1 {
   ::Simple::Package::information aliasesall
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information aliasesall Foo1 1]]
   puts [lsort [::Simple::Package::information aliasesall Foo2 1]]
   puts [lsort [::Simple::Package::information aliasesall Foo3 1.0]]

} -output {
{Foo2-Alias ::Simple::Foo2::Foo2-Public-1}
{Foo2-Alias ::Simple::Foo2::Foo2-Public-1}\
{Foo3-1-Alias ::Simple::Foo3-1::Foo3-1-Public}
}

test-case aliasesall-2 {
   ::Simple::Package::information aliasesall
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared package information
   puts [lsort [::Simple::Package::information aliasesall Foo1 1]]
   puts [lsort [::Simple::Package::information aliasesall Foo2 1.0]]
   puts [lsort [::Simple::Package::information aliasesall Foo3 1]]

} -output {
{Foo2-Alias ::Simple::Foo2::Foo2-Public-1}
{Foo2-Alias ::Simple::Foo2::Foo2-Public-1}\
{Foo3-1-Alias ::Simple::Foo3-1::Foo3-1-Public}
}

test-case aliasesall-3 {
   ::Simple::Package::information aliasesall
   error, incorrect package version number format
} -script {
   ::Simple::Package::information aliasesall Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case aliasesall-4 {
   ::Simple::Package::information aliasesall
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information aliasesall Undeclared
   } result
   puts $result
   ::Simple::Package::information aliasesall Foo1 2.0
} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2.0"}

### ===========================================================================
### -test     : ::Simple::Package::information collisions
test-case collisions-1 {
   ::Simple::Package::information collisions
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -setup {

   # Create some procedures
   proc non-colliding1 {} {}
   proc non-colliding2 {} {}
   proc {Foo2 Exported-1} {} {}
   proc Foo3-1-Exported {} {}
   proc Foo2-Alias {} {}

   # Create a namespace with the same procedures
   namespace eval ::foo {
      proc non-colliding1 {} {}
      proc non-colliding2 {} {}
      proc {Foo2 Exported-1} {} {}
      proc Foo3-1-Exported {} {}
      proc Foo2-Alias {} {}
   }

} -script {

   # Declared packages collisions into the global namespace
   puts [lsort [::Simple::Package::information collisions Foo1]]
   puts [lsort [::Simple::Package::information collisions Foo2]]

   # Importing Foo3 would result in collisions from
   # a Foo2 procedure and a Foo2 alias
   puts [lsort [::Simple::Package::information collisions Foo3]]

   # Install Foo2
   ::Simple::Package::Priv::add-installed {Foo2 1.0}

   # Importing Foo3 would result in collisions
   # from its own procedures only
   puts [lsort [::Simple::Package::information collisions Foo3]]

   # Declared packages collisions into ::foo
   puts [lsort [::Simple::Package::information collisions Foo1 ::foo]]
   puts [lsort [::Simple::Package::information collisions Foo2 ::foo]]
   puts [lsort [::Simple::Package::information collisions Foo3 ::foo]]

} -cleanup {

   # Uninstall Foo2
   ::Simple::Package::Priv::delete-installed {Foo2 1.0}

} -output {
{::Simple::Foo2::Foo2 Exported-1} Foo2-Alias
{::Simple::Foo2::Foo2 Exported-1} ::Simple::Foo3-1::Foo3-1-Exported Foo2-Alias
::Simple::Foo3-1::Foo3-1-Exported

{::Simple::Foo2::Foo2 Exported-1} Foo2-Alias
::Simple::Foo3-1::Foo3-1-Exported
}

test-case collisions-2 {
   ::Simple::Package::information collisions
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Declared packages collisions into the global namespace
   puts [lsort [::Simple::Package::information collisions Foo1]]
   puts [lsort [::Simple::Package::information collisions Foo2]]

   # Importing Foo3 would result in a
   # collision from a Foo2 procedure 
   puts [lsort [::Simple::Package::information collisions Foo3]]

   # Install Foo2
   ::Simple::Package::Priv::add-installed {Foo2 1.0}

   # Importing Foo3 would result in collisions
   # from its own procedures only
   puts [lsort [::Simple::Package::information collisions Foo3]]

} -output {
{::Simple::Foo2::Foo2 Exported-1} Foo2-Alias
{::Simple::Foo2::Foo2 Exported-1} ::Simple::Foo3-1::Foo3-1-Exported Foo2-Alias
::Simple::Foo3-1::Foo3-1-Exported
}

test-case collisions-3 {
   ::Simple::Package::information collisions
   error, non-existing namespace
} -script {
   ::Simple::Package::information collisions Foo3 ::Non-existing
} -error {non-existing namespace "::Non-existing"}

test-case collisions-4 {
   ::Simple::Package::information collisions
   error, undeclared package
} -script {
   ::Simple::Package::information collisions Undeclared
} -error {undeclared package "Undeclared"}

### ===========================================================================
### -test     : ::Simple::Package::information importedinto
test-case importedinto-1 {
   ::Simple::Package::information importedinto
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Import Foo1 into :: and ::foo
   ::Simple::Package::Priv::add-importedinto {Foo1 1.0} ::
   ::Simple::Package::Priv::add-importedinto {Foo1 1.0} ::foo

   # Uninstalled package
   puts [::Simple::Package::information importedinto Foo1 1.0]

   # Install Foo1
   ::Simple::Package::Priv::add-installed {Foo1 1.0}

   # Installed package
   puts [lsort [::Simple::Package::information importedinto Foo1 1]]

} -output {
:: ::foo
}

test-case importedinto-2 {
   ::Simple::Package::information importedinto
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 3}
} -script {

   # Installed package
   puts [lsort [::Simple::Package::information importedinto Foo1]]

   # Uninstall Foo1
   ::Simple::Package::Priv::delete-installed {Foo1 1.0}

   # Uninstalled package
   puts [::Simple::Package::information importedinto Foo1 1.0]

} -cleanup {

   # Delete Foo1 import information
   ::Simple::Package::Priv::delete-importedinto {Foo1 1.0}

} -output {:: ::foo

}

test-case importedinto-3 {
   ::Simple::Package::information importedinto
   error, incorrect package version number format
} -script {
   ::Simple::Package::information importedinto Foo1 incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case importedinto-4 {
   ::Simple::Package::information importedinto
   error, undeclared package
} -script {
   catch {
      ::Simple::Package::information importedinto Undeclared
   } result
   puts $result
   ::Simple::Package::information importedinto Foo1 2
} -cleanup {

   # Delete the procedures
   rename non-colliding1 {}
   rename non-colliding2 {}
   rename {Foo2 Exported-1} {}
   rename Foo3-1-Exported {}

   # Delete the namespace
   namespace delete ::foo

   # Clean the packages
   package forget Foo0
   package forget Foo1
   ::Simple::Package::Priv::delete-declared {Foo1 1.0}
   ::Simple::Package::Priv::delete-namespaces-info {Foo1 1.0}
   package forget Foo2
   ::Simple::Package::Priv::delete-declared {Foo2 1.0}
   ::Simple::Package::Priv::delete-installed {Foo2 1.0}
   namespace delete ::Simple::Foo2
   ::Simple::Package::Priv::delete-namespaces-info {Foo2 1.0}
   package forget Foo3
   ::Simple::Package::Priv::delete-declared {Foo3 1.0}
   namespace delete ::Simple::Foo3-1
   namespace delete ::Simple::Foo3-2
   ::Simple::Package::Priv::delete-namespaces-info {Foo3 1.0}

} -output {undeclared package "Undeclared"
} -error {undeclared package "Foo1 2"}

### ===========================================================================
### -test     : ::Simple::Package::install
test-case install-1 {
   ::Simple::Package::install
} -prerequisites {
   {[llength [namespace children ::Simple Foo*]] == 0}
} -setup {

   # First declared package
   ::Simple::Package::declare Foo1 1.0 -namespaces {
      ::Simple::Foo1
   } -export {
      ::Simple::Foo1::Foo1-Exported
   } -install {puts {Installing Foo1 1.0}} -firsttimeinstall {
      puts {   for the first time!}
      proc ::Simple::Foo1::Foo1-Exported {} {puts [info level 0]}
   }

   # Second declared package
   ::Simple::Package::declare Foo2 1 -required {
      {Foo1 1}
   } -namespaces {
      ::Simple::Foo2
      ::Simple::Foo2::Priv
   } -export {
      {::Simple::Foo2::Foo2 Exported}
      ::Simple::Foo2::Priv::Foo2-Priv-Exported
   } -install {puts {Installing Foo2 1}} -firsttimeinstall {
      puts {   for the first time!}
      namespace eval ::Simple::Foo2 {
         proc {::Simple::Foo2::Foo2 Exported} {} {puts [info level 0]}
         proc ::Simple::Foo2::Priv::Foo2-Priv-Exported {} {puts [info level 0]}
      }
   }

} -script {

   # No packages intalled
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Install the second package
   ::Simple::Package::install Foo2

   # Two packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Imported procedures
   Foo1-Exported
   {Foo2 Exported}
   Foo2-Priv-Exported

} -cleanup {

   # Clean the second package
   package forget Foo2
   ::Simple::Package::Priv::delete-declared {Foo2 1}
   ::Simple::Package::Priv::delete-installed {Foo2 1}
   namespace delete ::Simple::Foo2
   ::Simple::Package::Priv::delete-namespaces-info {Foo2 1}
   ::Simple::Package::Priv::delete-importedinto {Foo2 1}

} -output {installed:
Installing Foo1 1.0
   for the first time!
Installing Foo2 1
   for the first time!
installed:{Foo1 1.0} {Foo2 1}
Foo1-Exported
{Foo2 Exported}
Foo2-Priv-Exported
}

test-case install-2 {
   ::Simple::Package::install
   error, package not declared
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo1]] == 1}
} -script {

   # One package installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed installations
   catch {
      ::Simple::Package::install Foo
   } result
   puts $result

   # One package installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -output {installed:{Foo1 1.0}
undeclared package "Foo"
installed:{Foo1 1.0}
}

test-case install-3 {
   ::Simple::Package::install
   error, package installed
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo1]] == 1}
} -script {

   # Failed installation
   catch {
      ::Simple::Package::install Foo1
   } result
   puts $result

   # One package installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the first package
   package forget Foo1
   ::Simple::Package::Priv::delete-declared {Foo1 1.0}
   ::Simple::Package::Priv::delete-installed {Foo1 1.0}
   namespace delete ::Simple::Foo1
   ::Simple::Package::Priv::delete-namespaces-info {Foo1 1.0}
   ::Simple::Package::Priv::delete-importedinto {Foo1 1.0}

} -output {installed package "Foo1 1.0"
installed:{Foo1 1.0}
}

test-case install-4 {
   ::Simple::Package::install
   error, error evaluating the install script
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Third declared package
   ::Simple::Package::declare Foo3 1.0 -namespaces {
      ::Simple::Foo3
   } -export {
      ::Simple::Foo3::Foo3-Exported
   } -install {
      puts {Installing Foo3 1.0}
      foo bar
   } -firsttimeinstall {
      puts {   for the first time!}
   }

} -script {

   # Failed installation
   catch {
      ::Simple::Package::install Foo3
   } result
   puts $result

   # Ensure the package has not been imported into the global namespace
   catch {
      ::Foo3-Exported
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the third package
   package forget Foo3
   namespace delete ::Simple::Foo3
   ::Simple::Package::Priv::delete-declared {Foo3 1.0}

} -output {Installing Foo3 1.0
error evaluating package "Foo3 1.0" install script: invalid command name "foo"
invalid command name "::Foo3-Exported"
installed:
}

test-case install-5 {
   ::Simple::Package::install
   error, error evaluating the first time install script
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Third declared package
   ::Simple::Package::declare Foo3 1.0 -namespaces {
      ::Simple::Foo3
   } -export {
      ::Simple::Foo3::Foo3-Exported
   } -install {
      puts {Installing Foo3 1.0}
   } -firsttimeinstall {
      puts {   for the first time!}
      foo bar
   }

} -script {

   # Failed installation
   catch {
      ::Simple::Package::install Foo3
   } result
   puts $result

   # Ensure the package has not been imported into the global namespace
   catch {
      ::Foo3-Exported
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the third package
   package forget Foo3
   namespace delete ::Simple::Foo3
   ::Simple::Package::Priv::delete-declared {Foo3 1.0}

} -output {Installing Foo3 1.0
   for the first time!
error evaluating package "Foo3 1.0" first time install script:\
invalid command name "foo"
invalid command name "::Foo3-Exported"
installed:
}

test-case install-6 {
   ::Simple::Package::install
   error, exported commands collision with target namespace
} -setup {

   # First declared package
   ::Simple::Package::declare Foo1 1.0 -namespaces {
      ::Simple::Foo1
   } -export {
      ::Simple::Foo1::Exported
   } -install {puts {Installing Foo1 1.0}} -firsttimeinstall {
      proc ::Simple::Foo1::Exported {} {puts [info level 0]}
   }

   # Second declared package
   ::Simple::Package::declare Foo2 1 -required {
      {Foo1 1.0}
   } -namespaces {
      ::Simple::Foo2
   } -aliases {
      {Alias ::Simple::Foo2::Alias}
   } -export {
      ::Simple::Foo2::Exported
   } -install {puts {Installing Foo2 1.0}} -firsttimeinstall {
      proc ::Simple::Foo2::Exported {} {puts [info level 0]}
   }

   # Colliding procedure
   proc Exported {} {}

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed installation
   catch {
      ::Simple::Package::install Foo1
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Remove the colliding procedure
   rename Exported {}

} -output {installed:
couldn't install package "Foo1 1.0": command (or alias) "Exported" in the\
global namespace collides with exported command "::Simple::Foo1::Exported"\
(package "Foo1 1.0")
installed:
}

test-case install-7 {
   ::Simple::Package::install
   error, exported commands collision
} -prerequisites {
   {[llength [namespace children ::Simple Foo*]] == 2}
} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed installation
   catch {
      ::Simple::Package::install Foo2
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the second package
   package forget Foo2
   ::Simple::Package::Priv::delete-declared {Foo2 1}
   namespace delete ::Simple::Foo2
   ::Simple::Package::Priv::delete-namespaces-info {Foo2 1}

} -output {installed:
couldn't install package "Foo2 1": command (or alias) "Exported" is exported\
(or created) by both package "Foo1 1.0" (namespace "::Simple::Foo1") and\
"Foo2 1" (namespace "::Simple::Foo2")
installed:
}

test-case install-8 {
   ::Simple::Package::install
   error, aliases collision with target namespace
} -prerequisites {
   {[llength [namespace children ::Simple Foo*]] == 1}
} -setup {

   # Third declared package
   ::Simple::Package::declare Foo3 1.0 -namespaces {
      ::Simple::Foo3
   } -aliases {
      {Alias ::Simple::Foo3::Foo3-Exported}
   }

   # Colliding procedure
   proc Alias {} {}

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed installation
   catch {
      ::Simple::Package::install Foo3
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the third package
   package forget Foo3
   namespace delete ::Simple::Foo3
   ::Simple::Package::Priv::delete-declared {Foo3 1.0}

   # Remove the colliding procedure
   rename Alias {}

} -output {installed:
couldn't install package "Foo3 1.0": command (or alias) "Alias" in the global\
namespace collides with alias "Alias" (package "Foo3 1.0")
installed:
}

test-case install-9 {
   ::Simple::Package::install
   error, aliases collision
} -prerequisites {
   {[llength [namespace children ::Simple Foo*]] == 1}
} -setup {

   # Third declared package
   ::Simple::Package::declare Foo3 1.0 -namespaces {
      ::Simple::Foo3
   } -aliases {
      {Alias ::Simple::Foo3::Alias}
   }

   # Fourth declared package
   ::Simple::Package::declare Foo4 1.0 -required {
      Foo3
   } -namespaces {
      ::Simple::Foo4
   } -aliases {
      {Alias ::Simple::Foo4::Alias}
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed installation
   catch {
      ::Simple::Package::install Foo4
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the fourth package
   package forget Foo4
   namespace delete ::Simple::Foo4
   ::Simple::Package::Priv::delete-declared {Foo4 1.0}

} -output {installed:
couldn't install package "Foo4 1.0": command (or alias) "Alias" is exported (or\
created) by both package "Foo3 1.0" and "Foo4 1.0"
installed:
}

test-case install-10 {
   ::Simple::Package::install
   error, alias and exported command collision
} -prerequisites {
   {[llength [namespace children ::Simple Foo*]] == 2}
} -setup {

   # Fourth declared package
   ::Simple::Package::declare Foo4 1.0 -required {
      Foo3
   } -namespaces {
      ::Simple::Foo4
   } -export {
      ::Simple::Foo4::Alias
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed installation
   catch {
      ::Simple::Package::install Foo4
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the third package
   package forget Foo3
   namespace delete ::Simple::Foo3
   ::Simple::Package::Priv::delete-declared {Foo3 1.0}

   # Clean the fourth package
   package forget Foo4
   namespace delete ::Simple::Foo4
   ::Simple::Package::Priv::delete-declared {Foo4 1.0}

} -output {installed:
couldn't install package "Foo4 1.0": command (or alias) "Alias" is exported (or\
created) by both package "Foo3 1.0" and "Foo4 1.0" (namespace "::Simple::Foo4")
installed:
}

### ===========================================================================
### -test     : ::Simple::Package::require-and-install
test-case requireandinstall-1 {
   ::Simple::Package::require-and-install
   undeclared package
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Undeclared package
   package ifneeded Undeclared 1 {
      package provide Undeclared 1
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require and install the package
   puts [::Simple::Package::require-and-install Undeclared 1.0]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -cleanup {

   # Clean the package
   package forget Undeclared

} -output {installed:
1
installed:{Undeclared 1}
}

test-case requireandinstall-2 {
   ::Simple::Package::require-and-install
   declared package
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Declared package
   package ifneeded Declared 1.0 {
      ::Simple::Package::declare Declared 1.0\
         -install {puts {Installing Declared 1.0}}\
         -firsttimeinstall {puts {   for the first time!}}
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require and install the package
   puts [::Simple::Package::require-and-install Declared 1]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -cleanup {

   # Clean the package
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   ::Simple::Package::Priv::delete-installed {Declared 1.0}

} -output {installed:
Installing Declared 1.0
   for the first time!
1.0
installed:{Declared 1.0}
}

test-case requireandinstall-3 {
   ::Simple::Package::require-and-install
   undeclared and installed package
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Undeclared package
   package ifneeded Undeclared 1 {
      package provide Undeclared 1
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require and install the package
   puts [::Simple::Package::require-and-install Undeclared 1]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require and install the package again
   puts [::Simple::Package::require-and-install Undeclared 1]

} -cleanup {

   # Clean the package
   package forget Undeclared

} -output {installed:
1
installed:{Undeclared 1}
1
}

test-case requireandinstall-4 {
   ::Simple::Package::require-and-install
   declared and installed package
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Declared package
   package ifneeded Declared 1.0 {
      ::Simple::Package::declare Declared 1.0\
         -install {puts {Installing Declared 1.0}}\
         -firsttimeinstall {puts {   for the first time!}}
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require and install the package
   puts [::Simple::Package::require-and-install Declared 1]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -cleanup {

   # Clean the package
   package forget Declared
   ::Simple::Package::Priv::delete-declared {Declared 1.0}
   ::Simple::Package::Priv::delete-installed {Declared 1.0}

} -output {installed:
Installing Declared 1.0
   for the first time!
1.0
installed:{Declared 1.0}
}

test-case requireandinstall-5 {
   ::Simple::Package::require-and-install
   package version not given
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Undeclared package
   package ifneeded Undeclared 1 {
      package provide Undeclared 1
   }

} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Require and install the package
   puts [::Simple::Package::require-and-install Undeclared]

   # One package installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -cleanup {

   # Clean the package
   package forget Undeclared

} -output {installed:
1
installed:{Undeclared 1}
}

test-case requireandinstall-6 {
   ::Simple::Package::require-and-install
   error, incorrect package version number format
} -script {
   ::Simple::Package::require-and-install Undeclared incorrectVersion
} -error {expected version number but got "incorrectVersion"}

test-case requireandinstall-7 {
   ::Simple::Package::require-and-install
   error, required package not found
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -script {

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

   # Failed require
   catch {
      ::Simple::Package::require-and-install SimpleNonExisting
   } result
   puts $result

   # No packages installed
   puts "installed:[::Simple::Package::information installed *eclared]"

} -output {installed:
can't find package SimpleNonExisting
installed:
}

### ===========================================================================
### -test     : ::Simple::Package::import
test-case import-1 {
   ::Simple::Package::import
} -prerequisites {
   {[llength [namespace children ::Simple Foo*]] == 1}
} -setup {

   # Second declared package
   ::Simple::Package::declare Foo2 1 -required {
      {Foo1 1.0}
   } -namespaces {
      ::Simple::Foo2
   } -export {
      {::Simple::Foo2::Foo2 Exported}
   } -aliases {
      {Alias ::Simple::Foo2::Foo2-Alias}
   } -install {puts {Installing Foo2 1}} -firsttimeinstall {
      proc {::Simple::Foo2::Foo2 Exported} {} {puts [info level 0]}
   } -uninstall {puts {Uninstalling Foo2 1}}

   # Install the second package
   ::Simple::Package::install Foo2

   # Create a namespace
   namespace eval ::foo {}

} -script {

   # Import the second package into ::foo
   ::Simple::Package::import Foo2 ::foo

   # Imported procedures
   {Foo2 Exported}
   {::foo::Foo2 Exported}
   Exported

   # Ensure the first package has not been imported into ::foo
   catch {
      ::foo::Exported
   } result
   puts $result

   # Import information
   puts "Foo1 imported into:\
      \{[::Simple::Package::information importedinto Foo1]\}"
   puts "Foo2 imported into:\
      \{[::Simple::Package::information importedinto Foo2]\}"

} -cleanup {

   # Remove the procedures imported into the global namespace
   rename {Foo2 Exported} {}
   rename Exported {}

   # Remove the aliases
   rename Alias {}

} -output {{Foo2 Exported}
{::foo::Foo2 Exported}
Exported
invalid command name "::foo::Exported"
Foo1 imported into: {::}
Foo2 imported into: {::foo ::}
}

test-case import-2 {
   ::Simple::Package::import
   error, package not declared
} -script {
   ::Simple::Package::import Bar ::foo
} -error {undeclared package "Bar"}

test-case import-3 {
   ::Simple::Package::import
   error, package not installed
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 2}
} -setup {

   # Uninstall the first package
   ::Simple::Package::Priv::delete-installed {Foo1 1.0}

} -script {
   ::Simple::Package::import Foo1 ::foo
} -error {uninstalled package "Foo1 1.0"}

test-case import-4 {
   ::Simple::Package::import
   error, package imported
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 1}
} -script {
   ::Simple::Package::import Foo2 ::foo
} -error {package "Foo2 1" already imported into namespace "::foo"}

test-case import-5 {
   ::Simple::Package::import
   error, non-existing namespace
} -script {
   ::Simple::Package::import Foo2 ::Non-existing
} -error {non-existing namespace "::Non-existing"}

test-case import-6 {
   ::Simple::Package::import
   error, can't import into the global namespace
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 1}
} -script {
   ::Simple::Package::import Foo2 ::
} -error {package "Foo2 1" already imported into namespace "::"}

test-case import-7 {
   ::Simple::Package::import
   error, exported commands collision with target namespace
} -setup {

   # Colliding procedure
   proc {::foo::Foo2 Exported} {} {}

   # Delete Foo2 import information
   ::Simple::Package::Priv::delete-importedinto {Foo2 1}

} -script {

   # Import information
   puts "Foo2 imported into:\
      \{[::Simple::Package::information importedinto Foo2]\}"

   # Failed import
   catch {
      ::Simple::Package::import Foo2 ::foo
   } result
   puts $result

   # Import information
   puts "Foo2 imported into:\
      \{[::Simple::Package::information importedinto Foo2]\}"

} -cleanup {

   # Remove the colliding procedure
   rename {::foo::Foo2 Exported} {}

   # Clean the second package
   package forget Foo2
   ::Simple::Package::Priv::delete-declared {Foo2 1}
   namespace delete ::Simple::Foo2
   ::Simple::Package::Priv::delete-namespaces-info {Foo2 1}
   ::Simple::Package::Priv::delete-installed {Foo2 1}

} -output {Foo2 imported into: {}
couldn't import package "Foo2 1": command (or alias) "Foo2 Exported" in\
namespace "::foo" collides with exported command\
"::Simple::Foo2::Foo2 Exported" (package "Foo2 1")
Foo2 imported into: {}
}

### ===========================================================================
### -test     : ::Simple::Package::unimport
test-case unimport-1 {
   ::Simple::Package::unimport
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Install the first package
   ::Simple::Package::install Foo1

   # Import the first package into ::foo
   ::Simple::Package::import Foo1 ::foo

} -script {

   # Import information
   puts "Foo1 imported into:\
      \{[::Simple::Package::information importedinto Foo1]\}"

   # Unimport the first package from ::foo
   ::Simple::Package::unimport Foo1 ::foo

   # Import information
   puts "Foo1 imported into:\
      \{[::Simple::Package::information importedinto Foo1]\}"

   # Ensure the first package exported commands
   # have been unimported from ::foo
   ::foo::Exported

} -output {Foo1 imported into: {::foo ::}
Foo1 imported into: {::}
} -error  {invalid command name "::foo::Exported"}

test-case unimport-2 {
   ::Simple::Package::unimport
   error, package not declared
} -script {
   ::Simple::Package::unimport Bar ::foo
} -error {undeclared package "Bar"}

test-case unimport-3 {
   ::Simple::Package::unimport
   error, package not imported
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 1}
} -setup {

   # Create a namespace
   namespace eval ::bar {}

} -script {
   ::Simple::Package::unimport Foo1 ::bar
} -cleanup {

   # Delete the namespace
   namespace delete ::bar

} -error {package "Foo1 1.0" not imported into namespace "::bar"}

test-case unimport-4 {
   ::Simple::Package::unimport
   error, non-existing namespace
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 1}
} -script {
   ::Simple::Package::unimport Foo1 ::Non-existing
} -error {non-existing namespace "::Non-existing"}

test-case unimport-5 {
   ::Simple::Package::unimport
   error, package not installed
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 1}
} -setup {

   # Uninstall the first package
   ::Simple::Package::Priv::delete-installed {Foo1 1.0}
   rename Exported {}

} -script {
   ::Simple::Package::unimport Foo1 ::foo
} -error {uninstalled package "Foo1 1.0"}

### ===========================================================================
### -test     : ::Simple::Package::uninstall
test-case uninstall-1 {
   ::Simple::Package::uninstall
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Second declared package
   ::Simple::Package::declare Foo2 1 -namespaces {
      ::Simple::Foo2
   } -export {
      {::Simple::Foo2::Exported}
   } -aliases {
      {Alias ::Simple::Foo2::Foo2-Alias}
   } -firsttimeinstall {
      proc ::Simple::Foo2::Exported {} {
         puts [info level 0]
      }
      proc ::Simple::Foo2::Foo2-Alias {} {
         puts Alias
      }
   } -uninstall {puts {Uninstalling Foo2 1}}

   # Install the second package
   ::Simple::Package::install Foo2

   # Import the second package into ::foo
   ::Simple::Package::import Foo2 ::foo

} -script {

   # One package installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Import information
   puts "Foo2 imported into:\
      \{[lsort [::Simple::Package::information importedinto Foo2]]\}"

   # Alias
   Alias

   # Exported procedure
   Exported

   # Uninstall the second package
   ::Simple::Package::uninstall Foo2

   # No packages installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Import information
   puts "Foo2 imported into:\
      \{[::Simple::Package::information importedinto Foo2]\}"

   # The alias and the exported procedure no longer exist
   catch Alias result
   puts $result
   catch Exported result
   puts $result

} -output {installed:{Foo2 1}
Foo2 imported into: {:: ::foo}
Alias
Exported
Uninstalling Foo2 1
installed:
Foo2 imported into: {}
invalid command name "Alias"
invalid command name "Exported"
}

test-case uninstall-2 {
   ::Simple::Package::uninstall
   install/uninstall/install cycle
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Third declared package
   ::Simple::Package::declare Foo3 1 -namespaces {
      ::Simple::Foo3
   } -install {
      # A package variable
      set ::Simple::Foo3::Variable 0
      puts {Installing Foo3 1}
   } -firsttimeinstall {
      puts {   for the first time!}
   } -uninstall {
      puts {Uninstalling Foo3 1}
   }

} -script {

   # Install the third package
   ::Simple::Package::install Foo3

   # Access the package variable
   puts $::Simple::Foo3::Variable

   # Modify it
   set ::Simple::Foo3::Variable 1

   # Access it
   puts $::Simple::Foo3::Variable

   # Uninstall the third package
   ::Simple::Package::uninstall Foo3

   # Install the third package
   ::Simple::Package::install Foo3

   # Access the package variable
   puts $::Simple::Foo3::Variable

} -cleanup {

   # Uninstall the third package
   ::Simple::Package::uninstall Foo3

   # Undeclare the third package
   ::Simple::Package::undeclare Foo3

} -output {Installing Foo3 1
   for the first time!
0
1
Uninstalling Foo3 1
Installing Foo3 1
0
}

test-case uninstall-3 {
   ::Simple::Package::uninstall
   error, package not declared
} -script {
   ::Simple::Package::uninstall Bar
} -error {undeclared package "Bar"}

test-case uninstall-4 {
   ::Simple::Package::uninstall
   error, package not installed
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -script {
   ::Simple::Package::uninstall Foo1
} -error {uninstalled package "Foo1 1.0"}

test-case uninstall-5 {
   ::Simple::Package::uninstall
   error, error evaluating the uninstall script
} -prerequisites {
   {[llength [::Simple::Package::information installed Foo*]] == 0}
} -setup {

   # Third declared package
   ::Simple::Package::declare Foo3 1.0 -uninstall {
      puts {Uninstalling Foo3 1.0}
      foo bar
   }

   # Install the third package
   ::Simple::Package::install Foo3

} -script {

   # One package installed
   puts "installed:[::Simple::Package::information installed Foo*]"

   # Failed uninstallation
   catch {
      ::Simple::Package::uninstall Foo3
   } result
   puts $result

   # One package installed
   puts "installed:[::Simple::Package::information installed Foo*]"

} -cleanup {

   # Clean the third package
   package forget Foo3
   ::Simple::Package::Priv::delete-declared {Foo3 1.0}

} -output {installed:{Foo3 1.0}
Uninstalling Foo3 1.0
error evaluating package "Foo3 1.0" uninstall script:\
invalid command name "foo"
installed:{Foo3 1.0}
}

### ===========================================================================
### -test     : ::Simple::Package::undeclare
test-case undeclare-1 {
   ::Simple::Package::undeclare
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 2}
} -setup {

   # Add a fake ifneeded script to the first package
   package ifneeded Foo1 1.0 script

} -script {

   # Two packages declared
   puts "declared:[lsort [::Simple::Package::information declared Foo*]]"

   # Packages namespaces
   puts [namespace children ::Simple Foo1]
   puts [namespace children ::Simple Foo2]

   # Packages states
   puts [::Simple::Package::information state Foo1]
   puts [::Simple::Package::information state Foo1]

   # Undeclare the packages
   ::Simple::Package::undeclare Foo1
   ::Simple::Package::undeclare Foo2

   # No packages declared
   puts "declared:[::Simple::Package::information declared Foo*]"

   # Ensure the package namespaces has been deleted
   puts [namespace children ::Simple Foo1]
   puts [namespace children ::Simple Foo2]

   # Package states
   puts [::Simple::Package::information state Foo1]
   puts [::Simple::Package::information state Foo2]

} -output {declared:{Foo1 1.0} {Foo2 1}
::Simple::Foo1
::Simple::Foo2
declared
declared
declared:


available
unknown
}

test-case undeclare-2 {
   ::Simple::Package::undeclare
   error, package not declared
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 0}
} -script {
   ::Simple::Package::undeclare Foo1
} -error {undeclared package "Foo1"}

test-case undeclare-3 {
   ::Simple::Package::undeclare
   error, package installed
} -prerequisites {
   {[llength [::Simple::Package::information declared Foo*]] == 0}
} -setup {

   # Declare a package
   ::Simple::Package::declare Foo1 1.0

   # Install it
   ::Simple::Package::install Foo1

} -script {
   ::Simple::Package::undeclare Foo1
} -cleanup {

   # Clean the package
   ::Simple::Package::uninstall Foo1
   ::Simple::Package::undeclare Foo1

} -error {installed package "Foo1 1.0"}

### ===========================================================================
### -test     : ::Simple::Package::uninstall-and-undeclare
test-case uninstallandunreq-1 {
   ::Simple::Package::uninstall-and-undeclare
   undeclared package
} -prerequisites {
   {[llength [::Simple::Package::information installed *eclared]] == 0}
} -setup {

   # Undeclared package
   package ifneeded Undeclared 1 {
      package provide Undeclared 1
   }

   # Load the package
   ::Simple::Package::require-and-install Undeclared

} -script {

   # Package state
   puts [::Simple::Package::information state Undeclared]

   # Uninstall and undeclare the package
   ::Simple::Package::uninstall-and-undeclare Undeclared

   # Package state
   puts [::Simple::Package::information state Undeclared]

} -cleanup {

   # Clean the package
   package forget Undeclared

} -output {installed
available
}

test-case uninstallandunreq-2 {
   ::Simple::Package::uninstall-and-undeclare
   declared package
} -prerequisites {
   {[llength [::Simple::Package::information installed *eclared]] == 0}
} -setup {

   # Declared package
   package ifneeded Declared 1.0 {
      ::Simple::Package::declare Declared 1.0 -uninstall {
         puts {Uninstalling Declared 1.0}
      }
   }

   # Require and install the package
   ::Simple::Package::require-and-install Declared

} -script {

   # Package state
   puts [::Simple::Package::information state Declared]

   # Uninstall and undeclare the package
   ::Simple::Package::uninstall-and-undeclare Declared

   # Package state
   puts [::Simple::Package::information state Declared]

} -cleanup {

   # Clean the package
   package forget Declared

} -output {installed
Uninstalling Declared 1.0
available
}

test-case uninstallandunreq-3 {
   ::Simple::Package::uninstall-and-undeclare
   error, package not installed
} -prerequisites {
   {[llength [::Simple::Package::information installed *eclared]] == 0}
} -setup {

   # Declared package
   ::Simple::Package::declare Declared 1.0\
      -uninstall {puts {Uninstalling Declared 1.0}}

} -script {

   # Undeclared package
   catch {
      ::Simple::Package::uninstall-and-undeclare Undeclared
   } result
   puts $result
      
   # Declared package
   ::Simple::Package::uninstall-and-undeclare Declared

} -output {uninstalled package "Undeclared"
} -error {uninstalled package "Declared 1.0"}

### ===========================================================================
### -test     : ::Simple::Package::configure
test-case configure-1 {
   ::Simple::Package::configure
} -script {
   ::Simple::Package::configure
} -error {this package has no options}

### ===========================================================================
### -test     : ::Simple::Package::cget
test-case cget-1 {
   ::Simple::Package::cget
} -script {
   ::Simple::Package::cget
} -return {}

} ;# End of regression testing section
