### ===========================================================================
### -package  : SimpleProc
# -version    : 0.2
# -purpose    : Procedures related utilities.
# -overview   :
#     This package provides utilities for procedures handling.  Procedures are
#  provided for obtaining the name of a caller procedure, copying a procedure,
#  assessing whether a procedure exists and obtaining a procedure argument
#  usage string or argument list.
#
# -keywords   : procedure argument usage caller
# -commands   :
#
#  * K<::Simple::Proc::caller> ?level?
#    Returns the caller procedure name.
#
#  * K<::Simple::Proc::copy> sourceProcedure targetProcedure
#    Copies a procedure.
#
#  * K<::Simple::Proc::information exists> procedure
#    Returns whether a procedure exists.
#
#  * K<::Simple::Proc::information argsusage> procedure
#    Returns a procedure argument usage string.
#
#  * K<::Simple::Proc::information argslist> procedure
#    Returns a procedure argument list.
#
#  * K<::Simple::Proc::configure>
#    Configures the package options.
#
#  * K<::Simple::Proc::cget>
#    Gets the package options.
#
# -variables  :
#
#  { ::Simple::Proc::Priv::SavedName
#                     -name          "Name saved by
#                                     K<::Simple::Proc::Pric::save-name>"
#
# -history    :
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleProc) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleProc 0.2 -required {
   SimpleError SimplePackage SimpleNamespace
} -namespaces {
   ::Simple::Proc
   ::Simple::Proc::Priv
} -firsttimeinstall {

### Declare the package errors
::Simple::Error::declare ::Simple::Proc::CANT-COPY,DELETE-OR-RENAME {
   can't %s "%s": command doesn't exist
} {
   The procedure "<procedure>" could
   not be <copy, delete or rename>d because it does not exist
} {} {<copy, delete or rename> <procedure>}

### ===========================================================================
### -command  : ::Simple::Proc::caller
# -purpose    : Returns the caller procedure name.
# -overview   :
#     This procedure returns the name of a procedure up the calling stack.  By
#  default, the scope of the procedure which called the one which calls this
#  one is used, that is two levels up the stack frame.
#
# -returns    : The caller procedure name or the empty string if not found.
# -arguments  :
#  {?level?           -int 1         {Levels up the stack in which to operate}}
#
proc ::Simple::Proc::caller {
   {level 1}
} {
   if {![catch {set answer [lindex [info level -[incr level]] 0]}]} {
      set answer
   }
}

### ===========================================================================
### -command  : ::Simple::Proc::Priv::save-name
# -purpose    : Saves a variable or procedure name to be accessed later.
# -overview   :
#     This procedure saves a variable or procedure name which can be later
#  obtained via the P<::Simple::Proc::Priv::restore-name> procedure.  The
#  purpose of this pair of functions is to save the original name of a
#  variable or procedure given by the user as an argument to a Simple Library
#  procedure in order to be used for error reporting.
#
# -arguments  :
#  { name             -name          {Variable or procedure name to save}}
#
# -seealso    :
#  * The P<::Simple::Proc::Priv::restore-name> procedure.
#
proc ::Simple::Proc::Priv::save-name {
   name
} {
   ### Save the name if two levels up the stack is outside the Simple library
   if {![string match ::Simple* [uplevel 2 namespace current]]} {
      set ::Simple::Proc::Priv::SavedName $name
   }
}

### ===========================================================================
### -command  : ::Simple::Proc::Priv::restore-name
# -purpose    : Restores a variable or procedure name previously saved.
# -overview   :
#     This procedure returns the previously saved variable or procedure name
#  via the P<::Simple::Proc::Priv::save-name> procedure
#
# -returns    : The previously saved variable or procedure name.
#
# -seealso    :
#  * The P<::Simple::Proc::Priv::save-name> procedure.
#
proc ::Simple::Proc::Priv::restore-name {
} {
   set ::Simple::Proc::Priv::SavedName
}
### ===========================================================================
### -command  : ::Simple::Proc::copy
# -purpose    : Copies a procedure.
# -arguments  :
#  { sourceProcedure  -name          {Source procedure name}}
#  { targetProcedure  -name          {Target procedure name}}
#
# -remarks    :
#  * If the target procedure namespace does not exists, it is created. 
#    This mimics the behaviour of the K<rename> command.
#
# -effects    :
#  * Creates a procedure named "A<targetProcedure>".
#
proc ::Simple::Proc::copy {
   sourceProcedure
   targetProcedure
} {
   ### Save the procedure name
   ::Simple::Proc::Priv::save-name $sourceProcedure

   ### Qualify the procedure name in the calling scope
   if {![string match ::* $sourceProcedure]} {
      set sourceProcedure [::Simple::Namespace::Priv::qualify $sourceProcedure]
   }

   ### Procedure does not exist
   if {![{::Simple::Proc::information exists} $sourceProcedure]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Proc::CANT-COPY,DELETE-OR-RENAME\
         copy [::Simple::Proc::Priv::restore-name]

   ### Procedure exists
   } else {

      ### Get the procedure necessary information
      set arguments\
         [{::Simple::Proc::information argslist} $sourceProcedure]
      set body [info body $sourceProcedure]

      ### Qualify the target procedure name in the calling scope
      if {![string match ::* $targetProcedure]} {
         set targetProcedure\
            [::Simple::Namespace::Priv::qualify $targetProcedure]
      }

      ### Create the target procedure namespace if it does not exist
      namespace eval [::Simple::Namespace::qualifiers $targetProcedure] {}

      ### Create the target procedure
      proc $targetProcedure $arguments $body
   }
}

### ===========================================================================
### -command  : ::Simple::Proc::information
# -purpose    : Base command for package K<information> subcommands.
# -access     : Private
# -overview   :
#     This is the base command for the K<::Simple::Proc::information>
#  subcommand procedures.
#
# -arguments  :
#  { args             -list          {Subcommand and subcommand arguments}}
#
proc ::Simple::Proc::information {
   args
} {
   #========= %BASE-COMMAND%

   ### Get the subcommand
   set subcommand [lindex $args 0]
   set fullCommand ::Simple::Proc::information\ $subcommand

   ### Evaluate the subcommand procedure
   ### Error from subcommand procedure
   #   The uplevel here ensures that the call to the base command
   #   is hidden to the command procedure body
   if {[catch {uplevel [list $fullCommand] [lrange $args 1 end]} result]} {

      ### No subcommand given
      if {[llength $args] == 0} {

         ### Throw error
         ::Simple::Error::throw ::Simple::NO-SUBCOMMAND\
            ::Simple::Proc::information

      ### Non-existing subcommand
      } elseif {![string compare "invalid command name \"$fullCommand\""\
         $result]} {

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND $subcommand\
            [::Simple::Proc::Priv::enumerate\
            [lsort {argslist argsusage exists}]]

      ### Invalid arguments
      } elseif {![string compare "called \"$fullCommand\" with too many\
         arguments" $result] || [string match {no value given for\
         parameter*} $result]} {

         ### Compose the correct arguments
         switch -exact -- $subcommand {
            argslist -
            argsusage -
            exists {
               set arguments { procedure}
            }
         }

         ### Throw error
         ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND-ARGS\
            $fullCommand $arguments

      ### Other error
      } else {

         ### Rethrow the subcommand procedure error
         return -code error -errorcode $::errorCode $result
      }

   ### No error from subcommand procedure
   } else {

      ### Return the subcommand procedure return value
      set result
   }
}

### ===========================================================================
### -command  : ::Simple::Proc::information exists
# -purpose    : Returns whether a procedure exists.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : Whether the procedure exists.
#
proc {::Simple::Proc::information exists} {
   procedure
} {
   ### Qualify the procedure name in the calling scope
   if {![string match ::* $procedure]} {
      set procedure [::Simple::Namespace::Priv::qualify $procedure]
   }

   ### Get the procedure namespace
   set namespace [::Simple::Namespace::qualifiers $procedure]

   ### Namespace exists
   #   The existance of the namespace is checked first so as to not
   #   create it with the "namespace eval" below
   if {[::Simple::Namespace::information exists $namespace]} {

      ### Get the procedure tail
      set tail [list [namespace tail $procedure]]

      ### Get whether the procedure exists
      set answer\
         [expr {[llength [namespace eval $namespace "info procs $tail"]] == 1}]

   ### Namespace does not exist
   } else {

      ### Return false
      set answer 0
   }

   ### Return whether the procedure exists
   set answer
}

### ===========================================================================
### -command  : ::Simple::Proc::information argsusage
# -purpose    : Returns a procedure argument usage string.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : The procedure argument usage string.
# -remarks    :
#  * The returned string is empty if the procedure has no arguments but its
#    first character is an space otherwise.
#
# -seealso    :
#  * The P<::Simple::ExtProc::information argsusage> procedure.
#
proc {::Simple::Proc::information argsusage} {
   procedure
} {
   set answer {}
   foreach argument [uplevel [list info args $procedure]] {
      if {![string compare $argument args] ||\
         [uplevel [list info default $procedure $argument default]]} {

         append answer " ?$argument?"
      } else {
         append answer " $argument"
      }
   }
   set answer
}

### ===========================================================================
### -command  : ::Simple::Proc::information argslist
# -purpose    : Returns a procedure argument list.
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
# -returns    : The procedure argument list.
#
# -seealso    :
#  * The P<::Simple::ExtProc::information argslist> procedure.
#
proc {::Simple::Proc::information argslist} {
   procedure
} {
   set answer [list]
   foreach argument [uplevel [list info args $procedure]] {
      if {[uplevel [list info default $procedure $argument\
         ::Simple::Proc::%TMP%]]} {

         lappend answer [list $argument ${::Simple::Proc::%TMP%}]
      } else {
         lappend answer $argument
      }
   }
   set answer
}

### ===========================================================================
### -command  : ::Simple::Proc::Priv::enumerate
# -purpose    : Enumerates the elements of a list.
# -overview   :
#     This procedure returns a string built by separating the input list
#  elements with commas except the last one which is preceded by the separator
#  argument which defaults to "or".  If the number of elements in the list is
#  lower than two the list is returned unmodified. 
#
# -arguments  :
#  { list             -list          {List}}
#  {?separator?       -string or     {Modifier for last item}}
#
# -returns    : String containing the enumerated list.
#
proc ::Simple::Proc::Priv::enumerate {
   list
   {separator or}
} {
   set nElements [llength $list]
   if {$nElements > 1} {
      concat [join [lrange $list 0 [expr {$nElements - 2}]] {, }]\
         $separator [lindex $list end]
   } else {
      set list
   }
}

### ===========================================================================
### -command  : ::Simple::Proc::configure
# -purpose    : Configures the package options.
#
proc ::Simple::Proc::configure {
} {
   ::Simple::Error::throw ::Simple::NO-OPTIONS
}

### ===========================================================================
### -command  : ::Simple::Proc::cget
# -purpose    : Gets the package options.
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Proc::cget {
} {
   list
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleProc-test 0.2

### ===========================================================================
### -test     : ::Simple::Proc::caller
test-case caller-1 {
   ::Simple::Proc::caller
} -setup {
   namespace eval foo {}
} -script {

   # No caller
   puts \{[::Simple::Proc::caller]\}

   # Caller is bar, no caller-of-caller
   proc foo {} {
      puts [::Simple::Proc::caller]
      puts [::Simple::Proc::caller 2]
   }
   proc bar {} {
      ::foo
   }
   bar

   # Caller is bar, caller-of-caller is ::foo::gee
   proc ::foo::gee {} {
      bar
   }
   ::foo::gee

} -cleanup {
   namespace delete foo
   rename foo {}
   rename bar {}
} -output {{}
bar

bar
::foo::gee
}

### ===========================================================================
### -test     : ::Simple::Proc::Priv::save-name
test-case save-restore-name-1 {
   ::Simple::Proc::save-name and ::Simple::Proc::restore-name
} -setup {

   # Create some procedures 

   # ::Simple::gee saves GEE
   proc ::Simple::gee {} {
      ::Simple::Proc::Priv::save-name GEE
   }

   # ::Simple::bar saves BAR and calls ::Simple::gee
   proc ::Simple::bar {} {
      ::Simple::Proc::Priv::save-name BAR
      ::Simple::gee
   }

   # foo saves FOO and calls ::Simple::bar
   proc foo {} {
      ::Simple::Proc::Priv::save-name FOO
      ::Simple::bar
   }

} -script {

   # Call foo
   foo

   # Get the saved name
   ::Simple::Proc::Priv::restore-name

} -cleanup {

   # Clean the procedures
   rename foo {}
   rename ::Simple::bar {}
   rename ::Simple::gee {}

} -return BAR

### ===========================================================================
### -test     : ::Simple::Proc::copy
test-case copy-1 {
   ::Simple::Proc::copy
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure
   proc ::foo::bar {one two {three 3}} {
      puts "This is [lindex [info level 0] 0]"
   }

} -script {

   ### Call the source procedure
   ::foo::bar 1 2

   ### Copy the source to a first target procedure
   ::Simple::Proc::copy ::foo::bar ::bar::gee

   ### Call the first target procedure
   ::bar::gee 1 2

   ### Copy the source to a second target procedure
   namespace eval ::foo {
      ::Simple::Proc::copy bar zii
   }

   ### Call the second target procedure
   ::foo::zii 1 2

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

   # Clean the procedures
   rename ::bar::gee {}

} -output {This is ::foo::bar
This is ::bar::gee
This is ::foo::zii
}

test-case copy-2 {
   ::Simple::Proc::copy
   error, non-existing procedure
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] == -1}
} -script {

   # Failed procedure copies
   catch {::Simple::Proc::copy foo bar} result
   puts $result
   catch {::Simple::Proc::copy ::foo::bar ::bar::foo} result
   puts $result

   # Ensure the namespace ::foo has not been created
   puts [lsearch -exact [namespace children] ::foo]

   # Failed procedure copies
   namespace eval ::foo {
      ::Simple::Proc::copy bar foo
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {can't copy "foo": command doesn't exist
can't copy "::foo::bar": command doesn't exist
-1
} -error {can't copy "bar": command doesn't exist}

test-case copy-3 {
   ::Simple::Proc::copy
   error, unknown namespace
} -prerequisites {
   {[lsearch -exact [namespace children] ::foo] == -1}
} -script {

   # Failed procedure copy
   catch {::Simple::Proc::copy ::foo::bar ::bar::foo} result
   puts $result

   # Ensure the namespace ::foo has not been created
   puts [lsearch -exact [namespace children] ::foo]

   # Failed procedure copy
   namespace eval ::foo {
      ::Simple::Proc::copy bar foo
   }

} -output {can't copy "::foo::bar": command doesn't exist
-1
} -error {can't copy "bar": command doesn't exist}

### ===========================================================================
### -test     : ::Simple::Proc::information
test-case info-1 {
   ::Simple::Proc::information
} -script {
   ::Simple::Proc::information exists ::Simple::Proc::information
} -return 1

test-case info-2 {
   ::Simple::Proc::information
   subcommand throws an error
} -setup {

   # Create a subcommand procedure
   proc {::Simple::Proc::information foo} {} {
      return -code error -errorcode errorCode errorMessage
   }

} -script {
   set return [catch {::Simple::Proc::information foo} result]
   puts "return: \{$return\}"
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"
} -cleanup {

   # Delete the subcommand procedure
   rename {::Simple::Proc::information foo} {}

} -output {return: {1}
result: {errorMessage}
errorCode: {errorCode}
}

test-case info-3 {
   ::Simple::Proc::information
   error, no subcommand given
} -script {
   ::Simple::Proc::information
} -error {wrong # args: should be "::Simple::Proc::information option ?arg\
...?"}

test-case info-4 {
   ::Simple::Proc::information
   error, non-existing subcommand
} -regexp -script {
   ::Simple::Proc::information foo
} -error {bad option "foo": *}

test-case info-5 {
   ::Simple::Proc::information
   error, invalid subcommand arguments
} -script {
   ::Simple::Proc::information exists foo bar
} -error {wrong # args: should be "::Simple::Proc::information exists\
procedure"}

### ===========================================================================
### -test     : ::Simple::Proc::information exists
test-case exists-1 {
   ::Simple::Proc::information exists
} -setup {

   # Create some procedures
   namespace eval foo {}
   proc foo {} {}
   proc ::foo::bar {} {}

} -script {

   puts -nonewline [::Simple::Proc::information exists foo]
   puts -nonewline [::Simple::Proc::information exists ::foo::bar]
   puts -nonewline [::Simple::Proc::information exists bar]
   puts            [::Simple::Proc::information exists ::gee::bar]

   # Ensure the namespace ::gee has not been created
   lsearch -exact [namespace children] ::gee

} -cleanup {

   # Delete the procedures
   namespace delete foo
   rename ::foo {}

} -output {1100
} -return -1

### ===========================================================================
### -test     : ::Simple::Proc::information argsusage
test-case argsusage-1 {
   ::Simple::Proc::information argsusage
} -setup {

   # Create one procedure
   proc foo {
      arg1
      arg2
      {arg3 default1}
      {arg4 default2}
      args
   } {}

} -script {
   {::Simple::Proc::information argsusage} foo
} -return { arg1 arg2 ?arg3? ?arg4? ?args?}

### ===========================================================================
### -test     : ::Simple::Proc::information argslist
test-case argslist {
   ::Simple::Proc::information argslist
} -prerequisites {
   {[llength [info args ::foo]] == 5}
} -script {
   {::Simple::Proc::information argslist} foo
} -cleanup {

   # Delete the procedure
   rename ::foo {}

} -return {arg1 arg2 {arg3 default1} {arg4 default2} args}

### ===========================================================================
### -test     : ::Simple::Proc::Priv::enumerate
test-case enumerate-1 {
   ::Simple::Proc::Priv::enumerate
} -script {
   puts [::Simple::Proc::Priv::enumerate {}]
   puts [::Simple::Proc::Priv::enumerate one]
   puts [::Simple::Proc::Priv::enumerate {one two}]
   puts [::Simple::Proc::Priv::enumerate {one two three four}]
   puts [::Simple::Proc::Priv::enumerate {one two three four} and]
} -output {
one
one or two
one, two, three or four
one, two, three and four
}

### ===========================================================================
### -test     : ::Simple::Proc::configure
test-case configure-1 {
   ::Simple::Proc::configure
} -script {
   ::Simple::Proc::configure
} -error {this package has no options}

### ===========================================================================
### -test     : ::Simple::Proc::cget
test-case cget-1 {
   ::Simple::Proc::cget
} -script {
   ::Simple::Proc::cget
} -return {}

} ;# End of regression testing section
