### ===========================================================================
### -package  : SimpleSubcommand
# -version    : 0.2
# -purpose    : Subcommand procedures handling.
# -overview   :
#     This package allows to handle subcommand procedures, that is, procedures
#  with a name (the base command) and a first argument (the subcommand)
#  followed by an optional list of arguments.  Procedures are provided to
#  create, delete, copy, rename and get information about subcommand
#  procedures.
#
# -usage      :
#     Subcommand procedures are create via K<::Simple::Subcommand::create>. 
#  Such subcommand procedures behave similarly to the built-in Tcl commands
#  with subcommands (such as K<string> or K<array>):  
#
#  * They are called simply by providing the command and subcommand name, such
#    as "foo bar".  Nevertheless, it is slightly faster to use "foo\ bar" or
#    "{foo bar}" as a level of distpach is avoided. 
#
#  * When called with no subcommand, an error is thrown informing of the
#    allowed subcommands. 
#
#  * When called with an incorrect number of arguments for a particular
#    subcommand an error is thrown informing of the argument list for that
#    subcommand.
#
#     Subcommand procedures can be deleted via K<::Simple::Subcommand::delete>
#  or renamed via K<::Simple::Subcommand::rename>.  The
#  K<::Simple::Subcommand::information> procedure provides several subcommands
#  to query information about subcommand procedures:  K<exists> and
#  K<subcommands>.  Refer to the description of each procedure below for
#  further information.
#
# -keywords   : procedure command subcommand
# -variables  :
#
#  { BaseCommandTag   -string %BASE-COMMAND%
#                                    "The base command tag. A procedure is
#                                     considered a base command procedure if
#                                     its body contains this tring"}
#
# -commands   :
#
#  * K<create> procedure arguments body
#    Creates a subcommand procedure.
#
#  * K<::Simple::Subcommand::delete> procedure
#    Deletes a subcommand procedure.
#
#  * K<::Simple::Subcommand::move> sourceProcedure targetProcedure
#    Renames a subcommand procedure.
#
#  * K<::Simple::Subcommand::copy> sourceProcedure targetProcedure
#    Copies a subcommand procedure.
#
#  * K<::Simple::Subcommand::information exists> command
#    Returns whether a procedure is a base command procedure.
#
#  * K<::Simple::Subcommand::information subcommands> command
#    Returns a base command procedure subcommand list.
#
#  * K<::Simple::Subcommand::configure>
#    Configures the package options.
#
#  * K<::Simple::Subcommand::cget>
#    Gets the package options.
#
# -examples   :
#
#  # Install the package
#  package require SimplePackage
#  ::Simple::Package::require-and-install SimpleSubcommand
#  
#  # Create some subcommands
#  proc-sub {foo bar} arg1 {}
#  proc-sub {foo gee} arg1 {}
#  
#  # Assess whether foo is a subcommand base command
#  # This displays the following:
#  #    Yes, foo is a subcommand base command
#  if {[::Simple::Subcommand::information exists foo]} {
#     puts {Yes, foo is a subcommand base command}
#  } else {
#     puts {No, foo is not a subcommand base command}
#  }
#  
#  # Display the list of subcommands for foo
#  # This displays the following:
#  #    gee bar
#  puts [::Simple::Subcommand::information subcommands foo]
#  
#  # Delete the "foo bar" subcommand
#  ::Simple::Subcommand::delete {foo bar}
#  
#  # Display the list of subcommands for foo
#  # This displays the following:
#  #    gee
#  puts [::Simple::Subcommand::information subcommands foo]
#  
#  # Rename the "foo gee" subcommand to "foo bar"
#  ::Simple::Subcommand::move {foo gee} {foo bar}
#  
#  # Display the list of subcommands for foo
#  # This displays the following:
#  #    bar
#  puts [::Simple::Subcommand::information subcommands foo]
#  
#  # Rename the "foo bar" subcommand to "zoo fii"
#  ::Simple::Subcommand::move {foo bar} {zoo fii}
#  
#  # Assess whether foo is a subcommand base command
#  # This displays the following:
#  #    No, foo is not a subcommand base command
#  if {[::Simple::Subcommand::information exists foo]} {
#     puts {Yes, foo is a subcommand base command}
#  } else {
#     puts {No, foo is not a subcommand base command}
#  }
#
# -details    : 
#  * Subcommand procedures are implemented with the aid of a base procedure
#    named as the command.  That is, if a subcommand procedure named K<foo
#    bar> is created, two procedures are actually cretated:  the subcommand
#    procedure itself (named "foo bar") and the base command (named "foo"). 
#    The base command simply acts as a dispatcher for the actual subcommand
#    procedures.  It also catches the subcommand procedure errors and throws
#    explanation errors when appropriate.
#
#  * For this reason, it is slightly faster to invoke a subcommand procedure
#    as "{foo bar}" or "foo\ bar" instead of "foo bar", as the dispatch from
#    the base command is avoided.
#
#  more esoteric features of the P<opt> package such as argument names
# -history    :
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleSubcommand) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleSubcommand 0.2 -required {
   SimpleError SimplePackage SimpleProc SimpleNamespace
} -namespaces {
   ::Simple::Subcommand
   ::Simple::Subcommand::Priv
} -aliases {
   {proc-sub ::Simple::Subcommand::create}
} -firsttimeinstall {

### Create the package variables
set ::Simple::Subcommand::BaseCommandTag %BASE-COMMAND%

### Declare the package errors
::Simple::Error::declare ::Simple::Subcommand::CANT-CREATE {
   can't create subcommand "%s %s": procedure "%s" already exists
} {
   The subcommand procedure "<command> <subcommand>" could not be created
   because a procedure named as the subcommand base command (<command>)
   already exists
} {
   Either delete that procedure or choose another name for the
   subcommand base command
}

::Simple::Error::declare ::Simple::Subcommand::NON-EXISTING-COMMAND {
   "%s" isn't a subcommand base command procedure
} {
   The subcommand list could not be retrieved because <command> is not a
   subcommand base command procedure.  Either it is a regular procedure, a
   subcommand procedure or does not exist.
}

### ===========================================================================
### -command  : ::Simple::Subcommand::Priv::parse-qualify-name
# -purpose    : Parses and fully-qualifies a subcommand procedure name.
# -overview   :
#     This procedure parses a subcommand procedure name which must be two
#  words, the base command and subcommand names.  In case the base command
#  name is unqualified, it is fully-qualified in the scope two levels up the
#  stack.  The procedure then returns a three element list containing the
#  fully-qualified full name, the fully-qualified base command name and the
#  subcommand name.
#
# -returns    : A list with the following elements:
#     0: full command
#     1: command
#     2: subcommand
#
# -arguments  :
#  { procedure        -name          {Procedure name}}
#
proc ::Simple::Subcommand::Priv::parse-qualify-name {
   procedure
} {
   ### Invalid number of words in name
   if {[llength $procedure] != 2} {

      ### Throw error
      ::Simple::Error::throw ::Simple::BAD-ITEM\
         {subcommand procedure name} $procedure
   }

   ### Get the base command and subcommand names
   foreach {command subcommand} $procedure break

   ### Assert the base command and subcommand names are not empty
   if {[llength $command] == 0} {
      ::Simple::Error::throw ::Simple::BAD-ITEM\
         {subcommand base command name} {}
   }
   if {[llength $subcommand] == 0} {
      ::Simple::Error::throw ::Simple::BAD-ITEM {subcommand name} {}
   }

   ### Qualify the base command in the scope two levels up the stack
   if {![string match ::* $command]} {
      set command [::Simple::Namespace::Priv::qualify $command 2]
   }

   ### Return the full command, the base command and the subcommand names
   list $command\ $subcommand $command $subcommand
}

### ===========================================================================
### -command  : ::Simple::Subcommand::create
# -purpose    : Creates a subcommand procedure.
# -alias      : proc-sub
# -overview   :
#     This procedure acts as an interface to the K<proc> Tcl command to create
#  subcommand procedures.  Such subcommand procedures behave similarly to the
#  built-in Tcl commands with subcommands (such as K<string> or K<array>) in
#  that when called with no subcommand, an error is thrown informing of the
#  allowed subcommands; also, when called with an incorrect number of
#  arguments for a particular subcommand, an error is thrown informing of the
#  argument list for that subcommand.
#
#     The syntax for the argumen list is identical to that accepted by the
#  K<proc> procedure.
#
# -access     : Export
# -arguments  :
#  { procedure        -name          {Base command and subcommand pair}}
#  { arguments        -list          {Subcommand procedure argument list}}
#  { body             -script        {Subcommand procedure body}}
#
# -keywords   : procedure command subcommand
# -examples   :
#
#     # Create some subcommands
#     proc-sub {foo bar} arg1 {puts $arg1}
#     proc-sub {foo gee} {arg1 {arg2 99} args} {puts "$arg1 $arg2 $args"}
#
# -details    : 
#     A base command is created with the given base command name.  The base
#  command body simply calls the P<::Simple::Subcommand::Priv::base-command>
#  procedure passing all arguments received.  The actual subcommand procedure
#  is then created with the name "<command> <subcommand>" where "<command>" is
#  the base command name and "<subcommand>" is the subcommand name.
#
# -seealso    :
#  * The P<::Simple::ExtProc::proc-ext> procedure.
#
# -effects    :
#  * Creates a command named K<procedure>.
#
proc ::Simple::Subcommand::create {
   procedure
   arguments
   body
} {
   ### Save the command
   ::Simple::Proc::Priv::save-name [lindex $procedure 0]

   ### Parse and qualify the procedure name
   foreach {qualifiedProcedure command subcommand}\
      [::Simple::Subcommand::Priv::parse-qualify-name $procedure] break

   ### Create the base command
   ::Simple::Subcommand::Priv::create-base-command $command $subcommand

   ### Create the subcommand
   proc $qualifiedProcedure $arguments $body
}

### ===========================================================================
### -command  : ::Simple::Subcommand::Priv::create-base-command
# -purpose    : Creates a subcommand base command.
# -overview   :
#     This procedure creates a subcommand base command if it does not exists,
#  otherwise it asserts that it is a base command indeed.
#
# -arguments  :
#  { command          -qualifiedname {Fully-qualified command name}}
#  { subcommand       -name          {Subcommand name}}
#
# -effects    :
#  * Creates a command named K<command>.
#
proc ::Simple::Subcommand::Priv::create-base-command {
   command
   subcommand
} {
   ### The base command does not exist
   if {[catch {set baseBody [info body $command]}]} {

      ### Create the base command
      proc $command args [format {

         #========= %%%s%%

         ### Evaluate the subcommand procedure
         ### Error from subcommand procedure
         #   The uplevel here ensures that the call to the base command
         #   is hidden to the user-supplied command procedure body
         if {[catch {uplevel [lindex [info level 0] 0]\\ $args} result]} {

            ### Process the subcommand procedure error
            ::Simple::Subcommand::Priv::subcommand-error %s $result

         ### No error from subcommand procedure
         } else {

            ### Return the user return value
            set result
         }

      } $::Simple::Subcommand::BaseCommandTag $command]

   ### The base command exists
   } else {

      ### Assert that the base command procedure is a base command indeed
      set tmp [::Simple::Proc::Priv::restore-name]
      if {![regexp -- $::Simple::Subcommand::BaseCommandTag $baseBody]} {
         ::Simple::Error::throw ::Simple::Subcommand::CANT-CREATE\
            $tmp $subcommand $tmp
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Subcommand::Priv::subcommand-error
# -purpose    : Processes a subcommand procedure call error.
# -overview   :
#     This procedure is called when a call to a subcommand procedure created
#  via the P<::Simple::Subcommand::create> procedure returns an error.  The
#  error is parsed to intercept the different "wrong number of arguments" Tcl
#  errors as well as the error generated in case the subcommand does not
#  exist; in that cases an error is thrown informing either about the argument
#  list for that subcommand or about the allowed subcommands, respectively. 
#  Otherwise the error is an user error from the subcommand procedure, thus
#  being passed up to the calling scope.
#
# -arguments  :
#  { command          -qualifiedname {Fully-qualified command name}}
#  { result           -string        {Error from the subcommand procedure call}}
#
# -returns    : The subcommand user error.
# -remarks    :
#  * This procedure is recreated by the P<SimpleExtProc> package.
#
proc ::Simple::Subcommand::Priv::subcommand-error {
   command
   result
} {
   ### Get args from the subcommand procedure
   #   K<upvar> is used instead of adding an args argument to this
   #   procedure because it is not possible to discriminate
   #   between an empty args and no args at all
   upvar args args

   ### Get the subcommand and full command names
   set subcommand [lindex $args 0]
   set fullCommand $command\ $subcommand

   ### No subcommand given
   if {[llength $args] == 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::NO-SUBCOMMAND [info level -1]

   ### Non-existing subcommand
   } elseif {![string compare "invalid command name \"$fullCommand\""\
      $result]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND $subcommand\
         [::Simple::Proc::Priv::enumerate\
         [lsort [::Simple::Subcommand::information subcommands $command]]]

   ### Invalid arguments
   } elseif {[string match "called \"*\" with too many arguments"\
      $result] || [string match {no value given for parameter*} $result] ||
      [string match {no parameter given for flag*} $result]} {

      ### Throw error
      #   To get the procedure argument list, use
      #   K<::Simple::ExtProc::information> if available or resort to
      #   K<::Simple::Proc::information> otherwise
      if {[{::Simple::Proc::information exists}\
         ::Simple::ExtProc::information]} {
         set package ExtProc
      } else {
         set package Proc
      }
      ::Simple::Error::throw ::Simple::BAD-SUBCOMMAND-ARGS\
         [concat [lindex [info level -1] 0] $subcommand]\
         [::Simple::${package}::information\ argsusage $fullCommand]

   ### User error
   } else {

      ### Rethrow the subcommand procedure error
      return -code error -errorcode $::errorCode $result
   }
}

### ===========================================================================
### -command  : ::Simple::Subcommand::delete
# -purpose    : Deletes a subcommand procedure.
# -overview   :
#     This command is similar to the K<rename> command when its second argument
#  is empty but works for subcommand procedures only.
#
# -arguments  :
#  { procedure        -name          {Base command and subcommand pair}}
#
# -details    : 
#     The subcommand procedure is deleted and, should it be the last one, the
#  base command is deleted as well.
#
# -effects    :
#  * Deletes the procedure named "A<procedure>".
#
proc ::Simple::Subcommand::delete {
   procedure
} {
   ### Save the command name
   ::Simple::Proc::Priv::save-name [lindex $procedure 0]

   ### Parse and qualify the procedure name
   foreach {qualifiedProcedure command subcommand}\
      [::Simple::Subcommand::Priv::parse-qualify-name $procedure] break

   ### Delete the subcommand procedure
   if {[catch {rename $qualifiedProcedure {}}]} {
      ::Simple::Error::throw ::Simple::Proc::CANT-COPY,DELETE-OR-RENAME\
         delete [list [::Simple::Proc::Priv::restore-name] $subcommand]
   }

   ### No subcommands remaining for the base command
   if {[catch {::Simple::Subcommand::Priv::subcommands $command}]} {

      ### Delete the base command
      rename $command {}
   }
}

### ===========================================================================
### -command  : ::Simple::Subcommand::Priv::subcommands
# -purpose    : Returns a base command procedure subcommand list.
# -arguments  :
#  { command          -qualifiedname {Fully-qualified command name}}
#
# -returns    : The base command procedure subcommand list.
# -remarks    :
#  * Procedures containing a percent sign in its name are assumed to be
#    internal to the Simple Library and are excluded from the output.
#
proc ::Simple::Subcommand::Priv::subcommands {
   command
} {
   ### Get the base command procedure list
   set subcommandProcs [namespace eval [::Simple::Namespace::qualifiers\
      $command] [list info procs [namespace tail $command]\ *]]

   ### Tidy up the subcommand list
   set answer [list]
   foreach fullCommand $subcommandProcs {

      ### Get the subcommand name
      set subcommand [lindex $fullCommand 1]

      ### The subcommand name does not contain a percent sign
      #   This occurs, for example, with "after flags" auxiliary
      #   procedures from the P<SimpleExtProc> package
      if {[string first % $subcommand] == -1} {

         ### Add the subcommand to the subcommand list
         lappend answer $subcommand
      }
   }

   ### Empty subcommand list
   if {[llength $answer] == 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Subcommand::NON-EXISTING-COMMAND\
         [::Simple::Proc::Priv::restore-name]

   ### Non-empty subcommand list
   } else {

      ### Return the subcommand list
      set answer
   }
}

### ===========================================================================
### -command  : ::Simple::Subcommand::move
# -purpose    : Renames a subcommand procedure.
# -overview   :
#     This command is similar to the K<rename> command when its second
#  argument is non-empty but works for subcommand procedures only.
#
# -arguments  :
#  { sourceProcedure  -name          {Source base command and subcommand pair}}
#  { targetProcedure  -name          {Target base command and subcommand pair}}
#
# -details    : 
#     The source subcommand procedure is deleted and the target is created.
#
# -remarks    :
#  * If the target procedure namespace does not exists, it is created. 
#    This mimics the behaviour of the K<rename> command.
#
# -todo       :
#  * Allow to rename from subcommand to regular procedure and viceversa.
#
# -effects    :
#  * Renames the procedure named "A<sourceProcedure>".
#
proc ::Simple::Subcommand::move {
   sourceProcedure
   targetProcedure
} {
   ### Save the source command name
   ::Simple::Proc::Priv::save-name [lindex $sourceProcedure 0]

   ### Parse and qualify the procedure name
   foreach {qualifiedOldProcedure sourceCommand sourceSubcommand}\
      [::Simple::Subcommand::Priv::parse-qualify-name $sourceProcedure] break

   ### Command procedure does not exists
   if {![{::Simple::Proc::information exists} $qualifiedOldProcedure]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Proc::CANT-COPY,DELETE-OR-RENAME\
         rename [list [::Simple::Proc::Priv::restore-name] $sourceSubcommand]

   ### Command procedure exists
   } else {

      ### Get the subcommand procedure necessary information
      set arguments\
         [{::Simple::Proc::information argslist} $qualifiedOldProcedure]
      set body [info body $qualifiedOldProcedure]

      ### Delete the subcommand procedure
      rename $qualifiedOldProcedure {}

      ### Parse and qualify the target procedure name
      foreach {qualifiedNewProcedure targetCommand targetSubcommand}\
         [::Simple::Subcommand::Priv::parse-qualify-name $targetProcedure] break

      ### Create the target command namespace if it does not exist
      namespace eval [::Simple::Namespace::qualifiers $targetCommand] {}

      ### Create the target subcommand procedure
      ::Simple::Subcommand::create\
         $targetCommand\ $targetSubcommand $arguments $body

      ### No subcommands remaining for the source command
      if {[catch {::Simple::Subcommand::Priv::subcommands $sourceCommand}]} {

         ### Delete the base command
         rename $sourceCommand {}
      }
   }
}

### ===========================================================================
### -command  : ::Simple::Subcommand::copy
# -purpose    : Copies a subcommand procedure.
# -arguments  :
#  { sourceProcedure  -name          {Source base command and subcommand pair}}
#  { targetProcedure  -name          {Target base command and subcommand pair}}
#
# -remarks    :
#  * If the target procedure namespace does not exists, it is created. 
#    This mimics the behaviour of the K<rename> command.
#
# -todo       :
#  * Allow to copy from subcommand to regular procedure and viceversa.
#
# -effects    :
#  * Creates a command named K<targetProcedure>.
#
proc ::Simple::Subcommand::copy {
   sourceProcedure
   targetProcedure
} {
   ### Save the source command name
   ::Simple::Proc::Priv::save-name [lindex $sourceProcedure 0]

   ### Parse and qualify the procedure name
   foreach {qualifiedOldProcedure sourceCommand sourceSubcommand}\
      [::Simple::Subcommand::Priv::parse-qualify-name $sourceProcedure] break

   ### Command procedure does not exists
   if {![{::Simple::Proc::information exists} $qualifiedOldProcedure]} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Proc::CANT-COPY,DELETE-OR-RENAME\
         copy [list [::Simple::Proc::Priv::restore-name] $sourceSubcommand]

   ### Command procedure exists
   } else {

      ### Get the subcommand procedure necessary information
      set arguments\
         [{::Simple::Proc::information argslist} $qualifiedOldProcedure]
      set body [info body $qualifiedOldProcedure]

      ### Parse and qualify the target procedure name
      foreach {qualifiedNewProcedure targetCommand targetSubcommand}\
         [::Simple::Subcommand::Priv::parse-qualify-name $targetProcedure] break

      ### Create the target command namespace if it does not exist
      namespace eval [::Simple::Namespace::qualifiers $targetCommand] {}

      ### Create the target subcommand procedure
      ::Simple::Subcommand::create\
         $targetCommand\ $targetSubcommand $arguments $body
   }
}

### ===========================================================================
### -command  : ::Simple::Subcommand::information exists
# -purpose    : Returns whether a procedure is a base command procedure.
# -arguments  :
#  { command          -name          {Base command name}}
#
# -returns    : Whether the procedure is a base command procedure.
# -seealso    :
#  * The P<::Simple::ExtProc::information is-subcommand> procedure.
#
# -details    : 
#     A command is considered a base command procedure if its body contains
#  the base command tag stored in V<::Simple::Subcommand::BaseCommandTag>.
#
::Simple::Subcommand::create {::Simple::Subcommand::information exists} {
   command
} {
   ### Qualify the command name in the calling scope
   if {![string match ::* $command]} {
      set command [::Simple::Namespace::Priv::qualify $command]
   }

   ### The procedure does not exist
   if {[catch {set body [info body $command]}]} {

      ### Return false
      set answer 0

   ### The procedure exists
   } else {

      ### Return whether the procedure is a base command procedure
      set answer\
         [expr {[regexp -- $::Simple::Subcommand::BaseCommandTag $body]}]
   }

   ### Return whether the procedure is a base command procedure.
   set answer
}

### ===========================================================================
### -command  : ::Simple::Subcommand::information subcommands
# -purpose    : Returns a base command procedure subcommand list.
# -arguments  :
#  { command          -name          {Base command name}}
#
# -returns    : The base command procedure subcommand list.
# -seealso    :
#  * The P<::Simple::ExtProc::information subcommands> procedure.
#
# -remarks    :
#  * Procedures containing a percent sign in its name are assumed to be
#    internal to the Simple Library and are excluded from the output.
#
::Simple::Subcommand::create {::Simple::Subcommand::information subcommands} {
   command
} {
   ### Save the command name
   ::Simple::Proc::Priv::save-name $command

   ### Qualify the command name in the calling scope
   if {![string match ::* $command]} {
      set command [::Simple::Namespace::Priv::qualify $command]
   }

   ### Return the subcommand list
   ::Simple::Subcommand::Priv::subcommands $command
}

### ===========================================================================
### -command  : ::Simple::Subcommand::configure
# -purpose    : Configures the package options.
#
proc ::Simple::Subcommand::configure {
} {
   ::Simple::Error::throw ::Simple::NO-OPTIONS
}

### ===========================================================================
### -command  : ::Simple::Subcommand::cget
# -purpose    : Gets the package options.
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Subcommand::cget {
} {
   list
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleSubcommand-test 0.2

### ===========================================================================
### -test     : ::Simple::Subcommand::create
test-case proc-sub-1 {
   ::Simple::Subcommand::create
} -setup {

   # Create two namespaces
   namespace eval ::foo {}
   namespace eval ::bar {}

} -script {

   # Create a hierarchy of procedures
   proc-sub {::foo::command subcommand} arg1 {
      puts [namespace current]
      puts [uplevel namespace current]
      puts [info level]
      puts [uplevel info level]
      puts [info level 0]
      puts [uplevel info level 0]
   }
   proc ::bar::gee arg1 {
      ::foo::command subcommand $arg1
   }

   # Call the hierarchy of procedures
   ::bar::gee zoo

} -cleanup {

   # Delete the namespaces
   namespace delete ::foo
   namespace delete ::bar

} -output {::foo
::bar
2
1
{::foo::command subcommand} zoo
::bar::gee zoo
}

test-case proc-sub-2 {
   ::Simple::Subcommand::create
   no arguments
} -setup {

   # Create a namespace
   namespace eval ::foo {}

   # Create a procedure which displays its arguments 
   proc display-args args {
      puts [info level -1]
      foreach arg $args {
         set command "set $arg"
         puts "$arg = \{[uplevel $command]\}"
      }
   }

} -script {

   # Create a subcommand procedure
   proc-sub {::foo::bar subcommand-1} {} display-args

   # One subcommand
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Incorrect calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   namespace eval ::foo {
      catch {bar subcommand-1 arg1} result
      puts $result
   }

   # Correct call
   ::foo::bar subcommand-1

} -output {subcommand-1
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1
wrong # args: should be "bar subcommand-1"
{::foo::bar subcommand-1}
}

test-case proc-sub-3 {
   ::Simple::Subcommand::create
   arguments
} -prerequisites {
   {[llength [::Simple::Subcommand::information subcommands ::foo::bar]] == 1}
} -script {

   # Create a subcommand procedure
   namespace eval ::foo {
      proc-sub {bar subcommand-2} arg1 {display-args arg1}
   }

   # Two subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Incorrect calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   namespace eval ::foo {
      catch {bar subcommand-2 arg1 arg2} result
      puts $result
   }

   # Correct call
   ::foo::bar subcommand-2 arg1

} -output {subcommand-1 subcommand-2
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1 or subcommand-2
wrong # args: should be "bar subcommand-2 arg1"
{::foo::bar subcommand-2} arg1
arg1 = {arg1}
}

test-case proc-sub-4 {
   ::Simple::Subcommand::create
   arguments, optional 
} -prerequisites {
   {[llength [::Simple::Subcommand::information subcommands ::foo::bar]] == 2}
} -script {

   # Create a subcommand procedure
   namespace eval ::foo {
      proc-sub {bar subcommand-3} {arg1 {arg2 999}} {display-args arg1 arg2}
   }

   # Three subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Incorrect calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   namespace eval ::foo {
      catch {bar subcommand-3} result
      puts $result
      catch {bar subcommand-3 arg1 arg2 arg3} result
      puts $result
   }

   # Correct calls
   ::foo::bar subcommand-3 arg1
   ::foo::bar subcommand-3 arg1 arg2

} -output {subcommand-1 subcommand-2 subcommand-3
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1, subcommand-2 or subcommand-3
wrong # args: should be "bar subcommand-3 arg1 ?arg2?"
wrong # args: should be "bar subcommand-3 arg1 ?arg2?"
{::foo::bar subcommand-3} arg1
arg1 = {arg1}
arg2 = {999}
{::foo::bar subcommand-3} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
}

test-case proc-sub-5 {
   ::Simple::Subcommand::create
   arguments, args
} -prerequisites {
   {[llength [::Simple::Subcommand::information subcommands ::foo::bar]] == 3}
} -script {

   # Create a subcommand procedure
   proc-sub {::foo::bar subcommand-4} {arg1 args} {display-args arg1 args}

   # Four subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Incorrect calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   catch {::foo::bar subcommand-4} result
   puts $result

   # Correct calls
   namespace eval ::foo {
      bar subcommand-4 arg1
   }
   ::foo::bar subcommand-4 arg1 arg2 arg3

} -output {subcommand-1 subcommand-2 subcommand-3 subcommand-4
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1, subcommand-2, subcommand-3 or\
subcommand-4
wrong # args: should be "::foo::bar subcommand-4 arg1 ?args?"
{bar subcommand-4} arg1
arg1 = {arg1}
args = {}
{::foo::bar subcommand-4} arg1 arg2 arg3
arg1 = {arg1}
args = {arg2 arg3}
}

test-case proc-sub-6 {
   ::Simple::Subcommand::create
   arguments, optional, args
} -prerequisites {
   {[llength [::Simple::Subcommand::information subcommands ::foo::bar]] == 4}
} -script {

   # Create a subcommand procedure
   proc-sub {::foo::bar subcommand-5} {arg1 {arg2 999} args} {
      display-args arg1 arg2 args
   }

   # Five subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Incorrect calls
   catch ::foo::bar result
   puts $result
   catch {::foo::bar subcommand-0 arg1} result
   puts $result
   catch {::foo::bar subcommand-5} result
   puts $result

   # Correct calls
   namespace eval ::foo {
      bar subcommand-5 arg1
      bar subcommand-5 arg1 arg2
   }
   ::foo::bar subcommand-5 arg1 arg2 arg3 arg4

} -output {subcommand-1 subcommand-2 subcommand-3 subcommand-4 subcommand-5
wrong # args: should be "::foo::bar option ?arg ...?"
bad option "subcommand-0": must be subcommand-1, subcommand-2, subcommand-3,\
subcommand-4 or subcommand-5
wrong # args: should be "::foo::bar subcommand-5 arg1 ?arg2? ?args?"
{bar subcommand-5} arg1
arg1 = {arg1}
arg2 = {999}
args = {}
{bar subcommand-5} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
args = {}
{::foo::bar subcommand-5} arg1 arg2 arg3 arg4
arg1 = {arg1}
arg2 = {arg2}
args = {arg3 arg4}
}

test-case proc-sub-7 {
   ::Simple::Subcommand::create
   user error
} -prerequisites {
   {[llength [::Simple::Subcommand::information subcommands ::foo::bar]] == 5}
} -script {

   # Create a subcommand procedure
   namespace eval ::foo {
      proc-sub {bar subcommand-6} {} {
         return -code error -errorcode ERRORCODE {ERROR MESSAGE}
      }
   }

   # Six subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Call the subcommand
   catch {::foo::bar subcommand-6} result
   puts "result: \{$result\}"
   puts "errorCode: \{$::errorCode\}"

} -output {subcommand-1 subcommand-2 subcommand-3 subcommand-4 subcommand-5\
subcommand-6
result: {ERROR MESSAGE}
errorCode: {ERRORCODE}
}

test-case proc-sub-8 {
   ::Simple::Subcommand::create
   user return
} -prerequisites {
   {[llength [::Simple::Subcommand::information subcommands ::foo::bar]] == 6}
} -script {

   # Create a subcommand procedure
   proc-sub {::foo::bar subcommand-7} {} {
      return {RETURN VALUE}
   }

   # Seven subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Call the subcommand
   ::foo::bar subcommand-7

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {subcommand-1 subcommand-2 subcommand-3 subcommand-4 subcommand-5\
subcommand-6 subcommand-7
} -return {RETURN VALUE}

test-case proc-sub-9 {
   ::Simple::Subcommand::create
   error, invalid subcommand procedure name
} -script {

   # Failed subcommand procedure creations
   catch {proc-sub {} {} {}} result
   puts $result
   catch {proc-sub foo {} {}} result
   puts $result
   catch {proc-sub {foo {}} {} {}} result
   puts $result
   proc-sub {{} foo} {} {}

} -output {expected subcommand procedure name but got ""
expected subcommand procedure name but got "foo"
expected subcommand name but got ""
} -error {expected subcommand base command name but got ""}

test-case proc-sub-10 {
   ::Simple::Subcommand::create
   error, unknown namespace
} -script {

   # Failed subcommand procedure creation
   catch {proc-sub {::foo::command subcommand} {} {}} result
   puts $result

   # Ensure the namespace ::foo has not been created
   lsearch -exact [namespace children] ::gee

} -output {can't create procedure "::foo::command": unknown namespace
} -return -1

test-case proc-sub-11 {
   ::Simple::Subcommand::create
   error, existing base command
} -script {

   # Create a procedure
   proc foo {} {}

   # Failed subcommand procedure creation
   proc-sub {foo bar} {} {}

} -error {can't create subcommand "foo bar": procedure "foo" already exists}

### ===========================================================================
### -test     : ::Simple::Subcommand::delete
test-case delete-1 {
   ::Simple::Subcommand::delete
} -setup {

   # Create some subcommand procedures
   namespace eval ::foo {
      proc-sub {bar subcommand-1} {} {}
      proc-sub {bar subcommand-2} {} {}
      proc-sub {bar subcommand-3} {} {}
   }

} -script {

   # Three subcommands
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Delete two subcommand procedures
   ::Simple::Subcommand::delete {::foo::bar subcommand-1}
   namespace eval ::foo {
      ::Simple::Subcommand::delete {::foo::bar subcommand-2}
   }

   # One subcommand
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]

   # Delete the last subcommand
   ::Simple::Subcommand::delete {::foo::bar subcommand-3}

   # Ensure the base command has been deleted
   ::foo::bar

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {subcommand-1 subcommand-2 subcommand-3
subcommand-3
} -error {invalid command name "::foo::bar"}

test-case delete-2 {
   ::Simple::Subcommand::delete
   error, invalid subcommand procedure name
} -script {

   # Failed subcommand procedure deletions
   catch {::Simple::Subcommand::delete {}} result
   puts $result
   catch {::Simple::Subcommand::delete foo} result
   puts $result
   catch {::Simple::Subcommand::delete {foo {}}} result
   puts $result
   ::Simple::Subcommand::delete {{} foo}

} -output {expected subcommand procedure name but got ""
expected subcommand procedure name but got "foo"
expected subcommand name but got ""
} -error {expected subcommand base command name but got ""}

test-case delete-3 {
   ::Simple::Subcommand::delete
   error, non-existing subcommand
} -script {

   # Failed subcommand procedure deletions
   catch {::Simple::Subcommand::delete {foo bar}} result
   puts $result
   catch {::Simple::Subcommand::delete {::foo::bar gee}} result
   puts $result
   namespace eval ::foo {
      ::Simple::Subcommand::delete {bar gee}
   }

} -output {can't delete "foo bar": command doesn't exist
can't delete "::foo::bar gee": command doesn't exist
} -error {can't delete "bar gee": command doesn't exist}

test-case delete-4 {
   ::Simple::Subcommand::delete
   error, unknown namespace
} -script {

   # Failed subcommand procedure deletion
   catch {::Simple::Subcommand::delete {::gee::foo bar}} result
   puts $result

   # Ensure the namespace ::gee has not been created
   lsearch -exact [namespace children] ::gee

} -output {can't delete "::gee::foo bar": command doesn't exist
} -return -1

### ===========================================================================
### -test     : ::Simple::Subcommand::move
test-case move-1 {
   ::Simple::Subcommand::move
} -setup {

   # Create some subcommand procedures
   namespace eval ::foo {
      proc-sub {bar subcommand-1} {arg1 arg2} {
         display-args arg1 arg2
      }
   }
   proc-sub {::foo::bar subcommand-2} {arg1 {arg2 999} args} {
      display-args arg1 arg2 args
   }

} -script {

   # Call the first subcommand procedure
   ::foo::bar subcommand-1 arg1 arg2

   # Two subcommands
   puts [::Simple::Subcommand::information subcommands ::foo::bar]

   # Move the first subcommand procedure to a different command
   ::Simple::Subcommand::move\
      {::foo::bar subcommand-1} {::bar::gee subcommand-A}

   # The first subcommand procedure no longer exists
   catch {::foo::bar subcommand-1 arg1 arg2} result
   puts $result

   # Call the target subcommand procedure
   ::bar::gee subcommand-A arg1 arg2

   # One subcommand
   puts [::Simple::Subcommand::information subcommands ::foo::bar]

   # One subcommand
   namespace eval ::bar {
      puts [::Simple::Subcommand::information subcommands gee]
   }

   # Move the last subcommand
   namespace eval ::foo {
      ::Simple::Subcommand::move {bar subcommand-2} {::bar::gee subcommand-B}
   }

   # Ensure the base command has been deleted
   catch ::foo::bar result
   puts $result

   # Call the target subcommand procedure
   ::bar::gee subcommand-B arg1

   # Two subcommands
   namespace eval ::bar {
      puts [::Simple::Subcommand::information subcommands gee]
   }

} -cleanup {

   # Delete the namespaces
   namespace delete ::foo
   namespace delete ::bar

} -output {{::foo::bar subcommand-1} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
subcommand-1 subcommand-2
bad option "subcommand-1": must be subcommand-2
{::bar::gee subcommand-A} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
subcommand-2
subcommand-A
invalid command name "::foo::bar"
{::bar::gee subcommand-B} arg1
arg1 = {arg1}
arg2 = {999}
args = {}
subcommand-A subcommand-B
}

test-case move-2 {
   ::Simple::Subcommand::move
   error, invalid subcommand procedure name
} -script {

   # Failed subcommand procedure movements
   catch {::Simple::Subcommand::move {} bar} result
   puts $result
   catch {::Simple::Subcommand::move foo bar} result
   puts $result
   catch {::Simple::Subcommand::move {foo {}} bar} result
   puts $result
   ::Simple::Subcommand::move {{} foo} bar

} -output {expected subcommand procedure name but got ""
expected subcommand procedure name but got "foo"
expected subcommand name but got ""
} -error {expected subcommand base command name but got ""}

test-case move-3 {
  ::Simple::Subcommand::move
    error, non-existing subcommand
} -setup {

   # Create a namespace
   namespace eval ::foo {}

} -script {

   # Failed subcommand procedure movements
   catch {::Simple::Subcommand::move {foo bar} {gee zuu}} result
   puts $result
   catch {::Simple::Subcommand::move {::foo::bar gee} {zuu fii}} result
   puts $result
   namespace eval ::foo {
      ::Simple::Subcommand::move {bar gee} {zuu fii}
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {can't rename "foo bar": command doesn't exist
can't rename "::foo::bar gee": command doesn't exist
} -error {can't rename "bar gee": command doesn't exist}

test-case move-4 {
   ::Simple::Subcommand::move
   error, unknown namespace
} -script {

   # Failed subcommand procedure movement
   catch {::Simple::Subcommand::move {::gee::foo bar} {zuu fii}} result
   puts $result

   # Ensure the namespace ::gee has not been created
   lsearch -exact [namespace children] ::gee

} -output {can't rename "::gee::foo bar": command doesn't exist
} -return -1

### ===========================================================================
### -test     : ::Simple::Subcommand::copy
test-case copy-1 {
   ::Simple::Subcommand::copy
} -setup {

   # Create some subcommand procedures
   namespace eval ::foo {
      proc-sub {bar subcommand-1} {arg1 arg2} {
         display-args arg1 arg2
      }
   }
   proc-sub {::foo::bar subcommand-2} {arg1 {arg2 999} args} {
      display-args arg1 arg2 args
   }

} -script {

   # Call the first subcommand procedure
   ::foo::bar subcommand-1 arg1 arg2

   # Two subcommands
   puts [::Simple::Subcommand::information subcommands ::foo::bar]

   # Copy the first subcommand procedure to a different command
   ::Simple::Subcommand::copy\
      {::foo::bar subcommand-1} {::bar::gee subcommand-A}

   # The first subcommand procedure still exists
   ::foo::bar subcommand-1 arg1 arg2

   # Call the target subcommand procedure
   ::bar::gee subcommand-A arg1 arg2

   # Two subcommands
   puts [::Simple::Subcommand::information subcommands ::foo::bar]

   # One subcommand
   namespace eval ::bar {
      puts [::Simple::Subcommand::information subcommands gee]
   }

   # Copy the second subcommand to its same command
   namespace eval ::foo {
      ::Simple::Subcommand::copy {bar subcommand-2} {bar subcommand-3}
   }

   # Call the target subcommand procedure
   ::foo::bar subcommand-3 arg1

   # Three subcommands
   namespace eval ::foo {
      puts [::Simple::Subcommand::information subcommands bar]
   }

} -cleanup {

   # Delete the namespaces
   namespace delete ::foo
   namespace delete ::bar

} -output {{::foo::bar subcommand-1} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
subcommand-1 subcommand-2
{::foo::bar subcommand-1} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
{::bar::gee subcommand-A} arg1 arg2
arg1 = {arg1}
arg2 = {arg2}
subcommand-1 subcommand-2
subcommand-A
{::foo::bar subcommand-3} arg1
arg1 = {arg1}
arg2 = {999}
args = {}
subcommand-1 subcommand-2 subcommand-3
}

test-case copy-2 {
   ::Simple::Subcommand::copy
   error, invalid subcommand procedure name
} -script {

   # Failed subcommand procedure copies
   catch {::Simple::Subcommand::copy {} bar} result
   puts $result
   catch {::Simple::Subcommand::copy foo bar} result
   puts $result
   catch {::Simple::Subcommand::copy {foo {}} bar} result
   puts $result
   ::Simple::Subcommand::copy {{} foo} bar

} -output {expected subcommand procedure name but got ""
expected subcommand procedure name but got "foo"
expected subcommand name but got ""
} -error {expected subcommand base command name but got ""}

test-case copy-3 {
  ::Simple::Subcommand::copy
    error, non-existing subcommand
} -setup {

   # Create a namespace
   namespace eval ::foo {}

} -script {

   # Failed subcommand procedure copies
   catch {::Simple::Subcommand::copy {foo bar} {gee zuu}} result
   puts $result
   catch {::Simple::Subcommand::copy {::foo::bar gee} {zuu fii}} result
   puts $result
   namespace eval ::foo {
      ::Simple::Subcommand::copy {bar gee} {zuu fii}
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output {can't copy "foo bar": command doesn't exist
can't copy "::foo::bar gee": command doesn't exist
} -error {can't copy "bar gee": command doesn't exist}

test-case copy-4 {
   ::Simple::Subcommand::copy
   error, unknown namespace
} -script {

   # Failed subcommand procedure copyment
   catch {::Simple::Subcommand::copy {::gee::foo bar} {zuu fii}} result
   puts $result

   # Ensure the namespace ::gee has not been created
   lsearch -exact [namespace children] ::gee

} -output {can't copy "::gee::foo bar": command doesn't exist
} -return -1

### ===========================================================================
### -test     : ::Simple::Subcommand::information exists
test-case exists-1 {
   ::Simple::Subcommand::information exists
} -setup {

   # Create some procedures
   proc bar {} {}

   # Create a namespace
   namespace eval ::foo {
      proc-sub {bar subcommand-1} {} {}
   }

} -script {

   # Non-base command procedures
   puts -nonewline\
      [::Simple::Subcommand::information exists ::Simple::Subcommand::create]
   puts -nonewline [::Simple::Subcommand::information exists ::foo]
   puts -nonewline [::Simple::Subcommand::information exists ::bar]
   puts -nonewline [::Simple::Subcommand::information exists ::bar::gee]
   puts -nonewline [::Simple::Subcommand::information exists ::foo::gee]

   # Base subcommand procedures
   puts -nonewline [::Simple::Subcommand::information exists ::foo::bar]
   namespace eval ::foo {
      puts -nonewline [::Simple::Subcommand::information exists bar]
   }

} -cleanup {

   # Delete the namespace
   namespace delete ::foo

} -output 0000011

### ===========================================================================
### -test     : ::Simple::Subcommand::information subcommands
test-case subcommands-1 {
   ::Simple::Subcommand::information subcommands
} -setup {

   # Create some procedures
   proc gee {} {}
   namespace eval ::foo {
      proc gee {} {}
      proc-sub {bar subcommand-1} {} {}
      proc-sub {bar subcommand-2} {} {}
      proc-sub {bar subcommand-3} {} {}
   }

} -script {

   # Base subcommand procedures
   puts [lsort [::Simple::Subcommand::information subcommands ::foo::bar]]
   namespace eval ::foo {
      puts [lsort [::Simple::Subcommand::information subcommands bar]]
   }

   # Non-base command procedures
   catch {::Simple::Subcommand::information subcommands gee} result
   puts $result
   catch {::Simple::Subcommand::information subcommands ::foo::gee} result
   puts $result
   namespace eval ::foo {
      catch {::Simple::Subcommand::information subcommands gee} result
      puts $result
   }

} -cleanup {

   # Delete the procedures
   rename gee {}

   # Delete the namespace
   namespace delete ::foo

} -output {subcommand-1 subcommand-2 subcommand-3
subcommand-1 subcommand-2 subcommand-3
"gee" isn't a subcommand base command procedure
"::foo::gee" isn't a subcommand base command procedure
"gee" isn't a subcommand base command procedure
}

### ===========================================================================
### -test     : ::Simple::Subcommand::configure
test-case configure-1 {
   ::Simple::Subcommand::configure
} -script {
   ::Simple::Subcommand::configure
} -error {this package has no options}

### ===========================================================================
### -test     : ::Simple::Subcommand::cget
test-case cget-1 {
   ::Simple::Subcommand::cget
} -script {
   ::Simple::Subcommand::cget
} -return {}

} ;# End of regression testing section
