### ===========================================================================
### -package  : SimpleType
# -version    : 0.2
# -purpose    : Types handling.
# -overview   :
#     This package allows to handle variable types.  Procedures are provided
#  to declare, delete and modify types, get information about a type and
#  obtain the list of types, as well as several others to convert between
#  different types.  The key procedure of this package is
#  K<::Simple::Type::is>, which returns whether a value conforms to a type.
#
# -usage      :
#     Types are declared via K<::Simple::Type::declare> and can be deleted via
#  K<::Simple::Type::delete> or modified via K<::Simple::Type::modify>.  The
#  K<::Simple::Type::information> procedure provides several subcommands to
#  query information about types:  K<exists>, K<declared>, K<description> and
#  K<matchingscript>.  Refer to the description of each procedure below for
#  further information.
#
#     The K<::Simple::Type::convert> set of procedures convert a value between
#  different types.  The K<::Simple::Type::is> procedure returns whether a
#  value conforms to a type.
#
# -keywords   : type decimal hexadecimal binary octal base convert
# -commands   :
#
#  * K<::Simple::Type::declare> type description ?matchingScript?
#    Declares a type.
#
#  * K<::Simple::Type::modify description> type description
#    Modifies a type description.
#
#  * K<::Simple::Type::modify matchingscript> type matchingScript
#    Modifies a type matching script.
#
#  * K<::Simple::Type::delete> type
#    Deletes a type.
#
#  * K<::Simple::Type::information declared> ?typePattern?
#    Returns the list of declared types.
#
#  * K<::Simple::Type::information exists> type
#    Returns whether a type exists.
#
#  * K<::Simple::Type::information description> type
#    Returns a type description.
#
#  * K<::Simple::Type::information matchingscript> type
#    Returns a type matching script.
#
#  * K<::Simple::Type::is> type value ?extra?
#    Returns whether a value conforms to a type.
#
#  * K<::Simple::Type::convert type> type
#    Converts a type to an unqualified name.
#
#  * K<::Simple::Type::convert flag> flag
#    Converts a flag to an unqualified name.
#
#  * K<::Simple::Type::convert optional> optional
#    Converts an optional value to an unqualified name.
#
#  * K<::Simple::Type::convert boolean> booleanExpr
#    Converts an extended boolean to its equivalent boolean value.
#
#  * K<::Simple::Type::convert dec> number
#    Converts an integer in any base to decimal.
#
#  * K<::Simple::Type::convert hex> number
#    Converts an integer in any base to hexadecimal.
#
#  * K<::Simple::Type::convert bin> number
#    Converts an integer in any base to binary.
#
#  * K<::Simple::Type::convert octal> number
#    Converts an integer in any base to octal.
#
#  * K<::Simple::Type::configure>
#    Configures the package options.
#
#  * K<::Simple::Type::cget>
#    Gets the package options.
#
# -variables  :
#  { Description      -array         {Array containing for each type its
#                                     description}}
#
# -examples   :
#
#  # Install the package
#  package require SimplePackage
#  ::Simple::Package::require-and-install SimpleType
#  
#  # Declare a type -bar which can only hold either bar or BAR
#  ::Simple::Type::declare -bar {can only hold either bar or BAR}\
#     {if {![string match $value bar] && ![string match $value BAR]}\
#     {error {}}}
#  
#  # Assert the type -bar no exists
#  # This displays the following:
#  #    The type -bar exists
#  if {[::Simple::Type::information exists -bar]} {
#     puts {The type -bar exists}
#  }
#  
#  # Get some information about the type -bar
#  # This displays the following:
#  #    The type -bar can only hold either bar or BAR
#  puts "The type -bar [::Simple::Type::information description -bar]"
#  
#  # Check the allowed contents for a variable of type -bar
#  # This displays the following:
#  #    "bar" is of type -bar
#  #    "foo" is NOT of type -bar
#  #    "BAR" is of type -bar
#  #    "bAR" is NOT of type -bar
#  foreach contents {bar foo BAR bAR} {
#     puts -nonewline "\"$contents\" is "
#     if {![::Simple::Type::is -bar $contents]} {
#        puts -nonewline {NOT }
#     }
#     puts {of type -bar}
#  }
#  
#  # Get rid of the type -bar
#  ::Simple::Type::delete -bar
#  
#  # Assert the type -bar no longer exists
#  # This displays the following:
#  #    The type -bar no longer exists
#  if {![::Simple::Type::information exists -bar]} {
#     puts {The type -bar no longer exists}
#  }
#  
#  # More fun. A compose type made of three elements:
#  # an integer, a boolean and an alphabetic
#  ::Simple::Type::declare -foo {int + boolean + alpha} {
#     foreach theType [list -int -boolean -alpha] theValue $value {
#        if {![::Simple::Type::is $theType $theValue]} {
#           error {}
#        } 
#     }
#  }
#  
#  # Check the allowed contents for a variable of type -foo
#  # This displays the following:
#  #    "2 1 bar" is of type -foo
#  #    "2 x bar" is NOT of type -foo
#  foreach contents {{2 1 bar} {2 x bar}} {
#     puts -nonewline "\"$contents\" is "
#     if {![::Simple::Type::is -foo $contents]} {
#        puts -nonewline {NOT }
#     }
#        puts {of type -foo}
#  }
#
# -todo       :
#  * It would be nice to have a procedure which guesses the type of a given
#    value.
#  * The basic name regular expression "[-_a-zA-Z0-9%]+" should be a package
#    option.  All type matching scipts requiring it should reference the
#    package option.  Alternatively, for maximum performance, upon changing
#    this option, all types matching scripts using it should be recreated.
#
# -history    :
#  19-feb-1999   Unreleased first version 0.1
#  23-apr-2000   First public release, version 0.2
#
# -copyright  :
#  Copyright (C) 1999, 2000, Juan C. Gil (jgil@gmv.es)
#
### ===========================================================================

if {[string compare $::SIMPLE(PackageMode,SimpleType) test]} {

###
### PACKAGE DEFINITION SECTION
###

### Declare the package
package require SimplePackage
::Simple::Package::declare SimpleType 0.2 -required {
   SimpleError SimplePackage SimpleSubcommand
} -namespaces {
   ::Simple::Type
   ::Simple::Type::Priv
} -firsttimeinstall {

### Declare the package errors
::Simple::Error::declare ::Simple::Type::NON-EXISTING-TYPE {
   non-existing type "%s"
} {
   Type <type> does not exist
} {
   Use K<::Simple::Type::declare> to declare the type
}

::Simple::Error::declare ::Simple::Type::NEGATIVE-INT {
   negative integer "%s"
} {
   The integer <integer> is negative but a positive integer was expected.
}

### ===========================================================================
### -command  : ::Simple::Type::declare
# -purpose    : Declares a type.
# -overview   :
#     This procedure is used to declare a new type.
#
#     A type is made of three elements:  the type name, the type description
#  and the type matching script.  The type name must start with an hyphen such
#  as I<-int>.  The type matching script is a Tcl script used to check whether
#  a value conforms to the type.  It shall assume that the value to be checked
#  is passed in the "value" variable and shall raise an error upon invalid
#  values.  A type with an empty matching script may hold any value.
#
# -arguments  :
#  { type             -type          {Type}}
#  { description      -string        {Description}}
#  {?matchingScript?  -script  {}    {Matching script}}
#
# -examples   :
#
#     # Declare a type -foo which may hold any value
#     ::Simple::Type::declare -foo {Can hold any value}
#
#     # Declare a type -bar which can only hold either bar or BAR
#     ::Simple::Type::declare -bar {Can only hold either bar or BAR}\
#        {if {![string match $value bar] && ![string match $value BAR]}\
#        {error {}}}
#
# -effects    :
#  * Creates the type matching procedure named
#    K<::Simple::Type::Priv::%MATCHING%-<type>>.
#
proc ::Simple::Type::declare {
   type
   description
   {matchingScript {}}
} {
   ### Assert the type name is valid
   ::Simple::Type::Priv::assert-valid-type $type

   ### Store the type description and matching script
   set ::Simple::Type::Description($type) $description

   ### Create the matching procedure
   proc ::Simple::Type::Priv::%MATCHING%-$type {value {extra {}}}\
      $matchingScript
}

### ===========================================================================
### -command  : ::Simple::Type::Priv::assert-valid-type
# -purpose    : Throws an error if type name is not valid.
# -arguments  :
#  { type             -type          {Type}}
#
proc ::Simple::Type::Priv::assert-valid-type {
   type
} {
   # This condition should be
   #    if {![::Simple::Type::is -type $type]}
   # but the '-type' type itself may not be declared yet ...
   if {![regexp {^-[-_a-zA-Z0-9%]+$} $type]} {
      ::Simple::Error::throw ::Simple::BAD-ITEM {type name} $type
   }
}

### Declare the Simple library types
# fixme: many of the below matching scripts should be reimplemented using
#        K<string is> as present in Tcl 8.3

# Numbers
::Simple::Type::declare -int       {Integer number}\
   {if {![regexp {^[-+]?(0|[1-9][0-9]*)$} $value]} {error {}}}

::Simple::Type::declare -float     {Floating point number}\
   {if {![regexp\
   {^[-+]?(0|[1-9][0-9]*|[0-9]*\.[0-9]+|[0-9]+\.[0-9]*)([eE][-+]?[0-9]+)?$}\
   $value]} {error {}}}

::Simple::Type::declare -dec       {Decimal number}\
   {if {![regexp {^(0|[1-9][0-9]*)$} $value]} {error {}}}

::Simple::Type::declare -hex       {Hexadecimal number}\
   {if {![regexp {^(0[xX][0-9a-fA-F]+)$} $value]} {error {}}}

::Simple::Type::declare -bin       {Binary number}\
   {if {![regexp {^(0[bB][01]+)$} $value]} {error {}}}

::Simple::Type::declare -octal     {Octal number}\
   {if {![regexp {^0[0-7]*$} $value]} {error {}}}

::Simple::Type::declare -anybase\
   {Decimal, hexadecimal, binary or octal number}\
   {if {![regexp\
   {^((0|[1-9][0-9]*)|(0[xX][0-9a-fA-F]+)|(0[bB][01]+)|0[0-7]*)$}\
   $value]} {error {}}}

::Simple::Type::declare -number    {Any valid number} {expr {0 + $value}}

# Alphabetic and alphanumeric
::Simple::Type::declare -alpha     Alphabetic\
   {if {![regexp {^[a-zA-Z]+$} $value]} {error {}}}

::Simple::Type::declare -alphanum  Alphanumeric\
   {if {![regexp {^[a-zA-Z0-9]+$} $value]} {error {}}}

# Boolean
::Simple::Type::declare -boolean   {Boolean value}\
   {if {![regexp ^(0|1)$ $value]} {error {}}}

# Tcl/Tk specific
::Simple::Type::declare -extbool   {Tcl extended boolean}\
   {if {![regexp ^(0|1)$ [{::Simple::Type::convert boolean} $value]]}\
   {error {}}}

::Simple::Type::declare -word      {Tcl word}\
   {if {[string wordend $value 0] != [string length $value]} {error {}}}

::Simple::Type::declare -string    {Tcl string} {}

::Simple::Type::declare -script    {Tcl script} {llength $value}

::Simple::Type::declare -expr      {Tcl expression} {expr $value}

::Simple::Type::declare -list      {Tcl list} {llength $value}

::Simple::Type::declare -channel   {Open Tcl channel} {tell $value}

::Simple::Type::declare -namespace {Existing Tcl namespace}\
   {if {[string compare $value {}]} {namespace parent $value}}

::Simple::Type::declare -namespacename {Tcl namespace name}\
   {if {![regexp {^(::|[-_a-zA-Z0-9%]+)+$} $value]} {error {}}}

::Simple::Type::declare -qualifiedname {Tcl qualified name}\
   {if {![regexp {^(::([-_a-zA-Z0-9%]+)?|(::[-_a-zA-Z0-9%]+)+)$} $value]}\
   {error {}}}

::Simple::Type::declare -unqualifiedname {Tcl unqualified name}\
   {if {![regexp {^([-_a-zA-Z0-9%]+|)$} $value]} {error {}}}

::Simple::Type::declare -name      {Tcl name}\
   {if {![regexp {^(((::)?[-_a-zA-Z0-9%]+)+|)$} $value]} {error {}}}

::Simple::Type::declare -array     {Tcl array}\
   {if {![uplevel 2 [list array exists $value]]} {error {}}}

::Simple::Type::declare -pattern   {Tcl pattern} {string match $value {}}

::Simple::Type::declare -regexp    {Tcl regular expression}\
   {regexp $value {}}

::Simple::Type::declare -version   {Tcl package version}\
   {package vcompare 0 $value}

::Simple::Type::declare -widgetname {Tk widget name}\
   {if {![regexp {^(\.|(\.[-_a-zA-Z0-9%]+)+)$} $value]} {error {}}}

# Simple library types
::Simple::Type::declare -type      {Simple library data Type}\
   {if {![regexp {^-[-_a-zA-Z0-9%]+$} $value]} {error {}}}

::Simple::Type::declare -flag      {Simple library flag}\
   {if {![regexp {^-[-_a-zA-Z0-9%]+$} $value]} {error {}}}

::Simple::Type::declare -optional  {Simple library optional variable}\
   {if {![regexp {^\?([-_a-zA-Z0-9%]+|)\?$} $value]} {error {}}}

# Other types
::Simple::Type::declare -date      Date {clock scan $value}

# Special types
::Simple::Type::declare -choice    Choice\
   {if {[lsearch -exact $extra $value] == -1} {error {}}}

::Simple::Type::declare -any       {Any value} {}

::Simple::Type::declare -boolflag  {Boolean flag} {}

### ===========================================================================
### -command  : ::Simple::Type::modify description
# -purpose    : Modifies a type description.
# -arguments  :
#  { type             -type          {Type}}
#  { description      -string        {New escription}}
#
proc-sub {::Simple::Type::modify description} {
   type
   description
} {
   ### Type exists
   if {[info exist ::Simple::Type::Description($type)]} {

      ### Set the type description
      set ::Simple::Type::Description($type) $description

   ### Type does not exist
   } else {

      ### Assert the type name is valid
      ::Simple::Type::Priv::assert-valid-type $type

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type
   }
}

### ===========================================================================
### -command  : ::Simple::Type::modify matchingscript
# -purpose    : Modifies a type matching script.
# -arguments  :
#  { type             -type          {Type}}
#  { matchingScript   -script        {New matching script}}
#
# -effects    :
#  * Recreates the type matching procedure named
#    K<::Simple::Type::Priv::%MATCHING%-<type>>
#
proc-sub {::Simple::Type::modify matchingscript} {
   type
   matchingScript
} {
   ### Type exists
   if {[info exist ::Simple::Type::Description($type)]} {

      ### Recreate the matching procedure
      proc ::Simple::Type::Priv::%MATCHING%-$type {value {extra {}}}\
         $matchingScript

   ### Type does not exist
   } else {

      ### Assert the type name is valid
      ::Simple::Type::Priv::assert-valid-type $type

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type
   }
}

### ===========================================================================
### -command  : ::Simple::Type::delete
# -purpose    : Deletes a type.
# -arguments  :
#  { type             -type          {Type}}
#
# -effects    :
#  * Deletes the type matching procedure
#    K<::Simple::Type::Priv::%MATCHING%-<type>>
#
proc ::Simple::Type::delete {
   type
} {
   ### Type exists
   if {[info exist ::Simple::Type::Description($type)]} {

      ### Delete the type description and matching procedure
      unset ::Simple::Type::Description($type)
      rename ::Simple::Type::Priv::%MATCHING%-$type {}

   ### Type does not exist
   } else {

      ### Assert the type name is valid
      ::Simple::Type::Priv::assert-valid-type $type

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type
   }
}

### ===========================================================================
### -command  : ::Simple::Type::information declared
# -purpose    : Returns the list of declared types.
# -overview   :
#     This procedure returns the list of declared types matching the given
#  pattern.  This includes native Simple library types as well as others
#  declared via the P<::Simple::Type::declare> procedure.
#
# -arguments  :
#  {?typePattern?     -pattern {}    {Pattern}}
#
# -returns    : The list of types matching the given pattern.
# -remarks    :
#  * Use no pattern to obtain the complete list of types.
#
proc-sub {::Simple::Type::information declared} {
   {typePattern {}}
} {
   if {[string compare $typePattern {}]} {
      array names ::Simple::Type::Description $typePattern
   } else {
      array names ::Simple::Type::Description
   }
}

### ===========================================================================
### -command  : ::Simple::Type::information exists
# -purpose    : Returns whether a type exists.
# -overview   :
#     This procedure returns whether a type exists.  This includes native
#  Simple library types as well as others declared via the
#  P<::Simple::Type::declare> procedure.
#
# -arguments  :
#  { type             -type          {Type}}
#
# -returns    : Whether the type exists.
#
proc-sub {::Simple::Type::information exists} {
   type
} {
   ### Assert the type name is valid
   ::Simple::Type::Priv::assert-valid-type $type

   ### Return whether the type exists
   info exist ::Simple::Type::Description($type)
}

### ===========================================================================
### -command  : ::Simple::Type::information description
# -purpose    : Returns a type description.
# -arguments  :
#  { type             -type          {Type}}
#
# -returns    : The type description.
#
proc-sub {::Simple::Type::information description} {
   type
} {
   ### Type exists
   if {[info exist ::Simple::Type::Description($type)]} {

      ### Return the type description
      set ::Simple::Type::Description($type)

   ### Type does not exist
   } else {

      ### Assert the type name is valid
      ::Simple::Type::Priv::assert-valid-type $type

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type
   }
}

### ===========================================================================
### -command  : ::Simple::Type::information matchingscript
# -purpose    : Returns a type matching script.
# -arguments  :
#  { type             -type          {Type}}
#
# -returns    : The type matching script.
#
proc-sub {::Simple::Type::information matchingscript} {
   type
} {
   ### Type exists
   if {[info exist ::Simple::Type::Description($type)]} {

      ### Return the type matching script
      info body ::Simple::Type::Priv::%MATCHING%-$type

   ### Type does not exist
   } else {

      ### Assert the type name is valid
      ::Simple::Type::Priv::assert-valid-type $type

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type
   }
}

### ===========================================================================
### -command  : ::Simple::Type::is
# -purpose    : Returns whether a value conforms to a type.
# -keywords   : type
# -arguments  :
#  { type             -type          {Type}}
#  { value            -any           {Value}}
#  {?extra?           -any {}        {Extra data required to check the
#                                     conformation}}
#
# -returns    : Whether the value conforms to the type.
# -example    :
#
proc ::Simple::Type::is {
   type
   value
   {extra {}}
} {
   ### Type does not exist
   if {![info exist ::Simple::Type::Description($type)]} {

      ### Assert the type name is valid
      ::Simple::Type::Priv::assert-valid-type $type

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NON-EXISTING-TYPE $type

   }

   ### Return whether the value conforms to the type
   expr {![catch {::Simple::Type::Priv::%MATCHING%-$type $value $extra}]}
}

### ===========================================================================
### -command  : ::Simple::Type::convert type
# -purpose    : Converts a type to an unqualified name.
# -arguments  :
#  { type             -type          {Type}}
#
# -returns    : The type name.
#
proc-sub {::Simple::Type::convert type} {
   type
} {
   ### Assert the type name is valid
   ::Simple::Type::Priv::assert-valid-type $type

   ### Return the type name
   string range $type 1 end
}

### ===========================================================================
### -command  : ::Simple::Type::convert flag
# -purpose    : Converts a flag to an unqualified name.
# -arguments  :
#  { flag             -flag          {Flag}}
#
# -returns    : The flag name.
#
proc-sub {::Simple::Type::convert flag} {
   flag
} {
   ### Assert the flag name is valid
   if {![::Simple::Type::is -flag $flag]} {
      ::Simple::Error::throw ::Simple::BAD-ITEM {flag name} $flag
   }

   ### Return the flag name
   string range $flag 1 end
}

### ===========================================================================
### -command  : ::Simple::Type::convert optional
# -purpose    : Converts an optional value to an unqualified name.
# -arguments  :
#  { optional         -optional      {Optional value}}
#
# -returns    : The optional value name.
#
proc-sub {::Simple::Type::convert optional} {
   optional
} {
   ### Assert the optional value name is valid
   if {![::Simple::Type::is -optional $optional]} {
      ::Simple::Error::throw ::Simple::BAD-ITEM {optional value name} $optional
   }

   ### Return the optional value name
   string trim $optional ?
}

### ===========================================================================
### -command  : ::Simple::Type::convert boolean
# -purpose    : Converts an extended boolean to its equivalent boolean value.
# -overview   :
#     Use this procedure to convert from extended booleans (true/false or
#  on/off) to 1/0 because the Tcl K<expr> and K<if> commands can't use
#  non-numeric string as operand of "!" (sic).
#
# -keywords   : boolean convert
# -arguments  :
#  { booleanExpr      -extbool       {Boolean expression}}
#
# -returns    : The converted boolean.
# -examples   :
#
#     # A robust negation test
#     if {![::Simple::Type::convert boolean $extbool]} {
#        puts "$extbool is false"
#     }
#
proc-sub {::Simple::Type::convert boolean} {
   booleanExpr
} {
   if {$booleanExpr} {
      return 1
   } else {
      return 0
   }
}

### ===========================================================================
### -command  : ::Simple::Type::Priv::bin-to-dec
# -purpose    : Converts a binary integer to decimal.
# -arguments  :
#  { binary           -bin           {Binary integer}}
#
# -returns    : The converted integer.
#
proc ::Simple::Type::Priv::bin-to-dec {
   binary
} {
   ### Get rid of the prefix
   set binary [string range $binary 2 end]

   ### Compute the base
   set base [expr {1 << [string length $binary]}]

   ### Loop over binary digits
   # fixme: this should be reimplemented using
   #        K<binary scan> as present in Tcl 8.3
   set answer 0
   foreach digit [split $binary {}] {

      ### Reduce the base
      set base [expr {$base / 2}]

      ### Binary digit set
      if {$digit == 1} {
         incr answer $base
      }
   }

   ### Return the converted integer
   set answer
}

### ===========================================================================
### -command  : ::Simple::Type::convert dec
# -purpose    : Converts an integer in any base to decimal.
# -overview   :
#     Converts a decimal, hexadecimal, binary or octal integer to decimal base
#
# -keywords   : decimal hexadecimal binary octal base convert
# -arguments  :
#  { integer          -anybase       {Positive integer}}
#
# -returns    : The converted integer.
#
proc-sub {::Simple::Type::convert dec} {
   integer
} {
   ### Binary integer
   if {[::Simple::Type::is -bin $integer]} {

      ### Convert the binary integer
      set answer [::Simple::Type::Priv::bin-to-dec $integer]

   ### Decimal, hexadecimal or octal integer
   } else {
 
      ### Convert the integer
      set answer [format %d $integer]
   }

   ### Negative integer
   if {$answer < 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NEGATIVE-INT $integer
   }

   ### Return the converted integer
   set answer
}

### ===========================================================================
### -command  : ::Simple::Type::convert hex
# -purpose    : Converts an integer in any base to hexadecimal.
# -overview   :
#     Converts a decimal, hexadecimal, binary or octal integer to hexadecimal
#  base.  The resulting hexadecimal constant is of the form 0x<hex-digits>
#  where <hex-digits> are B<uppercase> hexadecimal digits, that is, the "0x"
#  prefix contains a B<lowercase> "x" but the actual digits are uppercase. 
#  This is different from the result of the "%#X" P<format> specifier which
#  returns an uppercase "X".
#
# -keywords   : decimal hexadecimal binary octal base convert
# -arguments  :
#  { integer          -anybase       {Positive integer}}
#
# -returns    : The converted integer.
#
proc-sub {::Simple::Type::convert hex} {
   integer
} {
   ### Binary integer
   set originalInteger $integer
   if {[::Simple::Type::is -bin $integer]} {

      ### Convert the binary integer
      set integer [::Simple::Type::Priv::bin-to-dec $integer]
   }

   ### Negative integer
   if {$integer < 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NEGATIVE-INT $originalInteger
   }

   ### Return the converted integer
   format 0x%X $integer
}

### ===========================================================================
### -command  : ::Simple::Type::convert bin
# -purpose    : Converts an integer in any base to binary.
# -overview   :
#     Converts a decimal, hexadecimal, binary or octal integer to binary base.
#  The resulting binary constant is of the form 0b<bin-digits>.
#
# -keywords   : decimal hexadecimal binary octal base convert
# -arguments  :
#  { integer          -anybase       {Positive integer}}
#
# -returns    : The converted integer.
#
proc-sub {::Simple::Type::convert bin} {
   integer
} {
   ### Binary integer
   set originalInteger $integer
   if {[::Simple::Type::is -bin $integer]} {

      ### Convert the binary integer to decimal
      set integer [::Simple::Type::Priv::bin-to-dec $integer]

   ### Decimal, hexadecimal or octal integer
   } else {

      ### Convert the integer to decimal
      set integer [format %d $integer]
   }

   ### Negative integer
   if {$integer < 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NEGATIVE-INT $originalInteger
   }

   ### Zero integer
   if {$integer == 0} {

      ### Binary digit not set
      set answer 0

   ### Greater than zero integer
   } else {

      ### Loop over binary digits
      # fixme: this should be reimplemented using
      #        K<binary format> as present in Tcl 8.3
      set answer {}
      while {$integer > 0} {

         ### Binary digit set
         if {[expr {$integer % 2}]} {
            set answer 1$answer

         ### Binary digit not set
         } else {
            set answer 0$answer
         }

         ### Reduce the integer
         set integer [expr {$integer / 2}]
      }
   }

   ### Return the converted integer
   set answer 0b$answer
}

### ===========================================================================
### -command  : ::Simple::Type::convert octal
# -purpose    : Converts an integer in any base to octal.
# -overview   :
#     Converts a decimal, hexadecimal, binary or octal integer to octal base.
#
# -keywords   : decimal hexadecimal binary octal base convert
# -arguments  :
#  { integer          -anybase       {Positive integer}}
#
# -returns    : The converted integer.
#
proc-sub {::Simple::Type::convert octal} {
   integer
} {
   ### Binary integer
   set originalInteger $integer
   if {[::Simple::Type::is -bin $integer]} {

      ### Convert the binary integer
      set integer [::Simple::Type::Priv::bin-to-dec $integer]
   }

   ### Negative integer
   if {$integer < 0} {

      ### Throw error
      ::Simple::Error::throw ::Simple::Type::NEGATIVE-INT $originalInteger
   }

   ### Return the converted integer
   format 0%o $integer
}

### ===========================================================================
### -command  : ::Simple::Type::configure
# -purpose    : Configures the package options.
#
proc ::Simple::Type::configure {
} {
   ::Simple::Error::throw ::Simple::NO-OPTIONS
}

### ===========================================================================
### -command  : ::Simple::Type::cget
# -purpose    : Gets the package options.
# -returns    : The requested option value or the whole list of options if none
#  specified.
#
proc ::Simple::Type::cget {
} {
   list
}

} ;# End of package declaration

} else { ;# End of package definition section

###
### REGRESSION TESTING SECTION
###

### Provide the package
package provide SimpleType-test 0.2

### ===========================================================================
### -test     : ::Simple::Type::declare
test-case declare-1 {
   ::Simple::Type::declare
} -script {

   # Type does not exist
   puts -nonewline [::Simple::Type::information exists -foo]

   # Declare a type
   ::Simple::Type::declare -foo {Can hold any value}

   # Type exists
   puts -nonewline [::Simple::Type::information exists -foo]

} -cleanup {

   # Delete the type
   ::Simple::Type::delete -foo

} -output 01

test-case declare-2 {
   ::Simple::Type::declare
   with matching script
} -script {

   # Type does not exist
   puts -nonewline [::Simple::Type::information exists -bar]

   # Declare a type
   ::Simple::Type::declare -bar {Can only hold either bar or BAR} {}

   # Type exists
   puts -nonewline [::Simple::Type::information exists -bar]

} -cleanup {

   # Delete the type
   ::Simple::Type::delete -bar

} -output 01

test-case declare-3 {
   ::Simple::Type::declare
   error, invalid type name
} -script {
   ::Simple::Type::declare foo {Invalid type name}
} -error {expected type name but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::modify description
test-case modify-descriptn-1 {
   ::Simple::Type::modify description
} -setup {

   # Declare a type
   ::Simple::Type::declare -foo {Original description}

} -script {

   # Get the type description
   puts [::Simple::Type::information description -foo]

   # Modify the type description
   ::Simple::Type::modify description -foo {Modified description}

   # Get the type description
   puts [::Simple::Type::information description -foo]

} -cleanup {

   # Delete the type
   ::Simple::Type::delete -foo

} -output {Original description
Modified description
}

test-case modify-descriptn-2 {
   ::Simple::Type::modify description
   error, invalid type name
} -script {
   ::Simple::Type::modify description foo {}
} -error {expected type name but got "foo"}

test-case modify-descriptn-3 {
   ::Simple::Type::modify description
   error, non-existing type
} -script {
   ::Simple::Type::modify description -foo {}
} -error {non-existing type "-foo"}

### ===========================================================================
### -test     : ::Simple::Type::modify matchingscript
test-case modify-matching-1 {
   ::Simple::Type::modify matchingscript
} -setup {

   # Declare a type
   ::Simple::Type::declare -foo {} {Original matchingscript}

} -script {

   # Get the type matching script
   puts [::Simple::Type::information matchingscript -foo]

   # Modify the type matching script
   ::Simple::Type::modify matchingscript -foo {Modified matchingscript}

   # Get the type matching script
   puts [::Simple::Type::information matchingscript -foo]

} -cleanup {

   # Delete the type
   ::Simple::Type::delete -foo

} -output {Original matchingscript
Modified matchingscript
}

test-case modify-matching-2 {
   ::Simple::Type::modify matchingscript
   error, invalid type name
} -script {
   ::Simple::Type::modify matchingscript foo {}
} -error {expected type name but got "foo"}

test-case modify-matching-3 {
   ::Simple::Type::modify matchingscript
   error, non-existing type
} -script {
   ::Simple::Type::modify matchingscript -foo {}
} -error {non-existing type "-foo"}

### ===========================================================================
### -test     : ::Simple::Type::delete
test-case delete-1 {
   ::Simple::Type::modify delete
} -setup {

   # Declare a type
   ::Simple::Type::declare -foo Description

} -script {

   # Get the type description
   puts [::Simple::Type::information description -foo]

   # Delete the type
   ::Simple::Type::delete -foo

   # Get the type description
   ::Simple::Type::information description -foo

} -output {Description
} -error {non-existing type "-foo"}

test-case delete-2 {
   ::Simple::Type::delete
   error, invalid type name
} -script {
   ::Simple::Type::delete foo
} -error {expected type name but got "foo"}

test-case delete-3 {
   ::Simple::Type::delete
   error, non-existing type
} -script {
   ::Simple::Type::delete foo
} -error {expected type name but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::information declared
test-case declared-1 {
   ::Simple::Type::information declared
} -script {

   # Number of types
   set nTypes [llength [::Simple::Type::information declared]]

   # Declare a type
   ::Simple::Type::declare -foo1 Description

   # Number of types
   puts [expr {[llength [::Simple::Type::information declared]] - $nTypes}]

   # Declare another type
   ::Simple::Type::declare -foo2 Description

   # Number of types matching -foo*
   lsort [::Simple::Type::information declared -foo*]

} -cleanup {

   # Delete the types
   ::Simple::Type::delete -foo1
   ::Simple::Type::delete -foo2

} -output {1
} -return {-foo1 -foo2}

### ===========================================================================
### -test     : ::Simple::Type::information exists
test-case exists-1 {
   ::Simple::Type::information exists
} -setup {

   # Declare a type
   ::Simple::Type::declare -foo Description

} -script {

   # Existing types
   puts -nonewline [::Simple::Type::information exists -int]
   puts -nonewline [::Simple::Type::information exists -channel]
   puts -nonewline [::Simple::Type::information exists -foo]

   # Delete the type
   ::Simple::Type::delete -foo

   # Non-existing types
   puts -nonewline [::Simple::Type::information exists -bar]
   puts -nonewline [::Simple::Type::information exists -gee]
   puts -nonewline [::Simple::Type::information exists -foo]

} -output 111000

test-case exists-2 {
   ::Simple::Type::information exists
   error, invalid type name
} -script {
   ::Simple::Type::information exists foo
} -error {expected type name but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::information description
test-case description-1 {
   ::Simple::Type::information description
} -setup {

   # Declare a type
   ::Simple::Type::declare -foo {Original description}

} -script {

   # Get the type description
   puts [::Simple::Type::information description -foo]

   # Modify the type description
   ::Simple::Type::modify description -foo {Modified description}

   # Get the type description
   puts [::Simple::Type::information description -foo]

} -cleanup {

   # Delete the type
   ::Simple::Type::delete -foo

} -output {Original description
Modified description
}

test-case description-2 {
   ::Simple::Type::information description
   error, invalid type name
} -script {
   ::Simple::Type::information description foo
} -error {expected type name but got "foo"}

test-case description-3 {
   ::Simple::Type::information description
   error, non-existing type
} -script {
   ::Simple::Type::information description -foo
} -error {non-existing type "-foo"}

### ===========================================================================
### -test     : ::Simple::Type::information matchingscript
test-case matchingscript-1 {
   ::Simple::Type::information matchingscript
} -setup {

   # Declare a type
   ::Simple::Type::declare -foo {} {Original matchingscript}

} -script {

   # Get the type matching script
   puts [::Simple::Type::information matchingscript -foo]

   # Modify the type matching script
   ::Simple::Type::modify matchingscript -foo {Modified matchingscript}

   # Get the type matching script
   puts [::Simple::Type::information matchingscript -foo]

} -cleanup {

   # Delete the type
   ::Simple::Type::delete -foo

} -output {Original matchingscript
Modified matchingscript
}

test-case matchingscript-2 {
   ::Simple::Type::information matchingscript
   error, invalid type name
} -script {
   ::Simple::Type::information matchingscript foo
} -error {expected type name but got "foo"}

test-case matchingscript-3 {
   ::Simple::Type::information matchingscript
   error, non-existing type
} -script {
   ::Simple::Type::information matchingscript -foo
} -error {non-existing type "-foo"}

### ===========================================================================
### -test     : is
test-case is-1 {
   is
} -setup {

   # Declare a couple of types
   ::Simple::Type::declare -foo {Can hold any value}
   ::Simple::Type::declare -bar {Can only hold either bar or BAR}\
      {if {![string match $value bar] && ![string match $value BAR]} {error {}}}

} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -foo foo123bar%-!]
   puts -nonewline [::Simple::Type::is -bar BAR]

   # No matches
   puts -nonewline [::Simple::Type::is -bar FOO]
   puts -nonewline [::Simple::Type::is -bar foobar]

} -cleanup {

   # Delete the types
   ::Simple::Type::delete -foo
   ::Simple::Type::delete -bar

} -output 1100

test-case is-2 {
   is
   -int
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -int 1]
   puts -nonewline [::Simple::Type::is -int +123]
   puts -nonewline [::Simple::Type::is -int 0]
   puts -nonewline [::Simple::Type::is -int -0]
   puts -nonewline [::Simple::Type::is -int -123]

   # No matches
   puts -nonewline [::Simple::Type::is -int 01]
   puts -nonewline [::Simple::Type::is -int 1-]
   puts -nonewline [::Simple::Type::is -int w23]

} -output 11111000

test-case is-3 {
   is
   -float
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -float 1]
   puts -nonewline [::Simple::Type::is -float +123]
   puts -nonewline [::Simple::Type::is -float 0]
   puts -nonewline [::Simple::Type::is -float -0]
   puts -nonewline [::Simple::Type::is -float -123]
   puts -nonewline [::Simple::Type::is -float .1]
   puts -nonewline [::Simple::Type::is -float 1.]
   puts -nonewline [::Simple::Type::is -float 0.1]
   puts -nonewline [::Simple::Type::is -float -.1]
   puts -nonewline [::Simple::Type::is -float -0.1]
   puts -nonewline [::Simple::Type::is -float 0.1e1]
   puts -nonewline [::Simple::Type::is -float -.1E1]
   puts -nonewline [::Simple::Type::is -float 0.1234e-10]

   # No matches
   puts -nonewline [::Simple::Type::is -float 01]
   puts -nonewline [::Simple::Type::is -float 1-]
   puts -nonewline [::Simple::Type::is -float w23]
   puts -nonewline [::Simple::Type::is -float 1e]
   puts -nonewline [::Simple::Type::is -float e1]
   puts -nonewline [::Simple::Type::is -float 1-e1]

} -output 1111111111111000000

test-case is-4 {
   is
   -dec
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -dec 1]
   puts -nonewline [::Simple::Type::is -dec 123]
   puts -nonewline [::Simple::Type::is -dec 0]
   puts -nonewline [::Simple::Type::is -dec 987654321]

   # No matches
   puts -nonewline [::Simple::Type::is -dec 01]
   puts -nonewline [::Simple::Type::is -dec -1]
   puts -nonewline [::Simple::Type::is -dec w23]
   puts -nonewline [::Simple::Type::is -dec +1]

} -output 11110000

test-case is-5 {
   is
   -hex
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -hex 0x1]
   puts -nonewline [::Simple::Type::is -hex 0X1a3]
   puts -nonewline [::Simple::Type::is -hex 0x0]
   puts -nonewline [::Simple::Type::is -hex 0XFFAB]

   # No matches
   puts -nonewline [::Simple::Type::is -hex 1]
   puts -nonewline [::Simple::Type::is -hex 0x]
   puts -nonewline [::Simple::Type::is -hex -1]
   puts -nonewline [::Simple::Type::is -hex 0Xw23]
   puts -nonewline [::Simple::Type::is -hex +1]

} -output 111100000

test-case is-6 {
   is
   -bin
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -bin 0b1]
   puts -nonewline [::Simple::Type::is -bin 0B0101]
   puts -nonewline [::Simple::Type::is -bin 0B0]
   puts -nonewline [::Simple::Type::is -bin 0b0000]

   # No matches
   puts -nonewline [::Simple::Type::is -bin 1]
   puts -nonewline [::Simple::Type::is -bin 0b]
   puts -nonewline [::Simple::Type::is -bin 0b123]
   puts -nonewline [::Simple::Type::is -bin 0x0101]

} -output 11110000

test-case is-7 {
   is
   -octal
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -octal 01]
   puts -nonewline [::Simple::Type::is -octal 0137]
   puts -nonewline [::Simple::Type::is -octal 0]
   puts -nonewline [::Simple::Type::is -octal 01]
   puts -nonewline [::Simple::Type::is -octal 00]

   # No matches
   puts -nonewline [::Simple::Type::is -octal 018]
   puts -nonewline [::Simple::Type::is -octal 1]
   puts -nonewline [::Simple::Type::is -octal w23]

} -output 11111000

test-case is-8 {
   is
   -anybase
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -anybase 1]
   puts -nonewline [::Simple::Type::is -anybase 123]
   puts -nonewline [::Simple::Type::is -anybase 987654321]
   puts -nonewline [::Simple::Type::is -anybase 01]
   puts -nonewline [::Simple::Type::is -anybase 0137]
   puts -nonewline [::Simple::Type::is -anybase 0]
   puts -nonewline [::Simple::Type::is -anybase 01]
   puts -nonewline [::Simple::Type::is -anybase 00]
   puts -nonewline [::Simple::Type::is -anybase 0x1]
   puts -nonewline [::Simple::Type::is -anybase 0X1a3]
   puts -nonewline [::Simple::Type::is -anybase 0x0]
   puts -nonewline [::Simple::Type::is -anybase 0XFFAB]
   puts -nonewline [::Simple::Type::is -anybase 0b01001]

   # No matches
   puts -nonewline [::Simple::Type::is -anybase 018]
   puts -nonewline [::Simple::Type::is -anybase w23]
   puts -nonewline [::Simple::Type::is -anybase -1]
   puts -nonewline [::Simple::Type::is -anybase +1]
   puts -nonewline [::Simple::Type::is -anybase 0x]
   puts -nonewline [::Simple::Type::is -anybase 0Xw23]
   puts -nonewline [::Simple::Type::is -anybase 0b012]

} -output 11111111111110000000

test-case is-9 {
   is
   -number
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -number +137]
   puts -nonewline [::Simple::Type::is -number -137]
   puts -nonewline [::Simple::Type::is -number  137]
   puts -nonewline [::Simple::Type::is -number -1.3e7]
   puts -nonewline [::Simple::Type::is -number 0x1A3]
   puts -nonewline [::Simple::Type::is -number 0XFF0]
   puts -nonewline [::Simple::Type::is -number 0137]

   # No matches
   puts -nonewline [::Simple::Type::is -number 018]
   puts -nonewline [::Simple::Type::is -number 1e]
   puts -nonewline [::Simple::Type::is -number 0xw23]
   puts -nonewline [::Simple::Type::is -number 0X]

} -output 11111110000

test-case is-10 {
   is
   -alpha
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -alpha aBcD]

   # No matches
   puts -nonewline [::Simple::Type::is -alpha a1b]
   puts -nonewline [::Simple::Type::is -alpha 123]
   puts -nonewline [::Simple::Type::is -alpha {aB cD}]

} -output 1000

test-case is-11 {
   is
   -alphanum
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -alphanum aBcD]
   puts -nonewline [::Simple::Type::is -alphanum a1b]
   puts -nonewline [::Simple::Type::is -alphanum 123]

   # No matches
   puts -nonewline [::Simple::Type::is -alphanum {aB cD}]

} -output 1110

test-case is-12 {
   is
   -boolean
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -boolean 1]
   puts -nonewline [::Simple::Type::is -boolean 0]

   # No matches
   puts -nonewline [::Simple::Type::is -boolean true]
   puts -nonewline [::Simple::Type::is -boolean false]
   puts -nonewline [::Simple::Type::is -boolean on]
   puts -nonewline [::Simple::Type::is -boolean off]
   puts -nonewline [::Simple::Type::is -boolean {2 > 1}]

} -output 1100000

test-case is-13 {
   is
   -extbool
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -extbool 1]
   puts -nonewline [::Simple::Type::is -extbool 0]
   puts -nonewline [::Simple::Type::is -extbool true]
   puts -nonewline [::Simple::Type::is -extbool false]
   puts -nonewline [::Simple::Type::is -extbool on]
   puts -nonewline [::Simple::Type::is -extbool off]

   # No matches
   puts -nonewline [::Simple::Type::is -extbool {2 > 1}]

} -output 1111110

test-case is-14 {
   is
   -word
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -word foobar]
   puts -nonewline [::Simple::Type::is -word ,]
   puts -nonewline [::Simple::Type::is -word foo_1_bar_2]

   # No matches
   puts -nonewline [::Simple::Type::is -word {foo bar}]
   puts -nonewline [::Simple::Type::is -word foo,bar]
   puts -nonewline [::Simple::Type::is -word foo-bar]

} -output 111000

test-case is-15 {
   is
   -string
} -script {

   # Matches
   puts -nonewline\
      [::Simple::Type::is -string {Everything matches the type -string}]

} -output 1

test-case is-16 {
   is
   -script
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -script {}]
   puts -nonewline\
      [::Simple::Type::is -script {if {$foo} {puts foo} else {puts bar}}]

   # No matches
   puts -nonewline [::Simple::Type::is -script \{]
   puts -nonewline [::Simple::Type::is -script \"]

} -output 1100

test-case is-17 {
   is
   -expr
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -expr 2]
   puts -nonewline [::Simple::Type::is -expr {2.1 / 3.4 + (3.0 - 10 % 3) * 4.4}]
   puts -nonewline [::Simple::Type::is -expr {acos(log(1 - cos(1.23)))}]
   puts -nonewline [::Simple::Type::is -expr {4 * [llength {6 2}]}]
   puts -nonewline [::Simple::Type::is -expr {{word one} < {word two}}]

   # No matches
   puts -nonewline [::Simple::Type::is -expr {}]
   puts -nonewline [::Simple::Type::is -expr {2.1 / 3.4 + 3.0 - 10 % 3) * 4.4}]
   puts -nonewline [::Simple::Type::is -expr {acos(log(1 - FOO(1.23)))}]
   puts -nonewline [::Simple::Type::is -expr foo]
   puts -nonewline [::Simple::Type::is -expr \{]

} -output 1111100000

test-case is-18 {
   is
   -list
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -list {}]
   puts -nonewline [::Simple::Type::is -list {foo {bar gee}}]

   # No matches
   puts -nonewline [::Simple::Type::is -list \{]
   puts -nonewline [::Simple::Type::is -list \"]

} -output 1100

test-case is-19 {
   is
   -channel
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -channel stdin]
   puts -nonewline [::Simple::Type::is -channel stdout]
   puts -nonewline [::Simple::Type::is -channel stderr]
   set channel [open | w]
   puts -nonewline [::Simple::Type::is -channel $channel]
   close $channel

   # No matches
   puts -nonewline [::Simple::Type::is -channel $channel]
   puts -nonewline [::Simple::Type::is -channel foo]

} -output 111100

test-case is-20 {
   is
   -namespace
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -namespace {}]
   puts -nonewline [::Simple::Type::is -namespace ::]
   puts -nonewline [::Simple::Type::is -namespace ::Simple]
   puts -nonewline [::Simple::Type::is -namespace ::Simple::]
   puts -nonewline [::Simple::Type::is -namespace ::::Simple::::]

   # No matches
   puts -nonewline [::Simple::Type::is -namespace ::foo]

} -output 111110

test-case is-21 {
   is
   -namespacename
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -namespacename ::]
   puts -nonewline [::Simple::Type::is -namespacename ::foo]
   puts -nonewline [::Simple::Type::is -namespacename foo1]
   puts -nonewline [::Simple::Type::is -namespacename foo::]
   puts -nonewline [::Simple::Type::is -namespacename ::foo::bar]
   puts -nonewline [::Simple::Type::is -namespacename foo::bar]
   puts -nonewline [::Simple::Type::is -namespacename foo::::bar]
   puts -nonewline [::Simple::Type::is -namespacename ::foo::bar::]
   puts -nonewline [::Simple::Type::is -namespacename foo::bar::]
   puts -nonewline [::Simple::Type::is -namespacename foo::bar::::]
   puts -nonewline [::Simple::Type::is -namespacename foo_bar]
   puts -nonewline [::Simple::Type::is -namespacename foo-bar]
   puts -nonewline [::Simple::Type::is -namespacename foo%bar]
   puts -nonewline [::Simple::Type::is -namespacename FoO::bAr]

   # No matches
   puts -nonewline [::Simple::Type::is -namespacename {}]
   puts -nonewline [::Simple::Type::is -namespacename foo.bar]
   puts -nonewline [::Simple::Type::is -namespacename foo/bar]
   puts -nonewline [::Simple::Type::is -namespacename foo\$bar]
   puts -nonewline [::Simple::Type::is -namespacename foo@bar]

} -output 1111111111111100000

test-case is-22 {
   is
   -qualifiedname
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo1::bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo_bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo-bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo%bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::FoO::bAr]
   puts -nonewline [::Simple::Type::is -qualifiedname ::]

   # No matches
   puts -nonewline [::Simple::Type::is -qualifiedname ::::foo]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo::bar::::]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo::bar::]
   puts -nonewline [::Simple::Type::is -qualifiedname foo]
   puts -nonewline [::Simple::Type::is -qualifiedname foo::]
   puts -nonewline [::Simple::Type::is -qualifiedname foo::bar]
   puts -nonewline [::Simple::Type::is -qualifiedname foo::::bar]
   puts -nonewline [::Simple::Type::is -qualifiedname foo::bar::]
   puts -nonewline [::Simple::Type::is -qualifiedname {}]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo.bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo/bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo\$bar]
   puts -nonewline [::Simple::Type::is -qualifiedname ::foo@bar]

} -output 11111110000000000000

test-case is-23 {
   is
   -unqualifiedname
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -unqualifiedname foo]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo_bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo-bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo1%bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname FoObAr]
   puts -nonewline [::Simple::Type::is -unqualifiedname {}]

   # No matches
   puts -nonewline [::Simple::Type::is -unqualifiedname foo::bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo::::bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname ::foo::bar::::]
   puts -nonewline [::Simple::Type::is -unqualifiedname ::foo::bar::]
   puts -nonewline [::Simple::Type::is -unqualifiedname ::]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo::]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo::bar::]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo.bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo/bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo\$bar]
   puts -nonewline [::Simple::Type::is -unqualifiedname foo@bar]

} -output 11111100000000000

test-case is-24 {
   is
   -name
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -name ::foo]
   puts -nonewline [::Simple::Type::is -name ::foo::bar]
   puts -nonewline [::Simple::Type::is -name ::foo_bar]
   puts -nonewline [::Simple::Type::is -name ::foo1-bar]
   puts -nonewline [::Simple::Type::is -name ::foo%bar]
   puts -nonewline [::Simple::Type::is -name ::FoO::bAr]
   puts -nonewline [::Simple::Type::is -name foo]
   puts -nonewline [::Simple::Type::is -name foo_bar]
   puts -nonewline [::Simple::Type::is -name foo-bar]
   puts -nonewline [::Simple::Type::is -name foo%bar]
   puts -nonewline [::Simple::Type::is -name FoObAr]
   puts -nonewline [::Simple::Type::is -name foo::bar]
   puts -nonewline [::Simple::Type::is -name {}]

   # No matches
   puts -nonewline [::Simple::Type::is -name ::foo::bar::::]
   puts -nonewline [::Simple::Type::is -name ::foo::bar::]
   puts -nonewline [::Simple::Type::is -name ::]
   puts -nonewline [::Simple::Type::is -name foo::]
   puts -nonewline [::Simple::Type::is -name foo::::bar]
   puts -nonewline [::Simple::Type::is -name foo::bar::]
   puts -nonewline [::Simple::Type::is -name ::foo.bar]
   puts -nonewline [::Simple::Type::is -name ::foo/bar]
   puts -nonewline [::Simple::Type::is -name ::foo\$bar]
   puts -nonewline [::Simple::Type::is -name ::foo@bar]
   puts -nonewline [::Simple::Type::is -name foo.bar]
   puts -nonewline [::Simple::Type::is -name foo/bar]
   puts -nonewline [::Simple::Type::is -name foo\$bar]
   puts -nonewline [::Simple::Type::is -name foo@bar]

} -output 111111111111100000000000000

test-case is-25 {
   is
   -array
} -setup {

   # Create a couple of arrays
   array set foo {bar gee}
   set bar(foo) gee

} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -array foo]
   puts -nonewline [::Simple::Type::is -array bar]

   # No matches
   unset foo
   puts -nonewline [::Simple::Type::is -array foo]

} -cleanup {

   # Delete the remaining array
   unset bar

} -output 110

test-case is-26 {
   is
   -pattern
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -pattern {}]
   puts -nonewline [::Simple::Type::is -pattern foo]
   puts -nonewline [::Simple::Type::is -pattern {[a-z]?}]
   puts -nonewline [::Simple::Type::is -pattern \{]

} -output 1111

test-case is-27 {
   is
   -regexp
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -regexp {}]
   puts -nonewline [::Simple::Type::is -regexp foo]
   puts -nonewline [::Simple::Type::is -regexp {[a-z]+}]
   puts -nonewline [::Simple::Type::is -regexp \{]

   # No matches
   puts -nonewline [::Simple::Type::is -regexp {[a-z]++}]
   puts -nonewline [::Simple::Type::is -regexp \(]
   puts -nonewline [::Simple::Type::is -regexp \[]

} -output 1111000

test-case is-28 {
   is
   -version
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -version 123]
   puts -nonewline [::Simple::Type::is -version 1.2.3]

   # No matches
   puts -nonewline [::Simple::Type::is -version {}]
   puts -nonewline [::Simple::Type::is -version foo]
   puts -nonewline [::Simple::Type::is -version .1.2.3]
   puts -nonewline [::Simple::Type::is -version 1.2.3.]

} -output 110000

test-case is-29 {
   is
   -widgetname
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -widgetname .]
   puts -nonewline [::Simple::Type::is -widgetname .foo]
   puts -nonewline [::Simple::Type::is -widgetname .foo1.bar]
   puts -nonewline [::Simple::Type::is -widgetname .FOO]
   puts -nonewline [::Simple::Type::is -widgetname .foo_bar]
   puts -nonewline [::Simple::Type::is -widgetname .foo-bar]
   puts -nonewline [::Simple::Type::is -widgetname .foo%bar]

   # No matches
   puts -nonewline [::Simple::Type::is -widgetname ..]
   puts -nonewline [::Simple::Type::is -widgetname .foo.]
   puts -nonewline [::Simple::Type::is -widgetname .foo/bar]
   puts -nonewline [::Simple::Type::is -widgetname .foo\$bar]
   puts -nonewline [::Simple::Type::is -widgetname .foo@bar]

} -output 111111100000

test-case is-30 {
   is
   -type
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -type -int]
   puts -nonewline [::Simple::Type::is -type -foo1-bar]

   # No matches
   puts -nonewline [::Simple::Type::is -type int-]
   puts -nonewline [::Simple::Type::is -type int]

} -output 1100

test-case is-31 {
   is
   -flag
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -flag -foo]
   puts -nonewline [::Simple::Type::is -flag -bar1]
   puts -nonewline [::Simple::Type::is -flag -foo-bar]

   # No matches
   puts -nonewline [::Simple::Type::is -flag foo]
   puts -nonewline [::Simple::Type::is -flag foo-]

} -output 11100

test-case is-32 {
   is
   -optional
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -optional ?foo?]
   puts -nonewline [::Simple::Type::is -optional ?foo_bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo-bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo1%bar?]
   puts -nonewline [::Simple::Type::is -optional ?FoObAr?]

   # No matches
   puts -nonewline [::Simple::Type::is -optional ?{}?]
   puts -nonewline [::Simple::Type::is -optional ?foo::bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo::::bar?]
   puts -nonewline [::Simple::Type::is -optional ?::foo::bar::::?]
   puts -nonewline [::Simple::Type::is -optional ?::foo::bar::?]
   puts -nonewline [::Simple::Type::is -optional ?::?]
   puts -nonewline [::Simple::Type::is -optional ?foo::?]
   puts -nonewline [::Simple::Type::is -optional ?foo::bar::?]
   puts -nonewline [::Simple::Type::is -optional ?foo.bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo/bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo\$bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo@bar?]
   puts -nonewline [::Simple::Type::is -optional ?foo]
   puts -nonewline [::Simple::Type::is -optional foo]
   puts -nonewline [::Simple::Type::is -optional foo?]

} -output 11111000000000000000

test-case is-33 {
   is
   -date
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -date {Sun May 30 22:05:51 GMT 1999}]
   puts -nonewline [::Simple::Type::is -date {Sun May 31 22:05:51 GMT 1999}]
   puts -nonewline [::Simple::Type::is -date 22:05:51]

   # No matches
   puts -nonewline [::Simple::Type::is -date {Sun May 32 22:05:51 GMT 1999}]
   puts -nonewline [::Simple::Type::is -date 24:05:51]

} -output 11100

test-case is-34 {
   is
   -choice
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -choice foo {foo bar}]
   puts -nonewline [::Simple::Type::is -choice bar {foo bar}]

   # No matches
   puts -nonewline [::Simple::Type::is -choice foo]
   puts -nonewline [::Simple::Type::is -choice FOO {foo bar}]

} -output 1100

test-case is-35 {
   is
   -any
} -script {

   # Matches
   puts -nonewline [::Simple::Type::is -any {Everything matches the type -any}]

} -output 1

test-case is-36 {
   is
   error, invalid type name
} -script {
   ::Simple::Type::is foo {}
} -error {expected type name but got "foo"}

test-case is-37 {
   is
   error, non-existing type
} -script {
   ::Simple::Type::is -foo {}
} -error {non-existing type "-foo"}

### ===========================================================================
### -test     : ::Simple::Type::convert type
test-case type-1 {
   ::Simple::Type::convert type
} -script {
   puts [::Simple::Type::convert type -foo]
   puts [::Simple::Type::convert type --foo]
} -output {foo
-foo
}

test-case type-2 {
   ::Simple::Type::convert type
   error, invalid type name
} -script {
   ::Simple::Type::convert type foo
} -error {expected type name but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::convert flag
test-case flag-1 {
   ::Simple::Type::convert flag
} -script {
   puts [::Simple::Type::convert flag -foo]
   puts [::Simple::Type::convert flag --foo]
} -output {foo
-foo
}

test-case flag-2 {
   ::Simple::Type::convert flag
   error, invalid flag name
} -script {
   ::Simple::Type::convert flag foo
} -error {expected flag name but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::convert optional
test-case optional-1 {
   ::Simple::Type::convert optional
} -script {
   ::Simple::Type::convert optional ?foo?
} -return foo

test-case optional-2 {
   ::Simple::Type::convert optional
   error, invalid optional value name
} -script {
   ::Simple::Type::convert optional foo
} -error {expected optional value name but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::convert
test-case to-boolean-1 {
   ::Simple::Type::convert
} -script {

   # True
   puts -nonewline [::Simple::Type::convert boolean true]
   puts -nonewline [::Simple::Type::convert boolean on]
   puts -nonewline [::Simple::Type::convert boolean 1]

   # False
   puts -nonewline [::Simple::Type::convert boolean false]
   puts -nonewline [::Simple::Type::convert boolean off]
   puts -nonewline [::Simple::Type::convert boolean 0]

} -output 111000

test-case to-boolean-2 {
   ::Simple::Type::convert boolean
   error, invalid boolean value
} -script {
   ::Simple::Type::convert boolean foo
} -error {expected boolean value but got "foo"}

### ===========================================================================
### -test     : ::Simple::Type::convert dec
test-case to-dec-1 {
   ::Simple::Type::convert dec
} -script {
   puts [::Simple::Type::convert dec 123]
   puts [::Simple::Type::convert dec 0x7B]
   puts [::Simple::Type::convert dec 0173]
   puts [::Simple::Type::convert dec 0b1111011]
} -output {123
123
123
123
}

test-case to-dec-2 {
   ::Simple::Type::convert dec
   error, invalid integer
} -script {
   ::Simple::Type::convert dec foo
} -error {expected integer but got "foo"}

test-case to-dec-3 {
   ::Simple::Type::convert dec
   error, negative integer
} -script {
   ::Simple::Type::convert dec -1
} -error {negative integer "-1"}

### ===========================================================================
### -test     : ::Simple::Type::convert hex
test-case to-hex-1 {
   ::Simple::Type::convert hex
} -script {
   puts [::Simple::Type::convert hex 123]
   puts [::Simple::Type::convert hex 0x7B]
   puts [::Simple::Type::convert hex 0173]
   puts [::Simple::Type::convert hex 0b1111011]
} -output {0x7B
0x7B
0x7B
0x7B
}

test-case to-hex-2 {
   ::Simple::Type::convert hex
   error, invalid integer
} -script {
   ::Simple::Type::convert hex foo
} -error {expected integer but got "foo"}

test-case to-hex-3 {
   ::Simple::Type::convert hex
   error, negative integer
} -script {
   ::Simple::Type::convert hex -1
} -error {negative integer "-1"}

### ===========================================================================
### -test     : ::Simple::Type::convert bin
test-case to-bin-1 {
   ::Simple::Type::convert bin
} -script {
   puts [::Simple::Type::convert bin 123]
   puts [::Simple::Type::convert bin 0x7B]
   puts [::Simple::Type::convert bin 0173]
   puts [::Simple::Type::convert bin 0b1111011]
   puts [::Simple::Type::convert bin 0B1111011]
   puts [::Simple::Type::convert bin 0]
} -output {0b1111011
0b1111011
0b1111011
0b1111011
0b1111011
0b0
}

test-case to-bin-2 {
   ::Simple::Type::convert bin
   error, invalid integer
} -script {
   ::Simple::Type::convert bin foo
} -error {expected integer but got "foo"}

test-case to-bin-3 {
   ::Simple::Type::convert bin
   error, negative integer
} -script {
   ::Simple::Type::convert bin -1
} -error {negative integer "-1"}

### ===========================================================================
### -test     : ::Simple::Type::convert octal
test-case to-octal-1 {
   ::Simple::Type::convert octal
} -script {
   puts [::Simple::Type::convert octal 123]
   puts [::Simple::Type::convert octal 0x7B]
   puts [::Simple::Type::convert octal 0173]
   puts [::Simple::Type::convert octal 0b1111011]
} -output {0173
0173
0173
0173
}

test-case to-octal-2 {
   ::Simple::Type::convert octal
   error, invalid integer
} -script {
   ::Simple::Type::convert octal foo
} -error {expected integer but got "foo"}

test-case to-octal-3 {
   ::Simple::Type::convert octal
   error, negative integer
} -script {
   ::Simple::Type::convert octal -1
} -error {negative integer "-1"}

### ===========================================================================
### -test     : ::Simple::Type::configure
test-case configure-1 {
   ::Simple::Type::configure
} -script {
   ::Simple::Type::configure
} -error {this package has no options}

### ===========================================================================
### -test     : ::Simple::Type::cget
test-case cget-1 {
   ::Simple::Type::cget
} -script {
   ::Simple::Type::cget
} -return {}

} ;# End of regression testing section
