/***********************************************************************
 *             Copyright (C) 1995-1999 Joe English
 *                   Freely redistributable
 ***********************************************************************
 *
 * File:	tclcost.c
 * Author:	Joe English
 * Created: 	28 Feb 1995
 * Description:	Cost/Tcl initialization and interface routines
 *
 * tclcost.c,v 1.50 1999/06/27 01:42:57 joe Exp
 */

#define COST_VERSION_MAJOR "2"
#define COST_VERSION_MINOR "2"
#define COST_VERSION "2.2"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "tcl.h"

#include "project.h"
#include "strmgt.h"
#include "esis.h"
#include "costq.h"
#define COSTGLOBAL
#include "tclcost.h"

/*
 * CostEval: Evaluate Tcl expression with current node
 */

int CostEval(Tcl_Interp *interp, CostData *cd, ESISNode nd, char *expr)
{
    ESISNode old_node = cd->current_node;
    ESISDocument old_document = cd->current_document;
    int status;

    cd->current_node = nd;
    status = Tcl_Eval(interp, expr);
    cd->current_node = old_node;
    cd->current_document = old_document;
    return status;
}

/*
 * CostRegisterDocument --
 *	add new ESISDocument to document table,
 * 	Sets interp->result to new document handle.
 */
void CostRegisterDocument(Tcl_Interp *interp, CostData *cd, ESISDocument doc)
{
    char namebuf[20];
    Tcl_HashEntry *hptr;
    int new;

    sprintf(namebuf, "doc%04d", ++cd->number_documents);
    hptr = Tcl_CreateHashEntry(&cd->document_table, namebuf, &new);
    ASSERT(!new, "Duplicate document name ?!?")
    Tcl_SetHashValue(hptr, (ClientData)doc);
    esis_set_docname(doc, intern(namebuf));

    Tcl_SetResult(interp, namebuf, TCL_VOLATILE);
}

/*+++
 * Command procedures:
 */

#if HAVE_TCL_CHANNEL_IO
static int TclIOproc(void *closure, char *buf, int n)
{
    Tcl_Channel channel = (Tcl_Channel)closure;
    int count = Tcl_Read(channel, buf, n);
    return count;
}
#endif

static CMDPROC(CostLoadSGMLSProc)
{
    CostData *cd = (CostData *)clientData;
    char *handle;
    ESISInputStream stream;
    ESISDocument new_document;

    CHECKNARGS(1, "handle")

    handle = argv[1];

#if HAVE_TCL_CHANNEL_IO
    {
	int mode;
	Tcl_Channel channel = Tcl_GetChannel(interp, handle, &mode);

	if (!channel) {
	    Tcl_AppendResult(interp,"Invalid channel name ",handle,NULL);
	    return TCL_ERROR;
	}
	if ((mode & TCL_WRITABLE) || !(mode & TCL_READABLE)) {
	    Tcl_AppendResult(interp, handle, " wrong mode", NULL);
	    return TCL_ERROR;
	}
	stream = estream_create(TclIOproc, channel);
    }
#else
    {
	FILE *fp;
	int status = Tcl_GetOpenFile(interp, handle,
	    0,	/* forWriting */
	    1, 	/* checkUsage */
	    &fp);
	if (status != TCL_OK)
	    return TCL_ERROR;
	stream = estream_create(ESISIOstdio, fp);
    }
#endif

    if (!(new_document = estream_load_sgmls(stream))) {
	Tcl_AppendResult(interp, "Error reading ", handle, NULL);
	return TCL_ERROR;
    }
    estream_close(stream);

    cd->current_document = new_document;
    cd->current_node = esis_rootnode(cd->current_document);

    CostRegisterDocument(interp, cd, new_document);

    return TCL_OK;
}

static CQStatus CostQueryContinuation(
	ESISNode nd, const char *value, void *closure)
{
    Tcl_DString *dstr = (Tcl_DString *)closure;
    (void)nd;
    if (value)
	Tcl_DStringAppend(dstr, (/*!const*/char *)value, -1);
    return CQ_SUCCEED;
}

/*ARGSUSED*/
static CQStatus CostQueryAllContinuation(
	ESISNode nd,const char *value,void *closure)
{
    Tcl_DString *dstr = (Tcl_DString *)closure;
    if (value)
	Tcl_DStringAppendElement(dstr, (/*!const*/char *)value);
    return CQ_FAIL;	/* force query continuation */
}

/*ARGSUSED*/
static CQStatus CostQueryCountContinuation(
	ESISNode nd,const char *value,void *closure)
{
    int (*ip) = (int *)closure;
    ++(*ip);
    return CQ_FAIL;	/* force query continuation */
}

static CMDPROC(CostQueryProc)	/* %%% factor */
{
    CostData *cd = (CostData *)clientData;
    CQQuery q;
    Tcl_DString ans;

    CHECKCURNODE(cd)

    Tcl_DStringInit(&ans);
    q = cq_buildquery(argv+1, argc - 1);
    if (!q)
	return TCL_ERROR;
    (void)cq_doquery(cd->current_node, q, CostQueryContinuation, &ans);
    cq_destroyquery(q);
    Tcl_ResetResult(interp);	/* bug in Tcl */
    Tcl_DStringResult(interp, &ans);
    return TCL_OK;
}

static CMDPROC(CostQueryAllProc)	/* %%% factor */
{
    CostData *cd = (CostData *)clientData;
    CQQuery q;
    Tcl_DString ans;
    CHECKCURNODE(cd)
    Tcl_DStringInit(&ans);
    q = cq_buildquery(argv+1, argc - 1);
    if (!q)
	return TCL_ERROR;
    (void)cq_doquery(cd->current_node, q,
	CostQueryAllContinuation, &ans);
    cq_destroyquery(q);
    Tcl_ResetResult(interp);	/* bug in Tcl */
    Tcl_DStringResult(interp, &ans);
    return TCL_OK;
}

static CMDPROC(CostMatchProc)	/* %%% factor w/CostQueryProc */
{
    CostData *cd = (CostData *)clientData;
    CQQuery q;
    int status;
    CHECKCURNODE(cd)
    q = cq_buildquery(argv+1, argc - 1);
    if (!q)
	return TCL_ERROR;
    status = cq_testquery(cd->current_node, q);
    cq_destroyquery(q);
    Tcl_SetResult(interp, status ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

static CMDPROC(CostCountProc)	/* %%% factor */
{
    CostData *cd = (CostData *)clientData;
    CQQuery q;
    int count;
    char buf[16];
    CHECKCURNODE(cd)
    q = cq_buildquery(argv+1, argc - 1);
    if (!q) return TCL_ERROR;
    count = 0;
    (void)cq_doquery(cd->current_node, q, CostQueryCountContinuation, &count);
    cq_destroyquery(q);
    sprintf(buf, "%d", count);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}

  /*ARGSUSED*/
static CQStatus CostSelectNodeContinuation(
	ESISNode node, const char *value, void *closure)
{
    CostData *cd = (CostData *)closure;
    if (node && !value)
	cd->current_node = node;
    return CQ_SUCCEED;
}

static CMDPROC(CostSelectNodeProc)	/* %%% factor w/CostQueryProc */
{
    CostData *cd = (CostData *)clientData;
    CQQuery q;
    CQStatus status;

    /* %%% #args? */
    CHECKCURNODE(cd)
    q = cq_buildquery(argv+1, argc - 1);
    if (!q)
	return TCL_ERROR;
    status = cq_doquery(cd->current_node, q, CostSelectNodeContinuation, cd);
    cq_destroyquery(q);
    Tcl_SetResult(interp, status == CQ_SUCCEED ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

/* implementation of withNode and foreachNode
*/

typedef struct
{
    Tcl_Interp	*interp;
    CostData	*costData;
    char	*script;
    CQStatus	cq_default_status;
    int 	status;		/* Tcl result status */
} CQData;

  /*ARGSUSED*/
static CQStatus CostNodeLoopContinuation(
	ESISNode node, const char *value, void *closure)
{
    CQData *cqdata = (CQData *)closure;
    if (node && !value)
    {
	int status =
	    CostEval(cqdata->interp, cqdata->costData, node, cqdata->script);
	switch (status) {
	    case TCL_CONTINUE:
	    case TCL_OK:
		break;
	    case TCL_BREAK:
		status = TCL_OK;
		/*FALLTHRU*/
	    case TCL_ERROR :
	    default:
		cqdata->status = status;
		return CQ_SUCCEED;	/* stop evaluating query %%% */
	}
    }
    return cqdata->cq_default_status;
}

static CMDPROC(CostWithNodeProc)
{
    CostData *cd = (CostData*)clientData;
    CQQuery q;
    CQData cqdata;
    char *script = argv[argc - 1];

    if (argc < 3) {
	Tcl_AppendResult(interp, "wrong #args\n", NULL);
	Tcl_AppendResult(interp, "Usage: ", argv[0], " query... cmd", NULL);
	return TCL_ERROR;
    }
    CHECKCURNODE(cd)

    q = cq_buildquery(argv+1, argc - 2);
    if (!q) return TCL_ERROR;

    cqdata.interp = interp;
    cqdata.costData = cd;
    cqdata.script = script;
    cqdata.cq_default_status =  CQ_SUCCEED;
    cqdata.status = TCL_OK;

    (void)cq_doquery(cd->current_node, q, CostNodeLoopContinuation , &cqdata);
    cq_destroyquery(q);
    return cqdata.status;
}

static CMDPROC(CostForeachNodeProc)
{
    CostData *cd = (CostData *)clientData;
    CQQuery q;
    CQData cqdata;
    char *script = argv[argc - 1];

    if (argc < 3) {
	Tcl_AppendResult(interp, "wrong #args\n", NULL);
	Tcl_AppendResult(interp, "Usage: ", argv[0], " query... cmd", NULL);
	return TCL_ERROR;
    }
    CHECKCURNODE(cd)

    q = cq_buildquery(argv+1, argc - 2);
    if (!q) return TCL_ERROR;

    cqdata.interp = interp;
    cqdata.costData = cd;
    cqdata.script = script;
    cqdata.cq_default_status = CQ_FAIL;
    cqdata.status = TCL_OK;

    (void)cq_doquery(cd->current_node, q, CostNodeLoopContinuation , &cqdata);
    cq_destroyquery(q);
    return cqdata.status;
}

static CMDPROC(CostSelectDocumentProc)
{
    CostData *cd = (CostData *)clientData;
    Tcl_HashEntry *hptr;
    char *docname = argv[1];

    CHECKNARGS(1, "docname");
    hptr = Tcl_FindHashEntry(&cd->document_table, docname);
    if (!hptr) {
	Tcl_AppendResult(interp, docname, ": No such document", NULL);
	return TCL_ERROR;
    }

    cd->current_document = (ESISDocument)Tcl_GetHashValue(hptr);
    cd->current_node = esis_rootnode(cd->current_document);

    Tcl_SetResult(interp, docname, TCL_VOLATILE);

    return TCL_OK;
}

static CMDPROC(CostWithDocumentProc)
{
    CostData *cd = (CostData *)clientData;
    Tcl_HashEntry *hptr;
    char *docname = argv[1], *script = argv[2];
    ESISDocument old_document = cd->current_document;
    ESISNode old_node = cd->current_node;
    int status;

    CHECKNARGS(2, "docname script");
    hptr = Tcl_FindHashEntry(&cd->document_table, docname);
    if (!hptr) {
	Tcl_AppendResult(interp, docname, ": No such document", NULL);
	return TCL_ERROR;
    }
    cd->current_document = (ESISDocument)Tcl_GetHashValue(hptr);
    cd->current_node = esis_rootnode(cd->current_document);
    status = Tcl_Eval(interp, script);
    cd->current_document = old_document;
    cd->current_node = old_node;
    return status;
}

static CMDPROC(CostCurrentDocumentProc)
{
    CostData *cd = (CostData *)clientData;

    CHECKNARGS(0, "");
    if (!cd->current_document) {
	Tcl_AppendResult(interp, argv[0], ": No current document", NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, 
	(/*!const*/char*)esis_docname(cd->current_document), TCL_VOLATILE);
    return TCL_OK;
}

static CMDPROC(CostContentProc)
{
    CostData *cd = (CostData *)clientData;
    ESISString content;

    CHECKCURNODE(cd)
    CHECKNARGS(0,"");

    content = esis_text(cd->current_node);
    if (content)
    {
	Tcl_SetResult(interp, content, TCL_VOLATILE);
    }
    else
    {	/* Not a data node -- concatenate character data of descendants */
	ESISNode nd = esis_firstpreorder(cd->current_node);
	Tcl_ResetResult(interp);
	while (nd)
	{
	    ESISNodeType ndtype = esis_nodetype(nd);
	    if (ndtype == EN_CDATA || ndtype == EN_RE || ndtype == EN_SDATA)
		Tcl_AppendResult(interp, esis_text(nd), NULL);
	    nd = esis_nextpreorder(cd->current_node, nd);
	}
    }
    return TCL_OK;
}

static CMDPROC(CostSetpropProc)
{
    CostData *cd = (CostData *)clientData;
    char *propname, *propval;
    CHECKNARGS(2,"propname propval");
    CHECKCURNODE(cd)
    propname = argv[1];
    propval = argv[2];
    esis_setprop(cd->current_node, propname, propval);
    return TCL_OK;
}

static CMDPROC(CostUnsetpropProc)
{
    CostData *cd = (CostData *)clientData;
    int i;
    if (argc < 1) {
	Tcl_AppendResult(interp,
		"Usage: ",argv[0]," propname [propname..]",NULL);
	return TCL_ERROR;
    }
    CHECKCURNODE(cd)
    for (i=1; i<argc; ++i)
	esis_unsetprop(cd->current_node, argv[i]);
    return TCL_OK;
}

/* +++ Ilinks and relations:
 */

static CMDPROC(CostRelationProc)
{
    CostData *cd = (CostData *)clientData;
    ESISToken relname;
    char **anchnames;
    int nanchors;

    CHECKCURNODE(cd)

    if (argc <= 1) {
	Tcl_AppendResult(interp,
		"Usage: ", argv[0], " relname [anchname ...]",
		NULL);
	return TCL_ERROR;
    }

    relname = ucintern(argv[1]);
    anchnames = argv + 2;
    nanchors = argc - 2;
    if (!esis_define_relation(cd->current_document,relname,nanchors,anchnames))
    {
	Tcl_AppendResult(interp, "Error defining relation ", relname, NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static CMDPROC(CostAddLinkProc)
{
    CostData *cd = (CostData *)clientData;
    ESISToken relname;
    ESISNode ilink;
    ESISNode old_current_node = cd->current_node;

    if (argc <  1 || argc % 2 != 0) {
	Tcl_AppendResult(interp,
	    "Usage: ", argv[0], " relname [ anchname \"query...\" ... ]",
	    NULL);
	return TCL_ERROR;
    }

    ++argv; --argc;
    relname = ucintern(argv[0]);
    ++argv; --argc;

    ilink = esis_create_ilink(cd->current_document, relname, cd->current_node);
    if (!ilink) {
	Tcl_AppendResult(interp, "Error adding ", relname, " link",NULL);
	return TCL_ERROR;
    }

    while (argc > 0) {
	ESISToken anchname;
	char **qargs;
	int nqargs;
	CQQuery q;
	CQStatus status;
	ESISNode anchor;

	anchname = ucintern(argv[0]);
	if (TCL_OK != Tcl_SplitList(interp, argv[1], &nqargs, &qargs))
	    return TCL_ERROR;

	q = cq_buildquery(qargs, nqargs);
	Tcl_FreeSplitList(qargs);
	if (!q) return TCL_ERROR;
	status = cq_doquery(ilink, q, CostSelectNodeContinuation, cd);
	anchor = cd->current_node; cd->current_node = old_current_node;
	cq_destroyquery(q);
	if (status == CQ_SUCCEED) {
	    if (!esis_set_linkend(ilink, anchname, anchor)) {
		Tcl_AppendResult(interp,
		    "Error creating ", anchname," anchor in ", relname,
		    NULL);
		return TCL_ERROR;
	    }
	}
	argv += 2; argc -= 2;
    }
    return TCL_OK;

}

/* +++ Handlers
 * CommandEventHandler: closure is a Tcl cmdInfo structure.
 * Invokes Tcl proc with one argument, the event type name.
 */

typedef struct
{
    Tcl_Interp	*interp;
    Tcl_CmdInfo	cmdInfo;
    CostData	*costData;
    const char	*cmdName;
    int 	status;		/* return status from handler */
} EHData;

static int CommandEventHandler
	(ESISEventType ev, ESISNode node, void *closure)
{
    EHData *ehdata = (EHData *)closure;
    CostData *cd = ehdata->costData;
    ESISNode old_node = cd->current_node;
    ESISDocument old_document = cd->current_document;
    const char *evname = esis_evtype_name(ev);
    const char *argv[3];
    int argc = 2;
    int status;

    argv[0] = ehdata->cmdName;
    argv[1] = evname;
    argv[2] = 0;
    cd->current_node = node;
    Tcl_ResetResult(ehdata->interp);	/* another Tcl 7.3 bug %%% */
    status = (ehdata->cmdInfo.proc)
	(ehdata->cmdInfo.clientData, ehdata->interp, argc,
	(/*!const*/char **)argv);
    cd->current_node = old_node;
    cd->current_document = old_document;
    if (status == TCL_ERROR) {
	Tcl_AppendResult(ehdata->interp, "\nError in ",evname," handler",NULL);
	if (esis_gi(node))
	    Tcl_AppendResult(ehdata->interp,
		" (", esis_gi(node), " element)", NULL);
    }
    ehdata->status = status;
    return status == TCL_OK;
}

static CMDPROC(CostProcessProc)
{
    CostData *cd = (CostData *)clientData;
    char *cmdName;
    EHData ehdata;

    CHECKCURNODE(cd)
    CHECKNARGS(1, "handler")
    cmdName = argv[1];
    if (!Tcl_GetCommandInfo(interp, cmdName, &ehdata.cmdInfo)) {
	Tcl_AppendResult(interp,argv[0],": no proc ",argv[1],NULL);
	return TCL_ERROR;
    }
    ehdata.interp = interp;
    ehdata.cmdName = cmdName;
    ehdata.costData = cd;
    ehdata.status = TCL_OK;

    esis_traverse(cd->current_node, CommandEventHandler, &ehdata);
    return ehdata.status;
}

static CMDPROC(CostDefineSpecificationProc)
{
    CostData *cd = (CostData *)clientData;
    CQAssoc assoc;

    CHECKNARGS(2,"name alist");
    assoc = assocConstructor(interp, cd, argv[2]);
    if (!assoc) {
	Tcl_AppendResult(interp, "Error defining ", argv[1], NULL);
	return TCL_ERROR;
    }

    Tcl_CreateCommand(interp,argv[1],assocProc,
	(ClientData)assoc, assocDestructor);

    Tcl_SetResult(interp, argv[1], TCL_VOLATILE);
    return TCL_OK;
}

/*+++ Event handler routines:
*/

typedef struct
{
    Tcl_Interp	*interp;		/* Tcl interpreter to use */
    CostData	*costData;		/* Cost data */
    int		status;			/* return status */
    int  	global;			/* flag: evaluate in global scope? */
    char 	*handlerScripts[ESISEventTypeMax];
} CostEventHandlerRec;

static int CostEventHandler(ESISEventType ev, ESISNode node, void *closure)
{
    CostEventHandlerRec *tr = (CostEventHandlerRec *)closure;
    CostData *cd = tr->costData;
    char *expr = tr->handlerScripts[ev];

    if (expr!= NULL)
    {
	ESISNode old_current_node = cd->current_node;
	ESISDocument old_current_document = cd->current_document;
	int status;

	cd->current_node = node;
	status = tr->global
		? Tcl_GlobalEval(tr->interp, expr)
		: Tcl_Eval(tr->interp, expr);
	cd->current_node = old_current_node;
	cd->current_document = old_current_document;

	switch (status)
	{
	    case TCL_OK:
		return 1;
	    case TCL_BREAK:
		status = TCL_OK;
		/* FALLTHRU */
	    default:
		tr->status = status;
		return 0;
	}
    }
    /* else */
    return 1;
}

static CMDPROC(CostEventHandlerProc)
{
    CostEventHandlerRec tr = *(CostEventHandlerRec *)clientData;
    /* Note: make local copy of event handler record; this makes
     * it re-entrant.
     */
    CostData *cd = tr.costData;

    CHECKCURNODE(cd)

    if (argc == 1)
    {
	esis_traverse(cd->current_node, CostEventHandler, &tr);
    }
    else if (argc == 2)	/* called as an event handler from 'process' */
    {
	ESISEventType ev = esis_string_to_evtype(argv[1]);
	if (ev == EV_ERROR) {
	    Tcl_AppendResult(interp, "\nBad event name ", argv[1], NULL);
	    return TCL_ERROR;
	}
	CostEventHandler(ev, cd->current_node, &tr);
    }
    else
    {
	Tcl_AppendResult(interp, "Usage: ", argv[0], NULL);
	return TCL_ERROR;
    }

    return tr.status;
}

static void CostEventHandlerDestructor(ClientData closure)
{
    CostEventHandlerRec *tr = (CostEventHandlerRec *)closure;
    int i;
    for (i=0; i<ESISEventTypeMax; ++i)
	if (tr->handlerScripts[i])
	    free(tr->handlerScripts[i]);
    free(tr);
}

static CMDPROC(CostDefineEventHandler)
{
    CostData *cd = (CostData *)clientData;
    CostEventHandlerRec *tr = malloc(sizeof(*tr));
    char **speclist;
    char *procname, *handlers;
    int i, speclistlen, status;

    if (argc < 3) {
usage:
	Tcl_AppendResult(interp, "Usage: ", argv[0],
	    " procname [ -global ] { EVENT script ... }", NULL);
	return TCL_ERROR;
    }

    tr->interp = interp;
    tr->costData = cd;
    tr->status = TCL_OK;
    tr->global = 0;
    for (i=0; i<ESISEventTypeMax; ++i)
	tr->handlerScripts[i] = NULL;

    i = 1;
    procname = argv[i]; ++i;

    if (*argv[i] == '-')
    {
	if (!strcmp(argv[i], "-global"))
		tr->global = 1;
	else
		goto usage;
	++i;
    }
    if (i != argc - 1)
	goto usage;

    handlers = skip_comments(argv[i]);
    if (TCL_OK != Tcl_SplitList(interp, handlers, &speclistlen, &speclist))
	return TCL_ERROR;

    Tcl_SetResult(interp, procname, TCL_VOLATILE);
    status = TCL_OK;
    if (speclistlen % 2) {
	--speclistlen;
	status = TCL_ERROR;
    }
    for (i=0; i<speclistlen; i += 2) {
	ESISEventType ev = esis_string_to_evtype(speclist[i]);
	if (ev == EV_ERROR) {
	    Tcl_AppendResult(interp, "\nBad event name ", speclist[i], NULL);
	    status = TCL_ERROR;
	}
	else if (tr->handlerScripts[ev]) {
	    Tcl_AppendResult(interp, "\n event multiply specified",
		speclist[i], NULL);
	    status = TCL_ERROR;
	}
	else {
	    tr->handlerScripts[ev] = savestring(speclist[i+1]);
	}
    }

    Tcl_CreateCommand(interp,procname, CostEventHandlerProc,
	(ClientData)tr, CostEventHandlerDestructor);

    Tcl_FreeSplitList(speclist);
    return status;
}

/* InitCostData -- initialize package data.
 */
static CostData *InitCostData(void)
{
    CostData *cd = malloc(sizeof(*cd));
    cd->current_node = NULL;
    cd->current_document = NULL;
    cd->number_documents = 0;
    Tcl_InitHashTable(&cd->document_table, TCL_STRING_KEYS);
    return cd;
}

/* Tcl_InterpDeleteProc for CostData:
 */
static void DeleteCostData(ClientData clientData, Tcl_Interp *interp)
{
    CostData *cd = (CostData *)clientData;
    Tcl_HashSearch hsrch;
    Tcl_HashEntry *hptr; 

    /*
     * Free all loaded documents:
     */
    for (hptr = Tcl_FirstHashEntry(&cd->document_table, &hsrch); 
	 hptr != NULL;
	 hptr = Tcl_NextHashEntry(&hsrch) )
    {
	ESISDocument doc = (ESISDocument)Tcl_GetHashValue(hptr);
	if (doc)
	    esis_free_document(cd->current_document);
    }

    free(cd);
}

/*+++ Utility routines:
*/

/* skip_comments:
 * 	Skip past an initial sequence of Tcl-style comments.
 *	Comments are only recognized on consequitive sequence of line.
 *	First non-comment line terminates comment sequence.
 */

char *skip_comments(char *p)
{
    /* Skip past '(\W*(#.*))*'
    */
    while (*p && isspace(*p)) ++p;
    while (*p == '#')
    {
	++p;
	while (*p && *p != '\n')  ++p;
	while (*p && isspace(*p)) ++p;
    }
    return p;
}

/*+++
 *----------------------------------------------------------------------
 *
 * Cost_Init --
 *	Package initialization for CoST
 *	Register commands, load initialization file.
 *
 *----------------------------------------------------------------------
 */

/*
 * Black magic to make this work as a Windows .DLL
 */
#ifdef __WIN32__
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
BOOL APIENTRY
DllEntryPoint(HINSTANCE hInst, DWORD reason, LPVOID reserved)
{ return TRUE; }
#endif

/* If the stub library is not available or if we're not using stubs,
 * redefine Tcl_InitStubs to call Tcl_PkgRequre instead:
 */
#if !TCL_USE_STUBS
#   ifndef Tcl_InitStubs
#	define	Tcl_InitStubs(interp, version, exact) \
		Tcl_PkgRequire(interp, "Tcl", version, exact)
#   endif
#endif

int EXPORT Cost_Init(Tcl_Interp *interp)
{
    ClientData cd = (ClientData)InitCostData();
    char *libdir = NULL; 
    Tcl_DString startup_file;
    int status;

    Tcl_InitStubs(interp, TCL_VERSION, 0);
    Tcl_SetAssocData(interp, "Cost", DeleteCostData, cd);

    Tcl_SetVar2(interp, "COST", "VERSION", COST_VERSION, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "COST", "MAJOR", COST_VERSION_MAJOR, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "COST", "MINOR", COST_VERSION_MINOR, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, "COST", "TCLVERSION", TCL_VERSION, TCL_GLOBAL_ONLY);

    /* Register Cost commands: */
    Tcl_CreateCommand(interp, "loadsgmls", 	CostLoadSGMLSProc, cd, NULL);
    Tcl_CreateCommand(interp, "loadxml", 	CostLoadXMLProc, cd, NULL);
    Tcl_CreateCommand(interp, "query",   	CostQueryProc, cd, NULL);
	Tcl_CreateCommand(interp, "q",   	CostQueryProc, cd, NULL);
    Tcl_CreateCommand(interp, "queryall", 	CostQueryAllProc, cd, NULL);
	Tcl_CreateCommand(interp, "query*", 	CostQueryAllProc, cd, NULL);
	Tcl_CreateCommand(interp, "q*", 	CostQueryAllProc, cd, NULL);
    Tcl_CreateCommand(interp, "query?",  	CostMatchProc, cd, NULL);
	Tcl_CreateCommand(interp, "q?",  	CostMatchProc, cd, NULL);
    Tcl_CreateCommand(interp, "query#",  	CostCountProc, cd, NULL);
	Tcl_CreateCommand(interp, "q#",  	CostCountProc, cd, NULL);
	Tcl_CreateCommand(interp, "countq",  	CostCountProc, cd, NULL);
    Tcl_CreateCommand(interp, "withNode",
				CostWithNodeProc,cd, NULL);
    Tcl_CreateCommand(interp, "foreachNode",
				CostForeachNodeProc,cd, NULL);

    Tcl_CreateCommand(interp, "content",  	CostContentProc, cd, NULL);
    Tcl_CreateCommand(interp, "setprop",  	CostSetpropProc, cd, NULL);
    Tcl_CreateCommand(interp, "unsetprop",  	CostUnsetpropProc, cd, NULL);

    Tcl_CreateCommand(interp, "specification", 	CostDefineSpecificationProc,
								cd,NULL);
    Tcl_CreateCommand(interp, "process",   	CostProcessProc, cd, NULL);
    Tcl_CreateCommand(interp, "eventHandler", 	CostDefineEventHandler,
								cd, NULL);
    Tcl_CreateCommand(interp, "selectNode", 	CostSelectNodeProc, cd, NULL);
    Tcl_CreateCommand(interp, "selectDocument",	CostSelectDocumentProc,cd,NULL);
    Tcl_CreateCommand(interp, "withDocument",	CostWithDocumentProc,cd,NULL);
    Tcl_CreateCommand(interp, "currentDocument",
					    CostCurrentDocumentProc,cd,NULL);
    /* Aliases for BT: */
	Tcl_CreateCommand(interp, "setstream",	CostSelectDocumentProc,cd,NULL);
	Tcl_CreateCommand(interp, "savestream",CostCurrentDocumentProc,cd,NULL);

    Tcl_CreateCommand(interp, "relation", 	CostRelationProc, cd, NULL);
    Tcl_CreateCommand(interp, "addlink", 	CostAddLinkProc, cd, NULL);

    /* Misc: */
    Tcl_CreateCommand(interp, "substitution", 	DefineSubstProc,NULL,NULL);
    Tcl_CreateCommand(interp, "environment",
			    DefineEnvironmentProc,NULL,NULL);

    /*
     * Load startup file:
     */
    Tcl_DStringInit(&startup_file);
    if (    (libdir = getenv("COSTLIB")) != NULL
         || (libdir = Tcl_GetVar(interp, "COSTLIB", TCL_GLOBAL_ONLY)) != NULL 
       )
    {
	Tcl_DStringAppend(&startup_file, libdir, -1);
	Tcl_DStringAppend(&startup_file, "/", 1);
    } 
    Tcl_DStringAppend(&startup_file, "costinit.tcl", -1);
    status = Tcl_EvalFile(interp, Tcl_DStringValue(&startup_file));
    Tcl_DStringFree(&startup_file);

#if HAVE_TCL_PACKAGES
    Tcl_PkgProvide(interp, "Cost", COST_VERSION_MAJOR "." COST_VERSION_MINOR);
#endif
    return status;
}

/*EOF*/
