/* -*-C++-*-
 * ###################################################################
 *  Cpptcl - connecting C++ with Tcl
 * 
 *  FILE: "tcl_base.h"
 *                                    created: 16/10/97 {2:09:40 pm} 
 *                                last update: 17/12/97 {7:02:18 pm} 
 *  Author: Vince Darley
 *  E-mail: <darley@fas.harvard.edu>
 *    mail: Division of Engineering and Applied Sciences, Harvard University
 *          Oxford Street, Cambridge MA 02138, USA
 *     www: <http://www.fas.harvard.edu/~darley/>
 *  
 * ===================================================================
 * Copyright (c) 1997  Vince Darley
 *  
 * See the file "license.terms" for information on usage and 
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * ===================================================================
 *  Description: 
 * 
 *  History
 * 
 *  modified by  rev reason
 *  -------- --- --- -----------
 *  16/10/97 VMD 1.0 original
 * ###################################################################
 */

#ifndef _Cpptcl_tcl_base_
#define _Cpptcl_tcl_base_

#include <tcl.h>
#include "tcl_obj.h"
#include "cpptclInt.h"
#include "cpptcl_type.h"
#include "tcl_args.h"
#include "meta_type.h"
#include "tcl_interaction.h"
#include "cpptcl_private.h"

//@Section: Cpptcl library
//@Man: C interface between Tcl and C++ code
/**
 * -------------------------------------------------------------------------
 *	   
 * "Cpptcl_ParseObjectCommand", "Cpptcl_DeleteObject" --
 *	  
 *  These two functions are the only ones actually called by Tcl.
 *  They simply act as a go-between with the C++ class.  It would be nice
 *  if they were static member functions, but not all compilers let such
 *  functions have "C" linkage, so this is the most compatible bet.
 * -------------------------------------------------------------------------
 */
//@{
/// Tcl passes the command line of an meta_object 'clientData' to this function.
extern "C" Tcl_ObjCmdProc Cpptcl_ParseObjectCommand;

/// Tcl calls this when 'clientData' should be deleted.
extern "C" void Cpptcl_DeleteObject(ClientData clientData);
//@}

/* 
 * If there's no exception handling, you _must_ put
 * 'NO_EXCEPTIONS(arg,TCL_ERROR)' or 'NO_EXCEPTIONS(arg,0)'
 * after 'arg >> done;', as	 appropriate.
 */
#ifdef NO_EXCEPTION_HANDLING
#define throw(x)
#define NO_EXCEPTIONS(a) if(a.state()) return a.state();
#else
#define NO_EXCEPTIONS(a)
#endif

//@Section: Cpptcl library
/// Useful define to simplify code
#define DONE(a) a >> done; NO_EXCEPTIONS(a);

//@Section: Cpptcl library
//@Man:
/** 
 * -------------------------------------------------------------------------
 *   
 * "class tcl_base" --
 *  
 *   Common base class shared by real objects ('tcl_object') and 
 *   meta-objects ('tcl_class').  
 *   
 *   Any tcl command implmented in C++ ultimately derives from this
 *   class, although most functionality is embedded in either of the
 *   aforementioned classes from one of which it must derive.
 *   
 * --Version--Author------------------Changes-------------------------------  
 *    1.0     <darley@fas.harvard.edu> original
 * -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
class tcl_base : public tcl_interaction {
	friend Tcl_ObjCmdProc Cpptcl_ParseObjectCommand;
	friend class meta_object;
  public:
	
	/// We need to know our interpreter and the name of our tcl command
	tcl_base(tcl_obj&, Tcl_Obj* name);
	/// We need to know our interpreter and the name of our tcl command
	tcl_base(tcl_obj&, const char* name);
	/// Although these can be read from an arg list if desired.
	tcl_base(tcl_args& arg);

	/// Destroys my associated command in the Tcl interpreter
  	virtual ~tcl_base(void);

	/// To be supplied by derived classes
   /** 
	* Derived classes should supply	this function. It looks through 
	* the arguments	and	does whatever it wants.	 The intention is for 
	* parse_tcl_command	to decide which	member function	to call, 
	* prepare the arguments	to that	member function	(perhaps 
	* converting strings to	ints or	suchlike), then	call the member	
	* function to do the actual	work.  #parse_tcl_command# should just 
	* be a parser, as the name implies.
	*/
    virtual int parse_tcl_command(tcl_args& arg)=0;
	///
    int parse_meta_commands(tcl_obj&, tcl_args& arg);
		
    /// Compares the next argument with any possible method
	/** 
	 * Most often used internally by 'member<type> i; arg >> i'
	 */
	const cpp_mem* configuration_option(tcl_args& arg, const meta_object& o,
										const char* member_type=0) {
		return o.configuration_option(arg,member_type);
	}
	///
	const cpp_mem* find_member(const cpp_private& data, const meta_object& o,
				const char* member_type) const {
		return o.find_member(data,member_type);
	}
							   	
	
    /// query our stream
    tcl_obj& get_tcl_obj(void) const  { return tcl_;}
    
	/// query the interpreter we were defined under
    Tcl_Interp* get_interp(void) const  { return tcl_.interpreter();}
	///
	Cpptcl_BaseClass(tcl_base);
	///
	static void update_metaobject(tcl_obj& i, cpptcl_metaobject_fn f=0);
	///
	virtual tcl_object* the_object(void) {return (tcl_object*)0;}
	///
	tcl_obj& tcl_command(tcl_obj& o) const;
	///
	bool embedded(void) const {return _embedded;}
	///
    char* char_tcl_command(void) const {
    	return (embedded() ? 0 : Tcl_GetCommandName(tcl_,cmd_info_));
	}
    Tcl_Command get_cmd_info(void) const {
    	return (embedded() ? NULL : cmd_info_);
	}
	/// Most of these objects are toplevel (hence no container)
	tcl_object* objcontainer(void) const {return 0;}
  protected:
	/// stream to evaluate tcl commands
    tcl_obj& tcl_;
	///
	bool _embedded;
    /// Copy constructor
    tcl_base(const tcl_base&);
  private:
	///
	void change_name_to(Tcl_Obj* newname);
	/// Tell tcl to create a new command bound to this object
	/** 
	 * Tcl uses	an approach	of automatically over-riding commands.
	 * So if you call this procedure with a	command	name that already
	 * exists, the old command is deleted.	It would be	easy to
	 * implement a check here, but in Tcl fashion that is left to
	 * the caller (use 'if	{ [info	commands <name>] ==	<name> } ...')
	 */
	Tcl_Command make_tcl_command(const char *cmd) {
		cmd_info_ = Tcl_CreateObjCommand(tcl_, (char*)cmd,
							 Cpptcl_ParseObjectCommand, (ClientData) this,
							 Cpptcl_DeleteObject);
		return cmd_info_;
	}
	/// Tell tcl to create a new command bound to this object
	Tcl_Command make_tcl_command(Tcl_Obj* cmd) {
		register int len;
		return make_tcl_command(Tcl_GetStringFromObj(cmd,&len));
	}	
#ifdef CPPTCL_USE_SCOPED_OBJECTS
	bool tracing_for_scope;
#endif
};

//@Section: Cpptcl library
/// Find a Tcl meta_object with a given name, or return zero.
/** 
 * -------------------------------------------------------------------------
 *       
 * "Cpptcl_getObjectByName" --
 *      
 *  A utility function that returns a tcl_object given the name of a tcl 
 *  command (or NULL if no such command exists).  
 * -------------------------------------------------------------------------
 */
tcl_base* Cpptcl_getObjectByName(Tcl_Interp* interp, const char* name);
tcl_base* Cpptcl_getObjectByName(Tcl_Interp* interp, Tcl_Obj* name);

/* 
 * Copy a string
 */
char* mystrdup(const char* source);
char* mystrdup(char* &to, int& len, const char* source);

#include "tcl_base.icc"

#endif


