/* -*-C++-*-
 * ############################################################################
 *	Cpptcl - Integrating C++ with Tcl								
 *	
 *	FILE: "cpptcl_metaobject.cc"
 *					   created:	19/7/96 {7:23:14 pm}
 *					   last update: 05/21/98 {14:55:19 PM}
 *	  by: Vince	Darley
 *			 E-mail: darley@fas.harvard.edu
 *			   mail:   Divison of Applied Sciences,	Harvard	University
 *					Cambridge MA 02138
 * 
 *	See	header file	for	further	information
 * 
 * ############################################################################
 */

#include "cpptcl_metaobject.h"
#include "cpptcl_subcommands.h"
#include "meta_object.h"
#include "meta_type.h"
#include "cpptcl_member_object.h"
#include "cpptcl_data_members.h"
#include "cpptcl_fn_members.h"

const meta_object& cpptcl_metaobject::new_object_type(const meta_object& o, 
						      const meta_object& parent) {
    if(!o.parent_list.contains((meta_object*)&parent)) {
	o.parent_list.append((meta_object*)&parent);
    }
    if(!parent.contains((meta_object*)&o)) {
	((meta_object&)parent).append((meta_object*)&o);
    }
    o.check_for_members();
    if(o.instantiable() && attaching) {
	o.attach_to_tcl_interp(tcl_);
    }
    return o;
}

const char* cpptcl_metaobject::command_for(const char* type) const {
    const meta_object* o = find_meta_info(type);
    return (o ? o->tcl_command : 0 );
}

bool cpptcl_metaobject::is_of_type(const meta_object * o, const char* type) const {
    if(o->type == type)
		return true;
    else {
		if(o->parent_list.isNonEmpty())
			return is_of_type(o->parent_list.headConst().item(),type);
		else
			return false;
    }
}

cpx_type cpptcl_metaobject::find_type(const char* type_name) const {
    // we must loop through everyone, trying 
    // to find a type which looks the same.
    const meta_object* o = find_meta_info(type_name);
    return (o ? o->type : 0);
}

cpptcl_metaobject::~cpptcl_metaobject(void) {
    // we used to have to delete the base_object since it was the only
    // copy of a pointer.  But now it's just the pointer to a real object
    // which sits inside another object and will be deleted naturally.
    // Hence we don't do:
    //     if(base_object) delete base_object;
    // but rather:
    if(base_object) {
    	base_object = 0;
    }
}

const meta_object* cpptcl_metaobject::find_meta_info(const char* parent_name,
		const meta_object* from) const {
/* 
 * Note	this first comparison could	just be	'==' not 'strcmp' except
 * we don't	need speed here, but more importantly, 'find_type' needs a 
 * strcmp, so we just use the one function for both tasks.
 */
	if(from == 0) {
		from = base_object;
		if(from == 0) return 0;
	}
    if(!strcmp(parent_name,from->type)) {
		return from;
	} else  {
		for(list_pos<meta_object*> scan = from->headConst();scan;++scan) {
	    	if(const meta_object* ch = find_meta_info(parent_name,scan.item()))
				return ch;
		}
		return 0;
    }
}

tcl_obj cpptcl_metaobject::list_descendants(const meta_object* o) const {
	tcl_obj t;
	for (list_pos<meta_object*> p = o->headConst(); p; ++p) {
		if (p.item()->length()) {
			tcl_obj t2;
			t2 << p.item()->type << lappend 
			   << list_descendants(p.item()) << lappend;
			t << t2 << lappend;
		} else {
			t << p.item()->type << lappend;
		}			
	}
	return t;
}

void cpptcl_metaobject::attach_all_objects(bool attach, const meta_object* o) const {
	if(!o) o = base_object;
	o->attach_to_tcl_interp(tcl_,attach);
	for (list_pos<meta_object*> p = o->headConst(); p; ++p) {
		attach_all_objects(attach,p.item());
	}
}

void cpptcl_metaobject::list_members(const meta_object *o, tcl_obj& t, 
		const char* member_type) const {
	for(int i = 0; i< o->member_info_size(); i++) {
		if(!member_type || is_of_type(o->member_info()[i].type(),member_type)) {
			t << o->member_info()[i].name() << lappend;
		}
	}
	for (list_pos<meta_object*> p = o->parent_list.headConst(); p; ++p) {
		list_members(p.item(),t,member_type);
	}
}

int cpptcl_metaobject::parse_tcl_command(tcl_args& arg){	
    if (arg("?fromType?","list of direct descendants")=="listTypes") {
		// Note this could return nothing at all
		const meta_object* o = base_object;
		arg >> optional >> o >> done;
		NO_EXCEPTIONS(arg);		
		tcl_ << *o << result;
		return TCL_OK;
    } else if (arg("type","list all descendant types")=="listDescendants") {
		// Note this could return nothing at all
		const meta_object* o = base_object;
		arg >> optional >> o >> done;
		NO_EXCEPTIONS(arg);
		tcl_ << list_descendants(o) << result;
		return TCL_OK;
    } else if (arg("type","test if this cpx_type has further derived types")
    			=="hasDescendants") {
	  	const meta_object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg);
		tcl_ << (o->length() != 0) << result;
		return TCL_OK;
    } else if (arg("type","list parent types")=="listParents") {
		// Note this could return nothing at all
	  	const meta_object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg);
		// list parents of the meta_object whose type we're given
		tcl_ << o->parent_list << result;
		return TCL_OK;
    } else if (arg("type","get required container type")=="containerType") {
		// Note this could return nothing at all
	  	const meta_object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg);
		const meta_object* cont = o->needs_container();
		if(cont) {
			tcl_ << cont;
		}
		tcl_ << result;
		return TCL_OK;
    } else if (arg("type","list all ancestral types")=="listAncestry") {
		// Note this could return nothing at all
	  	const meta_object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg);
		// list parents of the meta_object whose type we're given
		while (o->parent_list.isNonEmpty()) {
			o = o->parent_list.headConst().item();
			tcl_ <<  o->type << lappend;
		};
		tcl_ << result;
		return TCL_OK;
    } else if (arg("type1 type2","tests whether one type descends from another")
    			=="isa") {
		cpx_type t1,t2;
		arg >> t1 >> t2 >> done;
		NO_EXCEPTIONS(arg);
		tcl_ << is_of_type(t1,t2) << result;
		return TCL_OK;
    } else if (arg("type","the creation command for an object of the given type")
    			=="commandFor") {
	  	const meta_object* o;
	  	arg >> o >> done;
		NO_EXCEPTIONS(arg);
		if(o->tcl_command) 
			tcl_ <<  o->tcl_command << result;
    	return TCL_OK;
    } else if (arg("type","create an object of the given type")
    			=="create") {
	  	const meta_object* meta;
	  	arg >> meta;
		NO_EXCEPTIONS(arg);
		if(meta->instantiable()) {
			int res = meta->parse_meta_commands(tcl_,arg);
			if(res == CPPTCL_NOT_HANDLED) {
				arg.haveErr = 0;
				return arg.no_match();
			} else {
				return res;
			}
		} else {
			arg >> done;
			NO_EXCEPTIONS(arg);
			tcl_ << "That object type cannot be created." << result;
		}
    	return TCL_OK;
    } else if (arg("","remove all object commands from the interpreter")
    			=="detach") {
	  	arg >> done;
		NO_EXCEPTIONS(arg);
		attaching = false;
		attach_all_objects(false);
    	return TCL_OK;
    } else if (arg("","add all object commands to the interpreter")
    			=="attach") {
	  	arg >> done;
		NO_EXCEPTIONS(arg);
		attaching = true;
		attach_all_objects(true);
    	return TCL_OK;
    } else if (arg("type member ?args?","query members")=="query") {
		ometa<tcl_object> m;
		arg >> m;
		NO_EXCEPTIONS(arg);
		cpp_mem* obj = (cpp_mem*) m.meta->configuration_option(arg,
			cpp_mem::_type.type);
		if(!obj)
			arg.no_match();
		NO_EXCEPTIONS(arg);
		int res = obj->parse_meta_commands(tcl_,arg);
		if(res == CPPTCL_NOT_HANDLED) {
			arg.haveErr = 0;
			return arg.no_match();
		} else {
			return res;
		}
    } else if (arg("type option ?args?","configure subcommands of a given object type")
    			=="ensemble") {
		ometa<tcl_object> m;
		arg >> m;
		NO_EXCEPTIONS(arg);
		list<cpptcl_subcommand*>& l = m.meta->sub_commands;
		if(arg("name proc ?syntax? ?help?","adds a subcommand to the given type")=="add") {
			Tcl_Obj * o1, *o2, *o3, *o4;
			o3 = o4 = 0;
			arg >> o1 >> o2 >> optional >> o3 >> optional >> o4 >> done;
			NO_EXCEPTIONS(arg);
			l.append(new cpptcl_subcommand(o1,o2,o3,o4));
			return tcl_;
		} else if(arg("name","contains the named subcommand")=="contains") {
			cpptcl_subcommand* sub = subcommand(arg,*(m.meta));
			arg >> done;
			NO_EXCEPTIONS(arg);
			if(!sub) {
				tcl_ << "no" << result;
			} else {
				tcl_ << (l.contains(sub) ? "yes" : "indirectly") << result;
			}
			return tcl_;
		} else if(arg("name","removes the named subcommand")=="remove") {
			cpptcl_subcommand* sub = subcommand(arg,*(m.meta));
			if(!sub) {
				return arg.no_match();
			}
			arg >> done;
			NO_EXCEPTIONS(arg);
			if(l.contains(sub)) {
				l.remove(sub);
				delete sub;
			} else {
				tcl_ << "That subcommand is attached to an ancestor object." 
					 << tcl_error;
			}
			return tcl_;
		} else if(arg("","list all attached subcommands")=="list") {
			arg >> done;
			NO_EXCEPTIONS(arg);
			for (list_pos<cpptcl_subcommand*> s = l.headConst(); s; ++s) {
				tcl_ << s.item()->subcommand
					 << " " << s.item()->associated_proc;
				if(s.item()->syntax)
					tcl_ << " " << s.item()->syntax;
				if(s.item()->help)
					tcl_ << " " << s.item()->help;
				tcl_ << endl;
			}
			tcl_ << result;
			return tcl_;
		} else {
			return arg.no_match();
		}
    } else  // if we don't recognize the command, see if our parent does
		return tcl_object::parse_tcl_command(arg);
}

const char* cpptcl_metaobject::_type = "metacontrol";

cpptcl_metaobject::cpptcl_metaobject(tcl_args& arg)
	:tcl_object(arg),attaching(true)
{
	initialise_meta_hierarchy();
	tcl_base::metaobject = this;
}

void cpptcl_metaobject::initialise_meta_hierarchy(void) {
	base_object = (meta_object*)&tcl_interaction::_type;
	
	new_object_type(tcl_base::_type,tcl_interaction::_type);
	new_object_type(cpp_mem::_type,tcl_interaction::_type);
	new_object_type(cpp_config_mem::_type,cpp_mem::_type);
	new_object_type(cpp_data_mem::_type,cpp_config_mem::_type);
	new_object_type(cpp_operator_mem::_type,cpp_mem::_type);
	new_object_type(cpp_fn_mem::_type,cpp_operator_mem::_type);
	new_object_type(cpp_object_mem::_type,cpp_config_mem::_type);
	new_object_type(cpp_objectbag_mem::_type,cpp_object_mem::_type);
	new_object_type(cpp_tclobject_mem::_type,cpp_object_mem::_type);
}

cpptcl_metaobject::cpptcl_metaobject(tcl_args& arg, cpptcl_metaobject* from)
	:tcl_object(arg),attaching(true)
{
	if(tcl_base::metaobject == 0) {
		initialise_meta_hierarchy();
	} else {
		assert(from == tcl_base::metaobject);
		base_object = from->base_object;
		from->base_object = 0;
		delete from;
	}
	tcl_base::metaobject = this;
}

