/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - Integrating C++ with Tcl
 * 
 *	FILE: "tcl_obj.h"
 *									  created: 29/12/96 {7:45:59 pm} 
 *								  last update: 05/06/98 {16:16:38 PM} 
 *	Author:	Vince Darley
 *	E-mail:	<darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 *	Description: 
 *	  Wrapper for a 'Tcl_Obj' struct which makes it a lot nicer
 *	  to use from C++.  You can assign and append to strings, lists
 *	  etc. in an efficient manner, yet need only use simple '=', '<<'
 *	  operators.
 * 
 * ===================================================================
 * Copyright (c) 1997  Vince Darley
 *  
 * See the file "license.terms" for information on usage and 
 * redistribution of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * ===================================================================
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	29/12/96 VMD 1.0 original
 * ###################################################################
 */


#ifndef _Cpptcl_tcl_obj_
#define _Cpptcl_tcl_obj_

#include <tcl.h>
#include "cpptclInt.h"
#include "tcl_strobj.h"
#include "olist.h"
class tcl_base;

//@Section: Cpptcl library
//@Man:
/** 
 * -------------------------------------------------------------------------
 *   
 * "tcl_obj" --
 *  
 *  Wrapper for Tcl_Obj structs to allow simple manipulation from C++.
 *  We store strings and Tcl_Obj's inside 'tcl_strobj' structures
 *  which don't care whether they contain a char* or a Tcl_Obj*.
 *  
 *  This class allows you to build up the contents of a Tcl_Obj piece
 *  by piece using 'append' and 'lappend' operators.  Furthermore that
 *  building is carried out lazily.  Every time we add something a pointer 
 *  to it is just added to a list we keep.  If you throw away the object,
 *  we never actually call the Tcl library at all, otherwise, when
 *  needed, we carry out the require Tcl_... calls and return the
 *  final string as needed.
 *  
 *  If the tcl_obj knows about a Tcl interpreter, various methods are
 *  available for manipulating results, evaluating scripts,...  If no
 *  interpreter is known, all these methods do nothing.  It is up to
 *  the caller to check in advance.
 * -------------------------------------------------------------------------
 */
DLL_IMPORT_EXPORT
class tcl_obj {
	//@Man: Manipulators
	//@{
	/// Add the last item as a list item
	friend tcl_obj& lappend(tcl_obj&);
	/// Append as a string item (only for backwards compatibility)
	friend tcl_obj& append(tcl_obj&);
	/// Discard my contents
    friend tcl_obj& discard(tcl_obj&);
	/// Add a new-line
    friend tcl_obj& endl(tcl_obj&);
	/// For backwards compatibility.  Does nothing.
    friend tcl_obj& ends(tcl_obj&);
	//@Man: Conversion manipulators
	//@{
	/// Convert my contents to a long integer
	friend tcl_obj& to_int(tcl_obj&);
	/// Convert my contents to a long floating-point value
	friend tcl_obj& to_double(tcl_obj&);
	/// Convert my contents to a boolean value
	friend tcl_obj& to_bool(tcl_obj&);
	//@}
	//@Man: Toggle list-mode on and off
	//@{
	/// List mode on: all << operators do lappend
	friend tcl_obj& list_mode_on(tcl_obj&);
	/// List mode off: all << operators append
	friend tcl_obj& list_mode_off(tcl_obj&);
	//@}
	//@Man: Start and end embedded '[...]' commands.  NOT CURRENTLY WORKING
	/**
	 * These aren't working yet, because I can't find an efficient way of
	 * implementing them.  Just insert '[', ']' explicitly.
	 */
	//@{
	/// Start embedded cmd. NOT CURRENTLY WORKING
	friend tcl_obj& cmd(tcl_obj&);
	/// End embedded cmd. NOT CURRENTLY WORKING
	friend tcl_obj& end_cmd(tcl_obj&);
	//@}
	//@Man: These only apply if we have an interpreter.
	//@{
	/// Evaluate my contents in my interpreter
	friend tcl_obj& eval(tcl_obj&);
	/// Set my contents to be the result in my interpreter
    friend tcl_obj& result(tcl_obj&);
	/// Set my contents to be the error result in my interpreter
    friend tcl_obj& tcl_error(tcl_obj&);
	/// Get the result of my interpreter and put it into me (discarding old value)
    friend tcl_obj& get_result(tcl_obj&);
	//@}
	//@}
    
  public:	
	//@Man: Creation and destruction
	//@{
    /// Creates an object which links with a given interpreter
	/** 
	 * An object like this can be used to return results, evalute scripts,...
	 * For instance	the	'tcl_' object contained	within all 'tcl_object's is	of
	 * this	form.
	 */
	tcl_obj(Tcl_Interp* interp, const Tcl_Obj* obj=0);
	/// Just a general object wrapper, unrelated to some interp/result/etc. 
	/** 
	 * If you try and use the 'result',	'eval',... manipulators	on an object
	 * like	this, they will	be ignored.
	 */
	tcl_obj(const Tcl_Obj* obj=0);
	/// Copies all relevant information
	tcl_obj(const tcl_obj&);
	virtual ~tcl_obj(void);
	//@}
	//@Man: Assignment operators
	//@{
	///
	const tcl_obj& operator= (const tcl_obj&);
	///
	const tcl_obj& operator= (const short&);
	///
	const tcl_obj& operator= (const int&);
	///
	const tcl_obj& operator= (const long&);
	///
	const tcl_obj& operator= (const float&);
	///
	const tcl_obj& operator= (const double&);
	///
	const tcl_obj& operator= (const char&);
	///
	const tcl_obj& operator= (const char* const&);
	///
	const tcl_obj& operator= (const bool&);
	//@}
	//@Man: Append to my contents
	//@{
	///
	tcl_obj& operator<< (const tcl_obj&);
	///
	tcl_obj& operator<< (const tcl_strobj& i) {lzobj.append(i); return *this;}
	///
	tcl_obj& operator<< (const short&);
	///
	tcl_obj& operator<< (const int& i) {lzobj.append(Tcl_NewIntObj(i));return *this;}
	///
	tcl_obj& operator<< (const long& i) {lzobj.append(Tcl_NewLongObj(i));return *this;}
	///
	tcl_obj& operator<< (const float&);
	///
	tcl_obj& operator<< (const double& i) {lzobj.append(Tcl_NewDoubleObj(i));return *this;}
	///
	tcl_obj& operator<< (const char* const& o) {lzobj.append(o);return *this;}	
	///
	tcl_obj& operator<< (const char& i) {
		lzobj.append(Tcl_NewStringObj((char*)&i,1));return *this;
	}
	///
	tcl_obj& operator<< (const bool& i) {lzobj.append(Tcl_NewBooleanObj(i));return *this;}
	///
	tcl_obj& operator<< (const Tcl_Obj* const& o) {lzobj.append((Tcl_Obj*)o);return *this;}
	//@}
	//@Man: binary operators
	//@{
	///
	void binary_append(const float* , int len);
	///
	void binary_append(const double* , int len);
	///
	void binary_append(const int* , int len);
	///
	void binary_append(const long* , int len);
	//@}
	/// If you know the length of a string, this is the fastest way
	void add_precise_string(const char* const& i, int len);
#if (TCL_MINOR_VERSION > 0)
	/// Add a binary string of a certain number of bytes
	void add_byte_array(const unsigned* const& bytes, int len) {
		lzobj.append(tcl_strobj(bytes,len));
	}
#endif
	/// Get the string value with length
	const char* get_precise_string(int& len) const;
	//@Man: Casts for ease of use
	//@{
	//operator const Tcl_Obj* () const {_make_up_to_date();return _obj;}
	//operator const Tcl_Interp* () const {return _interp;}
	///
	operator Tcl_Obj* () const {_make_up_to_date();return _obj;}
	///
	operator Tcl_Interp* () const {return _interp;}
	///
	operator int (void) const { return _status;}
	//@}
	tcl_obj& operator<< (tcl_obj& (*fn)(tcl_obj&));
	//@Man: Read from my value.  Can fail.
	//@{
	///
	tcl_obj& operator>> (long&);
	///
	tcl_obj& operator>> (int&);
	///
	tcl_obj& operator>> (bool&);
	///
	tcl_obj& operator>> (double&);
	//@}
	///
	int status(void) const {return _status;}
	///
	void reset(void) {_status = TCL_OK;}
	///
	bool ok(void) const { return (_status == TCL_OK ? true : false);}
	///
	Tcl_Interp* interpreter(void) const { return _interp;}
	///
	tcl_obj& ResetResult(void);
	///
	Tcl_Obj* result(void);
	///
	const Tcl_Obj* result(void) const;
	/// Reset to the empty string
	void clear_value(void);
	/// The use of this function is discouraged
	const char* str(void);
	
	/// Very useful procedure to find you package library files
	int PackageLibraryInit(const char* libVarName, const char* envVarName, 
	                       const char* pkgName, const char* pkgInitFile, 
	                       const char* version, 
                           const char* prettyPkgName =0,
	                       const char* compiledLocation=0);
	/** 
	 * It searches in all standard places, including resource forks	and
	 * extensions folder on	MacOS, and will	load the specified file	if
	 * successful.	The	parameters are as follows:
	 * 
	 *	   libVarName -- the Tcl variable in which to store	the	library	dir.
	 *	   envVarName -- the variable env(...) in which	to look
	 *	   pkgName -- the name of the package
	 *	   pkgInitFile -- file to source; usually pkgName.tcl
	 *	   version -- string representation	of the version
	 *	   prettyPkgName --	optional to	look nice in error messages
	 *	   compiledLocation	-- string value	of a compiler flag if desired
	 * 
	 * For example,	this package uses the following	call:
	 * 
	 *	   tcl_.PackageLibraryInit("cpx::library","CPPTCL_LIBRARY","cpptcl",
	 *							   "cpptcl.tcl",CPPTCL_VERSION,"Cpptcl");
	 */
	/// Find or make the given namespace
	int MakeNamespace(const char* name);
	/// check list mode:
	bool list_mode(void) const {return (_list_mode != 0);}
  private:
	/// Interpreter, may be zero.
	Tcl_Interp* _interp;
	///
	mutable olist<tcl_strobj> lzobj;
	///// Wrapped object.  Don't read directly, rather use the 'operator'
	mutable tcl_strobj _obj;
	/// Tcl status, used to store status of conversions and evaluations.
	int _status;
	/// if set then operator << always lappends
	mutable int _list_mode;
	/// Ensures the wrapped object is up to date, and the working object empty
	void _make_up_to_date(void) const { _make_up_to_date(_obj);}
	/// Ensures the wrapped object is up to date, and the working object empty
	void _make_up_to_date(tcl_strobj&) const;
	/// Discards the contents of the working object
	void _clear_working(void);
	//@Man: If I have an interpreter, these items are used by manipulators
	//@{
	/// Make my value the result of my Tcl interpreter.  Not to be used externally.
	void _push_obj_into_result(void);
	/// Extract my Tcl interpreter's result, and set me to its value.
	void pull_from_result(void);
	/// Get my value and push it into my Tcl interpreter's result.
	void push_into_result(void);
	//@}
	/// Add end of line character.  Virtual so we can derive from this class.
	virtual tcl_obj& endl(void);
  public:
	//@Man: These items are specific to the Cpptcl system
	//@{
	/// Find an object with the given name
	tcl_base* FindObject(const char* name) const;
	/// Find an object with the given name
	tcl_base* FindObject(Tcl_Obj* obj) const;
	/// Find an object with the given name, and given type
	tcl_base* FindObject(const char* name, const char* type) const;
	/// Find an object with the given name, and given type
	tcl_base* FindObject(Tcl_Obj* name, const char* type) const;
	//@}
	
  public:
	//@Man: Tcl library wrappers
	/**
	 * These wrappers provide a more standard C++ way of accessing some
	 * of Tcl's C library.  They are all inline so there is no overhead,
	 * and they try and use some of C++'s features like default arguments,
	 * better use of 'const' etc so that calling the library requires less
	 * hassle.
	 * 
	 * As time goes by, more of Tcl's library will be added here.
	 */
	//@{
  	///
	char* PkgRequire(const char* name, const char* version=NULL, int exact=0) {
		return Tcl_PkgRequire(_interp, (char*)name, (char*)version,exact);
	}
	///
	int PkgProvide(const char* name, const char* version) {
		return Tcl_PkgProvide(_interp, (char*)name, (char*)version);
	}
	//@}

};

// Pull in the inline functions
#include "tcl_obj.icc"

#include "list.h"
// Write a list-iterator element out to a tcl stream
template <class T>
inline tcl_obj& operator<< (tcl_obj& o, const list_pos<T*>& p) {
	return o << p.item();
}

// Write a list out into a tcl stream
template <class T> 
inline tcl_obj& operator<< (tcl_obj& o, const list<T*>& l) {
	for(list_pos<T*> p(l);p;++p){ 
		o << p << lappend;
	}
	return o;
}

#endif
