######################################################################
#
# program name: framesets.tcl
#   programmer: Cris A. Fugate
# date written: September 2, 1998 (wrote frames.tcl)
#      changed: September 28, 1998 (added floadf and fstoref to frames)
#      changed: November 25, 1998 (wrote framesets.tcl)
#      changed: February 10, 1999 (added fupdatef to frames,
#               added fsgetr, fsputr and fsmemberf to framesets)         
#      changed: April 16, 1999 (merged frames and framesets)
#
#  description: This program is an extension to the tcl scripting
#               language.  It provides a frame and frameset
#               mechanism which can be used to dynamically organize
#               and perform operations on values and procedures.
#
# Copyright (c) 1999 Cris A. Fugate
#
# Permission is hereby granted, free of charge, to any person obtaining 
# a copy of this software and associated documentation files (the 
# "Software"), to deal in the Software without restriction, including 
# without limitation the rights to use, copy, modify, merge, publish, 
# distribute, sublicense, and/or sell copies of the Software, and to 
# permit persons to whom the Software is furnished to do so, subject to 
# the following conditions:
#
# The above copyright notice and this permission notice shall be included 
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
# OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
# ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
# OTHER DEALINGS IN THE SOFTWARE. 
#
######################################################################       
#
#                              Variables
# 
# aname                        array name
# avalue                       array value
# cmp                          comparison flag
# created                      create flag
# elema                        element of a list
# dname                        demon type
# elcnt                        count of list elements
# executed                     execute flag
# fframes                      list of frames
# fh                           file handle
# fhbuf                        file handle buffer
# flist                        list of references in a frame 
# fname                        frame name
# fname1                       frame name
# fname2                       frame name
# found                        exist flag
# ftype                        facet type
# i                            loop variable
# lista                        first list to be processed 
# listb                        second list to be processed
# listx                        first temporary list
# listy                        second temporary list
# mlist                        list of framesets of which a frame 
#                              is a member
# name                         frameset name
# plist                        list of frames in a reference chain
# pname                        procedure name
# put                          put flag
# removed                      remove flag
# s                            list of frames in the frameset
# sname                        slot name
# sname2                       slot name
# x                            variable used in place of expression
# y                            variable used in place of expression
# <fname>(<ename>)             used in operations involving many elements
# <fname>(<fname>,set)         frames in a frameset
# <fname>(<fname>,slots)       slots in a frame
# <fname>(<sname>,<ftype>)     demon facet
# <fname>(<sname>,facets)      facets in a slot
# <fname>(<sname>,ifcreatem)   ifcreatem demon
# <fname>(<sname>,ifcreater)   ifcreater demon
# <fname>(<sname>,ifcreatev)   ifcreatev demon 
# <fname>(<sname>,ifexecm)     ifexecm demon
# <fname>(<sname>,ifexistm)    ifexistm demon
# <fname>(<sname>,ifexistr)    ifexistr demon
# <fname>(<sname>,ifexistv)    ifexistv demon
# <fname>(<sname>,ifgetm)      ifgetm demon
# <fname>(<sname>,ifgetr)      ifgetr demon
# <fname>(<sname>,ifgetv)      ifgetv demon
# <fname>(<sname>,ifputm)      ifputm demon
# <fname>(<sname>,ifputr)      ifputr demon
# <fname>(<sname>,ifputv)      ifputv demon
# <fname>(<sname>,ifref)       ifref demon
# <fname>(<sname>,ifremovem)   ifremovem demon
# <fname>(<sname>,ifremover)   ifremover demon
# <fname>(<sname>,ifremovev)   ifremovev demon
# <fname>(<sname>,method)      method facet
# <fname>(<sname>,ref)         reference facet
# <fname>(<sname>,value)       value facet
#
######################################################################   
#
#                              Procedures
#
# compress                     order and remove duplicates from a list
# equivalence                  determine is two lists are equivalent
# fcomparef                    compare slots of two frames
# fcompares                    compare two slots
# fcopyf                       make a copy of a frame
# fcopys                       make a copy of a slot in another frame
# fcreated                     create a demon facet 
# fcreatef                     create a frame
# fcreatefs                    create a frameset 
# fcreatem                     create a method facet
# fcreater                     create a reference facet
# fcreates                     create a slot
# fcreatev                     create a value facet
# fexecd                       directly execution a demon
# fexecm                       execute a method
# fexistd                      determine if a demon facet exists
# fexistf                      determine if a frame exists
# fexistm                      determine if a method facet exists
# fexistr                      determine if a reference facet exists
# fexistrx                     (same as fexistr without a demon call)
# fexists                      determine if a slot exists
# fexistv                      determine if a value facet exists
# ffilterf                     filter a frame based on another frame
# fgetd                        get the value of a demon facet
# fgetm                        get the value of a method facet
# fgetr                        get the value of a reference facet
# fgetv                        get the value of a value facet
# flistf                       get a list of existing frames
# flistr                       get a list of references in a frame
# flists                       get a list of slots for a frame
# flistt                       get a list of facet types for a slot
# floadf                       load a frame into memory
# floadfs                      load a frameset into memory
# fmergef                      merge slots of a frame into another frame
# fpathr                       get a list of frames in a reference chain
# fputd                        put a value into a demon facet
# fputm                        put a value into a method facet
# fputr                        put a value into a reference facet
# fputv                        put a value into a value facet
# fremoved                     destroy a demon facet
# fremovef                     destroy a frame
# fremovefs                    destroy a frameset
# fremovem                     destroy a method facet
# fremover                     destroy a reference facet
# fremoves                     destroy a slot
# fremovev                     destroy a value facet
# fscreated                    create a demon facet in a frameset
# fscreatem                    create a method facet in a frameset
# fscreater                    create a reference facet in a frameset
# fscreates                    create a slot in a frameset
# fscreatev                    create a value facet in a frameset
# fsexcludef                   exclude a frame from a frameset
# fsgetr                       get a value from a reference facet 
#                              in a frameset
# fsincludef                   include a frame in a frameset 
# fslistf                      get a list of frames in a frameset
# fsmemberf                    get list of framesets in which 
#                              a frame is a member
# fsputr                       put a value in a reference facet
#                              in a frameset
# fsremoved                    remove a demon facet from a frameset
# fsremovem                    remove a method facet from a frameset
# fsremover                    remove a reference facet from a frameset
# fsremoves                    remove a slot from a frameset
# fsremovev                    remove a value facet from a frameset
# fstoref                      store a frame on disk
# fstorefs                     store a frameset on disk
# fupdatef                     synchronize a frame based on another frame
# member                       determine if a value is a member of a list
# remove                       remove a value from a list
#

#
# compress - order and remove duplicates from a list
# modifies lista
#
proc compress lista {
    upvar $lista listx
    set listx [lsort $listx]
    set listy [lindex $listx 0]
    set elema $listy
    foreach i $listx {
	if {$elema != $i} {
	    lappend listy $i
	}
	set elema $i
    }
    set listx $listy
}

#
# equivalence - determine if two lists are equivalent
#
proc equivalence {lista listb} {
    set listx $lista
    set listy $listb
    compress listx
    compress listy
    if {$listx == $listy} {
	return 1
    } else {
	return 0
    }
}

#
# member - determine if an element is a member of a list
#
proc member {lista elema} {
    set elcnt 0
    foreach i $lista {
	if {$elema == $i} {
	    incr elcnt
	}
    }
    return $elcnt
}

#
# remove - remove all occurances of an element from a list
# modifies lista
#
proc remove {lista elema} {
    upvar $lista listx
    set listy {}
    foreach i $listx {
	if {$elema != $i} {
	    lappend listy $i
	}
    }
    set listx $listy
}

# initialize frames
set fframes {}

#
# fexistf - determine if a frame exists
#
proc fexistf fname {
    global fframes
    return [member $fframes $fname]
} 

#
# fcreatef - create a frame
# requires that fname() does not exist
# modifies fframes, fname(fname,slots)
#
proc fcreatef fname {
    global fframes
    if {![fexistf $fname]} {
	lappend fframes $fname
	uplevel \#0 "set $fname\($fname,slots) {}"
	return 1
    } else {
	return 0
    }
}

#
# fremovef - remove a frame
# requires that fname() exists
# modifies fframes,fname()
#
proc fremovef fname {
    global fframes
    if {[fexistf $fname]} {
	foreach i [uplevel \#0 "array names $fname"] {
	    uplevel \#0 "unset $fname\($i)"
	}
	remove fframes $fname
	return 1
    } else {
	return 0
    }

}

#
# flistf - return list of frames
#
proc flistf {} {
    global fframes
    return $fframes
}

#
# fcopyf - create a new frame based on another frame
# requires that frame1() exists
# modifies fframes,fname2()
#
proc fcopyf {fname1 fname2} {
    global fframes
    if {[fexistf $fname1]} {
	fremovef $fname2
	lappend fframes $fname2
	foreach i [uplevel \#0 "array names $fname1"] {
	    uplevel \#0 "set $fname2\($i) $$fname1\($i)"
	}
	return 1
    } else {
	return 0
    }
}

#
# fcomparef - determine if two frames are equivalent
# requires that fname1() and fname2() exist
#
proc fcomparef {fname1 fname2} {
    if {[fexistf $fname1] && [fexistf $fname2]} {
	set x [uplevel \#0 "set $fname1\($fname1,slots)"]
	set y [uplevel \#0 "set $fname2\($fname2,slots)"]
	if {[equivalence $x $y]} {
	    return 1
	} else {
	    return 0
	}
    } else {
	return 0
    }

}

#
# fmergef - merge slots of one frame into another other
# requires that fname1() and fname2() exist
# modifies fname2()
#
proc fmergef {fname1 fname2} {
    if {[fexistf $fname1] && [fexistf $fname2]} {
	set y [uplevel \#0 "set $fname2\($fname2,slots)"]
	foreach i [uplevel \#0 "array names $fname1"] {
	    if {$i != "$fname1,set" && $i != "$fname1,slots"} {
		scan $i "%\[^,]" sname
		if {![member $y $sname]} {
		    uplevel \#0 "set $fname2\($i) $$fname1\($i)"
		    uplevel \#0 "lappend $fname2\($fname2,slots) $sname"
		}
	    }
	}
	return 1
    } else {
	return 0
    }
}

#
# floadf - load a frame into memory
# requires that fname() exists on disk, but not in memory
#
proc floadf fname {
    global fframes
    if {[file exists $fname] && ![fexistf $fname]} {
	lappend fframes $fname
	set fh [open $fname r]
	while {![eof $fh]} {
	    gets $fh fhbuf
	    set aname [lindex $fhbuf 0]
	    set avalue [remove fhbuf $aname]
	    uplevel \#0 "set $fname\($aname) {$avalue}"
	}
	close $fh
	return 1
    } else {
	return 0
    }
}

#
# fstoref - store a frame on disk
# requires that fname() exists
#
proc fstoref fname {
    if {[fexistf $fname]} {
	set fh [open $fname w]
	foreach i [uplevel \#0 "array names $fname"] {
	    set avalue [uplevel \#0 "set $fname\($i)"]
	    puts $fh "$i $avalue"
	}
	close $fh
	return 1
    } else {
	return 0
    }
}

#
# fupdatef - update structure of a frame from another frame
# requires that both frames exist
# modifies frame2()
#
proc fupdatef {fname1 fname2} {
    if {[fexistf $fname1] && [fexistf $fname2]} {
	uplevel \#0 "set $fname2\($fname2,slots) $$fname1\($fname1,slots)"
	foreach i [uplevel \#0 "array names $fname2"] {
	    if {$i != "$fname2,set" && $i != "$fname2,slots"} {
		if {![uplevel \#0 "info exists $fname1\($i)"]} {
		    uplevel \#0 "unset $fname2\($i)"
		}
	    }
	}
	foreach i [uplevel \#0 "array names $fname1"] {
	    if {$i != "$fname1,set" && $i != "$fname1,slots"} {
		if {![uplevel \#0 "info exists $fname2\($i)"]} {
		    uplevel \#0 "set $fname2\($i) $$fname1\($i)"
		}
	    } 
	}
	return 1
    } else {
	return 0
    }
}

#
# ffilterf - filter slots of a frame based on another frame
# requires that both frames exist
# modifies frame2()
# 
proc ffilterf {fname1 fname2} {
    if {[fexistf $fname1] && [fexistf $fname2]} {
	foreach i [uplevel \#0 "array names $fname2"] {
	    if {$i != "$fname2,set" && $i != "$fname2,slots"} {
		if {![uplevel \#0 "info exists $fname1\($i)"]} {
		    uplevel \#0 "unset $fname2\($i)"
		}
	    }
	}
	return 1
    } else {
	return 0
    } 
}

#
# fexists - determine if a slot exists
# requires that fname() exists
#
proc fexists {fname sname} {
    if {[fexistf $fname]} {
	if {[uplevel \#0 "member $$fname\($fname,slots) $sname"]} {
	    return 1
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

#
# fcreates - create a slot
# requires that fname() exists
# modifies fname(fname,slot),fname(sname,facets)
#
proc fcreates {fname sname} {
    if {[fexistf $fname]} {
	if {[uplevel \#0 "member $$fname\($fname,slots) $sname"] == 0} {
	    uplevel \#0 "lappend $fname\($fname,slots) $sname"
	    uplevel \#0 "set $fname\($sname,facets) {}"
	    return 1
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

#
# fremoves - remove a slot
# requires that fname(sname,facets) exists
# modifies fname(fname,slots),fname(sname,)
#
proc fremoves {fname sname} {
    if {[fexists $fname $sname]} {
	foreach i [uplevel \#0 "array names $fname"] {
	    scan $i "%\[^,]" sname2
	    if {$sname == $sname2} {
		uplevel \#0 "unset $fname\($i)"
	    }
	}
	uplevel \#0 "remove $fname\($fname,slots) $sname"
	return 1
    } else {
	return 0
    }
}

#
# flists - list slots of a frame
# requires that fname() exists
#
proc flists fname {
    if {[fexistf $fname]} {
	return [uplevel \#0 "set $fname\($fname,slots)"]
    } else {
	return {}
    }
}

#
# fcopys - copy a slot into another frame
# requires that fname1() and fname2() exist
# modifies fname2(sname,)
#
proc fcopys {fname1 sname fname2} {
    if {[fexists $fname1 $sname] && [fexistf $fname2]} {
	if {[uplevel \#0 "member $$fname2\($fname2,slots) $sname"] == 0} {
	    uplevel \#0 "lappend $fname2\($fname2,slots) $sname"
	 }
	foreach i [uplevel \#0 "array names $fname1"] {
	    scan $i "%\[^,]" sname2
	    if {$sname == $sname2} {
		uplevel \#0 "set $fname2\($i) $$fname1\($i)"
	    }
	}
	return 1
    } else {
	return 0
    }
}

#
# fcompares - compare a slot in two frames
# requires that fname1(sname,facets) and fname2(sname,facets) exist
#
proc fcompares {fname1 sname fname2} {
    set cmp 1
    if {[fexists $fname1 $sname] && [fexists $fname2 $sname]} {
	set x [uplevel \#0 "set $fname1\($sname,facets)"]
	set y [uplevel \#0 "set $fname2\($sname,facets)"]
	if {[equivalence $x $y]} {
	    foreach i [uplevel \#0 "array names $fname1"] {
		scan $i "%\[^,]" sname2
		if {$sname == $sname2} {
		    set x [uplevel \#0 "set $fname1\($i)"]
		    set y [uplevel \#0 "set $fname2\($i)"]
		    if {$x != $y} {
			set cmp 0
		    }
		}		       
	    }
            return $cmp		    
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

#
# flistt - list of facet types in a slot
# requires that fname(sname,facets) exists
#
proc flistt {fname sname} {
    if {[fexists $fname $sname]} {
	return [uplevel \#0 "set $fname\($sname,facets)"]
    } else {
	return {}
    }
}

#
# fexistrx - determine if a reference facet exists (internal)
# requires that fname(sname,facets) exists
# 
proc fexistrx {fname sname} {
    if {[fexists $fname $sname]} {
        if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
            return 1
	} else {
            return 0
	}
    } else {
	return 0
    }
}

# 
# fexistr - determine if a reference facet exists
# requires that fname(sname,facets) exists
# calls ifexistr demon
#
proc fexistr {fname sname} {
    if {[fexistrx $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ifexistr"]} {
	    uplevel \#0 "eval $$fname\($sname,ifexistr)"
        }
	return 1
    } else {
	return 0
    }
}

#
# fcreater - create a reference facet
# requires that fname(sname,facets) exists
# modifies fname(sname,facets),fname(sname,ref)
# calls ifcreater demon
#
proc fcreater {fname sname} {
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"] == 0} { 
	    set x [uplevel \#0 "member $$fname\($sname,facets) method"]
	    set y [uplevel \#0 "member $$fname\($sname,facets) value"]
	    if {!($x || $y)} {
		uplevel \#0 "lappend $fname\($sname,facets) ref"
		uplevel \#0 "set $fname\($sname,ref) {}"
		if {[uplevel \#0 "member $$fname\($sname,facets) ifcreater"]} {
		    uplevel \#0 "eval $$fname\($sname,ifcreater)"
		}
		return 1
	    } else {
		return 0
	    }
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

#
# fremover - remove a reference facet
# requires that fname(sname,ref) exists
# modifies fname(sname,facets),fname(sname,ref)
# calls ifremover demon
#
proc fremover {fname sname} {
    if {[fexistrx $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ifremover"]} {
	    uplevel \#0 "eval $$fname\($sname,ifremover)"
	}
	uplevel \#0 "unset $fname\($sname,ref)"
	uplevel \#0 "remove $fname\($sname,facets) ref"
	return 1
    } else {
	return 0
    }
}

#
# fgetr - get a value from a reference facet
# requires that fname(sname,ref) exists
# calls ifgetr demon
#
proc fgetr {fname sname} {
    if {[fexistrx $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ifgetr"]} {
	    uplevel \#0 "eval $$fname\($sname,ifgetr)"
	}
	return [uplevel \#0 "set $fname\($sname,ref)"]
    } else {
	return {}
    }
}

#
# fputr - put a value in a reference facet
# requires that fname(sname,ref) exists
# modifies fname(sname,ref)
# calls ifputr demon
#
proc fputr {fname1 sname fname2} {
    if {[fexistrx $fname1 $sname]} {
	uplevel \#0 "set $fname1\($sname,ref) $fname2"
	if {[uplevel \#0 "member $$fname1\($sname,facets) ifputr"]} {
	    uplevel \#0 "eval $$fname1\($sname,ifputr)"
	}
	return 1
    } else {
	return 0
    }
}

#
# flistr - list of references in a frame
# requires that fname() exists
# 
proc flistr fname {
    set flist {}
    if {[fexistf $fname]} {
	foreach i [uplevel \#0 "array names $fname"] {
	    scan $i "%\[^,],%s" sname ftype
	    if {$ftype == "ref"} {
		lappend flist $sname
	    }
	}
    }
    return $flist
}

#
# fpathr - return chain of references
# requires that fname(sname,facets) exists
#
proc fpathr {fname sname {plist {}}} {
    if {[fexists $fname $sname]} {
	if {[member $plist $fname] == 0} {
	    lappend plist $fname
	    if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
		set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
		fpathr $fname2 $sname $plist
	    } else {
		return $plist
	    }
	} else {
	    return $plist
	}
    } else {
	return $plist
    }
}

# 
# fexistm - determine if a method facet exists
# requires that fname(sname,facets) exists
# calls ifref and ifexistm demons
#
proc fexistm {fname sname} {
    set found 0
    if {[fexists $fname $sname]} {
	if {[fexistrx $fname $sname]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
            if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set found [fexistm $fname2 $sname]
	}
	if {[uplevel \#0 "member $$fname\($sname,facets) method"]} {
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifexistm"]} {
		uplevel \#0 "eval $$fname\($sname,ifexistm)"
	    }
	    set found 1
	}
    }
    return $found
}
    
#
# fcreatem - create a method facet
# requires that fname(sname,facets) exists
# modifies fname(sname,facets),fname(sname,method) where fname is
#          the original or referenced frame
# calls ifref and ifcreatem demons
#
proc fcreatem {fname sname} {
    set created 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) method"] ||
	    [uplevel \#0 "member $$fname\($sname,facets) value"]} {
	    set created 0
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
		set fname2 [uplevel \#0 "set $fname\($sname,ref)"] 
		if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		    uplevel \#0 "eval $$fname\($sname,ifref)"
		}
		set created [fcreatem $fname2 $sname]
	    } else {
		uplevel \#0 "set $fname\($sname,method) {}"
		uplevel \#0 "lappend $fname\($sname,facets) method"
		if {[uplevel \#0 "member $$fname\($sname,facets) ifcreatem"]} {
		    uplevel \#0 "eval $$fname\($sname,ifcreatem)"
		}
		set created 1
	    }
	}
    }
    return $created
}

# 
# fremovem - remove a method facet
# requires sthat fname(sname,facets) exists
# modifies fname(sname,facets),fname(sname,method) where fname is
#          the original or referenced frame
# calls ifref and ifremovem demons
#
proc fremovem {fname sname} {
    set removed 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set removed [fremovem $fname2 $sname]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) method"]} {
		if {[uplevel \#0 "member $$fname\($sname,facets) ifremovem"]} {
		    uplevel \#0 "eval $$fname\($sname,ifremovem)"
		}
		uplevel \#0 "unset $fname\($sname,method)"
		uplevel \#0 "remove $fname\($sname,facets) method"
		set removed 1
	    } 
	}
    }
    return $removed
}

#
# fexecm - execute a method
# requires that fname(sname,facets) exists
# calls ifref and ifexecm demons
#
proc fexecm {fname sname} {
    set executed 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"] 
	    if {[uplevel \#0 "member $fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set executed [fexecm $fname2 $sname]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) method"]} {
		if {[uplevel \#0 "member $$fname\($sname,facets) ifexecm"]} {
		    uplevel \#0 "eval $$fname\($sname,ifexecm)"
		}
		uplevel \#0 "eval $$fname\($sname,method)"
		set executed 1
	    }
	}
    }
    return $executed
}

# 
# fgetm - get a value from a method facet
# requires that fname(sname,facets) exists
# calls ifref and ifgetm demons
#
proc fgetm {fname sname} {
    set pname {}
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set pname [fgetm $fname2 $sname]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) method"]} {
		if {[uplevel \#0 "member $$fname\($sname,facets) ifgetm"]} {
		    uplevel \#0 "eval $$fname\($sname,ifgetm)"
		}
		set pname [uplevel \#0 "set $fname\($sname,method)"]
	    }
	}
    }
    return $pname
}

#
# fputm - put a value in a method facet
# requires that fname(sname,facets) exists
# modifies fname(sname,method) where fname is the original or
#          referenced frame
# calls ifref and ifputm demons
#
proc fputm {fname sname pname} {
    set put 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"] 
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set put [fputm $fname2 $sname $pname]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) method"]} {
		if {[uplevel \#0 "member $$fname\($sname,facets) ifputm"]} {
		    uplevel \#0 "eval $$fname\($sname,ifputm)"
		}
		uplevel \#0 "set $fname\($sname,method) $pname"
		set put 1
	    }
	}
    }
    return $put
}

#
# fexistv - determine if a value facet exists
# requires that fname(sname,facets) exists
# calls ifref and ifexistv demons
#
proc fexistv {fname sname} {
    set found 0
    if {[fexists $fname $sname]} {
	if {[fexistrx $fname $sname]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
            if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set found [fexistv $fname2 $sname]
	}
	if {[uplevel \#0 "member $$fname\($sname,facets) value"]} {
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifexistv"]} {
		uplevel \#0 "eval $$fname\($sname,ifexistv)"
	    }
	    set found 1
	}
    }
    return $found
}

# 
# fcreatev - create a value facet
# requires that fname(sname,facets) exists
# modifies fname(sname,facets),fname(sname,value) where fname is
#          the original or referenced frame
# calls ifref and ifcreatev demons
# 
proc fcreatev {fname sname} {
    set created 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) method"] ||
	    [uplevel \#0 "member $$fname\($sname,facets) value"]} {
	    set created 0
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
		set fname2 [uplevel \#0 "set $fname\($sname,ref)"] 
		if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		    uplevel \#0 "eval $$fname\($sname,ifref)"
		}
		set created [fcreatev $fname2 $sname]
	    } else {
		uplevel \#0 "set $fname\($sname,value) {}"
		uplevel \#0 "lappend $fname\($sname,facets) value"
		if {[uplevel \#0 "member $$fname\($sname,facets) ifcreatev"]} {
		    uplevel \#0 "eval $$fname\($sname,ifcreatev)"
		}
		set created 1
	    }
	}
    }
    return $created
}

#
# fremovev - remove a value facet
# requires that fname(sname,facets) exists
# modifies fname(sname,facets),fname(sname,value) where fname is
#          the original or referenced frame
# calls ifref and ifremovev demons
#
proc fremovev {fname sname} {
    set removed 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set removed [fremovev $fname2 $sname]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) value"]} {
		if {[uplevel \#0 "member $$fname\($sname,facets) ifremovev"]} {
		    uplevel \#0 "eval $$fname\($sname,ifremovev)"
		}
		uplevel \#0 "unset $fname\($sname,value)"
		uplevel \#0 "remove $fname\($sname,facets) value"
		set removed 1
	    } 
	}
    }
    return $removed
}

#
# fgetv - get a value from a value facet
# requires that fname(sname,facets) exists
# calls ifref and ifgetv demons
#
proc fgetv {fname sname} {
    set pname {}
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"]
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set pname [fgetv $fname2 $sname]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) value"]} {
		if {[uplevel \#0 "member $$fname\($sname,facets) ifgetv"]} {
		    uplevel \#0 "eval $$fname\($sname,ifgetv)"
		}
		set pname [uplevel \#0 "set $fname\($sname,value)"]
	    }
	}
    }
    return $pname
}

#
# fputv - put a value in a value facet
# requires that fname(sname,facets) exists
# modifies fname(sname,value) where fname is the original or
#          referenced frame
# calls ifref and ifputv demons
#
proc fputv {fname sname value} {
    set put 0
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) ref"]} {
	    set fname2 [uplevel \#0 "set $fname\($sname,ref)"] 
	    if {[uplevel \#0 "member $$fname\($sname,facets) ifref"]} {
		uplevel \#0 "eval $$fname\($sname,ifref)"
	    }
	    set put [fputv $fname2 $sname $value]
	} else {
	    if {[uplevel \#0 "member $$fname\($sname,facets) value"]} {
		uplevel \#0 "set $fname\($sname,value) $value"
		if {[uplevel \#0 "member $$fname\($sname,facets) ifputv"]} {
		    uplevel \#0 "eval $$fname\($sname,ifputv)"
		}
		set put 1
	    }
	}
    }
    return $put
}

#
# fexistd - determine if a demon facet exists
# requires that fname(sname,facets) exists
# 
proc fexistd {fname sname dname} {
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) $dname"]} {
	    return 1
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

# 
# fcreated - create a demon facet
# requires that fname(sname,facets) exists
# modifies fname(sname,facets),fname(sname,dname)
# 
proc fcreated {fname sname dname} {
    if {[fexists $fname $sname]} {
	if {[uplevel \#0 "member $$fname\($sname,facets) $dname"] == 0} {
	    uplevel \#0 "set $fname\($sname,$dname) {}"
	    uplevel \#0 "lappend $fname\($sname,facets) $dname"
	    return 1
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

# 
# fremoved - remove a demon facet
# requires that fname(sname,dname) exists
# modifies fname(sname,facets),fname(sname,dname)
#
proc fremoved {fname sname dname} {
    if {[fexistd $fname $sname $dname]} {
	uplevel \#0 "unset $fname\($sname,$dname)"
	uplevel \#0 "remove $fname\($sname,facets) $dname"
	return 1
    } else {
	return 0
    }
}

#
# fgetd - get a value from a demon facet
# requires that fname(sname,dname) exists
#
proc fgetd {fname sname dname} {
    if {[fexistd $fname $sname $dname]} {
	return [uplevel \#0 "set $fname\($sname,$dname)"]
    } else {
	return {}
    }
}

#
# fputd - put a value in a demon facet
# requires that fname(sname,dname) exists
# modifies fname(sname,dname)
#
proc fputd {fname sname dname pname} {
    if {[fexistd $fname $sname $dname]} {
	uplevel \#0 "set $fname\($sname,$dname) $pname"
	return 1
    } else {
	return 0
    }
}

#
# fexecd - directly execute a demon
# requires that fname(sname,dname) exists
#
proc fexecd {fname sname dname} {
    if {[fexistd $fname $sname $dname]} {
	uplevel \#0 "eval $$fname\($sname,$dname)"
	return 1
    } else {
	return 0
    }
}

#
# fcreatefs - create a frameset
# requires that name() does not exist
# modifies fframes, name(name,set), name(name,slots)
#
proc fcreatefs {name} {
    global fframes
    if {![fexistf $name]} {
	lappend fframes $name
	uplevel \#0 "set $name\($name,set) {}"
	uplevel \#0 "set $name\($name,slots) {}"
	return 1
    } else {
	return 0
    }
}

# 
# fremovefs - remove a frameset
# requires that name() exists
# modifies fframes, name()
#
proc fremovefs {name} {
    if {[fremovef $name]} {
	return 1
    } else {
	return 0
    }
}

#
# fslistf - return a list of frames in a frameset
# requires that name() exists
#
proc fslistf {name} {
    if {[fexistf $name]} {
	return [uplevel \#0 "set $name\($name,set)"]
    } else {
	return {}
    }
}

#
# floadfs - load a frameset into memory
# requires that name() exist on disk, but not in memory
#
proc floadfs {name} {
    if {[floadf $name]} {
	set s [fslistf $name]
	foreach i $s {
	    floadf $i 
	}
	return 1
    } else {
	return 0
    }
}

# 
# fstorefs - store a frameset on disk
# requires that name() exists
#
proc fstorefs {name} {
    if {[fstoref $name]} {
	set s [fslistf $name]
	foreach i $s {
	    fstoref $i
	}
	return 1
    } else {
	return 0
    }
}

#
# fsincludef - include a frame in a frameset
# requires that name() and fname() exist
# modifies name(name,set)
#
proc fsincludef {name fname} {
    if {[fexistf $name] && [fexistf $fname]} {
	uplevel \#0 "lappend $name\($name,set) $fname"
	return 1
    } else {
	return 0
    }
}

# 
# fsexcludef - exclude a frame from a frameset
# requires that name() exists
# modifies name(name,set)
#
proc fsexcludef {name fname} {
    if {[fexistf $name]} {
	if {[uplevel \#0 "member $$name\($name,set) $fname"]} {
	    uplevel \#0 "remove $name\($name,set) $fname"
	    return 1
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

# 
# fscreates - create a slot in a frameset
# requires that name() exists
# modifies name(name,slots), name(sname,facets), associated frames
#
proc fscreates {name sname} {
    if {[fcreates $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fcreates $i $sname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsremoves - remove a slot from a frameset
# requires that name(sname,facets) exists
# modifies name(name,slots), name(sname,), associated frames
#
proc sremoves {name sname} {
    if {[fremoves $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fremoves $i $sname
	}
	return 1
    } else {
	return 0
    }
}

# 
# fscreated - create a demon facet in a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,dname), associated frames
#
proc fscreated {name sname dname} {
    if {[fcreated $name $sname $dname]} {
	set s [fslistf $name]
	foreach i $s {
	    fcreated $i $sname $dname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsremoved - remove a demon facet from a frameset
# requires that name(sname,dname) exists
# modifies name(name,slots), name(sname,dname), associated frames
#
proc sremoved {name sname dname} {
    if {[fremoved $name $sname $dname]} {
	set s [fslistf $name]
	foreach i $s {
	    fremoved $i $sname $dname
	}
	return 1
    } else {
	return 0
    }
}

# 
# fscreatem - create a method facet in a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,method), associated frames
#
proc fscreatem {name sname} {
    if {[fcreatem $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fcreatem $i $sname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsremovem - remove a method facet from a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,method), associated frames
#
proc fsremovem {name sname} {
    if {[fremovem $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fremovem $i $sname
	}
	return 1
    } else {
	return 0
    }
}

# 
# fscreater - create a reference facet in a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,ref), associated frames
#
proc fscreater {name sname} {
    if {[fcreater $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fcreater $i $sname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsremover - remove a reference facet from a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,ref), associated frames
#
proc fsremover {name sname} {
    if {[fremover $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fremover $i $sname
	}
	return 1
    } else {
	return 0
    }
}

# 
# fscreatev - create a value facet in a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,value), associated frames
#
proc fscreatev {name sname} {
    if {[fcreatev $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fcreatev $i $sname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsremovev - remove a value facet from of a frameset
# requires that name(sname,facets) exists
# modifies name(sname,facets), name(sname,value), associated frames
#
proc fsremovev {name sname} {
    if {[fremovev $name $sname]} {
	set s [fslistf $name]
	foreach i $s {
	    fremovev $i $sname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsputr - put a value in reference facet in a frameset
# requires that name(sname,facets) exists
# modifies the name(sname,ref)
#
proc fsputr {name sname fname} {
    if {[fexistr $name $sname]} {
	fputr $name $sname $fname
	set s [fslistf $name]
	foreach i $s {
	    fputr $i $sname $fname
	}
	return 1
    } else {
	return 0
    }
}

#
# fsgetr - get a value from a reference facet in a frameset
# requires that name(sname,ref) exists
# modifies nothing
#
proc fsgetr {name sname} {
    if {[fexistr $name $sname]} {
	set r [fgetr $name $sname]
	return $r
    } else {
	return ""
    }
}
    
#
# fsmemberf - get list of framesets in which a frame is a member
# requires that the frame exists
# modifies nothing
#
proc fsmemberf {name} {
    if {[fexistf $name]} {
	foreach i [flistf] {
	    if {[uplevel \#0 "info exists $i\($i,set)"]} {
		if {[member [uplevel \#0 "fslistf $i"] $name]} {
		    lappend mlist $i
		}
	    }
	}
	return $mlist
    } else {
	return {}
    }
}