/*
 * matmath.c
 *
 *	C-level matrix utilities
 *
 *  Questions, Bugs reports, and Creative direction can for now be sent to
 *  Sean Woods, yoda@drexel.edu
 *
 *  Copyright (c) 1999 Woods Design Services.
 *
 * See the file "odie.license" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "matrices.h"

/* General Case Matrix Routines */
double *Matrix_Create(int rows,int cols) {
	return (double *)Tcl_Alloc(sizeof(double) * rows * cols);
}

MATOBJ *Matrix_Copy(MATOBJ *sourceMat) {
	MATOBJ *destMat;
    int length;
    destMat = (MATOBJ *)ckalloc(sizeof(MATOBJ));
    memcpy(destMat,sourceMat,length);    
  	length = sizeof(double) * destMat->cols * destMat->rows;
    destMat->cells = (double *)ckalloc(length);
    memcpy(destMat->cells,sourceMat->cells,length);
    return destMat;
}

void Matrix_Identity(double *matrix,int cols) {
	register int i;
	for(i=0; i<cols*cols ;i++)
		matrix[i] = 0;
	for(i=0;i<cols;i++)
		matrix[i*cols + i] = 1;	
}

void Matrix_Fill(double *matrix,int rows,int cols, double value) {
	register int i;
	for(i=0;i<cols*rows;i++)
		matrix[i] = value;	
}

void Matrix_Negate(double *result, double *data,int rows,int cols) {
	register int i;
	for(i=0;i<rows*cols;i++)	
		result[i] = -data[i]; 
}

void Matrix_Recipricol(double *result, double *data,int rows,int cols) {
	register int i;
	for(i=0;i<rows*cols;i++)	
		result[i] = 1.0/data[i];
}

void Matrix_Scale(double *result, double *data,int rows,int cols,double scaler) {
	register int i;
	for(i=0;i<rows*cols;i++)
		result[i] = data[i] * scaler;
}

/* WARNING RESULT MUST ME DIFFERENT THAN DATA!!!*/
void Matrix_Multiply(double *result, double *a, double *b, int rows, int cols) {
	register int i,j,k,offset;
	for(i=0;i<rows;i++) {		
		for(j=0;j<cols;j++) {
			offset = i*cols + j;
			result[offset] = 0.0;
			for(k=0;k<rows;k++)
				result[offset] += a[i*cols+k] * b[k*rows+j];
		} 
	}
}

/* WARNING RESULT MUST ME DIFFERENT THAN DATA!!!*/
void Matrix_Transpose(double *result,double *data,int rows,int cols) {
	register int i,j;
	for(i=0;i<rows;i++) {		
		for(j=0;j<cols;j++) {
			result[j*cols + i] = data[i*rows + j];
		} 
	}
}

/* WARNING RESULT MUST ME DIFFERENT THAN DATA!!!*/
/* Compute the minors of a square matrix */
void Matrix_Adjoint(double *result,double *data,int rows) {
	int i,j,brows;
	double *transpose;
	double *buffer;
	
	transpose = Matrix_Create(rows,rows);
	Matrix_Transpose(transpose,data,rows,rows);

	for(i=0;i<rows;i++) {		
		// Calculate Box Size
		for(j=0;j<rows;j++) {
			if((i&1) ^ (j&1)) {
				transpose[i*rows + j] = -transpose[i*rows + j];
			}		
		}
	}

	brows = rows - 1;
	buffer = (double *)Tcl_Alloc(sizeof(double)*brows*brows);	
	
	/* Create partition buffer */
	for(i=0;i<rows;i++) {		
		for(j=0;j<rows;j++) {
			register int ii,ji;
			register int row,col;
			row = 0;
			col = 0;
			
			for(ii=0;ii<rows;ii++) {
				if(ii == i) continue;
				for(ji=0;ji<rows;ji++) {
					if(ji == j) continue;	
					buffer[row*brows + col] = transpose[ii*rows + ji];
					RINCR(col,brows);	
				}
				RINCR(row,brows);
			}
			result[i*rows + j] = Matrix_Determinate(buffer,brows);
		} 
	}
	Tcl_Free((char *)buffer);
	Tcl_Free((char *)transpose);
}

/* WARNING RESULT MUST ME DIFFERENT THAN DATA!!!*/
int Matrix_Inverse(double *result, double *data,int rows) {
	register double D;
	
	D = Matrix_Determinate(data,rows);
	if (D == 0.0)
	{
		Matrix_Error("Singular Matrix in Inverse");
		return 1;
	}
	Matrix_Adjoint(result,data,rows);
	Matrix_Scale(result,result,rows,rows,1.0/D);
	return 0;
}

/* Compute the minors of a square matrix */
double Matrix_Determinate(double *data,int rows) {
	register int k,len;
	register double result=0.0;
	//result = (double *)Tcl_Alloc(sizeof(double)*rows*rows);
	len = rows - 1;
	switch (rows) {
		case 1:	return *data;
		case 2: return data[0] * data[3] - data[1] * data[2];
		default:
			for(k=0;k<rows;k++) {
				register double sump = 1.0;	
				register double summ = 1.0;	
				register int l,bcol;
				bcol = k;
				for(l=0;l<rows;l++) {
					RINCR(bcol,rows);
					sump *= data[l*rows + bcol];
					summ *= data[(len - l)*rows + bcol];
				}
				result += sump;
				result -= summ;
			}
	}
	return result;
}

double *Matrix_Range(MATOBJ *data,int *range) {
	int i,j,rows,cols,offset;
	double *result;
	
	for(i=0;i<4;i++) {
		if(i & 1) {
			if((range[i] < 0) || (range[i] >= data->rows)) {
				range[i] = data->rows - 1;
			} 
		} else {
			if((range[i] < 0) || (range[i] >= data->cols)) {
				range[i] = data->cols - 1;
			} 
		}
	}
	//Ensure Positive indexes
	if(range[BOTTOM] < range[TOP]) 	SWAP(range[BOTTOM],range[TOP],offset);
	if(range[RIGHT] < range[LEFT]) SWAP(range[RIGHT],range[LEFT],offset);
	
	rows = range[BOTTOM] - range[TOP] + 1;
	cols = range[RIGHT] - range[LEFT] + 1;
	result = (double *) Tcl_Alloc(sizeof(double) * rows * cols);
	offset = 0;
	for(i=range[TOP];i<=range[BOTTOM];i++) {
		for(j=range[LEFT];j<=range[RIGHT];j++) {
			result[offset] = data->cells[i*data->cols + j];
			offset++;
		}
	}
	return result;
}

/* Produce a general case matrix from a TCL List */

int Matrix_FromObj(Tcl_Interp *interp, Tcl_Obj *listPtr,MATOBJ *matrix) {
	Tcl_Obj **rowPtrs;
    Tcl_Obj **elemPtrs;
    int result;
    int rows,cols;
    register int i,j;
    int len;
    double *data;

	/* Step one, Measure the matrix */
	result = Tcl_ListObjGetElements(interp, listPtr, &rows, &rowPtrs);
	if (result != TCL_OK) return result;
	result = Tcl_ListObjGetElements(interp, rowPtrs[0], &cols, &elemPtrs);
	if (result != TCL_OK) return result;
	/* Link what we have found so far */
	matrix->rows = rows;
	matrix->cols = cols;
	data = (double *)Tcl_Alloc(sizeof(double) * rows * cols);
	matrix->cells = data;

	for(i=0;i<rows;i++) {
		result = Tcl_ListObjGetElements(interp, rowPtrs[i], &len, &elemPtrs);
		if (result != TCL_OK) return result;
		if(len != cols) {
			Matrix_Error("Columns Not Uniform");	
			return TCL_ERROR;
		}
		for(j=0;j<len;j++) {
			double temp;
			result =  Tcl_GetDoubleFromObj(interp, elemPtrs[j], &temp);
			if (result != TCL_OK) return result;
			data[i*cols + j]=temp;
		}
	}

	return TCL_OK;
}


int Matrix_ToList(Tcl_Obj *dest, MATOBJ *matrix) {
	Tcl_Obj **row;
	Tcl_Obj **col;	
	int rows,cols;
	register int i,j;
	double *data;
	/* Step 1, dimension matrix */
	rows = matrix->rows;
	cols = matrix->cols;
	data = matrix->cells;
	row = (Tcl_Obj **)Tcl_Alloc(sizeof(Tcl_Obj *) * rows);
	col = (Tcl_Obj **)Tcl_Alloc(sizeof(Tcl_Obj *) * cols);
	for(i=0;i<rows;i++) {
		for(j=0;j<cols;j++) { 
			col[j] = Tcl_NewDoubleObj(data[i*cols + j]);
		}
		row[i] = Tcl_NewListObj(cols,col);				
	}
	Tcl_SetListObj(dest,rows,row);
	return TCL_OK;
}

int *Coords_FromObj(Tcl_Interp *interp, Tcl_Obj *listPtr, int *len) {
	Tcl_Obj **elements;
	int result,i,length,temp;
	int *data;
	result = Tcl_ListObjGetElements(interp, listPtr, &length, &elements);
	if (result != TCL_OK) return NULL;
	data = (int *) Tcl_Alloc(sizeof(int) * length);
	for(i-0;i<length;i++) {
		result = Tcl_GetIntFromObj(interp,elements[i],&temp);
		data[i] = temp;		
		if(result != TCL_OK) {
			if(strcmp(Tcl_GetStringFromObj(elements[i],len),"end") == 0) {
				data[i] = -1;
			} else return NULL;
		}
	}
	*len = length;
	return data;
}

/* rotate a series of vectors about a common origion 
*/
int Vector_BatchTransform(int vcount,double *veclist,double *rot,double *tran) {
	register int i,offset;
	double *point;
	TRANSFORM transform;
	Compute_Rotation_Transform(&transform,rot);
	
	offset = vcount * 3;
	for(i=0;i<offset;i+=3) {
		point = &veclist[i];
		VSubEq(point,tran);
 		MTransPoint(point,point,&transform);
		VAddEq(point,tran);
	}
	return TCL_OK;
}

double *Matrix_CramersRule(int rows,double *dataA, double *dataB) {
	register int h,i,j;
	double *temp,*result;
		
	result = (double *) Tcl_Alloc(sizeof(double) * rows);
	temp = (double *) Tcl_Alloc(sizeof(double) * rows * rows);
	
	for(h=0;h<rows;h++) {
		for(i=0;i<rows;i++) {		
			for(j=0;j<rows;j++) {
				if(j==h) {
					temp[i*rows + j] = dataB[i];	
				} else {
					temp[i*rows + j] = dataA[i*rows + j];
				}				
			} 
		}
		result[h] = Matrix_Determinate(temp,rows);
	}
	Tcl_Free((char *)temp);
	return result;
}