/*
 * matrices.h
 *
 *	Macros and structure used within the Odie matrix system
 *
 *  Questions, Bugs reports, and Creative direction can for now be sent to
 *  Sean Woods, yoda@drexel.edu
 *
 *  Copyright (c) 1999 Woods Design Services.
 *
 * See the file "odie.license" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */


#ifndef MATRICES_H
#define MATRICES_H

/* General case Matrix 
 * We will accomodate a system of up to 64 x 64 elements
*/

#define MATSTACKSIZE 64


/* Various numerical constants that are used in the calculations */
#ifndef EPSILON     /* A small value used to see if a value is nearly zero */
#define EPSILON 1.0e-10
#endif

#ifndef HUGE_VAL    /* A very large value, can be considered infinity */
#define HUGE_VAL 1.0e+17
#endif

/*
 * If the width of a bounding box in one dimension is greater than
 * the critical length, the bounding box should be set to infinite.
 */

#ifndef CRITICAL_LENGTH
#define CRITICAL_LENGTH 1.0e6
#endif

#ifndef BOUND_HUGE  /* Maximum lengths of a bounding box. */
#define BOUND_HUGE 2.0e10
#endif

/*
 * These values determine the minumum and maximum distances
 * that qualify as ray-object intersections.
 */

#define Small_Tolerance 0.001
#define Max_Distance 1.0e7


#ifndef DBL_FORMAT_STRING
#define DBL_FORMAT_STRING "%lf"
#endif

#ifndef DBL
#define DBL double
#endif

#ifndef SNGL
#define SNGL float
#endif

#ifndef COLC
#define COLC float
#endif

#ifndef M_PI
#define M_PI   3.1415926535897932384626
#endif

#ifndef M_PI_2
#define M_PI_2 1.57079632679489661923
#endif

#ifndef TWO_M_PI
#define TWO_M_PI 6.283185307179586476925286766560
#endif

#ifndef M_PI_180
#define M_PI_180 0.01745329251994329576
#endif

#ifndef M_PI_360
#define M_PI_360 0.00872664625997164788
#endif

/* Some implementations of scanf return 0 on failure rather than EOF */
#ifndef SCANF_EOF
#define SCANF_EOF EOF
#endif

/* Get minimum/maximum of two values. */

#ifndef min
#define min(x,y) (((x)>(y))?(y):(x))
#endif

#ifndef max
#define max(x,y) (((x)<(y))?(y):(x))
#endif

/* Get minimum/maximum of three values. */

#define max3(x,y,z) (((x)>(y))?(((x)>(z))?(x):(z)):(((y)>(z))?(y):(z)))
#define min3(x,y,z) (((x)<(y))?(((x)<(z))?(x):(z)):(((y)<(z))?(y):(z)))

#ifndef labs      /* Absolute value of the long integer x. */
#define labs(x) (long) (((x)<0)?-(x):(x))
#endif

#ifndef fabs      /* Absolute value of the double x. */
#define fabs(x) ((x) < 0.0 ? -(x) : (x))
#endif


#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

#ifndef CONST       /* How to define a local variable - normally 'const' */
#define CONST
#endif

#ifndef CDECL
#define CDECL
#endif

#ifndef NEW_LINE_STRING
#define NEW_LINE_STRING "\n"
#endif

/* If compiler version is undefined, then make it 'u' for unknown */
#ifndef COMPILER_VER
#define COMPILER_VER ".u"
#endif

#ifndef QSORT
#define QSORT(a,b,c,d) qsort((a),(b),(c),(d))
#endif

typedef DBL UV_VECT [2];
typedef DBL VECTOR [3];
typedef DBL MATRIX [4][4];
typedef DBL EXPRESS [5];
typedef COLC COLOUR [5];
typedef COLC RGB [3];
typedef short WORD;

/* Vector array elements. */
#define U 0
#define V 1

#define X 0
#define Y 1
#define Z 2
#define T 3


/* Colour array elements. */

#define RED    0
#define GREEN  1
#define BLUE   2
#define FILTER 3
#define TRANSM 4

#define TMAT 0
#define TINV 1

#define R0C0	0x0
#define R0C1	0x1
#define R0C2	0x2
#define R0C3	0x3

#define R1C0	0x4
#define R1C1	0x5
#define R1C2	0x6
#define R1C3	0x7

#define R2C0	0x8
#define R2C1	0x9
#define R2C2	0xA
#define R2C3	0xB

#define R3C0	0xC
#define R3C1	0xD
#define R3C2	0xE
#define R3C3	0xF


#define LEFT 	0
#define TOP 	1
#define RIGHT 	2
#define BOTTOM	3

/* Macros to manipulate scalars, vectors, and colors. */

#define Destroy_Float(x)    if ((x)!=NULL) POV_FREE(x)

#define Assign_Vector(d,s)  memcpy((d),(s),sizeof(VECTOR))
#define Destroy_Vector(x)   if ((x)!=NULL) POV_FREE(x)

#define Assign_UV_Vect(d,s) memcpy((d),(s),sizeof(UV_VECT))
#define Destroy_UV_Vect(x)  if ((x)!=NULL) POV_FREE(x)

#define Assign_Colour(d,s)  memcpy((d),(s),sizeof(COLOUR))
#define Make_Colour(c,r,g,b) {(c)[RED]=(r);(c)[GREEN]=(g);(c)[BLUE]=(b);(c)[FILTER]=0.0;(c)[TRANSM]=0.0;}
#define Make_ColourA(c,r,g,b,a,t) {(c)[RED]=(r);(c)[GREEN]=(g);(c)[BLUE]=(b);(c)[FILTER]=(a);(c)[TRANSM]=t;}
#define Make_Vector(v,a,b,c) { (v)[X]=(a);(v)[Y]=(b);(v)[Z]=(c); }
#define Destroy_Colour(x) if ((x)!=NULL) POV_FREE(x)
#define Make_RGB(c,r,g,b) {(c)[RED]=(r);(c)[GREEN]=(g);(c)[BLUE]=(b);}

#define RINCR(a,b) {a++;if(a >= b) a = a % b;}
#define RDECR(a,b) {a--;if(a <  0) a = (b+a) % b;}
#define SWAP(a,b,temp) {temp = a; a = b; b = temp;}

typedef struct GenMatrix {
	int rows,cols;
	double *cells;
} MATOBJ;

typedef struct GenOpcode {
	int opcode;
	int operands;
	int arguments;
	unsigned int opForm:3;	/* Required form of operand */
} MATVERB;

/*****************************************************************************
 *
 * Transformation stuff.
 *
 *****************************************************************************/

typedef struct Transform_Struct TRANSFORM;

struct Transform_Struct
{
  MATRIX matrix;
  MATRIX inverse;
};

#define Destroy_Transform(x) if ((x)!=NULL) Tcl_Free((char *)x);

/*****************************************************************************
* Global preprocessor defines
******************************************************************************/




/*****************************************************************************
* Global typedefs
******************************************************************************/




/*****************************************************************************
* Global variables
******************************************************************************/




/*****************************************************************************
* Global functions
******************************************************************************/
/* Affine Routines */
void MZero (MATRIX result);
void MIdentity (MATRIX result);
void MTimes (MATRIX result, MATRIX matrix1, MATRIX matrix2);
void MAdd (MATRIX result, MATRIX matrix1, MATRIX matrix2);
void MSub (MATRIX result, MATRIX matrix1, MATRIX matrix2);
void MScale (MATRIX result, MATRIX matrix1, DBL amount);
void MTranspose (MATRIX result, MATRIX matrix1);
void MTransPoint (VECTOR result, VECTOR vector, TRANSFORM *trans);
void MInvTransPoint (VECTOR result, VECTOR vector, TRANSFORM *trans);
void MTransDirection (VECTOR result, VECTOR vector, TRANSFORM *trans);
void MInvTransDirection (VECTOR result, VECTOR vector, TRANSFORM *trans);
void MTransNormal (VECTOR result, VECTOR vector, TRANSFORM *trans);
void MInvTransNormal (VECTOR result, VECTOR vector, TRANSFORM *trans);
void Compute_Matrix_Transform (TRANSFORM *result, MATRIX matrix);
void Compute_Scaling_Transform (TRANSFORM *result, VECTOR vector);
void Compute_Inversion_Transform (TRANSFORM *result);
void Compute_Translation_Transform (TRANSFORM *transform, VECTOR vector);
void Compute_Rotation_Transform (TRANSFORM *transform, VECTOR vector);
void Compute_Look_At_Transform (TRANSFORM *transform, VECTOR Look_At, VECTOR Up, VECTOR Right);
void Compose_Transforms (TRANSFORM *Original_Transform, TRANSFORM *New_Transform);
void Compute_Axis_Rotation_Transform (TRANSFORM *transform, VECTOR V1, DBL angle);
void Compute_Coordinate_Transform (TRANSFORM *trans, VECTOR origin, VECTOR up, DBL r, DBL len);
TRANSFORM *Create_Transform (void);
TRANSFORM *Copy_Transform (TRANSFORM *Old);
VECTOR *Create_Vector (void);
DBL *Create_Float (void);
void MInvers (MATRIX r, MATRIX m);

double *Matrix_Create(int rows,int cols);
MATOBJ *Matrix_Copy(MATOBJ *sourceMat);
void Matrix_Multiply(double *result, double *a, double *b, int rows, int cols);
void Matrix_Negate(double *result, double *data,int rows,int cols);
void Matrix_Recipricol(double *result, double *data,int rows,int cols);
void Matrix_Scale(double *result, double *data,int rows,int cols,double scaler);
void Matrix_Transpose(double *result,double *data,int rows,int cols);

void Matrix_Identity(double *matrix,int cols);
void Matrix_Fill(double *matrix,int rows,int cols, double value);

/* General Case Routines */
double *Matrix_Range(MATOBJ *data,int *range);
double Matrix_Determinate(double *data,int rows);
double *Matrix_CramersRule(int rows,double *dataA, double *dataB);

int Matrix_Inverse(double *result, double *data,int rows);
void Matrix_Adjoint(double *result,double *data,int rows);

double *Vec_Rotation_Transform(double *vector);
void Vector_Rotate(double *result,double *vec,double *rot);
int Vector_BatchTransform(int vcount,double *veclist,double *rot,double *tran);

Tcl_Interp *MatInterp(void);

void Matrix_Error(char *error);

int Matrix_Init(Tcl_Interp *tinterp);
int Matrix_FromObj	(Tcl_Interp *interp, Tcl_Obj *listPtr,MATOBJ *matrix);
int Matrix_ToList	(Tcl_Obj *dest, MATOBJ *matrix);
void Matrix_ToObj(Tcl_Obj *dest, MATOBJ *matrix);
int *Coords_FromObj(Tcl_Interp *interp, Tcl_Obj *listPtr, int *len);
#endif

#ifndef VECTOR_H
#define VECTOR_H


/* Misc. Vector Math Macro Definitions */

extern DBL VTemp;

/* Vector Add */
#define VAdd(a, b, c) {(a)[X]=(b)[X]+(c)[X];(a)[Y]=(b)[Y]+(c)[Y];(a)[Z]=(b)[Z]+(c)[Z];}
#define VAddEq(a, b) {(a)[X]+=(b)[X];(a)[Y]+=(b)[Y];(a)[Z]+=(b)[Z];}

/* Vector Subtract */
#define VSub(a, b, c) {(a)[X]=(b)[X]-(c)[X];(a)[Y]=(b)[Y]-(c)[Y];(a)[Z]=(b)[Z]-(c)[Z];}
#define VSubEq(a, b) {(a)[X]-=(b)[X];(a)[Y]-=(b)[Y];(a)[Z]-=(b)[Z];}

/* Scale - Multiply Vector by a Scalar */
#define VScale(a, b, k) {(a)[X]=(b)[X]*(k);(a)[Y]=(b)[Y]*(k);(a)[Z]=(b)[Z]*(k);}
#define VScaleEq(a, k) {(a)[X]*=(k);(a)[Y]*=(k);(a)[Z]*=(k);}

/* Inverse Scale - Divide Vector by a Scalar */
#define VInverseScale(a, b, k) {(a)[X]=(b)[X]/(k);(a)[Y]=(b)[Y]/(k);(a)[Z]=(b)[Z]/(k);}
#define VInverseScaleEq(a, k) {(a)[X]/=(k);(a)[Y]/=(k);(a)[Z]/=(k);}

/* Dot Product - Gives Scalar angle (a) between two vectors (b) and (c) */
#define VDot(a, b, c) {a=(b)[X]*(c)[X]+(b)[Y]*(c)[Y]+(b)[Z]*(c)[Z];}

/* Cross Product - returns Vector (a) = (b) x (c) 
   WARNING:  a must be different from b and c.*/
#define VCross(a,b,c) {(a)[X]=(b)[Y]*(c)[Z]-(b)[Z]*(c)[Y]; \
                       (a)[Y]=(b)[Z]*(c)[X]-(b)[X]*(c)[Z]; \
                       (a)[Z]=(b)[X]*(c)[Y]-(b)[Y]*(c)[X];}

/* Evaluate - returns Vector (a) = Multiply Vector (b) by Vector (c) */
#define VEvaluate(a, b, c) {(a)[X]=(b)[X]*(c)[X];(a)[Y]=(b)[Y]*(c)[Y];(a)[Z]=(b)[Z]*(c)[Z];}
#define VEvaluateEq(a, b) {(a)[X]*=(b)[X];(a)[Y]*=(b)[Y];(a)[Z]*=(b)[Z];}

/* Divide - returns Vector (a) = Divide Vector (b) by Vector (c) */
#define VDiv(a, b, c) {(a)[X]=(b)[X]/(c)[X];(a)[Y]=(b)[Y]/(c)[Y];(a)[Z]=(b)[Z]/(c)[Z];}
#define VDivEq(a, b) {(a)[X]/=(b)[X];(a)[Y]/=(b)[Y];(a)[Z]/=(b)[Z];}

/* Simple Scalar Square Macro */
#define Sqr(a)  ((a)*(a))

/* Square a Vector (b) and Assign to another Vector (a) */
#define VSquareTerms(a, b) {(a)[X]=(b)[X]*(b)[X];(a)[Y]=(b)[Y]*(b)[Y];(a)[Z]=(b)[Z]*(b)[Z];}

/* Vector Length - returs Scalar Euclidean Length (a) of Vector (b) */
#define VLength(a, b) {a=sqrt((b)[X]*(b)[X]+(b)[Y]*(b)[Y]+(b)[Z]*(b)[Z]);}

/* Vector Distance - returs Scalar Euclidean Distance (a) between two
 * points/Vectors (b) and (c) */
#define VDist(a, b, c) {VECTOR tmp; VSub(tmp, b, c); VLength(a, tmp);}

/* Vector Distance Inverse Squares - returns the inverse
 * square of the distanc between two
 * points/Vectors (b) and (c) */
#define VDistInvSqr(a, b, c) {VECTOR tmp; VSub(tmp, b, c); a = 1/(tmp[X]*tmp[X] + tmp[Y]*tmp[Y] + tmp[Z] * tmp[Z]);}

/* Normalize a Vector - returns a vector (length of 1) that points at (b) */
#define VNormalize(a,b) {VTemp=1./sqrt((b)[X]*(b)[X]+(b)[Y]*(b)[Y]+(b)[Z]*(b)[Z]);(a)[X]=(b)[X]*VTemp;(a)[Y]=(b)[Y]*VTemp;(a)[Z]=(b)[Z]*VTemp;}
#define VNormalizeEq(a) {VTemp=1./sqrt((a)[X]*(a)[X]+(a)[Y]*(a)[Y]+(a)[Z]*(a)[Z]);(a)[X]*=VTemp;(a)[Y]*=VTemp;(a)[Z]*=VTemp;}

/* Compute a Vector (a) Halfway Between Two Given Vectors (b) and (c) */
#define VHalf(a, b, c) {(a)[X]=0.5*((b)[X]+(c)[X]);(a)[Y]=0.5*((b)[Y]+(c)[Y]);(a)[Z]=0.5*((b)[Z]+(c)[Z]);}

/* Calculate the sum of the sqares of the components of a vector.  (the square of its length) */
#define VSumSqr(a)  ((a)[X]*(a)[X] + (a)[Y]*(a)[Y] + (a)[Z]*(a)[Z])



/*
 * Linear combination of 2 vectors. [DB 7/94]
 *
 *   V = k1 * V1 + k2 * V2
 */
#define VLinComb2(V, k1, V1, k2, V2)            \
  { (V)[X] = (k1) * (V1)[X] + (k2) * (V2)[X];   \
    (V)[Y] = (k1) * (V1)[Y] + (k2) * (V2)[Y];   \
    (V)[Z] = (k1) * (V1)[Z] + (k2) * (V2)[Z]; }



/*
 * Linear combination of 3 vectors. [DB 7/94]
 *
 *   V = k1 * V1 + k2 * V2 + k3 * V3
 */
#define VLinComb3(V, k1, V1, k2, V2, k3, V3)                     \
  { (V)[X] = (k1) * (V1)[X] + (k2) * (V2)[X] + (k3) * (V3)[X];   \
    (V)[Y] = (k1) * (V1)[Y] + (k2) * (V2)[Y] + (k3) * (V3)[Y];   \
    (V)[Z] = (k1) * (V1)[Z] + (k2) * (V2)[Z] + (k3) * (V3)[Z]; }



/*
 * Evaluate a ray equation. [DB 7/94]
 *
 *   IPoint = Initial + depth * Direction
 */
#define VEvaluateRay(IPoint, Initial, depth, Direction)      \
  { (IPoint)[X] = (Initial)[X] + (depth) * (Direction)[X];   \
    (IPoint)[Y] = (Initial)[Y] + (depth) * (Direction)[Y];   \
    (IPoint)[Z] = (Initial)[Z] + (depth) * (Direction)[Z]; }



/*
 * Add a scaled vector. [DB 7/94]
 *
 *   V  = V1 + k * V2;
 *   V += k * V2;
 */
#define VAddScaled(V, V1, k, V2)         \
  { (V)[X] = (V1)[X] + (k) * (V2)[X];    \
    (V)[Y] = (V1)[Y] + (k) * (V2)[Y];    \
    (V)[Z] = (V1)[Z] + (k) * (V2)[Z]; }

#define VAddScaledEq(V, k, V2)  \
  { (V)[X] += (k) * (V2)[X];    \
    (V)[Y] += (k) * (V2)[Y];    \
    (V)[Z] += (k) * (V2)[Z]; }



/*
 * Subtract a scaled vector. [DB 8/94]
 *
 *   V  = V1 - k * V2;
 *   V -= k * V2;
 */
#define VSubScaled(V, V1, k, V2)         \
  { (V)[X] = (V1)[X] - (k) * (V2)[X];    \
    (V)[Y] = (V1)[Y] - (k) * (V2)[Y];    \
    (V)[Z] = (V1)[Z] - (k) * (V2)[Z]; }

#define VSubScaledEq(V, k, V2)  \
  { (V)[X] -= (k) * (V2)[X];    \
    (V)[Y] -= (k) * (V2)[Y];    \
    (V)[Z] -= (k) * (V2)[Z]; }


#endif

