/*
 * matrixCmd.h
 *
 *	Impliments the "matrix" and "mexpr" commands
 *
 *  Questions, Bugs reports, and Creative direction can for now be sent to
 *  Sean Woods, yoda@drexel.edu
 *
 *  Copyright (c) 1999 Woods Design Services.
 *
 * See the file "odie.license" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */
 
#include <math.h>
#include <tcl.h>
#include "matrices.h"
 
/* Stack Macros */
#define StackLink(i,r,c,d) {r = Stack[index].rows;c = Stack[index].cols;d = Stack[index].cells;}
#define ISVECTOR(A) {if((A.rows != 3) || (A.cols != 1))	{Matrix_Error("Argument Must be a Vector");return TCL_ERROR;}

/* Local Variables */
enum {
	vexprAny,
	vexprScaler,
	vexprList,
	vexprScreen,
	vexprVector,
	vexprAffine,
	vexprMatrix
} vexprTypes;

/* Expression Stack */

static MATOBJ Stack[MATSTACKSIZE];
static int Idx;
static int mutex;
static int defRows;
static int defCols;
static int defMode;

static MATVERB Instructions[MATSTACKSIZE];

static Tcl_Interp *interp = NULL;
static int objc;
static Tcl_Obj *CONST *objv;
static int objIdx;
static char *ErrorString;
static int	ErrorResult;	/* Error if Non-Zero */
static double VTemp;
// For integration / differentiation
static double deltaT;

/* Command dictionaries */

typedef struct cmndEntry {
	char *name;
	int	code;
} VERB;

static char *matrixCmds[] = {
	//Matrix Functions
	"+",
	"*",	/* Scaler Product */
	"*.",	/* Dot Product */
	"*x",	/* Cross Product */
	"~",	/* Negation */
	"1/",	/* Recipricol */
	"-",	
	"/",
	"transpose",
	"inverse",
	"determinate",
	"adjoint",
	"noop", 
	(char *) NULL 
};

enum {
	MatAdd,
	MatScale,
	MatMultiply,
	MatCross,
	MatNegate,
	MatRecipricol,
	MatSub,
	MatInvScale,
	MatTranspose,
	MatInverse,
	MatDeterminate,
	MatAdjoint
} matrixOpcodes;

static char *vectorCmds[] = {
	"+",
	"*",	/* Scaler Product */
	"*.",	/* Dot Product */
	"*x",	/* Cross Product */
	"~",
	"-",	
	"/",
	"half",
	"distance",
	"distInvSq",
	"length",
	"normalize",
	"rad2deg",
	"deg2rad",
	"pi",
	(char *) NULL 
};

enum {
	VecAdd,
	VecScale,
	VecDot,
	VecCross,
	VecNegate,
	VecSub,
	VecDivide,
	VecHalf,
	VecDistance,
	VecDisInvSqr,
	VecLength,
	VecNormalize,
	VecConstRad2Deg,
	VecConstDeg2Rad,
	VecConstPi
} vectorOpcodes;
//
extern int MatExpr_ObjCmd(ClientData dummy, Tcl_Interp *interp, int tobjc, Tcl_Obj *CONST tobjv[]);
extern int MatFunct_ObjCmd(ClientData dummy, Tcl_Interp *interp, int tobjc, Tcl_Obj *CONST tobjv[]);
extern int VecExpr_ObjCmd(ClientData dummy, Tcl_Interp *interp, int tobjc, Tcl_Obj *CONST tobjv[]);

static int Pop(int idx);
static int Push_Empty(int rows,int cols,int index);
static int Push_Obj(Tcl_Obj *listPtr,int index);
static int Push_Scaler(double value,int index);
static int Push_Vector(double *value,int index);
/* Matrix Functions */

int Stack_MatrixCommand(int cmnd);
int Stack_VectorCommand(int cmnd);

int Vector_BatchTransform(int vcount,double *veclist,double *rot,double *tran);
int Matrix_Init( Tcl_Interp *tinterp );

Tcl_Interp *MatInterp(void) { return interp; }


int Matrix_Init( Tcl_Interp *tinterp ) {
	if(interp == NULL)
		interp = tinterp;
	Tcl_CreateObjCommand(tinterp, "mexpr", MatExpr_ObjCmd, (ClientData) 0, NULL);
	Tcl_CreateObjCommand(tinterp, "vector3", MatExpr_ObjCmd, (ClientData) 3, NULL);
	Tcl_CreateObjCommand(tinterp, "vector2", MatExpr_ObjCmd, (ClientData) 2, NULL);

	Tcl_CreateObjCommand(tinterp, "vexpr", VecExpr_ObjCmd, (ClientData) 0, NULL);
	Tcl_CreateObjCommand(tinterp, "matrix", MatFunct_ObjCmd, NULL, NULL);	
	return TCL_OK;
}

/*
 * Performs specialized matrix functions (canvas scaling, cramer's rule, etc.)
 */
int MatFunct_ObjCmd(dummy, tinterp, objc, objv) 
    ClientData dummy;		/* Not used. */
    Tcl_Interp *tinterp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
	int index, err;
	MATOBJ result;
	static char *specialCmds[] = {
		"range",
	 	"integrate",
	 	"differentiate",
	  	"cramer",
		"transform",
		"canscale",
		"dt",
		"set",
		"vars",
		(char *) NULL 	
	};
	enum {
		SpecRange,
		SpecIntegrate,
		SpecDiff,
		SpecCramer,
		SpecTransform,
		SpecCanscale,
		SpecDt,
		SpecGet,
		SpecSet,
		SpecVars
	};
	interp = tinterp;
	err = TCL_OK;
	if (objc < 2) {
		Tcl_WrongNumArgs(interp, 1, objv, "command ?arg?");		
		return TCL_ERROR;		
	}
    if (Tcl_GetIndexFromObj(interp, objv[1], specialCmds, "command", 0,
	    (int *) &index) != TCL_OK) 
	    return TCL_ERROR;
	
	switch(index) {
    	case SpecDt:
			if (objc < 3) {
				Tcl_WrongNumArgs(interp, 1, objv, "dt value");		
				return TCL_ERROR;		
			}
			err =  Tcl_GetDoubleFromObj(interp, objv[2], &deltaT);
			break;
    	case SpecRange:
	    	{	int *range, len;
	    		MATOBJ A;
				if (objc < 4) {
					Tcl_WrongNumArgs(interp, 1, objv, "range matrix size");		
					return TCL_ERROR;		
				}
				if(Matrix_FromObj(interp,objv[2],&A) != TCL_OK)
					return TCL_ERROR;
				if((range = Coords_FromObj(interp,objv[3],&len)) == NULL)
					return TCL_ERROR;
				if(len < 4) {
					Matrix_Error("Range must be: {top left bottom right}");		
					return TCL_ERROR;				
	    		}
	    		result.cells = Matrix_Range(&A,range);
				result.rows = range[BOTTOM] - range[TOP] + 1;
				result.cols = range[RIGHT] - range[LEFT] + 1;	
			}
    		break;	
 		case SpecCramer: 	
			 {
				int rows,cols;
				MATOBJ A,B;
				if (objc < 4) {
					Tcl_WrongNumArgs(interp, 1, objv, "cramer <equations> <solutions>");		
					return TCL_ERROR;		
				}
				if (Matrix_FromObj(interp,objv[2],&A) != TCL_OK) return TCL_ERROR;
				if (Matrix_FromObj(interp,objv[3],&B) != TCL_OK) return TCL_ERROR;		

				if((B.cols > 1)) {
					Matrix_Error("Solution argument must be a row matrix");		
					return TCL_ERROR;
				}

				rows = A.rows;		
				cols = A.cols;
				
				if((B.rows != rows)) {
					Matrix_Error("Solution and matrix must have the same number of rows");		
					return TCL_ERROR;
				}

				if((rows != cols)) {
					Matrix_Error("Matrix must be square");
					return TCL_ERROR;
				}

				result.cells = Matrix_CramersRule(rows,A.cells,B.cells);
				result.rows = rows;		
				result.cols = 1;
			}	
 			break;
		case SpecCanscale:
		{	register int i,offi,offj;
			register double xscale,yscale,xoffset,yoffset;
			int rows,cols;
			double *data,*dataTemp;
			MATOBJ A,B;
			if (objc < 4) {
				Tcl_WrongNumArgs(interp, 1, objv, "canscale vectorlist scaledata");		
				return TCL_ERROR;		
			}
			if (Matrix_FromObj(interp,objv[2],&A) != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),ErrorString,-1);
				return TCL_ERROR;
			}
			if (Matrix_FromObj(interp,objv[3],&B) != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),ErrorString,-1);
				return TCL_ERROR;
			}
			rows = B.rows;
			cols = B.cols;
			xscale = B.cells[0];
			yscale = B.cells[1];
			xoffset = B.cells[cols];
			yoffset = B.cells[cols+1];	
					
			data = A.cells;
			rows = A.rows;
			cols = A.cols;
			
			dataTemp = (double *)Tcl_Alloc(sizeof(double)*rows*2);
			
			for(i=0;i<rows;i++) {
				offi = i*2;
				offj = i*cols;
				dataTemp[offi] = data[offj] * xscale + xoffset;
				dataTemp[offi+1] = data[offj+1] * yscale + yoffset;				
			}
			Tcl_Free((char *)A.cells);	
			Tcl_Free((char *)B.cells);
			
			result.rows = rows;
			result.cols = 2;
			result.cells = dataTemp;
		}
		break;
		case SpecTransform:	
		{
			MATOBJ Rot,Trans;
			
			if (objc < 4) {
				Tcl_WrongNumArgs(interp, 1, objv, "transform <vectorlist> <rotation> <translation>");		
				return TCL_ERROR;		
			}
			if (Matrix_FromObj(interp,objv[2],&result) != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),ErrorString,-1);		
				return TCL_ERROR;
			}
			if (Matrix_FromObj(interp,objv[3],&Rot) != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),ErrorString,-1);
				return TCL_ERROR;
			}
			if (Matrix_FromObj(interp,objv[4],&Trans) != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),ErrorString,-1);
				return TCL_ERROR;
			}
			if((Trans.cols != 1) || (Trans.rows != 3)) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),"Translation must be a vector",-1);
				return TCL_ERROR;
			}
			if((Rot.cols != 1) || (Rot.rows != 3)) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),"Rotation must be a vector",-1);
				return TCL_ERROR;
			}
			if(result.cols != 3) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),"Vector arguments must be 3x1",-1);
				return TCL_ERROR;
			}
			if(Vector_BatchTransform(
				result.rows,
				result.cells,
				Rot.cells,
				Trans.cells
			) != TCL_OK) return TCL_ERROR;
			Tcl_Free((char *)Rot.cells);
			Tcl_Free((char *)Trans.cells);			
		}
		break;
		default:
			Tcl_SetStringObj(Tcl_GetObjResult(interp),
				"Unimplemented Command",-1);			
				return TCL_ERROR;			
	}
	if(err == TCL_OK) {
		Matrix_ToList(Tcl_GetObjResult(interp), &result);	
		Tcl_Free((char *)result.cells);		
		return TCL_OK;
	}
	return TCL_ERROR;
}

/* tclVector.c
 *
 *	implements a new command 'matrix' which performs matrix operations
 */

int MatExpr_ObjCmd(dummy, tinterp, tobjc, tobjv) 
    ClientData dummy;		/* Not used. */
    Tcl_Interp *tinterp;		/* Current interpreter. */
    int tobjc;			/* Number of arguments. */
    Tcl_Obj *CONST tobjv[];	/* Argument objects. */
{
	int i,result;
	int index;
    int resultForm = (int) dummy;
    interp = tinterp;
    objc   = tobjc;
    objv   = tobjv;
    
 	Idx=-1;   
	result = TCL_OK;
	Tcl_SetStringObj(Tcl_GetObjResult(interp),"",-1);
	for(i=1;i<objc;i++) {    
	    if (Tcl_GetIndexFromObj(interp, objv[i], matrixCmds, "verb", 0,
		    (int *) &index) != TCL_OK) 
		{
			/* Not an opcode, push value into stack */
			Idx++;
			if(Idx >= MATSTACKSIZE) {
				Matrix_Error("Matrix Stack Overflow");		
				return TCL_ERROR;			
			}
			if (Matrix_FromObj(interp,objv[i],&(Stack)[Idx]) != TCL_OK) {
				return TCL_ERROR;
			}
	    } else {
			result = Stack_MatrixCommand(index);
			if(result != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),
				ErrorString,-1);			
				return result;
			}
		}
	}
	Matrix_ToList(Tcl_GetObjResult(interp), (Stack+Idx));
	/* Destroy Stack */
	for(i=Idx;i>=0;i--) {
		Pop(i);
	}
	return TCL_OK;
}

int Stack_MatrixCommand(int cmnd) 
{
	int i=0,errCode;
	errCode = TCL_OK;
	/* Interpret Opcode */
    switch (cmnd) {
		case MatNegate: 	
			Matrix_Negate(Stack[Idx].cells,Stack[Idx].cells,
				Stack[Idx].rows,Stack[Idx].cols);
		 	break;
		case MatRecipricol: 
			Matrix_Recipricol(Stack[Idx].cells,Stack[Idx].cells,
				Stack[Idx].rows,Stack[Idx].cols);
			break;
		case MatTranspose:
		case MatAdjoint:
		case MatInverse:
		{
			double *dataTemp;
			if(Idx < 1) {
				Matrix_Error("Missing Opcode");
				return TCL_ERROR;
			}
			
			/* Negate and Recipricol Don't need a buffer */
			dataTemp = (double *)Tcl_Alloc(sizeof(double)*Stack[Idx].rows*Stack[Idx].cols);
			switch(cmnd) {	
				case MatTranspose:
					Matrix_Transpose(dataTemp,
						Stack[Idx].cells,
						Stack[Idx].rows,Stack[Idx].cols); 
					Stack[Idx].rows = Stack[Idx].cols;
					Stack[Idx].cols = Stack[Idx].rows;
					break;		
				case MatAdjoint:
					Matrix_Adjoint(dataTemp,Stack[Idx].cells,Stack[Idx].rows); 
					break;
				case MatInverse:
 					if(Stack[Idx].rows != Stack[Idx].cols) {
						Matrix_Error("Tried to invert non-square matrix");
						Tcl_Free((char *)dataTemp);
						return TCL_ERROR;							
					}
					if(Matrix_Inverse(dataTemp,Stack[Idx].cells,Stack[Idx].rows)) 
						Tcl_Free((char *)dataTemp);
						return TCL_ERROR;
					break;
			}
			Tcl_Free((char *)Stack[Idx].cells);
			Stack[Idx].cells = dataTemp;	
			return TCL_OK;
		}		
		case MatDeterminate:
		{
			double D;
			/* Allocate Transverse Buffer */
			if (Stack[Idx].rows != Stack[Idx].cols)
			{
				Matrix_Error("Argument Must be Square");
				return TCL_ERROR;	
			}

			D = Matrix_Determinate(Stack[Idx].cells,Stack[Idx].rows);
			Pop(Idx);
			Push_Scaler(D,Idx);	
		}
			break;
		case MatMultiply:
		{
			double *dataTemp;
			if(Idx < 1) {
				/* These Items Need Two From The Stack */
				Matrix_Error("Missing Op: <A> <B> *");
				return TCL_ERROR;						
			}
			if(
				(Stack[Idx - 1].cols != Stack[Idx].rows )
				 || 
				(Stack[Idx - 1].rows != Stack[Idx].cols)
			) {
				Matrix_Error("Non-Conforming Argumemts");
				return TCL_ERROR;
			}
			dataTemp = Matrix_Create(Stack[Idx].rows,Stack[Idx].cols);
			Matrix_Multiply(dataTemp,
				Stack[Idx-1].cells,
				Stack[Idx].cells,
				Stack[Idx].rows,Stack[Idx].cols);
			Pop(Idx);
			Idx--;
			Tcl_Free((char *)Stack[Idx].cells);
			Stack[Idx].cells = dataTemp;		
		}
			break;
		case MatAdd:
		{
			register int IdxLast,cells,offset;

			if(Idx < 1) {
			/* These Items Need Two From The Stack */
				Matrix_Error("Missing Op: <A> <B> +");	
				return TCL_ERROR;					
			}
			IdxLast = Idx-1;
			if(
				(Stack[IdxLast].cols != Stack[Idx].cols )
				 || 
				(Stack[IdxLast].rows != Stack[Idx].rows)
			) {
				Matrix_Error("Non-Conforming Argumemts");
				return TCL_ERROR;
			}

			cells = Stack[Idx].cols * Stack[Idx].rows;
			for(offset = 0;offset<cells;offset++) 
				Stack[IdxLast].cells[offset] += Stack[Idx].cells[offset];
				 
			Pop(Idx);
			Idx--;		
		}
			break;
		case MatSub: 
			Stack_MatrixCommand(MatNegate);
			Stack_MatrixCommand(MatAdd); 
			break;
		case MatScale:
			if(Idx < 1) {
				Matrix_Error("Missing Operands: <matrix> <scaler> *");
				return TCL_ERROR;
			}
			Matrix_Scale(Stack[Idx-1].cells,
				Stack[Idx-1].cells,
				Stack[Idx-1].rows,
				Stack[Idx-1].cols,
				Stack[Idx].cells[0]);
			Pop(Idx);
			Idx--;
			break;
		case MatInvScale:
			if(Idx < 1) {
				Matrix_Error("Missing Operands: <matrix> <scaler> /");
				return TCL_ERROR;
			}
			Stack_MatrixCommand(MatRecipricol);
			Matrix_Scale(Stack[Idx-1].cells,
				Stack[Idx-1].cells,
				Stack[Idx-1].rows,
				Stack[Idx-1].cols,
				Stack[Idx].cells[0]);
			Pop(Idx);
			Idx--;
			break;
		default:
			Matrix_Error("Unimplemented Command");	
			errCode = TCL_ERROR;					
   }
   return errCode;
}

int VecExpr_ObjCmd(dummy, tinterp, tobjc, tobjv) 
    ClientData dummy;		/* Not used. */
    Tcl_Interp *tinterp;		/* Current interpreter. */
    int tobjc;			/* Number of arguments. */
    Tcl_Obj *CONST tobjv[];	/* Argument objects. */
{
	int i,result;
	int index;
    int resultForm = (int) dummy;
    interp = tinterp;
    objc   = tobjc;
    objv   = tobjv;
    
 	Idx=-1;   
	result = TCL_OK;
	Tcl_SetStringObj(Tcl_GetObjResult(interp),"",-1);
	for(i=1;i<objc;i++) {    
	    if (Tcl_GetIndexFromObj(interp, objv[i], vectorCmds, "verb", 0,
		    (int *) &index) != TCL_OK) 
		{
			/* Not an opcode, push value into stack */
			Idx++;
			if(Idx >= MATSTACKSIZE) {
				Matrix_Error("Matrix Stack Overflow");		
				return TCL_ERROR;			
			}
			if (Matrix_FromObj(interp,objv[i],&(Stack)[Idx]) != TCL_OK) {
				return TCL_ERROR;
			}
	    } else {
			result = Stack_VectorCommand(index);
			if(result != TCL_OK) {
				Tcl_SetStringObj(Tcl_GetObjResult(interp),
				ErrorString,-1);			
				return result;
			}
		}
	}
	Matrix_ToList(Tcl_GetObjResult(interp), (Stack+Idx));
	/* Destroy Stack */
	for(i=Idx;i>=0;i--) {
		Pop(i);
	}
	return TCL_OK;
}

int Stack_VectorCommand(int cmnd) {
	int errCode;
	errCode = TCL_OK;	
	switch(cmnd) {
		case VecAdd:
		case VecSub:
		case VecScale:	
		case VecDivide:
			if ( (Stack[Idx-1].rows < 3) || (Stack[Idx].rows < 3)) {
				Matrix_Error("Nonvector operand");
				return TCL_ERROR;
			}
			switch(cmnd) {
				case VecAdd: VAddEq(Stack[Idx-1].cells,Stack[Idx].cells); break;
				case VecSub: VSubEq(Stack[Idx-1].cells,Stack[Idx].cells); break;
				case VecScale:	VScaleEq(Stack[Idx-1].cells,Stack[Idx].cells[0]); break;
				case VecDivide: VInverseScaleEq(Stack[Idx-1].cells,Stack[Idx].cells[0]); break;						
			}
			Pop(Idx);Idx--;
			break;	
		case VecCross:
		case VecHalf:
			{
				double *result;
				if ( (Stack[Idx-1].rows < 3) || (Stack[Idx].rows < 3)) {
					Matrix_Error("Nonvector operand");
					return TCL_ERROR;
				}
				result = (double *) Tcl_Alloc(sizeof(double) * 3);
				switch(cmnd) {
					case VecCross:
						VCross(result,Stack[Idx-1].cells,Stack[Idx].cells); break;
					case VecHalf:
						VHalf(result,Stack[Idx-1].cells,Stack[Idx].cells); break;
				}
				Pop(Idx);Idx--;
				Pop(Idx);
				Push_Vector(result,Idx);	
			}				
			break;
		case VecDistance:
		case VecDisInvSqr:
		case VecDot:
			{
				double result;
				if ( (Stack[Idx-1].rows < 3) || (Stack[Idx].rows < 3)) {
					Matrix_Error("Nonvector operand");
					return TCL_ERROR;
				}
				switch(cmnd) {
					case VecDistance:
						VDist(result,Stack[Idx-1].cells,Stack[Idx].cells);
						break;
					case VecDisInvSqr:
						VDistInvSqr(result,Stack[Idx-1].cells,Stack[Idx].cells);
						break;
					case VecDot:
						VDot(result,Stack[Idx-1].cells,Stack[Idx].cells);
						break;
				}	
				Pop(Idx);Idx--;
				Pop(Idx);
				Push_Scaler(result,Idx);	
			}				
			break;	
		case VecNormalize:
			if ( Stack[Idx].rows < 3 ) {
				Matrix_Error("Nonvector operand");
				return TCL_ERROR;
			}		
			VNormalizeEq(Stack[Idx].cells);
			break;
		case VecLength:
			{
				double result;
				if ( Stack[Idx].rows < 3 ) {
					Matrix_Error("Nonvector operand");
					return TCL_ERROR;
				}	
				VLength(result,Stack[Idx].cells);
				Pop(Idx);
				Push_Scaler(result,Idx);	
			}				
			break;
		case VecConstRad2Deg:	Idx++; Push_Scaler(1/M_PI_180,Idx);break;
		case VecConstDeg2Rad:	Idx++; Push_Scaler(M_PI_180,Idx);break;
		case VecConstPi:		Idx++; Push_Scaler(M_PI,Idx);break;
		default:
			Matrix_Error("Unimplemented Command");	
			errCode = TCL_ERROR;				
	}
	return errCode;
}

/* Produce a general case matrix from a TCL List */
int Pop(int idx) {
	Stack[idx].rows = 0;
	Stack[idx].cols = 0;
	if(Stack[idx].cells != NULL) {
		Tcl_Free((char *) Stack[idx].cells);
		Stack[idx].cells = (double *)NULL;
	}
	return TCL_OK;
}

int Push_Empty(int rows,int cols,int index) {
	if(Stack[index].cells != NULL) {
		Pop(index);
	}
	Stack[index].rows = rows;
	Stack[index].cols = cols;
	Stack[index].cells = (double *)Tcl_Alloc(sizeof(double) * rows * cols);
	return TCL_OK;
}

int Push_Scaler(double value,int index) {
	if(Stack[index].cells != NULL) {
		Pop(index);
	}
	Stack[index].rows = 1;
	Stack[index].cols = 1;
	Stack[index].cells = (double *)Tcl_Alloc(sizeof(double));
	Stack[index].cells[0] = value;  
	return TCL_OK;
}

int Push_Vector(double *value,int index) {
	if(Stack[index].cells != NULL) {
		Pop(index);
	}
	Stack[index].rows = 3;
	Stack[index].cols = 1;
	Stack[index].cells = value;
	return TCL_OK;
}

void Matrix_Error(char *error) {
	ErrorString = error;
	ErrorResult = TCL_ERROR;
}
