/* Task Manager
 *
 *	Impliments a realtime task switching mechanism within the Tcl/Tk event
 *	system
 * 
 *  Questions, Bugs reports, and Creative direction can for now be sent to
 *  Sean Woods, yoda@drexel.edu
 *
 * Copyright (c) 1998 Woods Design Services
 *
 * See the file "odie.license" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tcl.h"
#include "process.h"
#include <strings.h>

int Process_Init(Tcl_Interp *interp);

int Tcl_ProcessObjCmd(
    ClientData dummy,		/* Not used. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int objc,			/* Number of arguments. */
    Tcl_Obj *CONST objv[]	/* Argument objects. */
);
/*
 *
 */

static void ProcessInsert(OdieTask *deviceState);
static void ProcessRemove(OdieTask *fileState);
static void ProcessCmdDeletionProc();
static void ProcessSetupProc(ClientData data,int flags);
static void ProcessCheckProc(ClientData data,int flags);
static int	ProcessEventProc(Tcl_Event *evPtr,int flags);
static int	ProcessTaskScript(ClientData data);
static int	ProcessExists(OdieTask *deviceState);

/* Head Pointer to a linked list of tasks */
static OdieTask TaskList;
static int pidCount;	
static int initialized = 0;
static int mutex = 0;

static unsigned long 
			clicks,
			lastime=0;
static	long 	
			rounds 		= 	0, 
			comptime	= 	0;
static	int		rollover	= 	0;
static	int 	guiPriority = 	1;
static 	int		activeTasks = 0;

/*
 *----------------------------------------------------------------------
 * Tcl_ProcessObjCmd --
 *
 *      Tcl Command interface to process table
 *
 * Results: 
 *      See Documentation
 *
 * Side Effects:
 *      See Documentation
 *
 *----------------------------------------------------------------------
 */
 
int Tcl_ProcessObjCmd(dummy, interp, objc, objv) 
    ClientData dummy;		/* Not used. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
	int pid;
	OdieTask *infoPtr;
	Tcl_Obj	*result;
	static char buffer[120];
    static char *subCmds[] = {
    	"create",
    	"start", 
    	"stop",
    	"reset",
    	"kill",
    	"info",
    	"jobs",
    	"ps",
    	"configure",
    	"stats",
    	(char *) NULL 
    };
    enum {    	
    	psCreate,
    	psStart, 
    	psStop,
    	psReset,
    	psKill,
    	psInfo,
    	psJobs,
    	psPs,
    	psConfigure,
    	psStats
    } index;
    
    if (objc < 2) {
    	Tcl_WrongNumArgs(interp, 1, objv, "command ?arg arg ...?");
		return TCL_ERROR;
    }
    
    if (Tcl_GetIndexFromObj(interp, objv[1], subCmds, "command", 0,
	    (int *) &index) != TCL_OK) {
    	return TCL_ERROR;
    }
    
    Tcl_ResetResult(interp);
    result = Tcl_GetObjResult(interp);
    
    switch(index) {
    	case psCreate:
    	{	
		    if (objc < 3) {
		    	Tcl_WrongNumArgs(interp, 1, objv, "create command ?arg arg ...?");
				return TCL_ERROR;
		    }
    		pid = Tcl_ProcessCreate(interp,(Tcl_Obj *)Tcl_DuplicateObj(objv[2]));
			infoPtr = Odie_GetState(pid);
			
    		if (Tcl_ProcessConfigure(infoPtr,interp,objc-3,&objv[3]) == TCL_ERROR) {
    			return TCL_ERROR;
 			}
    		
			Tcl_ResetResult(interp);
			if(infoPtr == NULL) {
				Tcl_AppendToObj(result,"Error In Creating New Process", -1);
				return TCL_ERROR;					
			}
			Tcl_SetIntObj(result,pid);			    		
    		return TCL_OK;
    	}
    	case psJobs:
    	{	char temp[32];
    		for(infoPtr = TaskList.nextPtr ; infoPtr != NULL ; infoPtr = infoPtr->nextPtr) {
				sprintf(temp,"%d ",infoPtr->pid);
				Tcl_AppendToObj(result,temp, -1);
			}  
			return TCL_OK;	
    	}
    	case psPs:
    	{
			Tcl_AppendToObj(result,"PID\ttype\tstate\tperiod\tavgtime\tdesc\n", -1);
			for(infoPtr = TaskList.nextPtr ; infoPtr != NULL ; infoPtr = infoPtr->nextPtr) {
				sprintf(buffer,"%d",infoPtr->pid);
				Tcl_AppendToObj(result,buffer, -1);
   	    		if ((infoPtr ->cfunct) == (&ProcessTaskScript)) {	
					Tcl_AppendToObj(result,"\tproc", -1);
				} else {
					Tcl_AppendToObj(result,"\tfunc", -1);
				} 					
				
				switch(infoPtr->state) {
					case taskStateActive:	Tcl_AppendToObj(result,"\tactive", -1);	break;					
					case taskStateStopped:	Tcl_AppendToObj(result,"\tstopped", -1);	break;	
					case taskStateHalted:	Tcl_AppendToObj(result,"\thalted", -1);	break;
					case taskStateSleeping:	Tcl_AppendToObj(result,"\tsleep", -1);	break;		
					case taskStateKilled:	Tcl_AppendToObj(result,"\tkilled", -1);	break;		
				}

				sprintf(buffer,"\t%d\t%d\t",infoPtr->period,infoPtr->avgtime);
				Tcl_AppendToObj(result,buffer, -1);
   				Tcl_AppendToObj(result,infoPtr->desc, -1);	
				Tcl_AppendToObj(result,"\n", -1);	
			} 
			return TCL_OK;
    	}
    	case psInfo:
    	{	Tcl_AppendToObj(result,"GNDN", -1);
    		return TCL_OK;
    	
    	}
	}


    if (objc < 3) {
    	Tcl_WrongNumArgs(interp, 1, objv, "command pid option ?value?");
		return TCL_ERROR;    
    }

    if (Tcl_GetIntFromObj(interp,objv[2], &pid) != TCL_OK) return TCL_ERROR;
    
	switch(index) {
    	case psStart: 
    		Odie_StartProcess(pid);
			return TCL_OK;
    	case psStop:
    		Odie_StopProcess(pid);
			return TCL_OK;    	
    	case psKill: 
     		Odie_KillProcess(pid);
			return TCL_OK;
	} 
    
	infoPtr = Odie_GetState(pid);
	if(infoPtr == NULL) {    
		Tcl_AppendToObj(result,"Bad Process ID", -1);   
	    return TCL_ERROR;
    }		  
      
	switch(index) {
		case psStats:
		{
			sprintf(buffer,"max %d min %d avg %d",
				infoPtr->maxtime,infoPtr->mintime,infoPtr->avgtime);
			Tcl_AppendToObj(result,buffer, -1);
			return TCL_OK;				
		}
		case psConfigure: 
			return Tcl_ProcessConfigure(infoPtr,interp,objc-3,&objv[3]);
		default:	
			return TCL_ERROR;
	}

}

/*
 *----------------------------------------------------------------------
 * Tcl_ProcessConfigure --
 *
 *     Modifies the state of a process
 *
 * Results: 
 *      See Documentation
 *
 * Side Effects:
 *      See Documentation
 *
 *----------------------------------------------------------------------
 */
 
int Tcl_ProcessConfigure(infoPtr, interp, objc, objv) 
	OdieTask *infoPtr;		/* Process Information */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
	static char *psOptions[] = { 
    	"-period",
    	"-frequency",
    	"-desc",
    	"-command",
		(char *)NULL
	};
	enum {
    	psPeriod,
    	psFrequency,
    	psDesc,
    	psCommand
    } optidx;
    int argidx,len;
	char *buffer;
    Tcl_Obj *result;
    
	Tcl_ResetResult(interp);
	
    result = Tcl_GetObjResult(interp);
        
    if (objc == 0) {
      	/*
    	 * Produce report of settings
    	 */
    	int len;
		buffer = Tcl_Alloc(256);
		
   	    if ((infoPtr ->cfunct) == (&ProcessTaskScript)) {
   	    	OdieTclTask *temp = (OdieTclTask *)infoPtr->data;
			Tcl_AppendToObj(result,"-command {", -1);
			Tcl_AppendToObj(result,Tcl_GetStringFromObj(temp->script,&len),len); 
		}

		Tcl_AppendToObj(result,"} -desc {", -1);
		if(infoPtr->desc != NULL) {
			Tcl_AppendToObj(result,infoPtr->desc,-1);
		}
		
		sprintf(buffer,"} -period %d",infoPtr->period);
		Tcl_AppendToObj(result,buffer, -1);
		Tcl_Free(buffer);
		return TCL_OK;
    }

    if (Tcl_GetIndexFromObj(interp, objv[0], psOptions, "option", 0,
	    (int *) &optidx) != TCL_OK) {
    	return TCL_ERROR;
    }
    
    if(objc < 2)
    {
	    switch(optidx) {
	    	case psCommand:
   	    		if ((infoPtr ->cfunct) == (&ProcessTaskScript)) {
   	    			int len;
    	    		OdieTclTask *temp = (OdieTclTask *)infoPtr->data;
					Tcl_AppendToObj(result,Tcl_GetStringFromObj(temp->script,&len),len);
	    		}
	    		return TCL_OK;
	    	case psDesc:
				{
					if (infoPtr->desc != NULL) {
						Tcl_AppendToObj(result,infoPtr->desc,-1);
					}
	    		}
	    		return TCL_OK;
	    	case psPeriod:
				Tcl_SetLongObj(result, (long)infoPtr->period);
				return TCL_OK;    	
	    	case psFrequency:
				Tcl_SetLongObj(result, (long)(1e6/infoPtr->period));
				return TCL_OK;
	    }
	} 
	for(argidx=0;argidx<objc;argidx++) {
		buffer = Tcl_GetStringFromObj(objv[argidx],&len);
	    /*
	     * Determine if item is an option
	     */	
		if(buffer[0] == '-') {
		    /*
		     * See Which it is
		     */			
		    if (Tcl_GetIndexFromObj(interp, objv[argidx], psOptions, "option", 0,
			    (int *) &optidx) != TCL_OK) {
		    	return TCL_ERROR;
		    }
		    /*
		     * See if we have enough arguments
		     */
		     if((objc - argidx) < 0) {
			    Tcl_WrongNumArgs(interp, 1, objv, "-option value");
				return TCL_ERROR;    		     	
		     }
		} else {
	    switch(optidx) {
	    	case psCommand:
   	    		if ((infoPtr ->cfunct) == (&ProcessTaskScript)) {
    	    		OdieTclTask *temp = (OdieTclTask *)infoPtr->data;
					int clen;
					char *code;
					
    	    		Tcl_DecrRefCount(temp->script);
    	    		temp->script = Tcl_DuplicateObj(objv[argidx]);
					code = Tcl_GetStringFromObj(temp->script,&clen);
					Tcl_AppendToObj(result,code,clen);
	    		}
	    		break;
	    	case psDesc:
				{
					int clen;
					strncpy(
						infoPtr->desc,
						Tcl_GetStringFromObj(objv[argidx],&clen),
						31
					);
					Tcl_AppendToObj(result,infoPtr->desc,-1);
	    		}
				break;
	    	case psPeriod:
	    		{
	    			long period;
	    			Tcl_GetLongFromObj(interp, objv[argidx], &period);
					infoPtr->period = (unsigned long)period;
				}
				break;   	
	    	case psFrequency:
	    		{
	    			long period;
	    			Tcl_GetLongFromObj(interp, objv[argidx], &period);
					infoPtr->period = (unsigned long)(1e6 / period);
				}
				break;    
	    }		
	    }
	}
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 * Process_Init --
 *
 *      Initializes the devices package
 *
 * Results: 
 *      The device command is registered.
 *
 * Side Effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

int 
Process_Init(interp)
Tcl_Interp *interp;
{     
    Tcl_CreateObjCommand(interp, "process", Tcl_ProcessObjCmd,
    	NULL, ProcessCmdDeletionProc);

    TaskList.nextPtr = NULL;

    return Tcl_PkgProvide(interp, "Process", "1.0");
}

/*
 *----------------------------------------------------------------------
 * ProcessCmdDeletionProc --
 *
 *      Closes all the open devices when the device command is deleted.
 *
 * Results: 
 *      All the ports are closed.
 *
 * Side Effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
ProcessCmdDeletionProc()
{
    Tcl_DeleteEventSource(ProcessSetupProc, ProcessCheckProc, NULL);
}

/*
 *----------------------------------------------------------------------
 * ProcessInsert --
 *
 *      Head insert a process into the notification list
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      An entry is added to the device list.
 *		If this is the first entry, we register the process manager as
 *		an event source
 *
 *----------------------------------------------------------------------
 */
 
void ProcessInsert(OdieTask *deviceState) {
	/*
	 * Head Insert Into Notification List
	 */
	
	deviceState->nextPtr = TaskList.nextPtr;
	TaskList.nextPtr = deviceState;
}

/*
 *----------------------------------------------------------------------
 * ProcessExists --
 *
 *      Determines Presence of process in table
 *
 * Results:
 *      0 - Not present
 *		1 - Present
 *
 * Side Effects:
 *
 *----------------------------------------------------------------------
 */
int ProcessExists(deviceState)
OdieTask *deviceState;
{
    OdieTask *temp;
	
	/*
	 *  Body Deletion
	 */
	for(temp = TaskList.nextPtr ; temp != NULL ; temp = temp->nextPtr) {
		if(temp == deviceState) {
			return 1;
		} 
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 * ProcessRemove --
 *
 *      Removes a device from the notification list
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      A task is removed from the process table
 *		If this is the last process, the process event system is 
 *		unregistered
 *----------------------------------------------------------------------
 */
void ProcessRemove(deviceState)
OdieTask *deviceState;
{
    OdieTask *temp;
	
	if(TaskList.nextPtr == deviceState) {
		TaskList.nextPtr = deviceState->nextPtr;
		return;
	}
	
	/*
	 *  Body Deletion
	 */
	for(temp = TaskList.nextPtr ; temp != NULL ; temp = temp->nextPtr) {
		if(temp->nextPtr == deviceState) {
			temp->nextPtr = temp->nextPtr->nextPtr;
			Tcl_Free((char *)deviceState);
			return;
		} 
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ProcessSetupProc --
 *
 *	This procedure is invoked before Tcl_DoOneEvent blocks waiting
 *	for an event.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Adjusts the block time if needed.
 *
 *----------------------------------------------------------------------
 */

void
ProcessSetupProc(data, flags)
ClientData data;		/* Not used. */
int flags;				/* Event flags as passed to Tcl_DoOneEvent. */
{
    OdieTask *infoPtr = (OdieTask *)data;
    Tcl_Time blockTime = { 0, 0 };
    Tcl_Time zeroTime = { 0, 0 };
    
    long thistime = 0;
	long nexttime = 1000000;

    if (!(flags & TCL_TIMER_EVENTS)) {
		return;
	}
	
	/* Handle Periodic Tasks */
	comptime =  TclpGetClicks();

	if ( comptime < lastime ) {
   		Tcl_SetMaxBlockTime(&zeroTime);
		 return;	
	} 
    /*
     * Search through the process table for the one whose handle
     * matches the event.  We do this rather than simply dereferencing
     * the handle in the event so that process can be deleted while the
     * event is in the queue.
     */
    if(!ProcessExists(infoPtr)) {
    	return;
    }
    
    /*
     * Check to see how soon until the next process event.
     */

	thistime = infoPtr-> nextcall - comptime;
	if( thistime < 0 ) {
   		Tcl_SetMaxBlockTime(&zeroTime);
		return;						
	}
	blockTime.usec = thistime;
	Tcl_SetMaxBlockTime(&blockTime);
}

/*
 *----------------------------------------------------------------------
 *
 * ProcessCheckProc --
 *
 *	This procedure is called by Tcl_DoOneEvent to check the serial
 *	event source for events.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May queue an event.
 *
 *----------------------------------------------------------------------
 */

void ProcessCheckProc(data, flags)
    ClientData data;		/* Not used. */
    int flags;			/* Event flags as passed to Tcl_DoOneEvent. */
{
	OdieTask *infoPtr = (OdieTask *) data;
    Tcl_Time blockTime = { 0, 0 };
    OdieTaskEvent *evPtr;
      
	/*
	 * Abort if not processing timer events
	 */
    if (!(flags & TCL_TIMER_EVENTS)) {
		return;
    }
	
    /*
     * Search through the process table for the one whose handle
     * matches the event.  We do this rather than simply dereferencing
     * the handle in the event so that process can be deleted while the
     * event is in the queue.
     */
    if(!ProcessExists(infoPtr)) {
    	return;
    }
    
    /*
     * Queue events for any ready tasks that don't already have events
     * queued (caused by persistent states that won't generate WinSock
     * events).
     */
   	if(infoPtr->state == taskStateActive) {
		if( (comptime > infoPtr-> nextcall) && !(infoPtr->pending)) {
			infoPtr -> pending = 1;
		    evPtr = (OdieTaskEvent *) ckalloc(sizeof(OdieTaskEvent));
		    evPtr->header.proc = ProcessEventProc;
	   		evPtr->infoPtr = infoPtr;
	    	Tcl_QueueEvent((Tcl_Event *) evPtr, TCL_QUEUE_TAIL);
		}
	}
}

/*----------------------------------------------------------------------
 *
 * ProcessEventProc --
 *
 *	This function is invoked by Tcl_ServiceEvent when a process event
 *	reaches the front of the event queue.  This procedure invokes
 *	Tcl_NotifyChannel on the port.
 *
 * Results:
 *	Returns 1 if the event was handled, meaning it should be removed
 *	from the queue.  Returns 0 if the event was not handled, meaning
 *	it should stay on the queue.  The only time the event isn't
 *	handled is if the TCL_TIMER_EVENTS flag bit isn't set.
 *
 * Side effects:
 *	Whatever the notifier callback does.
 *
 *----------------------------------------------------------------------
 */

static int
ProcessEventProc(evPtr,flags)
    Tcl_Event *evPtr;	/* Event to service. */
    int flags;			/* Flags that indicate what events to
						 * handle, such as TCL_FILE_EVENTS. */
{
	
    OdieTaskEvent *taskEvPtr = (OdieTaskEvent *)evPtr;
    OdieTask *infoPtr = taskEvPtr->infoPtr;

	unsigned long startTime, sliceTime;

	
	
	/*
	 * Abort if not processing file events, or a mutual exclusion
	 * condition exists
	 */
    if (!(flags & TCL_TIMER_EVENTS)) {
		return 0;
    }
	
    /*
     * Search through the process table for the one whose handle
     * matches the event.  We do this rather than simply dereferencing
     * the handle in the event so that port can be deleted while the
     * event is in the queue.
     */
    if(!ProcessExists(infoPtr)) {
    	return 1;
    }
    
	startTime = TclpGetClicks();

	if ((infoPtr->cfunct)(infoPtr->data) == TCL_ERROR) {
		Odie_StopProcess(infoPtr->pid);
	}
	sliceTime = TclpGetClicks() - startTime;
	
	/* Compute Task Statistics */
	if(sliceTime > infoPtr->maxtime) infoPtr->maxtime = sliceTime; 
	if(sliceTime < infoPtr->mintime) infoPtr->mintime = sliceTime;					
	infoPtr->avgtime = (infoPtr->avgtime + sliceTime) >> 1;
	
		
	if(infoPtr->avgtime > infoPtr->period) {
		/*
		 * Catch an overtime process
		 */
		 Odie_StopProcess(infoPtr->pid);
		 infoPtr->state = taskStateHalted;
		 return 1;
	}
	
	/*
	 * Indicate next time to run task
	 * and clear the pending flag
	 */
	infoPtr-> nextcall += infoPtr->period;
	infoPtr -> pending = 0;

    return 1;
}

OdieTask * Odie_GetState(pid) 
int pid;
{
	OdieTask *infoPtr;

	for(infoPtr = TaskList.nextPtr ; infoPtr != NULL ; infoPtr = infoPtr -> nextPtr) {
		if(infoPtr->pid == pid) {
			return infoPtr;
		}
	}

	return NULL;
}

OdieTask * Odie_GetStateFromName(name) 
char *name;
{
	OdieTask *infoPtr;

	for(infoPtr = TaskList.nextPtr ; infoPtr != NULL ; infoPtr = infoPtr -> nextPtr) {
		if(strcmp(infoPtr->desc,name) == 0) {
			return infoPtr;
		}
	}
	return NULL;
}

int	ProcessTaskScript(data)
ClientData data;
{
	OdieTclTask *task = (OdieTclTask *)data;
	return Tcl_GlobalEvalObj(task->interp,task->script);
}

/*
 * External C Calls
 */

/* 	
 *  Odie_CreateProcess
 *	Create a new process, that will be called periodically (or is a batch task
 *	if period is zero).
 */
//Tcl_DuplicateObj
int Tcl_ProcessCreate(interp,script)
	Tcl_Interp *interp;
	Tcl_Obj *script;
 {
	OdieTclTask *task;

	task = (OdieTclTask *) Tcl_Alloc(sizeof(OdieTclTask));
	task->interp = interp;
	task->script = Tcl_DuplicateObj(script);	

	return ProcessCreate(ProcessTaskScript,(ClientData *)task);
}

int ProcessCreate(Task_Proc,data)
	Odie_ProcessProc *Task_Proc;
	ClientData data;
{
	OdieTask *infoPtr;
	infoPtr = (OdieTask *) Tcl_Alloc(sizeof(OdieTask));

	infoPtr -> pid = pidCount++;
	
	infoPtr -> desc[0] = NULL;
	infoPtr -> period = 0;
		
	infoPtr -> cfunct = Task_Proc;
	infoPtr -> data = data;
	
	infoPtr -> nextcall = 0;
	infoPtr -> avgtime = 0;
	infoPtr -> maxtime = 0;	
	infoPtr -> mintime = 0xFFFFFFFF;
		
	infoPtr -> state = taskStateStopped;
	infoPtr -> pending = 0;
	infoPtr -> nextPtr = (OdieTask *)NULL;

	ProcessInsert(infoPtr);
	return infoPtr -> pid;
}

void Odie_ResetProcess(pid) {
	OdieTask *infoPtr;
	infoPtr = Odie_GetState(pid);
	if(infoPtr == NULL) return;

	infoPtr -> nextcall = 0;
	infoPtr -> avgtime = 0;
	infoPtr -> maxtime = 0;	
	infoPtr -> mintime = 0xFFFFFFFF;	
	infoPtr -> state = taskStateStopped;
	infoPtr -> pending = 0;
}

void Odie_StartProcess(pid)
int pid;
{
	OdieTask *infoPtr;
	infoPtr = Odie_GetState(pid);
	if(infoPtr == NULL) return;

	Tcl_CreateEventSource(ProcessSetupProc, ProcessCheckProc, (ClientData)infoPtr);
	infoPtr->state = taskStateActive;
	infoPtr->nextcall = TclpGetClicks();
}

void Odie_StopProcess(pid) 
int pid;
{
	OdieTask *infoPtr;
	infoPtr = Odie_GetState(pid);
	if(infoPtr == NULL) return;

	Tcl_DeleteEventSource(ProcessSetupProc, ProcessCheckProc, (ClientData)infoPtr);
	infoPtr->state = taskStateStopped;
}

void Odie_SetProcessPeriod(pid,period)
int pid;
long period;
{
	OdieTask *infoPtr;
	infoPtr = Odie_GetState(pid);
	if(infoPtr == NULL) return;
	infoPtr->period = period;
}

void Odie_KillProcess(pid) 
int pid;
{
	OdieTask *infoPtr;
	infoPtr = Odie_GetState(pid);
	if(infoPtr == NULL) return;
	
	Odie_StopProcess(pid); 
	ProcessRemove(infoPtr);
	
	/*
	 * Free memory used by this system
	 */
	if((infoPtr->cfunct) == (ProcessTaskScript)) {
		OdieTclTask *temp = (OdieTclTask *)infoPtr->data;
		Tcl_DecrRefCount(temp->script);
		Tcl_Free((char *)temp);
	}
	
	Tcl_Free((char *)infoPtr);
}
