/*
 *	Macintosh Serial Support, written by Sean Woods
 *  Modifications Copyright (c) 1999 Woods Design Services.
 *
 *	
 *	Baud rate, parity, data bits, and stop bits are configurable from the
 *	fconfigure command.
 *	Supports blocking and non-blocking modes, as well as file events
 *	Does Not Support flow control (XON/XOFF, DTR, CTS and their ilk)
 *
 *	Questions, Bugs reports, and Creative direction can for now be sent to
 *	Sean Woods, yoda@drexel.edu
 *	(at least until Scriptics decides to make this legit)
 * 
 */
#include "tclInt.h"
#include "tclPort.h"
#include "tclMacInt.h"
#include <Aliases.h>
#include <Errors.h>
#include <Files.h>
#include <Gestalt.h>
#include <Processes.h>
#include <Strings.h>
#include <FSpCompat.h>
#include <MoreFiles.h>
#include <MoreFilesExtras.h>

#include <string.h>

#include <devices.h> 
#include <Serial.h>

#include "tclDevices.h"
			    
static int		ADB_GetHandle _ANSI_ARGS_((ClientData instanceData,
		            int direction, ClientData *handlePtr));

static int		ADB_BlockMode _ANSI_ARGS_((ClientData instanceData,
			    int mode));
static int		ADB_Input _ANSI_ARGS_((ClientData instanceData,
			    char *buf, int toRead, int *errorCode));
static int		ADB_Output _ANSI_ARGS_((ClientData instanceData,
			    char *buf, int toWrite, int *errorCode));
static int		ADB_Close _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp));
			    
static int		ADB_GetOptionProc  _ANSI_ARGS_((ClientData instanceData,Tcl_Interp *interp,
				char *optionName,Tcl_DString *dsPtr));
static int		ADB_SetOptionProc  _ANSI_ARGS_((ClientData instanceData,Tcl_Interp *interp,
				char *optionName,char *newVal));
				
static int 		ADB_Ready _ANSI_ARGS_((PortState *infoPtr));
static void 	ADB_Watch _ANSI_ARGS_((ClientData instanceData,int mask));

/* Internal Structures 
 *
 * Any device configureation information should be stored in here
 */
typedef struct ADB_Port {
	unsigned char *inref;
	unsigned char *outref;
    unsigned int serConfig;	// Port Configuration 
   SerShk serHShake; // Port Handshaking
} ADB_Port;

/*
 * This variable describes the channel type structure for serial based IO.
 */

static Tcl_ChannelType serialChannelType = {
    "serial",					/* Type name. */
    ADB_BlockMode,		/* Set blocking or
                             * non-blocking mode.*/
    ADB_Close,			/* Close proc. */
    ADB_Input,			/* Input proc. */
    ADB_Output,		/* Output proc. */
    NULL,					/* Seek proc. (none) */
    ADB_SetOptionProc,	/* Set option proc. */
    ADB_GetOptionProc,	/* Get option proc. */
    ADB_Watch,			/* Initialize notifier. */
    ADB_GetHandle		/* Get OS handles out of channel. */
};

/* 
 *	Used Internally during the opening sequence to
 *	map out Tcl Port ID numbers to platform specific names 
*/
static ADB_Port serialPortList[] = {
	{"\p.AIn","\p.AOut",0},
	{"\p.BIn","\p.BOut",0},	
	{"\p.CIn","\p.COut",0},
	{"\p.DIn","\p.DOut",0},
	{NULL,NULL}
};


int ADB_Ready(PortState *infoPtr) {
    long length;
    OSErr err;	
    
	if (infoPtr->watchMask & TCL_WRITABLE)
	    return 1;
	
    err = SerGetBuf(infoPtr->inputRef,&length);
    if (err != noErr) return 0;
	if ((infoPtr->watchMask & TCL_READABLE) && (length > 0))
		return 1;
    
 	return 0;
}
 /*
 *----------------------------------------------------------------------
 *
 * OpenADB_Channel--
 *
 *	Opens a Macintosh driver and creates a Tcl channel to control it.
 *
 * Results:
 *	A Tcl channel.
 *
 * Side effects:
 *	Will open a Macintosh device.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
ADB_Open(
	Tcl_Interp *interp,
    int port,			/* Name of file to open. */
    int *errorCodePtr)			/* Where to store error code. */
{
    int channelPermissions;
    Tcl_Channel chan;
    OSErr err;
    short inputRef=0,outputRef=0;
    PortState *fileState;
    char channelName[64];

	/* Detect an already opened port, and open it
	 */
    sprintf(channelName, "serial%d", (int) port);
    
	chan = Tcl_GetChannel(interp,channelName,&channelPermissions);
	if (chan != NULL) {
		return chan;
	} else {
		Tcl_SetStringObj(Tcl_GetObjResult(interp),NULL,0);		
	}
	/* ADB_ Channels Will always be Read/Write */

	channelPermissions = (TCL_READABLE | TCL_WRITABLE);

	err = 0;
	err = OpenDriver(serialPortList[port].inref,&inputRef); 
    if (err != noErr) {
    	printf("Error opening for Read\r\n");
    	goto Error;
    }
	err = OpenDriver(serialPortList[port].outref, &outputRef); 
    if (err != noErr) {
    	printf("Error opening for Write\r\n");
    	goto Error;
  	}  

    fileState = (PortState *) ckalloc((unsigned) sizeof(PortState));
    chan = Tcl_CreateChannel(&serialChannelType, channelName, 
    	(ClientData) fileState, channelPermissions);
    if (chan == (Tcl_Channel) NULL) {
		*errorCodePtr = errno = EFAULT;
		Tcl_SetErrno(errno);
	    CloseDriver(inputRef);
		CloseDriver(outputRef);
		ckfree((char *) fileState);
        return NULL;
    }
    // Default 9600,8,1,No Parity
    serialPortList[port].serConfig = baud9600 | data8 | noParity | stop10;


	/* No Flow Control! */
	serialPortList[port].serHShake.fXOn = 0;
	serialPortList[port].serHShake.fCTS = 0;
	serialPortList[port].serHShake.fInX = 0;	
	serialPortList[port].serHShake.fDTR = 0;
			
	/* Disable Hardware Errors */
	serialPortList[port].serHShake.errs = 0;	
	serialPortList[port].serHShake.evts = 0;	
	
    fileState->portID	= port;
    
	fileState->devChan = chan;
    fileState->inputRef = inputRef;
    fileState->outputRef = outputRef;
    fileState->pending = false;
    fileState->watchMask = 0;
    fileState->validMask = (TCL_READABLE | TCL_WRITABLE | TCL_EXCEPTION);
	fileState->blocking = false;

	fileState->readyProc = ADB_Ready;
	
	/* Clear Input Buffer */
	err = SerSetBuf(inputRef, 0, 0);
    if (err != noErr) goto Error;

	/* Port Handshake Settings  */    
	err = Control(inputRef,14,&(serialPortList[port].serHShake)); 
    if (err != noErr) goto Error;
	err = Control(outputRef,14,&(serialPortList[port].serHShake)); 
    if (err != noErr) goto Error;

	/* Port Communication Settings/Reset  */
	err = SerReset(inputRef,serialPortList[port].serConfig); 
    if (err != noErr) goto Error;
    err = SerReset(outputRef,serialPortList[port].serConfig); 
    if (err != noErr) goto Error;
    
    /* Add to notification manager */
    DeviceInsert(fileState);
    return chan;
    
Error:
	switch(err) {
		case controlErr:
		case statusErr:
		case readErr:
		case writErr:
			*errorCodePtr = errno = EIO;
			break;
		default:
			*errorCodePtr = errno = TclMacOSErrorToPosixError(err);
	}
	/* Be Sure to close out the ports before we bail */
	if (outputRef != 0)
    	CloseDriver(outputRef);
    if (inputRef != 0)
		CloseDriver(inputRef);	
	
	Tcl_SetErrno(errno);
	return NULL;
}

/* 
 *----------------------------------------------------------------------
 *
 */
void ADB_Reset (int port)
{
	short inputRef, outputRef;

	OpenDriver(serialPortList[port].inref,&inputRef); 
	OpenDriver(serialPortList[port].outref, &outputRef);  
	CloseDriver(inputRef);
	CloseDriver(outputRef);		
}

/*
 *----------------------------------------------------------------------
 *
 * ADB_BlockMode --
 *
 *	Set blocking or non-blocking mode on a serial channel.
 *  (Since this is done on the software level, never really errors out)
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

static int
ADB_BlockMode(
    ClientData instanceData,		/* Unused. */
    int mode)				/* The mode to set. */
{
	/* Seems that yes means no, 1 - nonblocking 0 - blocking */
    PortState *fileState = (PortState *) instanceData;
    fileState->blocking = !mode;	
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * DeviceWatch --
 *
 *	Initialize the notifier to watch handles from this channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
ADB_Watch(
    ClientData instanceData,		/* The file state. */
    int mask)				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    PortState *thisPtr = (PortState *) instanceData;
    /*
     * Since the file is always ready for events, we set the block time
     * to zero so we will poll.
     */
    thisPtr->watchMask = mask & thisPtr->validMask;
}


/*
 *----------------------------------------------------------------------
 *
 * ADB_Close --
 *
 *	Closes the IO channel.
 *
 * Results:
 *	0 if successful, the value of errno if failed.
 *
 * Side effects:
 *	Closes the physical channel
 *
 *----------------------------------------------------------------------
 */

static int
ADB_Close(
    ClientData instanceData,	/* Unused. */
    Tcl_Interp *interp)		/* Unused. */
{
    PortState *fileState = (PortState *) instanceData;

    CloseDriver(fileState->inputRef);
	CloseDriver(fileState->outputRef);

    /* Remove Notification List */
	DeviceRemove(fileState);

	return 0;
}
/*
 *----------------------------------------------------------------------
 *
 * ADB_Input --
 *
 *	Reads input from the IO channel into the buffer given. Returns
 *	count of how many bytes were actually read, and an error indication.
 *
 * Results:
 *	A count of how many bytes were read is returned and an error
 *	indication is returned in an output argument.
 *
 * Side effects:
 *	Reads input from the actual channel.
 *
 *----------------------------------------------------------------------
 */

int
ADB_Input(
    ClientData instanceData,	/* Unused. */
    char *buffer,				/* Where to store data read. */
    int bufSize,				/* How much space is available
                                 * in the buffer? */
    int *errorCodePtr)			/* Where to store error code. */
{
    PortState *fileState = (PortState *) instanceData;
    long length;
    OSErr err;

	if(fileState->blocking) {	
		do
		{
			/* Wait until information is available */
	    	err = SerGetBuf(fileState->inputRef,&length);
	    	if (err != noErr) return -1;
			Tcl_DoOneEvent(0);
	    } while(length == 0);
	}
	err = SerGetBuf(fileState->inputRef,&length);
	if (err != noErr) return -1;
		
    if(length > 0) {
    	if(bufSize > length)
    		bufSize = length;
    	length = bufSize;
		err = FSRead(fileState->inputRef, &length, buffer);
	    if ((err == noErr) || (err == eofErr)) {
			return length;
	    } else {
			*errorCodePtr = errno = TclMacOSErrorToPosixError(err);
			Tcl_SetErrno(errno);
			return -1;
	    }
	} else {
		return 0;
	}
}

/*
 *----------------------------------------------------------------------
 *
 * ADB_Output--
 *
 *	Writes the given output on the IO channel. Returns count of how
 *	many characters were actually written, and an error indication.
 *
 * Results:
 *	A count of how many characters were written is returned and an
 *	error indication is returned in an output argument.
 *
 * Side effects:
 *	Writes output on the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
ADB_Output(
    ClientData instanceData,		/* Unused. */
    char *buffer,			/* The data buffer. */
    int toWrite,			/* How many bytes to write? */
    int *errorCodePtr)			/* Where to store error code. */
{
    PortState *fileState = (PortState *) instanceData;
 	Byte dataTransmitted = 0, Listen, ADBReg;
 	
    long length = toWrite;
	unsigned short	ADBIOCompletionPPC[3] = {0x14BC, 0x0001, 0x4E75};
    OSErr err;

    *errorCodePtr = 0;
    errno = 0;
	if (CountADBs() == 0) return TCL_ERROR; //no devices connected to the bus
	Listen = (theAddress << 4) + 8 + ADBReg;     
    
	#if GENERATINGPOWERPC
	 myErr = ADBOp ((Ptr)&dataTransmitted,(UniversalProcPtr)ADBIOCompletionPPC,(Ptr)buffer,Listen);
    #else
	 myErr = ADBOp ((Ptr)&dataTransmitted,(ADBServiceRoutineUPP)ADBIOCompletion68k,(Ptr)buffer,Listen);
    #endif   
 
    if (err != noErr) {
	 	switch(err) {
			case controlErr:
			case statusErr:
			case readErr:
			case writErr:
				*errorCodePtr = errno = EIO;
				break;
			default:
				*errorCodePtr = errno = TclMacOSErrorToPosixError(err);		
		}
		Tcl_SetErrno(errno);
		return -1;
    }
    return length;
}


/*
 *----------------------------------------------------------------------
 *
 * ADB_GetHandle --
 *
 *	Called from Tcl_GetChannelFile to retrieve OS handles from inside
 *	a file based channel.
 *
 * Results:
 *	The appropriate handle or NULL if not present. 
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
ADB_GetHandle(
    ClientData instanceData,		/* The file state. */
    int direction,			/* Which handle to retrieve? */
    ClientData *handlePtr)
{
    if (direction == TCL_READABLE) {
		*handlePtr = (ClientData) ((PortState*)instanceData)->inputRef;
		return TCL_OK;
    }
    if (direction == TCL_WRITABLE) {
		*handlePtr = (ClientData) ((PortState*)instanceData)->outputRef;
		return TCL_OK;
    }
    return TCL_ERROR;
}
/*
 *----------------------------------------------------------------------
 *
 * ADB_GetOptionProc --
 *
 *	Computes an option value for a serial based channel, or a
 *	list of all options and their values.
 *
 *	Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *	A standard Tcl result. The value of the specified option or a
 *	list of all options and	their values is returned in the
 *	supplied DString.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
ADB_SetOptionProc(
    ClientData instanceData, 		/* Socket state. */
    Tcl_Interp *interp,                 /* For error reporting - can be NULL.*/
    char *optionName, 			/* Name of the option to
                                         * retrieve the value for, or
                                         * NULL to get all options and
                                         * their values. */
    char *newVal)			/* Where to store the computed
                                         * value; initialized by caller. */
{
	PortState *fileState = (PortState *) instanceData;
	int port;
	unsigned int temp;
	unsigned int dataconv;
	OSErr	err;
	
	port = fileState->portID;
	temp = serialPortList[port].serConfig;

    /*
     * Determine which options we need to modify.
     */

	if (!strcmp(optionName, "-baud")) {
		int baud;
		/* Mask Out Baud Bits */
		temp = temp & 0xFE00;
		dataconv = atoi(newVal);
		switch (dataconv) {
			case 150: 		baud = baud150; break;
			case 300: 		baud = baud300; break;
			case 600: 		baud = baud600; break;
			case 1200:		baud = baud1200; break;
			case 1800: 		baud = baud1800; break;
			case 2400: 		baud = baud2400; break;
			case 3600: 		baud = baud3600; break;
			case 4800:		baud = baud4800; break;
			case 7200: 		baud = baud7200; break;
			case 9600:		baud = baud9600; break;
			case 14400: 	baud = baud14400; break;
			case 19200:		baud = baud19200; break;
			case 28800: 	baud = baud28800; break;
			case 38400: 	baud = baud38400; break;
			case 57600: 	baud = baud57600; break;
			default:
			  if (interp) {
                Tcl_AppendResult(interp, "bad value for -baud: ",
                        "valid settings are",
                        "150 300 600 1200 ",
                        "1800 2400 3600 4800 ",
                        "7200 9600 14400 19200 ",
                        "28800 38400 57600",
                        (char *) NULL);
                return TCL_ERROR;
               }	
		}
		temp = temp | baud;	
	} else if (!strcmp(optionName, "-databits")) {
		int bits;
		temp = temp & 0xF3FF;
		dataconv = atoi(newVal);
		switch (dataconv) {
			case 5: 		bits = data5; break;
			case 6: 		bits = data6; break;
			case 7: 		bits = data7; break;
			case 8:			bits = data8; break;
			default:
			  if (interp) {
                Tcl_AppendResult(interp, "bad value for -databits: ",
                        "valid settings are",
                        "5 6 7 8 ",
                        (char *) NULL);
                return TCL_ERROR;
               }	
		}
		temp = temp | bits;	
	} else if (!strcmp(optionName, "-parity")) {
		temp = temp & 0xCFFF;	
		if (!strcmp(newVal, "none")) {
			temp = temp | noParity;	
		} else if (!strcmp(newVal, "odd")) {
			temp = temp | oddParity;	
		} else if (!strcmp(newVal, "even")) {
			temp = temp | evenParity;
		} else {		
			  if (interp) {
                Tcl_AppendResult(interp, "bad value for -parity: ",
                        "valid settings are",
                        "none odd even ",
                        (char *) NULL);
                return TCL_ERROR;
               }		
		}
	} else if (!strcmp(optionName, "-stopbits")) {
		temp = temp & 0x3FFF;
		
		if (!strcmp(newVal, "1")) {
			temp = temp | 0x4000;	
		} else if (!strcmp(newVal, "1.5")) {
			temp = temp | 0x8000;	
		} else if (!strcmp(newVal, "2")) {
			temp = temp | 0xC000;	
		} else {		
			  if (interp) {
                Tcl_AppendResult(interp, "bad value for -stopbits: ",
                        "valid settings are",
                        "1 1.5 2 ",
                        (char *) NULL);
                return TCL_ERROR;
               }		
		}
	} else {
		return Tcl_BadChannelOption(interp, optionName, 
		    	"baud databits parity stopbits");
	}
	serialPortList[port].serConfig = temp;
	/* Port Communication Settings/Reset  */
	err = SerReset(fileState->inputRef,serialPortList[port].serConfig); 
    if (err != noErr) goto Error;
    err = SerReset(fileState->outputRef,serialPortList[port].serConfig); 
    if (err != noErr) goto Error;
    
    return TCL_OK;
    
Error:
	switch(err) {
		case controlErr:
		case statusErr:
		case readErr:
		case writErr:
			errno = EIO;
			break;
		default:
			errno = TclMacOSErrorToPosixError(err);		
	}
	Tcl_SetErrno(errno);
	return TCL_ERROR;
}
/*
 *----------------------------------------------------------------------
 *
 * ADB_GetOptionProc --
 *
 *	Computes an option value for a serial based channel, or a
 *	list of all options and their values.
 *
 *	Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *	A standard Tcl result. The value of the specified option or a
 *	list of all options and	their values is returned in the
 *	supplied DString.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
ADB_GetOptionProc(
    ClientData instanceData, 		/* Socket state. */
    Tcl_Interp *interp,                 /* For error reporting - can be NULL.*/
    char *optionName, 			/* Name of the option to
                                         * retrieve the value for, or
                                         * NULL to get all options and
                                         * their values. */
    Tcl_DString *dsPtr)			/* Where to store the computed
                                         * value; initialized by caller. */
{
	PortState *fileState = (PortState *) instanceData;
	int doAll=0,
		doBaud=0,
		doDataBits=0,
		doParity=0,
		doStopBits=0;
    char buffer[128];
	int port;
	
	port = fileState->portID;
    /*
     * Determine which options we need to do.  Do all of them
     * if optionName is NULL.
     */

    if (optionName == (char *) NULL || optionName[0] == '\0') {
        doAll = true;
    } else {
		if (!strcmp(optionName, "-baud")) {
		    doBaud = true;
		} else if (!strcmp(optionName, "-databits")) {
		    doDataBits = true;
		} else if (!strcmp(optionName, "-parity")) {
		    doParity = true;
		} else if (!strcmp(optionName, "-stopbits")) {
		    doStopBits = true;
		} else {
		    return Tcl_BadChannelOption(interp, optionName, 
		    		"baud databits parity stopbits");
		}
    }


    /*
     * Get the sockname for the socket.
     */

    if (doAll || doBaud) {
    	int baud;
		if (doAll) {
		    Tcl_DStringAppendElement(dsPtr, "-baud");
		}
		switch (serialPortList[port].serConfig & 0x1FF) {
			case baud150: baud = 150; break;
			case baud300: baud = 300; break;
			case baud600: baud = 600; break;
			case baud1200: baud = 1200; break;
			case baud1800: baud = 1800; break;
			case baud2400: baud = 2400; break;
			case baud3600: baud = 3600; break;
			case baud4800: baud = 4800; break;
			case baud7200: baud = 7200; break;
			case baud9600: baud = 9600; break;
			case baud14400: baud = 14400; break;
			case baud19200: baud = 19200; break;
			case baud28800: baud = 28800; break;
			case baud38400: baud = 38400; break;
			case baud57600: baud = 57600; break;
		}
		sprintf(buffer, "%d", baud);
		Tcl_DStringAppendElement(dsPtr, buffer);
    }
    if (doAll || doDataBits) {
		if (doAll) {
		    Tcl_DStringAppendElement(dsPtr, "-databits");
		}
		switch (serialPortList[port].serConfig & 0xC00) {
			case data5:		sprintf(buffer, "5"); break;
			case data6: 	sprintf(buffer, "6"); break;
			case data7: 	sprintf(buffer, "7"); break;
			case data8: 	sprintf(buffer, "8"); break;
		}
		Tcl_DStringAppendElement(dsPtr, buffer);
    }
    if (doAll || doParity) {
		if (doAll) {
		    Tcl_DStringAppendElement(dsPtr, "-parity");
		}
		switch (serialPortList[port].serConfig & 0x3000) {
			case noParity:		sprintf(buffer, "none"); break;
			case oddParity: 	sprintf(buffer, "odd"); break;
			case evenParity: 	sprintf(buffer, "even"); break;
		}
		Tcl_DStringAppendElement(dsPtr, buffer);
    }
    if (doAll || doStopBits) {
		if (doAll) {
		    Tcl_DStringAppendElement(dsPtr, "-stopbits");
		}
		switch (serialPortList[port].serConfig & 0xC000) {
			case 0x4000:	sprintf(buffer, "1"); break;
			case 0x8000: 	sprintf(buffer, "1.5"); break;
			case 0xC000: 	sprintf(buffer, "2"); break;
		}
		Tcl_DStringAppendElement(dsPtr, buffer);
    }
    return TCL_OK;
}
