/*
 *  Macintosh Serial Support, written by Sean Woods
 *
 *  Baud rate, parity, data bits, and stop bits are configurable from the
 *  fconfigure command.
 *  Supports blocking and non-blocking modes, as well as file events
 *  Does Not Support flow control (XON/XOFF, DTR, CTS and their ilk)
 *
 *  Questions, Bugs reports, and Creative direction can for now be sent to
 *  Sean Woods, yoda@drexel.edu
 *
 *  Copyright (c) 1999 Woods Design Services.
 *
 * See the file "odie.license" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tcl.h"
#include "tclMac.h"
#include "tclErrno.h"

/*
 * Tk renames Status
 * and the compiler yells at you
 */
#undef Status

#include <Aliases.h>
#include <Errors.h>
#include <Gestalt.h>
#include <Processes.h>
#include <Strings.h>
#include <string.h>
#include <devices.h>
#include <LowMem.h>
#include <CommResources.h>
#include <CRMSerialDevices.h>
#include <Serial.h>

#include "odieDevices.h"
 
static int SerialGetHandle _ANSI_ARGS_((ClientData instanceData,
	int direction, ClientData *handlePtr));
static int SerialBlockMode _ANSI_ARGS_((ClientData instanceData,
	int mode));
static int SerialInput _ANSI_ARGS_((ClientData instanceData,
	char *buf, int toRead, int *errorCode));
static int SerialOutput _ANSI_ARGS_((ClientData instanceData,
	char *buf, int toWrite, int *errorCode));
static int  SerialClose _ANSI_ARGS_((ClientData instanceData,
	Tcl_Interp *interp));

static int SerialGetOptionProc  _ANSI_ARGS_((ClientData instanceData,
	Tcl_Interp *interp,
	char *optionName,Tcl_DString *dsPtr));
static int  SerialSetOptionProc  _ANSI_ARGS_((ClientData instanceData,
	Tcl_Interp *interp,
	char *optionName,char *newVal));

static int SerialReady _ANSI_ARGS_((PortState *infoPtr));
static void SerialWatch _ANSI_ARGS_((ClientData instanceData,int mask));

static int TclMacDevErrorToPosixError (OSErr err);

/*
 * This set is from the Serial Driver Apocrypha by Apple DTS
 */

static OSErr 		CloseSerialDrivers(SInt16 inRefNum, SInt16 outRefNum);
static Boolean 		DriverIsOpen(ConstStr255Param driverName);
static OSErr 		OpenOneSerialDriver(ConstStr255Param driverName,
	short *refNum);
static Tcl_Channel	OpenSerialChannel _ANSI_ARGS_((int port, 
	int *errorCodePtr));
static OSErr 		OpenSerialDrivers _ANSI_ARGS_((ConstStr255Param inName,
	ConstStr255Param outName,
	short *inRefNum, short *outRefNum));
static Boolean 		SerialArbitrationExists(void);


/*
 * Internal Structures
 *
 * Any device configureation information should be stored in here
 */

enum {
    kModem = 0,
    kPrinter,
    kCom1,
    kCom2,
    kInternalModem
} portList;

typedef struct SerialPort {
	unsigned char *inref;
	unsigned char *outref;
    unsigned int serConfig;	/* Port Configuration */
    SerShk serHShake; /* Port Handshaking */
} SerialPort;

/*
 * This variable describes the channel type structure for serial based IO.
 */

static Tcl_ChannelType serialChannelType = {
    "serial",					/* Type name. */
    SerialBlockMode,		/* Set blocking or
                             * non-blocking mode.*/
    SerialClose,			/* Close proc. */
    SerialInput,			/* Input proc. */
    SerialOutput,		/* Output proc. */
    NULL,					/* Seek proc. (none) */
    SerialSetOptionProc,	/* Set option proc. */
    SerialGetOptionProc,	/* Get option proc. */
    SerialWatch,			/* Initialize notifier. */
    SerialGetHandle		/* Get OS handles out of channel. */
};

/*
 *	Used Internally during the opening sequence to
 *	map out Tcl Port ID numbers to platform specific names
 *
 *	We have a special case for powerbook internal modems,
 *	which is a port 5, but will not map transparently over
 *	port 0 (modem) We really don't need to confuse users
 *	any more than we honestly have to.
 */

static SerialPort serialPortList[] = {
	{"\p.AIn","\p.AOut",0},
	{"\p.BIn","\p.BOut",0},
	{"\p.CIn","\p.COut",0},
	{"\p.DIn","\p.DOut",0},
	{"\p.InternalModemIn","\p.InternalModemOut",0},
	{NULL,NULL}
};

/*
 * Tell if we have a serial port arbitrator for working around problems
 * with ARA
 */

enum {
    gestaltSerialPortArbitratorAttr = 'arb ',
    gestaltSerialPortArbitratorExists = 0
};

extern int errno;

/*
 *----------------------------------------------------------------------
 *
 * SerialReady --
 *
 *      Tests whether the given serial port is readable or writable
 *      depending on the value of watchMask.
 *
 * Results:
 *	1 if readable or writable, 0 if not
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */

int
SerialReady(
    PortState *infoPtr)
{
    long length;
    OSErr err;

    /*
     * For now, we are always writable.
     */
    
    if (infoPtr->watchMask & TCL_WRITABLE) {
	return 1;
    }
    
    err = SerGetBuf(infoPtr->inputRef, &length);
    if (err != noErr) {
	return 0;
    }
    if ((infoPtr->watchMask & TCL_READABLE) && (length > 0)) {
	return 1;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * OpenSerialChannel --
 *
 *	Opens a Macintosh driver and creates a Tcl channel to control it.
 *
 * Results:
 *	A Tcl channel.
 *
 * Side effects:
 *	Will open a Macintosh device.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
SerialOpen(
    Tcl_Interp *interp,
    int port,			/* Name of file to open. */
    int *errorCodePtr)			/* Where to store error code. */
{
    int channelPermissions;
    Tcl_Channel chan;
    OSErr err;
    short inputRef=0,outputRef=0;
    PortState *serialState;
    char channelName[64];
    
    /* 
     * Detect a bad port address
     */
     if(port > kInternalModem || port < 0) {
    	*errorCodePtr = errno = EFAULT;
	Tcl_SetErrno(errno);
	return NULL;
    }
    
    /*
     * Detect an already opened port, and return it
     */
    
    sprintf(channelName, "serial%d", (int) port);
    chan = Tcl_GetChannel(interp, channelName, &channelPermissions);
    if (chan != NULL) {
	return chan;
    } else {
	Tcl_SetStringObj(Tcl_GetObjResult(interp), NULL, 0);
    }
    
    /*
     * Serial Channels Will always be Read/Write
     */

    channelPermissions = (TCL_READABLE | TCL_WRITABLE);

    err = OpenSerialDrivers(serialPortList[port].inref,
	    serialPortList[port].outref,
	    &inputRef,
	    &outputRef);

    if (err != noErr) {
    	if(port == kModem) {
	    /*
	     * Transparently map the modem port to the internal
	     * modem port on powerbooks
	     */
	    
	    port = kInternalModem;
	    err = OpenSerialDrivers(serialPortList[port].inref,
		    serialPortList[port].outref,
		    &inputRef, &outputRef);
	}
	if (err != noErr) {
	    goto Error;
	}
    }

    serialState = (PortState *) ckalloc((unsigned) sizeof(PortState));
    chan = Tcl_CreateChannel(&serialChannelType, channelName,
	    (ClientData) serialState, channelPermissions);
    if (chan == (Tcl_Channel) NULL) {
	*errorCodePtr = errno = EFAULT;
	Tcl_SetErrno(errno);
	CloseSerialDrivers(inputRef, outputRef);
	ckfree((char *) serialState);
        return NULL;
    }
    
    /*
     * Default 9600,8,1,No Parity
     */
    
    serialPortList[port].serConfig = baud9600 | data8 | noParity | stop10;


    /*
     * No Flow Control!
     */
    
    serialPortList[port].serHShake.fXOn = 0;
    serialPortList[port].serHShake.fCTS = 0;
    serialPortList[port].serHShake.fInX = 0;
    serialPortList[port].serHShake.fDTR = 0;

    /*
     * Disable Hardware Errors
     */
    
    serialPortList[port].serHShake.errs = 0;
    serialPortList[port].serHShake.evts = 0;

    serialState->portID	= port;

    serialState->devChan = chan;
    serialState->inputRef = inputRef;
    serialState->outputRef = outputRef;
    serialState->pending = false;
    serialState->watchMask = 0;
    serialState->validMask = channelPermissions;
    serialState->blocking = false;

    serialState->readyProc = SerialReady;

    /*
     * Clear Input Buffer
     */
    
    err = SerSetBuf(inputRef, 0, 0);
    if (err != noErr) goto Error;

    /*
     * Port Handshake Settings
     */
    
    err = Control(inputRef, 14, &(serialPortList[port].serHShake));
    if (err != noErr) goto Error;
    err = Control(outputRef, 14, &(serialPortList[port].serHShake));
    if (err != noErr) goto Error;

    /*
     * Port Communication Settings/Reset
     */
    
    err = SerReset(inputRef, serialPortList[port].serConfig);
    if (err != noErr) {
        goto Error;
    }
    err = SerReset(outputRef, serialPortList[port].serConfig);
    if (err != noErr) {
        goto Error;
    }

    return chan;

    Error:
    
    /* 
     * Map err to POSIX style error code
     */
     
     *errorCodePtr = errno = TclMacDevErrorToPosixError(err);
 
    /*
     * Be Sure to close out the ports before we bail
     */
    
    CloseSerialDrivers(inputRef, outputRef);

    Tcl_SetErrno(errno);
    
    /* 
     * Return an error condition
     */
     
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * SerialReset --
 *
 *      Resets the input port
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Closes the drivers on the given port (opening them first if
 *      they are not already open.
 *
 *----------------------------------------------------------------------
 */

void SerialReset (int port)
{
    short inputRef, outputRef;

    OpenSerialDrivers(serialPortList[port].inref,serialPortList[port].outref,
		    &inputRef, &outputRef);
    CloseSerialDrivers(inputRef, outputRef);
}

/*
 *----------------------------------------------------------------------
 *
 * SerialBlockMode --
 *
 *	Set blocking or non-blocking mode on a serial channel.
 *      (Since this is done on the software level, never really errors out)
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

static int
SerialBlockMode(
    ClientData instanceData,	/* Unused. */
    int mode)			/* The mode to set. */
{
    /*
     * Seems that yes means no, 1 - nonblocking 0 - blocking
     */
    
    PortState *serialState = (PortState *) instanceData;
    serialState->blocking = !mode;
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * DeviceWatch --
 *
 *	Initialize the notifier to watch handles from this channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
SerialWatch(
    ClientData instanceData,		/* The file state. */
    int mask)				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    PortState *infoPtr = (PortState *) instanceData;
    int oldMask = infoPtr->watchMask;


    infoPtr->watchMask = mask  & infoPtr->validMask;
    if (infoPtr->watchMask) {
	if (!oldMask) {
	    /*
	     * Add to notification manager
	     */
	    
	    DeviceInsert(infoPtr);
	}
    } else {
	if (oldMask) {
	    /*
	     * Remove the serial from the list of watched serials.
	     */
	    DeviceRemove(infoPtr);
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SerialClose --
 *
 *	Closes the IO channel.
 *
 * Results:
 *	0 if successful, the value of errno if failed.
 *
 * Side effects:
 *	Closes the physical channel
 *
 *----------------------------------------------------------------------
 */

static int
SerialClose(
    ClientData instanceData,	/* Unused. */
    Tcl_Interp *interp)		/* Unused. */
{
    PortState *serialState = (PortState *) instanceData;
    int errorCode = 0;
    OSErr err;
    
    SerialWatch(instanceData,0);
    err = CloseSerialDrivers(serialState->inputRef, serialState->outputRef);

    /*
     * Map err to a POSIX standard error code
     */
    if (err != noErr) {
	errorCode = errno = TclMacDevErrorToPosixError(err);
	/*
	 * Removed because we can recover with a port reset
	 * plus failure to close the port is peanuts compared
	 * to the havok the user will experience recovering from
	 * a crash
	 *
	 * (Usually ;)
	 *
	 panic("error during port close");
	 */
    }

    ckfree((char *) serialState);
    Tcl_SetErrno(errorCode);
    return errorCode;
}
/*
 *----------------------------------------------------------------------
 *
 * SerialInput --
 *
 *	Reads input from the IO channel into the buffer given. Returns
 *	count of how many bytes were actually read, and an error indication.
 *
 * Results:
 *	A count of how many bytes were read is returned and an error
 *	indication is returned in an output argument.
 *
 * Side effects:
 *	Reads input from the actual channel.
 *
 *----------------------------------------------------------------------
 */

int
SerialInput(
    ClientData instanceData,	/* Unused. */
    char *buffer,	        /* Where to store data read. */
    int bufSize,		/* How much space is available
				 * in the buffer? */
    int *errorCodePtr)		/* Where to store error code. */
{
    PortState *serialState = (PortState *) instanceData;
    long length;
    OSErr err;

    if(serialState->blocking) {
	do {
	    length = bufSize;

	    /*
	     * Wait until information is available
	     */

	    err = SerGetBuf(serialState->inputRef,&length);
	    Tcl_DoOneEvent(0);
	} while(length == 0);
    }
    err = SerGetBuf(serialState->inputRef,&length);

    if(length > 0) {
    	if(bufSize > length)
	    bufSize = length;

    	length = bufSize;
	err = FSRead(serialState->inputRef, &length, buffer);
	if ((err == noErr) || (err == eofErr)) {
	    return length;
	} else {
	    *errorCodePtr = errno = TclMacDevErrorToPosixError(err);
	    Tcl_SetErrno(errno);
	    return -1;
	}
    } else {
	return 0;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * SerialOutput--
 *
 *	Writes the given output on the IO channel. Returns count of how
 *	many characters were actually written, and an error indication.
 *
 * Results:
 *	A count of how many characters were written is returned and an
 *	error indication is returned in an output argument.
 *
 * Side effects:
 *	Writes output on the actual channel.
 *
 *----------------------------------------------------------------------
 */

static int
SerialOutput(
    ClientData instanceData,		/* Unused. */
    char *buffer,			/* The data buffer. */
    int toWrite,			/* How many bytes to write? */
    int *errorCodePtr)			/* Where to store error code. */
{
    PortState *serialState = (PortState *) instanceData;
    long length = toWrite;
    SerStaRec inSerStat, outSerStat;
    OSErr err;

    *errorCodePtr = 0;
    errno = 0;
    err = SerStatus(serialState->outputRef, &inSerStat);
    err = SerStatus(serialState->inputRef, &outSerStat);
    err = FSWrite(serialState->outputRef, &length, buffer);
    if (err != noErr) {
	switch(err) {
	    case controlErr:
	    case statusErr:
	    case readErr:
	    case writErr:
		*errorCodePtr = errno = EIO;
		break;
	    default:
		*errorCodePtr = errno =
		    TclMacDevErrorToPosixError(err);
	}
	Tcl_SetErrno(errno);
	return -1;
    }
    return length;
}


/*
 *----------------------------------------------------------------------
 *
 * SerialGetHandle --
 *
 *	Called from Tcl_GetChannelFile to retrieve OS handles from inside
 *	a file based channel.
 *
 * Results:
 *	The appropriate handle or NULL if not present.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SerialGetHandle(
    ClientData instanceData,		/* The file state. */
    int direction,			/* Which handle to retrieve? */
    ClientData *handlePtr)
{
    if (direction == TCL_READABLE) {
	*handlePtr = (ClientData) ((PortState*)instanceData)->inputRef;
	return TCL_OK;
    }
    if (direction == TCL_WRITABLE) {
	*handlePtr = (ClientData)
	    ((PortState*)instanceData)->outputRef;
	return TCL_OK;
    }
    return TCL_ERROR;
}
/*
 *----------------------------------------------------------------------
 *
 * SerialGetOptionProc --
 *
 *	Computes an option value for a serial based channel, or a
 *	list of all options and their values.
 *
 *	Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *	A standard Tcl result. The value of the specified option or a
 *	list of all options and	their values is returned in the
 *	supplied DString.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SerialSetOptionProc(
    ClientData instanceData, 		/* Socket state. */
    Tcl_Interp *interp,                 /* For error reporting - can be NULL.*/
    char *optionName, 			/* Name of the option to
                                         * retrieve the value for, or
                                         * NULL to get all options and
                                         * their values. */
    char *newVal)			/* Where to store the computed
                                         * value; initialized by caller. */
{
    PortState *serialState = (PortState *) instanceData;
    int port;
    unsigned int temp;
    unsigned int dataconv;
    OSErr	err;

    port = serialState->portID;
    temp = serialPortList[port].serConfig;

    /*
     * Determine which options we need to modify.
     */

    if (!strcmp(optionName, "-baud")) {
	int baud;
	/*
	 * Mask Out Baud Bits
	 */
	
	temp = temp & 0xFE00;
	dataconv = atoi(newVal);
	switch (dataconv) {
	    case 150: 		baud = baud150; break;
	    case 300: 		baud = baud300; break;
	    case 600: 		baud = baud600; break;
	    case 1200:		baud = baud1200; break;
	    case 1800: 		baud = baud1800; break;
	    case 2400: 		baud = baud2400; break;
	    case 3600: 		baud = baud3600; break;
	    case 4800:		baud = baud4800; break;
	    case 7200: 		baud = baud7200; break;
	    case 9600:		baud = baud9600; break;
	    case 14400: 	baud = baud14400; break;
	    case 19200:		baud = baud19200; break;
	    case 28800: 	baud = baud28800; break;
	    case 38400: 	baud = baud38400; break;
	    case 57600: 	baud = baud57600; break;
	    default:
		if (interp) {
		    Tcl_AppendResult(interp, "bad value for -baud: ",
			    "valid settings are",
			    "150 300 600 1200 ",
			    "1800 2400 3600 4800 ",
			    "7200 9600 14400 19200 ",
			    "28800 38400 57600",
			    (char *) NULL);
		    return TCL_ERROR;
		}
	}
	temp = temp | baud;
    } else if (!strcmp(optionName, "-databits")) {
	int bits;
	temp = temp & 0xF3FF;
	dataconv = atoi(newVal);
	switch (dataconv) {
	    case 5: 		bits = data5; break;
	    case 6: 		bits = data6; break;
	    case 7: 		bits = data7; break;
	    case 8:			bits = data8; break;
	    default:
		if (interp) {
		    Tcl_AppendResult(interp, "bad value for -databits: ",
			    "valid settings are",
			    "5 6 7 8 ",
			    (char *) NULL);
		    return TCL_ERROR;
		}
	}
	temp = temp | bits;
    } else if (!strcmp(optionName, "-parity")) {
	temp = temp & 0xCFFF;
	if (!strcmp(newVal, "none")) {
	    temp = temp | noParity;
	} else if (!strcmp(newVal, "odd")) {
	    temp = temp | oddParity;
	} else if (!strcmp(newVal, "even")) {
	    temp = temp | evenParity;
	} else {
	    if (interp) {
                Tcl_AppendResult(interp, "bad value for -parity: ",
                        "valid settings are",
                        "none odd even ",
                        (char *) NULL);
                return TCL_ERROR;
	    }
	}
    } else if (!strcmp(optionName, "-stopbits")) {
	temp = temp & 0x3FFF;

	if (!strcmp(newVal, "1")) {
	    temp = temp | 0x4000;
	} else if (!strcmp(newVal, "1.5")) {
	    temp = temp | 0x8000;
	} else if (!strcmp(newVal, "2")) {
	    temp = temp | 0xC000;
	} else {
	    if (interp) {
                Tcl_AppendResult(interp, "bad value for -stopbits: ",
                        "valid settings are",
                        "1 1.5 2 ",
                        (char *) NULL);
                return TCL_ERROR;
	    }
	}
    } else {
	return Tcl_BadChannelOption(interp, optionName,
		"baud databits parity stopbits");
    }
    serialPortList[port].serConfig = temp;
    /*
     * Port Communication Settings/Reset
     */
    
    err = SerReset(serialState->inputRef,serialPortList[port].serConfig);
    if (err != noErr) goto Error;
    err = SerReset(serialState->outputRef,serialPortList[port].serConfig);
    if (err != noErr) goto Error;

    return TCL_OK;

    Error:
    errno = TclMacDevErrorToPosixError(err);
    Tcl_SetErrno(errno);
    return TCL_ERROR;
}
/*
 *----------------------------------------------------------------------
 *
 * SerialGetOptionProc --
 *
 *	Computes an option value for a serial based channel, or a
 *	list of all options and their values.
 *
 *	Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *	A standard Tcl result. The value of the specified option or a
 *	list of all options and	their values is returned in the
 *	supplied DString.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SerialGetOptionProc(
    ClientData instanceData, 		/* Socket state. */
    Tcl_Interp *interp,                 /* For error reporting - can be NULL.*/
    char *optionName, 			/* Name of the option to
                                         * retrieve the value for, or
                                         * NULL to get all options and
                                         * their values. */
    Tcl_DString *dsPtr)			/* Where to store the computed
                                         * value; initialized by caller. */
{
    PortState *serialState = (PortState *) instanceData;
    int doAll=0,
	doBaud=0,
	doDataBits=0,
	doParity=0,
	doStopBits=0;
    char buffer[128];
    int port;

    port = serialState->portID;
    /*
     * Determine which options we need to do.  Do all of them
     * if optionName is NULL.
     */

    if (optionName == (char *) NULL || optionName[0] == '\0') {
        doAll = true;
    } else {
	if (!strcmp(optionName, "-baud")) {
	    doBaud = true;
	} else if (!strcmp(optionName, "-databits")) {
	    doDataBits = true;
	} else if (!strcmp(optionName, "-parity")) {
	    doParity = true;
	} else if (!strcmp(optionName, "-stopbits")) {
	    doStopBits = true;
	} else {
	    return Tcl_BadChannelOption(interp, optionName,
		    "baud databits parity stopbits");
	}
    }


    /*
     * Get the sockname for the socket.
     */

    if (doAll || doBaud) {
    	int baud;
	if (doAll) {
	    Tcl_DStringAppendElement(dsPtr, "-baud");
	}
	switch (serialPortList[port].serConfig & 0x1FF) {
	    case baud150: baud = 150; break;
	    case baud300: baud = 300; break;
	    case baud600: baud = 600; break;
	    case baud1200: baud = 1200; break;
	    case baud1800: baud = 1800; break;
	    case baud2400: baud = 2400; break;
	    case baud3600: baud = 3600; break;
	    case baud4800: baud = 4800; break;
	    case baud7200: baud = 7200; break;
	    case baud9600: baud = 9600; break;
	    case baud14400: baud = 14400; break;
	    case baud19200: baud = 19200; break;
	    case baud28800: baud = 28800; break;
	    case baud38400: baud = 38400; break;
	    case baud57600: baud = 57600; break;
	}
	sprintf(buffer, "%d", baud);
	Tcl_DStringAppendElement(dsPtr, buffer);
    }
    if (doAll || doDataBits) {
	if (doAll) {
	    Tcl_DStringAppendElement(dsPtr, "-databits");
	}
	switch (serialPortList[port].serConfig & 0xC00) {
	    case data5:		sprintf(buffer, "5"); break;
	    case data6: 	sprintf(buffer, "6"); break;
	    case data7: 	sprintf(buffer, "7"); break;
	    case data8: 	sprintf(buffer, "8"); break;
	}
	Tcl_DStringAppendElement(dsPtr, buffer);
    }
    if (doAll || doParity) {
	if (doAll) {
	    Tcl_DStringAppendElement(dsPtr, "-parity");
	}
	switch (serialPortList[port].serConfig & 0x3000) {
	    case noParity:		sprintf(buffer, "none"); break;
	    case oddParity: 	sprintf(buffer, "odd"); break;
	    case evenParity: 	sprintf(buffer, "even"); break;
	}
	Tcl_DStringAppendElement(dsPtr, buffer);
    }
    if (doAll || doStopBits) {
	if (doAll) {
	    Tcl_DStringAppendElement(dsPtr, "-stopbits");
	}
	switch (serialPortList[port].serConfig & 0xC000) {
	    case 0x4000:	sprintf(buffer, "1"); break;
	    case 0x8000: 	sprintf(buffer, "1.5"); break;
	    case 0xC000: 	sprintf(buffer, "2"); break;
	}
	Tcl_DStringAppendElement(dsPtr, buffer);
    }
    return TCL_OK;
}

/*
 * The following routine from from Apple DTS.
 *
 * It opens both the input and output serial drivers, and returns their
 * refNums.  Both refNums come back as an illegal value (0) if we
 * can't open either of the drivers.
 *
 */

/*
 * The one true way of opening a serial driver.  This routine
 * tests whether a serial port arbitrator exists.  If it does,
 * it relies on the SPA to do the right thing when OpenDriver is called.
 * If not, it uses the old mechanism, which is to walk the unit table
 * to see whether the driver is already in use by another program.
 */

static OSErr OpenOneSerialDriver(ConstStr255Param driverName, short *refNum)
{
    OSErr err;

    if ( SerialArbitrationExists() ) {
        err = OpenDriver(driverName, refNum);
    } else {
        if ( DriverIsOpen(driverName) ) {
            err = portInUse;
        } else {
            err = OpenDriver(driverName, refNum);
        }
    }
    return err;
}

static OSErr OpenSerialDrivers(ConstStr255Param inName, ConstStr255Param
	outName,
	short *inRefNum, short *outRefNum)
{
    OSErr err;

    err = OpenOneSerialDriver(outName, outRefNum);
    if (err == noErr) {
        err = OpenOneSerialDriver(inName, inRefNum);
        if (err != noErr) {
            (void) CloseDriver(*outRefNum);
        }
    }
    if (err != noErr) {
        *inRefNum = 0;
        *outRefNum = 0;
    }
    return err;
}

static OSErr CloseSerialDrivers(SInt16 inRefNum, SInt16 outRefNum)
{
    OSErr err;

    (void) KillIO(inRefNum);
    err = CloseDriver(inRefNum);
    if (err == noErr) {
        (void) KillIO(outRefNum);
        (void) CloseDriver(outRefNum);
    }

    return err;
}

/*
 * Test Gestalt to see if serial arbitration exists
 * on this machine.
 */

static Boolean SerialArbitrationExists(void)
{
    Boolean result;
    long    response;

    result = ((Gestalt(gestaltSerialPortArbitratorAttr, &response) ==
	    noErr) &&
	    (response & (1 << gestaltSerialPortArbitratorExists) != 0));
    return result;
}

/*
 * Walks the unit table to determine whether the
 * given driver is marked as open in the table.
 * Returns false if the driver is closed, or does
 * not exist.
 */

static Boolean DriverIsOpen(ConstStr255Param driverName)
{
    Boolean     found;
    Boolean     isOpen;
    short       unit;
    DCtlHandle  dceHandle;
    StringPtr   namePtr;

    found = false;
    isOpen = false;

    unit = 0;
    while ( ! found && ( unit < LMGetUnitTableEntryCount() ) ) {

        /*
         *  Get handle to a device control entry.  GetDCtlEntry
         * takes a driver refNum, but we can convert between
         * a unit number and a driver refNum using bitwise not.
         */

        dceHandle = GetDCtlEntry( ~unit );

        if ( dceHandle != nil && (**dceHandle).dCtlDriver != nil ) {

            /*
             * If the driver is RAM based, dCtlDriver is a handle,
             * otherwise it's a pointer.  We have to do some fancy
             * casting to handle each case.  This would be so much
             * easier to read in Pascal )-:
             */

            if ( ((**dceHandle).dCtlFlags & dRAMBasedMask) != 0 ) {
                namePtr = & (**((DRVRHeaderHandle)
			(**dceHandle).dCtlDriver)).drvrName[0];
            } else {
                namePtr = & (*((DRVRHeaderPtr)
			(**dceHandle).dCtlDriver)).drvrName[0];
            }

            /*
             * Now that we have a pointer to the driver name, compare
             * it to the name we're looking for.  If we find it,
             * then we can test the flags to see whether it's open or
             * not.
             */

            if ( EqualString(driverName, namePtr, false, true) ) {
                found = true;
                isOpen = ((**dceHandle).dCtlFlags & dOpenedMask) != 0;
            }
        }
        unit += 1;
    }

    return isOpen;
}

/*
 *----------------------------------------------------------------------
 *
 * TclMacDevErrorToPosixError --
 *
 *	Maps macintosh device errors into their proper Posix-style errors
 *
 * Results:
 *
 *	A standard POSIX error code
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
 
static int TclMacDevErrorToPosixError(
	OSErr error
) {
    switch(error) {
    	case noErr:
	    return 0;
	case paramErr:		/*error in user parameter list*/
	    return EINVAL;
	case userCanceledErr:
	    return EINTR;
	case qErr:		/*queue element not found during deletion*/
	case SlpTypeErr:	/*invalid queue element*/
	case vTypErr:		/*invalid queue element*/
	    return ESRCH;	
	case corErr:		/*core routine number out of range*/
	    return EDOM;
	case unimpErr:		/*unimplemented core routine*/
	case seNoDB:		/*no debugger installed to handle debugger command*/
	case dInstErr:		/*DrvrInstall couldn't find driver in resources*/
	    return ENOSYS;
	case badUnitErr:	/*I/O System Errors*/
	case unitEmptyErr:	/*I/O System Errors*/	
	    return ENODEV;
	case openErr:		/*I/O System Errors*/
	case closErr:		/*I/O System Errors*/
	    return EBUSY;
	case dRemovErr:		/*tried to remove an open driver*/
	    return EDEADLK;
	case controlErr:	/*I/O System Errors*/
	case statusErr:		/*I/O System Errors*/
	case readErr:		/*I/O System Errors*/
	case writErr:		/*I/O System Errors*/
	    return EIO;
	default:
	    return EIO;
    }
}
