/*
 * odieDevices.c
 *
 *	Channel drivers for Macintosh channels for the
 *	console fds.
 *
 *  Questions, Bugs reports, and Creative direction can for now be sent to
 *  Sean Woods, yoda@drexel.edu
 * Copyright (c) 1998 Woods Design Services
 *
 * See the file "odie.license" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */


#include "tcl.h"
#include "odieDevices.h"

/*
 * This is the init routine for this file.
 */

int Device_Init _ANSI_ARGS_ ((Tcl_Interp *interp));
int Device_SafeInit _ANSI_ARGS_ ((Tcl_Interp *interp));
#pragma export reset
/*
 * Static routines for this file:
 */

static void  DeviceCmdDeletionProc _ANSI_ARGS_((ClientData clientData)); 
static void  DeviceChannelExitHandler _ANSI_ARGS_((ClientData
	clientData));
static void  DeviceCheckProc _ANSI_ARGS_((ClientData clientData,
	int flags));
static int   DeviceEventProc _ANSI_ARGS_((Tcl_Event *evPtr,
	int flags));
static void  DeviceSetupProc _ANSI_ARGS_((ClientData clientData,
	int flags));
static int   DevicePort(char *devName);
static void	 DeviceCheckForDead(void);

/*
 * The following variable is used to tell whether this module has been
 * initialized.
 */

static int initialized = 0;

/*
 * The following pointer refers to the head of the list of ports managed
 * that are being watched for file events.
 */

static PortState *firstDevPtr;

static PortDesc portList[] = {
#ifdef MAC_TCL
    {"modem",	"serial",	"Modem Port",
     0,SerialOpen,SerialReset},
    {"printer",	"serial",	"Printer Port",
     1,SerialOpen,SerialReset},
    {"com1",	"serial",	"RS-232 Port 1",
     2,SerialOpen,SerialReset},
    {"com2",	"serial",	"RS-232 Port 2",
     3,SerialOpen,SerialReset},
    {"pbmodem", "serial",	"Powerbook Internal Modem Port",
     4,SerialOpen,SerialReset},
#endif
    {"serial.a","serial",	"Serial Port 1",
     0,SerialOpen,SerialReset},
    {"serial.b","serial",	"Serial Port 2",
     1,SerialOpen,SerialReset},
    {"serial.c","serial",	"Serial Port 3",
     2,SerialOpen,SerialReset},
    {"serial.d","serial",	"Serial Port 4",
     3,SerialOpen,SerialReset},
    {NULL,NULL,NULL,NULL,NULL}
};


/*
 *----------------------------------------------------------------------
 * Device_Init --
 *
 *      Initializes the devices package
 *
 * Results: 
 *      The device command is registered.
 *
 * Side Effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

int 
Device_Init(interp)
    Tcl_Interp *interp;
{     
    Tcl_CreateObjCommand(interp, "device", Tcl_DeviceObjCmd, 
    	NULL, DeviceCmdDeletionProc);
    
    /*
     * Initialize Notifier Table
     */
    firstDevPtr = (PortState *) ckalloc(sizeof(PortState));
    firstDevPtr->nextPtr = NULL;
    
	Tcl_CreateEventSource(DeviceSetupProc, DeviceCheckProc, NULL);    	
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * Device_SafeInit --
 *
 *      Initializes the devices package
 *
 * Results: 
 *      Nothing happens, since this is not a safe activity.
 *
 * Side Effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

int 
Device_SafeInit(interp)
    Tcl_Interp *interp;
{
    
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * Tcl_DeviceObjCmd --
 *
 *      This is the implementation of the device command.
 *
 * Results: 
 *      See the documentation for details.
 *
 * Side Effects:
 *      The device command is added to the interpreter.
 *
 *----------------------------------------------------------------------
 */

int 
Tcl_DeviceObjCmd(dummy, interp, objc, objv)
    ClientData dummy;		/* Not used. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int objc;			/* Number of arguments. */
    Tcl_Obj *CONST objv[];	/* Argument objects. */
{
    int index, err;
    int i=0,len;
    char *devName;
    Tcl_Channel chan;
    static char *sCmds[] = {
	"open",
	"names",
	"reset",
	"type",
	"desc",
	(char *) NULL
    };
    enum {
	DevOpen,
	DevNames,
	DevReset,
	DevType,
	DevDesc
    };

    err = TCL_OK;
    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "command ?arg?");

	return TCL_ERROR;
    }
    
    if (Tcl_GetIndexFromObj(interp, objv[1], sCmds, "command", 0,
	    (int *) &index) != TCL_OK) {
	return TCL_ERROR;
    }

    switch(index) {
	case DevOpen:
	    devName = Tcl_GetStringFromObj(objv[2], &len);
	    chan = Tcl_OpenDeviceChannel(interp, devName);
	    if (chan == (Tcl_Channel) NULL) {
		return TCL_ERROR;
	    }
	    Tcl_RegisterChannel(interp, chan);
	    Tcl_AppendResult(interp, Tcl_GetChannelName(chan),
		    (char *) NULL);
	    return TCL_OK;
	case DevReset:
	    devName = Tcl_GetStringFromObj(objv[2], &len);
	    if(Tcl_ResetDeviceChannel(interp, devName) != TCL_OK) {
			return TCL_ERROR;
	    }
	    return TCL_OK;
	case DevNames:
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), NULL, 0);
	    
	    while((portList[i].name) != NULL) {
		
		Tcl_AppendToObj(Tcl_GetObjResult(interp),
			(portList[i].name), -1);
		Tcl_AppendToObj(Tcl_GetObjResult(interp),
			" ", -1);
		i++;
	    }
	    return TCL_OK;
	    
	case DevType:
	case DevDesc:
	    devName = Tcl_GetStringFromObj(objv[2],&len);
	    i = DevicePort(devName);
	    
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), NULL, 0);
	    	    
	    if (i < 0) {
		    Tcl_AppendResult(interp, "Unknown port: \"", devName, "\"",
			    (char *) NULL);
		    return TCL_ERROR;	    
	    }
	    
	    switch(index) {    	    
   	        case DevType:
			    Tcl_AppendToObj(Tcl_GetObjResult(interp),
				    (portList[i].type), -1);
			    return TCL_OK;    	        
	        case DevDesc:
			    Tcl_AppendToObj(Tcl_GetObjResult(interp),
				    (portList[i].desc), -1);
			    return TCL_OK;
		}
    }
    return err;
}

/*
 *----------------------------------------------------------------------
 * DeviceCmdDeletionProc --
 *
 *      Closes all the open devices when the device command is deleted.
 *
 * Results: 
 *      All the ports are closed.
 *
 * Side Effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
DeviceCmdDeletionProc()
{
    Tcl_DeleteEventSource(DeviceSetupProc, DeviceCheckProc, NULL);
}

/*
 *----------------------------------------------------------------------
 * DevicePort --
 *
 *      Return the port ID based on a text representation
 *
 * Results: 
 *      The port ID, or -1 if error.
 *
 * Side Effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static int DevicePort(char *devName) {
    int i;
    for(i=0; portList[i].name != NULL; i++) {
	if( strcmp(devName,portList[i].name) == 0 ) {
	    return portList[i].portid;
	}
    }
    return -1;
}

/*
 *----------------------------------------------------------------------
 * DeviceInsert --
 *
 *      Head insert a device into the notification list
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      An entry is added to the device list.
 *
 *----------------------------------------------------------------------
 */
void DeviceInsert(PortState *deviceState) {
	/*
	 * Remove from list if already there
	 */

	DeviceRemove(deviceState);
	
    /*
     * Head Insert Into Notification List
     */ 

    deviceState->nextPtr = firstDevPtr->nextPtr;
    firstDevPtr->nextPtr = deviceState;
}

/*
 *----------------------------------------------------------------------
 * DeviceRemove --
 *
 *      Removes a device from the notification list
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      A device is removed from the internal device list.
 *----------------------------------------------------------------------
 */
void DeviceRemove(PortState *fileState) {
    PortState *temp;
	
	/*
	 * Body Remove
	 */
    temp = firstDevPtr;
    while(temp != NULL) {
		if(temp->nextPtr == fileState) {
			temp->nextPtr = temp->nextPtr->nextPtr;
		}
		temp = temp->nextPtr;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenDeviceChannel --
 *
 *	Open an hardware based channel on Mac systems.
 *
 * Results:
 *	The new channel or NULL. If NULL, the output argument
 *	errorCodePtr is set to a POSIX error.
 *
 * Side effects:
 *	May open the channel
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenDeviceChannel(
    Tcl_Interp *interp,		/* Interpreter for error reporting;
                                 * can be NULL. */
    char *devName		/* Name of port to open. */
    )
{
    Tcl_Channel chan;
    int errorCode;
    int port=-1;

    port = DevicePort(devName);

    if( port >= 0) {
	chan = (portList[port].openProc)(interp,
		portList[port].portid,
		&errorCode);
    } else {
	if (interp != (Tcl_Interp *) NULL) {
	    Tcl_AppendResult(interp, "unknown port \"",
		    devName, "\": ",
		    (char *) NULL);
	}
	return NULL;
    }

    if (chan == NULL) {
	Tcl_SetErrno(errorCode);
	if (interp != (Tcl_Interp *) NULL) {
	    Tcl_AppendResult(interp, "Couldn't open \"",
		    devName, "\": ",
		    Tcl_PosixError(interp), (char *) NULL);
	}
	return NULL;
    }
    return chan;
}

/* ----------------------------------------------------------------------
 *
 * Tcl_ResetDeviceChannel --
 *
 *	Forces closed a hardware based channel on Mac systems.
 *
 * Results:
 *	TCL_ERROR if device not found
 *
 * Side effects:
 *	Forces closed a device
 *
 *----------------------------------------------------------------------
 */

int
Tcl_ResetDeviceChannel(
    Tcl_Interp *interp,		/* Interpreter for error reporting;
                                 * can be NULL. */
    char *devName		/* Name of port to open. */
    )
{
    int port=-1;

    port = DevicePort(devName);

    if( port >= 0) {
	if ((portList[port].closeProc) != NULL) {
	    (portList[port].closeProc)(portList[port].portid);
	}
	return TCL_OK;
    } else {
	if (interp != (Tcl_Interp *) NULL) {
	    Tcl_AppendResult(interp, "unknown port \"",
		    devName, "\": ",
		    (char *) NULL);
	}
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DeviceCheckForDead --
 *
 *	This procedure is invoked before we scan the device event table
 *		it prevents core dumps in cases where a device is closed
 *		in the midst of processing an event
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Adjusts the block time if needed.
 *
 *----------------------------------------------------------------------
 */
 void DeviceCheckForDead(void) {
     PortState *infoPtr;
     infoPtr = firstDevPtr;
     
     while( infoPtr != NULL ) {
	    if (infoPtr->portID < 0) {
	  		DeviceRemove(infoPtr);
		} else {
			infoPtr = infoPtr->nextPtr;	
		}
	 }    
 }
 
/*
 *----------------------------------------------------------------------
 *
 * DeviceSetupProc --
 *
 *	This procedure is invoked before Tcl_DoOneEvent blocks waiting
 *	for an event.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Adjusts the block time if needed.
 *
 *----------------------------------------------------------------------
 */

void
DeviceSetupProc(
    ClientData data,		/* Not used. */
    int flags)			/* Event flags as passed to Tcl_DoOneEvent. */
{
    PortState *infoPtr;
    Tcl_Time blockTime = { 0, 0 };

    if (!(flags & TCL_FILE_EVENTS)) {
		return;
	}

    /*
     * Check to see if there is a ready file.  If so, poll.
     */

    for (infoPtr = firstDevPtr->nextPtr; infoPtr != NULL;
	 infoPtr = infoPtr->nextPtr) {
    	if((infoPtr->readyProc)(infoPtr)) {
	    Tcl_SetMaxBlockTime(&blockTime);
	    break;
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DeviceCheckProc --
 *
 *	This procedure is called by Tcl_DoOneEvent to check the serial
 *	event source for events.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May queue an event.
 *
 *----------------------------------------------------------------------
 */

static void
DeviceCheckProc(
    ClientData data,		/* Not used. */
    int flags)			/* Event flags as passed to Tcl_DoOneEvent. */
{
    PortEvent *evPtr;
    PortState *infoPtr;
    int sentMsg = 0;
    Tcl_Time blockTime = { 0, 0 };

	/*
	 * Abort if not processing file events, or a mutual exclusion
	 * condition exists
	 */
    if (!(flags & TCL_FILE_EVENTS)) {
	return;
    }

    /*
     * Queue events for any ready files that don't already have events
     * queued (caused by persistent states that won't generate WinSock
     * events).
     */

    for (infoPtr = firstDevPtr->nextPtr; infoPtr != NULL;
            infoPtr = infoPtr->nextPtr) {
		if ((infoPtr->watchMask & TCL_READABLE)
			&& (infoPtr->readyProc)(infoPtr)
			&& !infoPtr->pending) 
		{
		    infoPtr->pending = 1;
		    evPtr = (PortEvent *) ckalloc(sizeof(PortEvent));
		    evPtr->header.proc = DeviceEventProc;
	   		evPtr->infoPtr = infoPtr;
	    	Tcl_QueueEvent((Tcl_Event *) evPtr, TCL_QUEUE_TAIL);
		}
    }
}

/*----------------------------------------------------------------------
 *
 * DeviceEventProc --
 *
 *	This function is invoked by Tcl_ServiceEvent when a serial event
 *	reaches the front of the event queue.  This procedure invokes
 *	Tcl_NotifyChannel on the port.
 *
 * Results:
 *	Returns 1 if the event was handled, meaning it should be removed
 *	from the queue.  Returns 0 if the event was not handled, meaning
 *	it should stay on the queue.  The only time the event isn't
 *	handled is if the TCL_FILE_EVENTS flag bit isn't set.
 *
 * Side effects:
 *	Whatever the notifier callback does.
 *
 *----------------------------------------------------------------------
 */

static int
DeviceEventProc(
    Tcl_Event *evPtr,		/* Event to service. */
    int flags)			/* Flags that indicate what events to
				 * handle, such as TCL_FILE_EVENTS. */
{
    PortEvent *portEvPtr = (PortEvent *)evPtr;
    PortState *infoPtr;

	/*
	 * Abort if not processing file events, or a mutual exclusion
	 * condition exists
	 */
    if (!(flags & TCL_FILE_EVENTS)) {
	return 0;
    }
	
    /*
     * Search through the list of watched ports for the one whose handle
     * matches the event.  We do this rather than simply dereferencing
     * the handle in the event so that port can be deleted while the
     * event is in the queue.
     */
    
    for (infoPtr = firstDevPtr->nextPtr; infoPtr != NULL;
            infoPtr = infoPtr->nextPtr) 
    {
		if (portEvPtr->infoPtr == infoPtr) {
		    infoPtr->pending = 0;
	   		Tcl_NotifyChannel(infoPtr->devChan, infoPtr->watchMask);
	    	break;
		}
    }
    return 1;
}
