# prox.tcl --
#
#     Tools to help analyze TclX's profile data.
#
#  (The name comes from "PROfile tools for tclX" or a pun on "procs",
#  you decide.)
#
# Copyright 1998 Pinebush Technologies Inc.
#
# All rights reserved.  This material contains proprietary trade
# secrets.  Use of a copyright notice is precautionary only and does
# not imply publication or disclosure.
#
#
# Revision info: %W% %E%
#
# Author:        Chris Nelson (chris@pinebush.com)
#
# Exported procs:
#     prox::dumpArray - Dump TclX profile array in a format readable
#                       by prox::loadArray
#     prox::loadArray - Load a TclX profile array as written by prox::dumpArray
#     prox::buildTree - Convert TclX profile array to a tree to for
#                       prox::explore
#     prox::explore   - Interactive, text-based exploration of a profile tree
#
# Private procs:
#     prox::CmpStacks - Compare call stacks for buildTree
#     prox::SumTimes  - Sum execution time of all children for buildTree
#     prox::NodeTime  - Extract time from a node for SumTimes
#     prox::CmpTimes  - Sort execution time of children for SumTimes
#     prox::ListNode  - Display one node of tree for explore
#
# Testing procs:
#     prox::InitTest  - Initialize a profile array as TclX would
#     prox::Test      - Run a test
#
# Example:
#
#
# Global data:
#     None.
#
#-----------------------------------------------------------------------
# To Do:
#    It would be nice to have Tk-based explorer which displayed pie
#    charts or histograms and drilled down a level by clicking on the
#    segment or pie slice of interest.
#-----------------------------------------------------------------------

package provide prox

namespace eval ::prox {
    namespace export \
            dumpArray \
            loadArray \
            buildTree \
            explore

}
#
#=======================================================================
# Public procs
#=======================================================================
# prox::dumpArray --
#
#     Dump an array from TclX's [profile off] so it can be reloaded by 
#     prox::loadArray.
# 
# Arguments:
#     channel   - ID of an open channel to write to
#     arrayName - Name of the array to process
#
# Results:
#     
proc prox::dumpArray { channel arrayName } {
    upvar $arrayName profArray
    
    puts $channel "{"
    foreach e [array names profArray] {
        puts $channel "    {$e} {$profArray($e)}"
    }
    puts $channel "}"
}
# prox::dumpArray
#
#-----------------------------------------------------------------------
# prox::loadArray --
#
#     Read an array from TclX's [profile off] as written by 
#     prox::dumpArray.
# 
# Arguments:
#     channel   - ID of an open channel to read from
#     arrayName - Name of the array to process
#     
# Results:
#     The named array is initialized and it's name returned.  This allows:
#
#         prox::buildTree [prox::loadArray $fid fooArray] fooTree
#
proc prox::loadArray { channel arrayName } {
    upvar $arrayName profArray
    set values [gets $channel]
    while { ! [eof $channel] } {
        append values [gets $channel]
    }
    eval array set profArray $values
    return $arrayName
}
# prox::loadArray
#
#-----------------------------------------------------------------------
# prox::buildTree --
#
#     Turn array from TclX into a tree for prox::explore
# 
# Arguments:
#     arrayName - Name of array from TclX's [profile off arrayName]
#     treeName  - Name of tree to build from array contents
#
# Results:
#     The array named by $treeName is filled in and ready for [explore].
#     The name of the tree is returned so you can do:
#
#         prox::explore [prox::buildTree fooArray fooTree]
#
# Notes:
#     The profile tree is really a pseudo-2d array.  The first part of
#     each index is a stack frame (like the indices in TclX profile
#     array).  The second part is "children" to access a list of that
#     node's children, or "times" to access a list of the times for
#     the sub-trees rooted at the children.  The list of children is
#     built directly by buildTree.  The times are computed by
#     prox::SumTimes.
#
#     The list of children is a RELATIVE list, not a full path.  While
#     traversing the tree, the full path must be built in a stack-like
#     manner to index the profile array.
#
#     The tree is rooted at profTree(root) so the first generation is
#     listed in $profTree($profTree(root),children).  
#
proc prox::buildTree { arrayName treeName {what real}} {
    upvar $arrayName profArray
    upvar $treeName profTree

    # Build a tree to walk summing times and displaying results
    foreach s [lsort -command prox::CmpStacks [array names profArray]] {
	set profTree($s,children) {}
	if {[llength $s] == 1} {
	    set profTree(root) $s
	} else {
	    set parent [lrange $s 1 end]
	    set child  [lindex $s 0]
	    lappend profTree($parent,children) $child
	}
    }

    # Sum the times, starting at root
    SumTimes $what $arrayName $treeName $profTree(root)

    return $treeName
}
# prox::buildTree
#
#-----------------------------------------------------------------------
# prox::explore --
#
#     Interactive exploration of call stack.
# 
# Arguments:
#     treeName - Name of profile tree array
#     node     - (OPTIONAL) Node to explore (Defaults to root of tree)
#
# Results:
#     Returns 0 to continue, 1 to abort.
#
#     explore is recursive.  Returning 1 pops all the recursive levels.
#
proc prox::explore { treeName {node ROOT}} {
    upvar $treeName profTree

    # Handle default
    if {$node == "ROOT"} {
        set node $profTree(root)
    }
    
    # Loop until the user asks to quit
    while { 1 } {
        set max [expr [ListNode $treeName $node] -1]
        if {$max > 0} {
            set defaultInput 0
            puts -nonewline "0-$max, "
        } else {
            set defaultInput u
        }
        puts -nonewline "q, u, ? \[$defaultInput\]: "
        flush stdout
        set input [gets stdin]
        puts ""
        if {$input == ""} {
            set input $defaultInput
        }

        switch -- [string tolower $input] {
            q {
                # Quit
                return 1
            }
            u {
                # Up a level
                return 0
            }

            ? {
                # Help
                if {$max > 0} {
                puts "0-$max\tpick a child to explore.\
                        (Default to child that used the most time.)"
                }
                puts "q\tQuit"
                puts "u\tUp to previous level"
                puts "?\tThis help"
                puts ""
            }

            default {
                # Hopefully a level number
                set times $profTree($node,times)
                if {[catch {lindex $times $input} time]} {
                    puts "\n\"$input\" invalid!\n"
                } else {
                    set child [lindex $time 0]
                    if {$child == "."} {
                        puts "\nNothing there!\n"
                    } elseif {[explore $treeName "$child $node"]} {
                        return 1
                    }
                }
            }
        }
    }
    # while
}
# prox::explore
#
#=======================================================================
# Private procs only below this line
#=======================================================================
# prox::CmpStacks --
#
#     Compare call stacks.  Shorter is earlier.
#
#     This is an [lsort] command proc.
#
# Arguments:
#     a, b  - A list representing a call stack, "current" proc last
#
# Results:
#     -1,0,1 as appropriate.
# 
proc prox::CmpStacks { a b } {
    if { [llength $a] < [llength $b] } {
        return -1
    } elseif { [llength $a] > [llength $b] } {
        return 1
    } else {
        return 0
    }
}
# prox::CmpStacks
#
#-----------------------------------------------------------------------
# prox::SumTimes --
#
#     Recursive proc to sum times for a node.
# 
#     Does a depth-first-search to compute times of higher level nodes
#     as the total of the times of their children.
# 
# Arguments:
#     what      - "real" or "cpu"
#     arrayName - Name of array from TclX's [profile off]
#     treeName  - Name of tree being built
#     node      - Node in the tree to be processed
#
# Results:
#     
# Note:
#     Each node has an entry in the tree which records how it's time
#     it divided between it's children and overhead.
#
#     The index of the entry is "$node,times", where "$node" is a full
#     path to the node.
#
#     The entry is a list of lists.  Each sub-list is two elements:
#
#         {childname childtime}
#
#     The special childname "." is used to record the time used by the
#     node outside calls to it's children.
#
#     For example, if a is called from the global scope and it calls b
#     (among others) which calls c and d, $tree(b a,times) might
#     contain:
#
#         {{. 12} {c 33} {d 5}}
#
#     From this we can see that b took a total of 50 milliseconds and
#     that c dominated b's running time.
#
proc prox::SumTimes { what arrayName treeName node } {
    upvar $arrayName profArray
    upvar $treeName profTree

    set myTime [NodeTime $arrayName $node $what]
    set profTree($node,times) [list [list "." $myTime]]
    foreach c $profTree($node,children) {
        set childTime [SumTimes $what $arrayName $treeName "$c $node"]
        lappend profTree($node,times) [list $c $childTime]
        incr myTime $childTime
    }
    set profTree($node,times) \
            [lsort -command prox::CmpTimes $profTree($node,times)]
    return $myTime
}
# prox::SumTimes
#
#-----------------------------------------------------------------------
# prox::NodeTime --
#
#     Extract time for a node from profArray
# 
# Arguments:
#     arrayName - Name of array from TclX's [profile off]
#     node      - index to profile array
#     what      - "real" or "cpu"
#
# Results:
#     Returns real or CPU time based on value of $what
#
proc prox::NodeTime { arrayName node what} {
    upvar $arrayName profArray
    switch -- $what {
        cpu {
            return [lindex $profArray($node) 1]
        }
        real {
            return [lindex $profArray($node) 2]
        }
        default {
            error "$what is not a value time index"
        }
    }
}
# prox::NodeTime
#
#-----------------------------------------------------------------------
# prox::CmpTimes --
#
#     Compare times.  Slower is earlier in the list
# 
# Arguments:
#     
# Results:
#     
proc prox::CmpTimes { a b } {
    set t1 [lindex $a 1]
    set t2 [lindex $b 1]
    if {$t1 < $t2} {
        return 1
    } elseif {$t1 == $t2} {
        return 0 
    } else {
        return -1
    }
}
# prox::CmpTimes
#
#-----------------------------------------------------------------------
# prox::ListNode --
#
#     Helper for explore.  Show up to 10 children of a node with their
#     times and percentages.
# 
# Arguments:
#     treeName - Name of tree from [buildTree]
#     node     - Node to list children of
#
# Results:
#     
proc prox::ListNode { treeName node } {
    upvar $treeName profTree

    # Total times
    set total 0
    set times $profTree($node,times)
    foreach time $times {
        incr total [lindex $time 1]
    }
    # Put percentages in list
    foreach time $times { 
        set newTime [lappend time [expr 100.0 * [lindex $time 1] / $total]]
        lappend newTimes $newTime
    }
    set times $newTimes

    # Display list of children
    puts "Call stack: $node"
    puts "Total time: $total ms"
    set i 0
    foreach time $times {
        puts "  $i \
                [format "%5dms (%05.2f%%)" [lindex $time 1] [lindex $time 2]] \
                [lindex $time 0]"
        incr i
        if {$i > 9} {
            puts "...others..."
            break
        }
    }

    # Tell caller how many we displayed
    return $i
}
# prox::ListNode
#
#=======================================================================
# Testing procs only below this line (defined only when script is
# invoked from the command line).
#=======================================================================
if {[string match [info script] $argv0]} {
    # prox::InitTest --
    #
    #     Initialize a small array like what TclX's [profile off] does
    # 
    # Arguments:
    #     
    # Results:
    #     
    proc prox::InitTest { arrayName } {
        upvar $arrayName profArray

        # Build a call stack like:
        # 1 
        #   A
        #     i
        #     ii
        #     ...
        #   B
        #     ii
        #     iii
        #   C
        #     ii
        #     iv
        #
        #             Call stack        {calls real cpu}
        set profArray([list xii  A 1])  {0 11 11}
        set profArray([list xii  A 1])  {0 21 20}
        set profArray([list xi  A 1])   {0 31 30}
        set profArray([list x  A 1])    {0 41 40}
        set profArray([list ix  A 1])   {0 51 50}
        set profArray([list viii  A 1]) {0 62 60}
        set profArray([list vii  A 1])  {0 52 50}
        set profArray([list vi  A 1])   {0 53 50}
        set profArray([list v   A 1])   {0 54 50}
        set profArray([list ii  A 1])   {0 55 50}
        set profArray([list i   A 1])   {0 76 70}
        set profArray([list     A 1])   {0 37 30}

        set profArray([list iii B 1])   {0 22 21}
        set profArray([list ii  B 1])   {0 87 85}
        set profArray([list     B 1])   {0 43 40}

        set profArray([list iv  C 1])   {0 13 13}
        set profArray([list ii  C 1])   {0 22 22}
        set profArray([list     C 1])   {0 31 31}

        set profArray([list 1])         {0 59 59}
    }
    # prox::InitTest
    #
    #-----------------------------------------------------------------------
    # prox::Test --
    #
    #     Test buildTree and explore
    # 
    # Arguments:
    #     
    # Results:
    #     
    proc prox::Test { } {
        puts -nonewline "Initializing profile array..."
        flush stdout
        prox::InitTest profArray
        puts "done."

        puts -nonewline "Building tree..."
        flush stdout
        prox::buildTree profArray profTree
        puts "done."

        prox::explore profTree
    }
    # prox::Test
    #
    #-----------------------------------------------------------------------
    # Try it out
    prox::Test
}
