/*
 * tcl.h --
 *
 *	This header file describes the externally-visible facilities
 *	of the Tcl interpreter.
 *
 * Copyright (c) 1987-1994 The Regents of the University of California.
 * Copyright (c) 1994-1997 Sun Microsystems, Inc.
 * Copyright (c) 1993-1996 Lucent Technologies.
 * Copyright (c) 1998-1999 Scriptics Corporation.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tcl.h,v 1.33 1999/02/03 02:58:25 stanton Exp $
 */

#ifndef _TCL
#define _TCL

/*
 * When version numbers change here, must also go into the following files
 * and update the version numbers:
 *
 * README
 * library/init.tcl	(only if major.minor changes, not patchlevel)
 * unix/configure.in
 * win/makefile.bc	(only if major.minor changes, not patchlevel)
 * win/makefile.vc	(only if major.minor changes, not patchlevel)
 * win/README
 * win/README.binary
 * mac/README
 *
 * The release level should be  0 for alpha, 1 for beta, and 2 for
 * final/patch.  The release serial value is the number that follows the
 * "a", "b", or "p" in the patch level; for example, if the patch level
 * is 7.6b2, TCL_RELEASE_SERIAL is 2.  It restarts at 1 whenever the
 * release level is changed, except for the final release which is 0
 * (the first patch will start at 1).
 */

#define TCL_MAJOR_VERSION   8
#define TCL_MINOR_VERSION   0
#define TCL_RELEASE_LEVEL   2
#define TCL_RELEASE_SERIAL  5

#define TCL_VERSION	    "8.0"
#define TCL_PATCH_LEVEL	    "8.0.5+"

/*
 * The following definitions set up the proper options for Windows
 * compilers.  We use this method because there is no autoconf equivalent.
 */

#ifndef __WIN32__
#   if defined(_WIN32) || defined(WIN32)
#	define __WIN32__
#   endif
#endif

#ifdef __WIN32__
#   ifndef STRICT
#	define STRICT
#   endif
#   ifndef USE_PROTOTYPE
#	define USE_PROTOTYPE 1
#   endif
#   ifndef HAS_STDARG
#	define HAS_STDARG 1
#   endif
#   ifndef USE_PROTOTYPE
#	define USE_PROTOTYPE 1
#   endif

/*
 * Under Windows we need to call Tcl_Alloc in all cases to avoid competing
 * C run-time library issues.
 */

#   ifndef USE_TCLALLOC
#	define USE_TCLALLOC 1
#   endif
#endif /* __WIN32__ */

/*
 * The following definitions set up the proper options for Macintosh
 * compilers.  We use this method because there is no autoconf equivalent.
 */

#ifdef MAC_TCL
#   ifndef HAS_STDARG
#	define HAS_STDARG 1
#   endif
#   ifndef USE_TCLALLOC
#	define USE_TCLALLOC 1
#   endif
#   ifndef NO_STRERROR
#	define NO_STRERROR 1
#   endif
#endif

/*
 * Utility macros: STRINGIFY takes an argument and wraps it in "" (double
 * quotation marks), JOIN joins two arguments.
 */

#define VERBATIM(x) x
#ifdef _MSC_VER
# define STRINGIFY(x) STRINGIFY1(x)
# define STRINGIFY1(x) #x
# define JOIN(a,b) JOIN1(a,b)
# define JOIN1(a,b) a##b
#else
# ifdef RESOURCE_INCLUDED
#  define STRINGIFY(x) STRINGIFY1(x)
#  define STRINGIFY1(x) #x
#  define JOIN(a,b) JOIN1(a,b)
#  define JOIN1(a,b) a##b
# else
#  ifdef __STDC__
#   define STRINGIFY(x) #x
#   define JOIN(a,b) a##b
#  else
#   define STRINGIFY(x) "x"
#   define JOIN(a,b) VERBATIM(a)VERBATIM(b)
#  endif
# endif
#endif

/* 
 * A special definition used to allow this header file to be included 
 * in resource files so that they can get obtain version information from
 * this file.  Resource compilers don't like all the C stuff, like typedefs
 * and procedure declarations, that occur below.
 */

#ifndef RESOURCE_INCLUDED

#ifndef BUFSIZ
#include <stdio.h>
#endif

/*
 * Definitions that allow Tcl functions with variable numbers of
 * arguments to be used with either varargs.h or stdarg.h.  TCL_VARARGS
 * is used in procedure prototypes.  TCL_VARARGS_DEF is used to declare
 * the arguments in a function definiton: it takes the type and name of
 * the first argument and supplies the appropriate argument declaration
 * string for use in the function definition.  TCL_VARARGS_START
 * initializes the va_list data structure and returns the first argument.
 */

#if defined(__STDC__) || defined(HAS_STDARG)
#   define TCL_VARARGS(type, name) (type name, ...)
#   define TCL_VARARGS_DEF(type, name) (type name, ...)
#   define TCL_VARARGS_START(type, name, list) (va_start(list, name), name)
#else
#   ifdef __cplusplus
#	define TCL_VARARGS(type, name) (type name, ...)
#	define TCL_VARARGS_DEF(type, name) (type va_alist, ...)
#   else
#	define TCL_VARARGS(type, name) ()
#	define TCL_VARARGS_DEF(type, name) (va_alist)
#   endif
#   define TCL_VARARGS_START(type, name, list) \
	(va_start(list), va_arg(list, type))
#endif

/*
 * Macros used to declare a function to be exported by a DLL.
 * Used by Windows, maps to no-op declarations on non-Windows systems.
 * The default build on windows is for a DLL, which causes the DLLIMPORT
 * and DLLEXPORT macros to be nonempty. To build a static library, the
 * macro STATIC_BUILD should be defined.
 * The support follows the convention that a macro called BUILD_xxxx, where
 * xxxx is the name of a library we are building, is set on the compile line
 * for sources that are to be placed in the library. See BUILD_tcl in this
 * file for an example of how the macro is to be used.
 */

#ifdef __WIN32__
# ifdef STATIC_BUILD
#  define DLLIMPORT
#  define DLLEXPORT
# else
#  if defined(_MSC_VER) || (defined(__GNUC__) && defined(__declspec))
#   define DLLIMPORT __declspec(dllimport)
#   define DLLEXPORT __declspec(dllexport)
#  else
#   define DLLIMPORT
#   define DLLEXPORT
#  endif
# endif
#else
# define DLLIMPORT
# define DLLEXPORT
#endif

#ifdef TCL_STORAGE_CLASS
# undef TCL_STORAGE_CLASS
#endif
#ifdef BUILD_tcl
# define TCL_STORAGE_CLASS DLLEXPORT
#else
# define TCL_STORAGE_CLASS DLLIMPORT
#endif

/*
 * Definitions that allow this header file to be used either with or
 * without ANSI C features like function prototypes.
 */

#undef _ANSI_ARGS_
#undef CONST

#if ((defined(__STDC__) || defined(SABER)) && !defined(NO_PROTOTYPE)) || defined(__cplusplus) || defined(USE_PROTOTYPE)
#   define _USING_PROTOTYPES_ 1
#   define _ANSI_ARGS_(x)	x
#   define CONST const
#else
#   define _ANSI_ARGS_(x)	()
#   define CONST
#endif

#ifdef __cplusplus
#   define EXTERN extern "C" TCL_STORAGE_CLASS
#else
#   define EXTERN extern TCL_STORAGE_CLASS
#endif

/*
 * Macro to use instead of "void" for arguments that must have
 * type "void *" in ANSI C;  maps them to type "char *" in
 * non-ANSI systems.
 */
#ifndef __WIN32__
#ifndef VOID
#   ifdef __STDC__
#       define VOID void
#   else
#       define VOID char
#   endif
#endif
#else /* __WIN32__ */
/*
 * The following code is copied from winnt.h
 */
#ifndef VOID
#define VOID void
typedef char CHAR;
typedef short SHORT;
typedef long LONG;
#endif
#endif /* __WIN32__ */

/*
 * Miscellaneous declarations.
 */

#ifndef NULL
#define NULL 0
#endif

#ifndef _CLIENTDATA
#   if defined(__STDC__) || defined(__cplusplus)
    typedef void *ClientData;
#   else
    typedef int *ClientData;
#   endif /* __STDC__ */
#define _CLIENTDATA
#endif

/*
 * Data structures defined opaquely in this module. The definitions below
 * just provide dummy types. A few fields are made visible in Tcl_Interp
 * structures, namely those used for returning a string result from
 * commands. Direct access to the result field is discouraged in Tcl 8.0.
 * The interpreter result is either an object or a string, and the two
 * values are kept consistent unless some C code sets interp->result
 * directly. Programmers should use either the procedure Tcl_GetObjResult()
 * or Tcl_GetStringResult() to read the interpreter's result. See the
 * SetResult man page for details.
 * 
 * Note: any change to the Tcl_Interp definition below must be mirrored
 * in the "real" definition in tclInt.h.
 *
 * Note: Tcl_ObjCmdProc procedures do not directly set result and freeProc.
 * Instead, they set a Tcl_Obj member in the "real" structure that can be
 * accessed with Tcl_GetObjResult() and Tcl_SetObjResult().
 */

typedef struct Tcl_Interp {
    char *result;		/* If the last command returned a string
				 * result, this points to it. */
    void (*freeProc) _ANSI_ARGS_((char *blockPtr));
				/* Zero means the string result is
				 * statically allocated. TCL_DYNAMIC means
				 * it was allocated with ckalloc and should
				 * be freed with ckfree. Other values give
				 * the address of procedure to invoke to
				 * free the result. Tcl_Eval must free it
				 * before executing next command. */
    int errorLine;              /* When TCL_ERROR is returned, this gives
                                 * the line number within the command where
                                 * the error occurred (1 if first line). */
} Tcl_Interp;

typedef struct Tcl_AsyncHandler_ *Tcl_AsyncHandler;
typedef struct Tcl_Channel_ *Tcl_Channel;
typedef struct Tcl_Command_ *Tcl_Command;
typedef struct Tcl_Event Tcl_Event;
typedef struct Tcl_Pid_ *Tcl_Pid;
typedef struct Tcl_RegExp_ *Tcl_RegExp;
typedef struct Tcl_TimerToken_ *Tcl_TimerToken;
typedef struct Tcl_Trace_ *Tcl_Trace;
typedef struct Tcl_Var_ *Tcl_Var;

/*
 * When a TCL command returns, the interpreter contains a result from the
 * command. Programmers are strongly encouraged to use one of the
 * procedures Tcl_GetObjResult() or Tcl_GetStringResult() to read the
 * interpreter's result. See the SetResult man page for details. Besides
 * this result, the command procedure returns an integer code, which is 
 * one of the following:
 *
 * TCL_OK		Command completed normally; the interpreter's
 *			result contains	the command's result.
 * TCL_ERROR		The command couldn't be completed successfully;
 *			the interpreter's result describes what went wrong.
 * TCL_RETURN		The command requests that the current procedure
 *			return; the interpreter's result contains the
 *			procedure's return value.
 * TCL_BREAK		The command requests that the innermost loop
 *			be exited; the interpreter's result is meaningless.
 * TCL_CONTINUE		Go on to the next iteration of the current loop;
 *			the interpreter's result is meaningless.
 */

#define TCL_OK		0
#define TCL_ERROR	1
#define TCL_RETURN	2
#define TCL_BREAK	3
#define TCL_CONTINUE	4

/*
 * The following definitions are the error codes returned by the conversion
 * routines:
 *
 * TCL_OK:			All characters were converted.
 *
 * TCL_CONVERT_NOSPACE:		The output buffer would not have been large
 *				enough for all of the converted data; as many
 *				characters as could fit were converted though.
 *
 * TCL_CONVERT_MULTIBYTE:	The last few bytes in the source string were
 *				the beginning of a multibyte sequence, but
 *				more bytes were needed to complete this
 *				sequence.  A subsequent call to the conversion
 *				routine should pass the beginning of this
 *				unconverted sequence plus additional bytes
 *				from the source stream to properly convert
 *				the formerly split-up multibyte sequence.
 *
 * TCL_CONVERT_SYNTAX:		The source stream contained an invalid
 *				character sequence.  This may occur if the
 *				input stream has been damaged or if the input
 *				encoding method was misidentified.  This error
 *				is reported only if TCL_ENCODING_STOPONERROR
 *				was specified.
 * 
 * TCL_CONVERT_UNKNOWN:		The source string contained a character
 *				that could not be represented in the target
 *				encoding.  This error is reported only if
 *				TCL_ENCODING_STOPONERROR was specified.
 */

#define TCL_CONVERT_MULTIBYTE		-1
#define TCL_CONVERT_SYNTAX		-2
#define TCL_CONVERT_UNKNOWN		-3
#define TCL_CONVERT_NOSPACE		-4

#define TCL_RESULT_SIZE 200

/*
 * Argument descriptors for math function callbacks in expressions:
 */

typedef enum {TCL_INT, TCL_DOUBLE, TCL_EITHER} Tcl_ValueType;
typedef struct Tcl_Value {
    Tcl_ValueType type;		/* Indicates intValue or doubleValue is
				 * valid, or both. */
    long intValue;		/* Integer value. */
    double doubleValue;		/* Double-precision floating value. */
} Tcl_Value;

/*
 * Forward declaration of Tcl_Obj to prevent an error when the forward
 * reference to Tcl_Obj is encountered in the procedure types declared 
 * below.
 */

struct Tcl_Obj;

/*
 * Procedure types defined by Tcl:
 */

typedef int (Tcl_AppInitProc) _ANSI_ARGS_((Tcl_Interp *interp));
typedef int (Tcl_AsyncProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int code));
typedef void (Tcl_ChannelProc) _ANSI_ARGS_((ClientData clientData, int mask));
typedef void (Tcl_CloseProc) _ANSI_ARGS_((ClientData data));
typedef void (Tcl_CmdDeleteProc) _ANSI_ARGS_((ClientData clientData));
typedef int (Tcl_CmdProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int argc, char *argv[]));
typedef void (Tcl_CmdTraceProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int level, char *command, Tcl_CmdProc *proc,
	ClientData cmdClientData, int argc, char *argv[]));
typedef void (Tcl_DupInternalRepProc) _ANSI_ARGS_((struct Tcl_Obj *srcPtr, 
        struct Tcl_Obj *dupPtr));
typedef int (Tcl_EventProc) _ANSI_ARGS_((Tcl_Event *evPtr, int flags));
typedef void (Tcl_EventCheckProc) _ANSI_ARGS_((ClientData clientData,
	int flags));
typedef int (Tcl_EventDeleteProc) _ANSI_ARGS_((Tcl_Event *evPtr,
        ClientData clientData));
typedef void (Tcl_EventSetupProc) _ANSI_ARGS_((ClientData clientData,
	int flags));
typedef void (Tcl_ExitProc) _ANSI_ARGS_((ClientData clientData));
typedef void (Tcl_FileProc) _ANSI_ARGS_((ClientData clientData, int mask));
typedef void (Tcl_FileFreeProc) _ANSI_ARGS_((ClientData clientData));
typedef void (Tcl_FreeInternalRepProc) _ANSI_ARGS_((struct Tcl_Obj *objPtr));
typedef void (Tcl_FreeProc) _ANSI_ARGS_((char *blockPtr));
typedef void (Tcl_IdleProc) _ANSI_ARGS_((ClientData clientData));
typedef void (Tcl_InterpDeleteProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp));
typedef int (Tcl_MathProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tcl_Value *args, Tcl_Value *resultPtr));
typedef void (Tcl_NamespaceDeleteProc) _ANSI_ARGS_((ClientData clientData));
typedef int (Tcl_ObjCmdProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int objc, struct Tcl_Obj * CONST objv[]));
typedef int (Tcl_PackageInitProc) _ANSI_ARGS_((Tcl_Interp *interp));
typedef void (Tcl_TcpAcceptProc) _ANSI_ARGS_((ClientData callbackData,
        Tcl_Channel chan, char *address, int port));
typedef void (Tcl_TimerProc) _ANSI_ARGS_((ClientData clientData));
typedef int (Tcl_SetFromAnyProc) _ANSI_ARGS_((Tcl_Interp *interp,
	struct Tcl_Obj *objPtr));
typedef void (Tcl_UpdateStringProc) _ANSI_ARGS_((struct Tcl_Obj *objPtr));
typedef char *(Tcl_VarTraceProc) _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, char *part1, char *part2, int flags));

/*
 * The following structure represents a type of object, which is a
 * particular internal representation for an object plus a set of
 * procedures that provide standard operations on objects of that type.
 */

typedef struct Tcl_ObjType {
    char *name;			/* Name of the type, e.g. "int". */
    Tcl_FreeInternalRepProc *freeIntRepProc;
				/* Called to free any storage for the type's
				 * internal rep. NULL if the internal rep
				 * does not need freeing. */
    Tcl_DupInternalRepProc *dupIntRepProc;
    				/* Called to create a new object as a copy
				 * of an existing object. */
    Tcl_UpdateStringProc *updateStringProc;
    				/* Called to update the string rep from the
				 * type's internal representation. */
    Tcl_SetFromAnyProc *setFromAnyProc;
    				/* Called to convert the object's internal
				 * rep to this type. Frees the internal rep
				 * of the old type. Returns TCL_ERROR on
				 * failure. */
} Tcl_ObjType;

/*
 * One of the following structures exists for each object in the Tcl
 * system. An object stores a value as either a string, some internal
 * representation, or both.
 */

typedef struct Tcl_Obj {
    int refCount;		/* When 0 the object will be freed. */
    char *bytes;		/* This points to the first byte of the
				 * object's string representation. The array
				 * must be followed by a null byte (i.e., at
				 * offset length) but may also contain
				 * embedded null characters. The array's
				 * storage is allocated by ckalloc. NULL
				 * means the string rep is invalid and must
				 * be regenerated from the internal rep.
				 * Clients should use Tcl_GetStringFromObj
				 * to get a pointer to the byte array as a
				 * readonly value. */
    int length;			/* The number of bytes at *bytes, not
				 * including the terminating null. */
    Tcl_ObjType *typePtr;	/* Denotes the object's type. Always
				 * corresponds to the type of the object's
				 * internal rep. NULL indicates the object
				 * has no internal rep (has no type). */
    union {			/* The internal representation: */
	long longValue;		/*   - an long integer value */
	double doubleValue;	/*   - a double-precision floating value */
	VOID *otherValuePtr;	/*   - another, type-specific value */
	struct {		/*   - internal rep as two pointers */
	    VOID *ptr1;
	    VOID *ptr2;
	} twoPtrValue;
    } internalRep;
} Tcl_Obj;

/*
 * Macros to increment and decrement a Tcl_Obj's reference count, and to
 * test whether an object is shared (i.e. has reference count > 1).
 * Note: clients should use Tcl_DecrRefCount() when they are finished using
 * an object, and should never call TclFreeObj() directly. TclFreeObj() is
 * only defined and made public in tcl.h to support Tcl_DecrRefCount's macro
 * definition. Note also that Tcl_DecrRefCount() refers to the parameter
 * "obj" twice. This means that you should avoid calling it with an
 * expression that is expensive to compute or has side effects.
 */

EXTERN void		Tcl_IncrRefCount _ANSI_ARGS_((Tcl_Obj *objPtr));
EXTERN void		Tcl_DecrRefCount _ANSI_ARGS_((Tcl_Obj *objPtr));
EXTERN int		Tcl_IsShared _ANSI_ARGS_((Tcl_Obj *objPtr));

#ifdef TCL_MEM_DEBUG
#   define Tcl_IncrRefCount(objPtr) \
	Tcl_DbIncrRefCount(objPtr, __FILE__, __LINE__)
#   define Tcl_DecrRefCount(objPtr) \
	Tcl_DbDecrRefCount(objPtr, __FILE__, __LINE__)
#   define Tcl_IsShared(objPtr) \
	Tcl_DbIsShared(objPtr, __FILE__, __LINE__)
#else
#   define Tcl_IncrRefCount(objPtr) \
	++(objPtr)->refCount
#   define Tcl_DecrRefCount(objPtr) \
	if (--(objPtr)->refCount <= 0) TclFreeObj(objPtr)
#   define Tcl_IsShared(objPtr) \
	((objPtr)->refCount > 1)
#endif

/*
 * Macros and definitions that help to debug the use of Tcl objects.
 * When TCL_MEM_DEBUG is defined, the Tcl_New* declarations are 
 * overridden to call debugging versions of the object creation procedures.
 */

EXTERN Tcl_Obj *	Tcl_NewBooleanObj _ANSI_ARGS_((int boolValue));
EXTERN Tcl_Obj *	Tcl_NewByteArrayObj _ANSI_ARGS_((CONST unsigned char *bytes,
			    int length));
EXTERN Tcl_Obj *	Tcl_NewDoubleObj _ANSI_ARGS_((double doubleValue));
EXTERN Tcl_Obj *	Tcl_NewIntObj _ANSI_ARGS_((int intValue));
EXTERN Tcl_Obj *	Tcl_NewListObj _ANSI_ARGS_((int objc,
			    Tcl_Obj *CONST objv[]));
EXTERN Tcl_Obj *	Tcl_NewLongObj _ANSI_ARGS_((long longValue));
EXTERN Tcl_Obj *	Tcl_NewObj _ANSI_ARGS_((void));
EXTERN Tcl_Obj *	Tcl_NewStringObj _ANSI_ARGS_((CONST char *bytes,
			    int length));

#ifdef TCL_MEM_DEBUG
#  define Tcl_NewBooleanObj(val) \
     Tcl_DbNewBooleanObj(val, __FILE__, __LINE__)
#  define Tcl_NewByteArrayObj(bytes, len) \
     Tcl_DbNewByteArrayObj(bytes, len, __FILE__, __LINE__)
#  define Tcl_NewDoubleObj(val) \
     Tcl_DbNewDoubleObj(val, __FILE__, __LINE__)
#  define Tcl_NewIntObj(val) \
     Tcl_DbNewLongObj(val, __FILE__, __LINE__)
#  define Tcl_NewListObj(objc, objv) \
     Tcl_DbNewListObj(objc, objv, __FILE__, __LINE__)
#  define Tcl_NewLongObj(val) \
     Tcl_DbNewLongObj(val, __FILE__, __LINE__)
#  define Tcl_NewObj() \
     Tcl_DbNewObj(__FILE__, __LINE__)
#  define Tcl_NewStringObj(bytes, len) \
     Tcl_DbNewStringObj(bytes, len, __FILE__, __LINE__)
#endif /* TCL_MEM_DEBUG */

/*
 * The following definitions support Tcl's namespace facility.
 * Note: the first five fields must match exactly the fields in a
 * Namespace structure (see tcl.h). 
 */

typedef struct Tcl_Namespace {
    char *name;                 /* The namespace's name within its parent
				 * namespace. This contains no ::'s. The
				 * name of the global namespace is ""
				 * although "::" is an synonym. */
    char *fullName;             /* The namespace's fully qualified name.
				 * This starts with ::. */
    ClientData clientData;      /* Arbitrary value associated with this
				 * namespace. */
    Tcl_NamespaceDeleteProc* deleteProc;
                                /* Procedure invoked when deleting the
				 * namespace to, e.g., free clientData. */
    struct Tcl_Namespace* parentPtr;
                                /* Points to the namespace that contains
				 * this one. NULL if this is the global
				 * namespace. */
} Tcl_Namespace;

/*
 * The following structure represents a call frame, or activation record.
 * A call frame defines a naming context for a procedure call: its local
 * scope (for local variables) and its namespace scope (used for non-local
 * variables; often the global :: namespace). A call frame can also define
 * the naming context for a namespace eval or namespace inscope command:
 * the namespace in which the command's code should execute. The
 * Tcl_CallFrame structures exist only while procedures or namespace
 * eval/inscope's are being executed, and provide a Tcl call stack.
 * 
 * A call frame is initialized and pushed using Tcl_PushCallFrame and
 * popped using Tcl_PopCallFrame. Storage for a Tcl_CallFrame must be
 * provided by the Tcl_PushCallFrame caller, and callers typically allocate
 * them on the C call stack for efficiency. For this reason, Tcl_CallFrame
 * is defined as a structure and not as an opaque token. However, most
 * Tcl_CallFrame fields are hidden since applications should not access
 * them directly; others are declared as "dummyX".
 *
 * WARNING!! The structure definition must be kept consistent with the
 * CallFrame structure in tclInt.h. If you change one, change the other.
 */

typedef struct Tcl_CallFrame {
    Tcl_Namespace *nsPtr;
    int dummy1;
    int dummy2;
    char *dummy3;
    char *dummy4;
    char *dummy5;
    int dummy6;
    char *dummy7;
    char *dummy8;
    int dummy9;
    char* dummy10;
} Tcl_CallFrame;

/*
 * Information about commands that is returned by Tcl_GetCommandInfo and
 * passed to Tcl_SetCommandInfo. objProc is an objc/objv object-based
 * command procedure while proc is a traditional Tcl argc/argv
 * string-based procedure. Tcl_CreateObjCommand and Tcl_CreateCommand
 * ensure that both objProc and proc are non-NULL and can be called to
 * execute the command. However, it may be faster to call one instead of
 * the other. The member isNativeObjectProc is set to 1 if an
 * object-based procedure was registered by Tcl_CreateObjCommand, and to
 * 0 if a string-based procedure was registered by Tcl_CreateCommand.
 * The other procedure is typically set to a compatibility wrapper that
 * does string-to-object or object-to-string argument conversions then
 * calls the other procedure.
 */
     
typedef struct Tcl_CmdInfo {
    int isNativeObjectProc;	 /* 1 if objProc was registered by a call to
				  * Tcl_CreateObjCommand; 0 otherwise.
				  * Tcl_SetCmdInfo does not modify this
				  * field. */
    Tcl_ObjCmdProc *objProc;	 /* Command's object-based procedure. */
    ClientData objClientData;	 /* ClientData for object proc. */
    Tcl_CmdProc *proc;		 /* Command's string-based procedure. */
    ClientData clientData;	 /* ClientData for string proc. */
    Tcl_CmdDeleteProc *deleteProc;
                                 /* Procedure to call when command is
                                  * deleted. */
    ClientData deleteData;	 /* Value to pass to deleteProc (usually
				  * the same as clientData). */
    Tcl_Namespace *namespacePtr; /* Points to the namespace that contains
				  * this command. Note that Tcl_SetCmdInfo
				  * will not change a command's namespace;
				  * use Tcl_RenameCommand to do that. */

} Tcl_CmdInfo;

/*
 * The structure defined below is used to hold dynamic strings.  The only
 * field that clients should use is the string field, and they should
 * never modify it.
 */

#define TCL_DSTRING_STATIC_SIZE 200
typedef struct Tcl_DString {
    char *string;		/* Points to beginning of string:  either
				 * staticSpace below or a malloced array. */
    int length;			/* Number of non-NULL characters in the
				 * string. */
    int spaceAvl;		/* Total number of bytes available for the
				 * string and its terminating NULL char. */
    char staticSpace[TCL_DSTRING_STATIC_SIZE];
				/* Space to use in common case where string
				 * is small. */
} Tcl_DString;

#define Tcl_DStringLength(dsPtr) ((dsPtr)->length)
#define Tcl_DStringValue(dsPtr) ((dsPtr)->string)
#define Tcl_DStringTrunc Tcl_DStringSetLength

/*
 * Definitions for the maximum number of digits of precision that may
 * be specified in the "tcl_precision" variable, and the number of
 * characters of buffer space required by Tcl_PrintDouble.
 */
 
#define TCL_MAX_PREC 17
#define TCL_DOUBLE_SPACE (TCL_MAX_PREC+10)

/*
 * Definition for a number of bytes of buffer space sufficient to hold the
 * string representation of an integer in base 10 (assuming the existence
 * of 64-bit integers).
 */

#define TCL_INTEGER_SPACE	24

/*
 * Flag that may be passed to Tcl_ConvertElement to force it not to
 * output braces (careful!  if you change this flag be sure to change
 * the definitions at the front of tclUtil.c).
 */

#define TCL_DONT_USE_BRACES	1

/*
 * Flag that may be passed to Tcl_GetIndexFromObj to force it to disallow
 * abbreviated strings.
 */

#define TCL_EXACT	1

/*
 * Flag values passed to Tcl_RecordAndEval.
 * WARNING: these bit choices must not conflict with the bit choices
 * for evalFlag bits in tclInt.h!!
 */

#define TCL_NO_EVAL		0x10000
#define TCL_EVAL_GLOBAL		0x20000

/*
 * Special freeProc values that may be passed to Tcl_SetResult (see
 * the man page for details):
 */

#define TCL_VOLATILE	((Tcl_FreeProc *) 1)
#define TCL_STATIC	((Tcl_FreeProc *) 0)
#define TCL_DYNAMIC	((Tcl_FreeProc *) 3)

/*
 * Flag values passed to variable-related procedures.
 */

#define TCL_GLOBAL_ONLY		 1
#define TCL_NAMESPACE_ONLY	 2
#define TCL_APPEND_VALUE	 4
#define TCL_LIST_ELEMENT	 8
#define TCL_TRACE_READS		 0x10
#define TCL_TRACE_WRITES	 0x20
#define TCL_TRACE_UNSETS	 0x40
#define TCL_TRACE_DESTROYED	 0x80
#define TCL_INTERP_DESTROYED	 0x100
#define TCL_LEAVE_ERR_MSG	 0x200
#define TCL_PARSE_PART1		 0x400

/*
 * Types for linked variables:
 */

#define TCL_LINK_INT		1
#define TCL_LINK_DOUBLE		2
#define TCL_LINK_BOOLEAN	3
#define TCL_LINK_STRING		4
#define TCL_LINK_READ_ONLY	0x80

/*
 * The following declarations either map ckalloc and ckfree to
 * malloc and free, or they map them to procedures with all sorts
 * of debugging hooks defined in tclCkalloc.c.
 */

EXTERN char *		Tcl_Alloc _ANSI_ARGS_((unsigned int size));
EXTERN void		Tcl_Free _ANSI_ARGS_((char *ptr));
EXTERN char *		Tcl_Realloc _ANSI_ARGS_((char *ptr,
			    unsigned int size));
EXTERN void		Tcl_InitMemory _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN int		Tcl_DumpActiveMemory _ANSI_ARGS_((CONST char *fileName));
EXTERN void		Tcl_ValidateAllMemory _ANSI_ARGS_((CONST char *file,
			    int line));

#ifdef TCL_MEM_DEBUG

#  define Tcl_Alloc(x) Tcl_DbCkalloc(x, __FILE__, __LINE__)
#  define Tcl_Free(x)  Tcl_DbCkfree(x, __FILE__, __LINE__)
#  define Tcl_Realloc(x,y) Tcl_DbCkrealloc((x), (y),__FILE__, __LINE__)
#  define ckalloc(x) Tcl_DbCkalloc(x, __FILE__, __LINE__)
#  define ckfree(x)  Tcl_DbCkfree(x, __FILE__, __LINE__)
#  define ckrealloc(x,y) Tcl_DbCkrealloc((x), (y),__FILE__, __LINE__)

#else

/*
 * If USE_TCLALLOC is true, then we need to call Tcl_Alloc instead of
 * the native malloc/free.  The only time USE_TCLALLOC should not be
 * true is when compiling the Tcl/Tk libraries on Unix systems.  In this
 * case we can safely call the native malloc/free directly as a performance
 * optimization.
 */

#  if USE_TCLALLOC
#     define ckalloc(x) Tcl_Alloc(x)
#     define ckfree(x) Tcl_Free(x)
#     define ckrealloc(x,y) Tcl_Realloc(x,y)
#  else
#     define ckalloc(x) malloc(x)
#     define ckfree(x)  free(x)
#     define ckrealloc(x,y) realloc(x,y)
#  endif
#  define Tcl_InitMemory(x)
#  define Tcl_DumpActiveMemory(x)
#  define Tcl_ValidateAllMemory(x,y)

#endif /* TCL_MEM_DEBUG */

/*
 * Forward declaration of Tcl_HashTable.  Needed by some C++ compilers
 * to prevent errors when the forward reference to Tcl_HashTable is
 * encountered in the Tcl_HashEntry structure.
 */

#ifdef __cplusplus
struct Tcl_HashTable;
#endif

/*
 * Structure definition for an entry in a hash table.  No-one outside
 * Tcl should access any of these fields directly;  use the macros
 * defined below.
 */

typedef struct Tcl_HashEntry {
    struct Tcl_HashEntry *nextPtr;	/* Pointer to next entry in this
					 * hash bucket, or NULL for end of
					 * chain. */
    struct Tcl_HashTable *tablePtr;	/* Pointer to table containing entry. */
    struct Tcl_HashEntry **bucketPtr;	/* Pointer to bucket that points to
					 * first entry in this entry's chain:
					 * used for deleting the entry. */
    ClientData clientData;		/* Application stores something here
					 * with Tcl_SetHashValue. */
    union {				/* Key has one of these forms: */
	char *oneWordValue;		/* One-word value for key. */
	int words[1];			/* Multiple integer words for key.
					 * The actual size will be as large
					 * as necessary for this table's
					 * keys. */
	char string[4];			/* String for key.  The actual size
					 * will be as large as needed to hold
					 * the key. */
    } key;				/* MUST BE LAST FIELD IN RECORD!! */
} Tcl_HashEntry;

/*
 * Structure definition for a hash table.  Must be in tcl.h so clients
 * can allocate space for these structures, but clients should never
 * access any fields in this structure.
 */

#define TCL_SMALL_HASH_TABLE 4
typedef struct Tcl_HashTable {
    Tcl_HashEntry **buckets;		/* Pointer to bucket array.  Each
					 * element points to first entry in
					 * bucket's hash chain, or NULL. */
    Tcl_HashEntry *staticBuckets[TCL_SMALL_HASH_TABLE];
					/* Bucket array used for small tables
					 * (to avoid mallocs and frees). */
    int numBuckets;			/* Total number of buckets allocated
					 * at **bucketPtr. */
    int numEntries;			/* Total number of entries present
					 * in table. */
    int rebuildSize;			/* Enlarge table when numEntries gets
					 * to be this large. */
    int downShift;			/* Shift count used in hashing
					 * function.  Designed to use high-
					 * order bits of randomized keys. */
    int mask;				/* Mask value used in hashing
					 * function. */
    int keyType;			/* Type of keys used in this table. 
					 * It's either TCL_STRING_KEYS,
					 * TCL_ONE_WORD_KEYS, or an integer
					 * giving the number of ints that
                                         * is the size of the key.
					 */
    Tcl_HashEntry *(*findProc) _ANSI_ARGS_((struct Tcl_HashTable *tablePtr,
	    CONST char *key));
    Tcl_HashEntry *(*createProc) _ANSI_ARGS_((struct Tcl_HashTable *tablePtr,
	    CONST char *key, int *newPtr));
} Tcl_HashTable;

/*
 * Structure definition for information used to keep track of searches
 * through hash tables:
 */

typedef struct Tcl_HashSearch {
    Tcl_HashTable *tablePtr;		/* Table being searched. */
    int nextIndex;			/* Index of next bucket to be
					 * enumerated after present one. */
    Tcl_HashEntry *nextEntryPtr;	/* Next entry to be enumerated in the
					 * the current bucket. */
} Tcl_HashSearch;

/*
 * Acceptable key types for hash tables:
 */

#define TCL_STRING_KEYS		0
#define TCL_ONE_WORD_KEYS	1

/*
 * Macros for clients to use to access fields of hash entries:
 */

#define Tcl_GetHashValue(h) ((h)->clientData)
#define Tcl_SetHashValue(h, value) ((h)->clientData = (ClientData) (value))
#define Tcl_GetHashKey(tablePtr, h) \
    ((char *) (((tablePtr)->keyType == TCL_ONE_WORD_KEYS) ? (h)->key.oneWordValue \
						: (h)->key.string))

/*
 * Macros to use for clients to use to invoke find and create procedures
 * for hash tables:
 */

#define Tcl_FindHashEntry(tablePtr, key) \
	(*((tablePtr)->findProc))(tablePtr, key)
#define Tcl_CreateHashEntry(tablePtr, key, newPtr) \
	(*((tablePtr)->createProc))(tablePtr, key, newPtr)

/*
 * Flag values to pass to Tcl_DoOneEvent to disable searches
 * for some kinds of events:
 */

#define TCL_DONT_WAIT		(1<<1)
#define TCL_WINDOW_EVENTS	(1<<2)
#define TCL_FILE_EVENTS		(1<<3)
#define TCL_TIMER_EVENTS	(1<<4)
#define TCL_IDLE_EVENTS		(1<<5)	/* WAS 0x10 ???? */
#define TCL_ALL_EVENTS		(~TCL_DONT_WAIT)

/*
 * The following structure defines a generic event for the Tcl event
 * system.  These are the things that are queued in calls to Tcl_QueueEvent
 * and serviced later by Tcl_DoOneEvent.  There can be many different
 * kinds of events with different fields, corresponding to window events,
 * timer events, etc.  The structure for a particular event consists of
 * a Tcl_Event header followed by additional information specific to that
 * event.
 */

struct Tcl_Event {
    Tcl_EventProc *proc;	/* Procedure to call to service this event. */
    struct Tcl_Event *nextPtr;	/* Next in list of pending events, or NULL. */
};

/*
 * Positions to pass to Tcl_QueueEvent:
 */

typedef enum {
    TCL_QUEUE_TAIL, TCL_QUEUE_HEAD, TCL_QUEUE_MARK
} Tcl_QueuePosition;

/*
 * Values to pass to Tcl_SetServiceMode to specify the behavior of notifier
 * event routines.
 */

#define TCL_SERVICE_NONE 0
#define TCL_SERVICE_ALL 1

/*
 * The following structure keeps is used to hold a time value, either as
 * an absolute time (the number of seconds from the epoch) or as an
 * elapsed time. On Unix systems the epoch is Midnight Jan 1, 1970 GMT.
 * On Macintosh systems the epoch is Midnight Jan 1, 1904 GMT.
 */

typedef struct Tcl_Time {
    long sec;			/* Seconds. */
    long usec;			/* Microseconds. */
} Tcl_Time;

/*
 * Bits to pass to Tcl_CreateFileHandler and Tcl_CreateChannelHandler
 * to indicate what sorts of events are of interest:
 */

#define TCL_READABLE	(1<<1)
#define TCL_WRITABLE	(1<<2)
#define TCL_EXCEPTION	(1<<3)
#define TCL_ACTIVE	0

typedef struct Tcl_StaticFile {
    CONST char *name;
    char **content;
} Tcl_StaticFile;

/*
 * Flag values to pass to Tcl_OpenCommandChannel to indicate the
 * disposition of the stdio handles.  TCL_STDIN, TCL_STDOUT, TCL_STDERR,
 * are also used in Tcl_GetStdChannel.
 */

#define TCL_STDIN		(1<<1)	
#define TCL_STDOUT		(1<<2)
#define TCL_STDERR		(1<<3)
#define TCL_ENFORCE_MODE	(1<<4)

/*
 * Typedefs for the various operations in a channel type:
 */

typedef int	(Tcl_DriverBlockModeProc) _ANSI_ARGS_((
		    ClientData instanceData, int mode));
typedef int	(Tcl_DriverCloseProc) _ANSI_ARGS_((ClientData instanceData,
		    Tcl_Interp *interp));
typedef int	(Tcl_DriverInputProc) _ANSI_ARGS_((ClientData instanceData,
		    char *buf, int toRead, int *errorCodePtr));
typedef int	(Tcl_DriverOutputProc) _ANSI_ARGS_((ClientData instanceData,
		    char *buf, int toWrite, int *errorCodePtr));
typedef int	(Tcl_DriverSeekProc) _ANSI_ARGS_((ClientData instanceData,
		    long offset, int mode, int *errorCodePtr));
typedef int	(Tcl_DriverSetOptionProc) _ANSI_ARGS_((
		    ClientData instanceData, Tcl_Interp *interp,
	            char *optionName, char *value));
typedef int	(Tcl_DriverGetOptionProc) _ANSI_ARGS_((
		    ClientData instanceData, Tcl_Interp *interp,
		    char *optionName, Tcl_DString *dsPtr));
typedef void	(Tcl_DriverWatchProc) _ANSI_ARGS_((
		    ClientData instanceData, int mask));
typedef int	(Tcl_DriverGetHandleProc) _ANSI_ARGS_((
		    ClientData instanceData, int direction,
		    ClientData *handlePtr));

/*
 * Enum for different end of line translation and recognition modes.
 */

typedef enum Tcl_EolTranslation {
    TCL_TRANSLATE_AUTO,			/* Eol == \r, \n and \r\n. */
    TCL_TRANSLATE_CR,			/* Eol == \r. */
    TCL_TRANSLATE_LF,			/* Eol == \n. */
    TCL_TRANSLATE_CRLF			/* Eol == \r\n. */
} Tcl_EolTranslation;

/*
 * Definitions for different byteorders.
 */

typedef int Tcl_ByteOrder;

#define TCL_BIGENDIAN 0
#define TCL_SMALLENDIAN 1

/*
 * struct Tcl_ChannelType:
 *
 * One such structure exists for each type (kind) of channel.
 * It collects together in one place all the functions that are
 * part of the specific channel type.
 */

typedef struct Tcl_ChannelType {
    char *typeName;			/* The name of the channel type in Tcl
                                         * commands. This storage is owned by
                                         * channel type. */
    Tcl_DriverBlockModeProc *blockModeProc;
    					/* Set blocking mode for the
                                         * raw channel. May be NULL. */
    Tcl_DriverCloseProc *closeProc;	/* Procedure to call to close
                                         * the channel. */
    Tcl_DriverInputProc *inputProc;	/* Procedure to call for input
                                         * on channel. */
    Tcl_DriverOutputProc *outputProc;	/* Procedure to call for output
                                         * on channel. */
    Tcl_DriverSeekProc *seekProc;	/* Procedure to call to seek
                                         * on the channel. May be NULL. */
    Tcl_DriverSetOptionProc *setOptionProc;
    					/* Set an option on a channel. */
    Tcl_DriverGetOptionProc *getOptionProc;
    					/* Get an option from a channel. */
    Tcl_DriverWatchProc *watchProc;	/* Set up the notifier to watch
                                         * for events on this channel. */
    Tcl_DriverGetHandleProc *getHandleProc;
					/* Get an OS handle from the channel
                                         * or NULL if not supported. */
    VOID *reserved;			/* reserved for future expansion */
} Tcl_ChannelType;

/*
 * The following flags determine whether the blockModeProc above should
 * set the channel into blocking or nonblocking mode. They are passed
 * as arguments to the blockModeProc procedure in the above structure.
 */

#define TCL_MODE_BLOCKING 0		/* Put channel into blocking mode. */
#define TCL_MODE_NONBLOCKING 1		/* Put channel into nonblocking
					 * mode. */

/*
 * Enum for different types of file paths.
 */

typedef enum Tcl_PathType {
    TCL_PATH_ABSOLUTE,
    TCL_PATH_RELATIVE,
    TCL_PATH_VOLUME_RELATIVE
} Tcl_PathType;

/*
 * Exported Tcl procedures:
 */

EXTERN void		panic _ANSI_ARGS_(TCL_VARARGS(CONST char *,format));
EXTERN void		Tcl_AddErrorInfo _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *message));
EXTERN void		Tcl_AddObjErrorInfo _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *message, int length));
EXTERN void		Tcl_AllowExceptions _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN int		Tcl_AppendAllObjTypes _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *objPtr));
EXTERN void		Tcl_AppendElement _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string));
EXTERN void		Tcl_AppendResult _ANSI_ARGS_(
			    TCL_VARARGS(Tcl_Interp *,interp));
EXTERN void		Tcl_AppendToObj _ANSI_ARGS_((Tcl_Obj *objPtr,
			    char *bytes, int length));
EXTERN void		Tcl_AppendStringsToObj _ANSI_ARGS_(
			    TCL_VARARGS(Tcl_Obj *,objPtr));
EXTERN Tcl_AsyncHandler	Tcl_AsyncCreate _ANSI_ARGS_((Tcl_AsyncProc *proc,
			    ClientData clientData));
EXTERN void		Tcl_AsyncDelete _ANSI_ARGS_((Tcl_AsyncHandler async));
EXTERN int		Tcl_AsyncInvoke _ANSI_ARGS_((Tcl_Interp *interp,
			    int code));
EXTERN void		Tcl_AsyncMark _ANSI_ARGS_((Tcl_AsyncHandler async));
EXTERN int		Tcl_AsyncReady _ANSI_ARGS_((void));
EXTERN void		Tcl_BackgroundError _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN char		Tcl_Backslash _ANSI_ARGS_((CONST char *src,
			    int *readPtr));
EXTERN int		Tcl_BadChannelOption _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *optionName, CONST char *optionList));
EXTERN void		Tcl_CallWhenDeleted _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_InterpDeleteProc *proc,
			    ClientData clientData));
EXTERN void		Tcl_CancelIdleCall _ANSI_ARGS_((Tcl_IdleProc *idleProc,
			    ClientData clientData));
#define Tcl_Ckalloc Tcl_Alloc
#define Tcl_Ckfree Tcl_Free
#define Tcl_Ckrealloc Tcl_Realloc
EXTERN int		Tcl_Close _ANSI_ARGS_((Tcl_Interp *interp,
        		    Tcl_Channel chan));
EXTERN int		Tcl_CommandComplete _ANSI_ARGS_((char *cmd));
EXTERN char *		Tcl_Concat _ANSI_ARGS_((int argc, char **argv));
EXTERN Tcl_Obj *	Tcl_ConcatObj _ANSI_ARGS_((int objc,
			    Tcl_Obj *CONST objv[]));
EXTERN int		Tcl_ConvertCountedElement _ANSI_ARGS_((CONST char *src,
			    int length, char *dst, int flags));
EXTERN int		Tcl_ConvertElement _ANSI_ARGS_((CONST char *src,
			    char *dst, int flags));
EXTERN int		Tcl_ConvertToType _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, Tcl_ObjType *typePtr));
EXTERN int		Tcl_CreateAlias _ANSI_ARGS_((Tcl_Interp *slave,
			    char *slaveCmd, Tcl_Interp *target,
        		    char *targetCmd, int argc, char **argv));
EXTERN int		Tcl_CreateAliasObj _ANSI_ARGS_((Tcl_Interp *slave,
			    char *slaveCmd, Tcl_Interp *target,
        		    char *targetCmd, int objc,
		            Tcl_Obj *CONST objv[]));
EXTERN Tcl_Channel	Tcl_CreateChannel _ANSI_ARGS_((
    			    Tcl_ChannelType *typePtr, char *chanName,
                            ClientData instanceData, int mask));
EXTERN void		Tcl_CreateChannelHandler _ANSI_ARGS_((
			    Tcl_Channel chan, int mask,
                            Tcl_ChannelProc *proc, ClientData clientData));
EXTERN void		Tcl_CreateCloseHandler _ANSI_ARGS_((
			    Tcl_Channel chan, Tcl_CloseProc *proc,
                            ClientData clientData));
EXTERN Tcl_Command	Tcl_CreateCommand _ANSI_ARGS_((Tcl_Interp *interp,
			    char *cmdName, Tcl_CmdProc *proc,
			    ClientData clientData,
			    Tcl_CmdDeleteProc *deleteProc));
EXTERN void		Tcl_CreateEventSource _ANSI_ARGS_((
			    Tcl_EventSetupProc *setupProc,
			    Tcl_EventCheckProc *checkProc,
			    ClientData clientData));
EXTERN void		Tcl_CreateExitHandler _ANSI_ARGS_((Tcl_ExitProc *proc,
			    ClientData clientData));
EXTERN void		Tcl_CreateFileHandler _ANSI_ARGS_((
    			    int fd, int mask, Tcl_FileProc *proc,
			    ClientData clientData));
EXTERN Tcl_Interp *	Tcl_CreateInterp _ANSI_ARGS_((void));
EXTERN void		Tcl_CreateMathFunc _ANSI_ARGS_((Tcl_Interp *interp,
			    char *name, int numArgs, Tcl_ValueType *argTypes,
			    Tcl_MathProc *proc, ClientData clientData));
EXTERN Tcl_Command	Tcl_CreateObjCommand _ANSI_ARGS_((
			    Tcl_Interp *interp, CONST char *cmdName,
			    Tcl_ObjCmdProc *proc, ClientData clientData,
			    Tcl_CmdDeleteProc *deleteProc));
EXTERN Tcl_Interp *	Tcl_CreateSlave _ANSI_ARGS_((Tcl_Interp *interp,
		            char *slaveName, int isSafe));
EXTERN Tcl_TimerToken	Tcl_CreateTimerHandler _ANSI_ARGS_((int milliseconds,
			    Tcl_TimerProc *proc, ClientData clientData));
EXTERN Tcl_Trace	Tcl_CreateTrace _ANSI_ARGS_((Tcl_Interp *interp,
			    int level, Tcl_CmdTraceProc *proc,
			    ClientData clientData));
EXTERN char *		Tcl_DbCkalloc _ANSI_ARGS_((unsigned int size,
			    CONST char *file, int line));
EXTERN int		Tcl_DbCkfree _ANSI_ARGS_((char *ptr,
			    CONST char *file, int line));
EXTERN char *		Tcl_DbCkrealloc _ANSI_ARGS_((char *ptr,
			    unsigned int size, CONST char *file, int line));
EXTERN void		Tcl_DbDecrRefCount _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *file, int line));
EXTERN void		Tcl_DbIncrRefCount _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *file, int line));
EXTERN int		Tcl_DbIsShared _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewBooleanObj _ANSI_ARGS_((int boolValue,
                            CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewByteArrayObj _ANSI_ARGS_((CONST unsigned char *bytes,
			    int length, CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewDoubleObj _ANSI_ARGS_((double doubleValue,
                            CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewListObj _ANSI_ARGS_((int objc,
			    Tcl_Obj *CONST objv[], CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewLongObj _ANSI_ARGS_((long longValue,
                            CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewObj _ANSI_ARGS_((CONST char *file, int line));
EXTERN Tcl_Obj *	Tcl_DbNewStringObj _ANSI_ARGS_((CONST char *bytes,
			    int length, CONST char *file, int line));
EXTERN void		Tcl_DefineStaticFile _ANSI_ARGS_((
			    Tcl_StaticFile *table));
EXTERN void		Tcl_DeleteAssocData _ANSI_ARGS_((Tcl_Interp *interp,
                            CONST char *name));
EXTERN int		Tcl_DeleteCommand _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *cmdName));
EXTERN void		Tcl_DeleteCommandFromToken _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Command command));
EXTERN void		Tcl_DeleteChannelHandler _ANSI_ARGS_((
    			    Tcl_Channel chan, Tcl_ChannelProc *proc,
                            ClientData clientData));
EXTERN void		Tcl_DeleteCloseHandler _ANSI_ARGS_((
			    Tcl_Channel chan, Tcl_CloseProc *proc,
                            ClientData clientData));
EXTERN void		Tcl_DeleteEvents _ANSI_ARGS_((
			    Tcl_EventDeleteProc *proc,
                            ClientData clientData));
EXTERN void		Tcl_DeleteEventSource _ANSI_ARGS_((
			    Tcl_EventSetupProc *setupProc,
			    Tcl_EventCheckProc *checkProc,
			    ClientData clientData));
EXTERN void		Tcl_DeleteExitHandler _ANSI_ARGS_((Tcl_ExitProc *proc,
			    ClientData clientData));
EXTERN void		Tcl_DeleteFileHandler _ANSI_ARGS_((int fd));
EXTERN void		Tcl_DeleteHashEntry _ANSI_ARGS_((
			    Tcl_HashEntry *entryPtr));
EXTERN void		Tcl_DeleteHashTable _ANSI_ARGS_((
			    Tcl_HashTable *tablePtr));
EXTERN void		Tcl_DeleteInterp _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN void		Tcl_DeleteTimerHandler _ANSI_ARGS_((
			    Tcl_TimerToken token));
EXTERN void		Tcl_DeleteTrace _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Trace trace));
EXTERN void		Tcl_DetachPids _ANSI_ARGS_((int numPids, Tcl_Pid *pidPtr));
EXTERN void		Tcl_DontCallWhenDeleted _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_InterpDeleteProc *proc,
			    ClientData clientData));
EXTERN int		Tcl_DoOneEvent _ANSI_ARGS_((int flags));
EXTERN void		Tcl_DoWhenIdle _ANSI_ARGS_((Tcl_IdleProc *proc,
			    ClientData clientData));
EXTERN char *		Tcl_DStringAppend _ANSI_ARGS_((Tcl_DString *dsPtr,
			    CONST char *string, int length));
EXTERN char *		Tcl_DStringAppendElement _ANSI_ARGS_((
			    Tcl_DString *dsPtr, CONST char *string));
EXTERN void		Tcl_DStringEndSublist _ANSI_ARGS_((Tcl_DString *dsPtr));
EXTERN void		Tcl_DStringFree _ANSI_ARGS_((Tcl_DString *dsPtr));
EXTERN void		Tcl_DStringGetResult _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_DString *dsPtr));
EXTERN void		Tcl_DStringInit _ANSI_ARGS_((Tcl_DString *dsPtr));
EXTERN void		Tcl_DStringResult _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_DString *dsPtr));
EXTERN void		Tcl_DStringSetLength _ANSI_ARGS_((Tcl_DString *dsPtr,
			    int length));
EXTERN void		Tcl_DStringStartSublist _ANSI_ARGS_((
			    Tcl_DString *dsPtr));
EXTERN Tcl_Obj *	Tcl_DuplicateObj _ANSI_ARGS_((Tcl_Obj *objPtr));
EXTERN int		Tcl_Eof _ANSI_ARGS_((Tcl_Channel chan));
EXTERN char *		Tcl_ErrnoId _ANSI_ARGS_((void));
EXTERN char *		Tcl_ErrnoMsg _ANSI_ARGS_((int err));
EXTERN int		Tcl_Eval _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string));
EXTERN int		Tcl_EvalFile _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *fileName));
EXTERN void		Tcl_EventuallyFree _ANSI_ARGS_((ClientData clientData,
			    Tcl_FreeProc *freeProc));
EXTERN int		Tcl_EvalObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr));
EXTERN int		Tcl_EvalObj2 _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, int flags));
EXTERN void		Tcl_Exit _ANSI_ARGS_((int status));
EXTERN int		Tcl_ExposeCommand _ANSI_ARGS_((Tcl_Interp *interp,
        		    char *hiddenCmdToken, char *cmdName));
EXTERN int		Tcl_ExprBoolean _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, int *ptr));
EXTERN int		Tcl_ExprBooleanObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, int *ptr));
EXTERN int		Tcl_ExprDouble _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, double *ptr));
EXTERN int		Tcl_ExprDoubleObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, double *ptr));
EXTERN int		Tcl_ExprLong _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, long *ptr));
EXTERN int		Tcl_ExprLongObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, long *ptr));
EXTERN int		Tcl_ExprObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, Tcl_Obj **resultPtrPtr));
EXTERN int		Tcl_ExprString _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string));
EXTERN int		Tcl_ExternalToUtf _ANSI_ARGS_((Tcl_Interp *interp,
			    VOID *encoding, CONST char *src, int srcLen,
			    int flags, VOID *statePtr, char *dst,
			    int dstLen, int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));
EXTERN char *		Tcl_ExternalToUtfDString _ANSI_ARGS_((
			    VOID *encoding, CONST char *src,
			    int srcLen, Tcl_DString *dsPtr));
EXTERN void		Tcl_Finalize _ANSI_ARGS_((void));
EXTERN void		Tcl_FindExecutable _ANSI_ARGS_((CONST char *argv0));
EXTERN VOID *		Tcl_FindPackageHandle _ANSI_ARGS_((
			    CONST char *packageName));
EXTERN Tcl_HashEntry *	Tcl_FirstHashEntry _ANSI_ARGS_((
			    Tcl_HashTable *tablePtr,
			    Tcl_HashSearch *searchPtr));
EXTERN int		Tcl_Flush _ANSI_ARGS_((Tcl_Channel chan));
EXTERN void		TclFreeObj _ANSI_ARGS_((Tcl_Obj *objPtr));
EXTERN void		Tcl_FreeResult _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN int		Tcl_GetAlias _ANSI_ARGS_((Tcl_Interp *interp,
       			    char *slaveCmd, Tcl_Interp **targetInterpPtr,
                            char **targetCmdPtr, int *argcPtr,
			    char ***argvPtr));
EXTERN int		Tcl_GetAliasObj _ANSI_ARGS_((Tcl_Interp *interp,
       			    char *slaveCmd, Tcl_Interp **targetInterpPtr,
                            char **targetCmdPtr, int *objcPtr,
			    Tcl_Obj ***objv));
EXTERN ClientData	Tcl_GetAssocData _ANSI_ARGS_((Tcl_Interp *interp,
                            CONST char *name, Tcl_InterpDeleteProc **procPtr));
EXTERN int		Tcl_GetBoolean _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string, int *boolPtr));
EXTERN int		Tcl_GetBooleanFromObj _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *objPtr,
			    int *boolPtr));
EXTERN unsigned char *	Tcl_GetByteArrayFromObj _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int *lengthPtr));
EXTERN Tcl_Channel	Tcl_GetChannel _ANSI_ARGS_((Tcl_Interp *interp,
	        	    CONST char *chanName, int *modePtr));
EXTERN int		Tcl_GetChannelBufferSize _ANSI_ARGS_((
    			    Tcl_Channel chan));
  /* Andreas Kupries <a.kupries@westend.com>, 05/31/1997.
   * Support of Tcl-Trf (binio).
   */
EXTERN int		Tcl_GetChannelByteorder _ANSI_ARGS_((
			    Tcl_Channel chan));
EXTERN int		Tcl_GetHostByteorder _ANSI_ARGS_((void));
EXTERN int		Tcl_GetChannelHandle _ANSI_ARGS_((Tcl_Channel chan,
	        	    int direction, ClientData *handlePtr));
EXTERN ClientData	Tcl_GetChannelInstanceData _ANSI_ARGS_((
    			    Tcl_Channel chan));
EXTERN int		Tcl_GetChannelMode _ANSI_ARGS_((Tcl_Channel chan));
EXTERN char *		Tcl_GetChannelName _ANSI_ARGS_((Tcl_Channel chan));
EXTERN int		Tcl_GetChannelOption _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Channel chan, char *optionName,
			    Tcl_DString *dsPtr));
EXTERN Tcl_ChannelType * Tcl_GetChannelType _ANSI_ARGS_((Tcl_Channel chan));
EXTERN int		Tcl_GetCommandInfo _ANSI_ARGS_((Tcl_Interp *interp,
			    char *cmdName, Tcl_CmdInfo *infoPtr));
EXTERN char *		Tcl_GetCommandName _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Command command));
EXTERN int		Tcl_GetDouble _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, double *doublePtr));
EXTERN int		Tcl_GetDoubleFromObj _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *objPtr,
			    double *doublePtr));
EXTERN int		Tcl_GetErrno _ANSI_ARGS_((void));
EXTERN char *		Tcl_GetHostName _ANSI_ARGS_((void));
EXTERN int		Tcl_GetIndexFromObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, char **tablePtr,
			    CONST char *msg, int flags, int *indexPtr));
EXTERN int		Tcl_GetIndexFromObjStruct _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, char **tablePtr, int offset,
			    CONST char *msg, int flags, int *indexPtr));
EXTERN int		Tcl_GetInt _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string, int *intPtr));
EXTERN int		Tcl_GetInterpPath _ANSI_ARGS_((Tcl_Interp *askInterp,
			    Tcl_Interp *slaveInterp));
EXTERN int		Tcl_GetIntFromObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, int *intPtr));
EXTERN int		Tcl_GetLongFromObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, long *longPtr));
EXTERN Tcl_Interp *	Tcl_GetMaster _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN CONST char *	Tcl_GetNameOfExecutable _ANSI_ARGS_((void));
EXTERN Tcl_Obj *	Tcl_GetObjResult _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN Tcl_ObjType *	Tcl_GetObjType _ANSI_ARGS_((CONST char *typeName));
EXTERN Tcl_Obj *	Tcl_GetObjVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags));
EXTERN int		Tcl_GetOpenFile _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string, int write, int checkUsage,
			    ClientData *filePtr));
EXTERN Tcl_PathType	Tcl_GetPathType _ANSI_ARGS_((CONST char *path));
EXTERN int		Tcl_Gets _ANSI_ARGS_((Tcl_Channel chan,
        		    Tcl_DString *dsPtr));
EXTERN int		Tcl_GetsObj _ANSI_ARGS_((Tcl_Channel chan,
        		    Tcl_Obj *objPtr));
EXTERN int		Tcl_GetServiceMode _ANSI_ARGS_((void));
EXTERN Tcl_Interp *	Tcl_GetSlave _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *slaveName));
EXTERN char **		Tcl_GetStaticFile _ANSI_ARGS_((CONST char *rsrcName));
EXTERN Tcl_Channel	Tcl_GetStdChannel _ANSI_ARGS_((int type));
EXTERN char *		Tcl_GetStringFromObj _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int *lengthPtr));
EXTERN char *		Tcl_GetStringResult _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN char *		Tcl_GetString _ANSI_ARGS_((Tcl_Obj *objPtr));
#define Tcl_GetString(objPtr) Tcl_GetStringFromObj(objPtr, (int *) NULL)
EXTERN char *		Tcl_GetVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags));
EXTERN char *		Tcl_GetVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags));
EXTERN int		Tcl_GlobalEval _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *command));
EXTERN int		Tcl_GlobalEvalObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr));
EXTERN char *		Tcl_HashStats _ANSI_ARGS_((Tcl_HashTable *tablePtr));
EXTERN int		Tcl_HideCommand _ANSI_ARGS_((Tcl_Interp *interp,
		            CONST char *cmdName, CONST char *hiddenCmdToken));
EXTERN int		Tcl_Init _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN int		Tcl_InitStandAlone _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN void		Tcl_InitHashTable _ANSI_ARGS_((Tcl_HashTable *tablePtr,
			    int keyType));
EXTERN int		Tcl_InputBlocked _ANSI_ARGS_((Tcl_Channel chan));
EXTERN int		Tcl_InputBuffered _ANSI_ARGS_((Tcl_Channel chan));
EXTERN int		Tcl_InterpDeleted _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN int		Tcl_IsSafe _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN void		Tcl_InvalidateStringRep _ANSI_ARGS_((
			    Tcl_Obj *objPtr));
EXTERN char *		Tcl_JoinPath _ANSI_ARGS_((int argc, CONST char * CONST argv[],
			    Tcl_DString *resultPtr));
EXTERN int		Tcl_LinkVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, char *addr, int type));
EXTERN int		Tcl_ListObjAppendList _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *listPtr, 
			    Tcl_Obj *elemListPtr));
EXTERN int		Tcl_ListObjAppendElement _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *listPtr,
			    Tcl_Obj *objPtr));
EXTERN int		Tcl_ListObjGetElements _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *listPtr,
			    int *objcPtr, Tcl_Obj ***objvPtr));
EXTERN int		Tcl_ListObjIndex _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *listPtr, int index, 
			    Tcl_Obj **objPtrPtr));
EXTERN int		Tcl_ListObjLength _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *listPtr, int *intPtr));
EXTERN int		Tcl_ListObjReplace _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *listPtr, int first, int count,
			    int objc, Tcl_Obj *CONST objv[]));
EXTERN void		Tcl_Main _ANSI_ARGS_((int argc, char *argv[],
			    Tcl_AppInitProc *appInitProc));
EXTERN void		Tcl_MainLoop _ANSI_ARGS_((void));
EXTERN Tcl_Channel	Tcl_MakeFileChannel _ANSI_ARGS_((ClientData handle,
			    int mode));
EXTERN int		Tcl_MakeSafe _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN Tcl_Channel	Tcl_MakeTcpClientChannel _ANSI_ARGS_((
    			    ClientData tcpSocket));
EXTERN char *		Tcl_Merge _ANSI_ARGS_((int argc, char *argv[]));
EXTERN Tcl_HashEntry *	Tcl_NextHashEntry _ANSI_ARGS_((
			    Tcl_HashSearch *searchPtr));
EXTERN void		Tcl_NotifyChannel _ANSI_ARGS_((Tcl_Channel channel,
			    int mask));
EXTERN int		Tcl_NumUtfChars _ANSI_ARGS_((CONST char *src,
			    int len));
EXTERN Tcl_Obj *	Tcl_ObjGetVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *part1Ptr, Tcl_Obj *part2Ptr,
			    int flags));
EXTERN Tcl_Obj *	Tcl_ObjSetVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *part1Ptr, Tcl_Obj *part2Ptr,
			    Tcl_Obj *newValuePtr, int flags));
EXTERN Tcl_Channel	Tcl_OpenCommandChannel _ANSI_ARGS_((
    			    Tcl_Interp *interp, int argc, char **argv,
			    int flags));
EXTERN Tcl_Channel	Tcl_OpenFileChannel _ANSI_ARGS_((Tcl_Interp *interp,
        		    CONST char *fileName, CONST char *modeString,
                            int permissions));
EXTERN Tcl_Channel	Tcl_OpenTcpClient _ANSI_ARGS_((Tcl_Interp *interp,
			    int port, char *address, char *myaddr,
		            int myport, int async));
EXTERN Tcl_Channel	Tcl_OpenTcpServer _ANSI_ARGS_((Tcl_Interp *interp,
		            int port, char *host,
        		    Tcl_TcpAcceptProc *acceptProc,
			    ClientData callbackData));
EXTERN char *		Tcl_ParseVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, char **termPtr));
EXTERN int		Tcl_PkgProvide _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, CONST char *version));
EXTERN int		Tcl_PkgProvide2 _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, CONST char *version,
			    ClientData clientData));
EXTERN char *		Tcl_PkgRequire _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, CONST char *version, int exact));
EXTERN char *		Tcl_PkgRequire2 _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, CONST char *version, int exact,
			    ClientData *clientDataPtr));
EXTERN char *		Tcl_PosixError _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN void		Tcl_Preserve _ANSI_ARGS_((ClientData data));
EXTERN void		Tcl_PrintDouble _ANSI_ARGS_((Tcl_Interp *interp,
			    double value, char *dst));
EXTERN int		Tcl_PutEnv _ANSI_ARGS_((CONST char *string));
EXTERN void		Tcl_QueueEvent _ANSI_ARGS_((Tcl_Event *evPtr,
			    Tcl_QueuePosition position));
EXTERN int		Tcl_Read _ANSI_ARGS_((Tcl_Channel chan,
	        	    char *bufPtr, int toRead));
EXTERN void		Tcl_ReapDetachedProcs _ANSI_ARGS_((void));
EXTERN int		Tcl_RecordAndEval _ANSI_ARGS_((Tcl_Interp *interp,
			    char *cmd, int flags));
EXTERN int		Tcl_RecordAndEvalObj _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *cmdPtr, int flags));
EXTERN Tcl_RegExp	Tcl_RegExpCompile _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string));
EXTERN int		Tcl_RegExpExec _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_RegExp regexp, CONST char *string,
			    CONST char *start));
EXTERN int		Tcl_RegExpMatch _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, char *pattern));
EXTERN void		Tcl_RegExpRange _ANSI_ARGS_((Tcl_RegExp regexp,
			    int index, char **startPtr, char **endPtr));
EXTERN void		Tcl_RegisterChannel _ANSI_ARGS_((Tcl_Interp *interp,
	        	    Tcl_Channel chan));
EXTERN void		Tcl_RegisterObjType _ANSI_ARGS_((
			    Tcl_ObjType *typePtr));
EXTERN void		Tcl_Release _ANSI_ARGS_((ClientData clientData));
EXTERN void		Tcl_ResetResult _ANSI_ARGS_((Tcl_Interp *interp));
#define Tcl_Return Tcl_SetResult
EXTERN int		Tcl_ScanCountedElement _ANSI_ARGS_((CONST char *string,
			    int length, int *flagPtr));
EXTERN int		Tcl_ScanElement _ANSI_ARGS_((CONST char *string,
			    int *flagPtr));
#define Tcl_ScanElement(string, flagPtr) Tcl_ScanCountedElement(string, -1, flagPtr)
EXTERN int		Tcl_Seek _ANSI_ARGS_((Tcl_Channel chan,
        		    int offset, int mode));
EXTERN int		Tcl_ServiceAll _ANSI_ARGS_((void));
EXTERN int		Tcl_ServiceEvent _ANSI_ARGS_((int flags));
EXTERN void		Tcl_SetAssocData _ANSI_ARGS_((Tcl_Interp *interp,
                            CONST char *name, Tcl_InterpDeleteProc *proc,
                            ClientData clientData));
EXTERN void		Tcl_SetBooleanObj _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    int boolValue));
EXTERN unsigned char *	Tcl_SetByteArrayLength _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    int length));
EXTERN void		Tcl_SetByteArrayObj _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST unsigned char *bytes, int length));
EXTERN void		Tcl_SetChannelBufferSize _ANSI_ARGS_((
			    Tcl_Channel chan, int sz));
EXTERN int		Tcl_SetChannelOption _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Channel chan,
	        	    CONST char *optionName, CONST char *newValue));
EXTERN int		Tcl_SetCommandInfo _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *cmdName, Tcl_CmdInfo *infoPtr));
EXTERN void		Tcl_SetDoubleObj _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    double doubleValue));
EXTERN void		Tcl_SetErrno _ANSI_ARGS_((int err));
EXTERN void		Tcl_SetErrorCode _ANSI_ARGS_(
    			    TCL_VARARGS(Tcl_Interp *,arg1));
EXTERN void		Tcl_SetIntObj _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    int intValue));
EXTERN void		Tcl_SetListObj _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    int objc, Tcl_Obj *CONST objv[]));
EXTERN void		Tcl_SetLongObj _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    long longValue));
EXTERN void		Tcl_SetMaxBlockTime _ANSI_ARGS_((Tcl_Time *timePtr));
EXTERN void		Tcl_SetObjErrorCode _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *errorObjPtr));
EXTERN void		Tcl_SetObjLength _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int length));
EXTERN void		Tcl_SetObjResult _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *resultObjPtr));
EXTERN Tcl_Obj *	Tcl_SetObjVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, Tcl_Obj *newValuePtr,
			    int flags));
EXTERN void		Tcl_SetPanicProc _ANSI_ARGS_((void (*proc)
			    _ANSI_ARGS_(TCL_VARARGS(char *, format))));
EXTERN int		Tcl_SetRecursionLimit _ANSI_ARGS_((Tcl_Interp *interp,
			    int depth));
EXTERN void		Tcl_SetResult _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, Tcl_FreeProc *freeProc));
EXTERN int		Tcl_SetServiceMode _ANSI_ARGS_((int mode));
EXTERN void		Tcl_SetStdChannel _ANSI_ARGS_((Tcl_Channel channel,
			    int type));
EXTERN void		Tcl_SetStringObj _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    CONST char *bytes, int length));
EXTERN void		Tcl_SetTimer _ANSI_ARGS_((Tcl_Time *timePtr));
EXTERN char *		Tcl_SetVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, char *newValue, int flags));
EXTERN char *		Tcl_SetVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, char *newValue,
			    int flags));
EXTERN char *		Tcl_SignalId _ANSI_ARGS_((int sig));
EXTERN char *		Tcl_SignalMsg _ANSI_ARGS_((int sig));
EXTERN void		Tcl_Sleep _ANSI_ARGS_((int ms));
EXTERN void		Tcl_SourceRCFile _ANSI_ARGS_((Tcl_Interp *interp));
EXTERN int		Tcl_SplitList _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *list, int *argcPtr, char **argvPtr[]));
EXTERN void		Tcl_SplitPath _ANSI_ARGS_((CONST char *path,
			    int *argcPtr, CONST char **argvPtr[]));
EXTERN void		Tcl_StaticPackage _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *pkgName, Tcl_PackageInitProc *initProc,
			    Tcl_PackageInitProc *safeInitProc));
EXTERN int		Tcl_StringMatch _ANSI_ARGS_((CONST char *string,
			    CONST char *pattern));
EXTERN int		Tcl_Tell _ANSI_ARGS_((Tcl_Channel chan));
#define Tcl_TildeSubst Tcl_TranslateFileName
EXTERN int		Tcl_TraceVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags, Tcl_VarTraceProc *proc,
			    ClientData clientData));
EXTERN int		Tcl_TraceVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags,
			    Tcl_VarTraceProc *proc, ClientData clientData));
EXTERN char *		Tcl_TranslateFileName _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, Tcl_DString *bufferPtr));
EXTERN int		Tcl_Ungets _ANSI_ARGS_((Tcl_Channel chan, CONST char *str,
			    int len, int atHead));
EXTERN void		Tcl_UnlinkVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName));
EXTERN int		Tcl_UnregisterChannel _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Channel chan));
EXTERN int		Tcl_UnsetVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags));
EXTERN int		Tcl_UnsetVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags));
EXTERN void		Tcl_UntraceVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags, Tcl_VarTraceProc *proc,
			    ClientData clientData));
EXTERN void		Tcl_UntraceVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags,
			    Tcl_VarTraceProc *proc, ClientData clientData));
EXTERN void		Tcl_UpdateLinkedVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName));
EXTERN int		Tcl_UpVar _ANSI_ARGS_((Tcl_Interp *interp,
			    char *frameName, char *varName,
			    char *localName, int flags));
EXTERN int		Tcl_UpVar2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *frameName, char *part1, char *part2,
			    char *localName, int flags));
EXTERN char *		Tcl_UtfAtIndex _ANSI_ARGS_((CONST char *src,
			    int index));
EXTERN int		Tcl_UtfCharComplete _ANSI_ARGS_((CONST char *src,
			    int len));
EXTERN int		Tcl_UtfBackslash _ANSI_ARGS_((CONST char *src,
			    int *readPtr, char *dst));
EXTERN char *		Tcl_UtfFindFirst _ANSI_ARGS_((CONST char *src,
			    int ch));
EXTERN char *		Tcl_UtfFindLast _ANSI_ARGS_((CONST char *src,
			    int ch));
EXTERN char *		Tcl_UtfNext _ANSI_ARGS_((CONST char *src));
EXTERN char *		Tcl_UtfPrev _ANSI_ARGS_((CONST char *src,
			    CONST char *start));
EXTERN int		Tcl_UtfToExternal _ANSI_ARGS_((Tcl_Interp *interp,
			    VOID *encoding, CONST char *src, int srcLen,
			    int flags, VOID *statePtr, char *dst,
			    int dstLen, int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));
EXTERN char *		Tcl_UtfToExternalDString _ANSI_ARGS_((
			    VOID *encoding, CONST char *src,
			    int srcLen, Tcl_DString *dsPtr));
EXTERN int		Tcl_UtfToLower _ANSI_ARGS_((char *src));
EXTERN int		Tcl_UtfToTitle _ANSI_ARGS_((char *src));
EXTERN int		Tcl_UtfToUpper _ANSI_ARGS_((char *src));
EXTERN int		Tcl_VarEval _ANSI_ARGS_(
    			    TCL_VARARGS(Tcl_Interp *,interp));
EXTERN ClientData	Tcl_VarTraceInfo _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags,
			    Tcl_VarTraceProc *procPtr,
			    ClientData prevClientData));
EXTERN ClientData	Tcl_VarTraceInfo2 _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags,
			    Tcl_VarTraceProc *procPtr,
			    ClientData prevClientData));
EXTERN int		Tcl_WaitForEvent _ANSI_ARGS_((Tcl_Time *timePtr));
EXTERN Tcl_Pid		Tcl_WaitPid _ANSI_ARGS_((Tcl_Pid pid, int *statPtr, 
			    int options));
EXTERN int		Tcl_Write _ANSI_ARGS_((Tcl_Channel chan,
			    CONST char *src, int srcLen));
EXTERN void		Tcl_WrongNumArgs _ANSI_ARGS_((Tcl_Interp *interp,
			    int objc, Tcl_Obj *CONST objv[], CONST char *message));

/* Andreas Kupries <a.kupries@westend.com>, 05/31/1997.
 * Support of Tcl-Trf.
 */
EXTERN Tcl_Channel      Tcl_ReplaceChannel _ANSI_ARGS_ ((Tcl_Interp* interp,
                          Tcl_ChannelType* typePtr, ClientData instanceData,
                          int mask, Tcl_Channel prevChan));
EXTERN void             Tcl_UndoReplaceChannel _ANSI_ARGS_ ((Tcl_Interp* interp,
                          Tcl_Channel chan));

/*
 * Callback table definitions
 *
 *     Designed by:
 *           Paul Duffin <pduffin@hursley.ibm.com>
 *           Jan Nijtmans <Jan.Nijtmans@wxs.nl>
 *     with a lot of invaluable support from:
 *	     Jean-Claude Wippler <jcw@equi4.com>
 */

#define TCL_STUB_MAGIC 0xFCA1BACF

struct stat;

typedef struct Tcl_StubTable {
    int magic; 		/* should be TCL_STUB_MAGIC */

    struct Tcl_StubTableHooks *hooks;	/* hooks to internal functions */

    /* The two most important functions, for the
     * registering and addressing of stub pointers */
    int		(*pkgProvide2) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, CONST char *version,
			    ClientData stubTable));
    char *	(*pkgRequire2) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, CONST char *version,
			    int exact, ClientData *stubTable));

    /* memory management functions */

    void	(*panic) _ANSI_ARGS_(TCL_VARARGS(CONST char *,format));
    char *	(*alloc) _ANSI_ARGS_((unsigned int size));
    void	(*free) _ANSI_ARGS_((char *ptr));
    char *	(*realloc) _ANSI_ARGS_((char *ptr, unsigned int size));
    char *	(*dbCkalloc) _ANSI_ARGS_((unsigned int size,
			    CONST char *file, int line));
    int		(*dbCkfree) _ANSI_ARGS_((char *ptr,
			    CONST char *file, int line));
    char *	(*dbCkrealloc) _ANSI_ARGS_((char *ptr,
			    unsigned int size, CONST char *file, int line));

    /* notifier functions */

#if (!defined(__WIN32__) && !defined(MAC_TCL))
    void	(*createFileHandler) _ANSI_ARGS_((
    			    int fd, int mask, Tcl_FileProc *proc,
			    ClientData clientData));
    void	(*deleteFileHandler) _ANSI_ARGS_((int fd));
#endif
    void	(*setTimer) _ANSI_ARGS_((Tcl_Time *timePtr));
    void	(*sleep) _ANSI_ARGS_((int ms));
    int		(*waitForEvent) _ANSI_ARGS_((Tcl_Time *timePtr));

    /* object handling functions */

    int		(*appendAllObjTypes) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *objPtr));
    void	(*appendStringsToObj) _ANSI_ARGS_(
			    TCL_VARARGS(Tcl_Obj *,interp));
    void	(*appendToObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    char *bytes, int length));
    Tcl_Obj *	(*concatObj) _ANSI_ARGS_((int objc,
			    Tcl_Obj *CONST objv[]));
    int		(*convertToType) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, Tcl_ObjType *typePtr));
    void	(*dbDecrRefCount) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *file, int line));
    void	(*dbIncrRefCount) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *file, int line));
    int		(*dbIsShared) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *file, int line));
    Tcl_Obj *	(*dbNewBooleanObj) _ANSI_ARGS_((int boolValue,
                            CONST char *file, int line));
    Tcl_Obj *	(*dbNewByteArrayObj) _ANSI_ARGS_((CONST unsigned char *bytes,
			    int length, CONST char *file, int line));
    Tcl_Obj *	(*dbNewDoubleObj) _ANSI_ARGS_((double doubleValue,
                            CONST char *file, int line));
    Tcl_Obj *	(*dbNewListObj) _ANSI_ARGS_((int objc,
			    Tcl_Obj *CONST objv[], CONST char *file, int line));
    Tcl_Obj *	(*dbNewLongObj) _ANSI_ARGS_((long longValue,
                            CONST char *file, int line));
    Tcl_Obj *	(*dbNewObj) _ANSI_ARGS_((CONST char *file, int line));
    Tcl_Obj *	(*dbNewStringObj) _ANSI_ARGS_((CONST char *bytes,
			    int length, CONST char *file, int line));
    Tcl_Obj *	(*duplicateObj) _ANSI_ARGS_((Tcl_Obj *objPtr));
    void	(*freeObj) _ANSI_ARGS_((Tcl_Obj *objPtr));
    int		(*getBoolean) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string, int *boolPtr));
    int		(*getBooleanFromObj) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *objPtr,
			    int *boolPtr));
    unsigned char * (*getByteArrayFromObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int *lengthPtr));
    int		(*getDouble) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, double *doublePtr));
    int		(*getDoubleFromObj) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *objPtr,
			    double *doublePtr));
    int		(*getIndexFromObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, char **tablePtr,
			    CONST char *msg, int flags, int *indexPtr));
    int		(*getInt) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string, int *intPtr));
    int		(*getIntFromObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, int *intPtr));
    int		(*getLongFromObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, long *longPtr));
    Tcl_ObjType*(*getObjType) _ANSI_ARGS_((CONST char *typeName));
    char *	(*getStringFromObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int *lengthPtr));
    void	(*invalidateStringRep) _ANSI_ARGS_((
			    Tcl_Obj *objPtr));
    int		(*listObjAppendList) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *listPtr, 
			    Tcl_Obj *elemListPtr));
    int		(*listObjAppendElement) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *listPtr,
			    Tcl_Obj *objPtr));
    int		(*listObjGetElements) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Obj *listPtr,
			    int *objcPtr, Tcl_Obj ***objvPtr));
    int		(*listObjIndex) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *listPtr, int index, 
			    Tcl_Obj **objPtrPtr));
    int		(*listObjLength) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *listPtr, int *intPtr));
    int		(*listObjReplace) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *listPtr, int first, int count,
			    int objc, Tcl_Obj *CONST objv[]));
    Tcl_Obj *	(*newBooleanObj) _ANSI_ARGS_((int boolValue));
    Tcl_Obj *	(*newByteArrayObj) _ANSI_ARGS_((CONST unsigned char *bytes,
			    int length));
    Tcl_Obj *	(*newDoubleObj) _ANSI_ARGS_((double doubleValue));
    Tcl_Obj *	(*newIntObj) _ANSI_ARGS_((int intValue));
    Tcl_Obj *	(*newListObj) _ANSI_ARGS_((int objc,
			    Tcl_Obj *CONST objv[]));
    Tcl_Obj *	(*newLongObj) _ANSI_ARGS_((long longValue));
    Tcl_Obj *	(*newObj) _ANSI_ARGS_((void));
    Tcl_Obj *	(*newStringObj) _ANSI_ARGS_((
			    CONST char *string, int length));
    void	(*setBooleanObj) _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    int boolValue));
    unsigned char * (*setByteArrayLength) _ANSI_ARGS_((Tcl_Obj *objPtr, 
			    int length));
    void	(*setByteArrayObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST unsigned char *bytes, int length));
    void	(*setDoubleObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    double doubleValue));
    void	(*setIntObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int intValue));
    void	(*setListObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int objc, Tcl_Obj *CONST objv[]));
    void	(*setLongObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    long longValue));
    void	(*setObjLength) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    int length));
    void	(*setStringObj) _ANSI_ARGS_((Tcl_Obj *objPtr,
			    CONST char *bytes, int length));

    /* other functions */
    void	(*addErrorInfo) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *message));
    void	(*addObjErrorInfo) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *message, int length));
    void	(*allowExceptions) _ANSI_ARGS_((Tcl_Interp *interp));
    void	(*appendElement) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string));
    void	(*appendResult) _ANSI_ARGS_(
			    TCL_VARARGS(Tcl_Interp *,interp));
    Tcl_AsyncHandler (*asyncCreate) _ANSI_ARGS_((Tcl_AsyncProc *proc,
			    ClientData clientData));
    void	(*asyncDelete) _ANSI_ARGS_((Tcl_AsyncHandler async));
    int		(*asyncInvoke) _ANSI_ARGS_((Tcl_Interp *interp,
			    int code));
    void	(*asyncMark) _ANSI_ARGS_((Tcl_AsyncHandler async));
    int		(*asyncReady) _ANSI_ARGS_((void));
    void	(*backgroundError) _ANSI_ARGS_((Tcl_Interp *interp));
    char	(*backslash) _ANSI_ARGS_((CONST char *src,
			    int *readPtr));
    int		(*badChannelOption) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *optionName, CONST char *optionList));
    void	(*callWhenDeleted) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_InterpDeleteProc *proc,
			    ClientData clientData));
    void	(*cancelIdleCall) _ANSI_ARGS_((Tcl_IdleProc *idleProc,
			    ClientData clientData));
    int		(*close) _ANSI_ARGS_((Tcl_Interp *interp,
        		    Tcl_Channel chan));
    int		(*commandComplete) _ANSI_ARGS_((char *cmd));
    char *	(*concat) _ANSI_ARGS_((int argc, char **argv));
    int		(*convertElement) _ANSI_ARGS_((CONST char *src, char *dst,
			    int flags));
    int		(*convertCountedElement) _ANSI_ARGS_((CONST char *src,
			    int length, char *dst, int flags));
    int		(*createAlias) _ANSI_ARGS_((Tcl_Interp *slave,
			    char *slaveCmd, Tcl_Interp *target,
        		    char *targetCmd, int argc, char **argv));
    int		(*createAliasObj) _ANSI_ARGS_((Tcl_Interp *slave,
			    char *slaveCmd, Tcl_Interp *target,
        		    char *targetCmd, int objc,
		            Tcl_Obj *CONST objv[]));
    Tcl_Channel	(*createChannel) _ANSI_ARGS_((
    			    Tcl_ChannelType *typePtr, char *chanName,
                            ClientData instanceData, int mask));
    void	(*createChannelHandler) _ANSI_ARGS_((
			    Tcl_Channel chan, int mask,
                            Tcl_ChannelProc *proc, ClientData clientData));
    void	(*createCloseHandler) _ANSI_ARGS_((
			    Tcl_Channel chan, Tcl_CloseProc *proc,
                            ClientData clientData));
    Tcl_Command	(*createCommand) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *cmdName, Tcl_CmdProc *proc,
			    ClientData clientData,
			    Tcl_CmdDeleteProc *deleteProc));
    void	(*createEventSource) _ANSI_ARGS_((
			    Tcl_EventSetupProc *setupProc,
			    Tcl_EventCheckProc *checkProc,
			    ClientData clientData));
    void	(*createExitHandler) _ANSI_ARGS_((Tcl_ExitProc *proc,
			    ClientData clientData));
    Tcl_Interp *(*createInterp) _ANSI_ARGS_((void));
    void	(*createMathFunc) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *name, int numArgs, Tcl_ValueType *argTypes,
			    Tcl_MathProc *proc, ClientData clientData));
    Tcl_Command	(*createObjCommand) _ANSI_ARGS_((
			    Tcl_Interp *interp, CONST char *cmdName,
			    Tcl_ObjCmdProc *proc, ClientData clientData,
			    Tcl_CmdDeleteProc *deleteProc));
    Tcl_Interp * (*createSlave) _ANSI_ARGS_((Tcl_Interp *interp,
		            char *slaveName, int isSafe));
    Tcl_TimerToken (*createTimerHandler) _ANSI_ARGS_((int milliseconds,
			    Tcl_TimerProc *proc, ClientData clientData));
    Tcl_Trace	(*createTrace) _ANSI_ARGS_((Tcl_Interp *interp,
			    int level, Tcl_CmdTraceProc *proc,
			    ClientData clientData));
    void	(*deleteAssocData) _ANSI_ARGS_((Tcl_Interp *interp,
                            CONST char *name));
    void	(*deleteChannelHandler) _ANSI_ARGS_((
    			    Tcl_Channel chan, Tcl_ChannelProc *proc,
                            ClientData clientData));
    void	(*deleteCloseHandler) _ANSI_ARGS_((
			    Tcl_Channel chan, Tcl_CloseProc *proc,
                            ClientData clientData));
    int		(*deleteCommand) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *cmdName));
    void	(*deleteCommandFromToken) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Command command));
    void	(*deleteEvents) _ANSI_ARGS_((
			    Tcl_EventDeleteProc *proc,
                            ClientData clientData));
    void	(*deleteEventSource) _ANSI_ARGS_((
			    Tcl_EventSetupProc *setupProc,
			    Tcl_EventCheckProc *checkProc,
			    ClientData clientData));
    void	(*deleteExitHandler) _ANSI_ARGS_((Tcl_ExitProc *proc,
			    ClientData clientData));
    void	(*deleteHashEntry) _ANSI_ARGS_((
			    Tcl_HashEntry *entryPtr));
    void	(*deleteHashTable) _ANSI_ARGS_((
			    Tcl_HashTable *tablePtr));
    void	(*deleteInterp) _ANSI_ARGS_((Tcl_Interp *interp));
    void	(*detachPids) _ANSI_ARGS_((int numPids, Tcl_Pid *pidPtr));
    void	(*deleteTimerHandler) _ANSI_ARGS_((
			    Tcl_TimerToken token));
    void	(*deleteTrace) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Trace trace));
    void	(*dontCallWhenDeleted) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_InterpDeleteProc *proc,
			    ClientData clientData));
    int		(*doOneEvent) _ANSI_ARGS_((int flags));
    void	(*doWhenIdle) _ANSI_ARGS_((Tcl_IdleProc *proc,
			    ClientData clientData));
    char *	(*dStringAppend) _ANSI_ARGS_((Tcl_DString *dsPtr,
			    CONST char *string, int length));
    char *	(*dStringAppendElement) _ANSI_ARGS_((
			    Tcl_DString *dsPtr, CONST char *string));
    void	(*dStringEndSublist) _ANSI_ARGS_((Tcl_DString *dsPtr));
    void	(*dStringFree) _ANSI_ARGS_((Tcl_DString *dsPtr));
    void	(*dStringGetResult) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_DString *dsPtr));
    void	(*dStringInit) _ANSI_ARGS_((Tcl_DString *dsPtr));
    void	(*dStringResult) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_DString *dsPtr));
    void	(*dStringSetLength) _ANSI_ARGS_((Tcl_DString *dsPtr,
			    int length));
    void	(*dStringStartSublist) _ANSI_ARGS_((
			    Tcl_DString *dsPtr));
    int		(*eof) _ANSI_ARGS_((Tcl_Channel chan));
    char *	(*errnoId) _ANSI_ARGS_((void));
    char *	(*errnoMsg) _ANSI_ARGS_((int));
    int		(*eval) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *command));
    int		(*evalFile) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *fileName));
    int		(*evalObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr));
    void	(*eventuallyFree) _ANSI_ARGS_((ClientData clientData,
			    Tcl_FreeProc *freeProc));
    void	(*exit) _ANSI_ARGS_((int status));
    int		(*exposeCommand) _ANSI_ARGS_((Tcl_Interp *interp,
        		    char *hiddenCmdToken, char *cmdName));
    int		(*exprBoolean) _ANSI_ARGS_((Tcl_Interp *interp, char *,
			    int *ptr));
    int		(*exprBooleanObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, int *ptr));
    int		(*exprDouble) _ANSI_ARGS_((Tcl_Interp *, char *,
			    double *));
    int		(*exprDoubleObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, double *ptr));
    int		(*exprLong) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, long *ptr));
    int		(*exprLongObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, long *ptr));
    int		(*exprObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, Tcl_Obj **resultPtrPtr));
    int		(*exprString) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string));
    void	(*finalize) _ANSI_ARGS_((void));
    void	(*findExecutable) _ANSI_ARGS_((CONST char *argv0));
    Tcl_HashEntry * (*firstHashEntry) _ANSI_ARGS_((
			    Tcl_HashTable *tablePtr,
			    Tcl_HashSearch *searchPtr));
    int		(*flush) _ANSI_ARGS_((Tcl_Channel chan));
    void	(*freeResult) _ANSI_ARGS_((Tcl_Interp *interp));
    int		(*getAlias) _ANSI_ARGS_((Tcl_Interp *interp,
       			    char *slaveCmd, Tcl_Interp **targetInterpPtr,
                            char **targetCmdPtr, int *argcPtr,
			    char ***argvPtr));
    int		(*getAliasObj) _ANSI_ARGS_((Tcl_Interp *interp,
       			    char *slaveCmd, Tcl_Interp **targetInterpPtr,
                            char **targetCmdPtr, int *objcPtr,
			    Tcl_Obj ***objv));
    ClientData	(*getAssocData) _ANSI_ARGS_((Tcl_Interp *interp,
                            CONST char *name, Tcl_InterpDeleteProc **procPtr));
    Tcl_Channel	(*getChannel) _ANSI_ARGS_((Tcl_Interp *interp,
	        	    CONST char *chanName, int *modePtr));
    int		(*getChannelBufferSize) _ANSI_ARGS_((
    			    Tcl_Channel chan));
    int		(*getChannelHandle) _ANSI_ARGS_((Tcl_Channel chan,
			    int direction, ClientData *handlePtr));
    ClientData	(*getChannelInstanceData) _ANSI_ARGS_((
			    Tcl_Channel chan));
    int		(*getChannelMode) _ANSI_ARGS_((Tcl_Channel chan));
    char *	(*getChannelName) _ANSI_ARGS_((Tcl_Channel chan));
    int		(*getChannelOption) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Channel chan, char *optionName,
			    Tcl_DString *dsPtr));
    Tcl_ChannelType * (*getChannelType) _ANSI_ARGS_((Tcl_Channel chan));
    int		(*getCommandInfo) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *cmdName, Tcl_CmdInfo *infoPtr));
    char *	(*getCommandName) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Command command));
    int		(*getErrno) _ANSI_ARGS_((void));
    char *	(*getHostName) _ANSI_ARGS_((void));
    int		(*getInterpPath) _ANSI_ARGS_((Tcl_Interp *askInterp,
			    Tcl_Interp *slaveInterp));
    Tcl_Interp * (*getMaster) _ANSI_ARGS_((Tcl_Interp *interp));
    CONST char * (*getNameOfExecutable) _ANSI_ARGS_((void));
    Tcl_Obj *	(*getObjResult) _ANSI_ARGS_((Tcl_Interp *interp));
#if (!defined(__WIN32__) && !defined(MAC_TCL))
    int		(*getOpenFile) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *string, int write, int checkUsage,
			    ClientData *filePtr));
#endif
    Tcl_PathType (*getPathType) _ANSI_ARGS_((CONST char *path));
    int		(*gets) _ANSI_ARGS_((Tcl_Channel chan,
        		    Tcl_DString *dsPtr));
    int		(*getsObj) _ANSI_ARGS_((Tcl_Channel chan,
        		    Tcl_Obj *objPtr));
    int		(*getServiceMode) _ANSI_ARGS_((void));
    Tcl_Interp * (*getSlave) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *slaveName));
    Tcl_Channel	(*getStdChannel) _ANSI_ARGS_((int type));
    char *	(*getStringResult) _ANSI_ARGS_((Tcl_Interp *interp));
    char *	(*getVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags));
    char *	(*getVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags));
    int		(*globalEval) _ANSI_ARGS_((Tcl_Interp *, CONST char *));
    int		(*globalEvalObj) _ANSI_ARGS_((Tcl_Interp *, Tcl_Obj *));
    int		(*hideCommand) _ANSI_ARGS_((Tcl_Interp *interp,
		            CONST char *cmdName, CONST char *hiddenCmdToken));
    int		(*init) _ANSI_ARGS_((Tcl_Interp *interp));
    void	(*initHashTable) _ANSI_ARGS_((Tcl_HashTable *tablePtr,
			    int keyType));
    int		(*inputBlocked) _ANSI_ARGS_((Tcl_Channel chan));
    int		(*inputBuffered) _ANSI_ARGS_((Tcl_Channel chan));
    int		(*interpDeleted) _ANSI_ARGS_((Tcl_Interp *interp));
    int		(*isSafe) _ANSI_ARGS_((Tcl_Interp *interp));
    char *	(*joinPath) _ANSI_ARGS_((int argc, CONST char * CONST argv[],
			    Tcl_DString *resultPtr));
    int		(*linkVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, char *addr, int type));
    void	(*mainLoop) _ANSI_ARGS_((void));
#ifndef MAC_TCL
    Tcl_Channel	(*makeFileChannel) _ANSI_ARGS_((ClientData handle,
			    int mode));
#endif
    int		(*makeSafe) _ANSI_ARGS_((Tcl_Interp *interp));
    Tcl_Channel	(*makeTcpClientChannel) _ANSI_ARGS_((
    			    ClientData tcpSocket));
    char *	(*merge) _ANSI_ARGS_((int argc, char *argv[]));
    Tcl_HashEntry * (*nextHashEntry) _ANSI_ARGS_((
			    Tcl_HashSearch *searchPtr));
    void	(*notifyChannel) _ANSI_ARGS_((Tcl_Channel channel,
			    int mask));
    Tcl_Obj *	(*objGetVar2) _ANSI_ARGS_((Tcl_Interp *, Tcl_Obj *,
			    Tcl_Obj *, int));
    Tcl_Obj *	(*objSetVar2) _ANSI_ARGS_((Tcl_Interp *, Tcl_Obj *,
			    Tcl_Obj *, Tcl_Obj *, int));
    Tcl_Channel	(*openCommandChannel) _ANSI_ARGS_((
    			    Tcl_Interp *interp, int argc, char **argv,
			    int flags));
    Tcl_Channel	(*openFileChannel) _ANSI_ARGS_((Tcl_Interp *interp,
        		    CONST char *fileName, CONST char *modeString,
                            int permissions));
    Tcl_Channel	(*openTcpClient) _ANSI_ARGS_((Tcl_Interp *interp,
			    int port, char *address, char *myaddr,
		            int myport, int async));
    Tcl_Channel	(*openTcpServer) _ANSI_ARGS_((Tcl_Interp *interp,
		            int port, char *host,
        		    Tcl_TcpAcceptProc *acceptProc,
			    ClientData callbackData));
    void	(*preserve) _ANSI_ARGS_((ClientData data));
    void	(*printDouble) _ANSI_ARGS_((Tcl_Interp *interp,
			    double value, char *dst));
    int		(*putEnv) _ANSI_ARGS_((CONST char *string));
    char *	(*posixError) _ANSI_ARGS_((Tcl_Interp *interp));
    void	(*queueEvent) _ANSI_ARGS_((Tcl_Event *evPtr,
			    Tcl_QueuePosition position));
    int		(*read) _ANSI_ARGS_((Tcl_Channel chan,
	        	    char *bufPtr, int toRead));
    void	(*reapDetachedProcs) _ANSI_ARGS_((void));
    int		(*recordAndEval) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *cmd, int flags));
    int		(*recordAndEvalObj) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *cmdPtr, int flags));
    void	(*registerChannel) _ANSI_ARGS_((Tcl_Interp *interp,
	        	    Tcl_Channel chan));
    void	(*registerObjType) _ANSI_ARGS_((
			    Tcl_ObjType *typePtr));
    Tcl_RegExp	(*regExpCompile) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string));
    int		(*regExpExec) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_RegExp regexp, CONST char *string,
			    CONST char *start));
    int		(*regExpMatch) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, char *pattern));
    void	(*regExpRange) _ANSI_ARGS_((Tcl_RegExp regexp,
			    int index, char **startPtr, char **endPtr));
    void	(*release) _ANSI_ARGS_((ClientData clientData));
    void	(*resetResult) _ANSI_ARGS_((Tcl_Interp *interp));
    int		(*scanElement) _ANSI_ARGS_((CONST char *string, int *flagPtr));
    int		(*scanCountedElement) _ANSI_ARGS_((CONST char *string,
			    int length, int *flagPtr));
    int		(*seek) _ANSI_ARGS_((Tcl_Channel chan,
        		    int offset, int mode));
    int		(*serviceAll) _ANSI_ARGS_((void));
    int		(*serviceEvent) _ANSI_ARGS_((int flags));
    void	(*setAssocData) _ANSI_ARGS_((Tcl_Interp *interp,
                            CONST char *name, Tcl_InterpDeleteProc *proc,
                            ClientData clientData));
    void	(*setChannelBufferSize) _ANSI_ARGS_((
			    Tcl_Channel chan, int sz));
    int		(*setChannelOption) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Channel chan,
	        	    CONST char *optionName, CONST char *newValue));
    int		(*setCommandInfo) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *cmdName, Tcl_CmdInfo *infoPtr));
    void	(*setErrno) _ANSI_ARGS_((int err));
    void	(*setErrorCode) _ANSI_ARGS_(
    			    TCL_VARARGS(Tcl_Interp *,arg1));
    void	(*setMaxBlockTime) _ANSI_ARGS_((Tcl_Time *timePtr));
    void	(*setPanicProc) _ANSI_ARGS_((void (*proc)
			    _ANSI_ARGS_(TCL_VARARGS(char *, format))));
    int		(*setRecursionLimit) _ANSI_ARGS_((Tcl_Interp *interp,
			    int depth));
    void	(*setResult) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *string, Tcl_FreeProc *freeProc));
    int		(*setServiceMode) _ANSI_ARGS_((int mode));
    void	(*setObjErrorCode) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *errorObjPtr));
    void	(*setObjResult) _ANSI_ARGS_((Tcl_Interp *interp,
			Tcl_Obj *resultObjPtr));
    void	(*setStdChannel) _ANSI_ARGS_((Tcl_Channel channel,
			    int type));
    char *	(*setVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, char *newValue, int flags));
    char *	(*setVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, char *newValue,
			    int flags));
    char *	(*signalId) _ANSI_ARGS_((int sig));
    char *	(*signalMsg) _ANSI_ARGS_((int sig));
    void	(*sourceRCFile) _ANSI_ARGS_((Tcl_Interp *interp));
    int		(*splitList) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *list, int *argcPtr, char **argvPtr[]));
    void	(*splitPath) _ANSI_ARGS_((CONST char *path,
			    int *argcPtr, CONST char **argvPtr[]));
    void	(*staticPackage) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *pkgName, Tcl_PackageInitProc *initProc,
			    Tcl_PackageInitProc *safeInitProc));
    int		(*stringMatch) _ANSI_ARGS_((CONST char *string,
			    CONST char *pattern));
    int		(*tell) _ANSI_ARGS_((Tcl_Channel chan));
    int		(*traceVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags, Tcl_VarTraceProc *proc,
			    ClientData clientData));
    int		(*traceVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags,
			    Tcl_VarTraceProc *proc, ClientData clientData));
    char *	(*translateFileName) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, Tcl_DString *bufferPtr));
    int		(*ungets) _ANSI_ARGS_((Tcl_Channel chan, CONST char *str,
			    int len, int atHead));
    void	(*unlinkVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName));
    int		(*unregisterChannel) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Channel chan));
    int		(*unsetVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags));
    int		(*unsetVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags));
    void	(*untraceVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags, Tcl_VarTraceProc *proc,
			    ClientData clientData));
    void	(*untraceVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags,
			    Tcl_VarTraceProc *proc, ClientData clientData));
    void	(*updateLinkedVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName));
    int		(*upVar) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *frameName, char *varName,
			    char *localName, int flags));
    int		(*upVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *frameName, char *part1, char *part2,
			    char *localName, int flags));
    int		(*varEval) _ANSI_ARGS_(
    			    TCL_VARARGS(Tcl_Interp *,interp));
    ClientData	(*varTraceInfo) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *varName, int flags,
			    Tcl_VarTraceProc *procPtr,
			    ClientData prevClientData));
    ClientData	(*varTraceInfo2) _ANSI_ARGS_((Tcl_Interp *, char *,
			    char *, int, Tcl_VarTraceProc *, ClientData));
    int		(*write) _ANSI_ARGS_((Tcl_Channel chan,
			    CONST char *src, int srcLen));
    void	(*wrongNumArgs) _ANSI_ARGS_((Tcl_Interp *interp,
			    int objc, Tcl_Obj *CONST objv[], CONST char *message));

    /* Backported 8.1 functions */
    int		(*evalObj2) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, int flags));
    int		(*getIndexFromObjStruct) _ANSI_ARGS_((Tcl_Interp *interp,
			    Tcl_Obj *objPtr, char **tablePtr, int offset,
			    CONST char *msg, int flags, int *indexPtr));
    Tcl_Obj *	(*getObjVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, int flags));
    Tcl_Obj *	(*setObjVar2) _ANSI_ARGS_((Tcl_Interp *interp,
			    char *part1, char *part2, Tcl_Obj *newValuePtr,
			    int flags));

    /* UTF-related functions. Experimental!!! */
    int		(*externalToUtf) _ANSI_ARGS_((Tcl_Interp *interp,
			    VOID *encoding, CONST char *src, int srcLen,
			    int flags, VOID *statePtr, char *dst,
			    int dstLen, int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));
    char *	(*externalToUtfDString) _ANSI_ARGS_((
			    VOID *encoding, CONST char *src,
			    int srcLen, Tcl_DString *dsPtr));
    int		(*numUtfChars) _ANSI_ARGS_((CONST char *src,
			    int len));
    char *	(*utfAtIndex) _ANSI_ARGS_((CONST char *src,
			    int index));
    int		(*utfCharComplete) _ANSI_ARGS_((CONST char *src,
			    int len));
    int		(*utfBackslash) _ANSI_ARGS_((CONST char *src,
			    int *readPtr, char *dst));
    char *	(*utfFindFirst) _ANSI_ARGS_((CONST char *src,
			    int ch));
    char *	(*utfFindLast) _ANSI_ARGS_((CONST char *src,
			    int ch));
    char *	(*utfNext) _ANSI_ARGS_((CONST char *src));
    char *	(*utfPrev) _ANSI_ARGS_((CONST char *src,
			    CONST char *start));
    int		(*utfToExternal) _ANSI_ARGS_((Tcl_Interp *interp,
			    VOID *encoding, CONST char *src, int srcLen,
			    int flags, VOID *statePtr, char *dst,
			    int dstLen, int *srcReadPtr, int *dstWrotePtr,
			    int *dstCharsPtr));
    char *	(*utfToExternalDString) _ANSI_ARGS_((
			    VOID *encoding, CONST char *src,
			    int srcLen, Tcl_DString *dsPtr));
    int		(*utfToLower) _ANSI_ARGS_((char *src));
    int		(*utfToTitle) _ANSI_ARGS_((char *src));

    /*
     * Put new functions here, preferably.
     */

    VOID *reserved1;
    VOID *reserved2;
    VOID *reserved3;

    /*
     * All functions after here are secretly exported
     * for use in the Tcl tests. They can not be thrusted
     * to keep the same positions in succeeding versions
     * of the plus-patch. Use them at your own risk.
     *
     *                      Jan Nijtmans
     */

    int magic2;         /* should be TCL_STUB_MAGIC */
    VOID *platform;    /* pointer to tclPlatform variable */
    char **executableName; /* pointer to tclExecutableName variable */

    int		(*access) _ANSI_ARGS_((CONST char *path, 
			    int mode));
    int		(*accessDeleteProc) _ANSI_ARGS_((void *proc));
    int		(*accessInsertProc) _ANSI_ARGS_((void *proc));
    int		(*closeFile) _ANSI_ARGS_((VOID *file));
    int         (*copyDirectory) _ANSI_ARGS_((CONST char *source,
			    CONST char *dest, Tcl_DString *errorPtr));
    int		(*copyFile) _ANSI_ARGS_((CONST char *source,
			    CONST char *dest));
    int         (*createDirectory) _ANSI_ARGS_((CONST char *path));
    int         (*createPipe) _ANSI_ARGS_((VOID **readPipe,
			    VOID **writePipe));
    int         (*deleteFile) _ANSI_ARGS_((CONST char *path));
    Tcl_Namespace * (*findNamespace) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *name, Tcl_Namespace *contextNsPtr,
			    int flags));
    Tcl_Var	(*findNamespaceVar) _ANSI_ARGS_((
			    Tcl_Interp *interp, char *name,
			    Tcl_Namespace *contextNsPtr, int flags));
    void	(*getVariableFullName) _ANSI_ARGS_((
			    Tcl_Interp *interp, Tcl_Var variable,
			    Tcl_Obj *objPtr));
    void	(*getTime) _ANSI_ARGS_((VOID *timePtr));
    Tcl_Obj *	(*newProcBodyObj) _ANSI_ARGS_((VOID *procPtr));
    int		(*objInterpProc) _ANSI_ARGS_((ClientData clientData,
		    	    Tcl_Interp *interp, int objc,
			    Tcl_Obj *CONST objv[]));
    Tcl_Channel	(*pOpenFileChannel) _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *fileName, CONST char *modeString,
			    int permissions));
    int		(*openFileChannelDeleteProc) _ANSI_ARGS_((
			    void *proc));
    int		(*openFileChannelInsertProc) _ANSI_ARGS_((
			    void *proc));
    void	(*popCallFrame) _ANSI_ARGS_((Tcl_Interp* interp));
    int		(*procInterpProc) _ANSI_ARGS_((ClientData clientData,
		    	    Tcl_Interp *interp, int argc, char **argv));
    int		(*pushCallFrame) _ANSI_ARGS_((Tcl_Interp* interp,
			    Tcl_CallFrame *framePtr, Tcl_Namespace *nsPtr,
			    int isProcCallFrame)); 
    int         (*removeDirectory) _ANSI_ARGS_((CONST char *path,
			    int recursive, Tcl_DString *errorPtr));
    int         (*renameFile) _ANSI_ARGS_((CONST char *source,
			    CONST char *dest));
    int		(*stat) _ANSI_ARGS_((CONST char *path, 
			    struct stat *buf));
    int		(*statDeleteProc) _ANSI_ARGS_((void *proc));
    int		(*statInsertProc) _ANSI_ARGS_((void *proc));
    int		(*testChannelCmd) _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));
    int		(*testChannelEventCmd) _ANSI_ARGS_((
    			    ClientData clientData, Tcl_Interp *interp,
                            int argc, char **argv));
#if (!defined(__WIN32__) && !defined(MAC_TCL))
    int		(*unixWaitForFile) _ANSI_ARGS_((int fd, int mask,
		    int timeout));
#endif
} Tcl_StubTable;

#ifdef TCL_USE_STUBS

extern Tcl_StubTable *tclStubTable;

typedef struct Tcl_FakeInterp {
    char *result;		/* If the last command returned a string
				 * result, this points to it. */
    void (*freeProc) _ANSI_ARGS_((char *blockPtr));
				/* Zero means the string result is
				 * statically allocated. TCL_DYNAMIC means
				 * it was allocated with ckalloc and should
				 * be freed with ckfree. Other values give
				 * the address of procedure to invoke to
				 * free the result. Tcl_Eval must free it
				 * before executing next command. */
    int errorLine;              /* When TCL_ERROR is returned, this gives
                                 * the line number within the command where
                                 * the error occurred (1 if first line). */
    Tcl_StubTable *stub;	/* stub table */
} Tcl_FakeInterp;

#ifndef TCL_MEM_DEBUG

#define Tcl_Alloc(size) \
	(tclStubTable->alloc(size))
#define Tcl_Free \
	(tclStubTable->free)
#define Tcl_Realloc(ptr, size) \
	(tclStubTable->realloc(ptr, size))
#define Tcl_NewBooleanObj(boolValue) \
	(tclStubTable->newBooleanObj(boolValue))
#define Tcl_NewByteArrayObj(bytes, length) \
	(tclStubTable->newByteArrayObj(bytes, length))
#define Tcl_NewDoubleObj(doubleValue) \
	(tclStubTable->newDoubleObj(doubleValue))
#define Tcl_NewIntObj(intValue) \
	(tclStubTable->newIntObj(intValue))
#define Tcl_NewListObj(objc, objv) \
	(tclStubTable->newListObj(objc, objv))
#define Tcl_NewLongObj(longValue) \
	(tclStubTable->newLongObj(longValue))
#define Tcl_NewObj() \
	(tclStubTable->newObj())
#define Tcl_NewStringObj(string, length) \
	(tclStubTable->newStringObj(string, length))

#endif /* TCL_MEM_DEBUG */

#define Tcl_HasStubSupport(interp, exact) \
	(((Tcl_FakeInterp*) (interp))->stub && \
	((Tcl_FakeInterp*) (interp))->stub->magic == TCL_STUB_MAGIC)
#define panic \
	(tclStubTable->panic)
#define TclFreeObj \
	(tclStubTable->freeObj)
#define Tcl_AppendAllObjTypes \
	(tclStubTable->appendAllObjTypes)
#define Tcl_AppendElement \
	(tclStubTable->appendElement)
#define Tcl_AppendResult \
	(tclStubTable->appendResult)
#define Tcl_AppendStringsToObj \
	(tclStubTable->appendStringsToObj)
#define Tcl_AppendToObj \
	(tclStubTable->appendToObj)
#define Tcl_AsyncCreate \
	(tclStubTable->asyncCreate)
#define Tcl_AsyncDelete \
	(tclStubTable->asyncDelete)
#define Tcl_AsyncMark \
	(tclStubTable->asyncMark)
#define Tcl_AsyncReady \
	(tclStubTable->asyncReady)
#define Tcl_BadChannelOption \
	(tclStubTable->badChannelOption)
#define Tcl_CancelIdleCall \
	(tclStubTable->cancelIdleCall)
#define Tcl_Close \
	(tclStubTable->close)
#define Tcl_CreateChannelHandler \
	(tclStubTable->createChannelHandler)
#define Tcl_CreateCloseHandler \
	(tclStubTable->createCloseHandler)
#define Tcl_CommandComplete \
	(tclStubTable->commandComplete)
#define Tcl_Concat \
	(tclStubTable->concat)
#define Tcl_ConcatObj \
	(tclStubTable->concatObj)
#define Tcl_ConvertElement \
	(tclStubTable->convertElement)
#define Tcl_ConvertCountedElement \
	(tclStubTable->convertCountedElement)
#define Tcl_ConvertToType \
	(tclStubTable->convertToType)
#define Tcl_CreateChannel \
	(tclStubTable->createChannel)
#define Tcl_CreateEventSource \
	(tclStubTable->createEventSource)
#define Tcl_CreateExitHandler \
	(tclStubTable->createExitHandler)
#define Tcl_CreateFileHandler \
	(tclStubTable->createFileHandler)
#define Tcl_CreateInterp \
	(tclStubTable->createInterp)
#define Tcl_CreateTimerHandler \
	(tclStubTable->createTimerHandler)
#define Tcl_DbCkalloc \
	(tclStubTable->dbCkalloc)
#define Tcl_DbCkfree \
	(tclStubTable->dbCkfree)
#define Tcl_DbCkrealloc \
	(tclStubTable->dbCkrealloc)
#define Tcl_DbDecrRefCount \
	(tclStubTable->dbDecrRefCount)
#define Tcl_DbIncrRefCount \
	(tclStubTable->dbIncrRefCount)
#define Tcl_DbIsShared \
	(tclStubTable->dbIsShared)
#define Tcl_DbNewBooleanObj \
	(tclStubTable->dbNewBooleanObj)
#define Tcl_DbNewByteArrayObj \
	(tclStubTable->dbNewByteArrayObj)
#define Tcl_DbNewDoubleObj \
	(tclStubTable->dbNewDoubleObj)
#define Tcl_DbNewListObj \
	(tclStubTable->dbNewListObj)
#define Tcl_DbNewLongObj \
	(tclStubTable->dbNewLongObj)
#define Tcl_DbNewObj \
	(tclStubTable->dbNewObj)
#define Tcl_DbNewStringObj \
	(tclStubTable->dbNewStringObj)
#define Tcl_DeleteChannelHandler \
	(tclStubTable->deleteChannelHandler)
#define Tcl_DeleteCloseHandler \
	(tclStubTable->deleteCloseHandler)
#define Tcl_DeleteEventSource \
	(tclStubTable->deleteEventSource)
#define Tcl_DeleteExitHandler \
	(tclStubTable->deleteExitHandler)
#define Tcl_DeleteFileHandler \
	(tclStubTable->deleteFileHandler)
#define Tcl_DeleteHashEntry \
	(tclStubTable->deleteHashEntry)
#define Tcl_DeleteHashTable \
	(tclStubTable->deleteHashTable)
#define Tcl_DeleteTimerHandler \
	(tclStubTable->deleteTimerHandler)
#define Tcl_DoOneEvent \
	(tclStubTable->doOneEvent)
#define Tcl_DoWhenIdle \
	(tclStubTable->doWhenIdle)
#define Tcl_DStringAppend \
	(tclStubTable->dStringAppend)
#define Tcl_DStringAppendElement \
	(tclStubTable->dStringAppendElement)
#define Tcl_DStringEndSublist \
	(tclStubTable->dStringEndSublist)
#define Tcl_DStringFree \
	(tclStubTable->dStringFree)
#define Tcl_DStringInit \
	(tclStubTable->dStringInit)
#define Tcl_DStringSetLength \
	(tclStubTable->dStringSetLength)
#define Tcl_DStringStartSublist \
	(tclStubTable->dStringStartSublist)
#define Tcl_DuplicateObj \
	(tclStubTable->duplicateObj)
#define Tcl_Eof \
	(tclStubTable->eof)
#define Tcl_EventuallyFree \
	(tclStubTable->eventuallyFree)
#define Tcl_Exit \
	(tclStubTable->exit)
#define Tcl_ErrnoId \
	(tclStubTable->errnoId)
#define Tcl_Finalize \
	(tclStubTable->finalize)
#define Tcl_FindExecutable \
	(tclStubTable->findExecutable)
#define Tcl_FirstHashEntry \
	(tclStubTable->firstHashEntry)
#define Tcl_Flush \
	(tclStubTable->flush)
#define Tcl_GetBoolean \
	(tclStubTable->getBoolean)
#define Tcl_GetBooleanFromObj \
	(tclStubTable->getBooleanFromObj)
#define Tcl_GetByteArrayFromObj \
	(tclStubTable->getByteArrayFromObj)
#define Tcl_GetErrno \
	(tclStubTable->getErrno)
#define Tcl_GetHostName \
	(tclStubTable->getHostName)
#define Tcl_GetDouble \
	(tclStubTable->getDouble)
#define Tcl_GetDoubleFromObj \
	(tclStubTable->getDoubleFromObj)
#define Tcl_GetIndexFromObj \
	(tclStubTable->getIndexFromObj)
#define Tcl_GetIndexFromObjStruct \
	(tclStubTable->getIndexFromObjStruct)
#define Tcl_GetInt \
	(tclStubTable->getInt)
#define Tcl_GetIntFromObj \
	(tclStubTable->getIntFromObj)
#define Tcl_GetLongFromObj \
	(tclStubTable->getLongFromObj)
#define Tcl_GetObjType \
	(tclStubTable->getObjType)
#define Tcl_GetPathType \
	(tclStubTable->getPathType)
#define Tcl_GetServiceMode \
	(tclStubTable->getServiceMode)
#define Tcl_GetStdChannel \
	(tclStubTable->getStdChannel)
#define Tcl_GetStringFromObj \
	(tclStubTable->getStringFromObj)
#define Tcl_InitHashTable \
	(tclStubTable->initHashTable)
#define Tcl_InvalidateStringRep \
	(tclStubTable->invalidateStringRep)
#define Tcl_JoinPath \
	(tclStubTable->joinPath)
#define Tcl_ListObjAppendList \
	(tclStubTable->listObjAppendList)
#define Tcl_ListObjAppendElement \
	(tclStubTable->listObjAppendElement)
#define Tcl_ListObjGetElements \
	(tclStubTable->listObjGetElements)
#define Tcl_ListObjIndex \
	(tclStubTable->listObjIndex)
#define Tcl_ListObjLength \
	(tclStubTable->listObjLength)
#define Tcl_ListObjReplace \
	(tclStubTable->listObjReplace)
#define Tcl_MakeFileChannel \
	(tclStubTable->makeFileChannel)
#define Tcl_MakeTcpClientChannel \
	(tclStubTable->makeTcpClientChannel)
#define Tcl_Merge \
	(tclStubTable->merge)
#define Tcl_NextHashEntry \
	(tclStubTable->nextHashEntry)
#define Tcl_NotifyChannel \
	(tclStubTable->notifyChannel)
#define Tcl_OpenCommandChannel \
	(tclStubTable->openCommandChannel)
#define Tcl_OpenFileChannel \
	(tclStubTable->openFileChannel)
#define Tcl_OpenTcpClient \
	(tclStubTable->openTcpClient)
#define Tcl_OpenTcpServer \
	(tclStubTable->openTcpServer)
#define Tcl_Preserve \
	(tclStubTable->preserve)
#define Tcl_PrintDouble \
	(tclStubTable->printDouble)
#define Tcl_PutEnv \
	(tclStubTable->putEnv)
#define Tcl_QueueEvent \
	(tclStubTable->queueEvent)
#define Tcl_RegisterObjType \
	(tclStubTable->registerObjType)
#define Tcl_RegExpCompile \
	(tclStubTable->regExpCompile)
#define Tcl_RegExpExec \
	(tclStubTable->regExpExec)
#define Tcl_RegExpMatch \
	(tclStubTable->regExpMatch)
#define Tcl_RegExpRange \
	(tclStubTable->regExpRange)
#define Tcl_Release \
	(tclStubTable->release)
#define Tcl_ScanCountedElement \
	(tclStubTable->scanCountedElement)
#define Tcl_ServiceAll \
	(tclStubTable->serviceAll)
#define Tcl_ServiceEvent \
	(tclStubTable->serviceEvent)
#define Tcl_SetBooleanObj \
	(tclStubTable->setBooleanObj)
#define Tcl_SetDoubleObj \
	(tclStubTable->setDoubleObj)
#define Tcl_SetErrno \
	(tclStubTable->setErrno)
#define Tcl_SetErrorCode \
	(tclStubTable->setErrorCode)
#define Tcl_SetIntObj \
	(tclStubTable->setIntObj)
#define Tcl_SetListObj \
	(tclStubTable->setListObj)
#define Tcl_SetLongObj \
	(tclStubTable->setLongObj)
#define Tcl_SetMaxBlockTime \
	(tclStubTable->setMaxBlockTime)
#define Tcl_SetObjLength \
	(tclStubTable->setObjLength)
#define Tcl_SetServiceMode \
	(tclStubTable->setServiceMode)
#define Tcl_SetStringObj \
	(tclStubTable->setStringObj)
#define Tcl_SignalId \
	(tclStubTable->signalId)
#define Tcl_SignalMsg \
	(tclStubTable->signalMsg)
#define Tcl_SourceRCFile \
	(tclStubTable->sourceRCFile)
#define Tcl_Sleep \
	(tclStubTable->sleep)
#define Tcl_SplitList \
	(tclStubTable->splitList)
#define Tcl_SplitPath \
	(tclStubTable->splitPath)
#define Tcl_StaticPackage \
	(tclStubTable->staticPackage)
#define Tcl_StringMatch \
	(tclStubTable->stringMatch)
#define Tcl_TranslateFileName \
	(tclStubTable->translateFileName)
#define Tcl_VarEval \
	(tclStubTable->varEval)
#define Tcl_WaitForEvent \
	(tclStubTable->waitForEvent)

#define Tcl_PkgProvide(interp, name, version) \
	Tcl_PkgProvide2(interp, name, version, 0)
#define Tcl_PkgRequire(interp, name, version, exact) \
	Tcl_PkgRequire2(interp, name, version, exact, 0)
#define Tcl_Required(interp, version, exact) \
	Tcl_PkgRequire2(interp, "Tcl", version, exact, &tclStubTable)
#define Tcl_PkgRequire2(interp, name, version, exact, clientData) \
		(Tcl_HasStubSupport(interp, exact)?((Tcl_FakeInterp*) \
		(interp))->stub->pkgRequire2(interp, \
		name, version, exact, (ClientData *) clientData) : \
		(((interp)->result="Stub conflict between Tcl and this Package"), \
		((interp)->freeProc=0), NULL))
#define Tcl_PkgProvide2(interp, name, version, stubTable) \
	((Tcl_FakeInterp*) ((interp)))->stub->pkgProvide2(interp, \
		name, version, (ClientData) (stubTable))
#define Tcl_AddErrorInfo(interp, message) \
	((Tcl_FakeInterp*) (interp))->stub->addErrorInfo(interp, message)
#define Tcl_AddObjErrorInfo(interp, message, length) \
	((Tcl_FakeInterp*) (interp))->stub->addObjErrorInfo(interp, message, length)
#define Tcl_AllowExceptions(interp) \
	((Tcl_FakeInterp*) (interp))->stub->allowExceptions(interp)
#define Tcl_AsyncInvoke(interp, code) \
	((Tcl_FakeInterp*) (interp))->stub->asyncInvoke(interp, code)
#define Tcl_BackgroundError(interp) \
	((Tcl_FakeInterp*) (interp))->stub->backgroundError(interp)
#define Tcl_Backslash(src, readPtr) \
	((Tcl_FakeInterp*) (interp))->stub->backslash(src, readPtr)
#define Tcl_CallWhenDeleted(interp, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->callWhenDeleted(interp, \
		proc, clientData)
#define Tcl_CreateCommand(interp, cmdName, proc, clientData, deleteProc) \
	((Tcl_FakeInterp*) (interp))->stub->createCommand(interp, \
		cmdName, proc, clientData, deleteProc)
#define Tcl_ExposeCommand(interp, hiddenCmdToken, cmdName) \
	((Tcl_FakeInterp*) (interp))->stub->exposeCommand(interp, \
		hiddenCmdToken, cmdName)
#define Tcl_CreateMathFunc(interp, name, numArgs, argTypes, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->createMathFunc(interp, name, numArgs, \
		argTypes, proc, clientData)
#define Tcl_CreateObjCommand(interp, cmdName, proc, clientData, deleteProc) \
	((Tcl_FakeInterp*) (interp))->stub->createObjCommand(interp, \
		cmdName, proc, clientData, deleteProc)
#define Tcl_CreateSlave(interp, slaveName, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->createSlave(interp, slaveName, \
		clientData)
#define Tcl_CreateTrace(interp, level, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->createTrace(interp, \
		level, proc, clientData)
#define Tcl_CreateAlias(slave, slaveCmd, target, targetCmd, argc, argv) \
	((Tcl_FakeInterp*) (interp))->stub->createAlias(slave, \
		slaveCmd, target, targetCmd, argc, argv)
#define Tcl_CreateAliasObj(slave, slaveCmd, target, targetCmd, objc, objv) \
	((Tcl_FakeInterp*) (interp))->stub->createAliasObj(slave, \
		slaveCmd, target, targetCmd, objc, objv)
#define Tcl_DeleteAssocData(interp, name) \
	((Tcl_FakeInterp*) (interp))->stub->deleteAssocData(interp, name)
#define Tcl_DeleteCommand(interp, cmdName) \
	((Tcl_FakeInterp*) (interp))->stub->deleteCommand(interp, cmdName)
#define Tcl_DeleteCommandFromToken(interp, command) \
	((Tcl_FakeInterp*) (interp))->stub->deleteCommandFromToken(interp, command)
#define Tcl_DeleteInterp(interp) \
	((Tcl_FakeInterp*) (interp))->stub->deleteInterp(interp)
#define Tcl_DontCallWhenDeleted(interp, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->dontCallWhenDeleted(interp, \
		proc, clientData)
#define Tcl_DeleteTrace(interp, trace) \
	((Tcl_FakeInterp*) (interp))->stub->deleteTrace(interp, trace)
#define Tcl_DStringGetResult(interp, dsPtr) \
	((Tcl_FakeInterp*) (interp))->stub->dStringGetResult(interp, dsPtr)
#define Tcl_DStringResult(interp, dsPtr) \
	((Tcl_FakeInterp*) (interp))->stub->dStringResult(interp, dsPtr)
#define Tcl_Eval(interp, command) \
	((Tcl_FakeInterp*) (interp))->stub->eval(interp, command)
#define Tcl_EvalFile(interp, fileName) \
	((Tcl_FakeInterp*) (interp))->stub->evalFile(interp, fileName)
#define Tcl_EvalObj(interp, objPtr) \
	((Tcl_FakeInterp*) (interp))->stub->evalObj(interp, objPtr)
#define Tcl_ExprLong(interp, string, ptr) \
	((Tcl_FakeInterp*) (interp))->stub->exprLong(interp, string, ptr)
#define Tcl_ExprString(interp, string) \
	((Tcl_FakeInterp*) (interp))->stub->exprString(interp, string)
#define Tcl_ExprBooleanObj(interp, objPtr, ptr) \
	((Tcl_FakeInterp*) (interp))->stub->exprBooleanObj(interp, objPtr, ptr)
#define Tcl_ExprDoubleObj(interp, objPtr, ptr) \
	((Tcl_FakeInterp*) (interp))->stub->exprDoubleObj(interp, objPtr, ptr)
#define Tcl_ExprLongObj(interp, objPtr, ptr) \
	((Tcl_FakeInterp*) (interp))->stub->exprLongObj(interp, objPtr, ptr)
#define Tcl_FreeResult(interp) \
	((Tcl_FakeInterp*) (interp))->stub->freeResult(interp)
#define Tcl_GetAlias(interp, slaveCmd, targetInterpPtr, targetCmdPtr, argcPtr, argvPtr) \
	((Tcl_FakeInterp*) (interp))->stub->getAlias(interp, \
		slaveCmd, targetInterpPtr, targetCmdPtr, argcPtr, argvPtr)
#define Tcl_GetAliasObj(interp, slaveCmd, targetInterpPtr, targetCmdPtr, objcPtr, objvPtr) \
	((Tcl_FakeInterp*) (interp))->stub->getAliasObj(interp, \
		slaveCmd, targetInterpPtr, targetCmdPtr, objcPtr, objvPtr)
#define Tcl_GetAssocData(interp, name, procPtr) \
	((Tcl_FakeInterp*) (interp))->stub->getAssocData(interp, \
		name, procPtr)
#define Tcl_GetChannel(interp, chanName, modePtr) \
	((Tcl_FakeInterp*) (interp))->stub->getChannel(interp, \
		chanName, modePtr)
#define Tcl_GetCommandInfo(interp, cmdName, infoPtr) \
	((Tcl_FakeInterp*) (interp))->stub->getCommandInfo(interp, \
		cmdName, infoPtr)
#define Tcl_GetCommandName(interp, command) \
	((Tcl_FakeInterp*) (interp))->stub->getCommandName(interp, command)
#define Tcl_GetInterpPath(askInterp, slaveInterp) \
	((Tcl_FakeInterp*) (interp))->stub->getInterpPath(askInterp, slaveInterp)
#define Tcl_GetMaster(interp) \
	((Tcl_FakeInterp*) (interp))->stub->getMaster(interp)
#define Tcl_GetObjResult(interp) \
	((Tcl_FakeInterp*) (interp))->stub->getObjResult(interp)
#define Tcl_GetObjVar2(interp, part1, part2, flags) \
	((Tcl_FakeInterp*) (interp))->stub->getObjVar2(interp, part1, part2, flags)
#define Tcl_GetOpenFile(interp, string, write, checkUsage, filePtr) \
	((Tcl_FakeInterp*) (interp))->stub->getOpenFile(interp, \
		string, write, checkUsage, filePtr)
#define Tcl_GetSlave(interp, slaveName) \
	((Tcl_FakeInterp*) (interp))->stub->getSlave(interp, slaveName)
#define Tcl_GetStringResult(interp) \
	((Tcl_FakeInterp*) (interp))->stub->getStringResult(interp)
#define Tcl_GetVar(interp, varName, flags) \
	((Tcl_FakeInterp*) (interp))->stub->getVar(interp, varName, flags)
#define Tcl_GetVar2(interp, part1, part2, flags) \
	((Tcl_FakeInterp*) (interp))->stub->getVar2(interp, part1, part2, flags)
#define Tcl_GlobalEval(interp, command) \
	((Tcl_FakeInterp*) (interp))->stub->globalEval(interp, command)
#define Tcl_GlobalEvalObj(interp, objPtr) \
	((Tcl_FakeInterp*) (interp))->stub->globalEvalObj(interp, \
		objPtr)
#define Tcl_HideCommand(interp, cmdName, hiddenCmdToken) \
	((Tcl_FakeInterp*) (interp))->stub->hideCommand(interp, cmdName, \
		hiddenCmdToken)
#define Tcl_Init(interp) \
	((Tcl_FakeInterp*) (interp))->stub->init(interp)
#define Tcl_InterpDeleted(interp) \
	((Tcl_FakeInterp*) (interp))->stub->interpDeleted(interp)
#define Tcl_IsSafe(interp) \
	((Tcl_FakeInterp*) (interp))->stub->isSafe(interp)
#define Tcl_LinkVar(interp, varName, addr, type) \
	((Tcl_FakeInterp*) (interp))->stub->linkVar(interp, \
		varName, addr, type)
#define Tcl_MakeSafe(interp) \
	((Tcl_FakeInterp*) (interp))->stub->makeSafe(interp)
#define Tcl_ObjGetVar2(interp, part1Ptr, part2Ptr, flags) \
	((Tcl_FakeInterp*) (interp))->stub->getObjVar2(interp, \
		Tcl_GetString(part1Ptr), (part2Ptr)?Tcl_GetString(part2Ptr):NULL, flags)
#define Tcl_ObjSetVar2(interp, part1Ptr, part2Ptr, newValuePtr, flags) \
	((Tcl_FakeInterp*) (interp))->stub->setObjVar2(interp, \
		Tcl_GetString(part1Ptr), (part2Ptr)?Tcl_GetString(part2Ptr):NULL, \
		newValuePtr, flags)
#define Tcl_PosixError(interp) \
	((Tcl_FakeInterp*) (interp))->stub->posixError(interp)
#define Tcl_RecordAndEval(interp, cmd, flags) \
	((Tcl_FakeInterp*) (interp))->stub->recordAndEval(interp, cmd, flags)
#define Tcl_RecordAndEvalObj(interp, cmdPtr, flags) \
	((Tcl_FakeInterp*) (interp))->stub->recordAndEvalObj(interp, cmdPtr, flags)
#define Tcl_ResetResult(interp) \
	((Tcl_FakeInterp*) (interp))->stub->resetResult(interp)
#define Tcl_SetAssocData(interp, name, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->setAssocData(interp, \
		name, proc, clientData)
#define Tcl_SetCommandInfo(interp, cmdName, infoPtr) \
	((Tcl_FakeInterp*) (interp))->stub->setCommandInfo(interp, \
		cmdName, infoPtr)
#define Tcl_SetRecursionLimit(interp, depth) \
	((Tcl_FakeInterp*) (interp))->stub->setRecursionLimit(interp, depth)
#define Tcl_SetResult(interp, string, freeProc) \
	((Tcl_FakeInterp*) (interp))->stub->setResult(interp, \
		string, freeProc)
#define Tcl_SetObjErrorCode(interp, errorObjPtr) \
	((Tcl_FakeInterp*) (interp))->stub->setObjErrorCode(interp, \
		errorObjPtr)
#define Tcl_SetObjResult(interp, resultObjPtr) \
	((Tcl_FakeInterp*) (interp))->stub->setObjResult(interp, \
		resultObjPtr)
#define Tcl_SetObjVar2(interp, part1, part2, newValuePtr, flags) \
	((Tcl_FakeInterp*) (interp))->stub->setObjVar2(interp, \
		part1, part2, newValuePtr, flags)
#define Tcl_SetVar(interp, varName, newValue, flags) \
	((Tcl_FakeInterp*) (interp))->stub->setVar(interp, \
		varName, newValue, flags)
#define Tcl_SetVar2(interp, part1, part2, newValue, flags) \
	((Tcl_FakeInterp*) (interp))->stub->setVar2(interp, \
		part1, part2, newValue, flags)
#define Tcl_TraceVar(interp, varName, flags, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->traceVar(interp, \
		varName, flags, proc, clientData)
#define Tcl_TraceVar2(interp, part1, part2, flags, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->traceVar2(interp, \
		part1, part2, flags, proc, clientData)
#define Tcl_UnlinkVar(interp, varName) \
	((Tcl_FakeInterp*) (interp))->stub->unlinkVar(interp, varName)
#define Tcl_UnsetVar(interp, varName, flags) \
	((Tcl_FakeInterp*) (interp))->stub->unsetVar(interp, varName, flags)
#define Tcl_UnsetVar2(interp, part1, part2, flags) \
	((Tcl_FakeInterp*) (interp))->stub->unsetVar2(interp, \
		part1, part2, flags)
#define Tcl_UntraceVar(interp, varName, flags, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->untraceVar(interp, \
		varName, flags, proc, clientData)
#define Tcl_UntraceVar2(interp, part1, part2, flags, proc, clientData) \
	((Tcl_FakeInterp*) (interp))->stub->untraceVar2(interp, \
		part1, part2, flags, proc, clientData)
#define Tcl_UpdateLinkedVar(interp, varName) \
	((Tcl_FakeInterp*) (interp))->stub->updateLinkedVar(interp, \
		varName)
#define Tcl_UpVar(interp, frameName, varName, localName, flags) \
	((Tcl_FakeInterp*) (interp))->stub->upVar(interp, frameName, \
		varName, localName, flags)
#define Tcl_UpVar2(interp, frameName, part1, part2, localName, flags) \
	((Tcl_FakeInterp*) (interp))->stub->upVar2(interp, frameName, \
		part1, part2, localName, flags)
#define Tcl_VarTraceInfo(interp, varName, flags, procPtr, prevClientData) \
	((Tcl_FakeInterp*) (interp))->stub->varTraceInfo(interp, varName\
		flags, procPtr, prevClientData)
#define Tcl_WrongNumArgs(interp, objc, objv, message) \
	((Tcl_FakeInterp*) (interp))->stub->wrongNumArgs(interp, \
		objc, objv, message)

#define Tcl_GetChannelBufferSize \
	(tclStubTable->getChannelBufferSize)
#define Tcl_GetChannelHandle \
	(tclStubTable->getChannelHandle)
#define Tcl_GetChannelInstanceData \
	(tclStubTable->getChannelInstanceData)
#define Tcl_GetChannelMode \
	(tclStubTable->getChannelMode)
#define Tcl_GetChannelName \
	(tclStubTable->getChannelName)
#define Tcl_GetChannelOption \
	(tclStubTable->getChannelOption)
#define Tcl_GetChannelType \
	(tclStubTable->getChannelType)
#define Tcl_Gets \
	(tclStubTable->gets)
#define Tcl_GetsObj \
	(tclStubTable->getsObj)
#define Tcl_GetNameOfExecutable \
	(tclStubTable->getNameOfExecutable)
#define Tcl_InputBlocked \
	(tclStubTable->inputBlocked)
#define Tcl_InputBuffered \
	(tclStubTable->inputBuffered)
#define Tcl_Read \
	(tclStubTable->read)
#define Tcl_RegisterChannel \
	(tclStubTable->registerChannel)
#define Tcl_Seek \
	(tclStubTable->seek)
#define Tcl_SetChannelBufferSize \
	(tclStubTable->setChannelBufferSize)
#define Tcl_SetChannelOption \
	(tclStubTable->setChannelOption)
#define Tcl_Tell \
	(tclStubTable->tell)
#define Tcl_UnregisterChannel \
	(tclStubTable->unregisterChannel)
#define Tcl_Write \
	(tclStubTable->write)

#endif /* TCL_USE_STUBS */

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS

/*
 * Convenience declaration of Tcl_AppInit for backwards compatibility.
 * This function is not *implemented* by the tcl library, so the storage
 * class is neither DLLEXPORT nor DLLIMPORT
 */

EXTERN int             Tcl_AppInit _ANSI_ARGS_((Tcl_Interp *interp));

#endif /* RESOURCE_INCLUDED */

#undef TCL_STORAGE_CLASS
#define TCL_STORAGE_CLASS DLLIMPORT

#endif /* _TCL */
