##############################################################################
#    TCL Developer Studio
#
#    Copyright (C) 1999  Alexey Kakunin
#    small@star.spb.ru
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
##############################################################################

namespace eval editorWindows {

namespace export create openFile selectFile applyChanges isChanged deleteFile
namespace export copy cut paste delete selectAll
namespace export gotoMark gotoProc find findNext setCursor replace replaceAll
namespace export enableHL disableHL onTabSize onFontChange

variable This
variable TxtWidget ""
variable Text ""

# editorWindows::create --
#
#        Create editors frame of TCL DevStudio
#
#        Arguments:
#            this        frame for storing editors windows
#
#        Results: (no results)
proc create {this} {
    global text

    variable This

    frame $this.f -bg grey50 -relief sunken
    pack $this.f -expand 1 -fill both

    set This $this.f
    return
}

# editorWindows::openFile --
#
#        Open file. Add file to editted files.
#        Arguments:
#                fileName - name of file for loading
#
#        Results: (no results)
proc openFile {fileName} {
    global tclDevData
    variable Text
    variable This

    if {$Text == ""} {
        # create text widget
        CreateTextWidget
        }

    # Read Marks
    ReadMarks $fileName
    
    # set text to the widget
    selectFile $fileName

    return
}

# editorWindows::CreateTextWidget --
#
#        Create Text Widget
#        Arguments: (none)
#
#        Results: (none)
proc CreateTextWidget {} {
    global tcl_platform
    global tclDevData
    
    variable This
    variable TxtWidget
    variable Text

    if {$Text != ""} {
        error "Internal Error"
    }

    $This configure -bg grey
    set Text [tixScrolledText $This.text -borderwidth 10]
    
    set TxtWidget [$Text subwidget text]
        
    # create font for editor window
#    if {$tcl_platform(platform) == "unix"} {
#        set size 12
#    } else {
#        set size 10
#    }

    set tabSize [expr {$tclDevData(options,tabSize)*
              [font measure $tclDevData(options,fonts,editorFont) -displayof $TxtWidget " "]}]

    $TxtWidget configure -wrap none -font $tclDevData(options,fonts,editorFont) -tabs [list $tabSize]

    ConfigureTags
        
    pack $Text -expand 1 -fill both -padx 5 -pady 5

    # create bindings
    bind $TxtWidget <Tab> "editorWindows::OnTabPress; break"
    bind $TxtWidget <KeyRelease-Return> "editorWindows::Indent; editorWindows::ReadCursor"
    bind $TxtWidget <braceright> editorWindows::OnCloseBrace
    
    if {$tcl_platform(platform) != "windows"} {
        bind $TxtWidget <Control-Insert> "editorWindows::copy; break"
        bind $TxtWidget <Control-Delete> "editorWindows::cut; break"
        bind $TxtWidget <Shift-Insert>   "editorWindows::paste; break"
        bind $TxtWidget <Control-a> "editorWindows::selectAll; break"
    }
    
    bind $TxtWidget <KeyRelease> editorWindows::OnKeyRelease
    bind $TxtWidget <ButtonRelease> editorWindows::OnMouseRelease

    bind $TxtWidget <Control-n> "mainFrame::onFileNew; break"
    bind $TxtWidget <Control-o> "mainFrame::onFileOpen; break"
    bind $TxtWidget <Control-s> "mainFrame::onFileSave; break"
    bind $TxtWidget <Control-x> "editorWindows::cut; break"
    bind $TxtWidget <Control-c> "editorWindows::copy; break"
    bind $TxtWidget <Control-v> "editorWindows::paste; break"
    bind $TxtWidget <Control-f> "editorWindows::OnFind; break"
    bind $TxtWidget <Control-l> "editorWindows::findNext; break"
    bind $TxtWidget <Control-h> "editorWindows::OnReplace; break"
    bind $TxtWidget <Control-g> "mainFrame::goTo; break"
    
    #enable menus and buttons
    mainFrame::onCreateText
    
    return
}

# delete text widget
proc DeleteTextWidget {} {

    variable This
    variable Text
    variable TxtWidget
    
    if {$Text == ""} {
        error "Internal Error"
    }

    tixDestroy $Text
    set Text ""
    set TxtWidget ""
    
    $This configure -bg grey50

    #disable menus and buttons
    mainFrame::onDestroyText
    
    return
}

# editorWindows::selectFile --
#
#        Set file as current file
#        Arguments: fileName
#
#        Results: (none)
proc selectFile {fileName} {
    global tclDevData
    variable TxtWidget

    if {$fileName != $tclDevData(curFile)} {
 
        # this is not used now.       
#        if {$tclDevData(curFile) != ""} {
#            projectWindow::closeFile $tclDevData(curFile)
#        }
        
        # store information about current file
        applyChanges
        
        # set new text
        SetFile $fileName

        # set focus to the editor window
        focus $TxtWidget
    }
}

# set active file
proc SetFile {fileName} {
    global tclDevData
    variable TxtWidget

    #clear text widget
    $TxtWidget delete 0.0 end

    set tclDevData(curFile) $fileName
    $TxtWidget insert 0.0 $tclDevData(files,$tclDevData(curFile),text)
    $TxtWidget mark set insert $tclDevData(files,$tclDevData(curFile),cursor)

    if {$tclDevData(files,$tclDevData(curFile),selFirst) != ""} {
        $TxtWidget tag add sel \
            $tclDevData(files,$tclDevData(curFile),selFirst) \
            $tclDevData(files,$tclDevData(curFile),selLast)
    }

    applyMarks
    projectWindow::applyMarks $tclDevData(curFile)

    colorize
    
    $TxtWidget see insert
    ReadCursor
    
    return
}

# read changes from text widget to the internal buffer
proc applyChanges {} {
    global tclDevData
    variable TxtWidget

    if {$tclDevData(curFile) != ""} {
        set text [$TxtWidget get 0.0 end]
        if {$text != $tclDevData(files,$tclDevData(curFile),text)} {
            set tclDevData(files,$tclDevData(curFile),isChanged) 1
        }

        set tclDevData(files,$tclDevData(curFile),text) [$TxtWidget get 0.0 end]
        set tclDevData(files,$tclDevData(curFile),cursor) [$TxtWidget index insert]
        if [catch {$TxtWidget index sel.first}] {
            set tclDevData(files,$tclDevData(curFile),selFirst) ""
            set tclDevData(files,$tclDevData(curFile),selLast) ""
        } else {
            set tclDevData(files,$tclDevData(curFile),selFirst) [$TxtWidget index sel.first]
            set tclDevData(files,$tclDevData(curFile),selLast) [$TxtWidget index sel.last]
        }

        ReadMarks $tclDevData(curFile)
        applyMarks
        projectWindow::applyMarks $tclDevData(curFile)
        #colorize
        
    }

    return   
}

#is file Changed?
proc fileChanged {fileName} {
    global tclDevData

    variable TxtWidget

    if $tclDevData(files,$fileName,isChanged) {
        return 1
    }

    if {$fileName == $tclDevData(curFile)} {
        if {$tclDevData(files,$tclDevData(curFile),text) != [$TxtWidget get 0.0 end]} {
            return 1
        } 
    }

    return 0
}

# reaction on closing file.
proc deleteFile {fileName {newFile ""}} {
    global tclDevData
    variable TxtWidget

    if {$newFile == ""} {
        #delete Text Widget
        DeleteTextWidget

        #reset curFile variable
        set tclDevData(curFile) ""
    } else {
        # set new File
        SetFile $newFile

        # set focus to text widget
        focus $TxtWidget
    }

    return
}

# edit-copy
proc copy {} {
    variable TxtWidget

    if [catch {$TxtWidget index sel.first}] {
        return
    }
    
    set lineStart [lindex [split [$TxtWidget index sel.first] "."] 0]
    set lineEnd [lindex [split [$TxtWidget index sel.last] "."] 0]
    
    tk_textCopy $TxtWidget
    
    ReadCursor
    ColorizeLines $lineStart $lineEnd
    
    return
}

# edit-cut
proc cut {} {
    variable TxtWidget

    if {$TxtWidget == ""} {
        return
    }

    tk_textCut $TxtWidget

    applyChanges
    ReadCursor
    
    set lineNum [lindex [split [$TxtWidget index insert] "."] 0]

    ColorizeLines $lineNum $lineNum
    
    return
}

# edit-paste
proc paste {} {
    global tcl_platform
    variable TxtWidget

    if {$TxtWidget == ""} {
        return
    }

    if {"$tcl_platform(platform)" == "unix"} {
        catch { $TxtWidget delete sel.first sel.last }
    }

    set lineStart [lindex [split [$TxtWidget index insert] "."] 0]

    tk_textPaste $TxtWidget
    $TxtWidget see insert

    applyChanges
    ReadCursor
    
    set lineEnd [lindex [split [$TxtWidget index insert] "."] 0]
    ColorizeLines $lineStart $lineEnd
    
    return
}

# edit-delete
proc delete {} {
    global tcl_platform
    variable TxtWidget

    if {$TxtWidget == ""} {
        return
    }

    catch { $TxtWidget delete sel.first sel.last }

    applyChanges
    ReadCursor
    set lineNum [lindex [split [$TxtWidget index insert] "."] 0]

    ColorizeLines $lineNum $lineNum
    
    return
}

proc selectAll {} {
    variable TxtWidget

    if {$TxtWidget == ""} {
        return
    }
        
    $TxtWidget tag add sel 0.0 end
}

# set cursor to the function
proc gotoMark { markName } {    
    variable TxtWidget

    $TxtWidget mark set insert $markName
    $TxtWidget see insert
    focus $TxtWidget
    ReadCursor 0
}

proc gotoProc {procName} {
    variable TxtWidget
    
    set result [$TxtWidget search -regexp -- "^( |\t)*proc( |\t)+$procName" insert]
    
    if {$result != ""} {
        $TxtWidget mark set insert $result
        $TxtWidget see insert
        focus $TxtWidget
        ReadCursor 0
    }
    
    return
}

# parse file and create proc file
proc ReadMarks { fileName } {
    global tclDevData

    # clear all marks in this file
    foreach name [array names tclDevData files,$tclDevData(curFile),marks,] {
        unset $tclDevData($name)
    }

    set tclDevData(files,$fileName,marks) {}
    
    set lineNum 1
    #gets lines from text
    set lines [split $tclDevData(files,$fileName,text) '\n']
    
    # for each line
    foreach line $lines {
        # is it procedure definition
        set a ""
        set b ""
        regexp {^( |\t)*proc( |\t)+([a-z]|[A-Z]|[0-9]|_|:)+} $line a
        regexp {^( |\t)*proc( |\t)+} $line b
    
        if {$a != ""} {
            #read name of proc
            set nameStart [string length $b]
            set name [string range $a [string length $b] end]
            
            set markName procName
            append markName $lineNum
            
            lappend tclDevData(files,$fileName,marks) $markName
            set tclDevData(files,$fileName,marks,$markName,lineNum) $lineNum
            set tclDevData(files,$fileName,marks,$markName,name) $name
        }
    
        incr lineNum
    }

    return
}

# set procedure marks into the text widget
proc applyMarks { } {
    global tclDevData
    variable TxtWidget

    # delete old marks
    foreach markName [$TxtWidget mark names] {
        if {$markName != "insert"} {
            $TxtWidget mark unset $markName
        }
    }

    # set all marks
    foreach markName $tclDevData(files,$tclDevData(curFile),marks) {
        $TxtWidget mark set $markName \
            $tclDevData(files,$tclDevData(curFile),marks,$markName,lineNum).0
    }
}


# make indentation
proc Indent { } {
    variable TxtWidget
    global tclDevData
    
    set lineNum [lindex [split [$TxtWidget index insert] "."] 0]

    if {!$tclDevData(options,useIndent)} {
        return
    }
    
    if {$lineNum > 0} {
        #get text of prev line
        set prevLineNum [expr {$lineNum - 1}]
        set text [$TxtWidget get $prevLineNum.0 "$prevLineNum.0 lineend"]

        #count first spaces
        set spaces ""
        regexp "^\ *" $text spaces

        $TxtWidget insert insert $spaces

        if {!$tclDevData(options,useSintaxIndent)} {
            return
        }
        
        set spaces ""
        for {set i 0} {$i < $tclDevData(options,indentSize)} {incr i} {
            append spaces " "
        }
        
        # is last char on prev line - open brace
        set lastChar [$TxtWidget get "$prevLineNum.0 lineend -1 char" \
                                     "$prevLineNum.0 lineend"]
        if {$lastChar == "\{"} {
            #insert 4 spaces
            $TxtWidget insert insert $spaces
        }
    }

    return
}

# change tab bu spaces
proc OnTabPress {} {
    variable TxtWidget
    global tclDevData
    
    if {$tclDevData(options,changeTabs)} {
        set spaces ""
        
        for {set i 0} {$i < $tclDevData(options,tabSize)} {incr i} {
            append spaces " "
        }
        
        #insert spaces
        $TxtWidget insert insert $spaces
    } else {
        #insert tab
        $TxtWidget insert insert "\t"
    }
}

# unindent for close brace
proc OnCloseBrace {} {
    variable TxtWidget
    global tclDevData
    
    if {!$tclDevData(options,useSintaxIndent)} {
        return
    }

    # check, is it first character in the line?
    set text [$TxtWidget get "insert linestart" insert]
    set lt [string length $text]

    #count first spaces
    set spaces ""
    regexp "^\ *" $text spaces

    set l [string length $spaces]

    if {$spaces == $text && $l >= 4} {
        # delete four spaces
        $TxtWidget delete "insert -4 char" insert
    }

    return
}

# reaction on key releasing
proc OnKeyRelease {} {
    ReadCursor
    ColorizeCurLine
}

# reasction on mouse button release
proc OnMouseRelease {} {
    ReadCursor
}

# read information about cursor and set it to the global variables
proc ReadCursor {{selectProc 1}} {
    variable TxtWidget
    global tclDevData
    
    set insertPos [split [$TxtWidget index insert] "."]
    set tclDevData(cursor,line) [lindex $insertPos 0]
    set tclDevData(cursor,pos) [expr {[lindex $insertPos 1] + 1}]

    if $selectProc {
        # -1 for skipping some apecial marks, such as "anchor"    
        set procName [$TxtWidget mark previous "insert -1 char"]
    
        if {$procName != ""} {
            projectWindow::selectProc $tclDevData(curFile) $procName
        }
    }
   
    return
}

proc find {text {options ""}} {
    variable TxtWidget

    if {$text != "" && $TxtWidget != ""} {
        # get start position for searching
        if [catch {$TxtWidget index sel.first}] {
            set startPos insert
        } else {
            if {[string first $options "-backward"] != -1} {
                set startPos [$TxtWidget index sel.first]
            } else {
                set startPos [$TxtWidget index sel.last]
            }
        }
        
        set result [eval "$TxtWidget search -count l $options -- \"$text\" $startPos"]
    
        if {$result != ""} {
            $TxtWidget mark set insert "$result +$l chars"
            $TxtWidget mark set anchor "$result"
            $TxtWidget tag remove sel 0.0 end
            $TxtWidget tag add sel "insert -$l chars" insert
            $TxtWidget see insert
            focus $TxtWidget
            ReadCursor
        }
    }
    
    return
}

proc findNext {} {
    global tclDevData
    
    variable TxtWidget
    
    if [llength $tclDevData(find)] {
        find [lindex $tclDevData(find) 0] $tclDevData(find,lastOptions)
    }
    
    return
}

proc OnFind {} {
    variable TxtWidget

    #is has selection
    if {![catch {$TxtWidget index sel.first}]} {
        set text [$TxtWidget get [$TxtWidget index sel.first] [$TxtWidget index sel.last]]
    } else {
        # get current word
        set text [$TxtWidget get "insert wordstart" "insert wordend"]
    }
    
    mainFrame::find $text
}

proc OnReplace {} {
    variable TxtWidget

    # get current word
    set text [$TxtWidget get "insert wordstart" "insert wordend"]
    
    mainFrame::replace $text
}

proc enableHL {} {
    variable TxtWidget
    
    if {$TxtWidget != ""} {
        colorize
    }
    
    return
}

proc disableHL {} {
    variable TxtWidget
    
    if {$TxtWidget != ""} {
        # delete all tags
        $TxtWidget tag delete comment
        $TxtWidget tag delete keyword

        ConfigureTags
    }
    
    return
}

proc colorize {} {
    variable TxtWidget
    global tclDevData
    
    if {$TxtWidget == ""} {
        error "Internal error"
    }
    
    if {!$tclDevData(options,useHL)} {
        return
    }
    
    # get text from widget
    set text [split [$TxtWidget get 0.0 end] "\n"]

    set lineNum 1
    
    # delete all tags
    $TxtWidget tag delete comment
    $TxtWidget tag delete keyword

    ConfigureTags
    
    # for each line    
    foreach line $text {
        
        #is first symbol (except spaces and tabs) - comment symbol ?
        if [IsComment $line] {
            $TxtWidget tag add comment "$lineNum.0" "$lineNum.0 lineend"
        } else {
            set l [GetKeywordCoord $line]
            
            if {$l != {} } {
                set nameStart [lindex $l 0]
                set nameEnd [lindex $l 1]
                $TxtWidget tag add keyword "$lineNum.$nameStart" "$lineNum.$nameEnd"
            }
        }
        
        incr lineNum
    }
    
    return
}

# reaction on comment symbol
proc ColorizeCurLine {} {
    variable TxtWidget
    global tclDevData
    
    if {!$tclDevData(options,useHL)} {
        return
    }
    
    #get line
    set line [$TxtWidget get "insert linestart" "insert lineend"]
    
    if [IsComment $line] {
        # this is comment
        #set comment font
        $TxtWidget tag add comment "insert linestart" "insert lineend"
        $TxtWidget tag remove keyword "insert linestart" "insert lineend"
    } else {
        $TxtWidget tag remove comment "insert linestart" "insert lineend"

        set l [GetKeywordCoord $line]
            
        if {$l != {} } {
            set nameStart [lindex $l 0]
            set nameEnd [lindex $l 1]
            
            #get lineNum
            set lineNum [lindex [split [$TxtWidget index insert] "."] 0]
            
            $TxtWidget tag add keyword "$lineNum.$nameStart" "$lineNum.$nameEnd"
        } else {
            $TxtWidget tag remove keyword "insert linestart" "insert lineend"
        }
    }

    return    
}

proc ColorizeLines {lineStart lineEnd} {
    variable TxtWidget
    
    # delete all tags
    $TxtWidget tag remove comment "$lineStart.0" "$lineEnd.0 lineend"
    $TxtWidget tag remove keyword "$lineStart.0" "$lineEnd.0 lineend"

    for {set lineNum $lineStart} {$lineNum <= $lineEnd} {incr lineNum} {
        
        # get line
        set line [$TxtWidget get "$lineNum.0" "$lineNum.0 lineend"]
        
        #is first symbol (except spaces and tabs) - comment symbol ?
        if [IsComment $line] {
            $TxtWidget tag add comment "$lineNum.0" "$lineNum.0 lineend"
        } else {
            set l [GetKeywordCoord $line]
            
            if {$l != {} } {
                set nameStart [lindex $l 0]
                set nameEnd [lindex $l 1]
                $TxtWidget tag add keyword "$lineNum.$nameStart" "$lineNum.$nameEnd"
            }
        }
    }
    
    return
}

proc ConfigureTags {} {
    variable TxtWidget
    global tclDevData

    # blue is specially for Lapshin
    $TxtWidget tag configure comment -font $tclDevData(options,fonts,commentFont) -foreground blue
    $TxtWidget tag configure keyword -font $tclDevData(options,fonts,keywordFont)

    return    
}

proc IsComment { line } {
    set a ""
    regexp {^( |\t)*\#} $line a
        
    if {$a != ""} {
        return 1
    } else {
        return 0
    }
}

proc GetKeywordCoord {line} {
    global tclDevData
    
    set a ""
    regexp {^( |\t)*[a-z|A-Z|_]+} $line a
            
    if {$a != ""} {
        # gets name
        set b ""
        regexp {^( |\t)*} $line b
        set nameStart [string length $b]
        set nameEnd [string length $a]
        set name [string range $a [string length $b] end]
                
        # is it keyword?
        if {[lsearch $tclDevData(keywords) $name] != -1} {
            lappend l $nameStart
            lappend l $nameEnd
            return $l
        }
    } else {
        # check for else
        set a ""
        regexp {^( |\t)*\}( |\t)*else( |\t)*\{} $line a
        
        if {$a != ""} {
            set b ""
            regexp {^( |\t)*\}( |\t)*} $line b
            
            lappend l [string length $b]
            lappend l [expr {[string length $b] + 4}]
            
            return $l
        }
        
        # check for elseif
        set a ""
        regexp {^( |\t)*\}( |\t)*elseif( |\t)*\{} $line a
        
        if {$a != ""} {
            set b ""
            regexp {^( |\t)*\}( |\t)*} $line b
            
            lappend l [string length $b]
            lappend l [expr {[string length $b] + 6}]
            
            return $l
        }
    }
 
    return {}
}

proc setCursor {lineNum pos} {
    variable TxtWidget
    
    $TxtWidget mark set insert $lineNum.$pos
    $TxtWidget see insert
    focus $TxtWidget
    ReadCursor
    
    return
}

proc replace {findStr replaceStr in options} {
    variable TxtWidget
    
    if {$findStr != "" && $TxtWidget != "" && 
        ($in == "file" || ![catch {$TxtWidget index sel.first}])} {
        
        #determine start pos
        if {$in == "file"} {
            set startPos insert
        } else {
            set startPos sel.first
        }
        
        set result [eval "$TxtWidget search -count l $options -- \"$findStr\" $startPos"]
    
        if {$result != "" } {
            # check , what it is current result
            if {$in != "file" && [lsearch [$TxtWidget tag names $result] sel] == -1} {
                return
            }
            
            #delete old string
            $TxtWidget delete $result "$result +$l chars"
            #insert new string
            $TxtWidget insert $result $replaceStr
            
            # set cursor
            $TxtWidget mark set insert "$result +[string length $replaceStr] chars"
            $TxtWidget see insert
            focus $TxtWidget
            ReadCursor
        }
    }
    
    return
}

proc replaceAll {findStr replaceStr in options} {
    variable TxtWidget

    set count 0
        
    if {$findStr != "" && $TxtWidget != "" && 
        ($in == "file" || ![catch {$TxtWidget index sel.first}])} {
        
        #determine start pos
        if {$in == "file"} {
            set startPos insert
        } else {
            set startPos sel.first
        }
        
        set result [eval "$TxtWidget search -count l $options -- \"$findStr\" $startPos"]
    
        while {$result != "" } {
            # check , what it is current result
            if {$in != "file" && [lsearch [$TxtWidget tag names $result] sel] == -1} {
                return
            }
            
            #delete old string
            $TxtWidget delete $result "$result +$l chars"
            #insert new string
            $TxtWidget insert $result $replaceStr
            
            # set cursor
            $TxtWidget mark set insert "$result +[string length $replaceStr] chars"
            
            incr count
            
            set result [eval "$TxtWidget search -count l $options -- \"$findStr\" insert"]
        }
        
        $TxtWidget see insert
        focus $TxtWidget
        ReadCursor
    }

    statusText::setText "$count occurrence(s) have been replaces."
    
    return
}

#reaction on changing tab size
proc onTabSize {} {
    variable TxtWidget
    global tclDevData
    
    if {$TxtWidget != ""} {
        set size [expr {$tclDevData(options,tabSize)*
                    [font measure $tclDevData(options,fonts,editorFont) -displayof $TxtWidget " "]}]
        $TxtWidget configure -tabs [list $size]
    }
    
    return
}

# reaction on change font
proc onFontChange {} {
    variable TxtWidget
    global tclDevData
    
    if {$TxtWidget != ""} {
        $TxtWidget configure -font $tclDevData(options,fonts,editorFont)
        ConfigureTags
    }
    
    return
}

}








