##############################################################################
#    TCL Developer Studio
#
#    Copyright (C) 1999  Alexey Kakunin
#    small@star.spb.ru
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
##############################################################################

namespace eval projectWindow {

namespace export create addFile fileBrowse selectFile deleteFile renameFile
namespace export addProc deleteProc selectProc applyMarks closeFile

variable This
variable FilesTree

# projectWindow::create --
#
#        Create main frame of TCL DevStudio
#
#        Arguments:
#            this        frame for storing project tabs
#
#        Results: (no results)
proc create {this} {
    global text

    variable This
    variable FilesTree

    set This $this

    # create notebook
    tixNoteBook $This.nb
    $This.nb subwidget nbframe config -font nbFrameFont -tabpadx 0 -tabpady 0
    pack $This.nb -side top -fill both -expand 1
    
    # add tabs
#    set project [$This.nb add project -label "Proj" -underline 0]
#    set proc [$This.nb add proc -label "Proc" -underline 1]
    set files [$This.nb add files -label "Files" -underline 0]
#    set info [$This.nb add info -label "Info" -underline 0]


    # Create trees
#    tixTree $project.tree
#    pack $project.tree -expand 1 -fill both
    
#    tixTree $proc.tree
#    pack $proc.tree -expand 1 -fill both

    set FilesTree [tixTree $files.tree -options { 
        separator ";"
        }]
    $FilesTree config -browsecmd "projectWindow::fileBrowse"

    pack $files.tree -expand 1 -fill both

#    tixTree $info.tree
#    pack $info.tree -expand 1 -fill both

    # set Files as activ tab
    $This.nb raise files
    
    return
}

# projectWindow::addFile --
#
#       Open file. Add it to the Files Tree
#    Arguments:
#           fileName - name of File
#
#       Results: (no results)

proc addFile {fileName {isProject 0}} {
    variable FilesTree
    global image

    set hlist [$FilesTree subwidget hlist]

    #get file name from path
    set name [lindex [split $fileName "/"] end]

    # add file to the Files tree
    if $isProject {
        set picture [image create pixmap -data $image(project)]
    } else {
        set picture [tix getimage textfile]
    }
        
    $hlist add  $fileName -itemtype imagetext -text $name \
        -image $picture
    # select this entry
    selectFile $fileName
    
    $FilesTree autosetmode

    return
}

# projectWindow::fileBrowse --
#
#       Browse files Tree
#    Arguments:
#           dir - selected directory
#
#       Results: (no results)
proc fileBrowse {dir} {
    variable FilesTree

    set ldir [split $dir ";"]

    set fileName [lindex $ldir 0]
    editorWindows::selectFile $fileName

    if {[llength $ldir] == 2} {
        # this is function
        editorWindows::gotoMark [lindex $ldir 1]
    }
}

# selecting file in tree
proc selectFile {fileName} {
    variable FilesTree

    set hlist [$FilesTree subwidget hlist]

    # clear old selection
    $hlist anchor clear
    $hlist selection clear

    # set new selection
    $hlist anchor set $fileName
    $hlist selection set $fileName

    $hlist see $fileName
}

#delete file from tree
proc deleteFile {fileName {newFile ""}} {
    variable FilesTree

    set hlist [$FilesTree subwidget hlist]
    $hlist delete entry $fileName

    if {$newFile != ""} {
        selectFile $newFile
    }

    return
}

# rename file in tree
proc renameFile {oldName newName} {
    variable FilesTree

    set hlist [$FilesTree subwidget hlist]

    # find prev element    
    set prev $oldName
    
    while {$prev != "" && 
            ([llength [split $prev ";"]] != 1 || $prev == $oldName)} {
        set prev [$hlist info prev $prev]
    }

    set restoreSelection 0

    if {[$hlist info anchor] == $oldName ||
        [$hlist info selection] == $oldName} {
        set restoreSelection 1
    }

    #delete old entry
    $hlist delete entry $oldName

    #get file name from path
    set name [lindex [split $newName "/"] end]

    if {$prev == ""} {
        $hlist add  $newName -itemtype imagetext -text $name \
            -image [tix getimage textfile] -at 0

    } else {
        $hlist add  $newName -itemtype imagetext -text $name \
            -image [tix getimage textfile] -after $prev
    }

    if $restoreSelection {
        # set new selection
        $hlist anchor set $newName
        $hlist selection set $newName
    }
}

# add procedure to the tree
proc addProc { fileName procName markName lineNum} {
    global tclDevData
    global image
    variable FilesTree

    set hlist [$FilesTree subwidget hlist]

    set entryName $fileName
    append entryName ";"
    append entryName $markName
    
    # find place for this procedure
    foreach childName [$hlist info children $fileName] {
        # get proc name
        set childProcName [lindex [split $childName ";"] 1]

        # get line number for it prcedure
        set procLineNum $tclDevData(files,$fileName,marks,$childProcName,lineNum)

        if {$lineNum < $procLineNum} {
            # insert before it procedure
            $hlist add $entryName -itemtype imagetext -text $procName \
                -image [image create pixmap -data $image(proc)] -before $childName
            $FilesTree autosetmode
            return
        }
    }

    
    $hlist add $entryName -itemtype imagetext -text $procName \
        -image [image create pixmap -data $image(proc)]

    $FilesTree autosetmode

    return
}

# apply procedures marks
proc applyMarks { fileName } {
    global tclDevData
    variable FilesTree

    set hlist [$FilesTree subwidget hlist]

    set children {}

    # delete all unuset procedures
    foreach childName [$hlist info children $fileName] {
        # get mark name
        set markName [lindex [split $childName ";"] 1]

        #try to search it in marks
        if {[lsearch $tclDevData(files,$fileName,marks) $markName] == -1} {
            #delete it
            deleteProc $fileName $markName
        } else {
            lappend children $markName
        }
    }

    # apply all nonexisted
    foreach markName $tclDevData(files,$fileName,marks) {
        # try to found it in childrens
        if {[lsearch $children $markName] == -1} {
            #add it
            addProc $fileName \
                    $tclDevData(files,$fileName,marks,$markName,name) \
                    $markName \
                    $tclDevData(files,$fileName,marks,$markName,lineNum)
        }
    }
}

#delete procedure from the tree
proc deleteProc { fileName markName } {
    global tclDevData
    variable FilesTree

    set hlist [$FilesTree subwidget hlist]

    # make entry name
    set entryName $fileName
    append entryName ";"
    append entryName $markName

    $hlist delete entry $entryName

    return
}

# select procedure
proc selectProc { fileName procName } {
    variable FilesTree
    
    set hlist [$FilesTree subwidget hlist]

    set name $fileName
    append name ";"
    append name $procName
    
    if [$hlist info exists $name] {
        $FilesTree open $fileName
        
        $hlist anchor clear
        $hlist selection clear
        $hlist anchor set $name
        $hlist selection set $name
        
        $hlist see $name
    }
    
    return    
}

proc closeFile {fileName} {
    variable FilesTree
    
    $FilesTree close $fileName
}

}




















