##############################################################################
#    TCL Developer Studio
#
#    Copyright (C) 1999  Alexey Kakunin
#    small@star.spb.ru
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
##############################################################################

namespace eval mainFrame {

namespace export create enable disable onCreateText onDestroyText find replace openTextFile
namespace export onFileNew onFileOpen onFileSave goTo openFile
namespace export getGeometry getProjectWinSize getOutputWinSize
namespace export setProjectName

variable MainFrame
variable MenuFrame
variable ToolbarFrame
variable StatusFrame
variable PaneWindow1
variable PaneWindow2
variable RecentFilesMenu
variable RecentProjectsMenu

# mainFrame::create --
#
#        Create main frame of TCL DevStudio
#
#        Arguments: (no arguments)
#
#        Results: (no results)
proc create {} {
    global titlepane
    global text
    global tclDevData

    variable MainFrame
    variable StatusLabel
    variable PaneWindow1
    variable PaneWindow2

    # wait closing titlepane
    if {[winfo exist $titlepane]} {
        tkwait window $titlepane
    }

    # create frame
    set MainFrame .mainFrame
    toplevel $MainFrame

    # set geometry
    if [info exist tclDevData(positions,mainFrame)] {
        wm geometry $MainFrame $tclDevData(positions,mainFrame)
    }
    
    # set wm communication
    wm title $MainFrame "TCL Developer Studio"
    wm protocol $MainFrame WM_DELETE_WINDOW mainFrame::exit

    # Create Statusbar
    mainFrame::CreateStatusBar
    
    # Create menu for main frame
    mainFrame::CreateMenu
    # Create toolbar for main frame
    mainFrame::CreateToolBar

    # create paned windows
    tixPanedWindow $MainFrame.pane
    set PaneWindow1 $MainFrame.pane
    pack $MainFrame.pane -side top -expand 1 -fill both -padx 5 -pady 5

    if [info exist tclDevData(positions,mainFrame)] {
        set topWin [$MainFrame.pane add topWin -expand 1.0]
    } else {
        set topWin [$MainFrame.pane add topWin -size 200 -expand 1.0]
    }
    
    if [info exist tclDevData(positions,outputWinSize)] {
        set outWin [$MainFrame.pane add outWin -min 40 -size $tclDevData(positions,outputWinSize)]
    } else {
        set outWin [$MainFrame.pane add outWin -min 40 -size 120]
    }

    # Create new pane window in top window
    tixPanedWindow $topWin.pane -orientation horizontal
    set PaneWindow2 $topWin.pane
    pack $topWin.pane -side top -expand 1 -fill both

    #create project window and editors window
    if [info exist tclDevData(positions,projectWinSize)] {
        set projWin [$topWin.pane add project -size $tclDevData(positions,projectWinSize)]
    } else {
        set projWin [$topWin.pane add project -size 150]
    }
    
    if [info exist tclDevData(positions,mainFrame)] {
        set editsWin [$topWin.pane add editors -expand 1.0]
    } else {
        set editsWin [$topWin.pane add editors -size 200 -expand 1.0]
    }

    #Create sub-windows
    projectWindow::create $projWin
    editorWindows::create $editsWin
    outputWindow::create $outWin
    
    return
}

# mainFrame::exit --
#
#        Exit from TCL Developer Studio
#
#        Arguments: (no arguments)
#
#        Results: (no results)

proc exit {} {

    # save options
    options::save
    
    # close all files
    if [CloseAll] {
        ::exit
    }
}

# mainFrame::CreateMenu --
#
#        Exit from TCL Developer Studio
#
#        Arguments: (no arguments)
#
#        Results: (no results)
proc CreateMenu {} {
    global image

    variable MainFrame
    variable MenuFrame
    variable RecentFilesMenu
    variable RecentProjectsMenu
    
    #create frame for menu
    set MenuFrame $MainFrame.menu
    frame $MenuFrame
    pack $MenuFrame -side top -anchor w

    menubutton $MenuFrame.file -text "File" -underline 0 -menu $MenuFrame.file.m
    help::add $MenuFrame.file "File Operations"
    menu $MenuFrame.file.m -tearoff 0
    $MenuFrame.file.m add command -label "New..." -underline 0  -command "mainFrame::onFileNew" -accelerator "Ctrl-N"
    $MenuFrame.file.m add command -label "Open" -underline 0 -command "mainFrame::onFileOpen" -accelerator "Ctrl-O"
    $MenuFrame.file.m add command -label "Close" -underline 0 -state disabled -command "mainFrame::OnFileClose" 
    $MenuFrame.file.m add separator
    $MenuFrame.file.m add command -label "Save" -underline 0 -state disabled -command "mainFrame::onFileSave" -accelerator "Ctrl-S"
    $MenuFrame.file.m add command -label "Save As..." -underline 5 -state disabled -command "mainFrame::OnFileSaveAs"
    $MenuFrame.file.m add command -label "Save All" -underline 6 -state disabled -command "mainFrame::OnSaveAll"
    $MenuFrame.file.m add separator
    set RecentFilesMenu $MenuFrame.file.m.recentFiles
    $MenuFrame.file.m add cascade -label "Recent Files" -menu $RecentFilesMenu
    set RecentProjectsMenu $MenuFrame.file.m.recentProjects
    $MenuFrame.file.m add cascade -label "Recent Projects" -menu $RecentProjectsMenu
    UpdateRecentFilesMenu
    UpdateRecentProjectsMenu
    $MenuFrame.file.m add separator
    $MenuFrame.file.m add command -label "Exit" -underline 1 -command "mainFrame::exit"
    
    menubutton $MenuFrame.edit -text "Edit" -underline 0 -menu $MenuFrame.edit.m
    help::add $MenuFrame.edit "Edit Operations"
    menu $MenuFrame.edit.m -tearoff 0
    $MenuFrame.edit.m add command -label "Undo" -underline 0 -state disabled
    $MenuFrame.edit.m add command -label "Redo" -underline 0 -state disabled
    $MenuFrame.edit.m add separator
    $MenuFrame.edit.m add command -label "Cut" -underline 2 -state disabled -command "mainFrame::OnEditCut" -accelerator "Ctrl-X"
    $MenuFrame.edit.m add command -label "Copy" -underline 0 -state disabled -command "mainFrame::OnEditCopy" -accelerator "Ctrl-C"
    $MenuFrame.edit.m add command -label "Paste" -underline 0 -state disabled -command "mainFrame::OnEditPaste" -accelerator "Ctrl-V"
    $MenuFrame.edit.m add command -label "Delete" -underline 0 -state disabled -command "mainFrame::OnEditDelete" -accelerator "Del"
    $MenuFrame.edit.m add separator
    $MenuFrame.edit.m add command -label "Select All" -underline 8 -state disabled -command "editorWindows::selectAll" -accelerator "Ctrl-A"
    $MenuFrame.edit.m add separator
    $MenuFrame.edit.m add command -label "Find" -underline 0 -state disabled -command "mainFrame::OnEditFind" -accelerator "Ctrl-F"
    $MenuFrame.edit.m add command -label "Find Next" -underline 0 -state disabled -command "mainFrame::OnEditFindNext" -accelerator "Ctrl-L"
    $MenuFrame.edit.m add command -label "Find in Files..." -underline 1 -command "mainFrame::OnEditFindInFiles"
    $MenuFrame.edit.m add command -label "Replace" -underline 1 -state disabled -command "mainFrame::OnEditReplace" -accelerator "Ctrl-H"
    $MenuFrame.edit.m add separator
    $MenuFrame.edit.m add command -label "Go To..." -underline 0 -state disabled -accelerator "Ctrl-G" -command "mainFrame::goTo"
    $MenuFrame.edit.m add separator
    $MenuFrame.edit.m add cascade -label "Advanced" -underline 0 -menu $MenuFrame.edit.m.adv -state disabled
        menu $MenuFrame.edit.m.adv -tearoff 0
        $MenuFrame.edit.m.adv add command -label "Format Selection" -underline 0
        $MenuFrame.edit.m.adv add command -label "Tabify Selection" -underline 0
        $MenuFrame.edit.m.adv add command -label "Untabify Selection" -underline 1
        $MenuFrame.edit.m.adv add command -label "Make Selection Uppercade" -underline 15
        $MenuFrame.edit.m.adv add command -label "Make Selection Lowercade" -underline 15

    menubutton $MenuFrame.project -text "Project" -underline 0 -menu $MenuFrame.project.m
    help::add $MenuFrame.project "Project Operations"
    menu $MenuFrame.project.m -tearoff 0
    $MenuFrame.project.m add command -label "Run" -underline 0 -command "mainFrame::OnRun" -state disabled
    $MenuFrame.project.m add command -label "Debug" -underline 0 -command "mainFrame::OnDebug" -state disabled
    $MenuFrame.project.m add separator

    $MenuFrame.project.m add cascade -label "Add To Project" -underline 0 -menu $MenuFrame.project.m.add -state disabled
        menu $MenuFrame.project.m.add -tearoff 0
        $MenuFrame.project.m.add add command -label "New..." -underline 0
        $MenuFrame.project.m.add add command -label "New Folder..." -underline 2
        $MenuFrame.project.m.add add separator
        $MenuFrame.project.m.add add command -label "Files..." -underline 0
    $MenuFrame.project.m add separator
    $MenuFrame.project.m add cascade -label "Source Control" -underline 7 -menu $MenuFrame.project.m.cvs -state disabled
        menu $MenuFrame.project.m.cvs -tearoff 0
        $MenuFrame.project.m.cvs add command -label "Get Latest Version..." -underline 0
        $MenuFrame.project.m.cvs add command -label "Check Out..." -underline 6
        $MenuFrame.project.m.cvs add command -label "Check In..." -underline 6
        $MenuFrame.project.m.cvs add command -label "Undo Check Out..." -underline 0
        $MenuFrame.project.m.cvs add separator
        $MenuFrame.project.m.cvs add command -label "Add to Source Control..." -underline 0
        $MenuFrame.project.m.cvs add command -label "Remove from Source Control..." -underline 2
        $MenuFrame.project.m.add add separator
        $MenuFrame.project.m.cvs add command -label "Show History..." -underline 5
        $MenuFrame.project.m.cvs add command -label "Show Differences..." -underline 5
        $MenuFrame.project.m.cvs add command -label "CVS Properties..." -underline 4
        $MenuFrame.project.m.cvs add command -label "Refresh Status" -underline 0
    $MenuFrame.project.m add separator
    $MenuFrame.project.m add command -label "Settings..." -underline 0 -command "project::runDialog"
    
    menubutton $MenuFrame.build -text "Build" -underline 0 -menu $MenuFrame.build.m -state disabled
    help::add $MenuFrame.build "Build Operations"
    menu $MenuFrame.build.m -tearoff 0

    menubutton $MenuFrame.tools -text "Tools" -underline 0 -menu $MenuFrame.tools.m
    help::add $MenuFrame.tools "Tools Operations"
    menu $MenuFrame.tools.m -tearoff 0
        $MenuFrame.tools.m add command -label "Options" -command "options::runDialog"

    menubutton $MenuFrame.help -text "Help" -underline 0 -menu $MenuFrame.help.m -state disabled
    help::add $MenuFrame.help "Help Operations"
    menu $MenuFrame.help.m -tearoff 0

    foreach name {file edit project build tools help} {
        pack $MenuFrame.$name -side left -expand 1 -fill y
    }
    return
}


# mainFrame::CreateToolBar --
#
#        Exit from TCL Developer Studio
#
#        Arguments: (no arguments)
#
#        Results: (no results)
proc CreateToolBar {} {

    global image
    global tclDevData
    
    variable MainFrame
    variable ToolbarFrame

    #create frame for toolbar
    set ToolbarFrame $MainFrame.toolbar
    frame $ToolbarFrame
    pack $ToolbarFrame -side top -anchor w

    button $ToolbarFrame.filenew -image [image create pixmap -data $image(filenew)] -relief flat -padx 0 -pady 0 -command "mainFrame::NewTCLFile"
    help::add $ToolbarFrame.filenew "New TCL File"

    button $ToolbarFrame.0 -padx 0 -pady 0 -relief flat -state disabled

    button $ToolbarFrame.fileopen -image [image create pixmap -data $image(fileopen)] -relief flat -padx 0 -pady 0 -command "mainFrame::onFileOpen"
    help::add $ToolbarFrame.fileopen "Open"
    button $ToolbarFrame.filesave -image [image create pixmap -data $image(filefloppy)] -relief flat -padx 0 -pady 0 -command "mainFrame::onFileSave" -state disabled
    help::add $ToolbarFrame.filesave "Save"

    button $ToolbarFrame.1 -padx 0 -pady 0 -relief flat -state disabled

    button $ToolbarFrame.editcut -image [image create pixmap -data $image(editcut)] -relief flat -padx 0 -pady 0  -command "mainFrame::OnEditCut" -state disabled
    help::add $ToolbarFrame.editcut "Cut"
    button $ToolbarFrame.editcopy -image [image create pixmap -data $image(editcopy)] -relief flat -padx 0 -pady 0 -command "mainFrame::OnEditCopy" -state disabled
    help::add $ToolbarFrame.editcopy "Copy"
    button $ToolbarFrame.editpaste -image [image create pixmap -data $image(editpaste)] -relief flat -padx 0 -pady 0 -command "mainFrame::OnEditPaste" -state disabled
    help::add $ToolbarFrame.editpaste "Paste"

    button $ToolbarFrame.2 -padx 0 -pady 0 -relief flat -state disabled

    button $ToolbarFrame.undo -image [image create pixmap -data $image(undo)] -relief flat -padx 0 -pady 0 -state disabled
    help::add $ToolbarFrame.undo "Undo"
    button $ToolbarFrame.redo -image [image create pixmap -data $image(redo)] -relief flat -padx 0 -pady 0 -state disabled
    help::add $ToolbarFrame.redo "Redo"

    button $ToolbarFrame.3 -padx 0 -pady 0 -relief flat -state disabled
    button $ToolbarFrame.find -image [image create pixmap -data $image(find2)] -relief flat -padx 0 -pady 0 -command mainFrame::OnFindButton
    help::add $ToolbarFrame.find "Find"
    tixComboBox $ToolbarFrame.findStr -editable 1
    #create binding for updating listbox
    trace variable tclDevData(find) w mainFrame::UpdateFindStr
    
    #Do not use it. Because the Enter redirec to the edit Window.
#    bind [$ToolbarFrame.findStr subwidget entry] <Return> "mainFrame::OnFindButton; break"

    foreach name {filenew 0 
                  fileopen filesave 1 
                  editcut editcopy editpaste 2 
                  undo redo 3
                  find findStr} {
        pack $ToolbarFrame.$name -side left -padx 0
    }
    
    return
}

proc UpdateFindStr {name1 name2 op} {
    global tclDevData
    variable ToolbarFrame
    
    set lb [$ToolbarFrame.findStr subwidget listbox]
    
    # delete all entries in listbox
    $lb delete 0 end
    
    foreach l $tclDevData(find) {
        $lb insert end $l
    }

    if {$tclDevData(find) != {}} {
        $ToolbarFrame.findStr configure -value [lindex $tclDevData(find) 0]
    }
    
    return
}

# mainFrame::CreateStatusBar --
#
#        Sreate Status Bar at the bottom of the Main Frame.
#        Status bar used for displaying status string
#        and other usefull information
#
#        Arguments: (no arguments)
#
#        Results: (no results)

proc CreateStatusBar {} {
    global tclDevData

    variable MainFrame
    variable StatusFrame

    set StatusFrame [statusText::create $MainFrame]
    
    #create cursor position labels
    set tclDevData(cursor,line) 0
    set tclDevData(cursor,pos) 0
    
    label $StatusFrame.line -textvariable tclDevData(cursor,line) -width 6
    label $StatusFrame.d -text ":"
    label $StatusFrame.pos -textvariable tclDevData(cursor,pos) -width 6
    
    pack $StatusFrame.pos $StatusFrame.d $StatusFrame.line -side right
    
    bind $StatusFrame.line <Button> mainFrame::OnLineClick
    
    # create time field
    label $StatusFrame.time -width 5
    pack $StatusFrame.time -padx 3 -side right
    
    after 1000 mainFrame::UpdateTimeField

    return
}

proc UpdateTimeField {} {
    variable StatusFrame
    
    $StatusFrame.time configure -text [clock format [clock seconds] -format "%H:%M"]
    
    after 1000 mainFrame::UpdateTimeField
    
    return
}

proc OnLineClick {} {
    variable StatusFrame
    global tclDevData
    
    if {$tclDevData(curFile) != ""} {
        mainFrame::goTo [$StatusFrame.line cget -text]
    }
    
    return
}

# mainFrame::OnFileOpen --
#
#               Reaction on FileOpen Command.
#        Show Open File dialog
#        Arguments: (no arguments)
#
#        Results: (no results)
proc onFileOpen {} {
    global tclDevData

    # use native open dialog
    set types {
        {{TCL Scripts} {.tcl}}
        {{All files}    *    }
    }

    set fileName [tk_getOpenFile -filetypes $types]

    if {$fileName != ""} {
        openFile $fileName
    }
}

proc openFile {fileName} {
    set ext [file extension $fileName]
    switch $ext {
        .tcl {
            OpenTCLFile $fileName
        }

        default {
            openTextFile $fileName
        }
    }
    
    return
}

# This procedure simple open text file
proc openTextFile { fileName {isProject 0}} {
    global tclDevData

    if {[lsearch $tclDevData(files) $fileName] == -1} {
        # load new file
        LoadFile $fileName $isProject

        projectWindow::addFile $fileName $isProject
        editorWindows::openFile $fileName

        lappend tclDevData(files) $fileName
        AddToRecentFiles $fileName
    } else {
        # select already loaded file
        projectWindow::selectFile $fileName
        editorWindows::selectFile $fileName
    }

    return
}

# this procedure determine, is it project file and open file as project, or as tcl file
proc OpenTCLFile { fileName } {
    set isProject 0
    
    if {[IsProjectFile $fileName] && 
        [tk_messageBox -icon question \
                       -type yesno \
                       -title "TCL DevStudio" \
                       -message "Load $fileName as TCLDevStudio Project File?"] == "yes"} {
        if {![OpenProject $fileName]} {
            return 0
        }
        
        set isProject 1
    }
    
    # not implemented yet.
    openTextFile $fileName $isProject
    return
}

# Open TCL DevStudio Project
proc OpenProject { fileName } {
    global tclDevData
    
    if {$tclDevData(projectFile) != ""} {
        # close Project
        if {![closeFile $tclDevData(projectFile)]} {
            return 0
        }
    }
    
    project::loadFile $fileName

    enable Run
    enable Debug
    AddToRecentProjects $fileName
    set tclDevData(projectFile) $fileName
    
    return 1
}


# mainFrame::LoadFile --
#
#       Load File
#       Arguments: fileName - name of file for loading
#
#    Results: 1 - if OK, 0 - if Error
proc LoadFile {fileName {isProject 0}} {
    global tclDevData

    if [catch {set fileID [open $fileName "r"]}] {
        error "Can not load file"
    }

    lappend tclDevData(files,$fileName)
    
    set tclDevData(files,$fileName,text) [read $fileID]
    if {![info exist tclDevData(files,$fileName,cursor)]} {
        set tclDevData(files,$fileName,cursor) 0.0
    }
    if {![info exist tclDevData(files,$fileName,selFirst)]} {
        set tclDevData(files,$fileName,selFirst) ""
    }
    if {![info exist tclDevData(files,$fileName,selLast)]} {
        set tclDevData(files,$fileName,selLast) ""
    }
    set tclDevData(files,$fileName,isUntitled) 0
    set tclDevData(files,$fileName,isChanged) 0
    set tclDevData(files,$fileName,marks) {}
    set tclDevData(files,$fileName,isProject) $isProject

    close $fileID

    # convert all tabs to spaces
    if $tclDevData(options,changeTabs) {
        set index [string first "\t" $tclDevData(files,$fileName,text)]
    
        set spaces ""
        for {set i 0} {$i < $tclDevData(options,tabSize)} {incr i} {
            append spaces " "
        }
        
        while {$index != -1} {
            set text [string range $tclDevData(files,$fileName,text) 0 [expr {$index - 1}]]
            append text $spaces
            append text [string range $tclDevData(files,$fileName,text) [expr {$index + 1}] end]
        
            set tclDevData(files,$fileName,text) $text
        
            set index [string first "\t" $tclDevData(files,$fileName,text)]
        }
    }
    
    # delete all TCLDEV Comments
    if $isProject {
        set text [split $tclDevData(files,$fileName,text) "\n"]
        set lineNum 0
        foreach line $text {
            if {[string first "##TCLDevStudioProject" $line] == 0} {
                #remove it
                set text [lreplace $text $lineNum $lineNum]
            } else {
                incr lineNum
            }
        }
        
        set tclDevData(files,$fileName,text) [join $text "\n"]
    }
    
    return 1
}

# If file untitled, ask name and then save it on disk
proc SaveFile { fileName } {
    global tclDevData

    if $tclDevData(files,$fileName,isUntitled) {
        return [saveAs $fileName]
    } else {
        return [storeFile $fileName]
    }
}


# Save file on disk
proc storeFile { fileName } {
    global tclDevData

    if [catch {set fileID [open $fileName "w"]}] {
        return 0
    }

    puts -nonewline $fileID $tclDevData(files,$fileName,text)

    # is it project?
    if $tclDevData(files,$fileName,isProject) {
        #store project settings in the end  of file
        puts $fileID ""
        project::save $fileID
    }
    
    close $fileID

    set tclDevData(files,$fileName,isUntitled) 0
    set tclDevData(files,$fileName,isChanged) 0

    return 1
}

# Ask new name for file and save it
proc saveAs {fileName} {
    global tclDevData

    set types {
        {{TCL Scripts} {.tcl}}
        {{All files}    *    }
    }

    set newName [tk_getSaveFile -filetypes $types \
                                -title "Save $fileName as..."]

    if {$newName == ""} {
        return 0
    } else {
        RenameFile $fileName $newName

        return [storeFile $newName]
    }

}

# close file
proc closeFile { fileName } {
    global tclDevData

    editorWindows::applyChanges

    #always save project
    if $tclDevData(files,$fileName,isProject) {
        set reply [tk_messageBox -icon question \
                       -type yesnocancel \
                       -title "TCL DevStudio" \
                       -message "Save Project $fileName?"]

        # process reply
        switch $reply {
            "yes" {
                # save File
                if {[SaveFile $fileName] == 0} {
                    return 0
                }
            }
    
            "no" {
                # do not save file
            }

            "cancel" {
                # do not close file
                return 0
            }
        }
    } else {
        # is file was changed?
        if [editorWindows::fileChanged $fileName] {
            # ask for saving
            set reply [tk_messageBox -icon question \
                       -type yesnocancel \
                       -title "TCL DevStudio" \
                       -message "File $fileName was changed. Save it?"]

            # process reply
            switch $reply {
                "yes" {
                    # save File
                    if {[SaveFile $fileName] == 0} {
                        return 0
                    }
                }
    
                "no" {
                    # do not save file
                }

                "cancel" {
                    # do not close file
                    return 0
                }
            }
        }
    }

    # make closing

    # search file in list
    set index [lsearch $tclDevData(files) $fileName]
    # check result of searching
    if {$index == -1} {
        error "Internal Error"
    }

    # delete file from list
    set tclDevData(files) [lreplace $tclDevData(files) $index $index]

    # select another file
    if {$tclDevData(files) == {}} {
        set newFile ""
    } else {
        set newFile [lindex $tclDevData(files) 0]
    }
   
    projectWindow::deleteFile $fileName $newFile
    editorWindows::deleteFile $fileName $newFile

    if {$tclDevData(files,$fileName,isProject)} {
        set tclDevData(projectFile) ""
    }
    
    # delete information, associated with this file.
    unset tclDevData(files,$fileName,text)
    unset tclDevData(files,$fileName,isChanged)
    unset tclDevData(files,$fileName,isUntitled)
    unset tclDevData(files,$fileName,marks)
    unset tclDevData(files,$fileName,isProject)
    
    foreach name [array names tclDevData files,$fileName,marks,*] {
        unset tclDevData($name)
    }
    
    return 1
}

# Reaction on Select "Save" menu
proc onFileSave {} {
    global tclDevData

    if {$tclDevData(curFile) != ""} {
        editorWindows::applyChanges

        SaveFile $tclDevData(curFile)
    }
    
    return
}

# Reaction on select "Save As..." menu
proc OnSaveAll {} {
    global tclDevData

    # apply changes in current file
    if {$tclDevData(curFile) != ""} {
        editorWindows::applyChanges
    }

    # for all files
    foreach fileName $tclDevData(files) {
        # save file
        SaveFile $fileName
    }
}

# Reaction on selecting "Close"
proc OnFileClose {} {
    global tclDevData

    if {$tclDevData(curFile) != ""} {
        closeFile $tclDevData(curFile)
    }
}

# close all files
proc CloseAll {} {
    global tclDevData

    # for each file
    foreach name $tclDevData(files) {
        #close file
        if {[closeFile $name] == 0} {
            return 0
        }
    }

    return 1
}

# Reaction on selecting "New" menu
proc onFileNew {} {
    set result [newDialog::run]
    
    if {$result != {}} {
        array set newDialogData $result
        
        switch $newDialogData(what) {
            "file" {
                NewFile
            }
            "tcl" {
                NewTCLFile
            }
            "project" {
                NewProject
            }
        }
    }
    
    return
}

proc NewFile {} {
    global tclDevData

    #generate unique unknown
    for {set i 1} {1} {incr i} {
        #create file name
        set name "untitled"
        append name $i

        if {[lsearch $tclDevData(files) $name] == -1} {
            break
        }
    }

    #create data for this file
    lappend tclDevData(files) $name
    set tclDevData(files,$name,text) ""
    set tclDevData(files,$name,cursor) 0.0
    set tclDevData(files,$name,selFirst) ""
    set tclDevData(files,$name,selLast) ""
    set tclDevData(files,$name,isUntitled) 1
    set tclDevData(files,$name,isChanged) 0
    set tclDevData(files,$name,marks) {}
    set tclDevData(files,$name,isProject) 0

    projectWindow::addFile $name
    editorWindows::openFile $name
}

proc NewTCLFile {} {
    global tclDevData

    #generate unique unknown
    for {set i 1} {1} {incr i} {
        #create file name
        set name "untitled"
        append name $i
        append name ".tcl"

        if {[lsearch $tclDevData(files) $name] == -1} {
            break
        }
    }

    #create data for this file
    lappend tclDevData(files) $name
    set tclDevData(files,$name,text) ""
    set tclDevData(files,$name,cursor) 0.0
    set tclDevData(files,$name,selFirst) ""
    set tclDevData(files,$name,selLast) ""
    set tclDevData(files,$name,isUntitled) 1
    set tclDevData(files,$name,isChanged) 0
    set tclDevData(files,$name,marks) {}
    set tclDevData(files,$name,isProject) 0

    projectWindow::addFile $name
    editorWindows::openFile $name
}

proc NewProject {} {
    global tclDevData

    # close prev project
    if {$tclDevData(projectFile) != ""} {
        if {![closeFile $tclDevData(projectFile)]} {
            return
        }
    }
    
    #generate unique unknown
    for {set i 1} {1} {incr i} {
        #create file name
        set name "untitled"
        append name $i
        append name ".tcl"

        if {[lsearch $tclDevData(files) $name] == -1} {
            break
        }
    }

    #create data for this file
    lappend tclDevData(files) $name
    set tclDevData(files,$name,text) ""
    set tclDevData(files,$name,cursor) 0.0
    set tclDevData(files,$name,selFirst) ""
    set tclDevData(files,$name,selLast) ""
    set tclDevData(files,$name,isUntitled) 1
    set tclDevData(files,$name,isChanged) 0
    set tclDevData(files,$name,marks) {}
    set tclDevData(files,$name,isProject) 1
    
    set tclDevData(projectFile) $name

    projectWindow::addFile $name 1
    editorWindows::openFile $name
}

# Rename file. Used in SaveAs command
proc RenameFile {oldName newName} {
    global tclDevData

    set index [lsearch $tclDevData(files) $oldName]

    if {$index == -1} {
        error "Internal error"
    }

    set tclDevData(files) [lreplace $tclDevData(files) $index $index $newName]
    
    # rename all variables
    foreach name [array names tclDevData files,$oldName,*] {
       #change second field in variable
       set newArrayName [join [lreplace [split $name ","] 1 1 $newName] ","]

       set tclDevData($newArrayName) $tclDevData($name)
       unset tclDevData($name)
    }

    #rename curFile
    if {$tclDevData(curFile) == $oldName} {
        set tclDevData(curFile) $newName
    }

    projectWindow::renameFile $oldName $newName
    
    if {$tclDevData(projectFile) == $oldName} {
        set tclDevData(projectFile) $newName
    }
}

# Reaction on selecting "Save As" menu
proc OnFileSaveAs {} {
    global tclDevData

    editorWindows::applyChanges

    if {$tclDevData(curFile) != ""} {
        saveAs $tclDevData(curFile)
    }
}

# Reaction on selecting "Run" menu
proc OnRun {} {
    # save all files
    OnSaveAll

    if {[project::getDir] != ""} {
        cd [project::getDir]
    }

    if {[project::getRunCmd] != ""} {
        
        disable Run
        disable Debug
        
        # Execute
        set cmd [project::getRunCmd]
        
        # clear output window
        outputWindow::clearDebug
        
        set outputStream [open "| $cmd"]
        fconfigure $outputStream -blocking false
        fileevent $outputStream readable "outputWindow::onDebugOutput $outputStream"
    }

    return
}

# Reaction on selecting "Debug" menu
proc OnDebug {} {
    # save all files
    OnSaveAll

    if {[project::getDir] != ""} {
        cd [project::getDir]
    }

    if {[project::getDebugCmd] != ""} {
        
        disable Run
        disable Debug
        
        # Debug
        set cmd [project::getDebugCmd]
        eval "exec $cmd"
    }

    return
}

# Determine, is it file - TCLDevStudio project file
proc IsProjectFile { fileName } {
    if [catch {set fileID [open $fileName "r"]}] {
        error "Can not load project $fileName"
    }

    set line ""
    while {[gets $fileID line] != -1} {
        if { $line == "##TCLDevStudioProject" } {
            close $fileID
            return 1
        }
    }

    close $fileID
    return 0
}

# Reaction on edit-cut
proc OnEditCut {} {
    editorWindows::cut
}

# Reaction on edit-copy
proc OnEditCopy {} {
    editorWindows::copy
}

# Reaction on edit-paste
proc OnEditPaste {} {
    editorWindows::paste
}

# Reaction on edit-delete
proc  OnEditDelete {} {
    editorWindows::delete
}

proc enable {what} {
    variable MenuFrame
    variable ToolbarFrame
    
    switch $what {
        "Run" {
            if {[project::getRunCmd] != ""} {
                $MenuFrame.project.m entryconfigure 0 -state normal
            }
        }
            
        "Debug" {
            if {[project::getDebugCmd] != ""} {
                $MenuFrame.project.m entryconfigure 1 -state normal
            }
        }
        
        "EditCut" {
            $MenuFrame.edit.m entryconfigure 3 -state normal
            $ToolbarFrame.editcut configure -state normal
        }
        "EditCopy" {
            $MenuFrame.edit.m entryconfigure 4 -state normal
            $ToolbarFrame.editcopy configure -state normal
        }
        "EditPaste" {
            $MenuFrame.edit.m entryconfigure 5 -state normal
            $ToolbarFrame.editpaste configure -state normal
        }
        "EditDelete" {
            $MenuFrame.edit.m entryconfigure 6 -state normal
        }
        "EditFind" {
            $MenuFrame.edit.m entryconfigure 10 -state normal
        }
        "EditFindNext" {
            $MenuFrame.edit.m entryconfigure 11 -state normal
        }
        "EditReplace" {
            $MenuFrame.edit.m entryconfigure 13 -state normal
        }
        "EditGoTo" {
            $MenuFrame.edit.m entryconfigure 15 -state normal
        }
        "SelectAll" {
            $MenuFrame.edit.m entryconfigure 8 -state normal
        }
        
        "FileClose" {
            $MenuFrame.file.m entryconfigure 2 -state normal
        }
        "FileSave" {
            $MenuFrame.file.m entryconfigure 4 -state normal
            $ToolbarFrame.filesave configure -state normal
        }
        "FileSaveAs" {
            $MenuFrame.file.m entryconfigure 5 -state normal
        }
        "FileSaveAll" {
            $MenuFrame.file.m entryconfigure 6 -state normal
        }
        
    }
    
    return
}

proc disable {what} {
    variable MenuFrame
    variable ToolbarFrame
    
    switch $what {
        "Run" {
            $MenuFrame.project.m entryconfigure 0 -state disabled
        }
            
        "Debug" {
            $MenuFrame.project.m entryconfigure 1 -state disabled
        }
        "EditCut" {
            $MenuFrame.edit.m entryconfigure 3 -state disabled
            $ToolbarFrame.editcut configure -state disabled
        }
        "EditCopy" {
            $MenuFrame.edit.m entryconfigure 4 -state disabled
            $ToolbarFrame.editcopy configure -state disabled
        }
        "EditPaste" {
            $MenuFrame.edit.m entryconfigure 5 -state disabled
            $ToolbarFrame.editpaste configure -state disabled
        }
        "EditDelete" {
            $MenuFrame.edit.m entryconfigure 6 -state disabled
        }
        "EditFind" {
            $MenuFrame.edit.m entryconfigure 10 -state disabled
        }
        "EditFindNext" {
            $MenuFrame.edit.m entryconfigure 11 -state disabled
        }
        "EditReplace" {
            $MenuFrame.edit.m entryconfigure 13 -state disabled
        }
        "EditGoTo" {
            $MenuFrame.edit.m entryconfigure 15 -state disabled
        }
        "SelectAll" {
            $MenuFrame.edit.m entryconfigure 8 -state disabled
        }

        "FileClose" {
            $MenuFrame.file.m entryconfigure 2 -state disabled
        }
        "FileSave" {
            $MenuFrame.file.m entryconfigure 4 -state disabled
            $ToolbarFrame.filesave configure -state disabled
        }
        "FileSaveAs" {
            $MenuFrame.file.m entryconfigure 5 -state disabled
        }
        "FileSaveAll" {
            $MenuFrame.file.m entryconfigure 6 -state disabled
        }
    }
    
    return

}

proc OnFindButton {} {
    variable ToolbarFrame
    
    set text [[$ToolbarFrame.findStr subwidget entry] get]
    
    findInFiles $text
        
    return
}

proc onCreateText {} {
    enable EditCut
    enable EditCopy
    enable EditPaste
    enable EditDelete
    enable EditFind
    enable EditFindNext
    enable EditReplace
    enable EditGoTo
    enable SelectAll
    enable FileClose
    enable FileSave
    enable FileSaveAs
    enable FileSaveAll
    return
}

proc onDestroyText {} {
    disable EditCut
    disable EditCopy
    disable EditPaste
    disable EditDelete
    disable EditFind
    disable EditFindNext
    disable EditReplace
    disable EditGoTo
    disable SelectAll
    disable FileClose
    disable FileSave
    disable FileSaveAs
    disable FileSaveAll
    
    return
}

# reaction of selection "Find" menu
proc OnEditFind {} {
    find    
    return
}

proc OnEditFindInFiles {} {
    findInFiles
    return
}

proc OnEditFindNext {} {
    editorWindows::findNext
    return
}

proc find {{str ""}}  {
    global tclDevData
    
    set options ""
    
    set result [findDialog $str]
    
    set findStr ""
        
    if {$result != ""} {
        set findStr [lindex $result 0]
        set options [lindex $result 1]
    }
    
    if {$findStr != ""} {
        # append into the find list
        if {[llength $tclDevData(find)] == 0 || 
            $findStr != [lindex $tclDevData(find) 0]} {
            set tclDevData(find) [linsert $tclDevData(find) 0 $findStr]
        }
        set tclDevData(find,lastOptions) $options
        
        #find
        editorWindows::find $findStr $options
    }
    
    return
}

proc findInFiles {{findStr ""}} {
    global tclDevData
    
    set result [findInFilesDialog $findStr]
    
    if {$result != {}} {
        set findStr [lindex $result 0]
        set path [lindex $result 1]
        set fileMasks [lindex $result 2]
        set options [lindex $result 3]
        
        if {$findStr != ""} {
            # append into the find list
            if {[llength $tclDevData(find)] == 0 || 
                $findStr != [lindex $tclDevData(find) 0]} {
                set tclDevData(find) [linsert $tclDevData(find) 0 $findStr]
            }
            
            # call grep
            set tclDevData(findInFiles) [eval "grep \"$findStr\" \"$path\" $fileMasks $options"]
            statusText::setText "[llength $tclDevData(findInFiles)] occurrence(s) have been found."

            #clear old result
            outputWindow::clearFind
                        
            # apply result
            foreach entry $tclDevData(findInFiles) {
                outputWindow::addFind [lindex $entry 0] [lindex $entry 1] [lindex $entry 2] [lindex $entry 3]
            }
            
        }
    }
    
    return
}

# reaction on Replace menu
proc OnEditReplace {} {
    replace
    return
}

proc replace {{findStr ""}} {
    global tclDevData

    set result [replaceDialog $findStr]
    
    if {$result != {}} {
        set op [lindex $result 0]
        set findStr [lindex $result 1]
        set replaceStr [lindex $result 2]
        set in [lindex $result 3]
        set options [lindex $result 4]

        # store find and replace strings
        if {$tclDevData(find) != {} || 
            $findStr != [lindex $tclDevData(find) 0]} {
            set tclDevData(find) [linsert $tclDevData(find) 0 $findStr]
        }
        if {$tclDevData(replace) != {} || 
            $replaceStr != [lindex $tclDevData(replace) 0]} {
            set tclDevData(replace) [linsert $tclDevData(replace) 0 $replaceStr]
        }

        switch $op {
            "find" {
                set tclDevData(find,lastOptions) $options
        
                #find
                editorWindows::find $findStr $options                
            }
            
            "replace" {
                editorWindows::replace $findStr $replaceStr $in $options
            }
            
            "replaceAll" {
                editorWindows::replaceAll $findStr $replaceStr $in $options
            }
        }
    }
    
    return
}

proc goTo {{lineNum ""}} {
    set result [goToDialog $lineNum]
    
    if {$result != {}} {
        set what [lindex $result 0]
        set value [lindex $result 1]
        
        switch $what {
            "line" {
                editorWindows::setCursor $value 0
            }
            
            "procedure" {
                editorWindows::gotoProc $value
            }
        }
    }
    
    return
}

proc getGeometry {} {
    variable MainFrame
    
    return [wm geometry $MainFrame]
}

proc getProjectWinSize {} {
    variable PaneWindow2
    
    return [$PaneWindow2 panecget project -size]
}

proc getOutputWinSize {} {
    variable PaneWindow1
    
    return [$PaneWindow1 panecget outWin -size]
}

proc AddToRecentFiles {fileName} {
    global tclDevData
    
    if {![info exist tclDevData(recentFiles)]} {
        lappend tclDevData(recentFiles) $fileName
    }
    
    #delete prev instance in recent files
    set index [lsearch $tclDevData(recentFiles) $fileName]
    
    if {$index != -1} {
        set tclDevData(recentFiles) [lreplace $tclDevData(recentFiles) $index $index]
    }
    
    #insert to begin of list
    set tclDevData(recentFiles) [linsert $tclDevData(recentFiles) 0 $fileName]
    
    #check size of list
    if {[llength $tclDevData(recentFiles)] > 20} {
        #delete last entry
        set lastIndex [expr {[llength $tclDevData(recentFiles)] - 1}]
        set tclDevData(recentFiles) [lreplace $tclDevData(recentFiles) $lastIndex $lastIndex]
    }

    UpdateRecentFilesMenu
    
    return    
}

proc AddToRecentProjects {fileName} {
    global tclDevData
    
    if {![info exist tclDevData(recentProjects)]} {
        lappend tclDevData(recentProjects) $fileName
    }
    
    #delete prev instance in recent files
    set index [lsearch $tclDevData(recentProjects) $fileName]
    
    if {$index != -1} {
        set tclDevData(recentProjects) [lreplace $tclDevData(recentProjects) $index $index]
    }
    
    #insert to begin of list
    set tclDevData(recentProjects) [linsert $tclDevData(recentProjects) 0 $fileName]
    
    #check size of list
    if {[llength $tclDevData(recentProjects)] > 20} {
        #delete last entry
        set lastIndex [expr {[llength $tclDevData(recentProjects)] - 1}]
        set tclDevData(recentProjects) [lreplace $tclDevData(recentProjects) $lastIndex $lastIndex]
    }

    UpdateRecentProjectsMenu
    return    
}

proc UpdateRecentFilesMenu {} {
    variable RecentFilesMenu
    global tclDevData
    
    catch {destroy $RecentFilesMenu}
    
    menu $RecentFilesMenu -tearoff 0
    
    if [info exist tclDevData(recentFiles)] {
        foreach fileName $tclDevData(recentFiles) {
            $RecentFilesMenu add command -label $fileName -command "mainFrame::openFile $fileName"
        }
    }
    
    return
}

proc UpdateRecentProjectsMenu {} {
    variable RecentProjectsMenu
    global tclDevData
    
    catch {destroy $RecentProjectsMenu}
    
    menu $RecentProjectsMenu -tearoff 0
    
    if [info exist tclDevData(recentProjects)] {
        foreach fileName $tclDevData(recentProjects) {
            $RecentProjectsMenu add command -label $fileName -command "mainFrame::openFile $fileName"
        }
    }
    
    return
}

proc setProjectName {name1 name2 op} {
    variable MainFrame
    
    if {$MainFrame != ""} {
        if {[project::getName] != ""} {
            wm title $MainFrame "TCL Developer Studio - [project::getName]"
        } else {
            wm title $MainFrame "TCL Developer Studio"
        }
    }
    
    return
}

}

# Go To dialog
proc goToDialog { {lineNum ""}} {
    global goToDialog
    global tclDevData
    
    set w [toplevel .goToDialog -borderwidth 10]
    wm title $w "Go To:"
    
    if [info exist tclDevData(positions,goToDialog)] {
        wm geometry $w $tclDevData(positions,goToDialog)
    } else {
        set x [expr ([winfo screenwidth .] - 200)/2]
        set y [expr ([winfo screenheight .] - 50)/2]

        wm geometry $w 400x100+$x+$y
    }
    wm protocol $w WM_DELETE_WINDOW "set tclDevData(positions,goToDialog) [wm geometry $w];set goToDialog(ok) 0"
        
    set goToDialog(line) $lineNum
    
    set f [tixListNoteBook $w.f -dynamicgeometry 0]
    $f subwidget hlist add line -itemtype text -text "Line" -under 0
    $f subwidget hlist add procedure -itemtype text -text "Procedure" -under 0
    
    $f add line
    $f add procedure
    
    set p [$f subwidget line]
    set line [tixLabelEntry $p.line -label "Enter line number:" -labelside top -options {
        entry.textVariable goToDialog(line)
    }]
    
    pack $line -side top -expand 1 -fill x -padx 3
    
    set p [$f subwidget procedure]
    tixLabelEntry $p.procedure -label "Enter procedure name:" -labelside top -options {
        entry.textVariable goToDialog(procedure)
    }
    pack $p.procedure -side top -expand 1 -fill x -padx 3

    pack $f -side left -expand 1 -fill both
    
    set buttons [tixButtonBox $w.buttons -orientation vertical -pady 2]
    
    $buttons add goto -text "Go to" -command {set goToDialog(ok) 1}
    $buttons add cancel -text Cancel -command {set goToDialog(ok) 0}
    pack $buttons -fill y -expand 0 -side right -padx 3
    
    # run dialog
#    focus [$f.findStr subwidget entry]
    focus [$line subwidget entry]
    grab $w
    tkwait variable goToDialog(ok)
    grab release $w
    
    set what [$f raised]

    # store position
    set tclDevData(positions,goToDialog) [wm geometry $w]
    
    destroy $w
    
    if $goToDialog(ok) {
        if {$what == "line"} {
            set value $goToDialog(line)
        } else {
            set value $goToDialog(procedure)
        }
        
        return [list $what $value]
    } else {
        return {}
    }
}









