######################################################################
######################################################################

#
# cport:
#
# create a canvas viewport item in a toplevel window.
#

proc cport {cell top colormap visual args} {
 
  catch {destroy $top};

  toplevel $top	-bd 0 \
		-colormap $colormap \
		-visual $visual;
  wm geometry $top 400x340;

  canvas $top.c;
  pack $top.c	-expand 1 \
		-fill both \
		-padx 10 \
		-pady 10;

  $top.c configure -bg steelblue;
  
  set id [$top.c create viewport 30 20 350 260 \
		-minu -100 \
		-minv -100 \
		-maxu 100 \
		-maxv 100 \
		-fill black \
		-outline red \
		-projection perspective \
		-vrp {0 0 100} \
		-prp {0 0 5} \
		-vpn {0.1 0 1} \
		-vup {0 1 0} \
		-ambient #5fff5fff5fff \
		-depthcue {{0 0 0} linear 0 1000} \
		-light0 {{1 1 1} {1 1 1} {1 1 1} {50 50 50 0}} \
		-light1 {{1 1 1} {1 1 1} {1 1 1} {-50 50 50 0}} \
		];

  $top.c itemconfigure $id -cell $cell;
  if {$args != ""} {eval [concat "{$top.c} itemconfigure $id" $args]};
  update;
}


######################################################################
######################################################################

#
# vport:
#
# create a viewport widget in a toplevel window.
#

proc vport {cell top colormap visual args} {

  catch {destroy $top};

  toplevel $top	-relief sunken \
		-bd 0 \
		-colormap $colormap \
		-visual $visual;
  wm geometry $top 400x320;

  viewport $top.c 1 1;
  pack $top.c	-expand 1 \
		-fill both \
		-padx 10 \
		-pady 10;
  
  $top.c configure	-cell $cell \
			-minu -100 \
			-minv -100 \
			-maxu 100 \
			-maxv 100 \
			-fill black \
			-outline red \
			-width 1 \
			-projection perspective \
			-vrp {0 0 100} \
			-prp {0 0 5} \
			-vpn {0.1 0 1} \
			-vup {0 1 0} \
			-ambient #5fff5fff5fff \
			-depthcue {{0 0 0} linear 0 1000} \
			-light0 {{1 1 1} {1 1 1} {1 1 1} {50 50 50 0}} \
			-light1 {{1 1 1} {1 1 1} {1 1 1} {-50 50 50 0}};

  if {$args != ""} {eval [concat "$top.c config " $args]};
  update;

  bind $top.c <Button-1> {start_navigation %W {100 100 100} None 0.2};
  bind $top.c <ButtonRelease-1> {stop_navigation};

  bind $top.c <Control-Button-1> {start_navigation %W {100 100 100} Control 0.2};
  bind $top.c <ButtonRelease-1> {stop_navigation};

  bind $top.c <Shift-Button-1> {start_navigation %W {100 100 100} Shift 0.2};
  bind $top.c <ButtonRelease-1> {stop_navigation};

  bind $top.c <Button-2> {pick_model %W %x %y};
}

######################################################################
######################################################################

#
# primitives:
#
# populate a cell with primitive objects:
#
# polygon, cylinder, disk, sphere
#

proc primitives {cell} {

  if ![catch {cell create $cell}] {

    # type point

    $cell create point point1	-vertices [list {-10 -10 -10} \
					        { 10 -10 -10} \
					        { 10 -10  10} \
					        {-10 -10  10} \
					        {-10  10 -10} \
					        { 10  10 -10} \
					        { 10  10  10} \
					        {-10  10  10} \
					        ] \
				-pos {-75 0 0} \
				-fwd {-0.5 0 1} \
				-up {0 1 0} \
				-pointsize 4 \
				-shading smooth \
				-color green \
				-emissive {0 0 0} \
				-ambient {0.4 0.4 0.4} \
				-diffuse {1 1 1} \
				-specular {1 1 1 15};
    # type sphere

    $cell create sphere sphere1 \
	-radius 10 \
	-slices 20 \
	-stacks 15 \
	-pos {-45 0 0} \
	-fwd {0 0 1} -up {0 1 0} \
	-shading smooth \
	-normals smooth \
	-color red \
	-emissive {0.8 0.8 0.8} \
	-ambient {0 0 0} \
	-diffuse {0.5 0.5 0.5} \
	-specular {1 1 1 20};
	
    # type cylinder

    $cell create cylinder cyl1 \
	-topradius 0 \
	-baseradius 10 \
	-height 20 \
	-slices 20 \
	-stacks 15 \
	-pos {-15 -5 0} \
	-fwd {0 1 0} -up {0 0 1} \
	-shading smooth \
	-normals smooth \
	-color orange \
	-emissive {0 0 0} \
	-ambient {0.4 0.4 0.4} \
	-diffuse {1 1 1} \
	-specular {1 1 1 15};
	
    # type disk

    $cell create disk disk1 \
	-innerradius 5 \
	-outerradius 10 \
	-loops 20 \
	-pos {15 0 0} \
	-fwd {-1 2 2} -up {0 1 0} \
	-shading smooth \
	-normals smooth \
	-color palegreen \
	-emissive {0 0 0} \
	-ambient {0.4 0.4 0.4} \
	-diffuse {1 1 1} \
	-specular {1 1 1 15};

    # type polygon

    $cell create polygon cube1	-vertices [list {-10 -10 -10} \
					        { 10 -10 -10} \
					        { 10 -10  10} \
					        {-10 -10  10} \
					        {-10  10 -10} \
					        { 10  10 -10} \
					        { 10  10  10} \
					        {-10  10  10} \
					        ] \
			      	-surfaces [list {1 2 3 4} \
					        {8 7 6 5} \
					        {1 4 8 5} \
					        {2 6 7 3} \
					        {1 5 6 2} \
					        {3 7 8 4} \
					        ] \
				-pos {45 0 0} \
				-fwd {-0.5 0 1} \
				-up {0 1 0} \
				-shading smooth \
				-color blue \
				-emissive {0 0 0} \
				-ambient {0.4 0.4 0.4} \
				-diffuse {1 1 1} \
				-specular {1 1 1 15} \
				-average off;

    # type line

    $cell create line line1	-vertices [list {-10 -10 -10} \
					        { 10 -10 -10} \
					        { 10 -10  10} \
					        {-10 -10  10} \
					        {-10  10 -10} \
					        { 10  10 -10} \
					        { 10  10  10} \
					        {-10  10  10} \
					        ] \
			      	-lines [list {1 2 3 4} \
					     {8 7 6 5} \
					     {1 4 8 5} \
					     {2 6 7 3} \
					     {1 5 6 2} \
					     {3 7 8 4} \
					     ] \
				-pos {75 0 0} \
				-fwd {-0.5 0 1} \
				-up {0 1 0} \
				-linewidth 4 \
				-linestipple 0xcccc \
				-linestipplefactor 2 \
				-shading smooth \
				-color brown \
				-emissive {0 0 0} \
				-ambient {0.4 0.4 0.4} \
				-diffuse {1 1 1} \
				-specular {1 1 1 15};

  }
  return [$cell list];
}


######################################################################
######################################################################

proc pick_model {port x y} {

  $port configure -pick [list $x $y 1 1];
  puts [lindex [$port cget -pick] 1];

}


######################################################################
######################################################################

#
# getvisual:
#
# return the OpenGL compatible visual with the greatest depth
#

proc getvisual {which} {
  switch $which {

    canvas {
      set visuals [sminfo visuals canvas];
    }

    widget {
      set visuals [sminfo visuals widget];
    }

    default {
      error "bad option \"$which\": must be canvas or widget";
    }
  }

  set mydepth 0;
  set myvisual {};

  foreach visual $visuals {
    set depth [lindex $visual 2];
    if {$mydepth < $depth} {
      set mydepth $depth;
      set myvisual [lrange $visual 1 2];
    }
  }

  return $myvisual;
}


######################################################################
######################################################################

#
# magnitude_vector:
#
# compute the magnitude of a vector

proc magnitude_vector {vec3d} {

  set x [lindex $vec3d 0];
  set y [lindex $vec3d 1];
  set z [lindex $vec3d 2];

  expr sqrt($x * $x + $y * $y + $z * $z);
}


#
######################################################################
######################################################################

#
# normalize_vector:
#
# normalize a vector
#

proc normalize_vector {vec3d} {

  set x [lindex $vec3d 0];
  set y [lindex $vec3d 1];
  set z [lindex $vec3d 2];

  set d [expr sqrt($x * $x + $y * $y + $z * $z)];
  if {$d > 0} {
    set d [expr 1.0 / $d];
    list [expr $x * $d] [expr $y * $d] [expr $z * $d];
  } else {
    return {0 0 0};
  }
}


######################################################################
######################################################################

#
# dot_product:
#
# compute the dot product of two vectors
#

proc dot_product {a b} {

  set ax [lindex $a 0];
  set ay [lindex $a 1];
  set az [lindex $a 2];

  set bx [lindex $b 0];
  set by [lindex $b 1];
  set bz [lindex $b 2];

  expr $ax * $bx + $ay * $by + $az * $bz;
}


######################################################################
######################################################################

#
# cross_product:
#
# compute the cross product of two vectors
#

proc cross_product {a b} {

  set ax [lindex $a 0];
  set ay [lindex $a 1];
  set az [lindex $a 2];

  set bx [lindex $b 0];
  set by [lindex $b 1];
  set bz [lindex $b 2];

  list [expr $ay * $bz - $az * $by] \
       [expr $az * $bx - $ax * $bz] \
       [expr $ax * $by - $ay * $bx];
}


######################################################################
######################################################################

#
# rotate_about_x:
#
# rotate a vector about the x-axis
#

proc rotate_about_x {vec3d angle} {

  set x [lindex $vec3d 0];
  set y [lindex $vec3d 1];
  set z [lindex $vec3d 2];

  set sine [expr sin($angle)];
  set cosine [expr cos($angle)];

  return [list $x \
	       [expr $cosine * $y - $sine * $z] \
	       [expr $sine * $y + $cosine * $z] \
	  ];
}


######################################################################
######################################################################

#
# rotate_about_y:
#
# rotate a vector about the y-axis
#

proc rotate_about_y {vec3d angle} {

  set x [lindex $vec3d 0];
  set y [lindex $vec3d 1];
  set z [lindex $vec3d 2];

  set sine [expr sin($angle)];
  set cosine [expr cos($angle)];

  return [list [expr $cosine * $x + $sine * $z] \
	       $y \
	       [expr -$sine * $x + $cosine * $z] \
	  ];
}


######################################################################
######################################################################

#
# rotate_about_z:
#
# rotate a vector about the z-axis
#

proc rotate_about_z {vec3d angle} {

  set x [lindex $vec3d 0];
  set y [lindex $vec3d 1];
  set z [lindex $vec3d 2];

  set sine [expr sin($angle)];
  set cosine [expr cos($angle)];

  return [list [expr $cosine * $x - $sine * $y] \
	       [expr $sine * $x + $cosine * $y] \
	       $z \
	  ];
}


######################################################################
######################################################################

proc rotate {vec3d matrix} {

#
# rotate:
#
# rotate a vector using a pre-computed rotation matrix
#

  set x [lindex $vec3d 0];
  set y [lindex $vec3d 1];
  set z [lindex $vec3d 2];

  set m0 [lindex $matrix 0];
  set m1 [lindex $matrix 1];
  set m2 [lindex $matrix 2];
  set m3 [lindex $matrix 3];
  set m4 [lindex $matrix 4];
  set m5 [lindex $matrix 5];
  set m6 [lindex $matrix 6];
  set m7 [lindex $matrix 7];
  set m8 [lindex $matrix 8];

  return [list [expr $x * $m0 + $y * $m1 + $z * $m2] \
	       [expr $x * $m3 + $y * $m4 + $z * $m5] \
	       [expr $x * $m6 + $y * $m7 + $z * $m8] \
	       ];
}


######################################################################
######################################################################

#
# rotation_axis:
#
# compute the matrix for rotation about an arbitrary axis
#

proc rotation_axis {vec3d angle} {

  set cx [lindex $vec3d 0];
  set cy [lindex $vec3d 1];
  set cz [lindex $vec3d 2];

  set sine [expr sin($angle)];
  set cosine [expr cos($angle)];

  set m00 [expr $cx * $cx + $cosine * (1 - $cx * $cx)];
  set m01 [expr $cx * $cy * (1 - $cosine) - $cz * $sine];
  set m02 [expr $cz * $cx * (1 - $cosine) + $cy * $sine];

  set m10 [expr $cx * $cy * (1 - $cosine) + $cz * $sine];
  set m11 [expr $cy * $cy + $cosine * (1 - $cy * $cy)];
  set m12 [expr $cy * $cz * (1 - $cosine) - $cx * $sine];

  set m20 [expr $cz * $cx * (1 - $cosine) - $cy * $sine];
  set m21 [expr $cy * $cz * (1 - $cosine) + $cx * $sine];
  set m22 [expr $cz * $cz + $cosine * (1 - $cz * $cz)];

  list $m00 $m01 $m02 $m10 $m11 $m12 $m20 $m21 $m22;
}


######################################################################
######################################################################

#
# rotation_vectors:
#
# compute the rotation matrix about an axis in the direction of the
# cross-product of two vectors
#

proc rotation_vectors {u v} {

  set axis [normalize_vector [cross_product $u $v]];
  if {$axis == {}} {
    return {};
  }

  set cosine [dot_product $u $v];
  set sine [expr sqrt(1 - $cosine * $cosine)];

  set cx [lindex $axis 0];
  set cy [lindex $axis 1];
  set cz [lindex $axis 2];

  set m00 [expr $cx * $cx + $cosine * (1 - $cx * $cx)];
  set m01 [expr $cx * $cy * (1 - $cosine) - $cz * $sine];
  set m02 [expr $cz * $cx * (1 - $cosine) + $cy * $sine];

  set m10 [expr $cx * $cy * (1 - $cosine) + $cz * $sine];
  set m11 [expr $cy * $cy + $cosine * (1 - $cy * $cy)];
  set m12 [expr $cy * $cz * (1 - $cosine) - $cx * $sine];

  set m20 [expr $cz * $cx * (1 - $cosine) - $cy * $sine];
  set m21 [expr $cy * $cz * (1 - $cosine) + $cx * $sine];
  set m22 [expr $cz * $cz + $cosine * (1 - $cz * $cz)];

  return [list $m00 $m01 $m02 $m10 $m11 $m12 $m20 $m21 $m22];
}


######################################################################
######################################################################

#
# compute_angle_2D:
#
# compute the angle a line drawn from the center of the window <w>
# to the point (x,y) makes with the horizontal.
#

proc compute_angle_2D {w x y} {

  set PI 3.1415926535897931;
  set PI_2 1.5707963267948966;

  set width [winfo width $w];
  set height [winfo height $w];

  if {($width == -1) || ($height == -1)} {
    return 0;
  }

  set dx [expr $x - $width * 0.5];
  set dy [expr $y - $height * 0.5];

  if {$dx == 0} {
    if {$dy == 0} {
      return 0;
    } elseif {$dy > 0} {
      return $PI_2;
    } else {
      return -$PI_2;
    }
  } else {
    set angle [expr atan($dy / $dx)];

    if {$dx >= 0} {
      return $angle;
    } elseif {$dy >= 0} {
      return [expr $angle + $PI];
    } else {
      return [expr $angle - $PI];
    }
  }
}


######################################################################
######################################################################

proc compute_vector_3D {w x y maxx maxy} {

#
# compute_vector_3D:
#
# compute a vector from the cursor position (given by x,y) to be used as the axis
# of rotation.
#
  set rx [expr double($x) / $maxx];
  set ry [expr double($y) / $maxy];

  set d [expr $rx * $rx + $ry * $ry];
  if {$d >= 1} {
    set f [expr 1.0 / sqrt($d)];
    set vector [list [expr $rx * $f] [expr $ry * $f] 0];
  } else {
    set vector [list $rx $ry [expr sqrt(1.0 - $d)]];
  }
  return $vector;
}
