/********************************************************************************
 *
 * cone.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Cone
 *
 * Read in a Cone node.
 *
 * fields:	parts		ALL	SFBitMask
 *		bottomRadius	1	SFFloat
 *		height		2	SFFloat
 *
 * parts:	SIDES, BOTTOM, ALL
 *
 *******************************************************************************/

Node *
Vrml_read_Cone(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeCone *cone;

    cone = &node->node.cone;
    cone->parts = MASK_ALL;
    cone->bottomRadius = 1;
    cone->height = 2;
    
    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Cone", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Cone", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Cone format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Cone", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Cone", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad Cone format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "bottomRadius")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cone->bottomRadius, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cone->bottomRadius < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cone bottomRadius", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "height")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cone->height, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cone->height < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cone height", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "parts")) {
	    if (Vrml_read_SFBitMask(interp, channel, argv, (unsigned int) (MASK_ALL | MASK_SIDES | MASK_BOTTOM), &cone->parts) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Cone field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_Cone
 *
 * Render a Cone node.
 *
 * fields:	parts		ALL	SFBitMask
 *		bottomRadius	1	SFFloat
 *		height		2	SFFloat
 *
 * parts:	SIDES, BOTTOM, ALL
 *
 *******************************************************************************/

#define CONE_SLICES	15

int
Vrml_render_Cone(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j, comp, indices[6];

    double angle, delta;
    double base, step;

    char ambientColor[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuseColor[TCL_DOUBLE_SPACE * 3 + 10];
    char specularColor[TCL_DOUBLE_SPACE * 3 + 10];
    char emissiveColor[TCL_DOUBLE_SPACE * 3 + 10];
    char shininess[TCL_DOUBLE_SPACE];

    char vertices[(TCL_DOUBLE_SPACE * 3 + 10) * (CONE_SLICES + 1)];
    char surfaces[15 * (CONE_SLICES + 1) + 5 * CONE_SLICES];
    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];

    char *wrapS, *wrapT;

    char materials[300 * CONE_SLICES + 100 * CONE_SLICES];
    char textures[150 * CONE_SLICES + 40 * CONE_SLICES + 60];
    char normals[20 * CONE_SLICES + 10];

    char doubleW[TCL_DOUBLE_SPACE];
    char doubleX[TCL_DOUBLE_SPACE];
    char doubleY[TCL_DOUBLE_SPACE];
    char doubleZ[TCL_DOUBLE_SPACE];

    char *argv[] = {
	"-vertices",
	NULL,
	"-surfaces",
	NULL,
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-average",
	"on",
	"-materials",
	NULL,
	"-textures",
	NULL,
	"-normals",
	NULL,
    };
    NodeCone *cone = &node->node.cone;
    float r = cone->bottomRadius;
    float h = cone->height / 2;

    float s0, s1, s2;
    float x, y, z;
    SFVec3f pos, fwd, up, scale;
    SurfaceProperties *s = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];
    x = r * s0;
    y = h * s1;
    z = -r * s2;

    argv[1] = vertices;
    argv[3] = surfaces;
    argv[7] = ambient;
    argv[9] = diffuse;
    argv[11] = specular;
    argv[13] = emissive;
    argv[17] = materials;
    argv[19] = textures;
    argv[21] = normals;

    /* add vertices */
    angle = 1.57079632679489661923;
    delta = 2 * 3.14159265358979323846 / CONE_SLICES;

    Tcl_PrintDouble(interp, (double) y, doubleY);
    sprintf(vertices, "{0.0 %s 0.0} ", doubleY);
    Tcl_PrintDouble(interp, (double) -y, doubleY);

    for (i = 0; i < CONE_SLICES; i++) {

	Tcl_PrintDouble(interp, (double) x * cos(angle), doubleX);
	Tcl_PrintDouble(interp, (double) z * sin(angle), doubleZ);

	sprintf(vertices + strlen(vertices), "{%s %s %s} ",doubleX, doubleY, doubleZ);
	angle += delta;
    }

    /* add surfaces */
    *surfaces = '\0';
    if (cone->parts & MASK_SIDES) {
	for (i = 0; i < CONE_SLICES - 1; i++) {
	    sprintf(surfaces + strlen(surfaces), "{1 %d %d} ", i + 2, i + 3);
	}
	sprintf(surfaces + strlen(surfaces), "{1 %d 2} ", CONE_SLICES + 1);
    }
    if (cone->parts & MASK_BOTTOM) {
	sprintf(surfaces + strlen(surfaces), "{");
	for (i = CONE_SLICES + 1; i > 1; i--) {
	    sprintf(surfaces + strlen(surfaces), "%d ", i);
	}
	strcat(surfaces, "}");
    }

    /* add ambient */
    if (s->ambientColor) {
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[2], doubleZ);

	sprintf(ambient, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (s->diffuseColor) {
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[2], doubleZ);

	sprintf(diffuse, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (s->specularColor) {
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[2], doubleZ);
	Tcl_PrintDouble(interp, (double) s->shininess[0], doubleW);

	sprintf(specular, "%s %s %s %s", doubleX, doubleY, doubleZ, doubleW);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (s->emissiveColor) {
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[2], doubleZ);

	sprintf(emissive, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    /* add materials

       default material binding = OVERALL,
     */

    *materials = '\0';
    switch(s->materialBinding) {

      case ENUM_PER_PART:
      case ENUM_PER_PART_INDEXED:
	i = 1;
	Vrml_initialize_material_indices(state, indices, 0);
	Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	if (cone->parts & MASK_SIDES) {
	    for (; i <= CONE_SLICES; i++) {
		sprintf(materials + strlen(materials), "{%d ", i);
		for (j = 0; j < 3; j++) {
		    sprintf(materials + strlen(materials), "{%s %s %s %s %s} ",
			    ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		}
		strcat(materials, "} ");
	    }
	}
	if (cone->parts & MASK_BOTTOM) {
	    Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    sprintf(materials + strlen(materials), "{%d ", i);
	    for (j = 0; j < CONE_SLICES; j++) {
		sprintf(materials + strlen(materials), "{%s %s %s %s %s} ",
			ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    }
	    strcat(materials, "} ");
	}
	break;
    }

    /* add textures

       {id textureID comp function s-mode t-mode mag-filter min-filter bordercolor texcoords}

     */

    *textures = '\0';
    if (s->texture2) {
	comp = s->image->comp;
	if (comp == 2) comp = 1;
	else if (comp == 4) comp = 3;

	wrapS = (s->wrapS == ENUM_REPEAT) ? "repeat" : "clamp";
	wrapT = (s->wrapT == ENUM_REPEAT) ? "repeat" : "clamp";

	i = 1;
	base = 0.0;
	step = 1.0 / CONE_SLICES;
	if (cone->parts & MASK_SIDES) {
	    for (; i <= CONE_SLICES; i++) {

		Tcl_PrintDouble(interp, (double) base + (step / 2.0), doubleX);
		Tcl_PrintDouble(interp, (double) base, doubleY);
		Tcl_PrintDouble(interp, (double) base + step, doubleZ);

		sprintf(textures + strlen(textures),
			"{%d %s %d modulate %s %s nearest nearest {0 0 0} {{%s 1.0} {%s 0.0} {%s 0.0}}} ",
			i, s->texture2->label, comp, wrapS, wrapT, doubleX, doubleY, doubleZ);
		base += step;
	    }
	}
	if (cone->parts & MASK_BOTTOM) {
	    sprintf(textures + strlen(textures),
		    "{%d %s %d modulate %s %s nearest nearest {0 0 0} {",
		    i, s->texture2->label, comp, wrapS, wrapT);
	    angle = 1.57079632679489661923 - delta;
	    for (j = 0; j < CONE_SLICES; j++) {

		Tcl_PrintDouble(interp, (double) (cos(angle) + 1) / 2.0, doubleX);
		Tcl_PrintDouble(interp, (double) (1 - sin(angle)) / 2.0, doubleY);

		sprintf(textures + strlen(textures), "{%s %s} ", doubleX, doubleY);
		angle -= delta;
	    }
	    strcat(textures, "}}");
	}
    }

    /* add vertex normals for bottom, if necessary */
    *normals = '\0';
    if (cone->parts & MASK_BOTTOM) {
	sprintf(normals, "{%d", ((cone->parts & MASK_SIDES) ? CONE_SLICES : 0) + 1);
	for (i = 0; i < CONE_SLICES; i++) {
	    (void) strcat(normals, " {0.0 -1.0 0.0}");
	}
	strcat(normals, "}");
    }

    /* create polygon object */
    if (Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 22, argv) != TCL_OK) return 2;

    return 0;
}
