/********************************************************************************
 *
 * cube.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Cube
 *
 * Read in a Cube node.
 *
 * fields:	width		2	SFFloat
 *		height		2	SFFloat
 *		depth		2	SFFloat
 *
 *******************************************************************************/

Node *
Vrml_read_Cube(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeCube *cube;

    cube = &node->node.cube;
    cube->width = 2;
    cube->height = 2;
    cube->depth = 2;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Cube", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Cube", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Cube format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Cube", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Cube", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad Cube format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "width")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cube->width, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cube->width < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cube width", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "height")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cube->height, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cube->height < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cube height", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "depth")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cube->depth, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cube->depth < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cube depth", (char *) NULL);
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Cube field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_Cube
 *
 * Render a Cube node.
 *
 * fields:	width		2	SFFloat
 *		height		2	SFFloat
 *		depth		2	SFFloat
 *
 *******************************************************************************/

int
Vrml_render_Cube(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j, comp, indices[6];

    char ambientColor[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuseColor[TCL_DOUBLE_SPACE * 3 + 10];
    char specularColor[TCL_DOUBLE_SPACE * 3 + 10];
    char emissiveColor[TCL_DOUBLE_SPACE * 3 + 10];
    char shininess[TCL_DOUBLE_SPACE];

    char vertices[(TCL_DOUBLE_SPACE * 3 + 10) * 8];
    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];

    char materials[(((TCL_DOUBLE_SPACE * 3 + 10) * 5 + 10) * 4 + 20) * 6];
    char textures[2048];

    char doubleW[TCL_DOUBLE_SPACE];
    char doubleX[TCL_DOUBLE_SPACE];
    char doubleY[TCL_DOUBLE_SPACE];
    char doubleZ[TCL_DOUBLE_SPACE];

    char *argv[] = {
	"-vertices",
	NULL,
	"-surfaces",
	"{1 2 3} {1 3 4} {8 7 6} {8 6 5} {5 6 2} {5 2 1} {4 3 7} {4 7 8} {2 6 7} {2 7 3} {5 1 4} {5 4 8}",
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-average",
	"off",
	"-materials",
	NULL,
	"-textures",
	NULL,
    };
    NodeCube *cube = &node->node.cube;
    float w = cube->width / 2;
    float h = cube->height / 2;
    float d = cube->depth / 2;

    float s0, s1, s2;
    SFVec3f pos;
    SFVec3f fwd;
    SFVec3f up;
    SFVec3f scale;
    SFVec3f vin[8];
    SurfaceProperties *s = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];

    argv[1] = vertices;
    argv[7] = ambient;
    argv[9] = diffuse;
    argv[11] = specular;
    argv[13] = emissive;
    argv[17] = materials;
    argv[19] = textures;

    vin[0].v[0] =  w; vin[0].v[1] = -h; vin[0].v[2] =  d;
    vin[1].v[0] =  w; vin[1].v[1] =  h; vin[1].v[2] =  d;
    vin[2].v[0] = -w; vin[2].v[1] =  h; vin[2].v[2] =  d;
    vin[3].v[0] = -w; vin[3].v[1] = -h; vin[3].v[2] =  d;
    vin[4].v[0] =  w; vin[4].v[1] = -h; vin[4].v[2] = -d;
    vin[5].v[0] =  w; vin[5].v[1] =  h; vin[5].v[2] = -d;
    vin[6].v[0] = -w; vin[6].v[1] =  h; vin[6].v[2] = -d;
    vin[7].v[0] = -w; vin[7].v[1] = -h; vin[7].v[2] = -d;

    /* add vertices */
    *vertices = '\0';
    for (i = 0; i < 8; i++) {
	Tcl_PrintDouble(interp, (double) vin[i].v[0] * s0, doubleX);
	Tcl_PrintDouble(interp, (double) vin[i].v[1] * s1, doubleY);
	Tcl_PrintDouble(interp, (double) vin[i].v[2] * s2, doubleZ);

	sprintf(vertices + strlen(vertices), "{%s %s %s} ", doubleX, doubleY, doubleZ);
    }

    /* add surfaces - done */

    /* add ambient */
    if (s->ambientColor) {
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[2], doubleZ);

	sprintf(ambient, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (s->diffuseColor) {
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[2], doubleZ);

	sprintf(diffuse, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (s->specularColor) {
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[2], doubleZ);
	Tcl_PrintDouble(interp, (double) s->shininess[0], doubleW);

	sprintf(specular, "%s %s %s %s", doubleX, doubleY, doubleZ, doubleW);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (s->emissiveColor) {
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[2], doubleZ);

	sprintf(emissive, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    /* add materials

       default material binding = OVERALL,
     */

    *materials = '\0';
    switch(s->materialBinding) {

      case ENUM_PER_PART:
      case ENUM_PER_PART_INDEXED:
      case ENUM_PER_FACE:
      case ENUM_PER_FACE_INDEXED:
	Vrml_initialize_material_indices(state, indices, 0);
	for (i = 1; i < 7; i++) {
	    Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    sprintf(materials + strlen(materials), "{%d ", i * 2 - 1);
	    for (j = 0; j < 3; j++) {
		sprintf(materials + strlen(materials), "{%s %s %s %s %s} ",
			ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    }
	    strcat(materials, "} ");
	    sprintf(materials + strlen(materials), "{%d ", i * 2);
	    for (j = 0; j < 3; j++) {
		sprintf(materials + strlen(materials), "{%s %s %s %s %s} ",
			ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    }
	    strcat(materials, "} ");
	}
	break;
    }

    /* add textures

       {id textureID comp function s-mode t-mode mag-filter min-filter bordercolor texcoords}

     */

    *textures = '\0';
    if (s->texture2) {
	comp = s->image->comp;
	if (comp == 2) comp = 1;
	else if (comp == 4) comp = 3;
	for (i = 1; i < 7; i++) {

	    sprintf(textures + strlen(textures),
		    "{%d %s %d modulate %s %s nearest nearest {0 0 0} {{1.0 0.0} {1.0 1.0} {0.0 1.0}}} ",
		    2 * i - 1, s->texture2->label, comp,
		    (s->wrapS == ENUM_REPEAT) ? "repeat" : "clamp",
		    (s->wrapT == ENUM_REPEAT) ? "repeat" : "clamp");

	    sprintf(textures + strlen(textures),
		    "{%d %s %d modulate %s %s nearest nearest {0 0 0} {{1.0 0.0} {0.0 1.0} {0.0 0.0}}} ",
		    2 * i, s->texture2->label, comp,
		    (s->wrapS == ENUM_REPEAT) ? "repeat" : "clamp",
		    (s->wrapT == ENUM_REPEAT) ? "repeat" : "clamp");
	}
    }

    /* create polygon object */
    if (Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 20, argv) != TCL_OK) return 2;

    return 0;
}
