/********************************************************************************
 *
 * cylinder.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Cylinder
 *
 * Read in a Cylinder node.
 *
 * fields:	parts		ALL	SFBitMask
 *		radius		1	SFFloat
 *		height		2	SFFloat
 *
 * parts:	ALL, SIDES, TOP, BOTTOM
 *
 *******************************************************************************/

Node *
Vrml_read_Cylinder(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeCylinder *cyl;

    cyl = &node->node.cylinder;
    cyl->parts = MASK_ALL;
    cyl->radius = 1;
    cyl->height = 2;
    
    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Cylinder", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Cylinder", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Cylinder format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Cylinder", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Cylinder", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad Cylinder format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "radius")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cyl->radius, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cyl->radius < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cylinder radius", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "height")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &cyl->height, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (cyl->height < 0) {
		Tcl_AppendResult(interp, argv, ": negative Cylinder height", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "parts")) {
	    if (Vrml_read_SFBitMask(interp, channel, argv, (unsigned int) (MASK_TOP | MASK_SIDES | MASK_BOTTOM), &cyl->parts) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Cylinder field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_Cylinder
 *
 * Render a Cylinder node.
 *
 * fields:	parts		ALL	SFBitMask
 *		radius		1	SFFloat
 *		height		2	SFFloat
 *
 * parts:	ALL, SIDES, TOP, BOTTOM
 *
 *******************************************************************************/

#define CYLINDER_SLICES	15

int
Vrml_render_Cylinder(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j, comp, indices[6];

    double angle, delta;
    double base, step;

    Model *model;
    Surface *s;
    Material *mat, material;
    Normal3D *norm;
    TexCoords *texcoords;

    GLenum smode, tmode;

    char vertices[(TCL_DOUBLE_SPACE * 3 + 10) * 2 * CYLINDER_SLICES];
    char surfaces[15 * (CYLINDER_SLICES + 1) + 2 * 10 * CYLINDER_SLICES];
    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];

    char doubleW[TCL_DOUBLE_SPACE];
    char doubleX[TCL_DOUBLE_SPACE];
    char doubleY[TCL_DOUBLE_SPACE];
    char doubleZ[TCL_DOUBLE_SPACE];

    char *argv[] = {
	"-vertices",
	NULL,
	"-surfaces",
	NULL,
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-average",
	"on",
	"-materials",
	"",
	"-normals",
	"",
	"-textures",
	"",
    };
    NodeCylinder *cyl = &node->node.cylinder;
    float r = cyl->radius;
    float h = cyl->height / 2;

    float s0, s1, s2;
    float x, y, z;
    SFVec3f pos, fwd, up, scale;
    SurfaceProperties *prop = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];
    x = r * s0;
    y = -h * s1;
    z = -r * s2;

    argv[1] = vertices;
    argv[3] = surfaces;
    argv[7] = ambient;
    argv[9] = diffuse;
    argv[11] = specular;
    argv[13] = emissive;

    /* add vertices */
    delta = 2 * 3.14159265358979323846 / CYLINDER_SLICES;

    /* top */
    angle = 1.57079632679489661923;
    Tcl_PrintDouble(interp, (double) y, doubleY);
    *vertices = '\0';
    for (i = 0; i < CYLINDER_SLICES; i++) {

	Tcl_PrintDouble(interp, (double) x * cos(angle), doubleX);
	Tcl_PrintDouble(interp, (double) z * sin(angle), doubleZ);

	sprintf(vertices + strlen(vertices), "{%s %s %s} ", doubleX, doubleY, doubleZ);
	angle += delta;
    }

    /* bottom */
    angle = 1.57079632679489661923;
    Tcl_PrintDouble(interp, (double) -y, doubleY);
    for (i = 0; i < CYLINDER_SLICES; i++) {

	Tcl_PrintDouble(interp, (double) x * cos(angle), doubleX);
	Tcl_PrintDouble(interp, (double) z * sin(angle), doubleZ);

	sprintf(vertices + strlen(vertices), "{%s %s %s} ", doubleX, doubleY, doubleZ);
	angle += delta;
    }

    /* add surfaces */
    *surfaces = '\0';
    if (cyl->parts & MASK_SIDES) {
	for (i = 0; i < CYLINDER_SLICES - 1; i++) {
	    sprintf(surfaces + strlen(surfaces),
		    "{%d %d %d %d} ", i + 1, i + 2, i + 2 + CYLINDER_SLICES, i + 1 + CYLINDER_SLICES);
	}
	sprintf(surfaces + strlen(surfaces), "{%d 1 %d %d} ", i + 1, CYLINDER_SLICES + 1, i + 1 + CYLINDER_SLICES);
    }
    if (cyl->parts & MASK_TOP) {
	strcat(surfaces, "{");
	for (i = CYLINDER_SLICES + 1; i <= 2 * CYLINDER_SLICES; i++) {
	    sprintf(surfaces + strlen(surfaces), "%d ", i);
	}
	strcat(surfaces, "} ");
    }
    if (cyl->parts & MASK_BOTTOM) {
	strcat(surfaces, "{");
	for (i = CYLINDER_SLICES; i > 0; i--) {
	    sprintf(surfaces + strlen(surfaces), "%d ", i);
	}
	strcat(surfaces, "}");
    }

    /* add ambient */
    if (prop->ambientColor) {
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[2], doubleZ);

	sprintf(ambient, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (prop->diffuseColor) {
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[2], doubleZ);

	sprintf(diffuse, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (prop->specularColor) {
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[2], doubleZ);
	Tcl_PrintDouble(interp, (double) prop->shininess[0], doubleW);

	sprintf(specular, "%s %s %s %s", doubleX, doubleY, doubleZ, doubleW);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (prop->emissiveColor) {
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[2], doubleZ);

	sprintf(emissive, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    /* create polygon object */
    if (Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 22, argv) != TCL_OK) return 2;

    if ((model = SmFindModel(state->cell, node->label)) == NULL) return 2;
    (void) SmComputeVertexNormals(model, 1);

    /* add materials

       default material binding = OVERALL,
     */

    switch(prop->materialBinding) {

      case ENUM_PER_PART:
      case ENUM_PER_PART_INDEXED:
	s = model->s;
	Vrml_initialize_material_indices(state, indices, 0);

	/* SIDES */
	Vrml_get_material_values(interp, state, indices, &material);
	if (cyl->parts & MASK_SIDES) {
	    for (i = 0; i < CYLINDER_SLICES; i++, s++) {
		if ((s->materials = (Material *) ckalloc(sizeof(Material) * s->vcnt)) == NULL) {
		    Tcl_AppendResult(interp, "model \"", node->label,
				     "\" (Cylinder): out of memory while allocating materials array", (char *) NULL);
		    return 2;
		}
		for (j = 0, mat = s->materials; j < s->vcnt; j++, mat++) {
		    (void) memcpy((void *) mat, (void *) &material, sizeof(Material));
		}
	    }
	}

	/* TOP */
	Vrml_get_material_values(interp, state, indices, &material);
	if (cyl->parts & MASK_TOP) {
	    if ((s->materials = (Material *) ckalloc(sizeof(Material) * s->vcnt)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (Cylinder): out of memory while allocating materials array", (char *) NULL);
		return 2;
	    }
	    for (j = 0, mat = s->materials; j < s->vcnt; j++, mat++) {
		(void) memcpy((void *) mat, (void *) &material, sizeof(Material));
	    }
	    s++;
	}

	/* BOTTOM */
	if (cyl->parts & MASK_BOTTOM) {
	    Vrml_get_material_values(interp, state, indices, &material);
	    if ((s->materials = (Material *) ckalloc(sizeof(Material) * s->vcnt)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (Cylinder): out of memory while allocating materials array", (char *) NULL);
		return 2;
	    }
	    for (j = 0, mat = s->materials; j < s->vcnt; j++, mat++) {
		(void) memcpy((void *) mat, (void *) &material, sizeof(Material));
	    }
	}
	break;
    }

    /* add vertex normals for top and bottom, if necessary */
    if (cyl->parts & MASK_SIDES) {
	s = model->s + CYLINDER_SLICES;
    }
    else {
	s = model->s;
    }

    if (cyl->parts & MASK_TOP) {

	if ((s->lvn = (Normal3D *) ckalloc(sizeof(Normal3D) * s->vcnt)) == NULL) {
	    Tcl_AppendResult(interp, "model \"", node->label,
			     "\" (Cylinder): out of memory while allocating normal array", (char *) NULL);
	    return 2;
	}
	for (i = 0, norm = s->lvn; i < s->vcnt; i++, norm++) {
	    norm->v[0] = norm->v[2] = 0;
	    norm->v[1] = 1;
	}
	s->normal = SM_VERTEX_NORMAL_USER;
	s++;
    }

    if (cyl->parts & MASK_BOTTOM) {
	if ((s->lvn = (Normal3D *) ckalloc(sizeof(Normal3D) * s->vcnt)) == NULL) {
	    Tcl_AppendResult(interp, "model \"", node->label,
			     "\" (Cylinder): out of memory while allocating normal array", (char *) NULL);
	    return 2;
	}
	for (i = 0, norm = s->lvn; i < s->vcnt; i++, norm++) {
	    norm->v[0] = norm->v[2] = 0;
	    norm->v[1] = -1;
	}
	s->normal = SM_VERTEX_NORMAL_USER;
    }


    /* add textures

       {id textureID comp function s-mode t-mode mag-filter min-filter bordercolor texcoords}

     */

    if (prop->texture2 && prop->image) {
	comp = prop->image->comp;
	if (comp == 2) comp = 1;
	else if (comp == 4) comp = 3;

	smode = (prop->wrapS == ENUM_REPEAT) ? GL_REPEAT : GL_CLAMP;
	tmode = (prop->wrapT == ENUM_REPEAT) ? GL_REPEAT : GL_CLAMP;

	s = model->s;

	if (cyl->parts & MASK_SIDES) {

	    base = 0.0;
	    step = 1.0 / CYLINDER_SLICES;

	    for (i = 0; i < CYLINDER_SLICES; i++, s++) {
		if ((s->texcoords = (TexCoords *) ckalloc(sizeof(TexCoords) * s->vcnt)) == NULL) {
		    Tcl_AppendResult(interp, "model \"", node->label,
				     "\" (Cylinder): out of memory while allocating texture coordinates array", (char *) NULL);
		    return 2;
		}

		s->texture = prop->texture2;
		s->component = comp;
		s->func = GL_MODULATE;
		s->smode = smode;
		s->tmode = tmode;
		s->magFilter = GL_NEAREST;
		s->minFilter = GL_NEAREST;
		s->bdColor[0] = s->bdColor[1] = s->bdColor[2] = 0;
		s->bdColor[3] = 1;
		prop->texture2->refCount++;

		s->texcoords[0].v[0] = base;
		s->texcoords[0].v[1] = 0;

		s->texcoords[1].v[0] = base + step;
		s->texcoords[1].v[1] = 0;

		s->texcoords[2].v[0] = base + step;
		s->texcoords[2].v[1] = 1;

		s->texcoords[3].v[0] = base;
		s->texcoords[3].v[1] = 1;

		base += step;
	    }
	}

	if (cyl->parts & MASK_TOP) {

	    if ((s->texcoords = (TexCoords *) ckalloc(sizeof(TexCoords) * CYLINDER_SLICES)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (Cylinder): out of memory while allocating texture coordinates array", (char *) NULL);
		return 2;
	    }
	    s->texture = prop->texture2;
	    s->component = comp;
	    s->func = GL_MODULATE;
	    s->smode = smode;
	    s->tmode = tmode;
	    s->magFilter = GL_NEAREST;
	    s->minFilter = GL_NEAREST;
	    s->bdColor[0] = s->bdColor[1] = s->bdColor[2] = 0;
	    s->bdColor[3] = 1;
	    prop->texture2->refCount++;

	    angle = 1.57079632679489661923;
	    for (i = 0, texcoords = s->texcoords; i < CYLINDER_SLICES; i++, texcoords++) {

		texcoords->v[0] =  (cos(angle) + 1) / 2.0;
		texcoords->v[1] = (sin(angle) + 1) / 2.0;

		angle += delta;
	    }
	    s++;
	}

	if (cyl->parts & MASK_BOTTOM) {

	    if ((s->texcoords = (TexCoords *) ckalloc(sizeof(TexCoords) * CYLINDER_SLICES)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (Cylinder): out of memory while allocating texture coordinates array", (char *) NULL);
		return 2;
	    }
	    s->texture = prop->texture2;
	    s->component = comp;
	    s->func = GL_MODULATE;
	    s->smode = smode;
	    s->tmode = tmode;
	    s->magFilter = GL_NEAREST;
	    s->minFilter = GL_NEAREST;
	    s->bdColor[0] = s->bdColor[1] = s->bdColor[2] = 0;
	    s->bdColor[3] = 1;
	    prop->texture2->refCount++;

	    angle = 1.57079632679489661923 - delta;
	    for (i = 0, texcoords = s->texcoords; i < CYLINDER_SLICES; i++, texcoords++) {
		
		texcoords->v[0] = (cos(angle) + 1) / 2.0;
		texcoords->v[1] = (1 - sin(angle)) / 2.0;

		angle -= delta;
	    }
	}
    }

    return 0;
}
