/********************************************************************************
 *
 * faceset.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_IndexedFaceSet
 *
 * Read in an IndexedFaceSet node.
 *
 * fields:	coordIndex		[]	MFLong
 *		materialIndex		-1	MFLong
 *		normalIndex		-1	MFLong
 *		textureCoordIndex	-1	MFLong
 *
 *******************************************************************************/

Node *
Vrml_read_IndexedFaceSet(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    int i;
    char *field;
    NodeIndexedFaceSet *faceset;

    faceset = &node->node.indexedFaceSet;
    faceset->coordIndex = NULL;
    faceset->coordlen = 0;
    faceset->materialIndex = NULL;
    faceset->materiallen = 0;
    faceset->normalIndex = NULL;
    faceset->normallen = 0;
    faceset->textureCoordIndex = NULL;
    faceset->texturelen = 0;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading IndexedFaceSet", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading IndexedFaceSet", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad IndexedFaceSet format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading IndexedFaceSet", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading IndexedFaceSet", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad IndexedFaceSet format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "coordIndex")) {
	    if (faceset->coordIndex) {
		(void) ckfree((void *) faceset->coordIndex);
		faceset->coordIndex = NULL;
		faceset->coordlen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &faceset->coordIndex, &faceset->coordlen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < faceset->coordlen; i++) {
		if (faceset->coordIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedFaceSet coordIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else if (!strcmp(field, "materialIndex")) {
	    if (faceset->materialIndex) {
		(void) ckfree((void *) faceset->materialIndex);
		faceset->materialIndex = NULL;
		faceset->materiallen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &faceset->materialIndex, &faceset->materiallen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < faceset->materiallen; i++) {
		if (faceset->materialIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedFaceSet materialIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else if (!strcmp(field, "normalIndex")) {
	    if (faceset->normalIndex) {
		(void) ckfree((void *) faceset->normalIndex);
		faceset->normalIndex = NULL;
		faceset->normallen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &faceset->normalIndex, &faceset->normallen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < faceset->normallen; i++) {
		if (faceset->normalIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedFaceSet normalIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else if (!strcmp(field, "textureCoordIndex")) {
	    if (faceset->textureCoordIndex) {
		(void) ckfree((void *) faceset->textureCoordIndex);
		faceset->textureCoordIndex = NULL;
		faceset->texturelen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &faceset->textureCoordIndex, &faceset->texturelen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < faceset->texturelen; i++) {
		if (faceset->textureCoordIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedFaceSet textureCoordIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad IndexedFaceSet field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    Vrml_free_IndexedFaceSet(node);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_free_IndexedFaceSet
 *
 * Free an IndexedFaceSet node.
 *
 * fields:	coordIndex		0	MFLong
 *		materialIndex		-1	MFLong
 *		normalIndex		-1	MFLong
 *		textureCoordIndex	-1	MFLong
 *
 *******************************************************************************/

void
Vrml_free_IndexedFaceSet(node)
     Node *node;
{
    NodeIndexedFaceSet *faceset;
    
    faceset = &node->node.indexedFaceSet;
    if (faceset->coordIndex) (void) ckfree((void *) faceset->coordIndex);
    if (faceset->normalIndex) (void) ckfree((void *) faceset->normalIndex);
    if (faceset->materialIndex) (void) ckfree((void *) faceset->materialIndex);
    if (faceset->textureCoordIndex) (void) ckfree((void *) faceset->textureCoordIndex);
}


/********************************************************************************
 *
 * Vrml_render_IndexedFaceSet
 *
 * Render a IndexedFaceSet node.
 *
 * fields:	coordIndex		0	MFLong
 *		materialIndex		-1	MFLong
 *		normalIndex		-1	MFLong
 *		textureCoordIndex	-1	MFLong
 *
 *******************************************************************************/

int
Vrml_render_IndexedFaceSet(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j, scnt, vcnt;
    int dimInit;
    int head, tail;
    long *index, last;
    float x, y, z, dimensions[3][2];
    double epsilon;

    Model *model;
    Vertex *vertices, *v;
    Surface *surfaces, *s;

    int matIndex, matIndices[6];
    Material *mat, material;

    int normIndex, normIndices;
    Normal3D *norm;

    int dimS, dimT, defaultMap;
    int comp;
    long texIndex;
    float maxT, diffs[3];
    SFVec2f vin2, vout2;
    TexCoords *texcoords;

    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];
    char double0[TCL_DOUBLE_SPACE];
    char double1[TCL_DOUBLE_SPACE];
    char double2[TCL_DOUBLE_SPACE];
    char double3[TCL_DOUBLE_SPACE];

    long *indexToCount = NULL;
    long *countToIndex = NULL;

    char *argv[] = {
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-shading",
	"flat",
	"-cull",
	NULL,
	"-ccw",
	NULL,
	"-average",
	"off",
	"-bfcolor",
	"white",
	"-bfambient",
	NULL,
	"-bfdiffuse",
	NULL,
	"-bfspecular",
	NULL,
	"-bfemissive",
	NULL,
    };
    NodeIndexedFaceSet *set = &node->node.indexedFaceSet;

    float s0, s1, s2;
    SFVec3f *vin3;
    SFVec3f pos, fwd, up, scale;
    SurfaceProperties *prop = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];

    argv[3] = argv[21] = ambient;
    argv[5] = argv[23] = diffuse;
    argv[7] = argv[25] = specular;
    argv[9] = argv[27] = emissive;
    argv[13] = (prop->shapeType == ENUM_SOLID) ? "back" : "none";
    argv[15] = (prop->vertexOrdering == ENUM_CLOCKWISE) ? "off" : "on";

    /* add ambient */
    if (prop->ambientColor) {
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->ambientColor->rgb[2], double2);

	sprintf(ambient, "%s %s %s", double0, double1, double2);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (prop->diffuseColor) {
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->diffuseColor->rgb[2], double2);

	sprintf(diffuse, "%s %s %s", double0, double1, double2);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (prop->specularColor) {
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->specularColor->rgb[2], double2);
	Tcl_PrintDouble(interp, (double) prop->shininess[0], double3);

	sprintf(specular, "%s %s %s %s", double0, double1, double2, double3);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (prop->emissiveColor) {
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[0], double0);
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[1], double1);
	Tcl_PrintDouble(interp, (double) prop->emissiveColor->rgb[2], double2);

	sprintf(emissive, "%s %s %s", double0, double1, double2);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    /* create polygon model */
    if (Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 28, argv) != TCL_OK) {
	return 2;
    }
    if ((model = SmFindModel(state->cell, node->label)) == NULL) {
	return 2;
    }

    /* build up vertex coordinates */
    vertices = NULL;
    vcnt = 0;
    if ((set->coordlen > 0) && (prop->coord3len > 0)) {

	if (((indexToCount = (long *) ckalloc(prop->coord3len * sizeof(long))) == NULL) ||
	    ((countToIndex = (long *) ckalloc(prop->coord3len * sizeof(long))) == NULL)) {
	    if (indexToCount) (void) ckfree((void *) indexToCount);
	    if (countToIndex) (void) ckfree((void *) countToIndex);
	    Tcl_AppendResult(interp, "model \"", node->label,
			     "\" (IndexedFaceSet): out of memory", (char *) NULL);
	    return 2;
	}
      
	dimInit = 0;
	for (i = 0; i < prop->coord3len; i++) {
	    indexToCount[i] = -1;
	}

	/* count number of distinct vertices */
	for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {

	    if ((*index != -1) && ((*index < 0) || (*index >= prop->coord3len))) {
		sprintf(interp->result, "model \"%s\" (IndexedFaceSet): vertex index out of bounds: %ld", node->label, *index);
		(void) ckfree((void *) indexToCount);
		(void) ckfree((void *) countToIndex);
		return 2;
	    }
	    
	    if ((*index != -1) && (indexToCount[*index] == -1)) {
		countToIndex[vcnt] = *index;
		indexToCount[*index] = vcnt++;
	    }
	}

	if (vcnt > 0) {
	    if ((vertices = (Vertex *) ckalloc(sizeof(Vertex) * vcnt)) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (IndexedFaceSet): out of memory while allocating vertex array", (char *) NULL);
		(void) ckfree((void *) indexToCount);
		(void) ckfree((void *) countToIndex);
		return 2;
	    }
	    v = vertices;
	    vin3 = prop->coord3;
	    for (i = 0, v = vertices, vin3 = prop->coord3; i < (const) vcnt; i++, v++) {

		x = v->lv[0] = (DOUBLE) vin3[countToIndex[i]].v[0] * s0;
		y = v->lv[1] = (DOUBLE) vin3[countToIndex[i]].v[1] * s1;
		z = v->lv[2] = (DOUBLE) vin3[countToIndex[i]].v[2] * s2;

		/* expand bounding box if necessary */
		if (dimInit) {
		    if (x > dimensions[0][1]) dimensions[0][1] = x;
		    else if (x < dimensions[0][0]) dimensions[0][0] = x;
		    
		    if (y > dimensions[1][1]) dimensions[1][1] = y;
		    else if (y < dimensions[1][0]) dimensions[1][0] = y;
		    
		    if (z > dimensions[2][1]) dimensions[2][1] = z;
		    else if (z < dimensions[2][0]) dimensions[2][0] = z;
		}
		else {
		    dimensions[0][0] = dimensions[0][1] = x;
		    dimensions[1][0] = dimensions[1][1] = y;
		    dimensions[2][0] = dimensions[2][1] = z;
		    
		    dimInit = 1;
		}
	    }
	}
	(void) ckfree((void *) countToIndex);
    }
    else {
	return 0;
    }

    model->nv = vcnt;
    model->v = vertices;
    model->materials = NULL;
    model->normals = NULL;
    model->textures = NULL;

    /* count number of surfaces */
    last = -1;
    scnt = 0;
    surfaces = NULL;
    for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {
	if ((*index != -1) && (last == -1)) {
	    scnt++;
	}
	last = *index;
    }

    if (scnt > 0) {

	if ((surfaces = (Surface *) ckalloc(sizeof(Surface) * scnt)) == NULL) {
	    Tcl_AppendResult(interp, "model \"", node->label,
			     "\" (IndexedFaceSet): out of memory while allocating surface array ", (char *) NULL);
	    (void) ckfree((void *) indexToCount);
	    return 2;
	}

	/* prepare for surface materials */
	matIndex = 0;
	Vrml_initialize_material_indices(state, matIndices, 0);

	switch(prop->materialBinding) {
	  case ENUM_PER_PART_INDEXED:
	  case ENUM_PER_FACE_INDEXED:
	  case ENUM_PER_VERTEX_INDEXED:

	    /* make sure material indices are present */
	    if ((set->materiallen == 0) || ((set->materiallen == 1) && (set->materialIndex[0] == -1))) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (IndexedFaceSet): current material binding is INDEXED, but no material indices defined",
				 (char *) NULL);
		(void) ckfree((void *) surfaces);
		(void) ckfree((void *) indexToCount);
		return 2;
	    }

	    /* make sure enough material indices are present */
	    if ((prop->materialBinding == ENUM_PER_VERTEX_INDEXED) && (set->materiallen < set->coordlen)) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (IndexedFaceSet): number of material indices less than number of vertices for PER_VERTEX_INDEXED binding",
				 (char *) NULL);
		(void) ckfree((void *) surfaces);
		(void) ckfree((void *) indexToCount);
		return 2;
	    }
	}

	/* prepare for surface normals */
	normIndex = 0;
	normIndices = 0;
	epsilon = state->cell->epsilon;
	
	/* prepare for surface texture */
	if (prop->texture2) {

	    texIndex = 0;
	    comp = prop->image->comp;
	    if (comp == 2) comp = 1;
	    else if (comp == 4) comp = 3;

	    /* if textureCoordIndex or textureCoordinate2 is null, use default mapping */
	    if ((prop->texcoord2len == 0) || (set->texturelen == 0) ||
		((set->texturelen == 1) && (set->textureCoordIndex[0] == -1))) {
		defaultMap = 1;
		diffs[0] = dimensions[0][1] - dimensions[0][0];
		diffs[1] = dimensions[1][1] - dimensions[1][0];
		diffs[2] = dimensions[2][1] - dimensions[2][0];
		if (diffs[0] > diffs[1]) {
		    if (diffs[0] > diffs[2]) {
			dimS = 0;
			dimT = (diffs[2] > diffs[1]) ? 2 : 1;
		    }
		    else {
			dimS = 2;
			dimT = 0;
		    }
		}
		else {
		    if (diffs[0] > diffs[2]) {
			dimS = 1;
			dimT = 0;
		    }
		    else {
			dimS = (diffs[1] > diffs[2]) ? 1 : 2;
			dimT = (diffs[1] > diffs[2]) ? 2 : 1;
		    }
		}
	    }
	    else {
		
		/* using textureCoordIndex; make sure enough indices are present */
		if (set->texturelen < set->coordlen) {
		    Tcl_AppendResult(interp,
				     "number of textureCoord indices in IndexedFaceSet less than number of coordinates",
				     (char *) NULL);
		    if (surfaces) (void) ckfree((void *) surfaces);
		    if (indexToCount) (void) ckfree((void *) indexToCount);
		    return TCL_ERROR;
		}
	      defaultMap = 0;
	    }
	}

	head = tail = 0;
	index = set->coordIndex;
	for (i = 0, s = surfaces; i < (const) scnt; i++, s++) {

	    s->materials = NULL;
	    s->lvn = NULL;
	    s->texcoords = NULL;
	    s->texture = NULL;
	    s->normal = SM_VERTEX_NORMAL_DEFAULT;

	    /* count number of vertices */
	    head = tail;
	    while (index[head] == -1) head++;
	    tail = head + 1;
	    while ((tail < set->coordlen) && (index[tail] != -1)) tail++;

	    if (tail - head < 3) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (IndexedFaceSet): surface has less than 3 vertices", (char *) NULL);
		goto err;
	    }
	    /* generate array of surface vertex indices */
	    if ((s->index = (int *) ckalloc(sizeof(int) * (tail - head))) == NULL) {
		Tcl_AppendResult(interp, "model \"", node->label,
				 "\" (IndexedFaceSet): out of memory while allocating vertex index array", (char *) NULL);
		goto err;
	    }
	    s->vcnt = tail - head;
	    for (j = 0; j < (const) s->vcnt; j++) {
		s->index[j] = indexToCount[index[head + j]];
	    }

	    /* generate surface materials */
	    if ((prop->materialBinding != ENUM_DEFAULT) && (prop->materialBinding != ENUM_OVERALL)) {
		if ((s->materials = (Material *) ckalloc(sizeof(Material) * s->vcnt)) == NULL) {
		    Tcl_AppendResult(interp, "model \"", node->label,
				     "\" (IndexedFaceSet): out of memory while allocating material array", (char *) NULL);
		    goto err;
		}
	    }

	    switch(prop->materialBinding) {

	      case ENUM_PER_PART:
	      case ENUM_PER_FACE:
		Vrml_get_material_values(interp, state, matIndices, &material);
		for (j = 0, mat = s->materials; j < s->vcnt; j++, mat++) {
		    (void) memcpy((void *) mat, (void *) &material, sizeof(Material));
		}
		break;

	      case ENUM_PER_PART_INDEXED:
	      case ENUM_PER_FACE_INDEXED:
		if (matIndex >= set->materiallen) {
		    Tcl_AppendResult(interp, "model \"", node->label,
				     "\" (IndexedFaceSet): too few material indices", (char *) NULL);
		    goto err;
		}
		if ((j = (int) set->materialIndex[matIndex++]) < 0) {
		    sprintf(interp->result, "model \"%s\" (IndexedFaceSet): material index out of bounds: %d", node->label, j);
		    goto err;
		}
		Vrml_initialize_material_indices(state, matIndices, j);
		Vrml_get_material_values(interp, state, matIndices, &material);
		for (j = 0, mat = s->materials; j < s->vcnt; j++, mat++) {
		    (void) memcpy((void *) mat, (void *) &material, sizeof(Material));
		}
		break;

	      case ENUM_PER_VERTEX:
		for (j = 0,mat = s->materials; j < s->vcnt; j++, mat++) {
		    Vrml_get_material_values(interp, state, matIndices, mat);
		}
		break;

	      case ENUM_PER_VERTEX_INDEXED:
		for (j = head, mat = s->materials; j < tail; j++, mat++) {
		    if (j >= set->materiallen) {
			Tcl_AppendResult(interp, "model \"", node->label,
					 "\" (IndexedFaceSet): too few material indices", (char *) NULL);
			goto err;
		    }
		    if ((matIndex = (int) set->materialIndex[j]) < 0) {
			sprintf(interp->result, "model \"%s\" (IndexedFaceSet): material index out of bounds: %d", node->label, matIndex);
			goto err;
		    }
		    Vrml_initialize_material_indices(state, matIndices, matIndex);
		    Vrml_get_material_values(interp, state, matIndices, mat);
		}
		break;

	      case ENUM_OVERALL:
	      case ENUM_DEFAULT:
	      default:
		s->materials = NULL;
	    }

	    /* generate surface normals */
	    if (prop->normal) {

		switch(prop->normalBinding) {

		  case ENUM_PER_PART:
		  case ENUM_PER_FACE:
		    if (normIndices < prop->normallen) {

			if ((s->lvn = (Normal3D *) ckalloc(sizeof(Normal3D) * s->vcnt)) == NULL) {
			    Tcl_AppendResult(interp, "model \"", node->label,
					     "\" (IndexedFaceSet): out of memory while allocating normal array", (char *) NULL);
			    goto err;
			}

			for (j = 0, norm = s->lvn; j < s->vcnt; j++, norm++) {
			    norm->v[0] = prop->normal[normIndices].v[0] * s0;
			    norm->v[1] = prop->normal[normIndices].v[1] * s1;
			    norm->v[2] = prop->normal[normIndices].v[2] * s2;
			    (void) SmNormalizeVector3D(norm->v, norm->v, epsilon);
			}
			normIndices++;
			s->normal = SM_VERTEX_NORMAL_USER;
		    }
		    break;

		  case ENUM_PER_PART_INDEXED:
		  case ENUM_PER_FACE_INDEXED:
		    if ((normIndex < set->normallen) &&
			((normIndices = set->normalIndex[normIndex++]) >= 0) &&
			(normIndices < prop->normallen)) {

			if ((s->lvn = (Normal3D *) ckalloc(sizeof(Normal3D) * s->vcnt)) == NULL) {
			    Tcl_AppendResult(interp, "model \"", node->label,
					     "\" (IndexedFaceSet): out of memory while allocating normal array", (char *) NULL);
			    goto err;
			}

			for (j = 0, norm = s->lvn; j < s->vcnt; j++, norm++) {
			    norm->v[0] = prop->normal[normIndices].v[0] * s0;
			    norm->v[1] = prop->normal[normIndices].v[1] * s1;
			    norm->v[2] = prop->normal[normIndices].v[2] * s2;
			    (void) SmNormalizeVector3D(norm->v, norm->v, epsilon);
			}
			s->normal = SM_VERTEX_NORMAL_USER;
		    }
		    break;

		  case ENUM_PER_VERTEX:

		    /* make sure there exists enough normals in the state */
		    if (prop->normallen - normIndices >= s->vcnt) {

			if ((s->lvn = (Normal3D *) ckalloc(sizeof(Normal3D) * s->vcnt)) == NULL) {
			    Tcl_AppendResult(interp, "model \"", node->label,
					     "\" (IndexedFaceSet): out of memory while allocating normal array", (char *) NULL);
			    goto err;
			}

			for (j = 0, norm = s->lvn; j < s->vcnt; j++, norm++) {
			    norm->v[0] = prop->normal[normIndices++].v[0] * s0;
			    norm->v[1] = prop->normal[normIndices++].v[1] * s1;
			    norm->v[2] = prop->normal[normIndices++].v[2] * s2;
			    (void) SmNormalizeVector3D(norm->v, norm->v, epsilon);
			}
			s->normal = SM_VERTEX_NORMAL_USER;
		    }			
		    break;

		  case ENUM_DEFAULT:
		  case ENUM_PER_VERTEX_INDEXED:

		    /* make sure there exists enough indices */
		    if (tail <= set->normallen) {
			if ((s->lvn = (Normal3D *) ckalloc(sizeof(Normal3D) * s->vcnt)) == NULL) {
			    Tcl_AppendResult(interp, "model \"", node->label,
					     "\" (IndexedFaceSet): out of memory while allocating normal array", (char *) NULL);
			    goto err;
			}

			for (j = head, norm = s->lvn; j < tail; j++, norm++) {
			    if (((normIndex = (int) set->normalIndex[j]) < 0) ||
				(normIndex >= prop->normallen)) {
				sprintf(interp->result, "model \"%s\" (IndexedFaceSet): normal index out of bounds: %d", node->label, normIndex);
				goto err;
			    }
			    norm->v[0] = prop->normal[normIndex].v[0] * s0;
			    norm->v[1] = prop->normal[normIndex].v[1] * s1;
			    norm->v[2] = prop->normal[normIndex].v[2] * s2;
			    (void) SmNormalizeVector3D(norm->v, norm->v, epsilon);
			}
			s->normal = SM_VERTEX_NORMAL_USER;
		    }
		    break;
		}
	    }

	    /* generate surface textures */
	    if (prop->texture2) {

		if (defaultMap) {
		    maxT = diffs[dimT] / diffs[dimS];
		    s0 = scale.v[dimS];
		    s1 = scale.v[dimT];
		}

		if ((s->texcoords = (TexCoords *) ckalloc(sizeof(TexCoords) * s->vcnt)) == NULL) {
		    Tcl_AppendResult(interp, "model \"", node->label,
				     "\" (IndexedFaceSet): out of memory while allocating texture coordinates", (char *) NULL);
		    goto err;
		}
		if (defaultMap) {
		    for (j = 0, texcoords = s->texcoords; j < s->vcnt; j++, texcoords++) {
			vin2.v[0] = ((prop->coord3 + s->index[j])->v[dimS] * s0 - dimensions[dimS][0]) / diffs[dimS];
			vin2.v[1] = ((prop->coord3 + s->index[j])->v[dimT] * s1 - dimensions[dimT][0]) / diffs[dimT] * maxT;
			Vrml_texcoord2_transform(state, &vin2, &vout2);
			texcoords->v[0] = (DOUBLE) vout2.v[0];
			texcoords->v[1] = (DOUBLE) vout2.v[1];
		    }
		}
		else {
		    for (j = head, texcoords = s->texcoords; j < tail; j++, texcoords++) {
			texIndex = set->textureCoordIndex[j];
			if ((texIndex < 0) || (texIndex >= prop->texcoord2len)) {
			    sprintf(interp->result, "model \"%s\" (IndexedFaceSet):  texture coordinate index out of bounds: %ld",
				    node->label, texIndex);
			    goto err;
			}
			Vrml_texcoord2_transform(state, prop->texcoord2 + texIndex, &vout2);
			texcoords->v[0] = (DOUBLE) vout2.v[0];
			texcoords->v[1] = (DOUBLE) vout2.v[1];
		    }
		}
		s->component = comp;
		s->func = GL_MODULATE;
		s->smode = (prop->wrapS == ENUM_REPEAT) ? GL_REPEAT : GL_CLAMP;
		s->tmode = (prop->wrapT == ENUM_REPEAT) ? GL_REPEAT : GL_CLAMP;
		s->magFilter = GL_NEAREST;
		s->minFilter = GL_NEAREST;
		s->bdColor[0] = s->bdColor[1] = s->bdColor[2] = 0;
		s->bdColor[3] = 1;
		s->texture = prop->texture2;
		prop->texture2->refCount++;
	    }
	    s->id = i;
	    s->model = model;
	}
    }
    else {
	return 0;
    }

    model->ns = scnt;
    model->s = surfaces;
    model->useTexture = 1;
    (void) SmComputeVertexNormals(model, 0);
    (void) SmComputeBoundingBox(model);

    if (indexToCount) (void) ckfree((void *) indexToCount);
    return 0;

  err:
    for (j = 0, s = surfaces; j < i; j++, s++) {
	if (s->index) (void) ckfree((void *) s->index);
	if (s->materials) (void) ckfree((void *) s->materials);
	if (s->lvn) (void) ckfree((void *) s->lvn);
	if (s->texcoords) (void) ckfree((void *) s->texcoords);
	if (s->texture) s->texture->refCount--;
    }
    if (indexToCount) (void) ckfree((void *) indexToCount);
    if (surfaces) (void) ckfree((void *) surfaces);
    return 2;
}
