/********************************************************************************
 *
 * group.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Group
 *
 * Read in a Group node.
 *
 *******************************************************************************/

Node *
Vrml_read_Group(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    Node *child, *last;
    NodeGroup *group;

    group = &node->node.group;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Group", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Group", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Group format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse child nodes */
    last = NULL;
    child = Vrml_read_node(interp, channel, argv, names, inlines, textures, 1, NULL);
    while ((child != NULL) && (child != (Node *) -1)) {
	if (last) {
	    last->sibling = child;
	}
	else {
	    node->child = child;
	}
	last = child;
	child = Vrml_read_node(interp, channel, argv, names, inlines, textures, 1, NULL);
    }
    if (child == NULL) {
	return node;
    }
    else {
	Vrml_free_Group(node);
	return (Node *) -1;
    }
}


/********************************************************************************
 *
 * Vrml_free_Group
 *
 * Free a Group node.
 *
 *******************************************************************************/

void
Vrml_free_Group(node)
     Node *node;
{
    Node *child, *sibling;
    
    for (child = node->child; child; child = sibling) {
	sibling = child->sibling;
	Vrml_free_node(child);
    }
}


/********************************************************************************
 *
 * Vrml_render_Group
 *
 * Render a Group node.
 *
 *******************************************************************************/

int
Vrml_render_Group(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, j;
    float d[4];
    SFVec3f pos, fwd, up, scale;
    TransformProperties *xform;

    node->objects = state->objects;
    state->objects = node;
    
    Vrml_object_state(state, &pos, &fwd, &up, &scale);

    /* create a container object */
    if ((Vrml_new_object(interp, state, node, "polygon", &pos, &fwd, &up, 0, NULL) != TCL_OK) ||
	(Vrml_push_model(interp, state, node) != TCL_OK)) {
	return 2;
    }

    /* reset local transformation matrix */
    xform = state->transform;
    for (i = 0; i < 3; i++) {
	d[i] = (float) sqrt((double) xform->local[0][i] * xform->local[0][i] +
			    xform->local[1][i] * xform->local[1][i] +
			    xform->local[2][i] * xform->local[2][i]);
    }
    d[3] = 1;
    for (i = 0; i < 4; i++) {
	for (j = 0; j < 4; j++) {
	    xform->local[i][j] = (i == j) ? d[i] : 0;
	}
    }
    return 0;
}


/********************************************************************************
 *
 * Vrml_finish_Group
 *
 * Finish rendering a Group node.
 *
 *******************************************************************************/

int
Vrml_finish_Group(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    Vrml_pop_model(state);
    return 0;
}
