/********************************************************************************
 *
 * lineset.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * format_materials
 *
 * Generate string value for the -materials option.
 *
 * Each material element has the following format:
 *
 *	{id {ambient diffuse specular emissive shininess} ...}
 *
 *******************************************************************************/

static int
format_materials(interp, state, set, nseg, materials)
     Tcl_Interp *interp;
     State *state;
     NodeIndexedLineSet *set;
     int nseg;
     char **materials;
{
    int doit;
    int i, n, last, segCount, lineCount;
    int matIndex, indices[6];
    long *index;
    char *p, *buf;
    SurfaceProperties *s = state->surface;

    char ambientColor[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuseColor[TCL_DOUBLE_SPACE * 3 + 10];
    char specularColor[TCL_DOUBLE_SPACE * 3 + 10];
    char emissiveColor[TCL_DOUBLE_SPACE * 3 + 10];
    char shininess[TCL_DOUBLE_SPACE];

    /* if binding is _Indexed, make sure indices are present */
    switch(s->materialBinding) {
      case ENUM_PER_PART_INDEXED:
      case ENUM_PER_FACE_INDEXED:
      case ENUM_PER_VERTEX_INDEXED:
	if ((set->materiallen == 0) || ((set->materiallen == 1) && (set->materialIndex[0] == -1))) {
	    Tcl_AppendResult(interp,
			     "current material binding is indexed, but no material indices defined",
			     (char *) NULL);
	    return TCL_ERROR;
	}

	/* make sure enough indices are present */
	if ((s->materialBinding == ENUM_PER_VERTEX_INDEXED) && (set->materiallen < set->coordlen)) {
	    Tcl_AppendResult(interp,
			     "number of material indices in IndexedLineSet less than number of coordinates for PER_VERTEX_INDEXED binding",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	break;

      case ENUM_OVERALL:
      case ENUM_DEFAULT:
	*materials = NULL;
	return TCL_OK;
    }

    if ((buf = (char *) ckalloc(nseg * (TCL_DOUBLE_SPACE * 13 + 30)))) {
	p = buf;;
	*buf = '\0';
	last = -1;
	doit = n = 0;
	segCount = 0;
	lineCount = 1;

	Vrml_initialize_material_indices(state, indices, 0);
	for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {

	    doit = 0;
	    switch(s->materialBinding) {

	      case ENUM_PER_PART:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			doit = 1;
		    }
		}
		else {
		    if (segCount == 1) {
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_PART_INDEXED:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			if (n >= set->materiallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet material indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((matIndex = (int) set->materialIndex[n++]) < 0) {
			    sprintf(buf, "%d", matIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet material index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_material_indices(state, indices, matIndex);
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			doit = 1;
		    }
		}
		else {
		    if (segCount == 1) {
			if (n >= set->materiallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet material indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((matIndex = (int) set->materialIndex[n++]) < 0) {
			    sprintf(buf, "%d", matIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet material index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_material_indices(state, indices, matIndex);
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_FACE:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			doit = 1;
		    }
		    else {
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		    }
		}
		else {
		    if (segCount == 1) {
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_FACE_INDEXED:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			doit = 1;
		    }
		    else {
			if (n >= set->materiallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet material indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((matIndex = (int) set->materialIndex[n++]) < 0) {
			    sprintf(buf, "%d", matIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet material index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_material_indices(state, indices, matIndex);
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		    }
		}
		else {
		    if (segCount == 1) {
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_VERTEX:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			sprintf(p, "{%d {%s %s %s %s %s} ",
				lineCount++, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			p += strlen(p);
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			sprintf(p, " {%s %s %s %s %s}} ",
				ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			p += strlen(p);
		    }
		    else {
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		    }
		}
		else {
		    if (segCount == 1) {
			sprintf(p, "{%d {%s %s %s %s %s} {%s %s %s %s %s}} ",
				lineCount++, ambientColor, diffuseColor, specularColor, emissiveColor, shininess,
				ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			p += strlen(p);
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_VERTEX_INDEXED:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			sprintf(p, "{%d {%s %s %s %s %s} ",
				lineCount++, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			p += strlen(p);
			if (n >= set->materiallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet material indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((matIndex = (int) set->materialIndex[n++]) < 0) {
			    sprintf(buf, "%d", matIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet material index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_material_indices(state, indices, matIndex);
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			sprintf(p, " {%s %s %s %s %s}} ",
				ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			p += strlen(p);
		    }
		    else {
			if (n >= set->materiallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet material indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((matIndex = (int) set->materialIndex[n++]) < 0) {
			    sprintf(buf, "%d", matIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet material index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_material_indices(state, indices, matIndex);
			Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		    }
		}
		else {
		    n++;
		    if (segCount == 1) {
			sprintf(p, "{%d {%s %s %s %s %s} {%s %s %s %s %s}} ",
				lineCount++, ambientColor, diffuseColor, specularColor, emissiveColor, shininess,
				ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
			p += strlen(p);
		    }
		    segCount = 0;
		}
		break;
	    }

	    switch(s->materialBinding) {

	      case ENUM_PER_PART:
	      case ENUM_PER_PART_INDEXED:
	      case ENUM_PER_FACE:
	      case ENUM_PER_FACE_INDEXED:
		if (doit) {
		    sprintf(p, "{%d {%s %s %s %s %s} {%s %s %s %s %s}} ",
			    lineCount++, ambientColor, diffuseColor, specularColor, emissiveColor, shininess,
			    ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
		    p += strlen(p);
		}
		break;
	    }
	    last = *index;
	}

	if ((last != -1) && (segCount == 1)) {
	    if (s->materialBinding == ENUM_PER_PART_INDEXED) {
		if (n >= set->materiallen) {
		    Tcl_AppendResult(interp, "too few IndexedLineSet material indices", (char *) NULL);
		    (void) ckfree((void *) buf);
		    return TCL_ERROR;
		}
		if ((matIndex = (int) set->materialIndex[n++]) < 0) {
		    sprintf(buf, "%d", matIndex);
		    Tcl_AppendResult(interp, "IndexedLineSet material index out of bounds:  \"",
				     buf, "\"", (char *) NULL);
		    (void) ckfree((void *) buf);
		    return TCL_ERROR;
		}
		Vrml_initialize_material_indices(state, indices, matIndex);
		Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    }
	    else if (s->materialBinding == ENUM_PER_PART) {
		Vrml_get_materials(interp, state, indices, ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    }
	    sprintf(p, "{%d {%s %s %s %s %s} {%s %s %s %s %s}}",
		    lineCount++, ambientColor, diffuseColor, specularColor, emissiveColor, shininess,
		    ambientColor, diffuseColor, specularColor, emissiveColor, shininess);
	    p += strlen(p);
	}
    }
    *materials = buf;
    *p = '\0';
    return TCL_OK;
}


/********************************************************************************
 *
 * format_normals
 *
 * Generate string value for the -normals option.
 *
 * Each normal element has the following format:
 *
 *	{id {dx dy dz} ...}
 *
 *******************************************************************************/

static int
format_normals(interp, state, set, nseg, scale, normals)
     Tcl_Interp *interp;
     State *state;
     NodeIndexedLineSet *set;
     int nseg;
     SFVec3f *scale;
     char **normals;
{
    int doit;
    int i, n, last, segCount, lineCount;
    int normIndex, indices;
    long *index;
    char *p, *buf;
    char vector[TCL_DOUBLE_SPACE * 3 + 10];
    SurfaceProperties *s = state->surface;

    /* if binding is _Indexed, make sure indices are present */
    switch(s->normalBinding) {
      case ENUM_DEFAULT:
      case ENUM_PER_PART_INDEXED:
      case ENUM_PER_FACE_INDEXED:
      case ENUM_PER_VERTEX_INDEXED:
	if ((set->normallen == 0) || ((set->normallen == 1) && (set->normalIndex[0] == -1))) {

	    if (s->normalBinding == ENUM_DEFAULT) {

		/* no normals specified; we assume the user really wants ENUM_OVERALL */
		normals = NULL;
		return TCL_OK;
	    }
	    else {
		Tcl_AppendResult(interp,
				 "current normal binding is indexed, but no normal indices defined",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	}

	/* make sure enough indices are present */
	if (((s->normalBinding == ENUM_PER_VERTEX_INDEXED) || (s->normalBinding == ENUM_DEFAULT))
	    && (set->normallen < set->coordlen)) {
	    Tcl_AppendResult(interp,
			     "number of normal indices in IndexedLineSet less than number of coordinates for PER_VERTEX_INDEXED binding",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	break;

      case ENUM_OVERALL:
	*normals = NULL;
	return TCL_OK;
    }

    if ((buf = (char *) ckalloc(nseg * (TCL_DOUBLE_SPACE * 6 + 30)))) {
	p = buf;
	*buf = '\0';
	last = -1;
	doit = n = 0;
	segCount = 0;
	lineCount = 1;

	Vrml_initialize_normal_indices(state, &indices, 0);
	for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {

	    doit = 0;
	    switch(s->normalBinding) {

	      case ENUM_PER_PART:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			Vrml_get_normal(interp, state, &indices, scale, vector);
			doit = 1;
		    }
		}
		else {
		    if (segCount == 1) {
			Vrml_get_normal(interp, state, &indices, scale, vector);
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_PART_INDEXED:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			if (n >= set->normallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet normal indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((normIndex = (int) set->normalIndex[n++]) < 0) {
			    sprintf(buf, "%d", normIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet normal index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_normal_indices(state, &indices, normIndex);
			Vrml_get_normal(interp, state, &indices, scale, vector);
			doit = 1;
		    }
		}
		else {
		    if (segCount == 1) {
			if (n >= set->normallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet normal indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((normIndex = (int) set->normalIndex[n++]) < 0) {
			    sprintf(buf, "%d", normIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet normal index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_normal_indices(state, &indices, normIndex);
			Vrml_get_normal(interp, state, &indices, scale, vector);
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_FACE:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			doit = 1;
		    }
		    else {
			Vrml_get_normal(interp, state, &indices, scale, vector);
		    }
		}
		else {
		    if (segCount == 1) {
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_FACE_INDEXED:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			doit = 1;
		    }
		    else {
			if (n >= set->normallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet normal indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((normIndex = (int) set->normalIndex[n++]) < 0) {
			    sprintf(buf, "%d", normIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet normal index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_normal_indices(state, &indices, normIndex);
			Vrml_get_normal(interp, state, &indices, scale, vector);
		    }
		}
		else {
		    if (segCount == 1) {
			doit = 1;
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_PER_VERTEX:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			sprintf(p, "{%d {%s} ", lineCount++, vector);
			p += strlen(p);
			Vrml_get_normal(interp, state, &indices, scale, vector);
			sprintf(p, " {%s}} ", vector);
			p += strlen(p);
		    }
		    else {
			Vrml_get_normal(interp, state, &indices, scale, vector);
		    }
		}
		else {
		    if (segCount == 1) {
			sprintf(p, "{%d {%s} {%s}} ", lineCount++, vector, vector);
			p += strlen(p);
		    }
		    segCount = 0;
		}
		break;

	      case ENUM_DEFAULT:
	      case ENUM_PER_VERTEX_INDEXED:
		if (*index != -1) {
		    segCount++;
		    if (last != -1) {
			sprintf(p, "{%d {%s}", lineCount++, vector);
			p += strlen(p);
			if (n >= set->normallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet normal indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((normIndex = (int) set->normalIndex[n++]) < 0) {
			    sprintf(buf, "%d", normIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet normal index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_normal_indices(state, &indices, normIndex);
			Vrml_get_normal(interp, state, &indices, scale, vector);
			sprintf(p, " {%s}} ", vector);
			p += strlen(p);
		    }
		    else {
			if (n >= set->normallen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet normal indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			if ((normIndex = (int) set->normalIndex[n++]) < 0) {
			    sprintf(buf, "%d", normIndex);
			    Tcl_AppendResult(interp, "IndexedLineSet normal index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_initialize_normal_indices(state, &indices, normIndex);
			Vrml_get_normal(interp, state, &indices, scale, vector);
		    }
		}
		else {
		    n++;
		    if (segCount == 1) {
			sprintf(p, "{%d {%s} {%s}} ", lineCount++, vector, vector);
			p += strlen(p);
		    }
		    segCount = 0;
		}
		break;
	    }

	    switch(s->normalBinding) {

	      case ENUM_PER_PART:
	      case ENUM_PER_PART_INDEXED:
	      case ENUM_PER_FACE:
	      case ENUM_PER_FACE_INDEXED:
		if (doit) {
		    sprintf(p, "{%d {%s} {%s}} ", lineCount++, vector, vector);
		    p += strlen(p);
		}
		break;
	    }
	    last = *index;
	}

	if ((last != -1) && (segCount == 1)) {
	    if (s->normalBinding == ENUM_PER_PART_INDEXED) {
		if (n >= set->normallen) {
		    Tcl_AppendResult(interp, "too few IndexedLineSet normal indices", (char *) NULL);
		    (void) ckfree((void *) buf);
		    return TCL_ERROR;
		}
		if ((normIndex = (int) set->normalIndex[n++]) < 0) {
		    sprintf(buf, "%d", normIndex);
		    Tcl_AppendResult(interp, "IndexedLineSet normal index out of bounds:  \"",
				     buf, "\"", (char *) NULL);
		    (void) ckfree((void *) buf);
		    return TCL_ERROR;
		}
		Vrml_initialize_normal_indices(state, &indices, normIndex);
		Vrml_get_normal(interp, state, &indices, scale, vector);
	    }
	    else if (s->normalBinding == ENUM_PER_PART) {
		Vrml_get_normal(interp, state, &indices, scale, vector);
	    }
	    sprintf(p, "{%d {%s} {%s}} ", lineCount++, vector, vector);
	    p += strlen(p);
	}
    }
    *normals = buf;
    *p = '\0';
    return TCL_OK;
}


/********************************************************************************
 *
 * format_textures
 *
 * Generate string value for the -textures option.
 *
 * Each texture element has the following format:
 *
 *	{id textureID comp function s-mode t-mode mag-filter min-filter bordercolor {{s t} ...}}
 *
 *******************************************************************************/

static int
format_textures(interp, state, set, nseg, scale, dimensions, textures)
     Tcl_Interp *interp;
     State *state;
     NodeIndexedLineSet *set;
     int nseg;
     SFVec3f *scale;
     float (*dimensions)[2];
     char **textures;
{
    int i, n, comp, last, segCount, lineCount;
    int dimS, dimT, defaultMap;
    long *index, texIndex;
    char *p, *buf, *wrapS, *wrapT;
    char coordS[TCL_DOUBLE_SPACE], coordT[TCL_DOUBLE_SPACE];
    float maxT, diffs[3];
    float s0, s1;
    SFVec2f vin, vout;
    SurfaceProperties *s = state->surface;

    /* if textureCoordIndex or textureCoordinate2 is null, use default mapping */
    if ((set->texturelen == 0) || (s->texcoord2len == 0) ||
	((set->texturelen == 1) && (set->textureCoordIndex[0] == -1))) {
	defaultMap = 1;
	diffs[0] = dimensions[0][1] - dimensions[0][0];
	diffs[1] = dimensions[1][1] - dimensions[1][0];
	diffs[2] = dimensions[2][1] - dimensions[2][0];
	if (diffs[0] > diffs[1]) {
	    if (diffs[0] > diffs[2]) {
		dimS = 0;
		dimT = (diffs[2] > diffs[1]) ? 2 : 1;
	    }
	    else {
		dimS = 2;
		dimT = 0;
	    }
	}
	else {
	    if (diffs[0] > diffs[2]) {
		dimS = 1;
		dimT = 0;
	    }
	    else {
		dimS = (diffs[1] > diffs[2]) ? 1 : 2;
		dimT = (diffs[1] > diffs[2]) ? 2 : 1;
	    }
	}
	maxT = diffs[dimT] / diffs[dimS];
	s0 = scale->v[dimS];
	s1 = scale->v[dimT];
    }
    else {

	/* using textureCoordIndex; make sure enough indices are present */
	if (set->texturelen < set->coordlen) {
	    Tcl_AppendResult(interp,
			     "number of textureCoord indices in IndexedLineSet less than number of coordinates",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	defaultMap = 0;
    }

    if ((buf = (char *) ckalloc(nseg * (TCL_DOUBLE_SPACE * 2 + 150)))) {
	p = buf;
	*buf = '\0';
	last = -1;
	n = 0;
	segCount = 0;
	lineCount = 1;

	comp = s->image->comp;
	if (comp == 2) comp = 1;
	else if (comp == 4) comp = 3;

	wrapS = (s->wrapS == ENUM_REPEAT) ? "repeat" : "clamp";
	wrapT = (s->wrapT == ENUM_REPEAT) ? "repeat" : "clamp";

	for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {

	    if (*index != -1) {
		segCount++;
		if (last != -1) {
		    sprintf(p, "{%d %s %d modulate %s %s nearest nearest {0 0 0} {{%s %s} ",
			    lineCount++, s->texture2->label, comp, wrapS, wrapT, coordS, coordT);
		    p += strlen(p);
		    if (defaultMap) {
			vin.v[0] = ((s->coord3 + *index)->v[dimS] * s0 - dimensions[dimS][0]) / diffs[dimS];
			vin.v[1] = (((s->coord3 + *index)->v[dimT] * s1 - dimensions[dimT][0]) / diffs[dimT]) * maxT;
			Vrml_texcoord2_transform(state, &vin, &vout);
		    }
		    else {
			if (n >= set->texturelen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet texture coordinate indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			texIndex = set->textureCoordIndex[n++];
			if ((texIndex < 0) || (texIndex >= s->texcoord2len)) {
			    sprintf(buf, "%ld (%d)", texIndex, s->texcoord2len);
			    Tcl_AppendResult(interp, "IndexedLineSet texture coordinate index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_texcoord2_transform(state, s->texcoord2 + texIndex, &vout);
		    }
		    Tcl_PrintDouble(interp, (double) vout.v[0], coordS);
		    Tcl_PrintDouble(interp, (double) vout.v[1], coordT);
		    sprintf(p, "{%s %s}}} ", coordS, coordT);
		    p += strlen(p);
		}
		else {
		    if (defaultMap) {
			vin.v[0] = ((s->coord3 + *index)->v[dimS] * s0 - dimensions[dimS][0]) / diffs[dimS];
			vin.v[1] = (((s->coord3 + *index)->v[dimT] * s1 - dimensions[dimT][0]) / diffs[dimT]) * maxT;
			Vrml_texcoord2_transform(state, &vin, &vout);
		    }
		    else {
			if (n >= set->texturelen) {
			    Tcl_AppendResult(interp, "too few IndexedLineSet texture coordinate indices", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			texIndex = set->textureCoordIndex[n++];
			if ((texIndex < 0) || (texIndex >= s->texcoord2len)) {
			    sprintf(buf, "%ld (%d)", texIndex, s->texcoord2len);
			    Tcl_AppendResult(interp, "IndexedLineSet texture coordinate index out of bounds:  \"",
					     buf, "\"", (char *) NULL);
			    (void) ckfree((void *) buf);
			    return TCL_ERROR;
			}
			Vrml_texcoord2_transform(state, s->texcoord2 + texIndex, &vout);
		    }
		    Tcl_PrintDouble(interp, (double) vout.v[0], coordS);
		    Tcl_PrintDouble(interp, (double) vout.v[1], coordT);
		}
	    }
	    else {
		n++;
		if (segCount == 1) {
		    sprintf(p, "{%d %s %d modulate %s %s nearest nearest {0 0 0} {{%s %s} {%s %s}}}",
			    lineCount, s->texture2->label, comp, wrapS, wrapT, coordS, coordT, coordS, coordT);
		    p += strlen(p);
		}
		segCount = 0;
	    }
	    last = *index;
	}

	if ((last != -1) && (segCount == 1)) {
	    sprintf(p, "{%d %s %d modulate %s %s nearest nearest {0 0 0} {{%s %s} {%s %s}}}",
		    lineCount, s->texture2->label, comp, wrapS, wrapT, coordS, coordT, coordS, coordT);
	    p += strlen(p);
	}
    }
    *textures = buf;
    *p = '\0';
    return TCL_OK;
}


/********************************************************************************
 *
 * Vrml_read_IndexedLineSet
 *
 * Read in an IndexedLineSet node.
 *
 * fields:	colorIndex		[]	MFLong
 *		materialIndex		-1	MFLong
 *		normalIndex		-1	MFLong
 *		textureCoordIndex	-1	MFLong
 *
 *******************************************************************************/

Node *
Vrml_read_IndexedLineSet(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    int i;
    char *field;
    NodeIndexedLineSet *lineset;

    lineset = &node->node.indexedLineSet;
    lineset->coordIndex = NULL;
    lineset->coordlen = 0;
    lineset->materialIndex = (long *) ckalloc(sizeof(long));
    lineset->materialIndex[0] = -1;
    lineset->materiallen = 1;
    lineset->normalIndex = (long *) ckalloc(sizeof(long));
    lineset->normalIndex[0] = -1;
    lineset->normallen = 1;
    lineset->textureCoordIndex = (long *) ckalloc(sizeof(long));
    lineset->textureCoordIndex[0] = -1;
    lineset->texturelen = 1;
    
    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {
	
      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading IndexedLineSet", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading IndexedLineSet", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad IndexedLineSet format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading IndexedLineSet", (char *) NULL);
	    return (Node *) -1;

	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading IndexedLineSet", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad IndexedLineSet format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "coordIndex")) {
	    if (lineset->coordIndex) {
		(void) ckfree((void *) lineset->coordIndex);
		lineset->coordIndex = NULL;
		lineset->coordlen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &lineset->coordIndex, &lineset->coordlen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < lineset->coordlen; i++) {
		if (lineset->coordIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedLineSet coordIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else if (!strcmp(field, "materialIndex")) {
	    if (lineset->materialIndex) {
		(void) ckfree((void *) lineset->materialIndex);
		lineset->materialIndex = NULL;
		lineset->materiallen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &lineset->materialIndex, &lineset->materiallen) != TCL_OK) {
		goto err;
	    }

	    for (i = 0; i < lineset->materiallen; i++) {
		if (lineset->materialIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedLineSet materialIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else if (!strcmp(field, "normalIndex")) {
	    if (lineset->normalIndex) {
		(void) ckfree((void *) lineset->normalIndex);
		lineset->normalIndex = NULL;
		lineset->normallen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &lineset->normalIndex, &lineset->normallen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < lineset->normallen; i++) {
		if (lineset->normalIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedLineSet normalIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else if (!strcmp(field, "textureCoordIndex")) {
	    if (lineset->textureCoordIndex) {
		(void) ckfree((void *) lineset->textureCoordIndex);
		lineset->textureCoordIndex = NULL;
		lineset->texturelen = 0;
	    }
	    
	    if (Vrml_read_MFLong(interp, channel, argv, &lineset->textureCoordIndex, &lineset->texturelen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < lineset->texturelen; i++) {
		if (lineset->textureCoordIndex[i] < -1) {
		    Tcl_AppendResult(interp, argv, ": negative IndexedLineSet textureCoordIndex", (char *) NULL);
		    goto err;
		}
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad IndexedLineSet field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }
    
  err:
    Vrml_free_token(field);
    Vrml_free_IndexedLineSet(node);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_free_IndexedLineSet
 *
 * Free an IndexedLineSet node.
 *
 * fields:	colorIndex		0	MFLong
 *		materialIndex		-1	MFLong
 *		normalIndex		-1	MFLong
 *		textureCoordIndex	-1	MFLong
 *
 *******************************************************************************/

void
Vrml_free_IndexedLineSet(node)
     Node *node;
{
    NodeIndexedLineSet *lineset;
    
    lineset = &node->node.indexedLineSet;
    if (lineset->coordIndex) (void) ckfree((void *) lineset->coordIndex);
    if (lineset->normalIndex) (void) ckfree((void *) lineset->normalIndex);
    if (lineset->materialIndex) (void) ckfree((void *) lineset->materialIndex);
    if (lineset->textureCoordIndex) (void) ckfree((void *) lineset->textureCoordIndex);
}


/********************************************************************************
 *
 * Vrml_render_IndexedLineSet
 *
 * Render a IndexedLineSet node.
 *
 * fields:	colorIndex		0	MFLong
 *		materialIndex		-1	MFLong
 *		normalIndex		-1	MFLong
 *		textureCoordIndex	-1	MFLong
 *
 *******************************************************************************/

int
Vrml_render_IndexedLineSet(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    int i, vcnt, dimInit;
    int nseg, result, segCount;
    long *index, last;
    float x, y, z, dimensions[3][2];

    char ambient[TCL_DOUBLE_SPACE * 3 + 10];
    char diffuse[TCL_DOUBLE_SPACE * 3 + 10];
    char specular[TCL_DOUBLE_SPACE * 4 + 10];
    char emissive[TCL_DOUBLE_SPACE * 3 + 10];

    long *map = NULL;
    char *vertices = NULL;
    char *lines = NULL;
    char *materials = NULL;
    char *textures = NULL;
    char *normals = NULL;

    char doubleW[TCL_DOUBLE_SPACE];
    char doubleX[TCL_DOUBLE_SPACE];
    char doubleY[TCL_DOUBLE_SPACE];
    char doubleZ[TCL_DOUBLE_SPACE];

    char *argv[] = {
	"-vertices",
	NULL,
	"-lines",
	NULL,
	"-color",
	"white",
	"-ambient",
	NULL,
	"-diffuse",
	NULL,
	"-specular",
	NULL,
	"-emissive",
	NULL,
	"-shading",
	"smooth",
	"-materials",
	NULL,
	"-normals",
	NULL,
	"-textures",
	NULL,
    };
    NodeIndexedLineSet *set = &node->node.indexedLineSet;

    float s0, s1, s2;
    SFVec3f *vin;
    SFVec3f pos, fwd, up, scale;
    SurfaceProperties *s = state->surface;

    node->objects = state->objects;
    state->objects = node;

    Vrml_object_state(state, &pos, &fwd, &up, &scale);
    s0 = scale.v[0];
    s1 = scale.v[1];
    s2 = scale.v[2];

    argv[7] = ambient;
    argv[9] = diffuse;
    argv[11] = specular;
    argv[13] = emissive;

    nseg = 0;

    /* build up line coordinates */
    if ((set->coordlen > 0) && (s->coord3len > 0) &&
	((vertices = (char *) ckalloc(s->coord3len * (TCL_DOUBLE_SPACE * 3 + 10))) != NULL) &&
	((map = (long *) ckalloc(s->coord3len * sizeof(long))) != NULL)) {

	vcnt = 1;
	dimInit = 0;
	for (i = 0; i < s->coord3len; i++) {
	    map[i] = -1;
	}

	*vertices = '\0';
	for (i = 0, vin = s->coord3, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {

	    if ((*index != -1) && ((*index < 0) && (*index >= s->coord3len))) {
		sprintf(vertices, "%ld", *index);
		Tcl_AppendResult(interp, "IndexedLineSet vertex index out of bounds: \"", vertices, "\"", (char *) NULL);
		(void) ckfree((void *) vertices);
		(void) ckfree((void *) map);
		return 2;
	    }

	    if ((*index != -1) && (map[*index] == -1)) {

		map[*index] = vcnt++;

		x = vin[*index].v[0] * s0;
		y = vin[*index].v[1] * s1;
		z = vin[*index].v[2] * s2;

		Tcl_PrintDouble(interp, (double) x, doubleX);
		Tcl_PrintDouble(interp, (double) y, doubleY);
		Tcl_PrintDouble(interp, (double) z, doubleZ);

		sprintf(vertices + strlen(vertices), "{%s %s %s} ", doubleX, doubleY, doubleZ);

		/* expand bounding box if necessary */
		if (dimInit) {
		    if (x > dimensions[0][1]) dimensions[0][1] = x;
		    else if (x < dimensions[0][0]) dimensions[0][0] = x;

		    if (y > dimensions[1][1]) dimensions[1][1] = y;
		    else if (y < dimensions[1][0]) dimensions[1][0] = y;

		    if (z > dimensions[2][1]) dimensions[2][1] = z;
		    else if (z < dimensions[2][0]) dimensions[2][0] = z;
		}
		else {
		    dimensions[0][0] = dimensions[0][1] = s->coord3->v[0] * s0;
		    dimensions[1][0] = dimensions[1][1] = s->coord3->v[1] * s1;
		    dimensions[2][0] = dimensions[2][1] = s->coord3->v[2] * s2;

		    dimInit = 1;
		}
	    }
	}

	/* count the number of line segments */
	last = -1;
	segCount = 0;
	for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {
	    if (*index != -1) {
		segCount++;
		if (last != -1) nseg++;
	    }
	    else {
		if (segCount == 1) nseg++;
		segCount = 0;
	    }
	    last = *index;
	}
	if ((last != -1) && (segCount == 1)) nseg++;
    }
    argv[1] = vertices ? vertices : "";

    /* create polylines - each segment has format: {index index} */
    if ((nseg > 0) && (lines = (char *) ckalloc(20 * nseg))) {
	*lines = '\0';
	last = -1;
	segCount = 0;
	for (i = 0, index = set->coordIndex; i < (const) set->coordlen; i++, index++) {
	    if (*index != -1) {
		segCount++;
		if (last != -1) {
		    sprintf(lines + strlen(lines), "{%ld %ld} ", map[last], map[(*index)]);
		}
	    }
	    else {
		if (segCount == 1) {
		    sprintf(lines + strlen(lines), "{%ld %ld} ", map[last], map[last]);
		}
		segCount = 0;
	    }
	    last = *index;
	}
	if ((last != -1) && (segCount == 1)) {
	    sprintf(lines + strlen(lines), "{%ld %ld} ", map[last], map[last]);
	}
    }
    argv[3] = lines ? lines : "";

    /* add ambient */
    if (s->ambientColor) {
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->ambientColor->rgb[2], doubleZ);

	sprintf(ambient, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(ambient, "0.0 0.0 0.0");
    }

    /* add diffuse */
    if (s->diffuseColor) {
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->diffuseColor->rgb[2], doubleZ);

	sprintf(diffuse, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(diffuse, "0.0 0.0 0.0");
    }

    /* add specular */
    if (s->specularColor) {
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->specularColor->rgb[2], doubleZ);
	Tcl_PrintDouble(interp, (double) s->shininess[0], doubleW);

	sprintf(specular, "%s %s %s %s", doubleX, doubleY, doubleZ, doubleW);
    }
    else {
	sprintf(specular, "0.0 0.0 0.0 0.0");
    }

    /* add emissive */
    if (s->emissiveColor) {
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[0], doubleX);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[1], doubleY);
	Tcl_PrintDouble(interp, (double) s->emissiveColor->rgb[2], doubleZ);

	sprintf(emissive, "%s %s %s", doubleX, doubleY, doubleZ);
    }
    else {
	sprintf(emissive, "0.0 0.0 0.0");
    }

    if (nseg > 0) {

	/* add materials  */
	if (format_materials(interp, state, set, nseg, &materials) != TCL_OK) {
	    if (map) (void) ckfree((void *) map);
	    if (vertices) (void) ckfree((void *) vertices);
	    return 2;
	}

	/* add normals */
	if (format_normals(interp, state, set, nseg, &scale, &normals) != TCL_OK) {
	    if (map) (void) ckfree((void *) map);
	    if (vertices) (void) ckfree((void *) vertices);
	    if (materials) (void) ckfree((void *) materials);
	    return 2;
	}

	/* add textures */
	if (s->texture2) {
	    if (format_textures(interp, state, set, nseg, &scale, dimensions, &textures) != TCL_OK) {
		if (map) (void) ckfree((void *) map);
		if (vertices) (void) ckfree((void *) vertices);
		if (materials) (void) ckfree((void *) materials);
		if (normals) (void) ckfree((void *) normals);
		return 2;
	    }
	}
    }
    argv[17] = (materials) ? materials : "";
    argv[19] = (normals) ? normals : "";
    argv[21] = (textures) ? textures : "";

    /* create line model */
    result = Vrml_new_object(interp, state, node, "line", &pos, &fwd, &up, 22, argv);

    if (map) (void) ckfree((void *) map);
    if (vertices) (void) ckfree((void *) vertices);
    if (materials) (void) ckfree((void *) materials);
    if (textures) (void) ckfree((void *) textures);
    if (normals) (void) ckfree((void *) normals);

    return (result == TCL_OK) ? 0 : 2;
}
