/********************************************************************************
 *
 * lod.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_LOD
 *
 * Read in an LOD node.
 *
 * fields:	range		[ ]	MFFloat
 *		center		0 0 0	SFVec3f
 *
 *******************************************************************************/

Node *
Vrml_read_LOD(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    int dofield, nchild;
    char *field;
    Node *child, *last;
    NodeLOD *lod;

    lod = &node->node.LOD;
    lod->range = NULL;
    lod->rangelen = 0;
    lod->center.v[0] = lod->center.v[1] = lod->center.v[2] = 0;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading LOD", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading LOD", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad LOD format", (char *) NULL);
	return (Node *) -1;
  }
    
    /* parse all fields until first child node or close curly bracket */
    dofield = 1;
    while (dofield) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading LOD", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading LOD", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad LOD format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "center")) {
	    if (Vrml_read_SFVec3f(interp, channel, argv, &lod->center, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "range")) {
	    if (lod->range) (void) ckfree((void *) lod->range);
	    if (Vrml_read_MFFloat(interp, channel, argv, &lod->range, &lod->rangelen) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    dofield = 0;
	}
	if (dofield) {
	    Vrml_free_token(field);
	    field = NULL;
	}
    }
    
    /* parse child nodes */
    last = NULL;
    nchild = 0;
    child = Vrml_read_node(interp, channel, argv, names, inlines, textures, 1, field);
    Vrml_free_token(field);
    field = NULL;
    while ((child != NULL) && (child != (Node *) -1)) {
	nchild++;
	if (last) {
	    last->sibling = child;
	}
	else {
	    node->child = child;
	}
	last = child;
	child = Vrml_read_node(interp, channel, argv, names, inlines, textures, 1, NULL);
    }
    if (child == NULL) {
	return node;
    }
  
  err:
    Vrml_free_token(field);
    Vrml_free_LOD(node);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_free_LOD
 *
 * Free an LOD node.
 *
 * fields:	range		[ ]	MFFloat
 *		center		0 0 0	SFVec3f
 *
 *******************************************************************************/

void
Vrml_free_LOD(node)
     Node *node;
{
    Node *child, *sibling;
    NodeLOD *lod = &node->node.LOD;

    if (lod->range) (void) ckfree((void *) lod->range);
    for (child = node->child; child; child = sibling) {
	sibling = child->sibling;
	Vrml_free_node(child);
    }
}


/********************************************************************************
 *
 * Vrml_render_LOD
 *
 * Render an LOD node.
 *
 * fields:	range		[ ]	MFFloat
 *		center		0 0 0	SFVec3f
 *
 *******************************************************************************/

int
Vrml_render_LOD(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    CurrentModel *model = state->model;

    node->lods = state->lods;
    state->lods = node;

    node->parent = model->node;
    Vrml_coord3_transform(state, &node->node.LOD.center, &node->node.LOD.wcenter);

    return 0;
}


/********************************************************************************
 *
 * Vrml_finish_LOD
 *
 * Finish rendering an LOD node.
 *
 *******************************************************************************/

int
Vrml_finish_LOD(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    return 0;
}
