/********************************************************************************
 *
 * material.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_Material
 *
 * Read in a Material node.
 *
 * fields:	ambientColor	0.2 0.2 0.2	MFColor
 *		diffuseColor	0.8 0.8 0.8	MFColor
 *		specularColor	0.0 0.0 0.0	MFColor
 *		emissiveColor	0.0 0.0 0.0	MFColor
 *		shininess	0.2		MFFloat
 *		transparency	0.0		MFFloat
 *
 *******************************************************************************/

Node *
Vrml_read_Material(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    int i;
    char *field;
    NodeMaterial *material;

    material = &node->node.material;
    material->ambientColor = (SFColor *) ckalloc(sizeof(SFColor));
    material->diffuseColor = (SFColor *) ckalloc(sizeof(SFColor));
    material->specularColor = (SFColor *) ckalloc(sizeof(SFColor));
    material->emissiveColor = (SFColor *) ckalloc(sizeof(SFColor));
    material->shininess = (float *) ckalloc(sizeof(float));
    material->transparency = (float *) ckalloc(sizeof(float));
    material->ambientlen = 1;
    material->diffuselen = 1;
    material->specularlen = 1;
    material->emissivelen = 1;
    material->shininesslen = 1;
    material->transparencylen = 1;

    material->ambientColor->rgb[0] = 0.2;
    material->ambientColor->rgb[1] = 0.2;
    material->ambientColor->rgb[2] = 0.2;

    material->diffuseColor->rgb[0] = 0.8;
    material->diffuseColor->rgb[1] = 0.8;
    material->diffuseColor->rgb[2] = 0.8;

    material->emissiveColor->rgb[0] = 0.0;
    material->emissiveColor->rgb[1] = 0.0;
    material->emissiveColor->rgb[2] = 0.0;

    material->specularColor->rgb[0] = 0.0;
    material->specularColor->rgb[1] = 0.0;
    material->specularColor->rgb[2] = 0.0;

    *material->shininess = 0.2;
    *material->transparency = 0.0;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading Material", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Material", (char *) NULL);
	return (Node *) -1;
	
      case TOKEN_OPEN_CURLY:
	break;
	
      case TOKEN_WORD:
	Vrml_free_token(field);
	
      default:
	Tcl_AppendResult(interp, argv, ": bad Material format", (char *) NULL);
	return (Node *) -1;
    }
    
    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {
	    
	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading Material", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading Material", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:
	    return node;
	    
	  case TOKEN_WORD:
	    break;
	    
	  default:
	    Tcl_AppendResult(interp, argv, ": bad Material format", (char *) NULL);
	    return (Node *) -1;
	}

	if (!strcmp(field, "ambientColor")) {
	    if (material->ambientColor) {
		(void) ckfree((void *) material->ambientColor);
		material->ambientColor = NULL;
		material->ambientlen = 0;
	    }
	    if (Vrml_read_MFColor(interp, channel, argv, &material->ambientColor, &material->ambientlen) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "diffuseColor")) {
	    if (material->diffuseColor) {
		(void) ckfree((void *) material->diffuseColor);
		material->diffuseColor = NULL;
		material->diffuselen = 0;
	    }
	    if (Vrml_read_MFColor(interp, channel, argv, &material->diffuseColor, &material->diffuselen) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "specularColor")) {
	    if (material->specularColor) {
		(void) ckfree((void *) material->specularColor);
		material->specularColor = NULL;
		material->specularlen = 0;
	    }
	    if (Vrml_read_MFColor(interp, channel, argv, &material->specularColor, &material->specularlen) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "emissiveColor")) {
	    if (material->emissiveColor) {
		(void) ckfree((void *) material->emissiveColor);
		material->emissiveColor = NULL;
		material->emissivelen = 0;
	    }
	    if (Vrml_read_MFColor(interp, channel, argv, &material->emissiveColor, &material->emissivelen) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "shininess")) {
	    if (material->shininess) {
		(void) ckfree((void *) material->shininess);
		material->shininess = NULL;
		material->shininesslen = 0;
	    }
	    
	    if (Vrml_read_MFFloat(interp, channel, argv, &material->shininess, &material->shininesslen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < material->shininesslen; i++) {
		if (material->shininess[i] < 0) {
		    material->shininess[i] = 0;
		}
	    }
	}
	else if (!strcmp(field, "transparency")) {
	    if (material->transparency) {
		(void) ckfree((void *) material->transparency);
		material->transparency = NULL;
		material->transparencylen = 0;
	    }
	    
	    if (Vrml_read_MFFloat(interp, channel, argv, &material->transparency, &material->transparencylen) != TCL_OK) {
		goto err;
	    }
	    
	    for (i = 0; i < material->transparencylen; i++) {
		if (material->transparency[i] < 0) {
		    material->transparency[i] = 0;
		}
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad Material field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
    }

  err:
    Vrml_free_token(field);
    Vrml_free_Material(node);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_free_Material
 *
 * Free a Material node.
 *
 * fields:	ambientColor	0.2 0.2 0.2	MFColor
 *		diffuseColor	0.8 0.8 0.8	MFColor
 *		specularColor	0.0 0.0 0.0	MFColor
 *		emissiveColor	0.0 0.0 0.0	MFColor
 *		shininess	0.2		MFFloat
 *		transparency	0.0		MFFloat
 *
 *******************************************************************************/

void
Vrml_free_Material(node)
     Node *node;
{
    NodeMaterial *material;
    
    material = &node->node.material;
    
    if (material->ambientColor) (void) ckfree((void *) material->ambientColor);
    if (material->diffuseColor) (void) ckfree((void *) material->diffuseColor);
    if (material->specularColor) (void) ckfree((void *) material->specularColor);
    if (material->emissiveColor) (void) ckfree((void *) material->emissiveColor);
    if (material->shininess) (void) ckfree((void *) material->shininess);
    if (material->transparency) (void) ckfree((void *) material->transparency);
}


/********************************************************************************
 *
 * Vrml_render_Material
 *
 * Render a Material node.
 *
 * fields:	ambientColor	0.2 0.2 0.2	MFColor
 *		diffuseColor	0.8 0.8 0.8	MFColor
 *		specularColor	0.0 0.0 0.0	MFColor
 *		emissiveColor	0.0 0.0 0.0	MFColor
 *		shininess	0.2		MFFloat
 *		transparency	0.0		MFFloat
 *
 *******************************************************************************/

int
Vrml_render_Material(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    NodeMaterial *mat = &node->node.material;
    SurfaceProperties *s = state->surface;

    s->ambientColor = mat->ambientColor;
    s->ambientlen = mat->ambientlen;

    s->diffuseColor = mat->diffuseColor;
    s->diffuselen = mat->diffuselen;

    s->specularColor = mat->specularColor;
    s->specularlen = mat->specularlen;

    s->emissiveColor = mat->emissiveColor;
    s->emissivelen = mat->emissivelen;

    s->shininess = mat->shininess;
    s->shininesslen = mat->shininesslen;

    s->transparency = mat->transparency;
    s->transparencylen = mat->transparencylen;

    return 0;
}
