/********************************************************************************
 *
 * orthographic.c
 *
 ********************************************************************************/

#include <stdlib.h>
#include <tk.h>

#include "vrml.h"


/********************************************************************************
 *
 * Vrml_read_OrthographicCamera
 *
 * Read in a OrthographicCamera node.
 *
 * fields:	position	0 0 1		SFVec3f
 *		orientation	0 0 1 0		SFRotation
 *		nearDistance	0		SFFloat
 *		farDistance	100		SFFloat
 *		focalDistance	5		SFFloat
 *		aspectRatio	1		SFFloat
 *		height		2		SFFloat
 *
 *******************************************************************************/

Node *
Vrml_read_OrthographicCamera(interp, channel, argv, node, names, inlines, textures)
     Tcl_Interp *interp;
     Tcl_Channel channel;
     char *argv;
     Node *node;
     Node **names, **inlines, **textures;
{
    char *field;
    NodeOrthographicCamera *camera;

    camera = &node->node.orthographicCamera;
    camera->position.v[0] = camera->position.v[1] = 0;
    camera->position.v[2] = 1;
    camera->orientation[0] = camera->orientation[1] = camera->orientation[3] = 0;
    camera->orientation[2] = 1;
    camera->nearDistance = 0;
    camera->farDistance = 100;
    camera->focalDistance = 5;
    camera->aspectRatio = 1;
    camera->height = 2;

    /* get open curly bracket */
    switch(Vrml_get_token(channel, &field)) {

      case TOKEN_OUT_OF_MEMORY:
	Tcl_AppendResult(interp, argv, ": out of memory while reading OrthographicCamera", (char *) NULL);
	return (Node *) -1;

      case TOKEN_EOF:
      case TOKEN_END:
	Tcl_AppendResult(interp, argv, ": unexpected end of input while reading OrthographicCamera", (char *) NULL);
	return (Node *) -1;

      case TOKEN_OPEN_CURLY:
	break;

      case TOKEN_WORD:
	Vrml_free_token(field);

      default:
	Tcl_AppendResult(interp, argv, ": bad OrthographicCamera format", (char *) NULL);
	return (Node *) -1;
    }

    /* parse all fields until close curly bracket */
    while (1) {
	
	switch(Vrml_get_token(channel, &field)) {

	  case TOKEN_OUT_OF_MEMORY:
	    Tcl_AppendResult(interp, argv, ": out of memory while reading OrthographicCamera", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_EOF:
	  case TOKEN_END:
	    Tcl_AppendResult(interp, argv, ": unexpected end of input while reading OrthographicCamera", (char *) NULL);
	    return (Node *) -1;
	    
	  case TOKEN_CLOSE_CURLY:

	    /* make sure field values are legal */
	    if ((camera->orientation[0] == 0) && (camera->orientation[1] == 0) && (camera->orientation[2] == 0)) {
		Tcl_AppendResult(interp, argv, ": null rotation axis specified for OrthographicCamera node", (char *) NULL);
		goto err;
	    }
	    if (camera->aspectRatio <= 0) {
		Tcl_AppendResult(interp, argv, ": PerspectiveCamera aspect ratio must be positive", (char *) NULL);
		goto err;
	    }
	    if (camera->nearDistance < 0) {
		camera->nearDistance = 0;
	    }
	    if (camera->farDistance <= camera->nearDistance) {
		Tcl_AppendResult(interp, argv, ": far distance is smaller than near distance for OrthographicCamera node", (char *) NULL);
		goto err;
	    }
	    return node;

	  case TOKEN_WORD:
	    break;

	  default:
	    Tcl_AppendResult(interp, argv, ": bad OrthographicCamera format", (char *) NULL);
	    return (Node *) -1;
	}
	
	if (!strcmp(field, "height")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &camera->height, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "aspectRatio")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &camera->aspectRatio, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "farDistance")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &camera->farDistance, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "nearDistance")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &camera->nearDistance, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "focalDistance")) {
	    if (Vrml_read_SFFloat(interp, channel, argv, &camera->focalDistance, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	    if (camera->focalDistance <= 0) {
		Tcl_AppendResult(interp, argv, ": focal distance must be positive", (char *) NULL);
		goto err;
	    }
	}
	else if (!strcmp(field, "position")) {
	    if (Vrml_read_SFVec3f(interp, channel, argv, &camera->position, NULL, NULL) != TCL_OK) {
		goto err;
	    }
	}
	else if (!strcmp(field, "orientation")) {
	    if (Vrml_read_SFRotation(interp, channel, argv, camera->orientation) != TCL_OK) {
		goto err;
	    }
	}
	else {
	    Tcl_AppendResult(interp, argv, ": bad OrthographicCamera field \"", field, "\"", (char *) NULL);
	    goto err;
	}
	Vrml_free_token(field);
	field = NULL;
    }

  err:
    if (field) Vrml_free_token(field);
    return (Node *) -1;
}


/********************************************************************************
 *
 * Vrml_render_OrthographicCamera
 *
 * Render an OrthographicCamera node.
 *
 * fields:	position	0 0 1		SFVec3f
 *		orientation	0 0 1 0		SFRotation
 *		nearDistance	0		SFFloat
 *		farDistance	100		SFFloat
 *		focalDistance	5		SFFloat
 *		aspectRatio	1		SFFloat
 *		height		2		SFFloat
 *
 *******************************************************************************/

int
Vrml_render_OrthographicCamera(interp, node, state)
     Tcl_Interp *interp;
     Node *node;
     State *state;
{
    NodeOrthographicCamera *camera;
    SFVec3f vpn, vup;

    node->cameras = state->cameras;
    state->cameras = node;

    camera = &node->node.orthographicCamera;
    Vrml_rotate_vector(camera->orientation, 0, 0, 1, vpn.v);
    Vrml_rotate_vector(camera->orientation, 0, 1, 0, vup.v);

    Vrml_coord3_transform(state, &camera->position, &camera->vrp);
    Vrml_vector_transform(state, &vpn, &camera->vpn);
    Vrml_vector_transform(state, &vup, &camera->vup);

    return 0;
}
